      PROGRAM plotsel

!  this program looks through output file of chem solver to
!   create selected plots of species and/or groups.
!  The desired species and groups are listed in the input file 
!                    plot.species
!  In this file, explicit species are listed in the first four characters of
!  each line.  For groups (say group x) two options exist:
!       a(x)   -   all species containing this group will be used
!       d(x)   -   only species having group x as dominant will be used
!  for each group, output it written for the SUM of the group, and for the top
!  three members of the group (based on average value during a run)
!  Output fort.23 may be imported into a spreadsheet.

      IMPLICIT NONE

      INTEGER,PARAMETER :: nxspec=2000  
      INTEGER :: ntime

!  dimension species dictionary parameters
      CHARACTER(79),DIMENSION(nxspec) :: dict
      CHARACTER(40),DIMENSION(nxspec) :: chem
      CHARACTER(10),DIMENSION(nxspec) :: code
      CHARACTER(4),DIMENSION(nxspec) :: sdict
      CHARACTER(1),DIMENSION(nxspec) :: dom
      CHARACTER(1),DIMENSION(32) :: class
      CHARACTER(32) :: preced
      INTEGER,DIMENSION(nxspec) :: ic,ix,ih,in,io,is

! dimension solver output parameters 
      CHARACTER(4),DIMENSION(nxspec) :: ssolv
      CHARACTER(1),DIMENSION(10) :: a
      REAL,DIMENSION(6) :: adata
      INTEGER,ALLOCATABLE,DIMENSION(:) :: ratlab
      REAL,ALLOCATABLE,DIMENSION(:) :: time  
      REAL,ALLOCATABLE,DIMENSION(:,:) :: c
      REAL,ALLOCATABLE,DIMENSION(:,:) :: rate   

! desired time interval (units = days) for tabular output 
! fort.23, fort.28 
      REAL,PARAMETER :: dtime=1./24./60.*6.
      INTEGER :: ntabl     ! number of times for tabular output
      REAL,ALLOCATABLE,DIMENSION(:) :: ttabl
      REAL,ALLOCATABLE,DIMENSION(:,:) :: ctabl
      REAL,ALLOCATABLE,DIMENSION(:,:) :: rtabl
      CHARACTER(25),DIMENSION(nxspec) :: tabnam

!  dimension group parameters
      CHARACTER(4) :: sel
      CHARACTER(1) :: group,atom
      INTEGER,DIMENSION(nxspec) :: ifind,iorder
      REAL,DIMENSION(nxspec) :: gsum,gaver,aveord

!  dimension plot parameters
      REAL,ALLOCATABLE,DIMENSION(:) :: x,y  
      CHARACTER(25) :: title
      CHARACTER(10) :: runttl,finame, fil1

! internal variables
      INTEGER :: i,j,jj,k,jc,jx,jh,jn,jo,js
      INTEGER :: itime,iplot,igroup,irate
      INTEGER :: nrate,nt
      INTEGER :: ndict,ncspec,nsolv,nblock,nlast,nc,nr,nint,ngroup,nsel
      REAL :: yfirst,ylast,yint,yave,rsum
      CHARACTER(4),DIMENSION(nxspec) :: selnam
      CHARACTER(10) :: line
      INTEGER,DIMENSION(nxspec) :: isel

! input files
!     7  -  solver output: concentrations
!     8  -  solver output: rates
!     11 -  dictionary
!     12 -  selections
!     13 -  solver input file

        OPEN(UNIT=11,FILE='../MM/alphadict.dat',STATUS='OLD')
        OPEN(UNIT=12,FILE='plot.species',STATUS='OLD')

! output files
!        22  -  summary (initial, average, final concentrations)
!        23  -  concentration vs. time : spreadsheet format
!        28  -  reaction rate vs. time : spreadsheet format

! set order of precedence for attributes:

        preced = '0123456789dpghnekaoucfblsmrtvwiq'
        DO i = 1,32
           class(i) = preced(i:i)
        ENDDO

!    read dictionary of species
        ndict = 0
        DO i=1,nxspec
           READ(11,200) dict(i)
           IF(dict(i)(1:4) .EQ. '****') EXIT
           ndict = ndict + 1
           sdict(i) = dict(i)(1:4)
        ENDDO

! read solver output: time, concs
        READ(7,*)ntime
        ALLOCATE(time(ntime),x(ntime),y(ntime),c(nxspec,ntime))
        DO i = 1, nxspec
           READ(7,205) (a(j),j=1,6)
           IF(a(1).EQ.'E') EXIT
           nsolv = i

!  kill parentheses and rewrite in A4 format
           DO j = 3,5
              IF(a(j).EQ.')') a(j) = ' '
           ENDDO
           WRITE(ssolv(i),210) (a(j),j=2,5)

! find species characteristics from dictionary
           CALL SEARCH(ssolv(i),sdict,ndict,j)
           READ(dict(j),215) ic(i),ix(i),ih(i),in(i), &
                       io(i),is(i),chem(i),code(i)

!  find dominant attribute by searching backwards through hierarchy
           dom(i) = ' '
           DO jj = 1, 32
              k = 33 - jj
              IF(INDEX(code(i),class(k)).NE.0) dom(i) = class(k)
           ENDDO

        ENDDO

! compute total number of blocks and number of columns in last block
!  (each block is TIME and up to 5 DATA columns) - then read values

        nblock = 1 + (nsolv - 1)/5
        nlast = nsolv - 5*(nblock - 1) + 1
        nc = 6

        DO j = 1, nblock 
           IF(j .EQ. nblock) nc = nlast
           DO i = 1, ntime
              READ(7,*) (adata(k), k = 1,nc)
              time(i) = adata(1)/(24.*3600.)
              DO k = 2,nc
                 nr = 5*(j-1) + k-1
                 c(nr,i) = adata(k)
              ENDDO
           ENDDO
        ENDDO

        ntabl = IFIX((time(ntime)-time(1))/dtime)+1
        ALLOCATE(ttabl(ntabl),ctabl(nxspec,ntabl))
!_____________________________________ 
!  set plot constants:
        DO itime = 1, ntime
              x(itime) = time(itime)
        ENDDO
        iplot = 0
        nsel = 0

!_____ generate header for file 22
       WRITE(22,*) '# name initial  average  final    formula'

!_____ loop through selections (re-entry at 2000)
2000   CONTINUE

       READ(12,220) sel 
       IF (sel .EQ. '****') GO TO 135

!**** explicit species:
       IF ((sel(2:2).EQ.'(').AND.(sel(4:4).EQ.')').OR. &
                     sel(1:1).EQ.'#') GO TO 60

! find species to plot:
        DO j = 1, nsolv
           IF(sel.EQ.ssolv(j)) THEN
              title = sel
              nsel = nsel + 1
              selnam(nsel) = sel
              isel(nsel) = j
              DO itime = 1, ntime
                 y(itime) = c(j,itime)
              ENDDO

              yfirst = y(1)
              ylast = y(ntime)
              yint = 0.
! use all data for computing average values
              DO i = 2, ntime
                 yint = yint + (y(i) + y(i-1))*0.5*(x(i)-x(i-1))
                 yint = yint + y(i)
              ENDDO
              yave = yint/x(ntime)
!!!! use fifth day, from 10:30 to 14:30 only, for computing average values
!!!!              nint = 0
!!!!              DO i = 2, ntime
!!!!                 IF(x(i) .LT. 4.4375 .OR. x(i) .GT. 4.6042) CYCLE
!!!!                 nint = nint + 1
!!!!              ENDDO
!!!!              yave = yint/float(nint)

! output
              !WRITE(21,'(A25)') title
              !DO i = 1, ntime
              !   WRITE(21,2100)x(i),y(i)
              !ENDDO
              iplot = iplot + 1
              tabnam(iplot) = title
              WRITE(22,225) iplot, ssolv(j),yfirst,yave,ylast,chem(j)
              CALL INTERP(x,y,ntime,dtime,ttabl,ctabl(iplot,:),ntabl)
              GO TO 2000
           ENDIF
        ENDDO
        GO TO 2000

60      CONTINUE

!**** by functional group
! decide whether to include all species which have group x, or only those
! which have it as the dominant group:
!          to include all, specify a(x)
!          to include dominant only, specify d(x)
!  find index for all group members

        igroup = 0
        IF(sel(1:1) .EQ. '#') THEN
           atom = sel(2:2)
           IF     (atom .EQ. 'C') THEN
              READ(sel,'(2X,I2)') jc 
              DO j = 1, nsolv
                 IF (ic(j) .EQ. jc ) THEN
                    igroup = igroup + 1
                    ifind(igroup) = j            
                 ENDIF
              ENDDO
           ELSE IF(atom .EQ. 'X') THEN
              READ(sel,'(2X,I2)') jx
           ELSE IF(atom .EQ. 'H') THEN
              READ(sel,'(2X,I2)') jh
           ELSE IF(atom .EQ. 'N') THEN
              READ(sel,'(2X,I2)') jn
              DO J = 1, nsolv
                 IF (in(j) .EQ. JN ) THEN
                    IF(ssolv(j) .EQ. 'N2  ') CYCLE
                    igroup = igroup + 1
                    ifind(igroup) = j            
                 ENDIF
              ENDDO
           ELSE IF(atom .EQ. 'O') THEN
              READ(sel,'(2X,I2)') jo
           ELSE IF(atom .EQ. 'S') THEN
              READ(sel,'(2X,I2)') js
           ENDIF
        ELSE IF (sel(1:1) .EQ. 'a') THEN
           group = sel(3:3)
           DO j = 1, nsolv
              IF (INDEX(code(j),group) .NE. 0 ) THEN
                 igroup = igroup + 1
                 ifind(igroup) = j            
              ENDIF
           ENDDO
        ELSE IF (sel(1:1) .EQ. 'd') THEN
           group = sel(3:3)
           DO j = 1, nsolv
              IF (dom(j) .EQ. group) THEN
                 igroup = igroup + 1
                 ifind(igroup) = j            
              ENDIF
           ENDDO
        ENDIF

        ngroup = igroup
        If (ngroup .EQ. 0) GO TO 2000
        
!  find group sum as a function of time

        DO itime = 1, ntime        
           gsum(itime) = 0.
           DO igroup = 1, ngroup
              gsum(itime) = gsum(itime) + c(ifind(igroup),itime)
           ENDDO
        ENDDO

!  find time-average for each member of group
        DO igroup = 1, ngroup
           rsum = 0.
           DO itime = 2, ntime
              rsum = rsum +                                            &
                 0.5*(c(ifind(igroup),itime)+c(ifind(igroup),itime-1)) &
                 *(x(itime) - x(itime-1))
           ENDDO
           gaver(igroup) = rsum/x(ntime)
        ENDDO

! sort group member indices according to decreasing average value:
!  first, find old order of appearance:
        DO igroup = 1, ngroup
           aveord(igroup) = - gaver(igroup)
           iorder(igroup) = ifind(igroup)
        ENDDO

        IF (ngroup.NE.1) CALL SORT(ngroup,aveord,iorder)

! calculate initial, average, and final values of group sum
        DO itime = 1, ntime
           y(itime) = gsum(itime)
        ENDDO
        yfirst = y(1)
        ylast = y(ntime)
        yint = 0.
        DO i = 2, ntime
           yint = yint + (y(i) + y(i-1))*0.5*(x(i)-x(i-1))
        ENDDO
        yave = yint/x(ntime)
! output group sum:
        title = sel
        !WRITE(21,'(A25)') title
        !DO i = 1, ntime
        !   WRITE(21,2100)x(i),y(i)
        !ENDDO
        iplot = iplot + 1
        tabnam(iplot) = title
        WRITE(22,230) iplot,sel,yfirst,yave,ylast
        CALL INTERP(x,y,ntime,dtime,ttabl,ctabl(iplot,:),ntabl)

!** top four explicit members:
        DO igroup = 1, ngroup
           IF (igroup .GT. 4) EXIT
           title = ssolv(iorder(igroup))
           DO itime = 1, ntime
              y(itime) = c(iorder(igroup),itime)
           ENDDO
           yfirst = y(1)
           ylast = y(ntime)
           yint = 0.
           DO i = 2, ntime
              yint = yint + (y(i) + y(i-1))*0.5*(x(i)-x(i-1))
           ENDDO
           yave = yint/x(ntime)
! output
           iplot = iplot + 1
           tabnam(iplot) = title
           !WRITE(21,'(A25)') title
           !DO i = 1, ntime
           !   WRITE(21,2100)x(i),y(i)
           !ENDDO
           WRITE(22,225) iplot, ssolv(iorder(igroup)), &
                     yfirst,yave,ylast,chem(iorder(igroup))
           CALL INTERP(x,y,ntime,dtime,ttabl,ctabl(iplot,:),ntabl)
        ENDDO

        GO TO 2000
135     CONTINUE

        PRINT*,"Selected time-averaged concentrations in file fort.22"
        CLOSE(22)

! Output the  conc vs. x file (column format) => fort.23
! tabnam(i) = title bar x, conc1, conc2, etc.
        WRITE(23,300) 'time',(tabnam(i),i=1,iplot)
        DO j=1,ntabl
           WRITE(23,310) ttabl(j),(ctabl(i,j),i=1,iplot)
        ENDDO

        PRINT*,"Selected concentration timeseries in file fort.23"
        CLOSE(23)

!----------------------------------------
! read solver output: time, rates 
! first find nr by inspecting solv.inp
        OPEN(UNIT=13,FILE='solv.inp',STATUS='OLD')
          READ(13,*)line
          READ(13,*)nr
        CLOSE(13)

! then read averaged rates from fort.8 output
        DO i=1,nr+1
          READ(8,*)line
        ENDDO

! check whether fort.8 contains rate info vs time
! If no timeseries, the last line read will say "END OF RATES"
! If that line not present, the last line read = 
!       the first line of the first timeseries
        IF(INDEX(line,'END').EQ.0)THEN

          ! read that line again, as integers
          BACKSPACE(8)
          READ(8,*) nt,nrate

          ! check that nt = ntime
          IF(nt.NE.ntime)THEN
            PRINT*,"ERROR: nt in fort.8 <> ntime in fort.7"
            STOP
          ENDIF 
          ALLOCATE(ratlab(nrate),rate(nrate,ntime),rtabl(nrate,ntabl))
       
          ! read rates 
          DO irate=1,nrate
            READ(8,*)ratlab(irate)
            PRINT*, ratlab(irate)
            DO itime=1,ntime
              READ(8,*) time(itime),rate(irate,itime)
            ENDDO
            DO itime = 1, ntime
              y(itime) = rate(irate,itime)
            ENDDO
            CALL INTERP(x,y,ntime,dtime,ttabl,rtabl(irate,:),ntabl)
          ENDDO

          ! Output the rate vs. time file (column format) => fort.28
          WRITE(28,305) 'time',(ratlab(i),i=1,nrate)
          DO j=1,ntabl
            WRITE(28,310) ttabl(j),(rtabl(i,j),i=1,nrate)
          ENDDO

          !WRITE(28,*)'********************' 
          ! output rates at SELECTED times
          !DO j=1,ntabl 
          !  WRITE(28,310) ttabl(j),(rtabl(i,j),i=1,nrate)
          !ENDDO

          PRINT*,"Selected rate timeseries in file fort.28"
          CLOSE(28)

          DEALLOCATE(ratlab,rate,rtabl)

        ENDIF

200     FORMAT(A79)
205     FORMAT(6A1)
210     FORMAT(4A1)
215     FORMAT(4X,6(1X,I2),2X,A40,1X,1X,A10)
220     FORMAT(A4)
225     FORMAT(I2,1X,A4,3(1X,1PE8.2),1X,A40)
230     FORMAT(I2,1X,A4,3(1X,1PE8.2))
235     FORMAT(A10)
300     FORMAT(100(1X,A10))
305     FORMAT(1X,A10,99(1X,I10))
310     FORMAT(100(1X,1PE10.3))
2100    FORMAT(0pf8.4,1x,1pe12.4)

        DEALLOCATE(x,y,time,c,ctabl,ttabl)

        END
!***************************************************************
        SUBROUTINE SORT(NS,WS,S)
        DIMENSION WS(NS)
        INTEGER S(NS)
   20   I = 1
   30   J = I + 1
        IF (WS(I).LE.WS(J)) GO TO 10
        WSTORE = WS(J)
        STORE = S(J)
        WS(J) = WS(I)
        S(J) = S(I)
        WS(I) = WSTORE
        S(I) = STORE
        I = I - 1
        IF (I.EQ.0) GO TO 20
        GO TO 30
  10    IF (J.EQ.NS) GO TO 40
        I = I + 1
        GO TO 30
  40    RETURN
        END     
!***************************************************************
        SUBROUTINE SEARCH(ASEEK,ALIST,NLIST,JFOUND)
        CHARACTER*4 ASEEK, ALIST(1)
                JOLD = 0
                JLO = 1
                JHI = NLIST + 1
10                J = (JHI+JLO)/2
                IF(J.EQ.JOLD) GO TO 40
                JOLD = J
                IF(ASEEK.GT.ALIST(J)) GO TO 20
                IF(ASEEK.EQ.ALIST(J)) GO TO 30
                JHI = J
                GO TO 10
20                JLO = J
                GO TO 10
30                JFOUND = J
                RETURN
40                JFOUND = 0
                RETURN
                END
!***************************************************************
        SUBROUTINE INTERP(x0,y0,nx0,dx1,x1,y1,nx1)

! Interpolate the solver output to an equally spaced x interval
!
! nx0 = number of solver time points.
! x0  = solver times.
! y0  = solver concentrations
! dx1 = output time increment
! x1 = decimal day values of interpolated tabluar output
! y1 = interpolated concentration tabular output
! nx1 = number of tabular output grid points

        INTEGER :: nx0,nx1,j,i
        REAL :: lo,hi,f,dx1
        REAL,DIMENSION(nx0) :: x0,y0
        REAL,DIMENSION(nx1) :: x1,y1

! Initialize first grid point to first solver output
        x1(1) = x0(1)
        y1(1) = y0(1)

! Define equally spaced x grid
        DO j = 2,nx1
           x1(j)= x1(j-1) + dx1
        ENDDO

! Loop over each x grid point and perform integration
        DO j = 2, nx1
! Calculate the solver points on either side of x grid point
          DO i = 1,nx0
            IF (x1(j).LT.x0(i)) THEN
              lo = x0(i-1)
              hi = x0(i)
              EXIT
            ENDIF
          ENDDO
! Calculate the fraction, F , into the solver x increment
          f = (x1(j)-lo)/(hi-lo)
! Linearly interpolate y to y1
          y1(j) = y0(i-1) + f*(y0(i)-y0(i-1))
        ENDDO
   
        END

