
!     NCAR/ACD, PO Box 3000, Boulder CO 80307   --  Apr 2006
!
!    April 2006: gear solver routines updated to fortran 90 (JMLT)
!
!    > May 2005 : updated to fortran90 by Julia Lee-Taylor.
!    > - Revised subroutine NEWCHM to allow user constraint of any species.
!    > - Revised units of emission, deposition, dilution, entrainment.
!    > - Output all concs (fort.9), for initialisation of subsequent runs.
!    > - Updated HNO3 + OH rate constants as per JPL Eval #14, 2003
!
!    > (March 1998)
!    > This program was developed due to the efforts of Sasha Madronich (Lead
!    > Scientist) and Kent Sieckman (Head Programmer).  It is a major rewrite
!    > of a code referred to as chemk, written in the late 70s and early 80s.
!     
!    > This code should be very portable, it has been run on a Pyramid 90x,
!    > Vax 11/750 running Unix, MacIntosh SE, MacIntosh II, Cray-1 (COS),
!    > Sun-4 and VAX 8300 running VMS.
!
!     The program sets up and integrates chemical rate equations.  The 
!     chemical reactions are read from a mechanism file (solv.inp).
!     Parameters NRXN - number of reactions & NSPC - number of species
!     are read from module bounds_mod.f90 (created simultaneously with solv.inp)
!     Parameter MSAVE - number of time slots to be saveds - is set below.
!
!______________________________unit conversions______________________________
! It is assumed that all input is in molec, cm3, sec. units. HOWEVER,
! The program operates at ppm, min units.  Runs with molec., cm3, sec units
! were tried, and were ok for steady light.  But program failed when diurnal
! light cycle was put in.  It works ok with the ppm, min units. 
! Here, the units are converted to NTP ppm (2.45e13 molecules/ppm) and min.
! as soon as data is read in.  However, for special rate constant functions
! which are calculated in subroutine RATCO5, the unit conversion factors
! are passed in by way of MODULE UNITS.
! Units are left general in case future work identifies the problem with the 
! cm3, molec, sec units.  If output is desired in different units, the output 
! subroutine must be edited. (Conversions specified in MODULE UNITS)
!---------------------------------------------------------------------------

!*****************************************************
! MODULES: Formerly common blocks 
!*****************************************************
! Array size specifiers
      MODULE PARAMS
        USE BOUNDS_MOD  ! numbers of reactions & species
        IMPLICIT NONE
        SAVE
          INTEGER :: i,j
          INTEGER, PARAMETER :: &
                   nrx1=7*nrxn, &
                   nrx4=2*nrxn+1, &
                   nrx5=25*nrxn
      END MODULE PARAMS

!---------------------------------
! variables for differential equations:

      MODULE CONTRL
        IMPLICIT NONE
        SAVE
        INTEGER :: err
        REAL :: start,stopp
      END MODULE CONTRL

      MODULE MECH
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nr
          INTEGER, DIMENSION(nrxn) :: nreag, nprod
          INTEGER, DIMENSION(nrxn,7) :: kr
          REAL, DIMENSION(nrxn,7) :: sc
          CHARACTER(4) :: iblank = '    '
          CHARACTER(4),DIMENSION(nspc) :: specis
          CHARACTER(4),DIMENSION(nrxn,7) :: molec
      END MODULE MECH

      MODULE KINET
        USE PARAMS
        IMPLICIT NONE
        SAVE
          REAL :: T
          REAL,DIMENSION(nrxn) :: R
      END MODULE KINET
!---------------------------------
! variables for input/output:

!__________________convert units______________________________
!  It is assumed that all input is in molec, cm3, sec. units. HOWEVER,
! The program operates at ppm, min units.  Runs with molec., cm3, sec units
! were tried, and were ok for steady light.  But program failed when diurnal
! light cycle was put in.  It works ok with the ppm, min units. 
!  Here, the units are converter to NTP ppm (2.45e13 molecules/ppm) and min.
!  as soon as data is read in.  However, for special rate constant functions
!  which are calculated in subroutine RATCO5, the unit conversion factors
! are passed in by way of common block /UNITS/.  
!  Units are left general if future work identifies the problem with the 
! cm3, molec, sec units.                               If output is
! desired in different units, the output subroutine must be edited.
! NB: all values are back-converted after use in the solver so do not
! interact with the acutal model temperature, pressure etc.

      MODULE UNITS
        IMPLICIT NONE
        SAVE
          REAL,PARAMETER :: tunit=60., cunit=2.45E19*1.0E-06, &
                  u0=tunit/cunit, u1=tunit,                   &
                  u2=tunit*cunit, u3=tunit*cunit*cunit,       &
                  tday = 24.*3600./tunit, zunit = 1.e+5
      END MODULE UNITS

! INPUT/OUTPUT File Unit Numbers:
!     inp  = Input Data File (solv.inp)
!     iout = Output File, Results
!     iout2 = Output File, rates at simulation end
!     iout3 = Output File, concentrations at simulation end
!     lout = Output Unit For Special Messages, (eg. Errors, Comm. )
      MODULE IOFIL
        IMPLICIT NONE
        SAVE
          INTEGER :: inp=1, lout=6, iout=7, iout2=8, iout3=9
      END MODULE IOFIL

      MODULE SAVED
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nt
          REAL,DIMENSION(msave) :: savtim
          REAL,DIMENSION(msave,nspc) :: savcon
      END MODULE SAVED

      MODULE RATES
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nrates
          INTEGER,DIMENSION(mrsave) :: irates
          REAL,DIMENSION(nrxn) :: rts
          REAL,DIMENSION(msave,mrsave) :: savrat
      END MODULE
!---------------------------------
! variables internal to GEAR solver:
      MODULE GEARS
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: ipti2,ipti3,iptr2,iptr3,nzro
          INTEGER,DIMENSION(nrx5) :: iw2
          REAL :: h,h0,hmin,hmax,hmax0,eps,epsj,uround
          REAL,DIMENSION(nrxn) :: ymax
      END MODULE GEARS
!---------------------------------
! variables for environmental constraints:
      MODULE FLAGS                           
        IMPLICIT NONE
        SAVE
        CHARACTER(8),DIMENSION(8) ::  cflag = (/'pbl_ht  ','Temp (K)', &
                                     'cloud   ','dilution','entrain ', &
                                     'emission','userconc','jvalue  '/)
      END MODULE FLAGS

      MODULE JVALU
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nr1,nphot
          INTEGER,DIMENSION(nrxn) :: itype1
          REAL,ALLOCATABLE,DIMENSION(:) :: tjval
          REAL,ALLOCATABLE,DIMENSION(:,:) :: xjval,pslope
      END MODULE JVALU

      MODULE DENSIT
        IMPLICIT NONE
        SAVE
          REAL :: dens,cmsolv,cmblsti,cmblstv
      END MODULE DENSIT

      MODULE TEMPER
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: ntemp
          REAL :: temp
          REAL,DIMENSION(nenv) :: ttemp,xtemp
      END MODULE TEMPER

      MODULE DILUTE
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nbg,ndilut 
          REAL :: dilut,kdilut,entrn,kentrn
          REAL,DIMENSION(nenv) :: tdilut,xdilut
          !REAL,DIMENSION(nspc) :: cbg
          REAL,ALLOCATABLE,DIMENSION(:) :: cbg
      END MODULE DILUTE

      MODULE CLOUD
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: ncloud
          REAL :: fcloud
          REAL,DIMENSION(nenv) :: tcloud,xcloud
      END MODULE CLOUD

      MODULE EMIT
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nsem,ntemx
          INTEGER,DIMENSION(nspc) :: ntem,iemit
          REAL,DIMENSION(nspc) :: yem,kem
          REAL,ALLOCATABLE,DIMENSION(:,:) :: tem,xem
      END MODULE EMIT
!---------------------------------
! rate constant data base:
      MODULE ARRHEN
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nr2, nr3 
          INTEGER,DIMENSION(nrxn) :: itype2, itype3
          REAL,DIMENSION(nrxn) :: A, S
      END MODULE ARRHEN

      MODULE TROE
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nr4
          INTEGER,DIMENSION(nrxn) :: itype4
          REAL,DIMENSION(nrxn) :: ak0300,aki300,an,am,bas,aequil,tequil
      END MODULE TROE

      MODULE SPCIAL  !incl relevant variables from old COMMON /WATER/
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nr5,iH2O
          INTEGER,DIMENSION(nrxn) :: itype5, iwhich
          REAL :: cH2O
      END MODULE SPCIAL

      MODULE PBL
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nr6, nz 
          INTEGER,DIMENSION(nrxn) :: itype6
          REAL ::  z
          REAL,DIMENSION(nenv) ::  tz,xz
      END MODULE PBL

      MODULE DEPOS
        USE PARAMS
        IMPLICIT NONE
        SAVE 
          INTEGER :: nr7
          INTEGER,DIMENSION(nrxn) :: itype7
      END MODULE DEPOS
!---------------------------------
! variables for chemical constraints:
      MODULE USERSPC
        USE PARAMS
        IMPLICIT NONE
        SAVE
          INTEGER :: nusr,ntusrx
          INTEGER,PARAMETER :: nfspc=3
          INTEGER,DIMENSION(nspc) :: ntusr   ! # of t / conc pairs
          REAL,ALLOCATABLE,DIMENSION(:,:) :: tusr,xusr
! species index : allows for family members
          !INTEGER,DIMENSION(nfspc,nspc) :: ispc  
          INTEGER,ALLOCATABLE,DIMENSION(:,:) :: ispc  
          REAL,DIMENSION(nspc) :: cusr  ! current conc
          CHARACTER(4),DIMENSION(nspc) :: usrspec
      END MODULE USERSPC
!---------------------------------------------------------------------------
!---------------------------------------------------------------------------
      PROGRAM CHEM

      USE PARAMS
! modules for differential equations:
      USE CONTRL
      USE MECH
      USE KINET
      USE GEARS
! modules for input/output:
      USE UNITS
      USE IOFIL
      USE SAVED
      USE RATES
! module for chemical constraints:
      USE USERSPC
! modules for environmental constraints:
      USE DENSIT
      USE TEMPER
      USE DILUTE
      USE JVALU
      USE CLOUD
      USE EMIT
! rate constant data base:
      USE ARRHEN
      USE TROE
      USE SPCIAL
      USE PBL
      USE DEPOS

      IMPLICIT NONE

! emissions table (values that only appear in this subroutine)

      CHARACTER(2),DIMENSION(nrxn) ::   c2
      CHARACTER(4)                 ::   r1,r2,r3,hdr
      CHARACTER(4),DIMENSION(nspc) ::   icspec,bgspec,emspec
      CHARACTER(20)                ::   finame
      CHARACTER(30)                ::   inpname
      CHARACTER(1)                 ::   response

      INTEGER :: ii,jj,k,m1,m2,m3,mr1,mr2,mr3,mr4,mr5,mr7, &
                 mphot,mtot,nics,ns,nval
      INTEGER,DIMENSION(nrxn) :: rtyp
      INTEGER                 :: ia(nrxn),ja(nrx1)

      REAL,DIMENSION(nspc) ::  ci,bgconc
      REAL,ALLOCATABLE,DIMENSION(:) ::  c

!---------------------------------------------------------------------------
! INITIALIZE DEFAULTS:
!     nt is the current number of times that have saved, initialized to zero.
      nt  = 0
      dilut = 0.  
      fcloud = 1.
      temp = 298.
      z = zunit

      print*,'nrx1=',nrx1
      print*,'nrx4=',nrx4
      print*,'nrx5=',nrx5
!---------------------------------------------------------------------------
!     Open the filename, if there is a problem OPNDAT will give error
!     message and terminate.

      finame = 'solv.inp'
      CALL OPNDAT( inp, finame )
 
!___________________________ START DATA READ _______________________________
!---------------------------------------------------------------------------
! REACTION NUMBERS - BY TYPE
!     NR     - TOTAL NUMBER OF THE REACTIONS
!     MR1    - NUMBER OF PHOTOLYSIS REACTIONS
!     MR2    - NUMBER OF ARRHENIUS REACTIONS
!     MR3    - NUMBER OF THREE-BODY REACTIONS
!     MR4    - NUMBER OF TROE REACTIONS
!     MR5    - NUMBER OF REACTIONS WITH SPECIAL FUNCTION FOR R.CONST.
!     MR6    - NUMBER OF EMISSION RATE REACTIONS 
!     MR7    - NUMBER OF DEPOSITION REACTIONS 
!     NPHOT  - NUMBER OF TIME VS. PHOTOLYSIS RATE CONST. DATA PAIRS
      inpname = ' header'
      READ(inp,*,ERR=2000,END=2010) hdr
      inpname = ' line 2'
      READ(inp,*,ERR=2000,END=2010) nr,mr1,mr2,mr3,mr4,mr5,mr7,nphot

      IF (nr .GT. nrxn) THEN
          WRITE(6,*) &
             'STOPPING:  2nd declaration of # reactions ',&
             '> nrxn defined in  bounds_mod.f90'
          STOP
      ENDIF

      ! write(6,*)'done read NR, etc.'

!---------------------------------------------------------------------------
! REAL OPTIONS
!     START  - STARTING TIME FOR INTEGRATION, IN SECONDS
!     STOPP  - ENDING TIME FOR INTEGRATION, IN SECONDS

      READ(inp,*) hdr
      inpname = ' start, end times'
      READ(inp,*,ERR=2000,END=2010) start,stopp
      write(6,*)'done read',inpname
        start = start/tunit
        stopp = stopp/tunit

!---------------------------------------------------------------------------
!!! read temperatures:
!    ttemp - times, in seconds
!    xtemp - temperatures, in Kelvin

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      ntemp = nval
      IF (ntemp .GT. 0) THEN
        IF(ntemp.GT.nenv) WRITE(6,*) ntemp,nenv, &
             'warning:  nenv < ntemp'
        DO i = 1, ntemp
          READ(inp,*,ERR=2000,END=2010) ttemp(i), xtemp(i)
        ENDDO
        ttemp = ttemp/tunit
        temp = xtemp(1)
      ENDIF

!---------------------------------------------------------------------------
!!! read cloud factors:
!     tcloud - times, in seconds
!     xcloud - multipliers for effect of cloud on j values

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      ncloud = nval
      IF (ncloud .GT. 0) THEN
        IF(ncloud.GT.nenv) WRITE(6,*) ncloud,nenv, &
           'warning:  nenv < ncloud'
        DO i = 1, ncloud
          READ(inp,*,ERR=2000,END=2010) tcloud(i), xcloud(i)
        ENDDO
        tcloud = tcloud/tunit
      ENDIF

!---------------------------------------------------------------------------
! read PBL information and convert:
! xz = pbl heights (in:km, out:cm)
! tz = transition times (in:s, out:min)
! entrainment rate calculation done in subroutine NEWENV

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      nz = nval
      IF (nz .GT. 0) THEN
        IF(nz.GT.nenv) WRITE(6,*) nz,nenv, &
             'warning:  nenv < nz (pbl)'
        DO i = 1, nz
          READ(inp,*,ERR=2000,END=2010) tz(i), xz(i)
        ENDDO
        tz = tz/tunit
        xz = xz*zunit
      ENDIF

!---------------------------------------------------------------------------
!!! read dilution:
!    tdilut - times, in seconds : convert to minutes
!    xdilut - dilution rate constants, in cm/s : convert to cm/min

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      ndilut = nval
      IF (ndilut .GE. 1) THEN
        IF(ndilut.GT.nenv) WRITE(6,*) ndilut,nenv, &
             'warning:  too many dilution inputs'
        DO i = 1, ndilut
          READ(inp,*,ERR=2000,END=2010) tdilut(i), xdilut(i)
        ENDDO
        tdilut = tdilut/tunit
        xdilut = xdilut*tunit
      ENDIF

!---------------------------------------------------------------------------
! read emissions table:
! nsem = number of species
! emspec(nsem) = name of species
! ntem(nsem) = number of times for each species
! tem(nsem,ntem) = time (input seconds, converted to mins)
! xem(nsem,ntem) = emission rate, molec.cm-2.s-1 (convert to cm.min-1)

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

! Find array sizes, allocate arrays, reposition input file
      nsem = nval
      IF(nsem.GT.0) THEN
        DO i = 1, nsem
           READ(inp,'(a4,1x,i3)') emspec(i), ntem(i)
           DO j = 1, ntem(i)
             READ(inp,*,ERR=2000,END=2010)
           ENDDO
        ENDDO
        ntemx = MAXVAL(ntem)
        ALLOCATE(tem(ntemx+1,nsem),xem(ntemx+1,nsem))
        DO i = 1, nsem+SUM(ntem)
          BACKSPACE inp
        ENDDO
! Read input data into arrays
        DO j = 1, nsem
          READ(inp,'(a4,1x,i3)') emspec(i), ntem(i)
          PRINT*, emspec(j),ntem(j)
          DO i = 1, ntem(j)
            READ(inp,*) tem(i,j), xem(i,j)
          ENDDO
        ENDDO
        tem = tem/tunit
        xem = xem*u0
      ELSE
        ALLOCATE(tem(1,1),xem(1,1))
      ENDIF

!---------------------------------------------------------------------------
!!! read user-constrained species
!    nusr   - number of user-constrained species
!    ntusr - number of time vs. user-defined data pairs
!    tusr - times, in seconds, converted to minutes
!    xusr - concentration, in molecules/cm3

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

! Find array sizes, allocate arrays, reposition input file
      nusr = nval
      IF(nusr.GT.0) THEN
        DO j = 1, nusr
          READ(inp,'(a4,1x,i3)',ERR=2000,END=2010) usrspec(j),ntusr(j)
            DO i = 1, ntusr(j)
              READ(inp,*,ERR=2000,END=2010)
            ENDDO
        ENDDO
        ntusrx = MAXVAL(ntusr)
        ALLOCATE(tusr(ntusrx+1,nusr),xusr(ntusrx+1,nusr))
        DO i = 1, nusr+SUM(ntusr)
          BACKSPACE inp
        ENDDO
! Read input data into arrays
        DO j = 1, nusr
          READ(inp,*,ERR=2000,END=2010) usrspec(j),ntusr(j)
          IF(ntusr(j).EQ.1)THEN
            PRINT*, usrspec(j),ntusr(j),' datapoint'
          ELSE
            PRINT*, usrspec(j),ntusr(j),' datapoints'
          ENDIF
          DO i = 1, ntusr(j)
            READ(inp,*,ERR=2000,END=2010) tusr(i,j), xusr(i,j)
          ENDDO
        ENDDO
        tusr = tusr/tunit
        xusr = xusr/cunit
      ELSE
        ALLOCATE(tusr(1,1),xusr(1,1))
      ENDIF

!---------------------------------------------------------------------------
! INITIAL CONCENTRATIONS
!   NICS   - NUMBER OF INITIAL CONDITION SPECIES
!   ICSPEC - NAME OF SPECIES TO BE INTIALIZED
!   CI     - CORRESPONDING INITAL CONCENTRATION, MOLECULES/CM3

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      nics = nval
      DO I = 1, nics
        READ (inp,1110,ERR=2000,END=2010) icspec(i), ci(i)
        PRINT*, icspec(i)
      ENDDO
      ci = ci/cunit

!---------------------------------------------------------------------------
! BACKROUND CONCENTRATIONS FOR DILUTION
!   NBG    - NUMBER OF NON-ZERO BACKGROUND  SPECIES
!   BGSPEC - NAME OF SPECIES TO BE INTIALIZED
!   BGCONC - CORRESPONDING INITAL CONCENTRATION, MOLECULES/CM3

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      nbg = nval
      DO i = 1, nbg
        READ (inp,*,ERR=2000,END=2010) bgspec(i), bgconc(i)
        WRITE(6,*) bgspec(i),bgconc(i)
      ENDDO
      bgconc = bgconc/cunit

!----------------------------------------------------------------------------
! NRATES = total number of output rates requested
! IRATES(NR) = index of reactions requested

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      WRITE(6,*)nval,inpname

      nrates= nval
      IF (nrates .GT. 0) THEN
        READ(inp,'(15i5)')(irates(i), i=1,nrates)
!        WRITE(6,*) (irates(i), i=1,nrates)
      ENDIF

!---------------------------------------------------------------------------
! REACTION MECHANISM
! A(J) - RATE CONSTANT AT 298K, IN MOLEC,CM3,SEC UNITS
!        (except for deposition: cm.s-1)
!        convert to minute units 
! S(J) - ACTIVATION TEMPERATURE, IN KELVIN

      READ(inp,*,ERR=2000,END=2010) nval,inpname

      nr= nval
      IF (nr .GT. nrxn) THEN
          WRITE(6,*) &
             'STOPPING:  2nd declaration of # reactions ',&
             '> nrxn defined in  bounds_mod.f90'
          STOP
      ENDIF

      DO J = 1, NR
         READ(INP,1091,ERR=2000,END=2010) C2(J), &
         (MOLEC(J,I),I=1,3),(SC(J,I),MOLEC(J,I),I=4,7),A(J),S(J)
         R1 = MOLEC(J,1)
         R2 = MOLEC(J,2)
         R3 = MOLEC(J,3)
         M1 = 1
         M2 = 1
         M3 = 1
         IF(R1.EQ.'    '.OR.R1.EQ.'HV  '.OR.R1.EQ.'(M) ') M1 = 0
         IF(R2.EQ.'    '.OR.R2.EQ.'HV  '.OR.R2.EQ.'(M) ') M2 = 0
         IF(R3.EQ.'    '.OR.R3.EQ.'HV  '.OR.R3.EQ.'(M) ') M3 = 0
         MTOT = M1 + M2 + M3

         IF(MTOT.EQ.0)   A(J) = A(J)*u0
         IF(MTOT.EQ.1)   A(J) = A(J)*u1
         IF(MTOT.EQ.2)   A(J) = A(J)*u2 
         IF(MTOT.EQ.3)   A(J) = A(J)*u3
   
! write reaction to screen if rates will be saved
         DO k=1,nrates
           IF(j.EQ.irates(k))THEN
             WRITE(6,*)irates(k),": ", &
                (molec(j,i),i=1,3),'=> ', (molec(j,i),i=4,7)
           ENDIF
         ENDDO

      ENDDO 
      WRITE(6,*)" "
      WRITE(6,*)nval,inpname

!---------------------------------------------------------------------------
!  TROE REACTION DATA, MOLECULE, CM3, SEC UNITS
!  ITYPE4 - REACTION LIST INDEX OF TROE REACTION
!  AK0300 - ZERO PRESSURE 300K RATE CONSTANT, THIRD ORDER
!  AN     - TEMPERATURE EXPONENT FOR ZERO PRESSURE RATE CONSTANT
!  AKI300 - HIGH PRESSURE 300K RATE CONSTANT, SECOND ORDER
!  AM     - TEMPERATURE EXPONENT FOR HIGH PRESSURE RATE CONSTANT
!  BAS    - BASE OF EXPONENTIATION, 0.6 FOR MOST
!  AEQUIL - PRE-EXPONENTIAL OF EQUILIBRIUM CONSTANT
!  TEQUIL - ACTIVATION TEMPERATURE OF EQUILIBRIUM CONSTANT

      READ(inp,*,ERR=2000,END=2010) nval,inpname
      write(6,*)nval,inpname

      mr4= nval
      DO I = 1, MR4
         READ(INP,1080,ERR=2000,END=2010) ITYPE4(I),  &
              AK0300(I),AN(I),AKI300(I),AM(I),BAS(I), &
              AEQUIL(I),TEQUIL(I)
           IF (AEQUIL(I) .NE. 1.)  AEQUIL(I) = AEQUIL(I)*CUNIT 
      ENDDO
      AK0300 = AK0300*U3
      AKI300 = AKI300*U2

!---------------------------------------------------------------------------
! TABLE OF J VALUES
!      ITYPE1(I) - ARRAY OF REACTION LIST INDEX FOR PHOTOLYSIS REACTIONS
!      TJVAL - TIMES, IN SECONDS
!      XJVAL - J-VALUES, IN 1/SECONDS

      inpname ='Photolysis reactions'
      IF (NPHOT .GT. 0) THEN
         ALLOCATE(tjval(nphot+1),xjval(nphot+1,mr1),pslope(nphot,mr1))
         read(inp,*)
         READ(inp,*,ERR=2000,END=2010) mr1, mphot
         read(inp,*)
         READ(inp,*,ERR=2000,END=2010) (itype1(i),i=1,mr1)
         read(inp,*)
         READ(inp,*,ERR=2000,END=2010) (tjval(j), j=1, nphot)
         tjval = tjval/tunit
         DO i = 1, mr1
            read(inp,*)
            READ(inp,*,ERR=2000,END=2010) (xjval(j,i), j=1, nphot)
         ENDDO
         xjval = xjval*u1
      ELSE
         ALLOCATE(tjval(1),xjval(1,1),pslope(1,1))
      ENDIF
      write(6,*)mr1,inpname

      write(6,*)'************done read all'
      CLOSE (inp)

!_____________________________ END DATA READ _______________________________
!---------------------------------------------------------------------------
! Check that inputs cover time of run; 
! extend to end of run or 1 day, whichever is earlier.
! (If only one value, used for whole day irrespective of time)

      IF(nz.EQ.1)     z = xz(1)
      IF(nz.GT.1)     CALL USRINP(nz,tz,xz,1)
      IF(ntemp.EQ.1)  temp = xtemp(1)
      IF(ntemp.GT.1)  CALL USRINP(ntemp,ttemp,xtemp,2)
      IF(ncloud.EQ.1) fcloud = xcloud(1)
      IF(ncloud.GT.1) CALL USRINP(ncloud,tcloud,xcloud,3)
      IF(ndilut.EQ.1) dilut = xdilut(1)
      IF(ndilut.GT.1) CALL USRINP(ndilut,tdilut,xdilut,4)
      DO j=1,nsem
        IF(ntem(j).EQ.1) yem(j) = xem(1,j)
        IF(ntem(j).GT.1) CALL USRINP(ntem(j),tem(:,j),xem(:,j),6)
      ENDDO
      DO j=1,nusr
        IF(ntusr(j).GT.1) CALL USRINP(ntusr(j),tusr(:,j),xusr(:,j),7)
      ENDDO

!---------------------------------------------------------------------------
!  check reactions and verify assignements of types

!    ITYPE1(I) - photolysis - already assigned at read
!    ITYPE2(I) - simple Arrhenius
!    ITYPE3(I) - A + M or A + B + M, possibly also Arrhenius
!    ITYPE4(I) - Troe  -  already assigned at read
!    ITYPE5(I) - Special functions for rate constants 
!    ITYPE6(I) - Emission 'reactions'
!    ITYPE7(I) - Deposition 'reactions'

      nr1 = 0
      nr2 = 0
      nr3 = 0
      nr4 = 0
      nr5 = 0
      nr6 = 0
      nr7 = 0

      DO j = 1, nr
! Count photolysis reactions:
         IF (molec(j,2) .EQ. 'HV  ') THEN
            nr1 = nr1 + 1
            rtyp(j) = 1
         ENDIF

! Identify and count simple Arrhenius type:
         IF (s(j) .NE. 0.) THEN
            nr2 = nr2 + 1
            itype2(nr2) = j
            rtyp(j) = 2
         ENDIF
        
! Identify and count reactions with M as a reagent (not Troe):
         DO k = 1, 3
            IF (molec(j,k) .EQ. 'M   ') THEN
               nr3 = nr3 + 1
               itype3(nr3) = j
               rtyp(j) = 3
            ENDIF
         ENDDO

! Count Troe reactions
         IF ((molec(j,2).EQ.'(M) ').OR.(molec(j,3).EQ.'(M) ')) THEN
            nr4 = nr4 + 1
            rtyp(j) = 4
         ENDIF

! Identify and count reactions requiring special treatment:
!   these have a lower case 's' in column 2
         IF (c2(j) .EQ. 's') THEN
            nr5 = nr5 + 1
            itype5(nr5) = j
            rtyp(j) = 5
         ENDIF

! Emission reactions:
         IF (c2(j) .EQ. 'e') THEN
            nr6 = nr6 + 1
            itype6(nr6) = j
            rtyp(j) = 6
         ENDIF

! Deposition reactions:
         IF (c2(j) .EQ. 'd') THEN
            nr7 = nr7 + 1
            itype7(nr7) = j
            rtyp(j) = 7
         ENDIF

! Initalize rate constant to NTP value:
         r(j) = a(j)

! Convert blank stoichiometry coefficients to unity (except if species is
!   also blank).  Note: specific products cannot be "turned off" by
!   simply giving them zero stoichiometry coefficients.

         DO k = 1, 7
            IF (molec(j,k) .EQ. 'HV  ') molec(j,k) = iblank
            IF (molec(j,k) .EQ. 'M   ') molec(j,k) = iblank
            IF (molec(j,k) .EQ. '(M) ') molec(j,k) = iblank
            IF (sc(j,k) .EQ. 0.) sc(j,k)=1.
            IF (molec(j,k) .EQ. iblank) sc(j,k) = 0.
         ENDDO

       ENDDO ! j = 1,nr

!---------------------------------------------------------------------------
!  verify number of each type of reaction against input data

      IF (nr1 .NE. mr1) WRITE(6,*) nr1, mr1, &
           'warning:  incorrect number of photolysis reactions'
      IF (nr2 .NE. mr2) WRITE(6,*) nr2, mr2, &
           'warning:  incorrect number of Arrhenius reactions'
      IF (nr3 .NE. mr3) WRITE(6,*) nr3, mr3, &
           'warning:  incorrect number of M reactions'
      IF (nr4 .NE. mr4) WRITE(6,*) nr4, mr4, &
           'warning:  incorrect number of Troe reactions'
      IF (nr5 .NE. mr5) WRITE(6,*) nr5, mr5, &
           'warning:  incorrect number of special reactions'
!---------------------------------------------------------------------------
!   find correct index for special reactions:
!   1  -  HO2 + HO2 -> H2O2 + O2
!   2  -  HNO3 + HO  ->  H2O + NO3
!   3  -  CO + HO -> CO2 + H

      iwhich = 0  ! array initialisation
      DO i = 1, nr5

         IF((molec(itype5(i),1) .EQ. 'HO2 ') .AND. &
            (molec(itype5(i),2) .EQ. 'HO2 '))  iwhich(i) = 1

         IF((molec(itype5(i),1) .EQ. 'HNO3') .AND. &
            (molec(itype5(i),2) .EQ. 'HO  '))  iwhich(i) = 2

         IF((molec(itype5(i),1) .EQ. 'CO  ') .AND. &
            (molec(itype5(i),2) .EQ. 'HO  '))  iwhich(i) = 3

         IF(iwhich(i) .EQ. 0) WRITE(6,*) &
                'warning: could not identify special reaction'

      ENDDO

!---------------------------------------------------------------------------
! Establish reaction index matrix and set up sparse jacobian vectors:
      CALL MATRX (ns)
      !IF(ns.NE.nspc)THEN
      !  PRINT*,'CAUTION! ns <> nspc',ns,nspc
      !ENDIF
      CALL SPARS (ia,ja,ns)

!---------------------------------------------------------------------------
! Check for N2, O2 inputs (require both, even if = zero)
      ii = 0
      jj = 0
      IF(nusr.GT.0)THEN
          ii=COUNT(usrspec.EQ.'N2  ')
          jj=COUNT(usrspec.EQ.'O2  ')
      ENDIF
      IF(nics.GT.0)THEN
          ii=ii+COUNT(icspec.EQ.'N2  ')
          jj=jj+COUNT(icspec.EQ.'O2  ')
      ENDIF

      IF(ii.EQ.0) THEN 
         PRINT*,'!! N2 CONSTRAINT OR INITIALIZATION REQUIRED !!'
         STOP
      ENDIF
      IF(jj.EQ.0) THEN 
         PRINT*,'!! O2 CONSTRAINT OR INITIALIZATION REQUIRED !!'
         STOP
      ENDIF

!---------------------------------------------------------------------------
! Check that emitted species appear in the mechanism
      DO i = 1, nsem     ! (will not loop if nsem = 0)
        j = COUNT(specis.EQ.emspec(i))
        IF(j.EQ.0)THEN
          PRINT*,"!!ERROR!! "
          PRINT*,"Emitted species ",emspec(i),  &
                " is not in reaction list & will not be considered." 
          PRINT*,"Is this intentional? (y/anykey)"
          READ(*,'(a1)') response
          IF(response.EQ."y".OR.response.EQ."Y")THEN
            CONTINUE
          ELSE 
            PRINT*,">> Add species to file input.species ", &
                   "and start again at Step 1."
            STOP
          ENDIF
        ENDIF
      ENDDO

! Check that background species appear in the mechanism
      DO i = 1, nbg     ! (will not loop if nbg = 0)
        j = COUNT(specis.EQ.bgspec(i))
        IF(j.EQ.0)THEN
          PRINT*,"!!ERROR!! "
          PRINT*,"Background species ",bgspec(i),  &
                " is not in reaction list & will not be considered." 
          PRINT*,"Is this intentional? (y/anykey)"
          READ(*,'(a1)') response
          IF(response.EQ."y".OR.response.EQ."Y")THEN
            CONTINUE
          ELSE 
            PRINT*,">> Add species to file input.species ", &
                   "and start again at Step 1."
            STOP
          ENDIF
        ENDIF
      ENDDO

!---------------------------------------------------------------------------
! Identify 'ballast' gases (those that do not appear in the mechanism 
! but contribute to concentration of 'M', the atmospheric density)
! cmblsti = the sum of initial concs of ballast species.
! cmblstv = the sum of initial concs of time-varying ballast species.

      cmblsti = 0.
      cmblstv = 0.

! time-varying user-defined species 
      DO i = 1, nusr
        j = COUNT(specis.EQ.usrspec(i))
        IF(j.EQ.0.)THEN
! multiple times specified
          IF(ntusr(i).GT.1)THEN
            PRINT*,"found time-constrained ballast species: ",usrspec(i)
            cmblstv = cmblstv + xusr(1,i)
          ELSE
! only initial time specified 
            PRINT*,"found initialised ballast species: ",usrspec(i)
            cmblsti = cmblsti + xusr(1,i)
          ENDIF
        ENDIF
      ENDDO

! initially-constrained species (but only if not identified above)
      DO i = 1, nics
        j = COUNT(specis.EQ.icspec(i))
        !PRINT*,"initial species ",i,j
        IF(j.EQ.0)THEN
          ii=COUNT(usrspec.EQ.icspec(i))
          IF(ii.EQ.0)THEN
            PRINT*,"found initialised ballast species: ",icspec(i)
            cmblsti = cmblsti + ci(i)
          ENDIF
        ENDIF
      ENDDO
!---------------------------------------------------------------------------
! Initialize solver concentrations and identify indices for special species:

! array initialisations
      ALLOCATE(ispc(nfspc,ns),c(ns),cbg(ns),STAT=ERR)
      ispc = 0    
      c = 0.
      cbg = 0.
! initialisations for water only (see RATCO5)
      iH2O = 0    
      cH2O = 0.

      DO j = 1, ns
! Default concentrations and family indices initialized from 
! initial concentration file or previous run's output, 
! and [M] as sum of solver species initial concentrations = cmsolv
!  - assumes that overriding constrained concentrations (below)
!    make negligible contribution to initial [M]

        IF(nics.GT.0)THEN
          DO i = 1, nics
            IF(specis(j).EQ.icspec(i))  THEN
              c(j) = ci(i)
            ENDIF
            IF(icspec(i).EQ.'NOX '.OR.icspec(i).EQ.'NOx ') THEN
              IF(specis(j).EQ.'NO  ') ispc(1,i)=j
              IF(specis(j).EQ.'NO2 ') ispc(2,i)=j
            ENDIF
            IF(icspec(i).EQ.'HOX '.OR.icspec(i).EQ.'HOx ') THEN
              IF(specis(j).EQ.'HO  ') ispc(1,i)=j
              IF(specis(j).EQ.'HO2 ') ispc(2,i)=j
              IF(specis(j).EQ.'H   ') ispc(3,i)=j
            ENDIF
          ENDDO
        ENDIF

! Constrained concentrations and indices of constrained families.
! OVERRIDE initial concs:
        IF(nusr.GT.0)THEN
          DO i = 1,nusr
            IF(specis(j).EQ.usrspec(i)) THEN
              ispc(1,i) = j
              c(j) = xusr(1,i)
            ENDIF
            IF(usrspec(i).EQ.'NOX '.OR.usrspec(i).EQ.'NOx ') THEN
              IF(specis(j).EQ.'NO  ') ispc(1,i)=j
              IF(specis(j).EQ.'NO2 ') ispc(2,i)=j
            ENDIF
            IF(usrspec(i).EQ.'HOX '.OR.usrspec(i).EQ.'HOx ') THEN
              IF(specis(j).EQ.'HO  ') ispc(1,i)=j
              IF(specis(j).EQ.'HO2 ') ispc(2,i)=j
              IF(specis(j).EQ.'H   ') ispc(3,i)=j
            ENDIF
          ENDDO
        ENDIF


! Background concentrations (for dilution):
! N2, O2 defaults equal boundary layer conditions
        IF(specis(j) .EQ.'N2  '.OR.specis(j).EQ.'O2  ') cbg(j)=c(j)
        IF(nbg.GT.0)THEN
          DO i = 1, nbg
            IF(specis(j) .EQ. bgspec(i)) THEN
              cbg(j) = bgconc(i)
              EXIT
            ENDIF
          ENDDO
        ENDIF
! Indices of emitted species:
        IF(nsem.GT.0)THEN
          DO i = 1, nsem
            IF(specis(j) .EQ. emspec(i)) THEN
               iemit(i) = j
               EXIT
            ENDIF
          ENDDO
        ENDIF

! H2O index & concentration:
        IF(specis(j).EQ.'H2O ') THEN
          iH2O = j
          cH2O  = c(iH2O)
        ENDIF

      ENDDO   ! j

! cmsolv = the sum of initial concentrations of solver species.
! dens = the sum of all initial concentrations = 'M' (To be updated in NEWENV)
      cmsolv = 0.
      dens = 0.

      cmsolv = SUM(c)
      dens = cmsolv + cmblsti + cmblstv

      !PRINT*,dens, cmsolv , cmblsti , cmblstv

!---------------------------------------------------------------------------
! Precalculate slopes for J values:

      CALL SLOPES

! Compute inital values of rate constants for each type of reaction:

      T = 0.
      IF (nr1 .GT. 0)       CALL RATCO1
      IF (nr2 .GT. 0)       CALL RATCO2
      IF (nr3 .GT. 0)       CALL RATCO3
      IF (nr4 .GT. 0)       CALL RATCO4
      IF (nr5 .GT. 0)       CALL RATCO5
      IF (nr6 .GT. 0)       CALL RATCO6
      IF (nr7 .GT. 0)       CALL RATCO7


!---------------------------------------------------------------------------
! Error control:
! Specification of the error bound 
      eps=1.E-3

! Specification of machine roundoff, chem vax -- drw:
!      uround = 5.96e-8
! Specification of machine roundoff, pyramid 90x -- kds:
      uround = 1.193E-7

! Inital guess for time step:
      h0=1.E-10

! Maximum time step, in correct units (sec or min), use 20 min or 1200 s
      hmax0 = 1200./tunit

!---------------------------------------------------------------------------
! Use the gear routine:

      WRITE(6,*)'Beginning simulation ...'
      CALL DRIVES (ns,c,ia,ja) 

!---------------------------------------------------------------------------
! Output results and finish neatly:
! a) timeseries concentrations
      CALL OUTPT (ns)

! b) final rates, timeseries rates if requested:
      DO j = 1, nr
        WRITE(iout2,'(i4,1x,3(a4,1x),e10.3)') j,(molec(j,i),i=1,3),r(j)
      ENDDO
      IF(nrates .NE. 0) CALL OUTRAT
      WRITE(iout2,*)'END OF RATES'

! c) final concentrations, for initialisation of subsequent run:
      WRITE(iout3,*) ns
      print*,'nt = ',nt
      DO i=1,ns
        WRITE(iout3,'(a4,1x,e10.3)') specis(i),savcon(nt,i)*cunit
      ENDDO
 
      GOTO 2050

!---------------------------------------------------------------------------
!      Handle Read Errors and Unexpected EOFs gracefully:
!---------------------------------------------------------------------------
2000  WRITE(6,*) 'Error reading input files : ',inpname
      STOP

2010  WRITE(6,*) 'Unexpected End-Of-File in Input File: ',finame
      STOP
 
!---------------------------------------------------------------------------
!     Format Statements:
!---------------------------------------------------------------------------
!1010  FORMAT (A20)
!1015  FORMAT ('Reading Input from ',A20)
!1020  FORMAT (6(I5,1X))
!1025  FORMAT (8(I5,1X))
!1030  FORMAT (4(E10.3,1X))
!1040  FORMAT (12I10)
!1050  FORMAT (7E10.3)
!1060  FORMAT (2E15.5)
!1070  FORMAT ('RESERVED FOR FUTURE USE')
1080  FORMAT (I5,1X,2(E10.3,1X,F5.2,1X),F5.2,2(1X,E10.3))
!1090  FORMAT (1X,A1,3(A4,1X),2X, &
!                   3(F5.2,1X,A4,1X),F5.2,1X,A4,E8.1,1X,E8.1)
1091  FORMAT (1X,A1,3(A4,1X),2X, &
                   3(F5.2,1X,A4,1X),F5.2,1X,A4,E9.2,1X,E9.2)
!1100  FORMAT (I5)
1110  FORMAT (A4,1X,E10.3)

2050  DEALLOCATE(pslope,tjval,xjval,tusr,xusr,tem,xem)
      DEALLOCATE(ispc,c,cbg)

      END PROGRAM CHEM

!*****************************************************************************
      SUBROUTINE DIFFUN (ns,c,ct)
!-----------------------------------------------------------------------
!     THIS SUBROUTINE CALCULATES THE DERIVATIVE VECTOR OF THE ODE S
!     ON Entry:
!       N = NUMBER OF SPECIES  
!       T = CURRENT TIME
!       C = CURRENT CONCENTRATION
!       R = RATE CONSTANTS, INCLUDING PHOTOLYSIS RATES BY CALL TO DAILY
!       DILUT = DILUTION RATE FOR ALL SPECIES
!       KR = REACTION SPECIES INDEX
!       NREAG = ORDER OF REACTION
!     ON Exit:
!       CT = d[C]/dt TIME DERIVATIVE FOR EACH SPECIES
!-----------------------------------------------------------------------
      USE PARAMS
      USE MECH
      USE KINET
      USE DILUTE
      USE RATES
      USE EMIT

      USE PBL

      IMPLICIT NONE
      INTEGER,INTENT(IN) :: ns
      REAL,DIMENSION(ns),INTENT(INOUT) ::  ct, c

      INTEGER :: ir,k,l
      REAL ::  rt

      !write(6,*)'diffun, time = ',t/(24.*60.), t*60., z

! adjust concentrations to external constraints, if any
! compute time-dependent rate constants for new environment 
! (time dependent light, temp., dilution, water vapor, etc.)
      
      CALL NEWCHM(t,ns,c)
      CALL NEWENV(t)
      
      ct = 0.

! Environmental influences on dc/dt:
      DO i = 1, ns
! ... emissions rates (min-1):
        DO j = 1, nsem
          IF(i .EQ. iemit(j)) THEN
             ct(i) = ct(i) + kem(j)
          ENDIF
        ENDDO
      ENDDO

! ... dilution rate (min-1):
      ct  = ct - kdilut * (c - cbg)
! ... entrainment rate (min-1):
      ct  = ct - kentrn * (c - cbg)

! reaction rates:
      DO ir=1,nr
        rt=r(ir)
! multiply the rate constant by the reagent concentrations
        DO l=1,nreag(ir)
          i = kr(ir,l)
          rt=rt*c(i)
        ENDDO
! store rates for possible output
        rts(ir) = rt

! contribution to d[conc]/dt from reagent sides
         DO l=1,nreag(ir)
             i = kr(ir,l)
             ct(i)=ct(i)-rt
         ENDDO
! contribution to d[conc]/dt from product sides
          DO k=4,3+nprod(ir)
             i=kr(ir,k)
             ct(i)=ct(i)+sc(ir,k)*rt
          ENDDO
      ENDDO
      
      END SUBROUTINE DIFFUN

!*****************************************************************************
      SUBROUTINE OUTPT (ns)
!---------------------------------------------------------------------------
!  This subroutine produces the output of a simulation.  The number of
!  saved times is printed on the first line, followed by a list of the
!  species names ending with "END", and finally a table of time -vs-
!  concentrations (see output algorithm below.
!  Note: IOUT is the output unit number specified in the main program
!        CHEM.  Its value is passed in through the COMMON block below.
!---------------------------------------------------------------------------
        USE PARAMS
        USE SAVED
        USE IOFIL
        USE UNITS
        USE MECH
        IMPLICIT NONE

        INTEGER :: k,ns,nblock,nlast,nc
        CHARACTER(7) :: aname

! write out total number of save times, species
        WRITE(iout,*)nt, ns
        WRITE(6,*)nt,'output timesteps generated'

!  write out species names for chemical proccessing
 
        aname(1:1) = '('
        DO i = 1, ns
           aname(2:) = specis(i)
           DO  j = 3, 6
              IF(aname(j:j).EQ.' ') THEN
                 aname(j:j) = ')'
                 EXIT
              ENDIF
           ENDDO
           WRITE(iout,'(a7)')aname
        ENDDO
        WRITE(iout,'(a3)') 'END'
        
!  (each block is TIME and up to 5 DATA columns) - then read values
!  NBLOCK is number of data blocks
!  NLAST is number of columns in last block

        nblock = 1 + (ns-1)/5
        nlast = ns - 5*(nblock-1)
        nc = 5
        DO j = 1, nblock
           IF(j.EQ.nblock) nc = nlast
           DO i = 1, nt
              WRITE(iout,1020)  &
             tunit* savtim(i),(cunit*savcon(i,5*(j-1)+k),k=1,nc)
           ENDDO
        ENDDO
        WRITE(iout,*)'END DATA'

!       Format Statements.
1020    FORMAT(6(e10.4,'  '))
!1025    FORMAT(6(e14.8,'  '))

        END SUBROUTINE  OUTPT

!*****************************************************************************
        SUBROUTINE OUTRAT

! this subroutine prints selected reaction rates as a function of time
! it is controlled by the value of NRATES

        USE PARAMS
        USE RATES
        USE SAVED
        USE IOFIL
        USE UNITS
        IMPLICIT NONE

! write out total number of save times, rates (convert units to molec/cc-s) 

        WRITE(iout2,*)nt,nrates

        DO j = 1, nrates
        WRITE(iout2,*) irates(j)
           DO i = 1, nt
              WRITE(iout2,1000)savtim(i)*tunit, savrat(i,j)/u0
           ENDDO
        ENDDO

1000    FORMAT(1pe11.4,2x,1pe11.4)

        END SUBROUTINE OUTRAT

!****************************************************************************
      SUBROUTINE COSET(METH,NQ,EL,TQ)
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: METH, NQ
      REAL,INTENT(OUT) :: EL(13), TQ(4)

      INTEGER :: K
      REAL, DIMENSION(12,2,3) :: PERTST = &
       RESHAPE ((/1.,1.,2.,1.,.3158,.07407,.01391,.002182,.0002945, &
       .00003492,.000003692,.0000003524,1.,1.,.5,.1667,.04167,1.,   &
       1.,1.,1.,1.,1.,1.,2.,12.,24.,37.89,53.33,70.08,87.97,106.9,  &
       126.7,147.4,168.8,191.0,2.0,4.5,7.333,10.42,13.7,1.,1.,1.,   &
       1.,1.,1.,1.,12.0,24.0,37.89,53.33,70.08,87.97,106.9,126.7,   &
       147.4,168.8,191.0,1.,3.0,6.0,9.167,12.5,1.,1.,1.,1.,1.,1.,   &
       1.,1./), (/12,2,3/))

      !write(6,*)'            coset.f'
      DO K = 1, 3
         TQ(K) = PERTST(NQ,METH,K)
      ENDDO
      TQ(4) = .5*TQ(2)/FLOAT(NQ+2)
  
      EL(2) = 1.0

      IF (NQ.EQ.1) THEN
         EL(1) = 1.0

      ELSEIF (NQ.EQ.2) THEN
         EL(1) = 6.6666666666667E-01
         EL(3) = 3.3333333333333E-01

      ELSEIF (NQ.EQ.3) THEN
         EL(1) = 5.4545454545455E-01
         EL(3) = EL(1)
         EL(4) = 9.0909090909091E-02

      ELSEIF (NQ.EQ.4) THEN
         EL(1) = 0.48
         EL(3) = 0.7
         EL(4) = 0.2
         EL(5) = 0.02

      ELSEIF (NQ.EQ.5) THEN
         EL(1) = 4.3795620437956E-01
         EL(3) = 8.2116788321168E-01
         EL(4) = 3.1021897810219E-01
         EL(5) = 5.4744525547445E-02
         EL(6) = 3.6496350364964E-03
      ENDIF

      END SUBROUTINE COSET

!****************************************************************************
      SUBROUTINE DRIVES (NS,Y0,IA,JA)

      USE PARAMS
      USE CONTRL
      USE KINET
      USE GEARS
      USE IOFIL
      IMPLICIT NONE

      INTEGER,INTENT(IN) :: ns
      INTEGER,INTENT(IN) :: ia(nrxn), ja(nrx1)
      !REAL,INTENT(IN) :: y0(nspc)
      REAL,INTENT(IN) :: y0(ns)

      INTEGER :: ier, maxl, maxpl, maxpu
      INTEGER :: np1, nza, nzl, nzu
      INTEGER :: lenw2=nrx5, leniw2=nrx5
      INTEGER :: iw1(nrx4,9) 
      REAL :: tst, yreal(ns,13)

!      print*,'drives'

      IF (EPS.LE.0.) THEN
         WRITE (lout,*) ' ILLEGAL INPUT.. eps .LE. 0.', eps
         STOP
      ENDIF
      IF (NS.LE.0) THEN
         WRITE (lout,*) ' ILLEGAL INPUT.. ns .LE. 0.'
         STOP
      ENDIF
      IF ((START-STOPP)*H0.GE.0.) THEN
         WRITE (lout,*) ' ILLEGAL INPUT.. (start-stopp)*h0 .GE. 0.'
         STOP
      ENDIF
      NP1 = NS + 1
      NZA = IA(NP1) - 1
      MAXL = LENIW2/2
      IPTI2 = MAXL + 1

! as MM_2.1
      !CALL SORDER (NS,IA,JA,IW1,IW1(1,5),MAXL,IW2,IW2(IPTI2),IER)
! as V2.4.06
      CALL SORDER(NS,IA,JA,IW1(:,1),IW1(:,5),MAXL,IW2, &
                  IW2(IPTI2:NRX5),IER)

      IPTI2 = NZA + 1
      IF (IPTI2+NZA-1.GT.LENIW2) THEN
         WRITE (LOUT,*) ' INSUFFICIENT WORKING STORAGE IN IW2 OR W2'
         STOP
      ENDIF
      DO I = 1, NP1
         IW1(I,2) = IA(I)
      ENDDO
      DO I = 1, NZA
         IW2(I) = JA(I)
      ENDDO

! as MM_2.1
!      CALL NSCORD (NS,IW1(1,2),IW2,IW1(1,3),IW2(IPTI2),IW1,IW1(1,5), &
!                   IW1(1,8))
! as V2.4.06
      CALL NSCORD(NS,IW1(:,2),IW2,IW1(:,3),IW2(IPTI2:NRX5), &
                 IW1(:,1),IW1(:,5),IW1(:,8))

      MAXPL = (LENIW2-NZA)/2
      IPTI3 = IPTI2 + MAXPL
      MAXPU = LENIW2 - IPTI3 + 1

! as MM_2.1
!      CALL NSSFAC (NS,IW1(1,2),IW2,MAXPL,IW1(1,3),IW2(IPTI2),IW1(1,4), &
!                   MAXPU,IW1(1,5),IW2(IPTI3),IW1(1,6),YREAL(1,6), &
!                   IW1(1,9),YREAL,YREAL(1,2),YREAL(1,3),IW1(1,7), &
!                   IW1(1,8),YREAL(1,4),YREAL(1,5),IER)
! as MM_2.1, with dimension "1" changed to ":"
      CALL NSSFAC (NS,IW1(1,2),IW2,MAXPL,IW1(1,3),IW2(IPTI2),IW1(1,4), &
                   MAXPU,IW1(1,5),IW2(IPTI3),IW1(1,6),YREAL(1,6), &
                   IW1(1,9),YREAL,YREAL(1,2),YREAL(1,3),IW1(1,7), &
                   IW1(1,8),YREAL(1,4),YREAL(1,5),IER)
! as V2.4.06
!      CALL NSSFAC(NS,IW1(:,2),IW2,MAXPL,IW1(:,3),IW2(IPTI2:NRX5),  &
!                 IW1(:,4),MAXPU,IW1(:,5),IW2(IPTI3:NRX5),IW1(:,6),&
!                 IW1(:,9),IW1(:,7),IW1(:,8),IER)

      NZL = IW1(NS+1,3)
      NZU = IW1(NS+1,5)
      IPTR2 = NZA + 1
      IPTR3 = IPTR2 + MAX0(NZA,NZL)
      IF (IPTR3+MAX0(NZA,NZU)-1.GT.LENW2) THEN
         WRITE (LOUT,*) ' INSUFFICIENT WORKING STORAGE IN IW2 OR W2'
         STOP
      ENDIF
      DO I = 1, NS
         YMAX(I) = ABS(Y0(I))
         IF (YMAX(I).EQ.0.) YMAX(I) = 1.E-10
         YREAL(I,1) = Y0(I)
      ENDDO
      T = START
      H = H0
      NZRO = 0
      TST = EPS*1.E-10
      DO I = 1, NS
         IF (YREAL(I,1).GT.TST) NZRO = NZRO + 1
      ENDDO
      NZRO = MAX0(NZRO,1)
      HMIN = ABS(H0)
      HMAX = ABS(START-STOPP)*10.
      HMAX = AMIN1(HMAX,HMAX0)
      EPSJ = SQRT(UROUND)

      CALL STIFFS(YREAL,NS,IA,JA,IW1)

      END SUBROUTINE DRIVES

!*****************************************************************************
      SUBROUTINE LOCAL(t,nx,tx,x,y,slope,iflag)

!  This subroutine interpolates tabular time values to current time:
!  Usually used for user-defined forcings.
!  input:  
!     T - desired time
!     NX - number of data pairs
!     TX - user input times
!     X  - user input values
!     iflag - indicator for debugging. Corresponds to cflag(iflag)
!  output
!     Y - value of X at time t
!     slope - dy/dx at time t
!  internal
!     TT - time of day

      USE PARAMS
      USE UNITS
      USE CONTRL ! for testing
      USE FLAGS
      IMPLICIT NONE
      INTEGER :: iflag, nx
      REAL,DIMENSION(nx+1) :: tx,x
      REAL :: y,t,tt,dt,slope

! cycle input if < 1 day supplied, else use as is
      IF(tx(nx).LT.tday.AND.t.GT.tday) THEN 
        tt = t - tday*FLOAT(INT(t/tday))
      ELSE            
        tt = t
      ENDIF

! interpolate:
! if only initialisation supplied, i.e.  nx < 2, loop will not execute 
! and y is not affected => solver concentrations float free,
! environmental params stay fixed.

      DO i = 1, nx-1
        IF(tx(i+1) .GE. tt) THEN
          dt = tt - tx(i)
          slope = (x(i+1)-x(i)) / (tx(i+1)-tx(i))
          y = x(i) + slope*dt
          RETURN
        ENDIF
      ENDDO

      END SUBROUTINE LOCAL

!*******************************************************************
      SUBROUTINE MATRX(NS)

!     MATRX CREATES A NR X 7 MATRIX OF THE REACTION SCHEME WHERE EACH
!     ROW REPRESENTS A  REACTION.  THE FIRST THREE COLUMNS ARE REACTANTS
!     AND THE LAST FOUR ARE THE PRODUCTS.  THE ELEMENTS CORRESPOND
!     TO THE INDIVIDUAL SPECIES AND WILL BE USED AS SUBSCRIPTS.

      USE PARAMS
      USE MECH
      IMPLICIT NONE
      INTEGER,INTENT(OUT) :: NS

      INTEGER ::  L, NC
      REAL :: STEMP(7)
      CHARACTER(4), DIMENSION(7) :: MTEMP

      !write(6,*)'    matrx.f'

      NS = 0
!  initialize KR(I,J)
      KR = 0

      DO I = 1, NR

!     IF LESS THAN THREE REACTANTS, FILL FIRST SLOTS.
         NC = 0
         DO J = 1, 3
            IF (MOLEC(I,J).NE.IBLANK) THEN
               NC = NC + 1
               STEMP(NC) = SC(I,J)
               MTEMP(NC) = MOLEC(I,J)
               SC(I,J) = 0.
               MOLEC(I,J) = IBLANK
            ENDIF
         ENDDO
         IF (NC.NE.0) THEN
            DO J = 1, NC
               SC(I,J) = STEMP(J)
               MOLEC(I,J) = MTEMP(J)
            ENDDO
         ENDIF
         NREAG(I) = NC
!     if less than four products, fill first slots.

         NC = 0
         DO J = 4, 7
            IF (MOLEC(I,J).NE.IBLANK) THEN
               NC = NC + 1
               STEMP(3+NC) = SC(I,J)
               MTEMP(3+NC) = MOLEC(I,J)
               SC(I,J) = 0.
               MOLEC(I,J) = IBLANK
            ENDIF
         ENDDO
         DO J = 4, 3 + NC
            SC(I,J) = STEMP(J)
            MOLEC(I,J) = MTEMP(J)
         ENDDO
         NPROD(I) = NC

!     process reactants here

         DO J = 1, 7
            IF (MOLEC(I,J).NE.IBLANK) THEN
               DO L = 1, NS
                  IF (MOLEC(I,J).EQ.SPECIS(L)) THEN
                     KR(I,J) = L
                     GOTO 50
                  ENDIF
               ENDDO
               NS = NS + 1
               SPECIS(NS) = MOLEC(I,J)
               KR(I,J) = NS
            ENDIF
 50      ENDDO
      ENDDO

      END SUBROUTINE MATRX

!*******************************************************************
      SUBROUTINE NEWCHM(t,ns,c)

! This subroutine imposes changes on concentrations, e.g.,
!   fixed NOx, HOx, experimentally determined ozone,  etc.
! On input:
!   ntusr(nusr) = number of tspecies,xspecies data pairs
!   If ntusr(nusr) = 0, conc. not constrained 
!   If ntusr(nusr) = 1, conc. initialised but not further constrained 
!   If ntusr(nusr) > 1, impose tabular conc. interpolated to local time
!   To get fixed concentration, must specify equal initial AND FINAL concs.
!  cusr = current concentration value of user-constrained species
!  ispc = species index 
!  tusr = time for constrained datum
!  xusr = species conc datum
! On output:
!  C(iNO), and C(iNO2) are changed so that they add up to cNOx
!  C(iHO), C(iHO2), C(iH) are changed so that they add up to cHOx
      
      USE USERSPC
      USE MECH
      USE DENSIT
      USE SPCIAL
      IMPLICIT NONE
   
      INTEGER :: ns
      REAL :: t,cNO,cNO2,cHO,cHO2,cH,dum,facfam
      REAL,DIMENSION(ns):: c

      cmblstv = 0.

! if want to constrain NOx, adjust NO and NO2 concentrations:
! peg other user-specifieds to input table of values

      facfam = 0.

      DO j=1,nusr
        IF (ntusr(j).GT.1) THEN
          CALL LOCAL(t,ntusr(j),tusr(:,j),xusr(:,j),cusr(j),dum,7)

! if usrspec is a ballast species, construct time-dependent M
          i = COUNT(specis.EQ.usrspec(j))
          IF(i.EQ.0)THEN
!            PRINT*,"ballast species found: ",usrspec(j)
            cmblstv = cmblstv + cusr(j)
          ELSE
! time-dependent concs of mechanism species 
            DO i=1,ns
              IF(specis(i).EQ.usrspec(j)) THEN
                c(i) = cusr(j)
                EXIT
              ENDIF
            ENDDO
          ENDIF

! NOx family
          IF (usrspec(j).EQ.'NOX '.OR.usrspec(j).EQ.'NOx ') THEN
            cNO  = c(ispc(1,j) )
            cNO2 = c(ispc(2,j) )
            IF(cNO+cNO2.NE.0) THEN
              facfam=cusr(j)/(cNO+cNO2)
              c(ispc(1,j))=cNO*facfam
              c(ispc(2,j))=cNO2*facfam
            ENDIF
          ENDIF
! HOx family
          IF (usrspec(j).EQ.'HOX '.OR.usrspec(j).EQ.'HOx ') THEN
            cHO  = c(ispc(1,j) )
            cHO2 = c(ispc(2,j) )
            cH   = c(ispc(3,j) )
            IF(cHO+cHO2+cH.NE.0) THEN
              facfam=cusr(j)/(cHO+cHO2+cH)
              c(ispc(1,j))=cHO*facfam
              c(ispc(2,j))=cHO2*facfam
              c(ispc(3,j))=cH*facfam
            ENDIF
          ENDIF
        ENDIF
      ENDDO

      cH2O=c(iH2O)

!  compute total number of molecules: floats with sum of all concs.

      cmsolv = SUM(c)
      dens = cmsolv + cmblsti + cmblstv
      !PRINT*,dens, cmsolv , cmblsti , cmblstv

      END SUBROUTINE NEWCHM

!*******************************************************************
      SUBROUTINE NEWENV(t)
!----------------------------------------------------------------------
!     THIS ROUTINE CALCULATES CURRENT VALUES OF TOTAL CONCENTRATION,
!     TEMPERATURE, AND DILUTION, AND THEN CALLS SUBROUTINES WHICH 
!     COMPUTE THE RATE CONSTANTS FOR DIFFERENT REACTON TYPES
!----------------------------------------------------------------------
      USE PARAMS
      USE JVALU
      USE CLOUD
      USE ARRHEN
      USE TROE
      USE TEMPER
      USE DILUTE
      USE PBL
      USE EMIT
      USE DEPOS
      USE UNITS

      IMPLICIT NONE
      REAL :: dum,t

! Find PBL height (cm):
      IF (nz .GT. 1)     CALL LOCAL(t,nz,tz,xz,z,dum,1)

! current temperature
      IF (ntemp .GT. 1 ) CALL LOCAL(t,ntemp,ttemp,xtemp,temp,dum,2)

! current cloud correction to photolysis rates:
      IF (ncloud .GT. 1) CALL LOCAL(t,ncloud,tcloud,xcloud,fcloud,dum,3)

! current dilution rate (cm min-1, convert to min-1):
      kdilut = 0.
      IF (ndilut .GT. 1 ) CALL LOCAL(t,ndilut,tdilut,xdilut,dilut,dum,4)
      IF (dilut.GT.0.) kdilut = dilut /z

! mixing with free troposphere during entrainment (cm min-1, convert to min-1):
      kentrn = 0.
      IF (nz .GT. 1) CALL LOCAL(t,nz,tz,xz,dum,entrn,5)
      IF (entrn.GT.0.) kentrn = entrn/z

! emissions rates:
      DO j = 1,nsem
        IF(ntem(j).GT.1) &
                  CALL LOCAL(t,ntem(j),tem(:,j),xem(:,j),yem(j),dum,6)
      ENDDO
      kem = yem/z

! Call rate constant calculation by type
!  call only if changing with time:
!  1 - call if photolysis changes with time
!  2-5 - call if temperature changes with time
!  6 - call if emissions present
!  7 - call if deposition present

        IF (NPHOT.GT.1) CALL RATCO1
        IF (NTEMP.GT.1) THEN
                        CALL RATCO2
                        CALL RATCO3
                        CALL RATCO4
                        CALL RATCO5
        ENDIF
        IF (NR6.GT.0) CALL RATCO6
        IF (NR7.GT.0) CALL RATCO7

        END SUBROUTINE NEWENV

!***********************************************************************
      SUBROUTINE NSBSLV(N,R,IC,IL,JL,ISL,LREAL,D,IU,JU,ISU,U,X,B,WY)
      USE PARAMS
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N
      INTEGER,DIMENSION(NRX4),INTENT(IN) :: R, IC, IL, ISL, IU, ISU
      INTEGER,DIMENSION(NRX5),INTENT(IN) :: JL, JU
      REAL,DIMENSION(NRXN),INTENT(IN) :: B, D
      REAL,DIMENSION(NRX5),INTENT(IN) :: LREAL, U
      REAL,DIMENSION(NRXN),INTENT(INOUT) :: X
      REAL,DIMENSION(NRX5),INTENT(INOUT) :: WY

      INTEGER :: K, JMIN, JMAX, JLJ,JUJ, ISLB, ISUB
      REAL :: TOT, YK

      !write(6,*)'            nsbslv.f'
      DO K = 1, N
         WY(K) = B(R(K))         !real
      ENDDO
      DO K = 1, N
         JMIN = IL(K)          !int
         JMAX = IL(K+1) - 1    !int
         YK = -D(K)*WY(K)      !real
         WY(K) = -YK           !real
         IF (JMIN.LE.JMAX) THEN
            ISLB = ISL(K) - 1  !int
            DO J = JMIN, JMAX
               ISLB = ISLB + 1 !int
               JLJ = JL(ISLB)  !int
               WY(JLJ) = WY(JLJ) + YK*LREAL(J)  !real
            ENDDO
         ENDIF
      ENDDO

      DO K = N, 1, -1
         TOT = -WY(K)          !real
         JMIN = IU(K)          !int
         JMAX = IU(K+1) - 1    !int
         IF (JMIN.LE.JMAX) THEN
            ISUB = ISU(K) - 1  !int
            DO J = JMIN, JMAX
               ISUB = ISUB + 1 !int
               JUJ = JU(ISUB)  !int
               TOT = TOT + U(J)*WY(JUJ) !real
            ENDDO
         ENDIF
         WY(K) = -TOT          !real
         X(IC(K)) = -TOT       !real
      ENDDO

      END SUBROUTINE NSBSLV

!***********************************************************************
      !SUBROUTINE NSCORA(N,IA,JA,A,IAP,JAWORK,AWORK,IC,IR,ICT)
      SUBROUTINE NSCORA(N,IA,JA,A,IAP,AWORK,IC,IR,ICT)
      USE PARAMS
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N
      INTEGER,DIMENSION(NRXN),INTENT(IN) :: IA
      INTEGER,DIMENSION(NRX1),INTENT(IN) :: JA
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IAP, IC
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IR, ICT
      REAL,DIMENSION(NRX5),INTENT(INOUT) :: A, AWORK 

      INTEGER :: IAINK, ICTJ, JAOUTJ, JMAX, JMIN, K
      INTEGER,DIMENSION(NRX5) :: JAWORK

      !write(6,*)'            nscora.f'
      DO K = 1, N
         IR(IC(K)) = K
      ENDDO
      JMIN = 1
      DO K = 1, N
         JMAX = JMIN + IA(IC(K)+1) - IA(IC(K)) - 1
         IF (JMIN.LE.JMAX) THEN
            IAINK = IA(IC(K)) - 1
            DO J = JMIN, JMAX
               IAINK = IAINK + 1
               JAOUTJ = JA(IAINK)
               JAOUTJ = IR(JAOUTJ)
               JAWORK(J) = JAOUTJ
               AWORK(J) = A(IAINK)
            ENDDO
         ENDIF
         JMIN = JMAX + 1
      ENDDO

      DO K = 1, N
         ICT(K) = IAP(K)
      ENDDO
      JMIN = 1
      DO K = 1, N
         JMAX = JMIN + IA(IC(K)+1) - IA(IC(K)) - 1
         IF (JMIN.LE.JMAX) THEN
            DO J = JMIN, JMAX
               JAOUTJ = JAWORK(J)
               ICTJ = ICT(JAOUTJ)
               A(ICTJ) = AWORK(J)
               ICT(JAOUTJ) = ICTJ + 1
            ENDDO
         ENDIF
         JMIN = JMAX + 1
      ENDDO
      END SUBROUTINE NSCORA

!***********************************************************************
      SUBROUTINE NSCORD(N,IA,JA,IAWORK,JAWORK,IC,IR,ICT)
      USE PARAMS
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IC, IR
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IA
      INTEGER,DIMENSION(NRX5),INTENT(INOUT) :: JA, JAWORK
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IAWORK,ICT

      INTEGER :: IAINK, ICK, ICTJ, JAOUTJ, JMAX, JMIN, K

      !write(6,*)'        nscord.f'
      DO I = 1, N
         ICT(I) = 0
      ENDDO
      IAWORK(1) = 1
      DO K = 1, N
         ICK = IC(K)
         JMIN = IAWORK(K)
         JMAX = JMIN + IA(ICK+1) - IA(ICK) - 1
         IAWORK(K+1) = JMAX + 1
         IF (JMIN.LE.JMAX) THEN
            IAINK = IA(ICK) - 1
            DO J = JMIN, JMAX
               IAINK = IAINK + 1
               JAOUTJ = JA(IAINK)
               JAOUTJ = IR(JAOUTJ)
               JAWORK(J) = JAOUTJ
               ICT(JAOUTJ) = ICT(JAOUTJ) + 1
            ENDDO
         ENDIF
      ENDDO
      IA(1) = 1
      DO I = 1, N
         IA(I+1) = IA(I) + ICT(I)
         ICT(I) = IA(I)
      ENDDO
      DO I = 1, N
         JMIN = IAWORK(I)
         JMAX = IAWORK(I+1) - 1
         IF (JMIN.LE.JMAX) THEN
            DO J = JMIN, JMAX
               JAOUTJ = JAWORK(J)
               ICTJ = ICT(JAOUTJ)
               JA(ICTJ) = I
               ICT(JAOUTJ) = ICTJ + 1
            ENDDO
         ENDIF
      ENDDO

      END SUBROUTINE NSCORD

!***********************************************************************
      SUBROUTINE NSNFAC(N,IA,JA,A,IL,JL,ISL,LREAL,D,IU,JU,ISU,U,X,IRL,&
                       JRL,IER)
      USE PARAMS
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N
      INTEGER,INTENT(OUT) :: IER
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IA, IL, ISL, IU, ISU
      INTEGER,DIMENSION(NRX5),INTENT(IN) :: JA, JL, JU
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IRL, JRL
                                         
      REAL,DIMENSION(NRX5),INTENT(INOUT) :: A, LREAL, U
      REAL,DIMENSION(NRXN),INTENT(INOUT) :: X
      REAL,DIMENSION(NRXN),INTENT(OUT) :: D

      INTEGER :: I1,I2,IRLI,ISLB,ISLK,ISUB, JAJ,JMAX,JMIN,JUJ, K
      REAL :: DK, LKI

      !write(6,*)'            nsnfac.f'
      IER = 0
      DO K = 1, N
         IRL(K) = IL(K)
         JRL(K) = 0
      ENDDO
      DO K = 1, N
         X(K) = 0.
         I1 = 0
         IF (JRL(K).NE.0) THEN
            I = JRL(K)
 20         I2 = JRL(I)
            JRL(I) = I1
            I1 = I
            X(I) = 0.
            I = I2
            IF (I.NE.0) GOTO 20
         ENDIF
         JMIN = ISU(K)
         JMAX = JMIN + IU(K+1) - IU(K) - 1
         IF (JMIN.LE.JMAX) THEN
            DO J = JMIN, JMAX
               JUJ = JU(J)
               X(JUJ) = 0.
            ENDDO
         ENDIF
         JMIN = IA(K)
         JMAX = IA(K+1) - 1
         DO J = JMIN, JMAX
            JAJ = JA(J)
            X(JAJ) = A(J)
         ENDDO
         I = I1
         IF (I.NE.0) THEN
 40         IRLI = IRL(I)
            LKI = -X(I)
            LREAL(IRLI) = -LKI
            JMIN = IU(I)
            JMAX = IU(I+1) - 1
            IF (JMIN.LE.JMAX) THEN
               ISUB = ISU(I) - 1
               DO J = JMIN, JMAX
                  ISUB = ISUB + 1
                  JUJ = JU(ISUB)
                  X(JUJ) = X(JUJ) + LKI*U(J)
               ENDDO
            ENDIF
            I = JRL(I)
            IF (I.NE.0) GOTO 40
         ENDIF
         IF (X(K).EQ.0.) GOTO 100
         DK = 1./X(K)
         D(K) = DK
         IF (K.NE.N) THEN
            JMIN = IU(K)
            JMAX = IU(K+1) - 1
            IF (JMIN.LE.JMAX) THEN
               ISUB = ISU(K) - 1
               DO J = JMIN, JMAX
                  ISUB = ISUB + 1
                  JUJ = JU(ISUB)
                  U(J) = X(JUJ)*DK
               ENDDO
            ENDIF
            I = I1
            IF (I.NE.0) THEN
 50            IRL(I) = IRL(I) + 1
               I1 = JRL(I)
               IF (IRL(I).LT.IL(I+1)) THEN
                  ISLB = IRL(I) - IL(I) + ISL(I)
                  J = JL(ISLB)
 55               IF (I.GT.JRL(J)) THEN
                     JRL(I) = JRL(J)
                     JRL(J) = I
                  ELSE
                     J = JRL(J)
                     GOTO 55
                  ENDIF
               ENDIF
               I = I1
               IF (I.NE.0) GOTO 50
            ENDIF
            ISLK = ISL(K)
            IF (IRL(K).LT.IL(K+1)) THEN
               J = JL(ISLK)
               JRL(K) = JRL(J)
               JRL(J) = K
            ENDIF
         ENDIF
      ENDDO
      RETURN

 100  IER = K

      END SUBROUTINE NSNFAC

!***********************************************************************
! as MM_2.1
      SUBROUTINE NSSFAC(N,IA,JA,MAXPL,IL,JL,ISL,MAXPU,IU,JU,ISU,P,V,&
                       IRA,JRA,IRAC,IRL,JRL,IRU,JRU,IER)
! as V2.4.06
      !SUBROUTINE NSSFAC(N,IA,JA,MAXPL,IL,JL,ISL,MAXPU,IU,JU,ISU,V,&
      !                 IRL,JRL,IER)
      USE PARAMS
      USE IOFIL
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N, MAXPL, MAXPU
      INTEGER,INTENT(OUT) :: IER
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IA
      INTEGER,DIMENSION(NRX5),INTENT(INOUT) :: JA
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: IL, IU, IRL, JRL
      INTEGER,DIMENSION(NRX5),INTENT(INOUT) :: JL, JU
      INTEGER,DIMENSION(NRX4),INTENT(INOUT) :: ISL, ISU, V

      INTEGER VI, VJ, PK, PPK, PI, CEND, REND, I1, IAK, IRAI,      &
              IRLI, IRUI, IRLL, IRUL, JAIAK, JAIRAI, JLPTR, JUPTR, &
              JMIN, JMAX, K, LASTI, LASTID, LSFS
      !INTEGER,DIMENSION(NRXN) :: IRA, JRA, IRU, JRU, P, IRAC
      INTEGER,DIMENSION(N) :: IRA, JRA, IRU, JRU, P, IRAC

      !write(6,*)'        nssfac.f'
      IER = 0
      DO K = 1, N
         IRA(K) = 0    ! jmlt
         IRL(K) = 0    ! jmlt
         IRU(K) = 0    ! jmlt
         IRAC(K) = 0
         JRA(K) = 0
         JRL(K) = 0
         JRU(K) = 0
      ENDDO
      DO K = 1, N
         IAK = IA(K)
         IF (IAK.GT.IA(K+1)) GOTO 300
         IF (JA(IAK).LE.K) THEN
            JAIAK = JA(IAK)
            JRA(K) = IRAC(JAIAK)
            IRAC(JAIAK) = K
         ENDIF
         IRA(K) = IAK
      ENDDO
      JLPTR = 0
      IL(1) = 1
      JUPTR = 0
      IU(1) = 1
      DO K = 1, N
         P(1) = 1
         V(1) = N + 1
         LSFS = 2
         VJ = IRAC(K)
         IF (VJ.NE.0) THEN
            PPK = 1
 20         PK = PPK
            PPK = P(PK)
            IF (V(PPK).LT.VJ) GOTO 20
            IF (V(PPK).EQ.VJ) GOTO 700
            P(PK) = LSFS
            V(LSFS) = VJ
            P(LSFS) = PPK
            LSFS = LSFS + 1
            VJ = JRA(VJ)
            IF (VJ.NE.0) THEN
               PPK = 1
               GOTO 20
            ENDIF
         ENDIF
         LASTI = 0
         I = K
 50      I = JRU(I)
         IF (I.EQ.0) THEN
            PI = P(1)
            IF (V(PI).NE.K) GOTO 800
            IF (LASTI.NE.0) THEN
               IF (LASTID.EQ.LSFS-3) THEN
                  IRLL = IRL(LASTI)
                  ISL(K) = IRLL + 1
                  IF (JL(IRLL).NE.K) ISL(K) = ISL(K) - 1
                  IL(K+1) = IL(K) + LASTID
                  IRL(K) = ISL(K)
                  GOTO 80
               ENDIF
            ENDIF
            ISL(K) = JLPTR + 1
            PI = P(1)
            PI = P(PI)
            VI = V(PI)
 60         IF (VI.GT.N) THEN
               IRL(K) = ISL(K)
               IL(K+1) = IL(K) + JLPTR - ISL(K) + 1
            ELSE
               JLPTR = JLPTR + 1
               IF (JLPTR.GT.MAXPL) GOTO 900
               JL(JLPTR) = VI
               PI = P(PI)
               VI = V(PI)
               GOTO 60
            ENDIF
 80         P(1) = 1
            V(1) = N + 1
            LSFS = 2
            JMIN = IRA(K)
            JMAX = IA(K+1) - 1
            IF (JMIN.LE.JMAX) THEN
               DO J = JMIN, JMAX
                  VJ = JA(J)
                  PPK = 1
 85               PK = PPK
                  PPK = P(PK)
                  IF (V(PPK).LT.VJ) GOTO 85
                  IF (V(PPK).EQ.VJ) GOTO 400
                  P(PK) = LSFS
                  V(LSFS) = VJ
                  P(LSFS) = PPK
                  LSFS = LSFS + 1
               ENDDO
            ENDIF
            LASTI = 0
            I = K
 100        I = JRL(I)
            IF (I.EQ.0) THEN
               PI = P(1)
               IF (V(PI).NE.K) GOTO 500
               IF (LASTI.NE.0) THEN
                  IF (LASTID.EQ.LSFS-3) THEN
                     IRUL = IRU(LASTI)
                     ISU(K) = IRUL + 1
                     IF (JU(IRUL).NE.K) ISU(K) = ISU(K) - 1
                     IU(K+1) = IU(K) + LASTID
                     IRU(K) = ISU(K)
                     I = K
                     GOTO 150
                  ENDIF
               ENDIF
               ISU(K) = JUPTR + 1
               PI = P(1)
               PI = P(PI)
               VI = V(PI)
 110           IF (VI.GT.N) THEN
                  IRU(K) = ISU(K)
                  IU(K+1) = IU(K) + JUPTR - ISU(K) + 1
                  I = K
               ELSE
                  JUPTR = JUPTR + 1
                  IF (JUPTR.GT.MAXPU) GOTO 600
                  JU(JUPTR) = VI
                  PI = P(PI)
                  VI = V(PI)
                  GOTO 110
               ENDIF
            ELSE
               PPK = 1
               JMIN = IRU(I)
               JMAX = ISU(I) + IU(I+1) - IU(I) - 1
               IF (LASTI.LE.I) THEN
                  LASTI = I
                  LASTID = JMAX - JMIN
                  IF (JU(JMIN).NE.K) LASTID = LASTID + 1
               ENDIF
               IF (JMIN.LE.JMAX) THEN
                  DO J = JMIN, JMAX
                     VJ = JU(J)
 112                 PK = PPK
                     PPK = P(PK)
                     IF (V(PPK).LT.VJ) GOTO 112
                     IF (V(PPK).NE.VJ) THEN
                        P(PK) = LSFS
                        V(LSFS) = VJ
                        P(LSFS) = PPK
                        PPK = LSFS
                     ENDIF
                     LSFS = LSFS + 1
                  ENDDO
               ENDIF
               GOTO 100
            ENDIF
         ELSE
            PPK = 1
            JMIN = IRL(I)
            JMAX = ISL(I) + IL(I+1) - IL(I) - 1
            IF (LASTI.LE.I) THEN
               LASTI = I
               LASTID = JMAX - JMIN
               IF (JL(JMIN).NE.K) LASTID = LASTID + 1
            ENDIF
            IF (JMIN.LE.JMAX) THEN
               DO J = JMIN, JMAX
                  VJ = JL(J)
 115              PK = PPK
                  PPK = P(PK)
                  IF (V(PPK).LT.VJ) GOTO 115
                  IF (V(PPK).NE.VJ) THEN
                     P(PK) = LSFS
                     V(LSFS) = VJ
                     P(LSFS) = PPK
                     PPK = LSFS
                     LSFS = LSFS + 1
                  ENDIF
               ENDDO
            ENDIF
            GOTO 50
!XXXXXXX
         ENDIF
 150     I1 = JRL(I)
         CEND = ISL(I) + IL(I+1) - IL(I)
         IF (IRL(I).LT.CEND) THEN
            IRLI = IRL(I)
            J = JL(IRLI)
            JRL(I) = JRL(J)
            JRL(J) = I
         ENDIF
         I = I1
         IF (I.EQ.0) THEN
            I = K
         ELSE
            IRL(I) = IRL(I) + 1
            GOTO 150
         ENDIF
 200     I1 = JRU(I)
         REND = ISU(I) + IU(I+1) - IU(I)
         IF (IRU(I).LT.REND) THEN
            IRUI = IRU(I)
            J = JU(IRUI)
            JRU(I) = JRU(J)
            JRU(J) = I
         ENDIF
         I = I1
         IF (I.EQ.0) THEN
            I = IRAC(K)
            IF (I.NE.0) THEN
 210           I1 = JRA(I)
               IRA(I) = IRA(I) + 1
               IF (IRA(I).LT.IA(I+1)) THEN
                  IRAI = IRA(I)
                  IF (JA(IRAI).LE.I) THEN
                     JAIRAI = JA(IRAI)
                     JRA(I) = IRAC(JAIRAI)
                     IRAC(JAIRAI) = I
                  ENDIF
               ENDIF
               I = I1
               IF (I.NE.0) GOTO 210
            ENDIF
         ELSE
            IRU(I) = IRU(I) + 1
            GOTO 200
         ENDIF
      ENDDO
      ISL(N) = JLPTR
      ISU(N) = JUPTR
      RETURN
 300  WRITE (LOUT,'("ROW ",I6," OF A IS NULL")') K
      GOTO 1000
 400  WRITE (LOUT,'("ROW ",I6," HAS DUPLICATE ENTRY")') K
      GOTO 1000
 500  WRITE (LOUT,'("ROW ",I6," HAS A NULL PIVOT")') K
      GOTO 1000
 600  WRITE (LOUT,'("ROW ",I6," EXCEEDS JU STORAGE")') K
      GOTO 1000
 700  WRITE (LOUT,'("COLUMN ",I6," HAS DUPILCATE ENTRY")') K
      GOTO 1000
 800  WRITE (LOUT,'("COLUMN ",I6," HAS A NULL PIVOT")') K
      GOTO 1000
 900  WRITE (LOUT,'("COLUMN ",I6," EXCEEDS JL STORAGE")') K
 1000 IER = 1

      END SUBROUTINE NSSFAC

!*****************************************************************************
      SUBROUTINE OPNDAT( iucl, clfile )
!-----------------------------------------------------------
!     Open clfile, on error write error number and terminate.
!-----------------------------------------------------------

      CHARACTER*20 clfile

      nend=index(clfile,' ')
      open(unit=iucl, file=clfile(1:nend-1),status='OLD',  &
          access='sequential',form='formatted',iostat=ios)
      IF (ios .NE. 0) THEN
         print 1000, ios
         write(19,1000) ios
         STOP
      ENDIF

1000  Format(1x,'error opening data file   ',i5)

      END SUBROUTINE OPNDAT

!**********************************************************************
       SUBROUTINE RATCO1
!----------------------------------------------------------------------
!      THIS ROUTINE CALCULATES RATE CONSTANTS FOR PHOTOLYSIS REACTIONS
!       (J VALUES) BY INTERPOLATING TO CURRENT TIME
!  On input:
!       TJVAL  - times at which J vals are specified
!       NPHOT  - number of such times
!       XJVAL  - corresponding J vals
!       PSLOPE - local slopes of J values
!       T      - current integration time
!       NR1    - number of photolysis reactions
!       ITYPE1 - reaction list index of photolysis reactions
!  On exit:
!       R      - first order rate coefficients at current time
!  Internal:
!       TT     - time of day    
!----------------------------------------------------------------------
      USE PARAMS
      USE KINET
      USE JVALU
      USE CLOUD
      USE UNITS
      IMPLICIT NONE
      REAL :: tt,dt
        
! assume cyclic one day data
      tt = t
      IF(t .GT. tday) tt = t - tday*FLOAT(INT(t/tday))

      DO j = 1, nphot
         IF (tjval(j+1) .GE. tt) THEN
            dt = tt - tjval(j)
            DO i = 1, nr1
               r(itype1(i)) = fcloud*(xjval(j,i) + pslope(j,i)*dt)
            ENDDO
            RETURN
         ENDIF
      ENDDO

      END SUBROUTINE RATCO1

!**********************************************************************
      SUBROUTINE RATCO2
!----------------------------------------------------------------------
!     THIS ROUTINE CALCULATES RATES CONSTANTS FOR SIMPLE ARRHENIUS 
!     EXPRESSIONS FOR SELECTED REACTIONS
! On Input:
!   TEMP   -  current temperature
!   NR2    -  number of such reactions
!   ITYPE2 -  reaction list index number for these reactions
!   A      -  rate constant at 298 K
!   S      -  Arrehnius temperature (Ea/R)
! On Exit
!   R      -  rate constant at current temperature
!----------------------------------------------------------------------
      USE PARAMS
      USE KINET
      USE ARRHEN
      USE TEMPER

      !write(6,*)'ratco2'
      FACT = 3.3556E-03  -  1./TEMP
      DO I = 1, NR2
         R(ITYPE2(I)) = A(ITYPE2(I))*EXP(FACT*S(ITYPE2(I)))
      ENDDO
        
      END SUBROUTINE RATCO2

!*********************************************************************
      SUBROUTINE RATCO3
!----------------------------------------------------------------------
!     THIS ROUTINE CALCULATES RATES CONSTANTS FOR SIMPLE ARRHENIUS 
!      EXPRESSIONS WHICH INCLUDE THE COLLISION PARTNER M
!      NOTE THAT TROE AND REVERSE TROE ARE EXCLUDED.
! On Input:
!   T      -  time, not used
!   TEMP   -  current temperature
!   NR3    -  number of such reactions
!   ITYPE2 -  reaction list index number for these reactions
!   A      -  rate constant at 298 K
!   S      -  Arrhenius temperature (Ea/R)
!   DENS     -  current value of the total concentration
! On Exit
!   R      -  rate constant at current temperature, [M]
!----------------------------------------------------------------------
      USE PARAMS
      USE KINET
      USE TEMPER
      USE DENSIT
      USE ARRHEN

      FACT = 3.3556E-03  -  1./TEMP
      DO I = 1, NR3
         R(ITYPE3(I)) = DENS*A(ITYPE3(I))*EXP(FACT*S(ITYPE3(I)))
      ENDDO
        
      END SUBROUTINE RATCO3

!**************************************************************************
      SUBROUTINE RATCO4
!----------------------------------------------------------------------
!     THIS ROUTINE CALCULATES RATE CONSTANTS FOR REACTIONS OF THE TYPE
!     A + B + (M)  ->  C + (M)      and
!     C + (M)  ->  A + B + (M)
!     USING TROE EXPRESSIONS AND EQUILIBRIUM CONSTANTS FOR THE REVERSE
!  On entry:
!       TEMP   - current temperature
!       DENS     - current total concentration
!       NR4    - number of Troe type reactions
!       ITYPE4  - reaction list index for these reactions
!       AK0300 - low pressure lim k at 300 K
!       AKI300 - high pressure lim k at 300 K
!       AN     - temp dependence coeff. for low pressure lim k
!       AM     - temp dependence coeff. for high pressure lim k
!       BAS    - base of Troe expression - e.g. JPL87 gives 0.6 for all reacts.
!       AEQUIL - pre-exponential of the equilibrium constant
!       TEQUIL - Arrhenius temperature (T/R) of the equilibrium constant
!  On exit:
!       R      - rate constants at current temperature and total concentr.
!  Internal:
!       TREL   - temperature / 300.
!       AK0T   - low press lim k at current temp
!       AKIT   - high press. lim k at current temp
!       F1,F2,F3,F4 - temporary values
!       FORW   - forward rate const.
!----------------------------------------------------------------------
      USE PARAMS
      USE KINET
      USE TEMPER
      USE DENSIT
      USE TROE

      TREL = TEMP/300.
      DO I = 1, NR4
         AK0T = AK0300(I) * TREL**(-AN(I))
         AKIT = AKI300(I) * TREL**(-AM(I))
         F1 = AK0T*DENS/AKIT
         F2 = AK0T*DENS/(1. + F1)
         F3 = ALOG(F1)/2.303
         F4 = 1./(1. + F3*F3)
         FORW = F2 * BAS(I)**F4
         R(ITYPE4(I)) = FORW / ( AEQUIL(I) *  &
             EXP(TEQUIL(I)*(1./TEMP - 1./298.)) )
      ENDDO

      END SUBROUTINE RATCO4

!*****************************************************************************
      SUBROUTINE RATCO5
!----------------------------------------------------------------------
!     THIS ROUTINE CALCULATES RATE CONSTANTS FOR SPECIAL REACTIONS
!     USING CUSTOM-MADE EXPRESSIONS
!  On entry:
!     TEMP   - current temperature
!     DENS   - current total concentration
!     cH2O   - current concentration of H2O
!     NR5    - number of special reactions
!     ITYPE5 - reaction list index for these reactions
!     IWHICH - index relating special equations to list 1 through NR5.
!  On exit:
!     R      - rate constants at current temperature and total concentr.
!  Internal:

!----------------------------------------------------------------------
      USE PARAMS
      USE KINET
      USE TEMPER
      USE DENSIT
      USE SPCIAL
      USE UNITS

      DO I = 1, NR5

        IF (IWHICH(I) .EQ. 1) THEN
! HO2+HO2 (JPL Eval #14, 2003)
!          RK2 = U2* 1.7e-12*EXP( 600.*(1./TEMP - 1./298.))
!          RK3 = U3* 4.9e-32*EXP(1000.*(1./TEMP - 1./298.))
! HO2+HO2 (JPL Eval #17, 2011)
          RK2 = U2* 3.0e-13*EXP(460./TEMP)
          RK3 = U3* 2.1e-33*EXP(920./TEMP)

          FH2O =CUNIT*  1.4e-21*EXP(2200./TEMP)
          R(ITYPE5(I)) = (RK2 + RK3*DENS)*(1. + FH2O*cH2O)
        ENDIF

! HNO3+HO (JPL Eval #14, 2003)
        IF (IWHICH(I) .EQ. 2) THEN
          rk0 = u2* 2.4e-14*EXP(460./temp)
          rk2 = u2* 2.7e-17*EXP(2199./temp)
          rk3 = u3* 6.5e-34*EXP(1335./temp)
          R(ITYPE5(I)) = RK0 + RK3*DENS/(1. + RK3*DENS/RK2)
        ENDIF
  
! CO+HO (JPL Eval #14, 2003)
        IF (IWHICH(I) .EQ. 3) THEN
           PRESS = (CUNIT* DENS/2.45e+19)*(TEMP/298.)
           R(ITYPE5(I)) = U2* 1.5e-13*(1.+0.6*PRESS)
        ENDIF

      ENDDO

      END SUBROUTINE RATCO5

!*******************************************************************
      SUBROUTINE RATCO6

! INPUTS:
!   nr6  = number of emission reactions
!   itype6 = reaction index for each emission reaction
!   temp = temperature (K) at solver time T
!   a    = rate constant at standard conditions, read directly from mechanism
!   s    = temperature dependence, in Arrhenius (exponential) form
!        (ignore all other variables in common block /arrhen/
!   z    = PBL height (cm) at T (from NEWENV)
! OUTPUT:
!   r = rate constant for each emission reaction      

      USE PARAMS
      USE KINET
      USE TEMPER
      USE ARRHEN
      USE PBL
      USE UNITS ! previous version specified z in km, now it's cm.

      DO i = 1, nr6
         j = itype6(i)
         r(j) = a(j)  * exp (s(j)*(temp - 298.)) *  1./(z/zunit)
      ENDDO

      END SUBROUTINE RATCO6

!*******************************************************************
      SUBROUTINE RATCO7

      USE PARAMS
      USE ARRHEN
      USE KINET
      USE PBL
      USE DEPOS
      USE UNITS ! previous version specified z in km, now it's cm.

! INPUTS:
! nr7 = number of deposition reactions
! itype7 = reaction index for each deposition reaction
! a = rate constant (in a 1 km boundary layer) at standard conditions, 
!     read from mechanism and converted to min-1 
! z = PBL height (cm) at t (from NEWENV)
! OUTPUT:
! r = rate constant for each deposition reaction (min-1) 

      DO i = 1, nr7
         j = itype7(i)
         r(j) = a(j)  *  1./(z/zunit)
      ENDDO  

      END SUBROUTINE RATCO7

!*********************************************************************
      SUBROUTINE SLOPES
!----------------------------------------------------------------------
!      Subroutine SLOPES calculates the slopes for J values
!      On Entry:
!          nr1    =  NUMBER OF PHOTOLYSIS REACTIONS
!          nphot  =  NUMBER OF PHOTOLYSIS TIMES
!          tjval  =  ARRAY OF TIMES AT WHICH J VALUES ARE SPECIFIED
!          xjval  =  CORRESPONDING J VALUES
!      On Exit:
!          pslope =  ARRAY OF THE SLOPES FOR A LINEAR FIT OF THE J VALUES
!----------------------------------------------------------------------
      USE PARAMS
      USE JVALU
      USE UNITS
      IMPLICIT NONE
      REAL tdif

      !write(6,*)'    slope.f'
      IF(nphot.LE.1) RETURN

! extend input data forward to next day for wraparound
      tjval(nphot+1) = tday + tjval(1)
      DO i = 1, nr1 
         xjval(nphot+1,i) = xjval(1,i)
      ENDDO
      
! calculate differences, slopes 
      DO j = 1, nphot
         tdif = tjval(j+1) - tjval(j)
         DO i = 1, nr1 
            pslope(j,i) = (xjval(j+1,i) - xjval(j,i)) / tdif
         ENDDO
      ENDDO

      END SUBROUTINE SLOPES

!****************************************************************************
      SUBROUTINE SORDER(N,IA,JA,P,Q,MAXL,V,LINT,IER)
      USE PARAMS
      USE IOFIL
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N, MAXL
      INTEGER,INTENT(IN) :: IA(NRXN), JA(NRX1)
      INTEGER,INTENT(OUT) :: IER
      INTEGER,DIMENSION(NRX4),INTENT(OUT) ::  P, Q
      INTEGER,DIMENSION(NRX5),INTENT(OUT) ::  V, LINT

      INTEGER :: JMAX, JMIN, K, KDIAG, LI, LJ, LK, LLK
      INTEGER :: S, SFS, PI, PJ, VI, VJ, VK, QVK, DTHR, DMIN

      !write(6,*)'        sorder.f'
      IER = 0
      DO S = 1, MAXL
         LINT(S) = S + 1
      ENDDO
      SFS = 1
      LINT(MAXL) = 0
      DO K = 1, N
         P(K) = K
         Q(K) = K
         V(K) = 1
         LINT(K) = 0
      ENDDO
      SFS = SFS + N
      DO K = 1, N
         JMIN = IA(K)
         JMAX = IA(K+1) - 1
         IF (JMIN.GT.JMAX+1) GOTO 700
         KDIAG = 0
         DO J = JMIN, JMAX
            VJ = JA(J)
            IF (VJ.NE.K) THEN
               LLK = K
 10            LK = LLK
               LLK = LINT(LK)
               IF (LLK.NE.0) THEN
                  IF (V(LLK).LT.VJ) GOTO 10
                  IF (V(LLK).EQ.VJ) THEN
                     LLK = VJ
                     GOTO 20
                  ENDIF
               ENDIF
               LLK = SFS
               IF (LLK.EQ.0) GOTO 800
               SFS = LINT(SFS)
               V(K) = V(K) + 1
               V(LLK) = VJ
               LINT(LLK) = LINT(LK)
               LINT(LK) = LLK
               LLK = VJ
            ELSE
               KDIAG = 1
               CYCLE
            ENDIF
 20         LK = LLK
            LLK = LINT(LK)
            IF (LLK.NE.0) THEN
               IF (V(LLK).LT.K) GOTO 20
               IF (V(LLK).EQ.K) CYCLE
            ENDIF
            LLK = SFS
            IF (LLK.EQ.0) GOTO 800
            SFS = LINT(SFS)
            V(VJ) = V(VJ) + 1
            V(LLK) = K
            LINT(LLK) = LINT(LK)
            LINT(LK) = LLK
         ENDDO
         IF (KDIAG.EQ.0) GOTO 900
      ENDDO
      J = 0
      DTHR = 0
      DMIN = N
      I = 0
 100  I = I + 1
      IF (I.GT.N) THEN
         RETURN
      ELSE
         JMIN = MAX0(J+1,I)
         IF (JMIN.GT.N) GOTO 300
      ENDIF
 200  DO J = JMIN, N
         VI = P(J)
         IF (V(VI).LE.DTHR) GOTO 400
         IF (V(VI).LT.DMIN) DMIN = V(VI)
      ENDDO
 300  DTHR = DMIN
      DMIN = N
      JMIN = I
      GOTO 200
 400  PJ = P(I)
      P(J) = PJ
      Q(PJ) = J
      PI = VI
      P(I) = PI
      Q(PI) = I
      LI = VI
 500  LI = LINT(LI)
      IF (LI.EQ.0) THEN
         LI = VI
 550     IF (LINT(LI).EQ.0) THEN
            LINT(LI) = SFS
            SFS = LINT(VI)
            GOTO 100
         ELSE
            LI = LINT(LI)
            VK = V(LI)
            LLK = VK
            QVK = MIN0(Q(VK),I)
 560        LK = LLK
            LLK = LINT(LK)
            IF (LLK.EQ.0) THEN
               IF (Q(VK).LE.I) THEN
                  LINT(LK) = SFS
                  SFS = LINT(VK)
               ELSEIF (V(VK).LE.DTHR) THEN
                  J = MIN0(Q(VK)-1,J)
                  DTHR = V(VK)
               ELSE
                  IF((DTHR.LT.V(VK)).AND.(V(VK).LT.DMIN)) DMIN = V(VK)
               ENDIF
               GOTO 550
            ELSE
               VJ = V(LLK)
               IF (Q(VJ).LE.QVK) THEN
                  V(VK) = V(VK) - 1
                  LINT(LK) = LINT(LLK)
                  LINT(LLK) = SFS
                  SFS = LLK
                  LLK = LK
               ENDIF
               GOTO 560
            ENDIF
         ENDIF
      ELSE
         VK = V(LI)
         LLK = VK
         LJ = VI
 600     LJ = LINT(LJ)
         IF (LJ.EQ.0) THEN
            IF (V(VK).LE.V(VI)) THEN
               I = I + 1
               QVK = Q(VK)
               PI = P(I)
               P(QVK) = PI
               Q(PI) = QVK
               P(I) = VK
               Q(VK) = I
            ENDIF
            GOTO 500
         ELSE
            VJ = V(LJ)
            IF (VJ.EQ.VK) GOTO 600
 620        LK = LLK
            LLK = LINT(LK)
            IF (LLK.NE.0) THEN
               IF (V(LLK).LT.VJ) GOTO 620
               IF (V(LLK).EQ.VJ) GOTO 600
            ENDIF
            LLK = SFS
            IF (LLK.EQ.0) THEN
               WRITE (LOUT,'("VERTEX ",I6," EXCEEDS STORAGE")') VI
               GOTO 1000
            ELSE
               SFS = LINT(SFS)
               V(VK) = V(VK) + 1
               V(LLK) = VJ
               LINT(LLK) = LINT(LK)
               LINT(LK) = LLK
               GOTO 600
            ENDIF
         ENDIF
      ENDIF
 700  WRITE (LOUT,'("ROW ",I6," OF A IS NULL")') K
      GOTO 1000
 800  WRITE (LOUT,'("ROW ",I6," EXCEEDS STORAGE")') K
      GOTO 1000
 900  WRITE (LOUT,'("COLUMN ",I6," .. DIAGONAL MISSING")') K
 1000 IER = 1
      END SUBROUTINE SORDER

!*****************************************************************************
      SUBROUTINE SPARS(IA,JA,N)
      USE PARAMS
      USE MECH
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N
      INTEGER,INTENT(OUT) :: IA(NRXN), JA(NRX1)

      INTEGER IR, K, K1, K2, KD, KM, KMAX, KMIN, KNOW, KT, &
              L, LL, M, MT, NM, NN

      LOGICAL SWITCH

      !write(6,*)'    spars.f'
      IA(:) = 1
      SWITCH = .FALSE.

      KT = 0
      IA(N+1) = 1
      JA(1) = 0
      DO IR = 1, NR
         DO K = 1, NREAG(IR)
            I = KR(IR,K)
            DO L = 1, NREAG(IR)
               J = KR(IR,L)
               K1 = IA(J)
               K2 = IA(J+1) - 1
               IF (K1.LE.K2) THEN
                  DO M = K1, K2
                     IF (I.EQ.JA(M)) SWITCH = .TRUE.
                  ENDDO
               ENDIF
               IF (SWITCH) THEN
                 SWITCH = .FALSE.
                 CYCLE  ! (use next L)
               ENDIF
               DO M = J, N
                  IA(M+1) = IA(M+1) + 1
               ENDDO
               KT = KT + 1
               KD = KT - K2
               K2 = K2 + 1
               DO M = 1, KD
                  JA(KT+2-M) = JA(KT+1-M)
               ENDDO
               JA(K2) = I
            ENDDO
! label 20
            K1 = IA(I)
            DO L = 4, 7
               K2 = IA(I+1) - 1
               J = KR(IR,L)
               IF (J.LT.0) THEN
               ELSEIF (J.EQ.0) THEN
                  EXIT
               ELSE
                  IF (K1.LE.K2) THEN
                     DO M = K1, K2
                        IF (J.EQ.JA(M)) SWITCH = .TRUE.
                     ENDDO
                  ENDIF
                  IF (SWITCH) THEN
                    SWITCH = .FALSE.
                    CYCLE  ! (use next L)
                  ENDIF
                  DO M = I, N
                     IA(M+1) = IA(M+1) + 1
                  ENDDO
                  KT = KT + 1
                  KD = KT - K2
                  K2 = K2 + 1
                  DO M = 1, KD
                     JA(KT+2-M) = JA(KT+1-M)
                  ENDDO
                  JA(K2) = J
               ENDIF
            ENDDO
! label 40 
         ENDDO
      ENDDO

      DO I = 1, N
         K1 = IA(I) + 1
         K2 = IA(I+1) - 1
         IF (K1.LE.K2) THEN
            MT = K2 - K1 + 1
            DO K = 1, MT
               DO M = K1, K2
                  IF (JA(M).LE.JA(M-1)) THEN
                     J = JA(M-1)
                     JA(M-1) = JA(M)
                     JA(M) = J
                  ENDIF
               ENDDO
            ENDDO
         ENDIF
      ENDDO
      M = N
      DO I = 1, M
         IF (IA(I+1).LE.IA(I)) THEN
            NM = I + 1
            NN = N + 1
            KMIN = IA(NM)
            KMAX = IA(NN)
            DO J = KMIN, KMAX
               KM = KMAX + KMIN - J
               JA(KM) = JA(KM-1)
            ENDDO
            KNOW = IA(I)
            JA(KNOW) = I
            DO LL = NM, NN
               IA(LL) = IA(LL) + 1
            ENDDO
         ENDIF
      ENDDO

      END SUBROUTINE SPARS

!**************************************************************************
      SUBROUTINE STIFFS(YREAL,N,IA,JA,IW1)
      USE PARAMS
      USE MECH
      USE KINET
      USE IOFIL
      USE CONTRL
      USE GEARS
      IMPLICIT NONE
      INTEGER,INTENT(IN) :: N
      INTEGER,INTENT(IN) :: IA(NRXN), JA(NRX1), IW1(NRX4,9)
      REAL,DIMENSION(N,13),INTENT(INOUT) :: YREAL

      INTEGER :: IDOUB, IER, IR, IREDO, IRET, ISV, IWEVAL, J1, J2,    &
                 JSTART, JX, K, KFLAG, KGO, KMAX, KMIN, KX, L, LMAX,  &
                 LSV, M, MEO, METH, MF, MFOLD, MIO, MITER, MT, NQ,    &
                 NEWQ, NFE, NHCUT, NJE, NOLD, NSTEP, NSTEPJ, NZZ
      REAL :: AYI, BND, CON, CRATE, D, D1, E, EDN,  ENQ1, ENQ2, ENQ3, &
              EPSOLD, EUP, FN, HOLD, PR1, PR2, PR3, R1, RC, RH, RMAX, &
              TOLD, TST, XJ, XK, OLDL0
      REAL :: ERROR(NRXN), W1(NRXN,3), W2(NRX5)
      REAL :: EL(13), TQ(4), RT(3)
! JMLT - to avoid rank mismatch when passing array YREAL(:,1) to SAVER
      REAL :: YDUM(N)

      !write(6,*) "stiffs"

      OLDL0 = 1.
      JSTART = 0
      NHCUT = 0
      MF = 21
      METH = 0  !jmlt
      W2 = 0

 100  CONTINUE 
      KFLAG = 0
      TOLD = T
      IF (JSTART.GT.0) GOTO 700
      IF (JSTART.NE.0) THEN
         IF (MF.EQ.MFOLD) GOTO 400
         MEO = METH
         MIO = MITER
         METH = MF/10
         MITER = MF - 10*METH
         MFOLD = MF
         IF (MITER.NE.MIO) IWEVAL = MITER
         IF (METH.EQ.MEO) GOTO 400
         IDOUB = L + 1
         IRET = 1
      ELSE

         CALL DIFFUN(N,YREAL(:,1),W1(:,1))

         DO I = 1, N
            YREAL(I,2) = H*W1(I,1)
         ENDDO
         METH = MF/10
         MITER = MF - 10*METH
         NQ = 1
         L = 2
         IDOUB = 3
         RMAX = 1.E4
         RC = 0.
         CRATE = 1.
         HOLD = H
         MFOLD = MF
         NSTEP = 0
         NSTEPJ = 0
         NFE = 1
         NJE = 0
         IRET = 3
      ENDIF

 200  CONTINUE

      CALL COSET(METH,NQ,EL,TQ)

      LMAX = 6
      RC = RC*EL(1)/OLDL0
      OLDL0 = EL(1)

 300  CONTINUE
      FN = FLOAT(NZRO)
      EDN = FN*(TQ(1)*EPS)**2
      E = FN*(TQ(2)*EPS)**2
      EUP = FN*(TQ(3)*EPS)**2
      BND = FN*(TQ(4)*EPS)**2
      EPSOLD = EPS
      NOLD = NZRO
      IF (IRET.EQ.1) GOTO 500
      IF (IRET.EQ.2) THEN
         RH = AMAX1(RH,HMIN/ABS(H))
         GOTO 600
      ELSEIF (IRET.EQ.3) THEN
         GOTO 700
      ENDIF

 400  CONTINUE
      IF (EPS.NE.EPSOLD .OR. NZRO.NE.NOLD) THEN
         IRET = 1
         GOTO 300
      ENDIF

 500  CONTINUE
      IF (H.EQ.HOLD) GOTO 700
      RH = H/HOLD
      H = HOLD
      IREDO = 3

 600  CONTINUE
      RH = AMIN1(RH,HMAX/ABS(H),RMAX)
      R1 = 1.
      DO J = 2, L
         R1 = R1*RH
         DO I = 1, N
            YREAL(I,J) = YREAL(I,J)*R1
         ENDDO
      ENDDO
      H = H*RH
      RC = RC*RH
      IDOUB = L + 1
      IF (IREDO.EQ.0) THEN
         RMAX = 100.
         GOTO 1500
      ENDIF

 700  CONTINUE
      IF (ABS(RC-1.).GT.0.3) IWEVAL = MITER
      IF (NSTEP.GE.NSTEPJ+20) IWEVAL = MITER
      T = T + H
      DO J1 = 1, NQ
         DO J2 = J1, NQ
            J = (NQ+J1) - J2
            DO I = 1, N
               YREAL(I,J) = YREAL(I,J) + YREAL(I,J+1)
            ENDDO
         ENDDO
      ENDDO

 800  CONTINUE
      DO I = 1, N
         ERROR(I) = 0.
      ENDDO
      M = 0

      CALL DIFFUN(N,YREAL(:,1),W1(:,2))

      NFE = NFE + 1
      IF (IWEVAL.GT.0) THEN
         IWEVAL = 0
         RC = 1.
         NJE = NJE + 1
         NSTEPJ = NSTEP
         CON = -H*EL(1)
         ISV = M
         LSV = L
         NZZ = IA(N+1) - 1
         DO I = 1, NZZ
            W2(I) = 0.
         ENDDO
         DO IR = 1, NR
            MT = NREAG(IR)
            DO I = 1, MT
               JX = I + 1 - I/3*3
               KX = I + 2 - I/2*3
               J = KR(IR,JX)
               K = KR(IR,KX)
               L = KR(IR,I)
               XJ = 1.
               IF (J.NE.0) XJ = YREAL(J,1)
               XK = 1.
               IF (K.NE.0) XK = YREAL(K,1)
               RT(I) = R(IR)*XJ*XK
            ENDDO
            DO K = 1, MT
               I = KR(IR,K)
               DO L = 1, MT
                  J = KR(IR,L)
                  M = IA(J) - 1
 805              M = M + 1
                  IF (I.NE.JA(M)) GOTO 805
                  W2(M) = W2(M) - RT(L)
               ENDDO
               DO L = 4, 7
                  J = KR(IR,L)
                  M = IA(I) - 1
                  IF (J.LT.0) THEN
                  ELSEIF (J.EQ.0) THEN
                     EXIT
                  ELSE
 806                 M = M + 1
                     IF (J.NE.JA(M)) GOTO 806
                     W2(M) = W2(M) + RT(K)*SC(IR,L)
                  ENDIF
               ENDDO
            ENDDO
         ENDDO
         DO J = 1, N
            KMIN = IA(J)
            KMAX = IA(J+1) - 1
            DO K = KMIN, KMAX
               W2(K) = W2(K)*CON
               IF (JA(K).EQ.J) W2(K) = W2(K) + 1. - CON*0.
               IF (JA(K).EQ.J .AND. J.EQ.N) W2(K) = W2(K) + CON*0.
            ENDDO
         ENDDO

! as MM_2.1
         !CALL NSCORA (N,IA,JA,W2,IW1(1,2),W2(IPTR3),W2(IPTR2), &
         !            IW1,IW1(1,7), IW1(1,8))
! as V2.4.06
         CALL NSCORA(N,IA,JA,W2,IW1(:,2),W2(IPTR2:NRX5), &
                    IW1(:,1),IW1(:,7),IW1(:,8))

! as MM_2.1
         !CALL NSNFAC (N,IW1(1,2),IW2,W2,IW1(1,3),IW2(IPTI2),IW1(1,4), &
         !             W2(IPTR2),W1(1,3),IW1(1,5),IW2(IPTI3),IW1(1,6), &
         !             W2(IPTR3),W1,IW1(1,7),IW1(1,8),IER)
! as V2.4.06
         CALL NSNFAC(N,IW1(:,2),IW2,W2,IW1(:,3),IW2(IPTI2:NRX5),  &
                    IW1(:,4),W2(IPTR2:NRX5),W1(:,3),IW1(:,5),     &
                    IW2(IPTI3:NRX5),IW1(:,6),W2(IPTR3:NRX5),      &
                    W1(:,1),IW1(:,7),IW1(:,8),IER)

         M = ISV 
         L = LSV
         IF (IER.NE.0) GOTO 1100
      ENDIF

 900  CONTINUE
      DO I = 1, N
         IF (M.GT.0) THEN
            IF (-H*W1(I,2)*10..GT.YREAL(I,1)) GOTO 1000
         ENDIF
         W1(I,1) = H*W1(I,2) - (YREAL(I,2)+ERROR(I))
      ENDDO

! as MM_2.1
      !CALL NSBSLV (N,IW1,IW1,IW1(1,3),IW2(IPTI2),IW1(1,4),W2(IPTR2),&
      !             W1(1,3),IW1(1,5),IW2(IPTI3),IW1(1,6),W2(IPTR3), &
      !             W1(1,2),W1,W2)
! as V2.4.06
      CALL NSBSLV(N,IW1(:,1),IW1(:,1),IW1(:,3),IW2(IPTI2:NRX5), &
                  IW1(:,4),W2(IPTR2:NRX5),W1(:,3),IW1(:,5),     &
                  IW2(IPTI3:NRX5),IW1(:,6),W2(IPTR3:NRX5),      &
                  W1(:,2),W1(:,1),W2)

      D = 0.
      DO I = 1, N
         ERROR(I) = ERROR(I) + W1(I,2)
         D = D + (W1(I,2)/YMAX(I))**2
         W1(I,1) = YREAL(I,1) + EL(1)*ERROR(I)
      ENDDO
      IF (M.NE.0) CRATE = AMAX1(.9*CRATE,D/D1)
      IF ((D*AMIN1(1.,2.*CRATE)).LE.BND) THEN
         IF (MITER.NE.0) IWEVAL = -1
         NFE = NFE + M
         D = 0.
         DO I = 1, N
            D = D + (ERROR(I)/YMAX(I))**2
         ENDDO
         IF (D.GT.E) THEN
            KFLAG = KFLAG - 1
            T = TOLD
            DO J1 = 1, NQ
               DO J2 = J1, NQ
                  J = (NQ+J1) - J2
                  DO I = 1, N
                     YREAL(I,J) = YREAL(I,J) - YREAL(I,J+1)
                  ENDDO
               ENDDO
            ENDDO
            RMAX = 2.
            IF (ABS(H).LE.HMIN*1.00001) THEN
               KFLAG = -1
               GOTO 1500
            ELSEIF (KFLAG.LE.-3) THEN
               IF (KFLAG.EQ.-9) THEN
                  KFLAG = -2
                  GOTO 1500
               ELSE
                  RH = 10.**KFLAG
                  RH = AMAX1(HMIN/ABS(H),RH)
                  H = H*RH

                  CALL DIFFUN(N,YREAL(:,1),W1(:,1))

                  NFE = NFE + 1
                  DO I = 1, N
                     YREAL(I,2) = H*W1(I,1)
                  ENDDO
                  IWEVAL = MITER
                  IDOUB = 10
                  IF (NQ.EQ.1) GOTO 700
                  NQ = 1
                  L = 2
                  IRET = 3
                  GOTO 200
               ENDIF
            ELSE
               IREDO = 2
               PR3 = 1.E+20
               GOTO 1200
            ENDIF
         ELSE
            KFLAG = 0
            IREDO = 0
            NSTEP = NSTEP + 1
            DO J = 1, L
               DO I = 1, N
                  YREAL(I,J) = YREAL(I,J) + EL(J)*ERROR(I)
               ENDDO
            ENDDO
            IF (IDOUB.EQ.1) THEN
               PR3 = 1.E+20
               IF (L.NE.LMAX) THEN
                  D1 = 0.
                  DO I = 1, N
                     D1 = D1 + ((ERROR(I)-YREAL(I,LMAX))/YMAX(I))**2
                  ENDDO
                  ENQ3 = .5/FLOAT(L+1)
                  PR3 = ((D1/EUP)**ENQ3)*1.4 + 1.4E-6
               ENDIF
               GOTO 1200
            ELSE
               IDOUB = IDOUB - 1
               IF (IDOUB.LE.1) THEN
                  IF (L.NE.LMAX) THEN
                     DO I = 1, N
                        YREAL(I,LMAX) = ERROR(I)
                     ENDDO
                  ENDIF
               ENDIF
               GOTO 1500
            ENDIF
         ENDIF
      ELSE
         D1 = D
         M = M + 1
         IF (M.NE.3) THEN

            CALL DIFFUN(N,W1(:,1),W1(:,2))

            GOTO 900
         ENDIF
      ENDIF

 1000 CONTINUE
      NFE = NFE + 2
      IF (IWEVAL.EQ.-1) THEN
         IWEVAL = MITER
         GOTO 800
      ENDIF

 1100 CONTINUE
      T = TOLD
      RMAX = 2.
      DO J1 = 1, NQ
         DO J2 = J1, NQ
            J = (NQ+J1) - J2
            DO I = 1, N
               YREAL(I,J) = YREAL(I,J) - YREAL(I,J+1)
            ENDDO
         ENDDO
      ENDDO
      IF (ABS(H).LE.HMIN*1.00001) THEN
         KFLAG = -3
         GOTO 1500
      ELSE
         RH = .25
         IREDO = 1
         RH = AMAX1(RH,HMIN/ABS(H))
         GOTO 600
      ENDIF

 1200 CONTINUE
      ENQ2 = .5/FLOAT(L)
      PR2 = ((D/E)**ENQ2)*1.2 + 1.2E-6
      PR1 = 1.E+20
      IF (NQ.NE.1) THEN
         D = 0.
         DO I = 1, N
            D = D + (YREAL(I,L)/YMAX(I))**2
         ENDDO
         ENQ1 = .5/FLOAT(NQ)
         PR1 = ((D/EDN)**ENQ1)*1.3 + 1.3E-6
      ENDIF
      IF (PR2.LE.PR3) THEN  
         IF (PR2.LE.PR1) THEN 
            NEWQ = NQ           
            RH = 1./PR2
            GOTO 1300
         ENDIF
      ELSEIF (PR3.LT.PR1) THEN 
         NEWQ = L               
         RH = 1./PR3
         IF (RH.LT.1.1) THEN
            IDOUB = 10
            GOTO 1500
         ELSE
            DO I = 1, N
               YREAL(I,NEWQ+1) = ERROR(I)*EL(L)/FLOAT(L)
            ENDDO
            GOTO 1400
         ENDIF
      ENDIF
      NEWQ = NQ - 1             
      RH = 1./PR1

 1300 CONTINUE
      IF ((KFLAG.EQ.0) .AND. (RH.LT.1.1)) THEN
         IDOUB = 10
         GOTO 1500
      ELSEIF (NEWQ.EQ.NQ) THEN
         RH = AMAX1(RH,HMIN/ABS(H))
         GOTO 600
      ENDIF

 1400 CONTINUE
      NQ = NEWQ
      L = NQ + 1
      IRET = 2
      GOTO 200

 1500 CONTINUE
      HOLD = H
      JSTART = NQ
      KGO = 1 - KFLAG
      IF (KGO.EQ.2 .OR. KGO.EQ.4) THEN
         IF (NHCUT.EQ.10) THEN
            WRITE (LOUT,99001)
99001       FORMAT (' PROBLEM APPEARS UNSOLVABLE WITH GIVEN INPUT')
            IF (KGO.EQ.4) WRITE (LOUT,99004) T
            IF (KGO.EQ.4) WRITE (6,99004) T
            STOP
         ELSE
            NHCUT = NHCUT + 1
            HMIN = .1*HMIN
            H = .1*H
            JSTART = -1
            GOTO 100
         ENDIF
      ELSEIF (KGO.EQ.3) THEN
         WRITE (LOUT,99002) T, H
         WRITE (6,99002) T, H
         STOP
      ELSE 
         D = 0.
         NZRO = 0
         TST = EPS*1.E-10
         DO I = 1, N
            IF (YREAL(I,1).LT.0.) THEN
               DO J = 1, 6
                  K = (J-1)*N + I
                  !YREAL(K,1) = 0.
! JMLT: safeguard for array bounds
                  IF(K.LE.N)THEN
                    YREAL(K,1) = 0.
                  ELSE
                    EXIT
                  ENDIF
               ENDDO
            ENDIF
            IF (YREAL(I,1).GT.TST) NZRO = NZRO + 1
            AYI = ABS(YREAL(I,1))
            YMAX(I) = AMAX1(1.E-10,AYI)
            D = D + (AYI/YMAX(I))**2
         ENDDO
         NZRO = MAX0(NZRO,1)
         IF (NZRO.NE.NOLD) JSTART = -1
         D = D*(UROUND/EPS)**2

!         CALL SAVER(T,N,YREAL(:,1))

!JMLT - avoid array rank mismatch when passing into SAVER
         YDUM = YREAL(:,1)
         CALL SAVER(T,N,YDUM)

         IF (D.GT.FLOAT(N)) THEN
            WRITE (LOUT,99003) T
            WRITE (6,99003) T
            STOP
         ELSE
            IF ((T-STOPP)*H.LT.0.) GOTO 100
            RETURN
         ENDIF
      ENDIF
99002 FORMAT (' KFLAG = -2 FROM INTEGRATOR AT T = ',E16.8,' H =',E16.8/ &
              ' THE REQUESTED ERROR IS SMALLER THAN CAN BE HANDLED')
99003 FORMAT (' INTEGRATION HALTED BY STIFFS AT T = ',E16.8/ &
              ' EPS TOO SMALL TO BE ATTAINED FOR THE MACHINE PRECISION')
99004 FORMAT (' KFLAG = -3 FROM INTEGRATOR AT T = ', &
              E16.8/' CORRECTOR CONVERGENCE COULD NOT BE ACHIEVED')
!99005 FORMAT (' INDEX = -1 ON INPUT WITH (T-STOPP)*H .GE. 0./4H T = ', &
!              E16.8,' STOPP = ',E16.8,' H =',E16.8)

      END SUBROUTINE STIFFS

!***************************************************************
      SUBROUTINE SAVER (t,ns,c)

      !write(6,*)'        saver.f'

! not in use now
!!      CALL  SAVOP1 (t,ns,c)

!  saves at approx equally spaced MSAVE times 
      CALL SAVOP2 (t,ns,c)

!  saves once per day near selected time e.g. noon
!      CALL SAVOP3 (t,ns,c)

      END SUBROUTINE SAVER

!***************************************************************
      SUBROUTINE SAVOP1 (T,NS,C)
!----------------------------------------------------------------------------
!  This subroutine SAVe OPtions 1 (SAVOPT) will save concentrations at all 
!     solver time steps
!  times which are selected from solver times.  The sampling is done on the
!  basis of the next occurrence of a solver time step following an equally
!  spaced grid of MSAVE times.
!
! On entry:
!    START  - start time for integration
!    STOPP  - final time for integration
!    T      - current integration time
!    MSAVE  - desired total number of saved times
!    NS     - number of species
!    C      - concentration of each species at T
! On exit
!    NT     - save index, incremented here. Last value is true number of
!             saved times (sometimes less than MSAVE)
!    SAVCON - saved concentrations
!    SAVTIM - time at which concentrations are saved
! Internal Variable
!    THRESH - threshold that T must .ge. in order to save concentrations
!----------------------------------------------------------------------------
      USE PARAMS
      USE SAVED
      USE RATES
      USE CONTRL

      REAL :: C(NSPC)

      IF (T .LT. 4.*24.*60.) RETURN

      NT = NT + 1
      SAVTIM(NT) = T
      DO J = 1, NS
        SAVCON(NT,J) = C(J)
      ENDDO

! save rates if requested
      IF (NRATES .GT. 0) THEN
        DO J = 1, NRATES
          SAVRAT(NT,J) = RTS(IRATES(J))
        ENDDO
      ENDIF

      END SUBROUTINE SAVOP1

!***************************************************************
      SUBROUTINE SAVOP2 (t,ns,c)
!----------------------------------------------------------------------------
!  This subroutine SAVe OPtions 2 (SAVOP2) will save concentrations at MSAVE
!  times which are selected from solver times.  The sampling is done on the
!  basis of the next occurrence of a solver time step following an equally
!  spaced grid of MSAVE times.
!
! On entry:
!    START  - start time for integration
!    STOPP  - final time for integration
!    T      - current integration time
!    MSAVE  - desired total number of saved times
!    NS     - number of species
!    C      - concentration of each species at T
! On exit
!    NT     - save index, incremented here. Last value is true number of
!             saved times (sometimes less than MSAVE)
!    SAVCON - saved concentrations
!    SAVTIM - time at which concentrations are saved
! Internal Variable
!    THRESH - threshold that T must .ge. in order to save concentrations
!----------------------------------------------------------------------------
        USE PARAMS
        USE SAVED
        USE RATES
        USE CONTRL

        IMPLICIT NONE

        INTEGER :: ns
        REAL :: thresh,t
        REAL,DIMENSION(ns) :: c

        !write(6,*)'            savop2.f'
         
        IF(nt.EQ.msave) RETURN

        thresh=(stopp-start)/FLOAT(msave-1)*FLOAT(nt)

        IF(t.GE.(thresh+start)) THEN
           nt = nt + 1
           savtim(nt) = t
           DO  j = 1, ns
              savcon(nt,j) = c(j)
           ENDDO
! save rates if requested
           IF (nrates .GT. 0) THEN
              DO j = 1, nrates
                 savrat(nt,j) = rts(irates(j))
              ENDDO
           ENDIF
        ENDIF

      END SUBROUTINE SAVOP2

!***************************************************************
      SUBROUTINE SAVOP3 (T,NS,C)
!----------------------------------------------------------------------------
!  This subroutine SAVe OPtions 3 (SAVOP3) will save concentrations at a
!   specific time of day, e.g., Noon
!   assuming that the starting time START is midnight. This routine is intended
!   for long-time simulations, e.g., years.
!   It actually saves the first solver time which occurs at or after the save
!    time
!
! On entry:
!    T      - current integration time
!    MSAVE  - dimension allotted to saved times
!    NS     - number of species
!    C      - concentration of each species at T
! On exit
!    NT     - save index, incremented here. Last value is true number of
!             saved times (sometimes less than MSAVE)
!    SAVCON - saved concentrations
!    SAVTIM - time at which concentrations are saved
! Internal
!    MONCE  - flags the first occurence after noon.     
!    TDAY   - time of day at which save is desired (e.g., Noon = 12.)
!----------------------------------------------------------------------------
      USE SAVED
      USE CONTRL

      SAVE MONCE
      REAL ::  C(NSPC)

!  For noon, TSAVE is 720. minutes = 4.32e4 seconds
      TSAVE = 4.32e4

      IF ( T .EQ. START ) NT = 0
      IF (NT .EQ. MSAVE ) RETURN

!  Find time of day:

      TIME = T - 8.64e4*FLOAT(INT(T/8.64e4))

!  Check if too soon
      IF (TIME.LT.TSAVE) THEN
          MONCE = 1
         RETURN
      ENDIF
  
!  Check if too late
      IF (MONCE.EQ.0) RETURN

! this should be first point on or after TSAVE
      NT = NT + 1
      SAVTIM(NT) = T
      DO J = 1, NS
         SAVCON(NT,J) = C(J)
      ENDDO
      MONCE = 0

      END SUBROUTINE SAVOP3

!*****************************************************************************
      SUBROUTINE USRINP(nx,tx,x,iflag)

!  interpolates input time-dependent values to cover one complete day:
!  input/output:  
!     NX - number of readin data pairs
!     TX - time : output augmented by one value if necessary
!     X  - values : output augmented by one value if necessary

      USE PARAMS
      USE CONTRL
      USE UNITS
      USE FLAGS
      IMPLICIT NONE
      INTEGER :: iflag, nx
      REAL,DIMENSION(nx+1) :: tx,x

      !write(6,*)"usrinp"
! check for initial values 
      IF(tx(1).GT.start) THEN
        PRINT*,&
        'INPUT ERROR : data must begin at or before start time :', &
        cflag(iflag)
        STOP
      ENDIF
      DO i=1,nx-1
        IF(tx(i).GT.tx(i+1))THEN
          PRINT*,&
          'INPUT ERROR : times must be in increasing order : ', &
          cflag(iflag)
          STOP
        ENDIF
      ENDDO
! extend input data forward for wraparound: 
! single values don't require tx/x notation and didn't invoke this subroutine.

! multiple values are wrapped around overnight if < 1 day is suppplied.
      IF(tx(nx).LT.tday)THEN
          tx(nx+1) = tx(1)+tday
          x(nx+1) = x(1)
      ELSE
! If > 1 day but < run length is supplied, run stops and returns error.
        IF(tx(nx).LT.stopp)THEN
          PRINT*,&
          'INPUT ERROR : last time must be < 1 day or > run length : ',&
          cflag(iflag)
          STOP
        ENDIF
! If supplied for entire run, values are left as is.
        tx(nx+1) = 1.e+38 
        x(nx+1) = x(nx)  ! safety value in case solver goes past end of run
      ENDIF
      
      END SUBROUTINE USRINP

!***************************************************************
