      PROGRAM jblock
*_______________________________________________________________________
*     January 2002 by Sasha Madronich
*     Calculate J-values for master mechanism simulations
*     Output is blocks of j-values and related information
*     J values can be calculated in one of two ways:
*     - using pre-tabulated values from TUV (Tropospheric Ultraviolet-Visible
*       radiation model), for those reactions available in TUV
*     - using blocks of estimated cross sections and quantum yields, as in
*       old jblock routines used in early version of the master mech.
*     The assignments (TUV or jblock based) are made in DATA/instr.dat
*     Pressure dependence of quantum yields is included.  For TUV-based 
*       jvalues, these are computed directly in tuv for the appropriate
*       altitude (and therefore pressure, temperature).  For jblock-based
*       jvalues, the pressure dependence is calculated here assuming a
*       Stern-Volmer form to unity yield at zero pressure (for multiple
*       channels, the total yield goes to unity at zero pressure, but the
*       branching ratio is constant).
*     Note that jblock data are specified on a fixed wavelength grid (Isaksen's)
*       while the tuv-based data may be on arbitrary wavelength grids. This code
*       interpolates all data to the tuv grid.
*     INPUT FILES:
*       tmp.mch - new mechanism for which jvalues are needed
*       tuv.jv  - precomputed tuv jvalues
*       DATA/instr.dat - instruction on how to compute jvalues for each reaction
*       DATA/rxs.dat - blocks of cross section data (for reactions not in tuv)
*       DATA/rqy.dat - blocks of quantum yield data (for reactions not in tuv)
*_______________________________________________________________________

      IMPLICIT NONE

      INTEGER kin
      PARAMETER(kin=12)
      INTEGER  j

      INTEGER ir, jr, nr, kr, krj
      PARAMETER (kr = 10000)
      PARAMETER (krj = 1000)
      INTEGER indexr(krj)
      CHARACTER*81 areact
      CHARACTER*5 molec(krj)
      CHARACTER*1 csplit(krj)

      INTEGER i, ninstr
      CHARACTER*75 ainstr
      LOGICAL usetuv(krj)
      CHARACTER*5 xmolec(krj)
      CHARACTER*50 xlabel(krj)
      REAL xscale(krj)

      INTEGER ij, nj, kj
      PARAMETER(kj = 100)
      CHARACTER*50 jlabel(kj)
      INTEGER itime, ntime, ktime
      PARAMETER(ktime = 100)
      REAL time(ktime)
      REAL mmjsav(ktime, kj)

      INTEGER iw, nwm1, nw, kw
      PARAMETER(kw = 200)
      REAL wl(kw), wc(kw), wu(kw)
      INTEGER izmm
      REAL z, tlev, airlev
      REAL mmfsav(ktime,kw)

      INTEGER ns, ks
      PARAMETER(ks=100)
      CHARACTER*4 title, slabel(ks)
      REAL yold(130)
      REAL xs(ks,kw), yg(kw)

      INTEGER nold, idum
      REAL dum, wold(130)

      INTEGER nq, kq
      PARAMETER (kq=100)
      CHARACTER*7 title7
      CHARACTER*5 qlabel(kq)
      REAL qy(kq,kw)

      INTEGER jfind(krj)
      REAL djdw
      REAL valj(krj, ktime)
      LOGICAL nofind

      CHARACTER*4 XSP1(krj), xsp2(krj), xsp3(krj)
      CHARACTER*5 xqy(krj)
      CHARACTER*1 xchan(krj), xqpflg(krj), xqtflg(krj)
      REAL xcoef1(krj), xcoef2(krj), xcoef3(krj)

      REAL xsect(krj, kw)
      REAL qyield(krj, kw)
      REAL qytntp(krj,kw), qyp(krj,kw)

      INTEGER isplit, nsplit

*************************** Initialise *********************************
      nofind = .FALSE.
*************************** Program Body *******************************
***** read mechanism for lines having HV:
* parse line for reagent r(nr) (including channel A, B, etc.), and for
* split reaction flag csplit(nr)

      OPEN(UNIT=kin,FILE='tmp.mch',STATUS='old')
      nr = 0
      DO 10 ir = 1, kr
         READ(kin,'(A81)') areact
         IF(areact(1:4) .EQ. '****') GO TO 15
         IF(areact(2:2) .EQ. 'X') GO TO 10
         IF(areact(8:11) .NE. 'HV  ') GO TO 10
         nr = nr + 1
         indexr(nr) = ir
         molec(nr) = areact(3:6)//areact(19:19)
         csplit(nr) = areact(18:18)
 10   CONTINUE
 15   CONTINUE
      CLOSE(kin)
      WRITE(*,*) 'no. of photoreactions, nr = ', nr

***** read instuctions file, parse 
* usetuv = T for tuv-based, = F for jblock-based

      OPEN(UNIT=kin,FILE='../JBLOCK/DATA/instr.dat',STATUS='old')
      DO 20 i = 1, krj
         READ(kin,'(a75)') ainstr
         IF (ainstr(1:4) .EQ. '****') GO TO 25

         xmolec(i)  = ainstr(1:4)//ainstr(6:6)
         READ(ainstr,'(9x,E9.3)') xscale(i)
         xlabel(i) = ainstr(20:69)

         usetuv(i) = .TRUE.
         IF (ainstr(8:8) .NE. 'J') usetuv(i) = .FALSE.

 20   CONTINUE
 25   ninstr = i - 1
      CLOSE(kin)
      WRITE(*,*) 'no. of instructions, ninstr = ', ninstr

***** read tuv-computed actinic fluxes:

      OPEN(UNIT=kin,FILE='usrout.txt',STATUS='old')
      READ(kin,*)
      READ(kin,*) nwm1, ntime
      nw = nwm1 + 1
      READ(kin,*) izmm, z, tlev, airlev
      READ(kin,*) (time(itime), itime = 1, ntime)
      DO iw = 1, nw-1
         READ(kin,*) idum, wl(iw), wc(iw), wu(iw)
         READ(kin,*) (mmfsav(itime,iw), itime = 1, ntime)
      ENDDO
      wl(nw) = wu(nw-1)
      write(*,*) 'no. of tuv actinic flux wavelengths, nw = ', nw

***** read tuv-computed J-values:

      READ(kin,*) ntime, nj
      DO ij = 1, nj
         READ(kin,'(a50)') jlabel(ij)
         READ(kin,*) (mmjsav(itime,ij), itime = 1, ntime)
      ENDDO
      CLOSE(kin)
      WRITE(*,*) 'no. of tuv j-values, nj = ', nj

***** read cross-section blocks.  These are on Isaksen w-grid (130 bins)
* read old wavelength grid:

      OPEN(UNIT=kin,FILE='../JBLOCK/DATA/grid_isak.dat',STATUS='old')
      DO I = 1, 4
         READ(KIN,*)
      ENDDO
      nold = 130
      DO i = 1, nold
         READ(kin,*) idum, dum, wold(i), dum, dum
      ENDDO
      CLOSE(kin)

* read cross sections:

      OPEN(UNIT=kin,FILE='../JBLOCK/DATA/rxs.dat',STATUS='old')
      DO i = 1, 28
         READ(kin,*)
      ENDDO
      DO 30 i = 1, ks
         READ(kin,*) title
         IF (title .EQ. '****') GO TO 35
         slabel(i) = title
         READ(kin,*) (yold(j), j = 1, nold)
         CALL inter2(nw,wl,yg,nold,wold,yold)
         DO iw = 1, nw - 1
            xs(i,iw) = yg(iw)
         ENDDO
 30   CONTINUE
 35   ns = i - 1
      CLOSE(kin)

***** read quantum yield blocks
* values are assumed to be valid for NTP (298K, 2.45e19 molec cm-3)

      OPEN(UNIT=kin,FILE='../JBLOCK/DATA/rqy.dat',STATUS='old')
      DO i = 1, 10
         READ(kin,*)
      ENDDO
      DO 40 i = 1, kq
         READ(kin,*) title7
         IF (title7 .EQ. '*******') GO TO 45
         qlabel(i) = title7(1:4)//title7(7:7)
         READ(kin,*) (yold(j), j = 1, nold)
         CALL inter2(nw,wl,yg,nold,wold,yold)
         DO iw = 1, nw - 1
            qy(i,iw) = yg(iw)
         ENDDO
 40   CONTINUE
 45   nq = i - 1
      CLOSE(kin)

********* locate instruction for each reaction:

      DO 50 ir = 1, nr
         jfind(ir) = 0
         DO i = 1, krj
            IF (molec(ir) .EQ. xmolec(i)) THEN
               jfind(ir) = i
               GO TO 51
            ENDIF
         ENDDO
         IF (jfind(ir) .EQ. 0) THEN
            nofind = .TRUE.
            WRITE(*,*) 'not found: ', molec(ir)
            GO TO 50
         ENDIF
 51      CONTINUE
         j = jfind(ir)

* if tuv-based, usetuv=true, then locate and assign j values

         IF (usetuv(j)) THEN
            DO 52 ij = 1, nj
               IF (jlabel(ij) .EQ. xlabel(j)) THEN
                  DO itime = 1, ntime
                     valj(ir, itime) = mmjsav(itime,ij) * xscale(j)
                  ENDDO
                  GO TO 50
               ENDIF
 52         CONTINUE

* if usetuv=true, but no TUV output exists for species (j)
            print*,'no TUV output found for:',xlabel(j) 
            GO TO 50
         ENDIF

* otherwise, jblock-based cross sections and quantum yields
* parse label instructions:

         xsp1(ir)   = xlabel(j)(13:16)
         xsp2(ir)   = xlabel(j)(21:24)
         xsp3(ir)   = xlabel(j)(29:32)
         xqy(ir)    = xlabel(j)(42:45)//xlabel(j)(48:48)
         xchan(ir)  = xlabel(j)(48:48)
         xqpflg(ir) = xlabel(j)(37:37)
         xqtflg(ir) = xlabel(j)(39:39)

* read real number fields:
           READ(XLABEL(J),100) XCOEF1(IR), XCOEF2(IR), XCOEF3(IR)
 100       FORMAT(8x,3(F4.2,4x))

* construct cross section:
         
         DO iw = 1, nw-1
            xsect(ir,iw) = 0.
         ENDDO
         DO 54 i = 1, ns
            IF (xsp1(ir) .EQ. slabel(i)) THEN
               DO iw = 1, nw-1
                  xsect(ir,iw) = xsect(ir,iw) + xcoef1(ir)*xs(i,iw)
               ENDDO
            ENDIF
            IF (xsp2(ir) .EQ. slabel(i)) THEN
               DO iw = 1, nw-1
                  xsect(ir,iw) = xsect(ir,iw) + xcoef2(ir)*xs(i,iw)
               ENDDO
            ENDIF
            IF (xsp3(ir) .EQ. slabel(i)) THEN
               DO iw = 1, nw-1
                  xsect(ir,iw) = xsect(ir,iw) + xcoef3(ir)*xs(i,iw)
               ENDDO
            ENDIF
 54      CONTINUE

* construct quantum yield at NTP:

         DO i = 1, nq
            IF (xqy(ir) .EQ. qlabel(i)) THEN
               DO iw = 1, nw-1
                  qyield(ir,iw) = qy(i,iw)*xscale(j)
               ENDDO
            ENDIF
         ENDDO

 50   CONTINUE
      IF (nofind) STOP

********* Correct quantum yields for pressure dependence:
* first compute total quantum yield at room-pressure (NTP) for multiple channels

      DO 60 ir = 1, nr
         IF(usetuv(jfind(ir))) GO TO 60
         DO iw = 1, nw-1
            qytntp(ir,iw) = 0.
         ENDDO
         DO 62 jr = 1, nr
            IF(usetuv(jfind(jr))) GO TO 62
            IF(molec(ir)(1:4) .EQ. molec(jr)(1:4)) THEN
               IF(csplit(jr) .EQ. '>' .OR. csplit(jr) .EQ. '1') THEN
                  DO iw = 1, nw-1
                     qytntp(ir,iw) = qytntp(ir,iw) + qyield(jr,iw)
                  ENDDO
               ENDIF
            ENDIF
 62      CONTINUE
 60   CONTINUE

* apply pressure correction, retaining same branching ratios:

      DO 64 ir = 1, nr
         IF(usetuv(jfind(ir))) GO TO 64
         IF(xqpflg(ir) .EQ. 'P') THEN
            DO iw = 1, nw-1
               IF(qytntp(ir,iw) .LE. 0.) THEN
                  qyp(ir,iw) = 0.
               ELSE
                  qyp(ir,iw) = (qyield(ir,iw)/qytntp(ir,iw)) /
     $                 (1. + (1./qytntp(ir,iw) - 1.)*airlev/2.45e19)
               ENDIF
            ENDDO
         ELSE
            DO iw = 1, nw-1
               qyp(ir,iw) = qyield(ir,iw)
            ENDDO
         ENDIF

* put safety limit between 0 and 1
* (was never encountered with mm2.2, as of jan 2002.)

         DO iw = 1, nw-1
            qyp(ir,iw) = max(qyp(ir,iw), 0.)
            qyp(ir,iw) = min(qyp(ir,iw), 1.)
         ENDDO

 64   continue

********* compute j-values for block-based qy and xs:

      DO 70 ir = 1, nr
         IF(usetuv(jfind(ir))) GO TO 70
         DO itime = 1, ntime
            valj(ir, itime) = 0.
            DO iw = 1, nw-1
               djdw = xsect(ir,iw)*qyp(ir,iw)*mmfsav(itime,iw)
               valj(ir, itime) = valj(ir, itime) + 
     $              djdw * (wu(iw)-wl(iw))
            ENDDO
         ENDDO
 70   CONTINUE
         
******** adjustment for split reactions:

      DO 80 ir = 1 , nr
         IF (csplit(ir) .EQ. '>') GO TO 80
         IF (csplit(ir) .NE. '1') GO TO 80

* count splits until next > or 1:
* assumes no surprises:

         DO isplit = 1, nr
            IF(csplit(ir+isplit) .EQ. '>' .OR. 
     $           csplit(ir+isplit) .EQ. '1') GO TO 85
         ENDDO
 85      CONTINUE
         nsplit = isplit

* scale by split:

         DO isplit = 1, nsplit
            DO itime = 1, ntime
               valj(ir + isplit - 1, itime) = 
     $              valj(ir + isplit - 1, itime)  / float(nsplit)
            ENDDO
         ENDDO

 80   CONTINUE

*****************

* OUTPUT

* output file

      open(unit=20,file='tmp.jv',status='unknown')

      WRITE(20,*) 'T, Atmospheric Number Density:'
      WRITE(20,*) tlev, airlev

      WRITE(20,*) 'Number of photolysis reactions, times:'
      WRITE(20,*) nr, ntime

      WRITE(20,*) 'Mechanism index map for photolysis reactions:'
      WRITE(20,251) (indexr(ir), ir = 1, nr)
 251  FORMAT(12(I6))

      WRITE(20,*) 'Time (sec):'
      WRITE(20,252) (time(itime), itime = 1, ntime)
 252  FORMAT(7(1PE11.3))

      DO ir = 1, nr
         WRITE(20,*) molec(ir)
         WRITE(20,253) (valj(ir,itime), itime = 1, ntime)
 253     FORMAT(7(1PE11.3))
      ENDDO

      WRITE(*,*) 'done: table created in file tmp.jv'

      END

      SUBROUTINE inter2(ng,xg,yg,n,x,y)
*_______________________________________________________________________
* Maps points n,x,y onto ng-1 intervals xg. Result is yg.
*  trapezoidal average y value for each 
*      grid interval
*  x  is an array of input x values (e.g., wavelength)
*  y  is an array of input y values (e.g., quantum yields, or fluxes)
*       corresponding to the values of x
*  n  is the number of input (x,y) data points
*  xg is the array of x values (e.g.,wavelengths) associated with 
*       the endpoints of the grid intervals
*  yg is the array of y values calculated for each grid interval
*  ng is the number of grid delimiters (one more than the number
*       of grid intervals)
*_______________________________________________________________________
*
*    Edit history:
*
*         01/04/95 - Subroutine has been completely rewritten      -SF-
*                    to order loops and eliminate confusing
*                    GOTOs
*_______________________________________________________________________

      IMPLICIT NONE

* input:
      INTEGER ng, n
      REAL x(n), y(n), xg(ng)

* output:
      REAL yg(ng)

* local:
      REAL area, xgl, xgu
      REAL darea, slope
      REAL a1, a2, b1, b2
      INTEGER ngintv
      INTEGER i, k, jstart
*_______________________________________________________________________

*  test for correct ordering of data, by increasing value of x

      DO 10, i = 2, n
         IF (x(i) .LE. x(i-1)) THEN
            WRITE(*,*)'data not sorted'
            STOP
         ENDIF
   10 CONTINUE     

*  find the integral of each grid interval and use this to 
*  calculate the average y value for the interval      
*  xgl and xgu are the lower and upper limits of the grid interval

      jstart = 1
      ngintv = ng - 1
      DO 50, i = 1,ngintv

*  if grid interval is outside data range, set yg to zero      

         IF ((xg(i).LT.x(1)) .OR. (xg(i+1).GT.x(n))) THEN
            yg(i) = 0.

         ELSE

* initalize:

            area = 0.0
            xgl = xg(i)
            xgu = xg(i+1)

*  discard data before the first grid interval and after the 
*  last grid interval
*  for internal grid intervals, start calculating area by interpolating
*  between the last point which lies in the previous interval and the
*  first point inside the current interval

            k = jstart
            IF (k .LE. n-1) THEN

*  if both points are before the first grid, go to the next point
   30         CONTINUE
                IF (x(k+1) .LE. xgl) THEN
                   jstart = k - 1
                   k = k+1
                   IF (k .LE. n-1) GO TO 30
                ENDIF


*  if the last point is beyond the end of the grid, complete and go to the next
*  grid
   40         CONTINUE
                 IF ((k .LE. n-1) .AND. (x(k) .LT. xgu)) THEN          

                    jstart = k-1

* compute x-coordinates of increment

                    a1 = MAX(x(k),xgl)
                    a2 = MIN(x(k+1),xgu)

*  if points coincide, contribution is zero

                    IF (x(k+1).EQ.x(k)) THEN
                       darea = 0.e0
                    ELSE
                       slope = (y(k+1) - y(k))/(x(k+1) - x(k))
                       b1 = y(k) + slope*(a1 - x(k))
                       b2 = y(k) + slope*(a2 - x(k))
                       darea = (a2 - a1)*(b2 + b1)/2.
                    ENDIF


*  find the area under the trapezoid from a1 to a2

                    area = area + darea

* go to next point
              
                    k = k+1
                    GO TO 40

                ENDIF

            ENDIF

*  calculate the average y after summing the areas in the interval
            yg(i) = area/(xgu - xgl)

         ENDIF
   50 CONTINUE
*_______________________________________________________________________

      RETURN
      END
