
#include <params.h>      

      module mo_cloud

      use mo_constants, only : pi

      implicit none

      private
      public  :: inimc, inimland, inicld, clouddr, ktop

      save

      integer :: ktop
      real :: rhonot
      real :: t0
      real :: cldmin
      real :: small
      real :: c
      real :: d
      real :: esi
      real :: esw
      real :: nos
      real :: prhonos
      real :: thrpd
      real :: gam3pd
      real :: gam4pd
      real :: rhoi
      real :: rhos
      real :: mcon01
      real :: mcon02
      real :: mcon03
      real :: mcon04
      real :: mcon05
      real :: mcon06
      real :: mcon07
      real :: mcon08
      real, allocatable :: landm(:,:,:)


      real :: &
        cappa, &      ! R/Cp
        gravit, &     ! Gravitational acceleration
        rair, &       ! Gas constant for dry air
        latvap, &     ! latent heat of vaporization for liq. water
        cpair, &      ! Specific heat of dry air
        rhow, &       ! water density (g/cm^3)
        tfh2o        ! freezing point of water at STP (K)

      integer :: &
        k700         ! Level closest to 700mb over ocean.

      contains

      subroutine pcond( qtend   ,ttend   ,omega   ,lat, ip, &
                        q       ,cwat    , t      ,p      ,pdel, &
                        oro     ,cldn    ,cldo    ,cme    ,evapr, &
                        prain   ,rmelt   ,deltat  ,pcflx, &
                        fwaut   ,fsaut   ,fracw   ,fsacw  ,fsaci, &
                        plonl,   platl )
!-----------------------------------------------------------------------
! 	... Calculate the prognostic condensate amount and tendencies
!           P. Rasch and J.E. Kristjansson, April 1997
!-----------------------------------------------------------------------

      use mo_mpi,       only : base_lat, thisnode
      use mo_grid,      only : plev, plevp
      use mo_constants, only : grav => gravit
      use eslookup,     only : hlatv, epsqs, cp, estblf, vqsatd

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
           plonl, &
           platl
      integer, intent(in) :: &
           lat, &
           ip

      real, intent(in) :: deltat               ! time step to advance solution over
      real, intent(in) :: cldn(plonl,plevp)    ! new value of cloud fraction    (fraction)
      real, intent(in) :: cldo(plonl,plevp)    ! old value of cloud fraction    (fraction)
      real, intent(in) :: cwat(plonl,plev)     ! cloud water (kg/kg)
      real, intent(in) :: omega(plonl,plev)    ! vert pressure vel (Pa/s)
      real, intent(in) :: oro(plonl)           ! flag for land/ice/ocean
      real, intent(in) :: p(plonl,plev)        ! pressure          (K)
      real, intent(in) :: pcflx(plonl,plevp)   ! convective precip level by level (kg/m2/s) (DISABLED)
      real, intent(in) :: pdel(plonl,plev)     ! pressure thickness (Pa)
      real, intent(in) :: q(plonl,plev)        ! water vapor (kg/kg)
      real, intent(in) :: qtend(plonl,plev)    ! mixing ratio tend (kg/kg/s)
      real, intent(in) :: t(plonl,plev)        ! temperature       (K)
      real, intent(in) :: ttend(plonl,plev)    ! temp tend         (K/s)

      real, intent(out) :: cme(plonl,plev)      ! rate of cond-evap within the cloud
      real, intent(out) :: evapr(plonl,plev)    ! rate of evaporation of falling precip (1/s)
      real, intent(out) :: prain(plonl,plev)    ! rate of conversion of condensate to precip (1/s)
      real, intent(out) :: rmelt(plonl,plev)    ! heating rate due to precip phase change (K/s) (DISABLED)
!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: cwn_eps   = 5.e-6
      real, parameter :: prain_eps = 5.e-6
      real, parameter :: conke = 1.e-5            ! rate of evaporation of precipitation:
      integer, parameter :: iter = 2              ! #iterations for precipitation calculation

      integer :: i                 ! work variable
      integer :: k                 ! work variable
      integer :: l                 ! work variable

      real :: cldm(plonl)          ! mean cloud fraction over the time step
      real :: cldmax(plonl)        ! max cloud fraction above
      real :: cldnt(plonl,plevp)   ! sane new cloud fraction    (temp , fraction)
      real :: cldot(plonl,plevp)   ! sane old cloud fraction    (temp , fraction)
      real :: coef(plonl)          ! conversion time scale for condensate to rain
      real :: convm(plonl,plev)    ! moistening rate   (kg/kg/s)
      real :: cwm(plonl)           ! cwat mixing ratio at midpoint
      real :: cwn(plonl)           ! cwat mixing ratio at end
      real :: denom                ! work variable
      real :: dqsdp                ! change in sat spec. hum. wrt pressure
      real :: dqsdt                ! change in sat spec. hum. wrt temperature
      real :: wrk1,wrk2,wrk3,wrk4  ! temp variable
      real :: es(plonl)            ! sat. vapor pressure
      real :: fice(plonl)          ! fraction of cwat that is ice
      real :: fracw(plonl,plev)    ! relative importance of collection of liquid by rain
      real :: fsaci(plonl,plev)    ! relative importance of collection of ice by  snow
      real :: fsacw(plonl,plev)    ! relative importance of collection of liquid by snow
      real :: fsaut(plonl,plev)    ! relative importance of ice auto conversion
      real :: fwaut(plonl,plev)    ! relative importance of warm cloud auto conversion
      real :: gamma(plonl)         ! d qs / dT
      real :: iceab(plonl)         ! rate of ice only from above
      real :: icwc                 ! in-cloud water content (kg/kg)
      real :: mincld               ! a small cloud fraction to avoid / zero
      real :: omeps                ! 1 minus epsilon
      real :: omsm                 ! a number just less than unity (for rounding)
      real :: pcme(plonl)          ! provisional condensation minus evaporation 
      real :: pcme1                ! work variable
      real :: precab(plonl)        ! rate of precipitation (kg / (m**2 * s))
      real :: prect(plonl)         ! rate of precipitation including convection (kg / (m**2 * s))
      real :: prprov(plonl)        ! provisional value of precip at btm of layer
      real :: prtmp                ! work variable
      real :: qn(plonl,plev)       ! mixing rat at end of time step ignoring condensate
      real :: qs(plonl)            ! spec. hum. of water vapor
      real :: qsn                  ! work variable
      real :: qsp(plonl,plev)      ! sat pt mixing ratio
      real :: qtl(plonl)           ! tendency which would saturate the grid box in deltat
      real :: qtmp                 ! work variable
      real :: relhum(plonl)        ! relative humidity
      real :: tc                   ! crit temp of transition to ice
      real :: tn(plonl,plev)       ! temp at end of time step ignoring condensate
      real :: tsp(plonl,plev)      ! sat pt temperature
      real :: pol                  ! work variable
      real :: cdt                  ! work variable
      real :: ttmp(plonl)          ! work variable
      real :: esn(plonl)           ! work variable

      omeps  = 1. - epsqs
      mincld = 1.e-10
      omsm   = .99999

!------------------------------------------------------------
! 	... Initialize a few single level fields
!------------------------------------------------------------
      precab(:) = 0.
      prect(:)  = 0.
      iceab(:)  = 0.            ! latent heat of precip above
      cldmax(:) = 0.

!------------------------------------------------------------
! 	... Initialize some multi-level fields
!------------------------------------------------------------
      do k = 1,plev
         qn(:,k) = q(:,k) + deltat*qtend(:,k)
         tn(:,k) = t(:,k) + deltat*ttend(:,k)
      end do

!------------------------------------------------------------
! 	... Add initialization for top layers
!------------------------------------------------------------
      do k = 1,ktop
	 cme(:,k)   = 0.
	 prain(:,k) = 0.
	 evapr(:,k) = 0.
	 rmelt(:,k) = 0.
      end do

!------------------------------------------------------------
! 	... Find the saturation point for the provisional t and q without condensation
!------------------------------------------------------------
      call findsp( qn, tn, p, tsp, qsp, plonl )
Vert_loop : &
      do k = ktop+1,plev
         call vqsatd( t(1,k), p(1,k), es, qs, gamma, plonl )
         do i = 1,plonl
            relhum(i) = q(i,k)/qs(i)
!------------------------------------------------------------
! 	... Sane limits on cloud fractions
!------------------------------------------------------------
            cldnt(i,k) = max( 0.,min( 1.,cldn(i,k),relhum(i) ) )
            cldot(i,k) = max( 0.,min( 1.,cldo(i,k),relhum(i) ) )
!------------------------------------------------------------
! 	... Mean cloud fraction over the time step
!------------------------------------------------------------
            cldm(i) = max( (cldnt(i,k) + cldot(i,k))*0.5,mincld )
!------------------------------------------------------------
! 	... Max cloud fraction above this level
!------------------------------------------------------------
            cldmax(i) = max( cldmax(i),cldm(i) )
         end do
         dqsdp = 0
         do i = 1,plonl
!------------------------------------------------------------
! 	... Fractions of ice at this level
!------------------------------------------------------------
            tc      = t(i,k) - 273.16
            fice(i) = max( 0.,min( -tc*.05,1. ) )
            dqsdt   = cp*gamma(i)/hlatv
!------------------------------------------------------------
! 	... Moistening term
!------------------------------------------------------------
            convm(i,k) = qtend(i,k) - cldm(i)*dqsdt*ttend(i,k) &
                                    + cldm(i)*dqsdp*omega(i,k)
!------------------------------------------------------------
! 	... Amount of water which must be evaporated to change the cloud
!           from fraction cldo to cldn keeping same incloud amount
!           jr this 0.0001 should really be mincld--fix later
!------------------------------------------------------------
            icwc  = cwat(i,k)/max( cldot(i,k),.0001 )
            pcme1 = min( (cldnt(i,k) - cldot(i,k))*icwc/deltat,0. )
!------------------------------------------------------------
!***** THE ABOVE SHOULD BE CHANGED TO ALLOW EITHER EVAP OR COND TO TAKE PLACE
! first guess on q-e, have not yet checked to see that there
! is enough cloud water to support evaporation
!------------------------------------------------------------
            denom   = 1. + cldm(i) * hlatv/cp * dqsdt
            pcme(i) = (cldm(i)*convm(i,k) + pcme1) / denom
!------------------------------------------------------------
! 	... Calculate the cooling due to a phase change of the
!           rainwater from above
!------------------------------------------------------------
            if( t(i,k) >= 273.16 ) then
               rmelt(i,k) = 0.
               iceab(i)   = 0.
            else
               rmelt(i,k) = 0.
            end if
         end do

!------------------------------------------------------------
! 	... Put reasonable bounds on the first guess
!------------------------------------------------------------
         do i = 1,plonl
            qtmp = q(i,k) + ( qtend(i,k) - pcme(i) )*deltat
!------------------------------------------------------------
! 	... qtl has the tend required to bring to saturation
!------------------------------------------------------------
            qtl(i) =  max((qsp(i,k) - qtmp)/deltat,0.)
!------------------------------------------------------------
! 	... Make sure we condense enough to bring it back to saturation
!           this guards (partially) against inconsistent cloud fractions
!           if no cloud, then make sure we evaporate all the water
!------------------------------------------------------------
            if( qtmp > qsp(i,k) ) then
               pcme(i) = pcme(i) + (qtmp - qsp(i,k))/deltat
            else
               if( cldm(i) > mincld ) then
               else
                  pcme(i) = - min( cwat(i,k)/deltat,qtl(i) )
               end if
            end if
            pcme(i) = max( -cwat(i,k)/deltat,pcme(i),-qtl(i) )*omsm
         end do
         do i = 1,plonl
!------------------------------------------------------------
! 	... Provisional value of the cloud water from cond-evap at midpoint of timestep
!------------------------------------------------------------
            cwm(i) = max( cwat(i,k) + pcme(i)*deltat*0.5,0. )
!------------------------------------------------------------
! 	... Provisional value at endpoint 
!------------------------------------------------------------
            cwn(i) = max( cwat(i,k) + pcme(i)*deltat,0. )
!------------------------------------------------------------
! 	... Provisional value of precip
!------------------------------------------------------------
            prain(i,k) = 0.
!------------------------------------------------------------
! 	... Move provisional cond-evap into final location
!------------------------------------------------------------
            cme(i,k) = pcme(i) 
         end do

!------------------------------------------------------------
! 	... Calculate the formation of precip. Since this is a highly nonlinear
!           calculation, we do it iteratively, using values from the midpoint of
!           the time step
!------------------------------------------------------------
iter_loop : &
         do l = 1,iter
            do i = 1,plonl
               prprov(i) = prect(i) + prain(i,k)*pdel(i,k)/grav
            end do

            call findmcnew( ip, lat, k, prprov, t, p, cwm, cldm, cldmax, &
                            fice, oro, coef, fwaut(1,k), fsaut(1,k), &
                            fracw(1,k), fsacw(1,k), fsaci(1,k), plonl )
!------------------------------------------------------------
! 	... Calculate the precip rate
!------------------------------------------------------------
            do i = 1,plonl
               if( cldm(i) > 0. ) then
!------------------------------------------------------------
! 	... First predict the cloud water
!------------------------------------------------------------
                  if( coef(i) > 0. ) then
                     wrk1 =  cwat(i,k) - cme(i,k)/coef(i)
                     wrk4 =  -coef(i)*deltat
                     if( wrk4 > -500. ) then
                        wrk2 =  exp( wrk4 )
                     else
                        wrk2 =  0.
                     end if
                     wrk3 =  cme(i,k)/coef(i)
!                    wrk1 =  wrk1*wrk2 + wrk3
!                    cwn(i) = max( 0.,wrk1 )
                     wrk4 =  wrk1*wrk2 + wrk3
                     if( wrk4 >= cwn_eps*max( abs( wrk1*wrk2 ),abs( wrk3 ) ) ) then
                        cwn(i) = wrk4
                     else
                        cwn(i) = 0.
                     end if
                  else
!                   cwn(i) = max( 0.,cwat(i,k) + cme(i,k)*deltat )
                    wrk1 = cme(i,k)*deltat
                    wrk2 = max( abs( cwat(i,k) ),abs( wrk1 ) )
                    wrk3 = cwat(i,k) + wrk1
                    if( wrk3 >= cwn_eps*wrk2 ) then
                       cwn(i) = wrk3
                    else
                       cwn(i) = 0.
                    end if
                  end if
!------------------------------------------------------------
! 	... Now back out the tendency
!------------------------------------------------------------
!                 prain(i,k) = max( 0.,cme(i,k) - (cwn(i) - cwat(i,k))/deltat )
                  wrk1 = (cwn(i) - cwat(i,k))/deltat
                  wrk2 = max( abs( cme(i,k) ),abs( wrk1 ) )
                  wrk3 = cme(i,k) - wrk1
                  if( wrk3 >= prain_eps*wrk2 ) then
                     prain(i,k) = wrk3
                  else
                     prain(i,k) = 0.
                  end if
               else
                  prain(i,k) = 0.
                  cwn(i)     = 0.
               end if
!------------------------------------------------------------
! 	... Update any remaining  provisional values
!------------------------------------------------------------
               cwm(i) = (cwn(i) + cwat(i,k))*0.5
            end do
         end do iter_loop

!------------------------------------------------------------
! 	... Now have estimates of the condensation-evaporation terms, and
!           the precipitation
!           so given decent estimates of pcme, calculate provisional value
!           of cloud water for evapr calculation
!------------------------------------------------------------
         ttmp(:) = t(:plonl,k) + (ttend(:plonl,k) + rmelt(:plonl,k)) &
                                *deltat + hlatv/cp*deltat*cme(:plonl,k)
         esn(:) = ESTBLF( ttmp(:) )
         do i = 1,plonl
            qtmp      = q(i,k) + (qtend(i,k) - cme(i,k))*deltat
            qsn       = min( epsqs*esn(i)/(p(i,k) - omeps*esn(i)),1. )
            qtl(i)    = max( (qsn - qtmp)/deltat,0. )
            relhum(i) = qtmp/qsn
         end do

!------------------------------------------------------------
! 	... Evaporation of rain
!------------------------------------------------------------
         do i = 1,plonl
             evapr(i,k) = conke*(1. - cldot(i,k))*SQRT( precab(i) ) &
                               *(1. - min( relhum(i),1. ))
!------------------------------------------------------------
! 	... Limit the evaporation to the amount which is entering the box
!           or saturates the box
!------------------------------------------------------------
            prtmp      = precab(i)*grav/pdel(i,k)
            evapr(i,k) = min( evapr(i,k),prtmp,qtl(i) )*omsm
         end do

!------------------------------------------------------------
! 	... Precipitation 
!------------------------------------------------------------
         do i = 1,plonl
           prtmp     = pdel(i,k) / grav *(prain(i,k) - evapr(i,k))
           iceab(i)  = iceab(i) + fice(i)*prtmp
           precab(i) = precab(i) + prtmp
           prect(i)  = prect(i) + prtmp + pcflx(i,k+1)
           if( abs( precab(i) ) < 1.e-10 ) then
             precab(i) = 0.
           end if
           if( abs( prect(i) ) < 1.e-10 ) then
             prect(i) = 0.
           end if
         end do
      end do Vert_loop

      end subroutine pcond

      subroutine findmcnew( ip, lat, k, precab, t, &
                            p, cwm, cldm, cldmax, fice, &
                            oro, coef, fwaut, fsaut, fracw, &
                            fsacw, fsaci, plonl )
!------------------------------------------------------------
! 	... Calculate the conversion of condensate to precipitate
!           written by phil rasch april 1997
!------------------------------------------------------------

      use MO_GRID, only : plev, plevp

      implicit none

!------------------------------------------------------------
! 	... Dummy arguments
!------------------------------------------------------------
      integer, intent(in) :: &
              k, &              ! level index 
              ip, &             ! long index
              lat, &            ! lat index
              plonl             ! lon tile dim

      real, dimension(plonl), intent(in) :: &
           precab, &            ! rate of precipitation from above (kg / (m**2 * s))
           cldm, &              ! cloud fraction
           cldmax, &            ! max cloud fraction above this level
           cwm, &               ! condensate mixing ratio (kg/kg)
           fice, &              ! fraction of cwat that is ice
           oro                  ! oro flag (1 is land)
      real, dimension(plonl,plev), intent(in) :: &
           t, &                 ! temperature       (K)
           p                    ! pressure          (Pa)

      real, dimension(plonl), intent(out) :: &
           coef, &              ! conversion rate (1/s)
           fwaut, &             ! relative importance of liquid autoconversion (a diagnostic)
           fsaut, &             ! relative importance of ice autoconversion (a diagnostic)
           fracw, &             ! relative  importance of rain accreting liquid (a diagnostic)
           fsacw, &             ! relative  importance of snow accreting liquid (a diagnostic)
           fsaci                ! relative  importance of snow accreting ice (a diagnostic)

!------------------------------------------------------------
! 	... Local variables
!------------------------------------------------------------
      integer :: i
      integer :: ii
      integer :: ind(plonl)
      integer :: nlons

      real :: alpha                ! ratio of 3rd moment radius to 2nd
      real :: capc                 ! constant for autoconversion
      real :: capn                 ! local cloud particles / cm3
      real :: capnc                ! cold and oceanic cloud particles / cm3
      real :: capnw                ! warm continental cloud particles / cm3
      real :: ciaut                ! coefficient of autoconversion of ice (1/s)
      real :: ciautb               ! coefficient of autoconversion of ice (1/s)
      real :: cldloc(plonl)        ! non-zero amount of cloud
      real :: cldpr(plonl)         ! assumed cloudy volume occupied by rain and cloud
      real :: con1                 ! work constant
      real :: con2                 ! work constant
      real :: convfw               ! constant used for fall velocity calculation
      real :: cracw                ! constant used for rain accreting water
      real :: critpr               ! critical precip rate collection efficiency changes
      real :: csacx                ! constant used for snow accreting liquid or ice
      real :: dtice                ! interval for transition from liquid to ice
      real :: effc                 ! collection efficiency
      real :: icemr(plonl)         ! in-cloud ice mixing ratio
      real :: icrit                ! threshold for autoconversion of ice
      real :: icritc               ! threshold for autoconversion of cold ice
      real :: icritw               ! threshold for autoconversion of warm ice
      real :: kconst               ! const for terminal velocity (stokes regime)
      real :: liqmr(plonl)         ! in-cloud liquid water mixing ratio
      real :: pracw                ! rate of rain accreting water
      real :: prlloc(plonl)        ! local rain flux in mm/day
      real :: prscgs(plonl)        ! local snow amount in cgs units
      real :: psaci                ! rate of collection of ice by snow (lin et al 1983)
      real :: psacw                ! rate of collection of liquid by snow (lin et al 1983)
      real :: psaut                ! rate of autoconversion of ice condensate
      real :: ptot                 ! total rate of conversion
      real :: pwaut                ! rate of autoconversion of liquid condensate
      real :: r3l                  ! volume radius
      real :: r3lcrit              ! critical radius at which autoconversion become efficient
      real :: rainmr(plonl)        ! in-cloud rain mixing ratio
      real :: rat1                 ! work constant
      real :: rat2                 ! work constant
      real :: rdtice               ! recipricol of dtice
      real :: rho(plonl)           ! density (mks units)
      real :: rhocgs               ! density (cgs units)
      real :: snowfr               ! fraction of precipate existing as snow
      real :: totmr(plonl)         ! in-cloud total condensate mixing ratio
      real :: vfallw               ! fall speed of precipitate as liquid
      real :: wp                   ! weight factor used in calculating pressure dep of autoconversion
      real :: wt                   ! fraction of ice

!------------------------------------------------------------------------------
!     	... Statement functions
!------------------------------------------------------------------------------
      real :: a1, a2
      real :: HEAVYM
      HEAVYM(a1,a2) = max( .01,SIGN(1.,a1-a2) )  ! modified heavyside function

!------------------------------------------------------------------------------
! 	... Critical precip rate at which we assume the collector drops can change the
!           drop size enough to enhance the auto-conversion process (mm/day)
!------------------------------------------------------------------------------
      critpr = .5
      convfw = 1.94*2.13*SQRT( rhow*1000.*9.81*2.7e-4 )

!------------------------------------------------------------------------------
! 	... Liquid microphysics
!------------------------------------------------------------------------------
      cracw = .884*SQRT( 9.81/(rhow*1000.*2.7e-4) ) ! tripoli and cotton

!------------------------------------------------------------------------------
! 	... Ice microphysics
!------------------------------------------------------------------------------
      ciautb = 5.e-4
      icritw = 4.e-4
      icritc = 5.e-6

      dtice  = 20.
      rdtice = 1./dtice
      capnw  = 400.              ! warm continental cloud particles / cm3
      capnc  =  80.              ! cold and oceanic cloud particles / cm3
      capnc  = 150.              ! cold and oceanic cloud particles / cm3
      kconst = 1.18e6            ! const for terminal velocity
      effc   = 0.55              ! autoconv collection efficiency following tripoli and cotton
      alpha  = 1.1**4
      capc   = pi**(-.333)*kconst*effc*(.75)**(1.333)*alpha     ! constant for autoconversion
      r3lcrit = 5.e-6           ! 5u  crit radius where liq conversion begins

!------------------------------------------------------------------------------
! 	... Find all the points where we need to do the microphysics
!           and set the output variables to zero
!------------------------------------------------------------------------------
      nlons = 0
      do i = 1,plonl
         coef(i) = 0.
         fwaut(i) = 0.
         fsaut(i) = 0.
         fracw(i) = 0.
         fsacw(i) = 0.
         fsaci(i) = 0.
         if( cwm(i) > 1.e-20 ) then
            nlons = nlons + 1
            ind(nlons) = i
         end if
      end do

#ifdef CRAY
!dir  ivdep
#endif
      do ii = 1,nlons
         i = ind(ii)
!------------------------------------------------------------------------------
! 	... Local cloudiness at this level
!------------------------------------------------------------------------------
         cldloc(i) = max( cldmin,cldm(i) )
!------------------------------------------------------------------------------
! 	... Weighted mean between max cloudiness above, and this layer
!------------------------------------------------------------------------------
         cldpr(i) = max( cldmin,(cldmax(i) + cldm(i))*.5 )
!------------------------------------------------------------------------------
! 	... Decompose the suspended condensate into 
!           an incloud liquid and ice phase component
!------------------------------------------------------------------------------
         totmr(i) = cwm(i)/cldloc(i)
         icemr(i) = totmr(i)*fice(i)
         liqmr(i) = totmr(i)*(1. - fice(i))
!------------------------------------------------------------------------------
! 	... Density
!------------------------------------------------------------------------------
         rho(i) = p(i,k)/(287.*t(i,k))
         rhocgs = rho(i)*1.e-3     ! density in cgs units
!------------------------------------------------------------------------------
! 	... Decompose the precipitate into a liquid and ice phase 
!------------------------------------------------------------------------------
         if( t(i,k) > t0 ) then
            vfallw    = convfw/SQRT( rho(i) )
            rainmr(i) = precab(i)/(rho(i)*vfallw*cldpr(i))
            snowfr    = 0.
         else
            snowfr    = 1.
            rainmr(i) = 0.
         end if
!------------------------------------------------------------------------------
! 	... Local snow amount in cgs units
!------------------------------------------------------------------------------
         prscgs(i) = precab(i)/cldpr(i)*0.1*snowfr
!------------------------------------------------------------------------------
! 	... Local rain amount in mm/day
!------------------------------------------------------------------------------
         prlloc(i) = precab(i)*86400./cldpr(i)
      end do

      con1 = 1./(1.333*pi)**0.333 * 0.01 ! meters
!------------------------------------------------------------------------------
! 	... Calculate the conversion terms
!------------------------------------------------------------------------------
#ifdef CRAY
!dir  ivdep
#endif
      do ii = 1,nlons
         i = ind(ii)
         rhocgs = rho(i)*1.e-3     ! density in cgs units
!------------------------------------------------------------------------------
! 	... Exponential temperature factor
!           efact = exp(0.025*(t(i,k)-t0))
!           some temperature dependent constants
!------------------------------------------------------------------------------
         wt = min( 1.,max( 0.,(t0 - t(i,k))*rdtice ) )
         icrit = icritc*wt + icritw*(1. - wt)
!------------------------------------------------------------------------------
! 	... Linear weight factor in pressure (1 near sfc, 0 at .8 of sfc) 
!------------------------------------------------------------------------------
         wp = min( 1.,max( 0.,(p(i,k) - .8*p(i,plev))/(.2*p(i,plev)) ) )
!------------------------------------------------------------------------------
! 	... Near land near sfc raise the number concentration
!------------------------------------------------------------------------------
         capn =  landm(i,lat,ip)*(capnw*wp + capnc*(1. - wp)) &
                 + (1. - landm(i,lat,ip))*capnc
!------------------------------------------------------------------------------
! 	... Useful terms in following calculations
!------------------------------------------------------------------------------
         rat1 = rhocgs/rhow
         rat2 = liqmr(i)/capn
         con2 = (rat1*rat2)**0.333 
!------------------------------------------------------------------------------
! 	... Volume radius
!------------------------------------------------------------------------------
         r3l = con1*con2 
!------------------------------------------------------------------------------
! 	... Critical threshold for autoconversion if modified for mixed phase
!           clouds to mimic a bergeron findeisen process
!           r3lc2 = r3lcrit*(1.-0.5*fice(i)*(1-fice(i)))
!------------------------------------------------------------------------------
!
!------------------------------------------------------------------------------
!      	... Autoconversion of liquid
!           pwaut is following tripoli and cotton (and many others)
!           we reduce the autoconversion below critpr, because these are regions where
!           the drop size distribution is likely to imply much smaller collector drops than
!           those relevant for a cloud distribution corresponding to the value of effc = 0.55 
!           suggested by cotton (see austin 1995 JAS, baker 1993)
!------------------------------------------------------------------------------
         pwaut = capc*liqmr(i)**2*rat1*con2 &
                     *HEAVYM( r3l,r3lcrit ) &
                     *max( .1,min( 1.,prlloc(i)/critpr ) )
!------------------------------------------------------------------------------
! 	... Autoconversion of ice
!------------------------------------------------------------------------------
         ciaut = ciautb
!------------------------------------------------------------------------------
! 	... Autoconversion of ice condensate
!------------------------------------------------------------------------------
         psaut = max( 0.,icemr(i) - icrit )*ciaut
!------------------------------------------------------------------------------
! 	... Collection of liquid by rain 
!------------------------------------------------------------------------------
         pracw = cracw*rho(i)*SQRT( rho(i) )*liqmr(i)*rainmr(i) !(tripoli and cotton)
!------------------------------------------------------------------------------
! 	... The following lines calculate the slope parameter and snow mixing ratio
!           from the precip rate using the equations found in lin et al 83
!           in the most natural form, but it is expensive, so after some tedious
!           algebraic manipulation you can use the cheaper following form
!            vfalls = c*gam4pd/(6*lamdas**d)*sqrt(rhonot/rhocgs)
!                     *0.01   ! convert from cm/s to m/s
!            snowmr(i) = snowfr*precab(i)/(rho(i)*vfalls*cldpr(i))
!            snowmr(i) = ( prscgs(i)*mcon02 * (rhocgs**mcon03) )**mcon04
!            lamdas = (prhonos/max(rhocgs*snowmr(i),small))**0.25
!            csacw = mcon01*sqrt(rhonot/rhocgs)/(lamdas**thrpd) 
!
!           coefficient for collection by snow independent of phase
!------------------------------------------------------------------------------
         csacx = mcon07*rhocgs**mcon08*prscgs(i)**mcon05
!------------------------------------------------------------------------------
! 	... Collection of liquid by snow (lin et al 1983)
!------------------------------------------------------------------------------
         psacw = csacx*liqmr(i)*esw
!------------------------------------------------------------------------------
! 	... Collection of ice by snow (lin et al 1983)
!------------------------------------------------------------------------------
         psaci = csacx*icemr(i)*esi
!------------------------------------------------------------------------------
! 	... Total conversion of condensate to precipitate 
!------------------------------------------------------------------------------
         ptot = pwaut + psaut + pracw + psacw + psaci
!------------------------------------------------------------------------------
! 	... Turn the tendency back into a loss rate (1/seconds)
!------------------------------------------------------------------------------
         if( totmr(i) > 0. ) then
            coef(i) = ptot/totmr(i)
         else
            coef(i) = 0.
         end if

         fwaut(i) = pwaut/max( ptot,small )
         fsaut(i) = psaut/max( ptot,small )
         fracw(i) = pracw/max( ptot,small )
         fsacw(i) = psacw/max( ptot,small )
         fsaci(i) = psaci/max( ptot,small )
      end do

      end subroutine findmcnew

      subroutine findsp( q, t, p, tsp, qsp, plonl )
!------------------------------------------------------------------------------
!   	... Find the saturation point for a given t and q
!           if q > qs(t) then tsp > t and qsp = qs(tsp) < q
!           if q < qs(t) then tsp < t and qsp = qs(tsp) > q
!------------------------------------------------------------------------------

      use MO_GRID,  only : plev, plevp
      use ESLOOKUP, only : hlatf, hlatv, pcf, rgasv, epsqs, ttrice, cp, ESTBLF

      implicit none

!------------------------------------------------------------------------------
!  	... Dummy arguments
!------------------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, dimension(plonl,plev), intent(in) :: &
           q, &                 ! water vapor (kg/kg)
           t, &                 ! temperature (K)
           p                    ! pressure    (Pa)

      real, dimension(plonl,plev), intent(out) :: &
           tsp, &               ! saturation temp (K)
           qsp                  ! saturation mixing ratio (kg/kg)

!------------------------------------------------------------------------------
! 	... Local variables
!------------------------------------------------------------------------------
      integer, parameter :: iter_max = 10
      real, parameter    :: eps1 = 1.e-4, eps2 = 1.e-4
      integer :: i                 ! work variable
      integer :: k                 ! work variable
      integer :: l                 ! work variable

      real :: t1, q1, dt, dq
      real :: dtm, dqm
      real :: qvd, a1, tmp
      real :: rair
      real :: r1b, c1, c2, c3
      real :: omeps                ! 1 minus epsilon
      real :: trinv                ! work variable
      real :: desdt                ! change in sat vap pressure wrt temperature
      real :: dqsdt                ! change in sat spec. hum. wrt temperature
      real :: dgdt                 ! work variable
      real :: g                    ! work variable
      real :: tterm                ! work var.
      real :: tc                   ! crit temp of transition to ice
      real :: hltalt               ! lat. heat. of vap.
      real :: es(plonl)            ! sat. vapor pressure
      real :: qs(plonl)            ! spec. hum. of water vapor
      real :: hlatsb               ! (sublimation)

      logical :: converged

      omeps = 1. - epsqs
      trinv = 1./ttrice
      a1    = 7.5*log( 10. )
      rair  =  287.04
      c3    = rair*a1/cp

!------------------------------------------------------------------------------
! 	... Do not calculate saturation above ktop
!------------------------------------------------------------------------------
      do k = 1,ktop
	 tsp(:,k) = t(:,k)
	 qsp(:,k) = 1.
      end do

Vert_loop : &
      do k = ktop+1,plev
!------------------------------------------------------------------------------
! 	... First guess on the saturation point
!------------------------------------------------------------------------------
         es(:) = ESTBLF( t(:,k) )
!------------------------------------------------------------------------------
! 	... Saturation specific humidity
!------------------------------------------------------------------------------
         qs(:) = min( epsqs*es(:)/(p(:,k) - omeps*es(:)),1. )
         do i = 1,plonl
            if( qs(i) < 0. ) then
               qsp(i,k) = 1.
               tsp(i,k) = t(i,k)
            else
!------------------------------------------------------------------------------
! 	... "Generalized" analytic expression for t derivative of es
!           accurate to within 1 percent for 173.16 < t < 373.16
!
!          Weighting of hlat accounts for transition from water to ice
!          polynomial expression approximates difference between es over
!          water and es over ice from 0 to -ttrice (C) (min of ttrice is
!          -40): required for accurate estimate of es derivative in transition 
!          range from ice to water also accounting for change of hlatv with t 
!          above 273.16 where const slope is given by -2369 j/(kg c) = cpv - cw
!------------------------------------------------------------------------------
               tc = t(i,k) - 273.16
               if( t(i,k) < 273.16 ) then
                  hltalt = hlatv + min( -tc*trinv,1. )*hlatf
               else
                  hltalt = hlatv - 2369.0*tc
               end if
               tmp      =  q(i,k) - qs(i)
               c1       = hltalt*c3
               c2       = (t(i,k) + 36.)**2
               r1b      = c2/(c2 + c1*qs(i))
               qvd      = r1b*tmp
               tsp(i,k) = t(i,k) + ((hltalt/cp)*qvd)
            end if
         end do
         es(:) = ESTBLF( tsp(:,k) )
	 where( qs(:) >= 0. )
            qsp(:,k) = min( epsqs*es(:)/(p(:,k) - omeps*es(:)),1. )
	 endwhere

!------------------------------------------------------------------------------
! 	... Iterate on first guess
!------------------------------------------------------------------------------
#ifndef NEC
Long_loop : &
         do i = 1,plonl
            if( qsp(i,k) < 1. ) then
	       converged = .false.
Iter_loop : &
               do l = 1,iter_max
#else
	 converged = .false.
         do l = 1,iter_max
            dtm = 0.
            dqm = 0.
            do i = 1,plonl
               if( qsp(i,k) < 1. ) then
#endif
                  es(i:i) = ESTBLF( tsp(i:i,k) )
!------------------------------------------------------------------------------
! 	... Saturation specific humidity
!------------------------------------------------------------------------------
                  qs(i) = min( epsqs*es(i)/(p(i,k) - omeps*es(i)),1. )
!------------------------------------------------------------------------------
! 	... "generalized" analytic expression for t derivative of es
!           accurate to within 1 percent for 173.16 < t < 373.16
!
!           Weighting of hlat accounts for transition from water to ice
!           polynomial expression approximates difference between es over
!           water and es over ice from 0 to -ttrice (C) (min of ttrice is
!           -40): required for accurate estimate of es derivative in transition 
!           range from ice to water also accounting for change of hlatv with t 
!           above 273.16 where const slope is given by -2369 j/(kg c) = cpv - cw
!------------------------------------------------------------------------------
                  tc = tsp(i,k) - 273.16
                  if( tsp(i,k) < 273.16 ) then
                     hltalt = hlatv + min( -tc*trinv,1. )*hlatf
                  else
                     hltalt = hlatv - 2369.0*tc
                  end if
                  if( tc >= ttrice .and. tc < 0. ) then
                     tterm = pcf(1) + tc*(pcf(2) + tc*(pcf(3) + tc*(pcf(4) + tc*pcf(5))))
                  else
                     tterm = 0.
                  end if
!------------------------------------------------------------------------------
! 	... The following check is to avoid the generation of negative
!           values that can occur in the upper stratosphere and mesosphere
!------------------------------------------------------------------------------
                  if( qs(i) < 0. ) then
                     es(i) = p(i,k)
                     dqsdt = 0.
                     qs(i) = q(i,k)
                  else
                     desdt = hltalt*es(i)/(rgasv*tsp(i,k)*tsp(i,k)) + tterm*trinv
                     dqsdt = (epsqs + omeps*qs(i))/(p(i,k) - omeps*es(i))*desdt
                  end if
                  g        = cp*(t(i,k) - tsp(i,k)) + hltalt*(q(i,k) - qsp(i,k))
                  dgdt     = -(cp + hltalt*dqsdt)
                  t1       = tsp(i,k) - g/dgdt
                  dt       = abs( t1 - tsp(i,k) )/t1
                  tsp(i,k) = t1
                  es(i:i)  = ESTBLF( tsp(i:i,k) )
                  q1       = min( epsqs*es(i)/(p(i,k) - omeps*es(i)),1. )
                  dq       = abs( q1 - qsp(i,k) )/max( q1,1.e-12 )
                  qsp(i,k) = q1
#ifdef NEC
                  dtm      = max( dtm,dt )
                  dqm      = max( dqm,dq )
               end if
#else
                  if( dt < eps1 .and. dq < eps2 ) then
	             converged = .true.
                     exit
                  end if
               end do Iter_loop
               if( .not. converged ) then
                  write(*,*) 'FINDSP: not converging ', dt, dq, i, k
               end if
            end if
#endif
#ifdef NEC
            end do
            if( dtm < eps1 .and. dqm < eps2 ) then
	       converged = .true.
               exit
            end if
         end do
         if( .not. converged ) then
            write(*,*) 'FINDSP: not converging ', dtm, dqm, k
         end if
#endif
#ifndef NEC
      end do Long_loop
#endif
      end do Vert_loop

      end subroutine findsp

      subroutine cldfrc( pmid    ,rpdeli  ,temp    ,q       ,omga, &
                         cldtop  ,cldbot  ,cloud   ,clc     ,pdel, &
                         cmfmc   ,oro     ,snowh   ,concld  ,cldst, &
                         ts      ,ps      ,zdu     ,lat     ,plonl )
!-----------------------------------------------------------------------
! 	... Compute cloud fraction using scheme of J.M.Slingo, 
!           as modified by J.J.Hack and J.T.Kiehl
!
!           This scheme is based on the operational scheme used in the ECMWF model
!           A full description of its development can be found in Slingo (1987),
!           which appears in the QJRMS July issue.  A number of modifications have
!           been introduced to the original scheme in the following implementation 
!-----------------------------------------------------------------------

      use MO_GRID,  only : plev, plevp
      use ESLOOKUP, only : AQSAT

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lat
      integer, intent(in) :: plonl
      real, dimension(plonl,plev), intent(in) :: &
           pmid, &               ! midpoint pressures
           rpdeli, &             ! 1./(pmid(k+1)-pmid(k))
           temp, &               ! temperature
           q, &                  ! specific humidity
           omga, &               ! vertical pressure velocity
           cmfmc, &              ! convective mass flux--m sub c
           pdel, &               ! pressure depth of layer
           zdu                   ! detrainment rate from deep convection
      real, dimension(plonl), intent(in) :: &
           cldtop, &             ! top level of convection
           cldbot, &             ! bottom level of convection
           snowh, &              ! snow depth (liquid water equivalent)
           oro, &                ! Land/ocean/seaice flag
           ts, &                 ! surface temperature
           ps                    ! surface pressure

      real, intent(out) :: &
           cloud(plonl,plevp), & ! cloud fraction
           clc(plonl), &         ! column convective cloud amount
           cldst(plonl,plev)     ! cloud fraction

      real, dimension(plonl,plev), intent(out) :: &
           concld                ! convective cloud cover

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: pnot = 1.e5              ! reference pressure

      integer :: i,k                ! longitude, level indices
      integer :: kp1
      integer :: kc
      integer :: kdthdp(plonl)

      real :: &
           dmudp, &              ! measure of mass detraining in a layer
           dthdp, &              ! lapse rate (intermediate variable)
           premib, &             ! bottom pressure bound of middle cloud
           pretop, &             ! pressure bounding high cloud
           rhb, &                ! intermediate scratch variable
           rhdif, &              ! intermediate scratch variable
           strat, &              ! intermediate scratch variable
           zrth, &               ! reciprocal of no. of convective layers
           bvf, &                ! brunt-vaisalla frequency
           rbvflim, &            ! bound on inverse of bvf
           rho, &                ! local density (used to calculate bvf)
           rhlim, &              ! local rel. humidity threshold estimate 
           rhden, &              ! intermediate scratch variable
           rhdif2, &             ! intermediate scratch variable
           pdepth, &             ! intermediate scratch variable
           stratfac, &           ! intermediate scratch variable
           rhminl, &             ! minimum rh for low stable clouds
           rhminh, &             ! minimum rh for high stable clouds
           coef1                 ! coefficient to convert mass flux to mb/d
      real, dimension(plonl,plev) :: &
           dthtdp, &             ! lapse rate (d theta/dp) below 750 mb
           es, &                 ! saturation vapor pressure
           qs, &                 ! saturation specific humidity
           rh, &                 ! relative humidity
           theta                 ! potential temperature
      real :: cld                ! intermediate scratch variable (low cld)
                                 ! random overlap in convective layer)
      real, dimension(plonl) :: &
           dtdpmn, &             ! most stable lapse rate below 750 mb
           cld8, &               ! low cloud fraction estimate
           cld9, &               ! mid and high cloud fraction estimate
           cck, &                ! convective cloud per level (assuming
           mcbar, &              ! mean convective scale motion in column
           dpsum, &              ! vertical sum of delta-p (k-1 levels)
           ccldt, &              ! estimate of total convective cloud
           clrsky, &             ! temporary used in random overlap calc
           thetas       

      logical, dimension(plonl) :: &
              lol, &             ! region of low level cloud
              cldbnd             ! region below high cloud boundary


!-----------------------------------------------------------------------
! 	... Statement functions
!-----------------------------------------------------------------------
      logical :: LAND
      LAND(i) = NINT( oro(i) ) == 1

!-----------------------------------------------------------------------
! 	... Set bound for inverse of brunt-vaisalla frequency and minimum relative
!           humidity thresholds for stable clouds.  These are the principal 
!           "disposable" parameters for the cloud fraction scheme
!-----------------------------------------------------------------------
      rbvflim = 1./.00035
      rhminl = .9
      rhminh = .9

!-----------------------------------------------------------------------
! 	... Evaluate potential temperature and relative humidity
!-----------------------------------------------------------------------
      call aqsat( temp, pmid, es, qs, plonl, & 
                  plonl, plev, 1, plev )
      do k = 1,plev
        do i = 1,plonl
          theta(i,k)  = temp(i,k)*(pnot/pmid(i,k))**cappa
          rh(i,k)     = q(i,k)/qs(i,k)
          cloud(i,k)  = 0.
          cldst(i,k)  = 0.
          concld(i,k) = 0.
        end do
      end do
      cloud(:,plevp)  = 0.

!-----------------------------------------------------------------------
! 	... Initialize other temporary variables
!-----------------------------------------------------------------------
      do i = 1,plonl
        thetas(i)  = ts(i)*(pnot/ps(i))**cappa
        cck(i) = 0.    
        clc(i) = 0.    
      end do
      coef1 = gravit*864.     ! conversion to millibars/day
      do i = 1,plonl
        mcbar(i) = 0. 
        dpsum(i) = 0. 
      end do

!-----------------------------------------------------------------------
! 	... Calculate mean convective motion throughout column (in units of mb/day)
!-----------------------------------------------------------------------
      do k = 1,plev-1
        do i = 1,plonl
          mcbar(i) = mcbar(i) + max( cmfmc(i,k+1)*coef1,0. )*pdel(i,k)
          dpsum(i) = dpsum(i) + pdel(i,k)
        end do
      end do

!-----------------------------------------------------------------------
! 	... Estimate of total convective cloud cover based on mean convective motion
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! 	... Make the convective cloud depend on the conv. mass detraining
!           for upper levels only (above 500mb), since Xu and Kreuger showed
!           rh is a very poor predictor of those clouds
!-----------------------------------------------------------------------
      do k = 1,plev-1
         do i = 1,plonl
            if( pmid(i,k) < 5.e4 ) then
               concld(i,k) = min( rh(i,k),MIN( 1.,max( 0.,zdu(i,k)*5.e4 ) ) )
            endif
         end do
      end do   

!-----------------------------------------------------------------------
! 	... Evaluate effective column-integrated convective cloud cover using
!           random overlap assumption (for diagnostic purposes only)
!-----------------------------------------------------------------------
      clrsky(:plonl) = 1.
      do k = plev,1,-1
         clrsky(:plonl) = clrsky(:plonl)*(1. - concld(:plonl,k))
      end do
      clc(:plonl) = 1. - clrsky(:plonl)

!-----------------------------------------------------------------------
!    	... Compute layer cloudiness
! 	There is effecively no top for high cloud formation
!       (all the way up to 1mb)
! 	The bottom of middle level cloud (or the top of low level cloud) is
! 	arbitrarily define to be 750 mb (premib)
!-----------------------------------------------------------------------
      premib = 750.e2
      pretop = 1.0e2                 ! top of cloud layer is at 1 mb
!-----------------------------------------------------------------------
! 	... Find most stable level below 750 mb for evaluating stratus regimes
!-----------------------------------------------------------------------
      do i = 1,plonl
        dtdpmn(i)   = 0.
        kdthdp(i)   = 0
        dthtdp(i,1) = 0.
      end do
      do k = 2,plev-2
        do i = 1,plonl
          if( pmid(i,k) >= premib ) then
            dthdp = 100.*(theta(i,k) - theta(i,k-1))*rpdeli(i,k-1)
          else
            dthdp = 0.
          end if
          if( dthdp < dtdpmn(i) ) then
            dtdpmn(i) = dthdp
            kdthdp(i) = k     ! index of interface of max inversion
          end if
          dthtdp(i,k) = dthdp
        end do
      end do
      do k = plev-1,plev
        do i = 1,plonl
          if( 0. < dtdpmn(i) ) then
            dtdpmn(i) = 0.
          end if
          dthtdp(i,k) = 0.
        end do
      end do

!-----------------------------------------------------------------------
! 	... bvf => brunt-vaisalla frequency (approx. 1-sided diff.)
!           this stability measure is used to set a local relative humidity 
!           threshold when evaluating the fractional area of layered cloud
!-----------------------------------------------------------------------
      do k = 2,plev
        kp1 = min( k + 1,plev )
        do i = 1,plonl
          if( dthtdp(i,k) > dtdpmn(i) ) then
            dthtdp(i,k) = 0.
          end if
          cldbnd(i) = pmid(i,k) >= pretop
          lol(i) = pmid(i,k) >= premib
          rho = pmid(i,k)/(rair*temp(i,k))
          bvf = -rho*gravit*gravit*((theta(i,k)-theta(i,k-1))*rpdeli(i,k-1))/theta(i,k)
          if( cldbnd(i) ) then
            rhlim = .999 - (1. - rhminh)*(1. - min( 1.,max( .0,bvf*rbvflim ) ))
            rhden = 1. - rhlim
          else
            rhlim = .999
            rhden = .001
          end if
          rhdif = (rh(i,k) - rhlim)/rhden
          cld9(i) = min( .999,(max( rhdif,0. ))**2 )
!-----------------------------------------------------------------------
! 	... Ignore brunt-vaisalla stability estimate of local relative humidity
!           threshold when evaluating low cloud where local vertical motion is 
!           less than some prescribed value (see low cloud section below)
!           Relative humidity threshold is fixed at rhminl for this case, except
!           over snow-free land, where it is reduced by 10%.  This distinction is
!           made to account for enhanced cloud drop nucleation ({\it i.e.,} at 
!           lower relative humidities) that can occur over CCN rich land areas.
!-----------------------------------------------------------------------
          if( lol(i) ) then
            if( land(i) .and. (snowh(i) <= .000001) ) then
              rhlim = rhminl - .1
            else
              rhlim = rhminl
            endif
            rhdif2 = (rh(i,k) - rhlim)/(1. - rhlim)
            cld8(i) = min( .999,(max( rhdif2,0. ))**2 )
          else
            cld8(i) = cld9(i)
          end if
        end do
!-----------------------------------------------------------------------
!   	... Final evaluation of layered cloud fraction
!-----------------------------------------------------------------------
        do i = 1,plonl
!-----------------------------------------------------------------------
! 	... Low cloud: non-zero only if vertical velocity requirements are satisfied
!           Current vertical velocity threshold is omega < +50 mb/day with a 50 mb/day 
!           linear ramp (other quantities in the class of "disposable" parameters)
!-----------------------------------------------------------------------
          if( lol(i) ) then
            if( omga(i,k) < .05787 ) then
              cld = cld8(i)*min( 1.,max( .0,(.05787 - omga(i,k))/.05787 ) )
            else
              cld = 0.
            end if
            cloud(i,k) = cld
          else                  ! Middle and high level cloud 
            if( cldbnd(i) ) then
              cloud(i,k) = cld9(i)
            else
              cloud(i,k) = 0.
            end if
          end if
        end do
      end do

!-----------------------------------------------------------------------
! 	... Add in the marine strat
!-----------------------------------------------------------------------
         do i = 1,plonl
            k = kdthdp(i)
            kc = max( k,1 )
            kp1 = min( k+1,plev )
            strat = min( 1.,max( 0.,(theta(i,k700) - thetas(i))*.057 - .5573 ) )
!-----------------------------------------------------------------------
! 	... Assign the stratus to the layer just below max inversion
!           the relative humidity changes so rapidly across the inversion
!           that it is not safe to just look immediately below the inversion
!           so limit the stratus cloud by rh in both layers below the inversion
!-----------------------------------------------------------------------
            if( NINT( oro(i) ) == 0 .and. dthtdp(i,kc) <= .125 .and. k /= 0 ) then
               cldst(i,k) = min( strat,max( rh(i,k),rh(i,kp1) ) )
               cloud(i,k) = max( cloud(i,k),cldst(i,k) )
            end if
         end do

!-----------------------------------------------------------------------
! 	... Merge convective and layered cloud fraction for total cloud
!-----------------------------------------------------------------------
      do k = 1,plev
        do i = 1,plonl
!-----------------------------------------------------------------------
! 	... Change to a max overlap assumption between convective and strat clouds
!-----------------------------------------------------------------------
            cloud(i,k) = max( .0,min( .999,max( concld(i,k),cloud(i,k) ) ) )
            if( rh(i,k) > .99 ) then
               cloud(i,k) = max( .01,cloud(i,k) )
            end if
        end do
      end do

      end subroutine cldfrc

      subroutine cldsav( cld, pmid, cldtot, cldlow, cldmed, cldhgh, plonl )
!-----------------------------------------------------------------------
! 	... Compute total & 3 levels of cloud fraction assuming random overlap.
!           These diagnostics must be done on the fly to correctly deal with
!           random overlap. Pressure ranges for the 3 cloud levels are specified.
!-----------------------------------------------------------------------

      use MO_GRID, only : plev, plevp

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in) :: cld(plonl,plevp)    ! Cloud fraction
      real, intent(in) :: pmid(plonl,plev)    ! Level pressures

      real, dimension(plonl), intent(out) :: &
           cldtot, &           ! Total random overlap cloud cover
           cldlow, &           ! Low random overlap cloud cover
           cldmed, &           ! Middle random overlap cloud cover
           cldhgh              ! High random overlap cloud cover

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: plowmax = 120000.   ! Max prs for low cloud cover range
      real, parameter :: plowmin = 70000.    ! Min prs for low cloud cover range
      real, parameter :: pmedmax = 70000.    ! Max prs for mid cloud cover range
      real, parameter :: pmedmin = 40000.    ! Min prs for mid cloud cover range
      real, parameter :: phghmax = 40000.    ! Max prs for hgh cloud cover range
      real, parameter :: phghmin = 5000.     ! Min prs for hgh cloud cover range

      integer :: i,k              ! Longitude,level indices
      real    :: clrsky(plonl)    ! Random overlap clear sky fraction

!-----------------------------------------------------------------------
! 	... Total cloud
!-----------------------------------------------------------------------
      clrsky(:plonl) = 1.
      do k = plev,1,-1
        do i = 1,plonl
          clrsky(i) = clrsky(i)*(1. - cld(i,k))
        end do
      end do
      cldtot(:plonl) = 1. - clrsky(:plonl)

!-----------------------------------------------------------------------
! 	... High cloud (400mb -> 50mb)
!-----------------------------------------------------------------------
      clrsky(:plonl) = 1.
      do k = plev,1,-1
        do i = 1,plonl
          if( phghmin <= pmid(i,k) .and. pmid(i,k) < phghmax ) then
            clrsky(i) = clrsky(i)*(1. - cld(i,k))
          end if
        end do
      end do
      cldhgh(:plonl) = 1. - clrsky(:plonl)

!-----------------------------------------------------------------------
! 	... Medium cloud (700mb -> 400mb)
!-----------------------------------------------------------------------
      clrsky(:plonl) = 1.
      do k = plev,1,-1
        do i = 1,plonl
          if( pmedmin <= pmid(i,k) .and. pmid(i,k) < pmedmax ) then
            clrsky(i) = clrsky(i)*(1. - cld(i,k))
          end if
        end do
      end do
      cldmed(:plonl) = 1. - clrsky(:plonl)

!-----------------------------------------------------------------------
! 	... Low cloud (sfc -> 700mb)
!-----------------------------------------------------------------------
      clrsky(:plonl) = 1.
      do k = plev,1,-1
        do i = 1,plonl
          if( plowmin <= pmid(i,k) .and. pmid(i,k) < plowmax ) then
            clrsky(i) = clrsky(i)*(1. - cld(i,k))
          end if
        end do
      end do
      cldlow(:plonl) = 1. - clrsky(:plonl)
      
      end subroutine cldsav

      subroutine inimc( ref_pmid )
!---------------------------------------------------------------------- 
! 	... Initialize the module variables for the prognostic
!           condensate parameterization
!---------------------------------------------------------------------- 

      use MO_GRID, only : plev

      implicit none

!---------------------------------------------------------------------- 
!	... Dummy arguments
!---------------------------------------------------------------------- 
      real, intent(in) :: ref_pmid(plev)     ! Pa

!---------------------------------------------------------------------- 
!	... Local variables
!---------------------------------------------------------------------- 
      real, parameter :: p_limit = 10.e2     ! findsp pressure limit (Pa)
      integer :: k

#ifdef CRAY
      real :: signgam              ! variable required by cray gamma function
      external gamma
#endif
      rhonot  = 1.275e-3           ! air density at surface (gm/cm3)
      rhos    = .1                 ! assumed snow density (gm/cm3)
      rhoi    = 1.                 ! ice density
      esi     = 1.0                ! collection efficient for ice by snow
      esw     = 0.1                ! collection efficient for water by snow
      t0      = 273.16             ! approximate freezing temp
      cldmin  = 0.02               ! assumed minimum cloud amount 
      small   = 1.e-22             ! a small number compared to unity
      c       = 152.93             ! constant for graupel like snow cm**(1-d)/s
      d       = 0.25               ! constant for graupel like snow
      nos     = 3.e-2              ! particles snow / cm**4
      prhonos = pi*rhos*nos
      thrpd   = 3. + d
#ifdef CRAY
      call gamma( 3.+d, signgam, gam3pd )
      gam3pd = sign( exp(gam3pd),signgam )
      call gamma( 4.+d, signgam, gam4pd )
      gam4pd = sign( exp(gam4pd),signgam )
#else
      if( d == .25 ) then
         gam3pd = 2.549256966718531 ! only right for d = 0.25
         gam4pd = 8.285085141835282
      else
         write(*,*) 'INIMC: can only use d ne 0.25 on a cray'
         call endrun
      end if
#endif
#ifdef DEBUG
      write(*,*) 'INIMC: d, gamma(3+d), gamma(4+d) =', gam3pd, gam4pd
#endif
      mcon01 = pi*nos*c*gam3pd/4.
      mcon02 = 1./(c*gam4pd*SQRT(rhonot)/(6*prhonos**(d/4.)))
      mcon03 = -(0.5 + d/4.)
      mcon04 = 4./(4. + d)
      mcon05 = (3. + d)/(4. + d)
      mcon06 = (3. + d)/4.
      mcon07 = mcon01*SQRT(rhonot)*mcon02**mcon05/prhonos**mcon06
      mcon08 = -0.5/(4. + d)

!---------------------------------------------------------------------- 
!	... Find level where etamids are all > 10 hPa
!---------------------------------------------------------------------- 
      ktop = 0
      if( ref_pmid(1) < p_limit ) then
         do k = 1,plev
	    if( ref_pmid(k) < p_limit ) then
	       ktop = k
	    end if
         end do
      end if
      write(*,*) 'INIMC: prognostic cloud water capped at level ',ktop+1
      write(*,*) '       whose midpoint is ',ref_pmid(ktop+1)*1.e-2,' hPa'
      write(*,*) 'INIMC: complete ' 

      end subroutine inimc

      subroutine inimland( oro, clat, plonl, platl, pplon, nodes )
!---------------------------------------------------------------------- 
!	... Initialize the land mask
!---------------------------------------------------------------------- 

      use mo_grid,      only : plon, plat
      use mo_constants, only : arad => rearth, pi, clon => lam
#ifdef USE_MPI
      use mo_mpi,       only : mpi_double_precision, mpi_comm_comp, mpi_success, &
			       masternode, base_lat
#else
      use mo_mpi,       only : masternode, base_lat 
#endif

      implicit none

!---------------------------------------------------------------------- 
!	... Dummy arguments
!---------------------------------------------------------------------- 
      integer, intent(in) :: plonl, platl, pplon, nodes
      real, intent(in)    :: clat(plat)                ! lat in radians
      real, intent(in)    :: oro(plonl,platl,pplon)

!---------------------------------------------------------------------- 
!	... Local variables
!---------------------------------------------------------------------- 
      real, parameter :: dmax = 2.e6                ! distance to carry the mask

      integer :: i
      integer :: j
      integer :: ii
      integer :: jj
      integer :: iplm1
      integer :: jof
      integer :: iof
      integer :: itmp
      integer :: ierr
      integer :: jmin, jmax
      real :: dist
      real :: sum
      real :: c1
      real :: s1
      real :: c2
      real :: s2
      real :: dx
      real :: dy
      real :: term
      real, dimension(plon,plat) :: &
              cont, &
              temp, &
              oro_glob, &
              landm_glob
      real :: oro_temp(plonl,platl,pplon,nodes)
      real, dimension(plat) :: &
              cs, &
              ss

!---------------------------------------------------------------------- 
!	... allocate landmask
!---------------------------------------------------------------------- 
      if( .not. allocated( landm ) ) then
         allocate( landm(plonl,platl,pplon),stat=ierr )
         if( ierr /= 0 ) then
	    write(*,*) 'inimland: failed to allocate land mask; error = ',ierr
	    call endrun
         end if
      end if

!---------------------------------------------------------------------- 
!	... Gather the node orography into one array one master node
!---------------------------------------------------------------------- 
#ifdef USE_MPI
      call mpi_gather( oro, plon*platl, mpi_double_precision, &
                       oro_temp, plon*platl, mpi_double_precision, &
                       0, mpi_comm_comp, ierr )
      if( ierr /= mpi_success ) then
	 write(*,*) 'INIMLAND: Gather for oro failed; code = ',ierr
	 call endrun
      end if
#endif

master_only : &
      if( masternode ) then
!---------------------------------------------------------------------- 
!	... Recast into "whole" domain array
!---------------------------------------------------------------------- 
#ifdef USE_MPI
         do jj = 0,nodes-1
	    do j = 1,platl
	       do ii = 0,pplon-1
		  do i = 1,plonl
		     oro_glob(i+ii*plonl,j+jj*platl) = oro_temp(i,j,ii+1,jj+1)
		  end do
	       end do
	    end do
	 end do
#else
         do ii = 0,pplon-1
            do i = 1,plonl
               oro_glob(i+ii*plonl,:) = oro(i,:,ii+1)
            end do
         end do
#endif
!---------------------------------------------------------------------- 
! 	.. First isolate the continents 
!          as land points not surrounded by ocean or ice
!---------------------------------------------------------------------- 
         do j = 1,plat
            cs(j) = COS( clat(j) )
            ss(j) = SIN( clat(j) )
            do i = 1,plon
               cont(i,j) = 0.
               if( NINT( oro_glob(i,j) ) == 1 ) then
                  cont(i,j) = 1.
               end if
            end do
            temp(1,j) = cont(1,j)
            temp(plon,j) = cont(plon,j)
         end do

         do i = 1,plon
            temp(i,1) = cont(i,1)
            temp(i,plat) = cont(i,plat)
         end do

!---------------------------------------------------------------------- 
! 	... Remove one and two point islands
!---------------------------------------------------------------------- 
         do j = 2,plat-1
            do i = 2,plon-1
               sum =  cont(i,j+1) + cont(i,j-1) &
                    + cont(i+1,j+1) + cont(i+1,j-1) &
                    + cont(i-1,j+1) + cont(i-1,j-1) & 
                    + cont(i+1,j) + cont(i-1,j) &
                    + cont(i,j)
               if( sum <= 2. ) then
                  temp(i,j) = 0.
               else
                  temp(i,j) = 1.
               end if
            end do
         end do

         do j = 1,plat
            cont(:plon,j) = temp(:plon,j)
         end do

!---------------------------------------------------------------------- 
! 	... Construct a function which is one over land, 
!           zero over ocean points beyond dmax from land
!---------------------------------------------------------------------- 
         iplm1 = 2*plon - 1
         dy = pi*arad/real(plat,kind=8)
         jof = int( dmax/dy ) + 1
         do j = 2,plat-1
            c1 = cs(j)
            s1 = ss(j)
            dx = 2.*pi*arad*cs(j)/real(plon,kind=8)
            iof = min( int(dmax/dx) + 1, plon )
            do i = 1,plon
               temp(i,j) = 0.
               landm_glob(i,j) = 0.
               jmin = max( 1,j-jof )
               jmax = min( plat,j+jof )
               do jj = jmin,jmax
                  s2 = ss(jj)
                  c2 = cs(jj)
                  do itmp = -iof,iof
                     ii = mod( i+itmp+iplm1,plon ) + 1
                     term = s1*s2 + c1*c2*cos( clon(ii) - clon(i) )
                     if( term > .9999999 ) then
		        term = 1.
		     end if
                     dist = arad*acos( term )
                     landm_glob(i,j) = max( landm_glob(i,j),(1. - dist/dmax)*cont(ii,jj) )
                  end do
               end do
            end do
         end do

!---------------------------------------------------------------------- 
!	... Just set these to zero land for fssl
!---------------------------------------------------------------------- 
         landm_glob(:plon,1)    = 0.
         landm_glob(:plon,plat) = 0.
      end if master_only
!---------------------------------------------------------------------- 
!	... Broadcast the whole array to each node
!---------------------------------------------------------------------- 
#ifdef USE_MPI
      call mpi_bcast( landm_glob, plon*plat, mpi_double_precision, 0, mpi_comm_comp, ierr )
      if( ierr /= mpi_success ) then
	 write(*,*) 'INIMLAND: Bcast for landmask failed; code = ',ierr
	 call endrun
      end if
!---------------------------------------------------------------------- 
!	... Extract node section of global land mask
!---------------------------------------------------------------------- 
      do j = 1,platl
	 jj = j + base_lat
	 do ii = 0,pplon-1
	    do i = 1,plonl
	       landm(i,j,ii+1) = landm_glob(i+ii*plonl,jj)
	    end do
	 end do
      end do
#else
      do ii = 0,pplon-1
         do i = 1,plonl
            landm(i,:,ii+1) = landm_glob(i+ii*plonl,:)
         end do
      end do
#endif

      end subroutine inimland

      subroutine inicld( xphi, xcappa, xgravit, xrair, xlatvap, &
                         xcpair, xrhow, xtfh2o, etamid )
!------------------------------------------------------------------------------
!	... Initialize cloud routines
!------------------------------------------------------------------------------

      use MO_GRID, only : plev, plat

      implicit none

!------------------------------------------------------------------------------
! 	... Dummy arguments
!------------------------------------------------------------------------------
      real, intent(in) :: &
        xphi(plat), &      ! Gaussian latitudes (radians)
        xcappa, &          ! R/Cp
        xgravit, &         ! Gravitational acceleration
        xrair, &           ! Gas constant for dry air
        xlatvap, &         ! latent heat of vaporization for liq. water
        xcpair, &          ! Specific heat of dry air
        xrhow, &           ! water density (g/cm^3)
        xtfh2o, &          ! freezing point of water at STP (K)
        etamid(plev)       ! eta coordinate at layer midpoints

!------------------------------------------------------------------------------
! 	... Local variables
!------------------------------------------------------------------------------
      integer :: k
      real    :: detamn, tmp

!----------------------------------------------------------------------------
!  	... Set the physical constants for cloud routines
!----------------------------------------------------------------------------
      cappa  = xcappa
      gravit = xgravit
      rair   = xrair
      latvap = xlatvap
      cpair  = xcpair
      rhow   = xrhow
      tfh2o  = xtfh2o

!----------------------------------------------------------------------------
!     	... Find level closest to 700mb over ocean
!----------------------------------------------------------------------------
      detamn = abs( .7 - etamid(1) )
      do k = 2, plev
         tmp = abs( .7 - etamid(k) )
         if( tmp < detamn ) then
            k700 = k
            detamn = tmp
         end if
      end do

      end subroutine inicld

      subroutine clouddr( ip, lat, dtime, oro, pint, &
                          pmid, pdel, rpdeli, ts, t, &
                          sh, omga, cmfmc, zdu, cldtop, &
                          cldbot, dlf, tm1, shm1, cmfdqr, &
                          cwat, cldt, cldc, cltot, cme, &
                          evapr, prain, precl, plonl, platl )
!-----------------------------------------------------------------------
! 	... Cloud physics calcs
!-----------------------------------------------------------------------

      use MO_MPI,     only : thisnode
      use MO_GRID,    only : plev, plevp
      use MO_HISTOUT, only : OUTFLD, sim_file_cnt

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        plonl, &             ! longitude tile dimension
        platl                ! longitude tile dimension
      integer, intent(in) :: &
        ip, &                ! longitude tile index
        lat                  ! latitude tile index

      real, intent(in) :: &
        dtime                ! timestep size (s)
      real, dimension(plonl), intent(in) :: &
        oro, &               ! orography
        ts, &                ! surface temperature
        cldtop, &            ! top level of convection
        cldbot               ! bottom level of convection
      real, dimension(plonl,plev), intent(in) :: &
        pmid, &              ! pressure at layer midpoints
        pdel, &              ! pressure difference across layers
        rpdeli, &            ! 1./(pmid(k+1)-pmid(k))
        omga, &              ! vertical pressure velocity
        cmfmc, &             ! convective mass flux (kg/m^2/s)
        zdu, &               ! du2 from conv_ccm, 1/s
        dlf, &               ! detraining cloud water from convection
        tm1, &               ! temperature (K), tdyn(n-1)
        shm1, &              ! specific humidity (kg/kg), shdyn(n-1)
        cmfdqr               ! dq/dt due to convective rainout 
      real, dimension(plonl,plevp), intent(in) :: &
        pint                 ! pressure at layer interfaces

      real, dimension(plonl,plev), intent(inout) :: &
        t, &                 ! temperature (K)
                             ! tdyn(n-1)+adv+vdiff+conv on input,
                             ! tendency due to cloud added on output
        sh, &                ! specific humidity (kg/kg)
                             ! shdyn(n-1)+adv+vdiff+conv on input
                             ! tendency due to cloud added on output
        cwat                 ! cloud water (kg/kg)

      real, dimension(plonl,plev), intent(out) :: &
        cldc, &              ! convective cloud fraction
        cme, &               ! rate of cond-evap within the cloud
        evapr, &             ! rate of evaporation of falling precipitation (kg/kg/s)
        prain                ! rate of conversion of condensate to precipitation (kg/kg/s)
      real, dimension(plonl,plevp), intent(out) :: &
        cldt                 ! total cloud fraction
      real, dimension(plonl), intent(out) :: &
        cltot, &             ! diagnosed tot random overlap cld cover
        precl                ! large scale precip (m/s)

!-----------------------------------------------------------------------
! 	...Local variables
!-----------------------------------------------------------------------
      integer :: i, k, file

!-----------------------------------------------------------------------
!     	... cloud fraction
!-----------------------------------------------------------------------
      real :: &
        clc(plonl), &           ! column convective cloud amount
        cldst(plonl,plev), &    ! cloud fraction (strat?)
        snowh(plonl)            ! snow depth (liquid water equivalent)
      real :: cllow(plonl)      ! diagnosed low random overlap cld cover
      real :: clmed(plonl)      ! diagnosed med random overlap cld cover
      real :: clhgh(plonl)      ! diagnosed hgh random overlap cld cover

!-----------------------------------------------------------------------
!     	... moist
!-----------------------------------------------------------------------
      real :: &
        qtend(plonl,plev), &   ! moisture tendencies (kg/kg/s)
        ttend(plonl,plev), &   ! temp tendencies (K/s)
        rmelt(plonl,plev), &   ! heating rate due to precip phase change (K/s) (DISABLED)
        pcflx(plonl,plevp), &  ! convective precip level by level (kg/m2/s) (DISABLED)
        fracw(plonl,plev), &   ! relative importance of collection of liquid by rain
        fsaci(plonl,plev), &   ! relative importance of collection of ice by snow
        fsacw(plonl,plev), &   ! relative importance of collection of liquid by snow
        fsaut(plonl,plev), &   ! relative importance of ice auto conversion
        fwaut(plonl,plev)      ! relative importance of warm cloud auto conversion

!-----------------------------------------------------------------------
!	... Cloud fraction
!-----------------------------------------------------------------------
      snowh(:) = 0.

!-----------------------------------------------------------------------
!     	... Compute cloud amount
!-----------------------------------------------------------------------
      call cldfrc( pmid, rpdeli, t, sh, omga, &
                   cldtop, cldbot, cldt, clc, pdel, &
                   cmfmc, oro, snowh, cldc, cldst, &
                   ts, pint(1,plevp), zdu, lat, plonl )
!-----------------------------------------------------------------------
!     	... Cloud cover diagnostics
!-----------------------------------------------------------------------
      call cldsav( cldt, pmid, cltot, cllow, clmed, clhgh, plonl )

!-----------------------------------------------------------------------
!     	... Put the detraining cloud water into the cloud and environement
!           in proportion to the cloud fraction.
!-----------------------------------------------------------------------
      do k = 1,plev
         do i = 1,plonl
            sh(i,k) = sh(i,k) + dtime*dlf(i,k)*(1. - cldt(i,k))
            t(i,k) = t(i,k) - latvap/cpair*dtime*dlf(i,k)*(1. - cldt(i,k))
            cwat(i,k) = cwat(i,k) + dtime*dlf(i,k)*cldt(i,k)
            if( cwat(i,k) < 1.e-12 ) then
	       cwat(i,k) = 0.
	    end if
            qtend(i,k) = (sh(i,k) - shm1(i,k))/dtime
            ttend(i,k) = (t(i,k) - tm1(i,k))/dtime
         end do
      end do

!-----------------------------------------------------------------------
!     	... Stratiform condensation via prognostic cloud water
!-----------------------------------------------------------------------
      precl(:) = 0.
      do k = 1,plevp
         pcflx(:,k) = 0.
      end do

      call pcond( qtend, ttend, omga, lat, ip, &
                  shm1, cwat, tm1, pmid, pdel, oro, &
                  cldt, cldt, cme, evapr, prain, rmelt, dtime, pcflx, &
                  fwaut, fsaut, fracw, fsacw, fsaci, &
		  plonl, platl )

      do file = 1,sim_file_cnt
         call outfld( 'CME',   cme,   plonl, ip, lat, file )
         call outfld( 'PRAIN', prain, plonl, ip, lat, file )
         call outfld( 'EVAPR', evapr, plonl, ip, lat, file )
      end do

      do k = 1,plev
         do i = 1,plonl
            t(i,k) = t(i,k) + latvap/cpair*(cme(i,k) - evapr(i,k))*dtime + rmelt(i,k)*dtime
            sh(i,k) = max( 1.e-12,sh(i,k) - (cme(i,k) - evapr(i,k))*dtime )
            cwat(i,k) = cwat(i,k) + (cme(i,k) - prain(i,k))*dtime
            precl(i) = precl(i) + (prain(i,k) - evapr(i,k))*pdel(i,k)/gravit
            if( cwat(i,k) < 1.e-12 ) then
	       cwat(i,k) = 0.
	    end if
         end do
      end do

!-----------------------------------------------------------------------
!    	... Convert precl from kg/m2/s to m/s
!-----------------------------------------------------------------------
      precl(:plonl) = precl(:plonl)*.001

      end subroutine clouddr

      end module mo_cloud
