
      module mo_control
!---------------------------------------------------------------------------
! 	... Control parameters
!---------------------------------------------------------------------------

      use m_types, only : filespec, pdiag

      implicit none

      save

      logical :: &
        incore, &              ! t => incore history buffer and nas buffer
        divdiff, &             ! t => diagnose vertical diffusion
        arvdiff, &             ! t => archived vertical diffusion
        diconvccm, &           ! t => diagnose CCM convection
        arconvccm, &           ! t => archived CCM convection
        dicldphys, &           ! t => diagnose cloud physics
        arcldphys, &           ! t => archived cloud physics
        ditiedtke, &           ! t => diagnose Tiedtke convection
        artiedtke, &           ! t => archived Tiedtke convection
        stratchem, &           ! t => stratospheric chemistry
        tropchem, &            ! t => tropospheric chemistry
        chembgt, &             ! t => perform budget computations for trop chem
        vdiffuse, &            ! t => vertical diffusion scheme enabled
        convect, &             ! t => convection scheme enabled
        chemistry, &           ! t => chemistry scheme enabled
        physics, &             ! t => vdiffuse or convect or chemistry or dicldphys
        fixmas, &              ! t => apply SLT mass fixer
        calcedot, &            ! t => calculate etadot
        advqth, &              ! t => advect Q and THETA (potential temperature)
        rdomega, &             ! t => read omega from dynamics input files
        satvap, &              ! t => enable saturated vapor pressure calculation
        tvbnd, &               ! t => emissions from time varying boundary dataset
        dowetdep, &            ! t => apply wet deposition parameterization
        addqrad, &             ! t => add radiative heating tendency to the temperature
                               !      field supplied to the convection routine
        limqin, &              ! t => limit input Q s.t. it does not exceed saturation
        use_dust, &            ! t => use specified dust
        dyn_soilw, &           ! t => soil moisture from dynamics datasets
        dyn_has_fsds_avg, &    ! t => average fsds from dynamics datasets
        dyn_has_ts_avg, &      ! t => average surface temp from dynamics datasets
        xactive_emissions, &   ! t => interactive emissions
        xactive_prates, &      ! t => interactive fast tuv photolysis
        xactive_drydep, &      ! t => interactive dry deposition velocities
        xactive_h2o            ! t => interactive water vapor

!---------------------------------------------------------------------------
!     	... Control time sequence
!     *** Note on terminology ***  timestep N advances the integration from
!     time index N-1 to time index N.
!---------------------------------------------------------------------------
      integer :: &
        rstflg, &      ! 0 for initial run, 1 for restart
        itim0, &       ! beginning time index for run
        nestep         ! ending timestep for run

      logical :: &
        begstep, &     ! t => current timestep is beginning timestep for run
        laststep, &    ! t => current timestep is final timestep for run
        endofrun, &    ! t => last timestep or shutdown requested
        wrestart       ! t => write a restart file this timestep

!-------------------------------------------------------------------
!	... Mozart control variables
!-------------------------------------------------------------------

      integer, parameter :: secpday = 86400
      integer :: nspday                    ! number of steps per day
      integer :: delt                      ! delt in seconds
      integer :: step_cnt = 0              ! time step counter
      integer :: time_step                 ! time step index
      integer :: total_steps
      logical :: first_step
      logical :: last_step
      logical :: diurnal_step = .false.

      type date_time
         integer :: date
         integer :: secs
      end type date_time
      type(date_time) :: time_t
      type(date_time) :: begin_time
      type(date_time) :: end_time

!-------------------------------------------------------------------
!	... Mozart namelist variables for file input output
!           indices for local_path, remote_path, nl_filename, and resstr
!-------------------------------------------------------------------
      type(filespec) :: dyn_flsp
      type(filespec) :: ic_flsp
      type(filespec) :: ubc_flsp
      type(filespec) :: emis_flsp
      type(filespec) :: frc_flsp
      type(filespec) :: photo_flsp
      type(filespec) :: photo_xs_long_flsp
      type(filespec) :: photo_rsf_flsp
      type(filespec) :: photo_xs_short_flsp
      type(filespec) :: photo_xs_o2_flsp
      type(filespec) :: solar_parms_flsp
      type(filespec) :: woods_flsp
      type(filespec) :: neckel_flsp
      type(filespec) :: sulf_flsp
      type(filespec) :: surf_flsp
      type(filespec) :: dvel_flsp
      type(filespec) :: lbc_flsp
      type(filespec) :: dust_flsp
      type(filespec) :: sad_flsp
      type(filespec) :: rest0_flsp
      type(filespec) :: rest1_flsp
      type(filespec) :: constit_flsp
      type(filespec) :: srcs_flsp

      type(pdiag) :: &
         pdiags

      contains

      subroutine ini_date( ncdate, ncsec, mdelt, begstep, endstep )
!-------------------------------------------------------------------
!	... Initialize the time_t structure to next time
!-------------------------------------------------------------------

      use mo_calendar, only : addsec2dat

      implicit none

!-------------------------------------------------------------------
!	... Dummy args
!-------------------------------------------------------------------
      integer, intent(in) :: ncdate, ncsec, begstep, mdelt, endstep

      time_step   = begstep
      time_t%date = ncdate
      time_t%secs = ncsec
      begin_time%date = ncdate
      begin_time%secs = ncsec
      delt        = mdelt
      total_steps = endstep - begstep
      nspday = nint( secpday/real(mdelt) )

      end subroutine ini_date

      subroutine update_time
!-------------------------------------------------------------------
!	... Update the time_t structure to next time
!-------------------------------------------------------------------
      use mo_calendar, only : newdate

      implicit none

!-------------------------------------------------------------------
!	... Local variables
!-------------------------------------------------------------------
      integer :: iday

      step_cnt    = step_cnt + 1
      time_step   = time_step + 1
      time_t%secs = time_t%secs + delt
      if( time_t%secs >= secpday ) then
         iday        = int(time_t%secs/secpday)
	 time_t%secs = mod( time_t%secs,INT(secpday) )
	 time_t%date = newdate( time_t%date, iday )
      end if
      first_step   = step_cnt == 1
      last_step    = step_cnt == total_steps
      diurnal_step = mod( step_cnt,nspday ) == 0

      end subroutine update_time

      end module mo_control

