
      module mo_ffsldr

      private
      public :: ffsldr

      contains

      subroutine ffsldr( nstep, dtime, u, v, w,          &
                         ps1, ps_end, ps_pred, as, sh0,  &
                         hw1, hw2, nadv, x2, plonl, platl, pplon )
!---------------------------------------------------------------------
!	... Advection "driver"
!---------------------------------------------------------------------

      use mo_mpi,       only : base_lat, thisnode
      use m_adv,        only : mmr, jle, jue
      use mo_grid,      only : plon, plev, plevp, plat, pcnst
      use mo_extyv,     only : extyv
      use plevs,        only : hyai, hybi, plevs0
      use mo_ffsladv,   only : adjust_press, advection
      use mass_diags,   only : fdiags, advn_flux, hsa_fac
      use mo_constants, only : rgrav, latwts
      use mo_timer,     only : time_diff
      use mo_mass,      only : gamdry
      use mo_histout,   only : moz_file_cnt

      implicit none

!---------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) ::  &
        nstep, &                                    ! current model timestep
        nadv                                        ! number of misc advected variables
      integer, intent(in) ::  &
        plonl, &
        platl, &
        pplon

      real, intent(in) ::  &
        dtime                                       ! time step (seconds)
      real, dimension(plonl,-3:platl+4,pplon), intent(in) ::  &
        ps1,   &                                    ! surface pressure at t(n)
        ps_end                                      ! surface pressure at t(n+1)
      real, intent(in) ::  &
        sh0(plonl,plev,platl,pplon)                 ! specific humidity at t(n) (kg/kg)

      real, intent(inout) ::              &
        as(plonl,plev,pcnst,platl,pplon), &         ! species mass mixing ratio
        u(plonl,plev,-3:platl+4,pplon),   &         ! u wind component (m/s)
        v(plonl,plev,-2:platl+3,pplon)              ! v wind component (m/s)
     
      real, intent(out) ::               &
        x2(plonl,plev,nadv,platl,pplon), &          ! miscellaneous variables after advection
        w(plonl,plevp,platl,pplon),      &          ! vertical wind ("eta-dot")
        ps_pred(plonl,-3:platl+4,pplon)             ! predicted surface pressure
      real, dimension(pcnst,platl), intent(out) ::  &
        hw1, &                                      ! global mass before advection
        hw2                                         ! global mass after advection

!---------------------------------------------------------------------
! 	... Local variables
!---------------------------------------------------------------------
      integer :: i, ip, j, k, m, n, ntp, ierr, jglob, off
      integer :: file                              ! history file index
      real :: v_fac
      real :: bad_mem
      real, allocatable, dimension(:,:,:) ::  &
              xwind(:,:,:), &                       ! u wind component (ffsl shape) (m/s)
              ywind(:,:,:), &                       ! v wind component (ffsl shape) (m/s)
              zwind(:,:,:), &                       ! vertical wind component (ffsl shape)
              xmass(:,:,:), &                       ! zonal mass flux
              ymass(:,:,:), &                       ! meridional mass flux
              crx(:,:,:), &                         ! zonal wind in courant number
              cry(:,:,:)                            ! meridional wind in courant number
      real, allocatable, dimension(:,:) ::  &
              pctm1, &                              ! surface pressure at t(n) (ffsl shape)
              pctm2, &                              ! predicted surface pressure at t(n+1) (ffsl shape)
              pmet2                                 ! surface pressure at t(n+1) from met fields (ffsl shape)
      real,  dimension(plonl,plev) ::  &
              pdel, &                               ! level pressure thicknesses
              pmid                                  ! level midpoint pressures
      real :: pint(plonl,plevp)
      real, allocatable, dimension(:,:,:,:) ::  &
              hw1j, &
              hw2j
      character(len=8)    :: cdate(2)
      character(len=10)   :: ctime(2)

      allocate( xwind(plon,-3:platl+4,plev), &
                ywind(plon,-2:platl+3,plev), &
                zwind(plon,-1:platl+2,plev), &
                xmass(plon,-3:platl+4,plev), &
		ymass(plon,-2:platl+3,plev), &
                pctm1(plon,-3:platl+4), &
		pctm2(plon,-3:platl+4), &
                pmet2(plon,-3:platl+4), &
                hw1j(plonl,pcnst,platl,pplon), &
                hw2j(plonl,pcnst,platl,pplon), &
                crx(plon,-3:platl+4,plev), &
		cry(plon,-2:platl+3,plev), &
                stat=ierr)
      if( ierr /= 0 ) then
        write(*,*) 'ffsldr: allocation error = ',ierr
        call endrun
      end if

#ifdef INIT_TEST
!---------------------------------------------------------------------
! 	... Set allocated variables to poison value
!---------------------------------------------------------------------
      bad_mem      = z'7ff7ffff7ff7ffff'
      xwind(:,:,:) = bad_mem
      ywind(:,:,:) = bad_mem
      zwind(:,:,:) = bad_mem
      xmass(:,:,:) = bad_mem
      ymass(:,:,:) = bad_mem
      crx(:,:,:)   = bad_mem
      cry(:,:,:)   = bad_mem
      pctm1(:,:)   = bad_mem
      pctm2(:,:)   = bad_mem
      pmet2(:,:)   = bad_mem
#endif

!---------------------------------------------------------------------
! 	... Form mass integral before advection
!---------------------------------------------------------------------
      if( pplon > 1 ) then
!$omp parallel do private( file, i, ip, j, k, m, n, pmid, pint, pdel )
         do ip = 1,pplon
            do j = 1,platl
               call plevs0( ps1(1,j,ip), pint, pmid, pdel, plonl )
               call gamdry( j, pcnst, as(1,1,1,j,ip), sh0(1,1,j,ip), pdel, hw1j(1,1,j,ip), plonl )
	       do file = 1,moz_file_cnt
	          if( fdiags(file) > 0 ) then
	             do n = 1,fdiags(file)
	                m = advn_flux(n,file)%index
                        do k = 1,plev
                           do i = 1,plonl
 	                      advn_flux(n,file)%mass_delta(i,k,j,ip) = as(i,k,m,j,ip)*pdel(i,k)
                             end do
                        end do
                     end do
                  end if
               end do
            end do
         end do
!$omp end parallel do
      else
!$omp parallel do private( file, i, j, k, m, n, pmid, pint, pdel )
         do j = 1,platl
            call plevs0( ps1(1,j,1), pint, pmid, pdel, plonl )
            call gamdry( j, pcnst, as(1,1,1,j,1), sh0(1,1,j,1), pdel, hw1j(1,1,j,1), plonl )
	    do file = 1,moz_file_cnt
	       if( fdiags(file) > 0 ) then
	          do n = 1,fdiags(file)
	             m = advn_flux(n,file)%index
                     do k = 1,plev
                        do i = 1,plonl
 	                   advn_flux(n,file)%mass_delta(i,k,j,1) = as(i,k,m,j,1)*pdel(i,k)
                        end do
                     end do
                  end do
               end if
            end do
         end do
!$omp end parallel do
      end if

      do m = 1,pcnst
         do j = 1,platl
            hw1(m,j) = sum( hw1j(:plonl,m,j,:pplon) )
         end do
      end do

!---------------------------------------------------------------------
! 	... Map incoming u,v to ffsl grid
!---------------------------------------------------------------------
      call extyv( u, v, plonl, platl, pplon )

!---------------------------------------------------------------------
! 	... Reorder variables for advection
!---------------------------------------------------------------------
      call reorder_in( u, v, ps1, ps_end, xwind, &
                       ywind, pctm1, pmet2, plonl, platl, pplon )

      ntp = pcnst - 1 + nadv

!---------------------------------------------------------------------
! 	... Average polar regions
!---------------------------------------------------------------------
      call average_press_poles( plon, plat, platl, pctm1 )
      call average_press_poles( plon, plat, platl, pmet2 )

!---------------------------------------------------------------------
! 	... Pressure fixer
!---------------------------------------------------------------------
      call adjust_press( pctm1, pmet2, pctm2, xwind, ywind, &
                         xmass, ymass, INT(dtime), crx, cry, &
			 plonl, platl, pplon )

!---------------------------------------------------------------------
! 	... Transfer data between nodes
!---------------------------------------------------------------------
      call ffsl_transfer_data( xmass, ymass, pctm2, platl )

!---------------------------------------------------------------------
! 	... Advection
!---------------------------------------------------------------------
      call advection( pctm1, pctm2, pmet2, xmass, ymass, &
                      zwind, INT(dtime), ntp, hyai, hybi, &
		      crx, cry, plonl, platl, pplon )

!---------------------------------------------------------------------
! 	... Reorder variables for return
!---------------------------------------------------------------------
      call reorder_out( pctm2, zwind, mmr, ps_pred, w, &
                        as, x2, dtime, nadv, plonl, platl, pplon )

!---------------------------------------------------------------------
! 	... Form mass integral after advection
!---------------------------------------------------------------------
      if( pplon > 1 ) then
!$omp parallel do private( file, i, ip, j, k, m, n, off, pint, pmid, pdel, v_fac )
         do ip = 1,pplon
            off = (ip - 1)*plonl
            do j = 1,platl
               call plevs0( ps_pred(1,j,ip), pint, pmid, pdel, plonl )
               call gamdry( j, pcnst, as(1,1,1,j,ip), x2(1,1,1,j,ip), pdel, hw2j(1,1,j,ip), plonl )
               do file = 1,moz_file_cnt
                  if( fdiags(file) > 0 ) then
                     do n = 1,fdiags(file)
                        m = advn_flux(n,file)%index
                        v_fac = hsa_fac*latwts(base_lat+j)*rgrav
                        do k = 1,plev
                           do i = 1,plonl
                              advn_flux(n,file)%adv_mass(i,k,j,ip) = as(i,k,m,j,ip)*pdel(i,k)
                              advn_flux(n,file)%mass_delta(i,k,j,ip) =  &
                                 (advn_flux(n,file)%adv_mass(i,k,j,ip) &
                                   - advn_flux(n,file)%mass_delta(i,k,j,ip) ) * v_fac
                           end do
                        end do
                     end do
                  end if
               end do
            end do
         end do
!$omp end parallel do
      else
!$omp parallel do private( file, i, j, k, m, n, pint, pmid, pdel, v_fac )
         do j = 1,platl
            call plevs0( ps_pred(1,j,1), pint, pmid, pdel, plonl )
            call gamdry( j, pcnst, as(1,1,1,j,1), x2(1,1,1,j,1), pdel, hw2j(1,1,j,1), plonl )
            do file = 1,moz_file_cnt
               if( fdiags(file) > 0 ) then
                  do n = 1,fdiags(file)
                     m = advn_flux(n,file)%index
                     v_fac = hsa_fac*latwts(base_lat+j)*rgrav
                     do k = 1,plev
                        do i = 1,plonl
                           advn_flux(n,file)%adv_mass(i,k,j,1) = as(i,k,m,j,1)*pdel(i,k)
                           advn_flux(n,file)%mass_delta(i,k,j,1) =  &
                              (advn_flux(n,file)%adv_mass(i,k,j,1) &
                                - advn_flux(n,file)%mass_delta(i,k,j,1) ) * v_fac
                        end do
                     end do
                  end do
               end if
            end do
         end do
!$omp end parallel do
      end if
      do m = 1,pcnst
         do j = 1,platl
            hw2(m,j) = sum( hw2j(:plonl,m,j,:pplon) )
         end do
      end do

      deallocate( xwind, ywind, zwind, &
		  xmass, ymass, crx, cry, &
		  pctm1, pctm2, pmet2, &
		  hw1j, hw2j, stat=ierr)
      if( ierr /= 0 ) then
        write(*,*) 'ffsldr: deallocation error = ',ierr
        call endrun
      end if

      end subroutine ffsldr

      subroutine reorder_in( u, v, ps1, ps_end, xwind, &
                             ywind, pctm1, pmet2, plonl, platl, pplon )
!---------------------------------------------------------------------
! 	... Reorder variables for advection
!---------------------------------------------------------------------

      use mo_mpi,  only : masternode, lastnode
      use mo_grid, only : plon, plev

      implicit none

!---------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      real, intent(in) :: &
        u(plonl,plev,-3:platl+4,pplon),   &         ! u wind component (m/s)
        v(plonl,plev,-2:platl+3,pplon)              ! v wind component (m/s)
      real, dimension(plonl,-3:platl+4,pplon), intent(in) ::  &
        ps1,   &                                    ! surface pressure at t(n)
        ps_end                                      ! surface pressure at t(n+1)
      real, intent(out) :: &
        xwind(plon,-3:platl+4,plev), &              ! u wind component (ffsl shape) (m/s)
        ywind(plon,-2:platl+3,plev)                 ! v wind component (ffsl shape) (m/s)
      real, dimension(plon,-3:platl+4), intent(out) :: &
        pctm1, &                                    ! surface pressure at t(n) (ffsl shape)
        pmet2                                       ! surface pressure at t(n+1) from met fields (ffsl shape)

!---------------------------------------------------------------------
! 	... Local variables
!---------------------------------------------------------------------
      integer :: i, ip, j, k, off
      integer, dimension(2) :: jl, ju

#ifdef USE_MPI
      jl(:) = (/ -2, -3 /)
      ju(:) = (/ platl + 3, platl + 4 /)
      if( masternode ) then
         jl(:) = 1
      end if
      if( lastnode ) then
         ju(:) = platl
      end if
#else
      jl(:) = 1
      ju(:) = platl
#endif
      do ip = 1,pplon
         off = (ip - 1)*plonl
         do j = jl(1),ju(1)
            do k = 1,plev
               do i = 1,plonl
                  ywind(i+off,j,k) = v(i,k,j,ip)
               end do
            end do
         end do
         do j = jl(2),ju(2)
            do i = 1,plonl
               pctm1(i+off,j) = ps1(i,j,ip)
               pmet2(i+off,j) = ps_end(i,j,ip)
            end do

            do k = 1,plev
               do i = 1,plonl
                  xwind(i+off,j,k) = u(i,k,j,ip)
               end do
            end do
         end do
      end do

      end subroutine reorder_in

      subroutine reorder_out( pctm2, zwind, mmr, ps_pred, w, &
                              as, x2,  dtime, nadv, plonl, platl, pplon )
!---------------------------------------------------------------------
! 	... Reorder variables for return
!---------------------------------------------------------------------

      use mo_grid, only : plon, plev, plevp, pcnst
      use m_adv,   only : jle, jue

      implicit none

!---------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      integer, intent(in) :: &
        nadv                                 ! number of misc advected variables
      real, intent(in) :: &
        pctm2(plon,-3:platl+4),            & ! predicted surface pressure at t(n+1) (ffsl shape)
        zwind(plon,-1:platl+2,plev),       & ! vertical wind component (ffsl shape)
        mmr(plon,-3:platl+4,plev,pcnst+nadv-1) ! tracer mixing ratios
      real, intent(inout) :: &
        as(plonl,plev,pcnst,platl,pplon)     ! species mass mixing ratio
      real, intent(out) :: &
        ps_pred(plonl,-3:platl+4,pplon),   & ! predicted surface pressure
        w(plonl,plevp,platl,pplon),        & ! vertical wind ("eta-dot")
        x2(plonl,plev,nadv,platl,pplon)      ! miscellaneous variables after advection
      real, intent(in) :: &
        dtime                                ! time step (seconds)

!---------------------------------------------------------------------
! 	... Local variables
!---------------------------------------------------------------------
      integer :: i, ip, j, k, m, off

      if( pplon > 1 ) then
!$omp parallel do private( i, ip, j, k, m, off )
         do ip = 1,pplon
            off = (ip - 1)*plonl
            do j = jle(7),jue(7)
               do i = 1,plonl
                  ps_pred(i,j,ip) = pctm2(i+off,j)
               end do
            end do
            do j = 1,platl
               do i = 1,plonl
                  ps_pred(i,j,ip) = pctm2(i+off,j)
                  w(i,1,j,ip)     = 0.
                  w(i,plevp,j,ip) = 0.
               end do
               do k = 1,plev-1
                  do i = 1,plonl
                     w(i,k+1,j,ip) = zwind(i+off,j,k)/dtime
                  end do
               end do
            end do
            do m = 1,pcnst-1
               do j = 1,platl
                  do k = 1,plev
                     do i = 1,plonl
                        as(i,k,m,j,ip) = mmr(i+off,j,k,m)
                     end do
                  end do
               end do
            end do
            if( nadv > 0 ) then
               do m = 1,nadv
                  do j = 1,platl
                     do k = 1,plev
                        do i = 1,plonl
                           x2(i,k,m,j,ip) = mmr(i+off,j,k,m+pcnst-1)
                        end do
                     end do
                  end do
               end do
            end if
         end do
!$omp end parallel do
      else
!$omp parallel private( i, j, k, m )
!$omp do
         do j = jle(7),jue(7)
            do i = 1,plonl
               ps_pred(i,j,1) = pctm2(i,j)
            end do
         end do
!$omp end do
!$omp do
         do j = 1,platl
            do i = 1,plonl
               ps_pred(i,j,1) = pctm2(i,j)
               w(i,1,j,1)     = 0.
               w(i,plevp,j,1) = 0.
            end do
            do k = 1,plev-1
               do i = 1,plonl
                  w(i,k+1,j,1) = zwind(i,j,k)/dtime
               end do
            end do
         end do
!$omp end do
!$omp do
         do m = 1,pcnst-1
            do j = 1,platl
               do k = 1,plev
                  do i = 1,plonl
                     as(i,k,m,j,1) = mmr(i,j,k,m)
                  end do
               end do
            end do
         end do
!$omp end do
         if( nadv > 0 ) then
!$omp do
            do m = 1,nadv
               do j = 1,platl
                  do k = 1,plev
                     do i = 1,plonl
                        x2(i,k,m,j,1) = mmr(i,j,k,m+pcnst-1)
                     end do
                  end do
               end do
            end do
!$omp end do
         end if
!$omp end parallel
      end if

      end subroutine reorder_out

      subroutine ffsl_transfer_data( xmass, ymass, pctm2, platl )
!---------------------------------------------------------------------
! 	... Transfer data between nodes
!---------------------------------------------------------------------

#ifdef USE_MPI
      use mo_mpi,  only : masternode, lastnode, thisnode, &
                          slab, xregion, yregion,         &
                          mpi_comm_comp, mpi_proc_null, mpi_status_size
#endif
      use mo_grid, only : plon, plev

      implicit none

!---------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) :: &
	platl
      real, intent(inout) ::          &
        xmass(plon,-3:platl+4,plev),  &      ! zonal mass flux
        ymass(plon,-2:platl+3,plev),  &      ! meridional mass flux
        pctm2(plon,-3:platl+4)               ! predicted surface pressure at t(n+1)

!---------------------------------------------------------------------
! 	... Local variables
!---------------------------------------------------------------------
#ifdef USE_MPI
      integer :: i, dest, ierr, request(12), status1(MPI_STATUS_SIZE)
#endif

#ifdef USE_MPI
!---------------------------------------------------------------------
! 	... Send/receive north
!---------------------------------------------------------------------
      if( .not. lastnode ) then
         dest = thisnode + 1
      else
         dest = mpi_proc_null
      end if
      call mpi_isend( xmass(1,platl-3,1), 1, xregion, dest, 21, mpi_comm_comp, request(1), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_isend xmass; error = ',ierr
         call endrun
      end if
      call mpi_irecv( xmass(1,platl+1,1), 1, xregion, dest, 22, mpi_comm_comp, request(2), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_irecv xmass; error = ',ierr
         call endrun
      end if
      call mpi_isend( ymass(1,platl-2,1), 1, yregion, dest, 23, mpi_comm_comp, request(3), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_isend ymass; error = ',ierr
         call endrun
      end if
      call mpi_irecv( ymass(1,platl+1,1), 1, yregion, dest, 24, mpi_comm_comp, request(4), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_irecv ymass; error = ',ierr
         call endrun
      end if
      call mpi_isend( pctm2(1,platl-3), 1, slab, dest, 25, mpi_comm_comp, request(5), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_isend pctm2; error = ',ierr
         call endrun
      end if
      call mpi_irecv( pctm2(1,platl+1), 1, slab, dest, 26, mpi_comm_comp, request(6), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_irecv pctm2; error = ',ierr
         call endrun
      end if


!---------------------------------------------------------------------
! 	... Send/receive south
!---------------------------------------------------------------------
      if( .not. masternode ) then
         dest = thisnode - 1
      else
         dest = MPI_PROC_NULL
      end if
      call mpi_isend( xmass(1,1,1), 1, xregion, dest, 22, mpi_comm_comp, request(7), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_isend xmass; error = ',ierr
         call endrun
      end if
      call mpi_irecv( xmass(1,-3,1), 1, xregion, dest, 21, mpi_comm_comp, request(8), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_irecv xmass; error = ',ierr
         call endrun
      end if
      call mpi_isend( ymass(1,1,1), 1, yregion, dest, 24, mpi_comm_comp, request(9), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_isend ymass; error = ',ierr
         call endrun
      end if
      call mpi_irecv( ymass(1,-2,1), 1, yregion, dest, 23, mpi_comm_comp, request(10), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_irecv ymass; error = ',ierr
         call endrun
      end if
      call mpi_isend( pctm2(1,1), 1, slab, dest, 26, mpi_comm_comp, request(11), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_isend pctm2; error = ',ierr
         call endrun
      end if
      call mpi_irecv( pctm2(1,-3), 1, slab, dest, 25, mpi_comm_comp, request(12), ierr )
      if( ierr /= 0 ) then 
         write(*,*) 'ffsldr: Failed to mpi_irecv pctm2; error = ',ierr
         call endrun
      end if

!---------------------------------------------------------------------
! 	... Wait for completion
!---------------------------------------------------------------------
      do i = 1,12
         call mpi_wait( request(i), status1, ierr )
         if( ierr /= 0 ) then 
            write(*,*) 'ffsldr: mpi_wait failed for i=',i,'; error = ',ierr
            call endrun
         end if
      end do

#ifdef DEBUG
      write(*,'(''*pctm2='',6ES20.12)') pctm2(10,-2:3)
      write(*,'(''*pctm2='',6ES20.12)') pctm2(10,platl-2:platl+3)
      write(*,'(''*xmass='',6ES20.12)') xmass(10,-2:3,plev)
      write(*,'(''*xmass='',6ES20.12)') xmass(10,platl-2:platl+3,plev)
      write(*,'(''*ymass='',6ES20.12)') ymass(10,-2:3,plev)
      write(*,'(''*ymass='',6ES20.12)') ymass(10,platl-2:platl+3,plev)
#endif
#endif
      end subroutine ffsl_transfer_data

      subroutine average_press_poles( plon, plat, platl, ps )
!-----------------------------------------------------------------------
!       ... Average pressure at the poles.
!           When the enlarged polar cap is used, make pressures at
!           the last two latitudes equal
!-----------------------------------------------------------------------

      use mo_mpi,       only : masternode, lastnode, base_lat
      use m_adv,        only : has_spole, has_npole
      use mo_constants, only : latwts

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plon, plat, platl
      real, intent(inout) :: ps(plon,-3:platl+4)    ! surface pressure

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: j, jb
      real    :: meanps

      if( has_spole ) then
	 jb = 1 - base_lat
         meanps = sum( ps(:plon,jb) ) * latwts(base_lat + jb) &
                  + sum( ps(:plon,jb+1) ) * latwts(base_lat + jb + 1)
         meanps = meanps / (real( plon ) * sum( latwts(base_lat+jb:base_lat+jb+1) ))
	 do j = jb,jb+1
            ps(:plon,j) = meanps
	 end do
      end if
      if( has_npole ) then
	 jb = plat - base_lat
         meanps = sum( ps(:plon,jb-1) ) * latwts(base_lat + jb - 1) &
                  + sum( ps(:plon,jb) ) * latwts( base_lat + jb)
         meanps = meanps / (real( plon ) * sum( latwts(base_lat+jb-1:base_lat+jb) ))
	 do j = jb-1,jb
            ps(:plon,j) = meanps
	 end do
      end if

      end subroutine average_press_poles

      end module mo_ffsldr
