
      module mo_histout

      use mo_grid, only : plon, plev, plat

      implicit none

      private
      public :: inihist, disphst, outfld, qhsti, qhstc, hst_gather, wrnchist, nexthist, addfld
      public :: hst_file, sim_file_cnt, match_file_cnt, moz_file_cnt, hst_file_max

      save

      integer, parameter :: mxdiagflds   = 500
      integer, parameter :: hst_file_max = 10
      integer, parameter :: hst_cat_max  = 22
!-----------------------------------------------------------------------
!       mxdiagflds    max number of fields in table containing info for the
!                     diagnostic fields (set in block data routine)
!-----------------------------------------------------------------------

!-----------------------------------------------------------
!	... history tape variables 
!-----------------------------------------------------------
      integer :: &
        ndiagfld, &            ! num of fields in table for diagnostic fields
        lon_id, &              ! netcdf longitude dimension id
        lat_id, &              ! netcdf latitude dimension id
        lev_id, &              ! netcdf midpoint level dimension id
        ilev_id, &             ! netcdf interface level dimension id
        time_id, &             ! netcdf time dimension id
        chr_id, &              ! netcdf character dimension id
        fld_idx = 0, &         ! count of dynamics fields active in flddat type
        match_file_cnt = 0, &  ! count of match files
        moz_file_cnt   = 0, &  ! count of mozart files
        sim_file_cnt   = 0     ! count of simulation files
!-----------------------------------------------------------
!	... history file user defined type
!-----------------------------------------------------------
      type hst_file            ! basic history file type
         integer :: &
               mxoutflds, &    ! count of max hist inst,timav outputs
               nfld, &         ! total number of user requested output fields
               ninstfld, &     ! number of user requested instantaneous output fields
               ntimavfld, &    ! number of user requested time averaged output fields
               lenhdi, &       ! length of integer header record (number of integers)
               lenhdc, &       ! length of character header record (number of character*8)
               ncid, &         ! netcdf file id
               nhtfrq, &       ! freq of hist tape writes in timesteps
               mfilt, &        ! max number of time samples written to hist tape
               ifcnt, &        ! counter for fields as they are initialized in inifld.
                               ! used to make sure number of fields specified in fldlst
                               ! is equal to the parameter nfld set above.
               ndcurf, &       ! day of first time sample on current history file
               nscurf, &       ! seconds relative to ndcurf
               ncdatf, &       ! date of first time sample on current history file
               ncsecf          ! seconds relative to ncdatf
         integer, dimension(2) :: &
              tifcnt, &        ! counter of single,multi lev instantaneous fields
              tafcnt, &        ! counter of single,multi lev time averaged fields
              match_cnt        ! count of inst,timav "match" outputs
         integer :: histout_cnt(hst_cat_max,2)                           ! history catagory counts
         integer :: histout_ind(hst_cat_max,2)                           ! history catagory begin indicies
         integer, dimension(:), pointer :: &
              hdi, &                                                      ! integer header record
              inst_map, &                                                 ! inst mozart field map
              timav_map                                                   ! timav mozart field map
         real :: &
              fnorm                                                       ! normalizer for time averaged fields
         real, dimension(:,:,:), pointer :: &
              ta_slev, ti_slev                                            ! single level averaged and instantaneous buffers
         real, dimension(:,:,:,:), pointer :: &
              ta_mlev, ti_mlev                                            ! multi level averaged and instantaneous buffers
         real, pointer, dimension(:,:,:,:)   ::  &
              gta_slev, gti_slev                                          ! single level averaged and instantaneous buffers
         real, pointer, dimension(:,:,:,:,:) :: &
              gta_mlev, gti_mlev                                          ! multi level averaged and instantaneous buffers
         logical :: &
              wrhstts, &                                                  ! t => write a history file time sample this timestep
              fullhst, &                                                  ! t => history file full after writing time sample
              closehst, &                                                 ! t => close history file after writing time sample
              partial_ta, &                                               ! t => history file has partial time average
              force_inst                                                  ! t => force a pure inst hst file to write
         character(len=1) :: &
              thtfrq                                                      ! type flag for history tape write frequency; m = monthly
         character(len=32), dimension(:), pointer :: &
              outinst, &                                                  ! inst wrk space
              outtimav, &                                                 ! timav wrk space
              hist_inst, &                                                ! inst mozart field names
              hist_timav, &                                               ! timav mozart field names
              hdc                                                         ! character header record
         character(len=80) :: &
              lnhstc, &                                                   ! mss pathname of current history file
              lnhstf, &                                                   ! mss pathname of first history file for this case
              mcstit                                                      ! case title
         character(len=168) :: &
              lpath, &                                                    ! local path for history files
              rpath                                                       ! remote path for history files
         character(len=168) :: &
              lpath_rhst, &                                               ! local path for history restart files
              rpath_rhst                                                  ! remote path for history restart files
      end type hst_file

      type(hst_file), public :: hfile(hst_file_max)

      integer :: &
        maxsiz  = 0, &         ! length of output data records (may be packed)
        unpksiz = 0, &         ! length of unpacked output data record
        nestep, &              ! ending timestep for model run
        ndens, &               ! write density for output history tapes
        irt                    ! retention period in days for mass store files.

      logical :: &
        async, &               ! t => output hist files xferred to ms asynchronously
        rmout, &               ! t => remove hist file after xferring to mass store
        stratchem, &           ! t => stratospheric chemistry
        tropchem               ! t => tropospheric chemistry

      character(len=8) :: &
        msvol, &               ! mass store virtual volume name
        wpasswd                ! mass store write password
      character(len=3) :: &
        msclass                ! mass store class of service

      type hst_fld_data
         character(len=32) :: name
         character(len=8)  :: lvl
         character(len=8)  :: units
      end type hst_fld_data
      type(hst_fld_data) :: flddat(mxdiagflds)
      character(len=32) :: &
        diagnams(mxdiagflds)   ! names for diagnostic fields

      contains

      subroutine inihist( rrootd, case, ntshst, xnestep, xnhtfrq, &
                          xthtfrq, xndens, xmfilt, stday, stsec, &
                          hst_flsp, icdate, icsec, dtime, rstflg, &
                          title, latrad, gw, xirt, xasync, &
                          xrmout, xmsvol, xwpasswd, xstratchem, xtropchem, &
                          partial_ta, rst_date, rst_datesec, rst_days, rst_secs, &
                          rst_lnhstf, xmsclass, xrhst_lpath, xrhst_rpath, plonl, platl )
!-----------------------------------------------------------------------
! 	... Initialize module.  Open first history file.  Initialize
!           history buffers.  For a restart run, the values in module that
!           are updated as the run progresses, and the history buffer,
!           will be initialized by the restart module.
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use mo_charutl,    only : lastsl, incstr, glc
      use m_types,       only : filespec

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: &
        rrootd, &            ! Default remote root directory
        case, &              ! case identifier
        title, &             ! case title
        xmsvol, &            ! mass store virtual volume name
        xwpasswd             ! MS write password
      character(len=*), dimension(hst_file_max), intent(in) :: &
        rst_lnhstf           ! path of first hst file for this case
      character(len=*), intent(in) :: xrhst_lpath(hst_file_max)
      character(len=*), intent(in) :: xrhst_rpath(hst_file_max)

      integer, intent(in) :: &
        xnestep, &           ! ending timestep for model run
        xndens, &            ! write density for output history tapes
        stday, &             ! starting day of initial run
        stsec, &             ! seconds relative to stday
        icdate, &            ! date of initial conditions in yymmdd format
        icsec, &             ! seconds relative to icdate
        dtime, &             ! timestep interval in seconds
        rstflg, &            ! 0 for initial run, 1 for restart run, 2 for branch run
        xirt                 ! retention period in days for mass store files.  If set to
                             ! 0 (default) then output files not transfered to mass store
      integer, intent(in) :: &
        plonl, &
        platl

      integer, dimension(hst_file_max), intent(in) :: &
        ntshst, &            ! number of time samples in the starting hist file
        xnhtfrq, &           ! freq of hist tape writes in timesteps
        xmfilt, &            ! max number of time samples written to hist tape
        rst_date, &          ! date retrieved from rstrt file (yyyymmdd) 
        rst_datesec, &       ! seconds in date retrieved from rstrt file (s) 
        rst_days, &          ! simulation elapsed days retrieved from rstrt file
        rst_secs             ! seconds in simulation elapsed days retrieved from rstrt file

      logical, intent(in) :: &
        xasync, &            ! true => output hist files transferred to ms asynchronously
        xrmout, &            ! true => remove hist file after linking and/or transferring to mass store
        xstratchem, &        ! t => stratospheric chemistry
        xtropchem            ! t => tropospheric chemistry

      real, dimension(plat), intent(in) :: &
        latrad, &            ! latitudes in radians
        gw                   ! Gauss weights

      character(len=*), dimension(hst_file_max), intent(in) :: &
        xthtfrq              ! type of history time output; m = monthly

      character(len=*), intent(in) :: &
        xmsclass             ! mass store class

      logical, dimension(hst_file_max), intent(in) :: &
        partial_ta           ! partial time average history file flags

      type(filespec), dimension(hst_file_max), intent(in) :: &
        hst_flsp             ! history paths, filenames

!-----------------------------------------------------------------------
! 	... local Variables
!-----------------------------------------------------------------------
      integer ::  i, j, k, m, spos     ! indicies
      integer ::  istat                ! return status
      integer ::  astat                ! allocation status
      integer ::  file                 ! history file index
      integer ::  var_id               ! netcdf variable id
      integer ::  ntrm                 ! spectral truncation parameter M
      integer ::  ntrn                 ! spectral truncation parameter N
      integer ::  ntrk                 ! spectral truncation parameter K
      integer ::  ndx(1)               ! netcdf variable index
      character(len=168) :: lfilepath  ! local filepath workspace
      character(len=168) :: filenm     ! local filename

!-----------------------------------------------------------------------
!     	... initialize module variables directly from input arguments
!-----------------------------------------------------------------------
      do file = 1,hst_file_max
         hfile(file)%ifcnt     = 0
         hfile(file)%tafcnt(:) = 0
         hfile(file)%tifcnt(:) = 0
      end do
      do file = 1,sim_file_cnt
         hfile(file)%lpath = trim( hst_flsp(file)%local_path ) // trim( hst_flsp(file)%nl_filename )
         if( hst_flsp(file)%remote_path(1:1) == ' ' ) then
            hfile(file)%rpath = '/' // rrootd(:GLC(rrootd)) // '/mozart/' // case(:GLC(case)) // &
                                '/hist/' // trim( hst_flsp(file)%nl_filename )
         else
            hfile(file)%rpath = trim( hst_flsp(file)%remote_path ) // trim( hst_flsp(file)%nl_filename )
         end if
         write(*,*) 'inihist: lpath = ',trim(hfile(file)%lpath)
         write(*,*) 'inihist: rpath = ',trim(hfile(file)%rpath)
      end do
      ndens     = xndens
      irt       = xirt
      async     = xasync
      rmout     = xrmout
      msvol     = xmsvol
      wpasswd   = xwpasswd
      if( trim( xmsclass ) == 'STD' .or. trim( xmsclass ) == 'EC' ) then
         msclass   = xmsclass
      else
         msclass   = 'EC'
      end if
      write(*,*) ' '
      write(*,*) 'inihist: mass store reliability = ',trim(msclass)
      write(*,*) ' '
      stratchem = xstratchem
      tropchem  = xtropchem
      nestep    = xnestep
      hfile(:sim_file_cnt)%nhtfrq = MAX( 1,xnhtfrq(:sim_file_cnt) )
      hfile(:sim_file_cnt)%thtfrq = xthtfrq(:sim_file_cnt)(1:1)
      hfile(:sim_file_cnt)%fnorm  = 1./REAL( MAX( 1,hfile(:sim_file_cnt)%nhtfrq ) )
      hfile(:sim_file_cnt)%mfilt  = MAX( 1,xmfilt(:sim_file_cnt) )

!-----------------------------------------------------------------------
!	... Define dynamical fields available for history output
!-----------------------------------------------------------------------
      call deffld
      ndiagfld            = fld_idx
      diagnams(:ndiagfld) = flddat(:ndiagfld)%name

!-----------------------------------------------------------------------
!     	... Check that the requested fields are known, and that there are no
!           duplicates.  Count the requested fields.
!-----------------------------------------------------------------------
      do file = 1,sim_file_cnt
         call fldparse( file )
         hfile(file)%nfld = hfile(file)%ninstfld + hfile(file)%ntimavfld
         if( hfile(file)%nfld == 0 ) then
            write(*,*) 'inihist: no output fields requested for file = ',file
            write(*,*) '         Use the namelist variables HSTINST and/or HSTTIMAV.'
            call endrun
         end if

!-----------------------------------------------------------------------
!     	... Allocate memory for field-dependent arrays
!-----------------------------------------------------------------------
         hfile(file)%lenhdi = 37 + 3*hfile(file)%nfld
         hfile(file)%lenhdc = 89 + 2*hfile(file)%nfld
         allocate( hfile(file)%hdi(hfile(file)%lenhdi),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'inihist: Failed to allocate hdi for file = ',file,'; error = ',astat
	    call endrun
         end if
         allocate( hfile(file)%hdc(hfile(file)%lenhdc),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'inihist: Failed to allocate hdc for file = ',file,'; error = ',astat
	    call endrun
         end if
      end do

      do file = 1,sim_file_cnt
         call fldlst( hfile(file)%outinst(1:hfile(file)%ninstfld), 0, file )
         call fldlst( hfile(file)%outtimav(1:hfile(file)%ntimavfld), 1, file )
!-----------------------------------------------------------------------
!     	... allocate memory for time averaged buffers
!-----------------------------------------------------------------------
         if( hfile(file)%tafcnt(1) > 0 ) then
            allocate( hfile(file)%ta_slev(plon,platl,hfile(file)%tafcnt(1)),stat=astat )
            if( astat /= 0 ) then
               write(*,*) 'inihist: Failed to allocate ta_slev for file = ',file,'; error = ',astat
	       call endrun
            end if
	    do m = 1,hfile(file)%tafcnt(1)
	       do j = 1,platl
	          hfile(file)%ta_slev(:plon,j,m) = 0.
	       end do
	    end do
         end if
         if( hfile(file)%tafcnt(2) > 0 ) then
            allocate( hfile(file)%ta_mlev(plon,platl,plev,hfile(file)%tafcnt(2)),stat=astat )
            if( astat /= 0 ) then
               write(*,*) 'inihist: Failed to allocate ta_mlev for file = ',file,'; error = ',astat
	       call endrun
            end if
	    do m = 1,hfile(file)%tafcnt(2)
	       do k = 1,plev
	          do j = 1,platl
	             hfile(file)%ta_mlev(:plon,j,k,m) = 0.
	          end do
	       end do
	    end do
         end if
!-----------------------------------------------------------------------
!     	... Integer header
!-----------------------------------------------------------------------
         hfile(file)%hdi(1)  = hfile(file)%lenhdi    ! length of integer header
         hfile(file)%hdi(3)  = ntshst(file)          ! counter for time samples in this file
         hfile(file)%hdi(4)  = hfile(file)%mfilt     ! max number of time samples in a file
         hfile(file)%hdi(9)  = plon                  ! number of longitudes
         hfile(file)%hdi(10) = plon                  ! number of longitude values written to data record
         hfile(file)%hdi(11) = plat                  ! number of latitudes
         hfile(file)%hdi(12) = plev                  ! number of vertical levels
      end do

!-----------------------------------------------------------------------
!     	... Spectral truncation parameters are set based on horizontal resolution.
!-----------------------------------------------------------------------
      call getspecres( plon, plat, ntrm, ntrn, ntrk )
Files_loop : &
      do file = 1,sim_file_cnt
         hfile(file)%hdi(13) = ntrm      ! spectral truncation parameter M
         hfile(file)%hdi(14) = ntrn      ! spectral truncation parameter N
         hfile(file)%hdi(15) = ntrk      ! spectral truncation parameter K
         hfile(file)%hdi(16) = hfile(file)%nfld      ! number of fields in each time sample
         hfile(file)%hdi(18) = 0         ! timestep number at start of run
         hfile(file)%hdi(20) = stday     ! base day 
         hfile(file)%hdi(21) = stsec     ! seconds relative to base day
         hfile(file)%hdi(24) = icdate    ! base date
         hfile(file)%hdi(25) = icsec     ! seconds relative to base date
         hfile(file)%hdi(28) = dtime     ! model timestep in seconds
         hfile(file)%hdi(29) = hfile(file)%nhtfrq    ! frequency of time sample writes in timesteps
         hfile(file)%hdi(30) = rstflg    ! flag for type of run
         hfile(file)%hdi(31) = hfile(file)%lenhdc    ! length of character header
         if( rstflg == 0 .or. rstflg == 2 ) then
            hfile(file)%hdi(17) = 0      ! current iteration number
            hfile(file)%hdi(19) = 0      ! timesteps since last time sample written
            hfile(file)%hdi(22) = stday  ! current day corresponding to current timestep
            hfile(file)%hdi(23) = stsec  ! seconds relative to current day
            hfile(file)%hdi(26) = icdate ! current date corresponding to current timestep
            hfile(file)%hdi(27) = icsec  ! current seconds relative to current date
         end if

!-----------------------------------------------------------------------
!     	... Character header
!-----------------------------------------------------------------------
         hfile(file)%hdc(:89) = ' '
         hfile(file)%hdc(1)   = case    ! case identifier
         hfile(file)%mcstit   = title   ! case title
         hfile(file)%lnhstc   = trim( hfile(file)%rpath )
         if( rstflg == 0 .or. rstflg == 2 ) then
            hfile(file)%lnhstf = hfile(file)%lnhstc
            do i = 1,10
               hfile(file)%hdc(24+i) = hfile(file)%lnhstf(8*(i-1)+1:8*i)
            end do
         end if

         if( io_node .and. ntshst(file) == 0 ) then
!-----------------------------------------------------------------------
!  	... Create new netcdf history file
!-----------------------------------------------------------------------
            call handle_ncerr( NF_CREATE( trim( hfile(file)%lpath ), or(nf_clobber,nf_64bit_offset), &
                               hfile(file)%ncid ), 'inihist: Failed to create ' // trim( hfile(file)%lpath ) )
!-----------------------------------------------------------------------
!  	... Initialize the netcdf file
!-----------------------------------------------------------------------
	    call hfile_inti( hfile(file)%hdi(38:hfile(file)%lenhdi), &
                             hfile(file)%hdc(90:hfile(file)%lenhdc), &
                             hfile(file)%nfld, file )
	 end if
Restarting : &
	 if( rstflg == 1 ) then
            if( ntshst(file) /= 0 ) then
!-----------------------------------------------------------------------
!  	... all nodes acquire most recent history file for this run
!-----------------------------------------------------------------------
	       if( .not. ded_io_node ) then
                  hfile(file)%ncid = open_netcdf_file( hst_flsp(file)%nl_filename, hst_flsp(file)%local_path, &
                                                       hst_flsp(file)%remote_path )
	       else
                  hfile(file)%ncid = open_netcdf_file( hst_flsp(file)%nl_filename, hst_flsp(file)%local_path, &
                                                       hst_flsp(file)%remote_path, use_io_node=.true. )
	       end if
	       lfilepath = trim( hst_flsp(file)%local_path ) // trim( hst_flsp(file)%nl_filename )
               if( io_node .or. masternode ) then
!-----------------------------------------------------------------------
!  	... master node get variable ids
!-----------------------------------------------------------------------
                  do m = 1,hfile(file)%nfld
                     call handle_ncerr( &
                           NF_INQ_VARID( hfile(file)%ncid, trim( hfile(file)%hdc(88+2*m) ), &
                                         hfile(file)%hdi(37+3*m) ), &
                           'inihist: Failed to get' // trim(hfile(file)%hdc(88+2*m)) // ' variable id' )
                  end do
		  if( ded_io_node .and. io_node ) then
                     call handle_ncerr( NF_CLOSE( hfile(file)%ncid ), 'inihist: Failed to close ' // trim( lfilepath ) )
		  end if
	       end if
               if( comp_node ) then
!-----------------------------------------------------------------------
!  	... all nodes close netcdf history file
!-----------------------------------------------------------------------
                  call handle_ncerr( nf_close( hfile(file)%ncid ), 'inihist: Failed to close ' // trim( lfilepath ) )
	       end if
            end if
	    hfile(file)%hdi(22) = rst_days(file)
	    hfile(file)%hdi(23) = rst_secs(file)
	    hfile(file)%hdi(26) = rst_date(file)
	    hfile(file)%hdi(27) = rst_datesec(file)
	    hfile(file)%lnhstf  = rst_lnhstf(file)
	    hfile(file)%ndcurf  = hfile(file)%hdi(22)
	    hfile(file)%nscurf  = hfile(file)%hdi(23)
	    hfile(file)%ncdatf  = hfile(file)%hdi(26)
	    hfile(file)%ncsecf  = hfile(file)%hdi(27)
!-----------------------------------------------------------------------
!  	... read history restart files
!-----------------------------------------------------------------------
	    if( comp_node .and. partial_ta(file) ) then
	       write(*,*) 'inihist: Before calling rhst_read for file ',file
	       call rhst_read( file, xrhst_lpath(file), xrhst_rpath(file), platl )
               write(*,*) 'inihist: After calling rhst_read for file ',file
	    end if
	 end if Restarting
         if( io_node ) then
            write(*,*) ' '
            write(*,*) '----------------------------------------------------------------------------'
            write(*,*) 'inihist: Diagnostics for file ',file
            write(*,*) 'The integer header'
            write(*,'(18i5)') hfile(file)%hdi(38:37+3*hfile(file)%nfld:3)
            write(*,*) 
            write(*,'(18i5)') hfile(file)%hdi(39:37+3*hfile(file)%nfld:3)
            write(*,*) 
            write(*,'(18i5)') hfile(file)%hdi(40:37+3*hfile(file)%nfld:3)
            write(*,*) 
            write(*,*) 'The character header'
            write(*,'(4(1x,a32))') hfile(file)%hdc(90:89+2*hfile(file)%nfld:2)
            write(*,*) 
            write(*,'(4(1x,a32))') hfile(file)%hdc(91:89+2*hfile(file)%nfld:2)
            write(*,*) '----------------------------------------------------------------------------'
         end if
      end do Files_loop

      end subroutine inihist

      subroutine deffld
!-----------------------------------------------------------------------
! 	... Define dynamics and miscellaneous fields that may be
!           written to the output history file
!-----------------------------------------------------------------------

      implicit none

      call addfld( 'PS',       'single',  'PA' )          ! surface pressure
      call addfld( 'U',        'multmid', 'M/S' )         ! zonal wind
      call addfld( 'V',        'multmid', 'M/S' )         ! meridional wind
      call addfld( 'ETADOT',   'multint', '1/S' )         ! vertical velocity in eta coord
      call addfld( 'OMEGA',    'multmid', 'pa/s' )        ! vertical velocity in pressure coord
      call addfld( 'T',        'multmid', 'K' )           ! temperature
      call addfld( 'TS',       'single',  'K' )           ! surface temperature
      call addfld( 'Q',        'multmid', 'KG/KG' )       ! advected specific humidity
      call addfld( 'RELHUM',   'multmid', 'fraction' )    ! relative humidity
      call addfld( 'H2O',   'multmid', 'VMR' )            ! water vmr
      call addfld( 'SAD',   'multmid', 'cm2/cm3' )        ! total surface area density
      call addfld( 'FRACDAY',     'single',  ' ' )           ! fraction of points during daytime
      call addfld( 'QIN',      'multmid', 'kg/kg' )       ! specific humidity from input data
      call addfld( 'TCWV',     'single',  'kg/m2' )       ! total column water vapor
      call addfld( 'Z',        'multmid', 'm' )           ! geopotential height above surface at layer midpoints
      call addfld( 'ZI',       'multint', 'm' )           ! geopotential height above surface at interfaces
      call addfld( 'ORO',      'single',  'FLAG' )        ! Orography
      call addfld( 'PHIS',     'single',  'M2/S2' )       ! surface geopotential
      call addfld( 'TAUX',     'single',  'N/m2' )        ! surface zonal stress (N/m2)
      call addfld( 'TAUY',     'single',  'N/m2' )        ! surface meridional stress (N/m2)
      call addfld( 'SHFLX',    'single',  'W/m2' )        ! surface sensible heat flux (W/m2)
      call addfld( 'QFLX',     'single',  'kg/m2/s' )     ! surface moisture flux (kg/m2/s)
      call addfld( 'FSDS',     'single',  'W/m2' )        ! downward solar flux at surface (W/m2)
      call addfld( 'FSUS',     'single',  'W/m2' )        ! upward solar flux at surface (W/m2)
      call addfld( 'FLDS',     'single',  'W/m2' )        ! downward longwave flux at surface (W/m2)
      call addfld( 'VWCSFC',   'single',  'fraction' )    ! volumetric soil water content at surface (frac)
      call addfld( 'SNOWH',    'single',  'm' )           ! water equivalent accumulated snow depth (m)
      call addfld( 'KVH',      'multint', 'M2/S' )        ! vertical diffusion coeff for heat
      call addfld( 'CGS',      'multint', 'S/M2' )        ! vertical diffusion counter gradient coeff
      call addfld( 'PBLH',     'single',  'M' )           ! PBL height
      call addfld( 'TPERT',    'single',  'K' )           ! tpert from pbl
      call addfld( 'QPERT',    'single',  'KG/KG' )       ! qpert from pbl
      call addfld( 'ZMU',      'multmid', 'KG/M2/S' )     ! upward mass flux (Zhang)
      call addfld( 'ZMD',      'multmid', 'KG/M2/S' )     ! downward mass flux (Zhang)
      call addfld( 'ZEU',      'multmid', '1/S' )         ! entrainment in updraft (Zhang)
      call addfld( 'ZDU',      'multmid', '1/S' )         ! detrainment in updraft (Zhang)
      call addfld( 'CONBETA',  'multmid', 'FRACTION' )    ! overshoot parameter (Hack)
      call addfld( 'CONETA',   'multmid', 'KG/M2/S' )     ! convective mass flux (Hack)
      call addfld( 'CMFDQR',   'multmid', 'KG/KG/S' )     ! convective rainout
      call addfld( 'CMFMC',    'multmid', 'KG/KG/S' )     ! moist convection cloud mass flux
      call addfld( 'CONICW',   'multmid', 'KG/KG' )       ! convective in-cloud water m.r.
      call addfld( 'PRECC',    'single',  'M/S' )         ! convective precip
      call addfld( 'PRECT',    'single',  'm/s' )         ! total precip
      call addfld( 'ZCMFDQR',  'multmid', 'KG/KG/S' )     ! convective rainout (Zhang)
      call addfld( 'ZCMFMC',   'multmid', 'KG/KG/S' )     ! moist convection cloud mass flux (Zhang)
      call addfld( 'ZPRECC',   'single',  'M/S' )         ! convective precip (Zhang)
      call addfld( 'CLOUD',    'multmid', 'FRAC' )        ! Total Cloud Fraction
      call addfld( 'CONCLD',   'multmid', 'FRAC' )        ! Cloud Fraction (convective)
      call addfld( 'CLDTOT',   'single',  'FRAC' )        ! Vertically-integrated  random overlap, total cloud amount
      call addfld( 'CLDLOW',   'single',  'FRAC' )        ! Vertically-integrated  random overlap, low cloud amount
      call addfld( 'CLDMED',   'single',  'FRAC' )        ! Vertically-integrated  random overlap, mid-level cloud amount
      call addfld( 'CLDHGH',   'single',  'FRAC' )        ! Vertically-integrated  random overlap, high cloud amount
      call addfld( 'CME',      'multmid', 'KG/KG/S' )     ! rate of cond-evap within cloud
      call addfld( 'EVAPR',    'multmid', 'KG/KG/S' )     ! rate of evap of falling precip
      call addfld( 'PRAIN',    'multmid', 'KG/KG/S' )     ! rate of conversion of condensate to precipitation
      call addfld( 'PRECL',    'single',  'M/S' )         ! stratiform precip
      call addfld( 'CWAT',     'multmid', 'KG/KG' )       ! Grid box cloud Water
      call addfld( 'XTHETA',   'multmid', 'K' )           ! advected THETA
      call addfld( 'DQR',      'multmid', 'kg/kg/s' )     ! total Q tendency due to rain
      call addfld( 'TAQ',      'multmid', 'kg/kg/s' )     ! total advection tendency of Q
      call addfld( 'TATHETA',  'multmid', 'kg/kg/s' )     ! total advection tendency of potential temperature
      call addfld( 'VDQ',      'multmid', 'kg/kg/s' )     ! vertical diffusion tendency of Q
      call addfld( 'DCQ',      'multmid', 'kg/kg/s' )     ! moist convection tendency of Q
      call addfld( 'PDMFUEN',  'multmid', 'KG/M2/S' )     ! entrainment rate for updraft (Tiedtke)
      call addfld( 'PDMFUDE',  'multmid', 'KG/M2/S' )     ! detrainment rate for updraft (Tiedtke)
      call addfld( 'PDMFDEN',  'multmid', 'KG/M2/S' )     ! entrainment rate for downdraft (Tiedtke)
      call addfld( 'PDMFDDE',  'multmid', 'KG/M2/S' )     ! detrainment rate for downdraft (Tiedtke)
      call addfld( 'PMFU',     'multmid', 'KG/M2/S' )     ! updraft mass flux (Tiedtke)
      call addfld( 'PMFD',     'multmid', 'KG/M2/S' )     ! downdraft mass flux (Tiedtke)
      call addfld( 'NOX',      'multmid', 'VMR' )         ! no+no2 volume mixing ratio
      call addfld( 'NOY',      'multmid', 'VMR' )         ! nox+no3+hno3+ho2no2+2*n2o5 volume mixing ratio
      call addfld( 'DUST1',    'multmid', 'mol/mol' )     ! dust concentration (mol/mol)
      call addfld( 'DUST2',    'multmid', 'mol/mol' )     ! dust concentration (mol/mol)
      call addfld( 'DUST3',    'multmid', 'mol/mol' )     ! dust concentration (mol/mol)
      call addfld( 'DUST4',    'multmid', 'mol/mol' )     ! dust concentration (mol/mol)
      call addfld( 'PCBOT',    'single',  'PA' )          ! pressure at bottom of cloud
      call addfld( 'PCTOP',    'single',  'PA' )          ! pressure at top of cloud
      call addfld( 'DEEPC',    'single',  'FRACTION' )    ! Deep Convection Diagnosed?  1= Deep, 0= not deep
      call addfld( 'SHALC',    'single',  'FRACTION' )    ! Shallow Convection Diagnosed? 1=Shallow, 0=not shallow
      call addfld( 'MIDC',     'single',  'FRACTION' )    ! Mid-level Convection Diagnosed?  1= Mid, 0= not mid
      call addfld( 'CFREQ',    'single',  'FRACTION' )    ! Convection at all diagnosed (either of above 3)?
      call addfld( 'LNO_COL_PROD', 'single',  'TG N/YR' ) ! lightning column no production
      call addfld( 'LNO_PROD', 'multmid', '/cm3/s' )      ! lightning insitu no production
      call addfld( 'SRFALB',   'single', 'FRACTION' )     ! surface albedo
      call addfld( 'FLASHFRQ', 'single', '1/MIN' )        ! flash frequency in grid box per minute (PPP)
      call addfld( 'CLDHGT',   'single', 'KM' )           ! cloud top height
      call addfld( 'DCHGZONE', 'single', 'KM' )           ! depth of discharge zone
      call addfld( 'CGIC',     'single', 'RATIO' )        ! ratio of cloud-ground/intracloud discharges
      call addfld( 'TROPLEV', 'single ', 'KM' )           ! tropopause level
      call addfld( 'DPS-CTM0', 'single', 'PA' )           ! CTM surface pressure change (original)
      call addfld( 'DPS-CTM',  'single', 'PA' )           ! CTM surface pressure change (after fixer)
      call addfld( 'DPS-MET',  'single', 'PA' )           ! met field surface pressure change
      call addfld( 'DDPS0',    'single', 'PA' )           ! surface pressure error (original)
      call addfld( 'DDPS',     'single', 'PA' )           ! surface pressure error (after fixer)
      call addfld( 'SA1_SRC',  'single', 'mmr')
      call addfld( 'SA2_SRC',  'single', 'mmr')
      call addfld( 'SA3_SRC',  'single', 'mmr')
      call addfld( 'SA4_SRC',  'single', 'mmr')
      call addfld( 'SA1_GSV',  'multmid', 'm/s')
      call addfld( 'SA2_GSV',  'multmid', 'm/s')
      call addfld( 'SA3_GSV',  'multmid', 'm/s')
      call addfld( 'SA4_GSV',  'multmid', 'm/s')
      call addfld( 'DTCBS',    'single', 'OD')
      call addfld( 'DTOCS',    'single', 'OD')
      call addfld( 'DTSO4',    'single', 'OD')
      call addfld( 'DTANT',    'single', 'OD')
      call addfld( 'DTSAL',    'single', 'OD')
      call addfld( 'DTDUST',   'single', 'OD')
      call addfld( 'DTTOTAL',  'single', 'OD')
      call addfld( 'U_10m',    'single', 'm/s')
      call addfld( 'H2SO4_VMR' , 'multmid', 'VMR')
      call addfld( 'TSO4_VMR' , 'multmid', 'VMR')
      call addfld( 'TNO3_VMR' , 'multmid', 'VMR')
      call addfld( 'OFFL_OH',  'multmid', 'vmr' )
      call addfld( 'INV_OH',   'multmid', 'molecules/cm3' )
      call addfld( 'CO_SRC1',  'multmid', 'molecules/cm3/sec' )
      call addfld( 'CO_SRC2',  'multmid', 'molecules/cm3/sec' )
!-----------------------------------------------------------------------
! add 2-d output fields of wet deposition
!-----------------------------------------------------------------------
      call addfld( 'HNO3_WD_LOSS'  ,'single','kg/m2/s' )
      call addfld( 'HO2NO2_WD_LOSS','single','kg/m2/s' )
      call addfld( 'ONIT_WD_LOSS'  ,'single','kg/m2/s' )
      call addfld( 'ONITR_WD_LOSS' ,'single','kg/m2/s' )
      call addfld( 'NH3_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'NH4_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'NH4NO3_WD_LOSS','single','kg/m2/s' )
      call addfld( 'SA1_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'SA2_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'SA3_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'SA4_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'SO2_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'SO4_WD_LOSS'   ,'single','kg/m2/s' )
      call addfld( 'SOA_WD_LOSS'   ,'single','kg/m2/s' )

      end subroutine deffld

      subroutine addfld( nm, lvl, units )
!-----------------------------------------------------------------------
! 	... Add field information to master field list
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: &
        nm, &         ! Field name
        lvl, &        ! Field level keyword
        units         ! Field dimensional units

      fld_idx = fld_idx + 1
      if( fld_idx > mxdiagflds ) then
         write(*,*) 'addfld: mxdiagflds must be set larger than ', mxdiagflds
         call endrun
      end if
      if( fld_idx > 1 .and. any( flddat(:fld_idx-1)%name == nm ) ) then
         write(*,*) 'addfld: field ',trim(nm),' already in table'
         call endrun
      end if
      flddat(fld_idx)%name  = nm
      flddat(fld_idx)%lvl   = lvl
      flddat(fld_idx)%units = units

      end subroutine addfld

      subroutine hfile_inti( mflds_in, mcflds_in, nfld, file )
!------------------------------------------------------------------------------------------
!	... Initialize the history netcdf file
!           Note : the netcdf file is expected to be open for variable definition at entry
!------------------------------------------------------------------------------------------

      use netcdf
      use mo_constants, only : r2d, phi, latwts
      use mo_calendar,  only : type
      use plevs,        only : hyam, hybm, hyai, hybi, p0 => ps0

      implicit none

!------------------------------------------------------------------------------------------
!	... dummy arguments
!------------------------------------------------------------------------------------------
      integer, intent(in)          ::  nfld                   ! field count for history file
      integer, intent(in)          ::  file                   ! history file index
      integer, intent(inout)       ::  mflds_in(:)
      character(len=*), intent(in) ::  mcflds_in(:)

!------------------------------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------------------------------
      integer :: dim_cnt, i, ind, lev_flg, acc_flg, var_id, ncid
      integer :: dims(4)
      integer :: mflds(3)
      real    :: latdeg(plat)
      real    :: lam(plon)
      character(len=80) :: attribute
      character(len=32) :: mcflds(2)

!-----------------------------------------------------------------------
!     	... define dimensions
!-----------------------------------------------------------------------
      ncid = hfile(file)%ncid
      call handle_ncerr( NF_DEF_DIM( ncid, 'lon', plon, lon_id ), &
                         'HFILE_INTI: Failed to create longitude dimension' )
      call handle_ncerr( NF_DEF_DIM( ncid, 'lev', plev, lev_id ), &
                         'HFILE_INTI: Failed to create level dimension' )
      call handle_ncerr( NF_DEF_DIM( ncid, 'ilev', plev+1, ilev_id ), &
                         'HFILE_INTI: Failed to create level dimension' )
      call handle_ncerr( NF_DEF_DIM( ncid, 'lat', plat, lat_id ), &
                         'HFILE_INTI: Failed to create latitude dimension' )
      call handle_ncerr( NF_DEF_DIM( ncid, 'time', NF_UNLIMITED, time_id ), &
                         'HFILE_INTI: Failed to create latitude dimension' )
      call handle_ncerr( NF_DEF_DIM( ncid, 'nchar', 80, chr_id ), &
                         'HFILE_INTI: Failed to create char length dimension' )
!-----------------------------------------------------------------------
!     	... Set global attributes
!-----------------------------------------------------------------------
      attribute = 'NCAR-CSM'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, NF_GLOBAL, 'Conventions', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create global conventions attribute' )
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, NF_GLOBAL, 'case', 32, hfile(file)%hdc(1) ), &
                         'HFILE_INTI: Failed to create global case attribute' )
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, NF_GLOBAL, 'title', 80, hfile(file)%mcstit ), &
                         'HFILE_INTI: Failed to create global title attribute' )
!-----------------------------------------------------------------------
!     	... Define variables and attributes; first the scalars
!-----------------------------------------------------------------------
      call handle_ncerr( NF_DEF_VAR( ncid, 'P0', NF_FLOAT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create P0 variable' )
      attribute = 'reference pressure'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create P0 attribute' )
      attribute = 'Pa'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create P0 attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'ntrm', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create ntrm variable' )
      attribute = 'spectral truncation parameter M'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ntrm attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'ntrn', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create ntrn variable' )
      attribute = 'spectral truncation parameter N'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ntrn attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'ntrk', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create ntrk variable' )
      attribute = 'spectral truncation parameter K'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ntrk attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'ndbase', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create ndbase variable' )
      attribute = 'base day for this case'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ndbase attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'nsbase', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create nsbase variable' )
      attribute = 'seconds to complete base day'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create nsbase attribute' )
      attribute = 's'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create nsbase attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'nbdate', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create nbdate variable' )
      attribute = 'base date as 6 digit integer (YYMMDD)'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create nbdate attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'nbsec', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create nbsec variable' )
      attribute = 'seconds to complete base date'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create nbsec attribute' )
      attribute = 's'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create nbsec attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'mdt', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create mdt variable' )
      attribute = 'model timestep'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create mdt attribute' )
      attribute = 's'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create mdt attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'mhisf', NF_INT, 0, dims, var_id ), &
                         'HFILE_INTI: Failed to create mhisf variable' )
      attribute = 'frequency of model writes (timesteps)'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create mhisf attribute' )

!-----------------------------------------------------------------------
!     	... The arrays
!-----------------------------------------------------------------------
      dims(1) = chr_id
      call handle_ncerr( NF_DEF_VAR( ncid, 'current_mss', NF_CHAR, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create current_mss variable' )
      attribute = 'MSS pathname of this file'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create current_mss attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'first_mss', NF_CHAR, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create first_mss variable' )
      attribute = 'MSS pathname of first file for this case'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create first_mss attribute' )
      dims(1) = ilev_id
      call handle_ncerr( NF_DEF_VAR( ncid, 'hyai', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create hyai variable' )
      attribute = 'hybrid A coefficient at layer interfaces'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create hyai attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'hybi', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create hybi variable' )
      attribute = 'hybrid B coefficient at layer interfaces'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create hybi attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'ilev', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create ilev variable' )
      attribute = 'hybrid level at layer interface (1000*(A+B))'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ilev attribute' )
      attribute = 'hybrid_sigma_pressure'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'down'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'positive', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ilev attribute' )
      attribute = 'hyam'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'A_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ilev attribute' )
      attribute = 'hybm'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'B_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ilev attribute' )
      attribute = 'P0'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'P0_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ilev attribute' )
      attribute = 'PS'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'PS_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create ilev attribute' )
      attribute = 'ilev'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'bounds', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      dims(1) = lev_id
      call handle_ncerr( NF_DEF_VAR( ncid, 'hyam', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create hyam variable' )
      attribute = 'hybrid A coefficient at layer midpoints'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create hyam attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'hybm', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create hybm variable' )
      attribute = 'hybrid B coefficient at layer midpoints'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create hybm attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'lev', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create lev variable' )
      attribute = 'hybrid level at layer midpoints (1000*(A+B))'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'hybrid_sigma_pressure'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'down'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'positive', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'hyam'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'A_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'hybm'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'B_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'P0'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'P0_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'PS'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'PS_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      attribute = 'ilev'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'bounds', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lev attribute' )
      dims(1) = lat_id
      call handle_ncerr( NF_DEF_VAR( ncid, 'lat', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create lat variable' )
      attribute = 'latitude'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lat attribute' )
      attribute = 'degrees_north'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lat attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'gw', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create gw variable' )
      attribute = 'gauss weights'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create gw attribute' )
      dims(1) = lon_id
      call handle_ncerr( NF_DEF_VAR( ncid, 'lon', NF_FLOAT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create lon variable' )
      attribute = 'longitude'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lon attribute' )
      attribute = 'degrees_east'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create lon attribute' )
      dims(1) = time_id
      call handle_ncerr( NF_DEF_VAR( ncid, 'days', NF_INT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create days variable' )
      attribute = 'elapsed simulation days for this case'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create days attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'secs', NF_INT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create secs variable' )
      attribute = 'seconds to complete elapsed days'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create secs attribute' )
      attribute = 's'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create secs attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'date', NF_INT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create date variable' )
      attribute = 'current date as 6 digit integer (YYMMDD)'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create date attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'datesec', NF_INT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create datesec variable' )
      attribute = 'seconds to complete current date'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create datesec attribute' )
      attribute = 's'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create datesec attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'time', NF_DOUBLE, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create time variable' )
      attribute = 'simulation time'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create time attribute' )
      attribute = 'days since 0000-01-01 00:00:00'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create time attribute' )
      if( type == '365' ) then
         attribute = '365_days'
      else if( type == 'gregorian' ) then
         attribute = 'gregorian'
      end if
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'calendar', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create time attribute' )
      call handle_ncerr( NF_DEF_VAR( ncid, 'timestep_index', NF_INT, 1, dims, var_id ), &
                         'HFILE_INTI: Failed to create timestep_index variable' )
      attribute = 'iteration counter for current case'
      call handle_ncerr( NF_PUT_ATT_TEXT( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'HFILE_INTI: Failed to create timestep_index attribute' )

      do i = 1,nfld
         mflds(1)  = mflds_in(3*(i-1)+1)
         mcflds(1) = mcflds_in(2*(i-1)+1)
         mcflds(2) = mcflds_in(2*(i-1)+2)
	 lev_flg = mod( mflds(1),10 ) + 1
	 acc_flg = mflds(1)/10 + 1
	 if( lev_flg == 1 ) then
	    dim_cnt = 3
            dims(:3) = (/ lon_id, lat_id, time_id /)
	 else
	    dim_cnt = 4
            dims(:4) = (/ lon_id, lat_id, lev_id, time_id /)
	 end if
#ifdef DEBUG
         write(*,*) 'HFILE_INTI: Creating field #',i,': ',trim(mcflds(1)), &
                    ', lev_flg,acc_flg,dim_cnt = ',lev_flg,acc_flg,dim_cnt
#endif
         call handle_ncerr( NF_DEF_VAR( ncid, trim( mcflds(1) ), &
                                          NF_FLOAT, dim_cnt, dims, mflds(3) ), &
                            'HFILE_INTI: Failed to create ' // trim( mcflds(1) ) // ' variable' )
         call handle_ncerr( NF_PUT_ATT_TEXT( ncid, mflds(3), 'units', &
                                          len_trim(mcflds(2)), trim(mcflds(2)) ), &
                            'HFILE_INTI: Failed to create ' // trim(mcflds(1)) // ' attribute' )
         mflds_in(3*(i-1)+3)   = mflds(3)
      end do

!-----------------------------------------------------------------------
!     	... Leave define and set spatial variables
!-----------------------------------------------------------------------
      call handle_ncerr( NF_ENDDEF( ncid ), 'HFILE_INTI: Failed to leave define mode' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'lon', var_id ), &
                         'HFILE_INTI: Failed to get longitudes variable id' )
      lam(:) = (/ (360.*REAL(i)/REAL(plon),i=0,plon-1) /)
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, lam ), &
                         'HFILE_INTI: Failed to write longitudes variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'lat', var_id ), &
                         'HFILE_INTI: Failed to get latitude variable id' )
      latdeg(:) = r2d * phi(:)
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, latdeg ), &
                         'HFILE_INTI: Failed to write latitudes variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'gw', var_id ), &
                         'HFILE_INTI: Failed to get gaussian weights variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, latwts ), &
                         'HFILE_INTI: Failed to write gaussian weights variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'hyam', var_id ), &
                         'HFILE_INTI: Failed to get hyam variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, hyam ), &
                         'HFILE_INTI: Failed to write hyam variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'hybm', var_id ), &
                         'HFILE_INTI: Failed to get hybm variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, hybm ), &
                         'HFILE_INTI: Failed to write hybm variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'lev', var_id ), &
                         'HFILE_INTI: Failed to get lev variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, 1000.*(hybm+hyam) ), &
                         'HFILE_INTI: Failed to write ilev variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'hyai', var_id ), &
                         'HFILE_INTI: Failed to get hyai variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, hyai ), &
                         'HFILE_INTI: Failed to write hyai variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'hybi', var_id ), &
                         'HFILE_INTI: Failed to get hybi variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, hybi ), &
                         'HFILE_INTI: Failed to write hybi variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'ilev', var_id ), &
                         'HFILE_INTI: Failed to get ilev variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, 1000.*(hybi+hyai) ), &
                         'HFILE_INTI: Failed to write ilev variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'P0', var_id ), &
                         'HFILE_INTI: Failed to get P0 variable id' )
      call handle_ncerr( NF_PUT_VAR_DOUBLE( ncid, var_id, p0 ), &
                         'HFILE_INTI: Failed to write P0 variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'nbdate', var_id ), &
                         'HFILE_INTI: Failed to get nbdate variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(24) ), &
                         'HFILE_INTI: Failed to write nbdate variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'nbsec', var_id ), &
                         'HFILE_INTI: Failed to get nbsec variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(25) ), &
                         'HFILE_INTI: Failed to write nbsec variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'ntrm', var_id ), &
                         'HFILE_INTI: Failed to get ntrm variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(13) ), &
                         'HFILE_INTI: Failed to write ntrm variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'ntrn', var_id ), &
                         'HFILE_INTI: Failed to get ntrn variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(14) ), &
                         'HFILE_INTI: Failed to write ntrn variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'ntrk', var_id ), &
                         'HFILE_INTI: Failed to get ntrk variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(15) ), &
                         'HFILE_INTI: Failed to write ntrk variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'mdt', var_id ), &
                         'HFILE_INTI: Failed to get mdt variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(28) ), &
                         'HFILE_INTI: Failed to write mdt variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'ndbase', var_id ), &
                         'HFILE_INTI: Failed to get ndbase variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(20) ), &
                         'HFILE_INTI: Failed to write ndbase variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'nsbase', var_id ), &
                         'HFILE_INTI: Failed to get nsbase variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%hdi(21) ), &
                         'HFILE_INTI: Failed to write nsbase variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'mhisf', var_id ), &
                         'HFILE_INTI: Failed to get mhisf variable id' )
      call handle_ncerr( NF_PUT_VAR_INT( ncid, var_id, hfile(file)%nhtfrq ), &
                         'HFILE_INTI: Failed to write mhisf variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'current_mss', var_id ), &
                         'HFILE_INTI: Failed to get current_mss variable id' )
      call handle_ncerr( NF_PUT_VAR_TEXT( ncid, var_id, hfile(file)%lnhstc ), &
                         'HFILE_INTI: Failed to write current_mss variable' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'first_mss', var_id ), &
                         'HFILE_INTI: Failed to get first_mss variable id' )
      call handle_ncerr( NF_PUT_VAR_TEXT( ncid, var_id, hfile(file)%lnhstf ), &
                         'HFILE_INTI: Failed to write first_mss variable' )

!-----------------------------------------------------------------------
!     	... Close file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), 'HFILE_INTI: Failed to close history netcdf file ' )

      end subroutine hfile_inti

      subroutine fldlst( outnam, timerep, file )
!-----------------------------------------------------------------------
! 	... Initialize data in /histout/ that is field specific.
!-----------------------------------------------------------------------

      use chem_mods, only : phtcnt, rxt_rate_map

      implicit none

!----------------------------------------------------------------------
! 	... Dummy arguments
!----------------------------------------------------------------------
      integer, intent(in) :: &
        file, &                 ! history file index
        timerep                 ! 0 => instant, 1 => averaged

      character(len=*), intent(in) :: &
        outnam(:)               ! field names

!----------------------------------------------------------------------
! 	... Local variables
!----------------------------------------------------------------------
      integer, parameter :: &
        single  = 0, &  ! flag for single level field
        multint = 1, &  ! flag for multi level field at layer interfaces
        multmid = 2     ! flag for multi level field at layer midpoints

      integer :: &
        i, m, &    ! index
        nz_cnt, &
        base, &
        indx, &
        levrep, &
        timetype
      character(len=32) :: fldname
      character(len=16) :: units

!-----------------------------------------------------------------------
!	... First the "standard" diagnostic fields
!-----------------------------------------------------------------------
      timetype = timerep + 1
      if( hfile(file)%match_cnt(timetype) /= 0 ) then
         do i = 1,hfile(file)%match_cnt(timetype)
            indx = lookup( outnam(i), ndiagfld, diagnams )
            if ( indx > 0 ) then
               if( flddat(indx)%lvl == 'single' ) then
                  levrep = single
               else if( flddat(indx)%lvl == 'multint' ) then
                  levrep = multint
               else if( flddat(indx)%lvl == 'multmid' ) then
                  levrep = multmid
               end if
               call inifld( outnam(i), levrep, timerep, flddat(indx)%units, file )
            end if
         end do
      end if
      do i = 1,hst_cat_max
	 if( i > 4 .and. i < 8 ) then
	    cycle
	 end if
	 if( hfile(file)%histout_cnt(i,timetype) == 0 ) then
	    cycle
	 end if
	 base = hfile(file)%histout_ind(i,timetype) - 1
	 do m = 1,hfile(file)%histout_cnt(i,timetype)
	    if( timetype == 1 ) then
	       indx = hfile(file)%inst_map(base+m)
	       fldname = hfile(file)%hist_inst(base+m)
	    else
	       indx = hfile(file)%timav_map(base+m)
	       fldname = hfile(file)%hist_timav(base+m)
	    end if
	    select case( i )
	       case( 1:2 )
                  levrep = multmid
		  if( stratchem .or. tropchem ) then
                     units  = 'VMR'
		  else
                     units  = 'KG/KG'
		  end if
	       case( 3,18 )
                  levrep = single
                  units  = 'KG/M^2/S'
	       case( 17 )
                  levrep = single
                  units  = 'KG'
	       case( 4 )
                  levrep = single
                  units  = 'CM/S'
	       case( 8 )
                  levrep = multmid
                  units  = '/S'
	       case( 9 )
                  levrep = multmid
                  nz_cnt = count( rxt_rate_map(indx+phtcnt,1:2) /= 0 ) &
                         + rxt_rate_map(indx+phtcnt,3)
                  select case( nz_cnt )
                  case( 1 )
                     units  = '/S'
                  case( 2 )
                     units  = 'CM^3/S'
                  case( 3 )
                     units  = 'CM^6/S'
                  end select
	       case( 11,14:15 )
                  levrep = multmid
                  units  = '/CM^3/S'
	       case( 10 )
                  levrep = multmid
                  units  = '/S'
	       case( 16 )
                  levrep = multmid
                  units  = 'KG'
	       case( 19 )
                  levrep = single
                  units  = 'S'
	       case( 20 )
                  write(*,*) '$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$'
                  write(*,*) 'fldlst: fldname, file = ',trim(fldname),file
                  write(*,*) '$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$'
                  if( index( fldname, '_emis' ) > 0 ) then
                     levrep = single
                     units  = 'KG/M^2/S'
                  else
                     levrep = multmid
                     units  = 'VMR'
                  end if
	       case( 21:22 )
                  levrep = multmid
                  units  = '/CM^3/S'
	    end select
            call inifld( fldname, levrep, timerep, units, file )
         end do
      end do

      end subroutine fldlst

      subroutine fldparse( file )
!-----------------------------------------------------------------------
! 	... Parse the output field lists to determine the number of output
!           fields.  Check lists for inconsistencies, such as duplicated
!           requests or fields that are not contained in the tracnam or
!           diagnosed field lists.
!-----------------------------------------------------------------------

      implicit none

!----------------------------------------------------------------------
! 	... Dummy arguments
!----------------------------------------------------------------------
      integer, intent(in) :: file

!----------------------------------------------------------------------
! 	... Local variables
!----------------------------------------------------------------------
      integer :: i, j
      logical :: found

      hfile(file)%ninstfld = 0
!----------------------------------------------------------------------
! 	... Look for "standard" instantaneous diagnostic fields
!----------------------------------------------------------------------
      if( hfile(file)%match_cnt(1) /= 0 ) then
         do i = 1,hfile(file)%match_cnt(1)
            call findfld( hfile(file)%outinst(1:hfile(file)%match_cnt(1)), i, ndiagfld, &
			  diagnams, hfile(file)%ninstfld, found )
            if( found ) then
	       cycle
	    end if
            write(*,*) &
              'FLDPARSE: requested instantaneous field ',hfile(file)%outinst(i), &
              ' not found... stopping; file = ',file
            call endrun
         end do
      end if
      hfile(file)%ninstfld = hfile(file)%ninstfld + sum( hfile(file)%histout_cnt(:4,1) ) &
                                                  + sum( hfile(file)%histout_cnt(8:hst_cat_max,1) )
      hfile(file)%ntimavfld = 0
!----------------------------------------------------------------------
! 	... Look for "standard" time averaged diagnostic fields
!----------------------------------------------------------------------
      if( hfile(file)%match_cnt(2) /= 0 ) then
         do i = 1,hfile(file)%match_cnt(2)
            call findfld( hfile(file)%outtimav(1:hfile(file)%match_cnt(2)), i, ndiagfld, &
			  diagnams, hfile(file)%ntimavfld, found )
            if( found ) then
	       cycle
	    end if
            write(*,*) &
              'FDLPARSE: requested time average field ',hfile(file)%outtimav(i), &
              ' not found... stopping; file = ',file
            call endrun
         end do
      end if
      hfile(file)%ntimavfld = hfile(file)%ntimavfld + sum( hfile(file)%histout_cnt(:4,2) ) &
                                                    + sum( hfile(file)%histout_cnt(8:hst_cat_max,2) )
!----------------------------------------------------------------------
!     	... Check for duplication between namelist and preprocessor fields
!----------------------------------------------------------------------
      if( hfile(file)%match_cnt(1) > 0 .and. hfile(file)%ninstfld > hfile(file)%match_cnt(1) ) then
         do i = 1,hfile(file)%match_cnt(1)
            do j = hfile(file)%match_cnt(1)+1,hfile(file)%ninstfld
               if( hfile(file)%outinst(i) == hfile(file)%outinst(j) ) then
                  write(*,*) &
                    'FLDPARSE: ',hfile(file)%outinst(i),' duplicate field in file = ',file
                  call endrun
               end if
            end do
         end do
      end if
      if( hfile(file)%match_cnt(2) > 0 .and. hfile(file)%ntimavfld > hfile(file)%match_cnt(2) ) then
         do i = 1,hfile(file)%match_cnt(2)
            do j = hfile(file)%match_cnt(2)+1,hfile(file)%ntimavfld
               if( hfile(file)%outtimav(i) == hfile(file)%outtimav(j) ) then
                  write(*,*) &
                    'FLDPARSE: ',hfile(file)%outtimav(i),' duplicate field in file = ',file
                  call endrun
               end if
            end do
         end do
      end if
!----------------------------------------------------------------------
!     	... Check for duplication between time average and instantaneous lists.
!----------------------------------------------------------------------
      if( hfile(file)%ninstfld > 0 .and. hfile(file)%ntimavfld > 0 ) then
         do i = 1,hfile(file)%ntimavfld
            do j = 1,hfile(file)%ninstfld
               if( hfile(file)%outtimav(i) == hfile(file)%outinst(j) ) then
                  write(*,*) &
                    'FLDPARSE: ',hfile(file)%outtimav(i),' requested as ' // &
                     'both instantaneous and time averaged... stopping; file = ',file
                  call endrun
               end if
            end do
         end do
      end if

      end subroutine fldparse

      subroutine findfld( reqlst, ireq, lenlook, looklst, nfld, found )
!-----------------------------------------------------------------------
! 	... The field at the current position in the request list (reqlst(ireq))
!           is being searched for in looklst.  If it is found, then search
!           through the previously updated part of the reqlst to make sure
!           the field hasn't already been specified.  If it hasn't, then update
!           the reqlst with the new field.  Updating the reqlst is done to
!           overwrite any duplicate specifications.  That way when the list is
!           used to construct the history tape header we dont need to go through
!           this again.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        ireq, &             ! current position in reqlist
        lenlook

      character(len=*), intent(in) :: &
        looklst(lenlook)    ! a list of fields that the model can output

      character(len=*), intent(inout) :: &
        reqlst(:)           ! user requested fields
      integer, intent(inout) :: &
        nfld                ! current number of valid fields in reqlist

      logical, intent(out) :: &
        found               ! t => valid user requested field
                            ! f => requested field not found in input looklst

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        j, k
      logical :: &
        duplicate

      found = .false.
      if( lenlook > 0 ) then
         do j = 1,lenlook
            if( reqlst(ireq) == looklst(j) ) then
               found = .true.
!-----------------------------------------------------------------------
!   	... Make sure field hasnt already been requested.
!-----------------------------------------------------------------------
               if( nfld > 0 ) then
		  duplicate = .false.
                  do k = 1,nfld
                     if( reqlst(ireq) == reqlst(k) ) then
		        duplicate = .true.
		        exit
		     end if
                  end do
!-----------------------------------------------------------------------
!    	... Update list
!-----------------------------------------------------------------------
		  if( .not. duplicate ) then
                     nfld = nfld + 1
                     reqlst(nfld) = reqlst(ireq)
		  end if
               else
                  nfld = 1
               end if
               exit
            end if
         end do
      end if

      end subroutine findfld

      integer function lookup( name, lenlook, looklst )
!-----------------------------------------------------------------------
! 	... Look for the input name in the array looklst.  Return array index.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lenlook

      character(len=*), intent(in) :: &
        name, &
        looklst(lenlook)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: j

      lookup = 0
      if( lenlook > 0 ) then
         do j = 1,lenlook
            if( name == looklst(j) ) then
               lookup = j
               exit
            end if
         end do
      end if

      end function lookup

      subroutine fldinfo( name, nfld, found, ihd, file )
!-----------------------------------------------------------------------
! 	... Get field info needed to write history tape and update counters
!-----------------------------------------------------------------------

      implicit none

!----------------------------------------------------------------------
! 	... Dummy arguments
!----------------------------------------------------------------------
      integer, intent(in) :: &
        nfld, &           ! number of output fields
        file              ! history file index

      character(len=*), intent(in) :: &
        name              ! requested field

      integer, intent(out) :: &
        ihd(3)            ! integer header info for requested field

      logical, intent(out) :: &
        found             ! t => requested field found

!----------------------------------------------------------------------
! 	... Local variables
!----------------------------------------------------------------------
      integer :: j, offset

      found = .false.
      do j = 1,nfld
         if( hfile(file)%hdc(89+2*(j-1)+1) == name ) then
            found = .true.
	    offset = 37 + 3*(j-1)
            ihd(1) = hfile(file)%hdi(offset+1)
            ihd(2) = hfile(file)%hdi(offset+2)
            exit
         end if
      end do

      end subroutine fldinfo

      subroutine inifld( fname, levflg, accflg, units, file )
!-----------------------------------------------------------------------
! 	... Initialize field list information.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        levflg, &   ! 0 = single level, 1 = multi at layer interfaces, 2 = multi at layer midpoints
        accflg, &   ! 0 = instantaneous, 1 = averaged
        file
      character(len=*), intent(in) :: &
        fname, &    !  name of field
        units       !  units of field


!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer, parameter :: inst = 0, avrg = 1, single = 0
      integer :: offset

      hfile(file)%ifcnt = hfile(file)%ifcnt + 1
!-----------------------------------------------------------------------
!	... Check field count number
!-----------------------------------------------------------------------
      if( hfile(file)%ifcnt > hfile(file)%nfld ) then
         write(*,*) 'inifld: parameter nfld not set correctly in inihist'
         write(*,*) 'inifld: file, field = ',file,fname
         write(*,*) 'inifld: ifcnt, nfld = ',hfile(file)%ifcnt,hfile(file)%nfld
         call endrun
      end if

!-----------------------------------------------------------------------
!	... Set output field type counters and indicies
!-----------------------------------------------------------------------
      offset = 37 + 3*(hfile(file)%ifcnt-1)
      hfile(file)%hdi(offset+1) = accflg*10 + levflg
      if( accflg == avrg ) then
	 if( levflg == single ) then
            hfile(file)%tafcnt(1)     = hfile(file)%tafcnt(1) + 1
            hfile(file)%hdi(offset+2) = hfile(file)%tafcnt(1)
	 else
            hfile(file)%tafcnt(2)     = hfile(file)%tafcnt(2) + 1
            hfile(file)%hdi(offset+2) = hfile(file)%tafcnt(2)
	 end if
      else if( accflg == inst ) then
	 if( levflg == single ) then
            hfile(file)%tifcnt(1)     = hfile(file)%tifcnt(1) + 1
            hfile(file)%hdi(offset+2) = hfile(file)%tifcnt(1)
	 else
            hfile(file)%tifcnt(2)     = hfile(file)%tifcnt(2) + 1
            hfile(file)%hdi(offset+2) = hfile(file)%tifcnt(2)
	 end if
      end if
!-----------------------------------------------------------------------
!	... Set output field character information
!-----------------------------------------------------------------------
      offset                    = 89 + 2*(hfile(file)%ifcnt-1)
      hfile(file)%hdc(offset+1) = fname
      hfile(file)%hdc(offset+2) = units

      end subroutine inifld

      subroutine nexthist( endofrun, file )
!-----------------------------------------------------------------------
! 	... Update history module.  Open next history file if we are not
!           at the end of the run.
!-----------------------------------------------------------------------

      use netcdf
      use mo_mpi
      use mo_calendar, only   : addsec2dat
      use mo_file_utils, only : make_filename
      use mo_control, only    : time_t, delt

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: file               ! history file index
      logical, intent(in) :: endofrun           ! true => last timestep or shutdown requested

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: spos, m
      integer :: date, sec
      character(len=168) :: filename
      logical :: monthly

!-----------------------------------------------------------------------
!     	... Reset counter for number of time samples in a file.
!-----------------------------------------------------------------------
      hfile(file)%hdi(3) = 0

      if( io_node ) then
!-----------------------------------------------------------------------
!     	... Build filenames for next history file
!-----------------------------------------------------------------------
        monthly = hfile(file)%thtfrq == 'm'
        date = time_t%date
        sec  = time_t%secs
        if( .not. monthly ) then
          call addsec2dat( delt*hfile(file)%nhtfrq, date, sec )
        end if
        call make_filename( date, sec, file, monthly, hfile(file)%hdc(1), filename )
        if( masternode ) then
          write(*,*) 'nexthist: ',trim( filename )
        end if
        spos = index( hfile(file)%rpath, '/', back=.true. ) + 1
        hfile(file)%rpath(spos:) = trim( filename )
        spos = index( hfile(file)%lpath, '/', back=.true. ) + 1
        hfile(file)%lpath(spos:) = trim( filename )

!-----------------------------------------------------------------------
!     	... Update remote filename of current history file.
!-----------------------------------------------------------------------
        hfile(file)%lnhstc = hfile(file)%rpath

        if( .not. endofrun ) then
!-----------------------------------------------------------------------
!  	... Create next netcdf history file
!-----------------------------------------------------------------------
           call handle_ncerr( nf_create( trim( hfile(file)%lpath ), nf_clobber, hfile(file)%ncid ), &
                              'nexthist: Failed to create ' // trim( hfile(file)%lpath ) )
!-----------------------------------------------------------------------
!  	... Initialize the netcdf file
!-----------------------------------------------------------------------
	   call hfile_inti( hfile(file)%hdi(38:hfile(file)%lenhdi), &
                            hfile(file)%hdc(90:hfile(file)%lenhdc), &
                            hfile(file)%nfld, file )
        end if
      end if

      end subroutine nexthist

      subroutine outfld( fname, field, idim, lonp, lat, file )
!-----------------------------------------------------------------------
! 	... For instantaneous fields, write to output buffer if a time
!           sample is to be written this timestep.  Otherwise return.
!           For time averaged fields, update the history buffer unless
!           it is time to write a time sample.  In that case, do the
!           averaging and write to the output buffer.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: &
        fname         ! name of output field

      integer, intent(in) :: &
        idim, &       ! first dimension of field array
        lonp, &       ! longitude plane index
        lat, &        ! local latitude index
        file          ! history file index

      real, intent(in) :: &
        field(idim,*) ! field data

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: slen
      integer :: mflds(3)      ! integer header info for requested field
      logical :: found         ! t => requested field found
      logical :: is_lt         ! t => requested field is of type "local time"

!-----------------------------------------------------------------------
!  	... Get field information
!-----------------------------------------------------------------------
      call fldinfo( fname, hfile(file)%nfld, found, mflds, file )

      if( found ) then
!-----------------------------------------------------------------------
!  	... Update history buffer
!-----------------------------------------------------------------------
#ifdef DEBUG
	 write(*,'(''outfld: field = '',a,1x,3i3)') trim(fname),lonp,lat,idim
#endif
         if( mflds(1)/10 == 0 .and. .not. hfile(file)%wrhstts ) then   ! instantaneous and not write time
            return
         else                                                          ! time average and/or write time
            slen  = len_trim( fname )
            if( slen > 3 ) then
               is_lt = fname(slen-2:slen) == '_LT'
            else
               is_lt = .false.
            end if
            call wrobuf( mflds, idim, field, lonp, lat, file, is_lt )
         end if
#ifdef DEBUG
      else
	 write(*,'(''outfld: field = '',a,'' not found'')') trim(fname)
#endif
      end if

      end subroutine outfld

      subroutine wrobuf( mflds, dim1, field, ip, lat, file, is_lt )
!-----------------------------------------------------------------------
! 	... Write a latitude slice of data to the output buffer
!-----------------------------------------------------------------------
      use mo_local_time, only : set_local_values
      use mo_grid, only       : plonl

       implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        dim1, &        ! 1st dimension of field array
        ip, &          ! longitude plane index
        lat, &         ! local latitude index
        file, &        ! history file index
        mflds(3)       ! integer header info for field being output

      real, intent(in) :: &
        field(dim1,*)  ! field values

      logical, intent(in) :: is_lt           ! field is local time type

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer, parameter :: single = 1
      integer, parameter :: inst = 1
      integer, parameter :: avrg = 2
      integer :: &
        i, k, &      ! indices
        ndx, &       ! history buffer index
        lev_flg, &   ! level flag
        acc_flg      ! time flag
      real :: wrk(plonl,plev)

      lev_flg = mod( mflds(1),10 ) + 1
      acc_flg = mflds(1)/10  + 1
      ndx     = mflds(2)
#ifdef DEBUG
      write(*,'(''wrobuf: mflds = '',2i3,''; ip, lat = '',2i3)') mflds(:2), ip, lat
#endif

!-----------------------------------------------------------------------
!     	... Move field to output data buffer
!-----------------------------------------------------------------------
      if( lev_flg == single ) then
!-----------------------------------------------------------------------
!     	... Single level
!-----------------------------------------------------------------------
         if( acc_flg == inst ) then
	    do i = 1,plonl
	       hfile(file)%ti_slev((ip-1)*plonl+i,lat,ndx) = field(i,1)
	    end do
	 else if( is_lt ) then
	    do i = 1,plonl
	       wrk(i,1) = hfile(file)%ta_slev((ip-1)*plonl+i,lat,ndx)
	    end do
            call set_local_values( file, wrk, field, dim1, 1, plonl, ip )
	    do i = 1,plonl
	       hfile(file)%ta_slev((ip-1)*plonl+i,lat,ndx) = wrk(i,1)
	    end do
	 else if( acc_flg == avrg ) then
	    do i = 1,plonl
	       hfile(file)%ta_slev((ip-1)*plonl+i,lat,ndx) = &
                 hfile(file)%ta_slev((ip-1)*plonl+i,lat,ndx) + field(i,1)
	    end do
	 end if
      else
!-----------------------------------------------------------------------
!     	... Multi level
!-----------------------------------------------------------------------
         if( acc_flg == inst ) then
	    do k = 1,plev
	       do i = 1,plonl
	          hfile(file)%ti_mlev((ip-1)*plonl+i,lat,k,ndx) = field(i,k)
	       end do
	    end do
	 else if( is_lt ) then
	    do k = 1,plev
	       do i = 1,plonl
	          wrk(i,k) = hfile(file)%ta_mlev((ip-1)*plonl+i,lat,k,ndx)
	       end do
	    end do
            call set_local_values( file, wrk, field, dim1, plev, plonl, ip )
	    do k = 1,plev
	       do i = 1,plonl
	          hfile(file)%ta_mlev((ip-1)*plonl+i,lat,k,ndx) = wrk(i,k)
	       end do
	    end do
	 else if( acc_flg == avrg ) then
	    do k = 1,plev
	       do i = 1,plonl
	          hfile(file)%ta_mlev((ip-1)*plonl+i,lat,k,ndx) = &
                     hfile(file)%ta_mlev((ip-1)*plonl+i,lat,k,ndx) + field(i,k)
	       end do
	    end do
	 end if
      end if

      end subroutine wrobuf

      subroutine hst_gather( nstep, file, pdate, psec, platl )
!-----------------------------------------------------------------------
! 	... Write the node buffers out to the netcdf file
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_calendar, only : newdate, diffdat
      use mo_control,  only : delt
      use mo_grid,     only : nodes

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        nstep, &                     ! Current timestep number
        file, &                      ! file index
        pdate, &                     ! current model date (yyyymmdd)
        psec, &                      ! current seconds in model date
        platl                        ! current seconds in model date

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer ::  j, k, m            ! indicies
      integer ::  itim               ! time index at the end of timestep nstep
      integer ::  nitslf             ! number of timesteps since last time sample
      integer ::  totsec             ! total seconds since last time sample
      integer ::  iday               ! days since last time sample
      integer ::  isec               ! seconds since last time sample relative to eday
      integer ::  ndcur              ! day of current time sample
      integer ::  nscur              ! seconds relative to ndcur
      integer ::  count              ! mpi wrk counter
      integer ::  istat              ! status
      real    ::  ndys0              ! number of days since 0/0/0
      logical ::  do_timav           ! allocate time average buffer
      character(len=10) :: ctime
      character(len=8)  :: cdate

is_hist_write : &
      if( hfile(file)%wrhstts .and. .not. hfile(file)%partial_ta ) then
         itim = nstep
!-----------------------------------------------------------------------
!     	... update index of this time sample in the current history file
!-----------------------------------------------------------------------
         hfile(file)%hdi(3) = hfile(file)%hdi(3) + 1

!-----------------------------------------------------------------------
!     	... update number of timesteps since last time 
!           sample written to history file
!-----------------------------------------------------------------------
         nitslf              = hfile(file)%nhtfrq
         hfile(file)%hdi(19) = nitslf

!-----------------------------------------------------------------------
!     	... update time index of the current history buffer.
!-----------------------------------------------------------------------
         hfile(file)%hdi(17) = itim

!-----------------------------------------------------------------------
!     	... time since last time sample written to history file
!-----------------------------------------------------------------------
         if( .not. hfile(file)%force_inst ) then
	    if( hfile(file)%thtfrq /= 'm' ) then
               totsec = nitslf * hfile(file)%hdi(28)
               iday   = totsec/86400
               isec   = mod( totsec,86400 )
!-----------------------------------------------------------------------
!     	... update day
!-----------------------------------------------------------------------
               ndcur = hfile(file)%hdi(22) + iday
               nscur = hfile(file)%hdi(23) + isec
               if ( nscur >= 86400 ) then
                  ndcur = ndcur + nscur/86400
                  nscur = mod( nscur, 86400 )
               end if
	    else
!-----------------------------------------------------------------------
!     	... monthly output
!-----------------------------------------------------------------------
               ndys0 = diffdat( hfile(file)%hdi(26), hfile(file)%hdi(27), pdate, psec )
	       hfile(file)%fnorm = 1./ real( nint( ndys0 * 86400. / real( delt ) ) )
	       iday = floor( ndys0 )
	       isec = nint( (ndys0 - real(iday))*86400. )
               ndcur = hfile(file)%hdi(22) + iday
               nscur = hfile(file)%hdi(23) + isec
               if ( nscur >= 86400 ) then
                  ndcur = ndcur + nscur/86400
                  nscur = mod( nscur, 86400 )
               end if
	    end if
            hfile(file)%hdi(22) = ndcur
            hfile(file)%hdi(23) = nscur
!-----------------------------------------------------------------------
!     	... update date
!-----------------------------------------------------------------------
            hfile(file)%hdi(26) = pdate
            hfile(file)%hdi(27) = psec
!-----------------------------------------------------------------------
!     	... save day and date from first time sample on a file for the comment string
!-----------------------------------------------------------------------
            if( hfile(file)%hdi(3) == 1 ) then
               hfile(file)%ndcurf = ndcur
               hfile(file)%nscurf = nscur
               hfile(file)%ncdatf = pdate
               hfile(file)%ncsecf = psec
            end if
         end if
      end if is_hist_write
#ifdef USE_MPI
      do_timav = hfile(file)%wrhstts .or. hfile(file)%partial_ta
#ifdef DEBUG
	 write(*,*) 'HST_GATHER: do_timav = ',do_timav
#endif
is_io_node : &
      if( io_node ) then
	 if( .not. ded_io_node ) then
	    count = maxnodes
	 else
	    count = maxnodes+1
	 end if
	 call date_and_time( cdate, ctime )
#ifdef DEBUG
	 write(*,*) 'HST_GATHER: Before buffer allocation for file ',file,' @ ',trim(ctime)
#endif
!-----------------------------------------------------------------------
!     	... allocate gather buffers on io node
!-----------------------------------------------------------------------
         if( hfile(file)%wrhstts .and. hfile(file)%tifcnt(1) > 0 ) then
#ifdef DEBUG
	    write(*,*) 'HST_GATHER: Before gti_slev allocate of ',plon*platl*hfile(file)%tifcnt(1)*maxnodes*8,' bytes'
#endif
	    allocate( hfile(file)%gti_slev(plon,platl,hfile(file)%tifcnt(1),count), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ti_slev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gti_slev )
         end if
         if( hfile(file)%wrhstts .and. hfile(file)%tifcnt(2) > 0 ) then
#ifdef DEBUG
	    write(*,*) 'HST_GATHER: Before gti_mlev allocate of ',plon*platl*plev*hfile(file)%tifcnt(2)*maxnodes*8,' bytes'
#endif
	    allocate( hfile(file)%gti_mlev(plon,platl,plev,hfile(file)%tifcnt(2),count), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ti_mlev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gti_mlev )
         end if
         if( do_timav .and. hfile(file)%tafcnt(1) > 0 ) then
#ifdef DEBUG
	    write(*,*) 'HST_GATHER: Before gta_slev allocate of ',plon*platl*hfile(file)%tafcnt(1)*maxnodes*8,' bytes'
            write(*,*) 'HST_GATHER: Before gta_slev allocate; maxnodes,species,count = ',maxnodes,hfile(file)%tafcnt(1),count
            write(*,*) 'HST_GATHER: Before gta_slev allocate; size ta_slev  = ',size( hfile(file)%ta_slev )
#endif
	    allocate( hfile(file)%gta_slev(plon,platl,hfile(file)%tafcnt(1),count), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ta_slev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gta_slev )
         end if
         if( do_timav .and. hfile(file)%tafcnt(2) > 0 ) then
#ifdef DEBUG
	    write(*,*) 'HST_GATHER: Before gta_mlev allocate of ',plon*platl*plev*hfile(file)%tafcnt(2)*maxnodes*8,' bytes'
            write(*,*) 'HST_GATHER: Before gta_mlev allocate; maxnodes,species,count = ',maxnodes,hfile(file)%tafcnt(2),count
            write(*,*) 'HST_GATHER: Before gta_mlev allocate; size ta_slev  = ',size( hfile(file)%ta_mlev )
#endif
	    allocate( hfile(file)%gta_mlev(plon,platl,plev,hfile(file)%tafcnt(2),count), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ta_mlev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gta_mlev )
         end if
#ifdef DEBUG
	 call date_and_time( cdate, ctime )
	 write(*,*) 'HST_GATHER: After buffer allocation for file ',file,' @ ',trim(ctime)
         write(*,*) 'HST_GATHER: Allocated gather buffers'
#endif
      else is_io_node
!-----------------------------------------------------------------------
!     	... allocate gather buffers on other nodes
!-----------------------------------------------------------------------
         if( hfile(file)%wrhstts .and. hfile(file)%tifcnt(1) > 0 ) then
	    allocate( hfile(file)%gti_slev(1,1,1,1), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ti_slev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gti_slev )
         end if
         if( hfile(file)%wrhstts .and. hfile(file)%tifcnt(2) > 0 ) then
	    allocate( hfile(file)%gti_mlev(1,1,1,1,1), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ti_mlev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gti_mlev )
         end if
         if( do_timav .and. hfile(file)%tafcnt(1) > 0 ) then
	    allocate( hfile(file)%gta_slev(1,1,1,1), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ta_slev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gta_slev )
         end if
         if( do_timav .and. hfile(file)%tafcnt(2) > 0 ) then
	    allocate( hfile(file)%gta_mlev(1,1,1,1,1), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'HST_GATHER: Failed to allocate local ta_mlev; error = ',istat
	       call endrun
	    end if
	 else
	    nullify( hfile(file)%gta_mlev )
         end if
      end if is_io_node
!-----------------------------------------------------------------------
!	... Synchronize
!-----------------------------------------------------------------------
#ifdef DEBUG
      write(*,*) 'HST_GATHER: Before Barrier @ node = ',thisnode
#endif
      call mpi_barrier( MPI_COMM_WORLD, istat )
      if( istat /= MPI_SUCCESS ) then
	 write(*,*) 'HST_GATHER: Barrier call failed; error = ',istat
	 call endrun
      end if
#ifdef DEBUG
      write(*,*) 'HST_GATHER: After Barrier @ node = ',thisnode
      call date_and_time( cdate, ctime )
      write(*,*) 'HST_GATHER: Before buffer gather for file ',file,' @ ',trim(ctime)
#endif

!-----------------------------------------------------------------------
!	... Gather history variables to io node
!-----------------------------------------------------------------------
      if( hfile(file)%wrhstts .and. hfile(file)%tifcnt(1) > 0 ) then
         count = plon*hfile(file)%tifcnt(1)*platl
#ifdef DEBUG
         write(*,*) 'HST_GATHER: Before ti_slev gather; size = ',count
#endif
         call mpi_gather( hfile(file)%ti_slev, count, MPI_DOUBLE_PRECISION, &
                          hfile(file)%gti_slev, count, MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
	 if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'HST_GATHER: MPI_GATHER failed; error code = ',istat
	    call endrun
#ifdef DEBUG
	 else
	    write(*,*) 'HST_GATHER: MPI_GATHER for ti_slev done'
#endif
	 end if
      end if
      if( hfile(file)%wrhstts .and. hfile(file)%tifcnt(2) > 0 ) then
	 count = plon*plev*hfile(file)%tifcnt(2)*platl
#ifdef DEBUG
         write(*,*) 'HST_GATHER: Before ti_mlev gather; size = ',count
#endif
         call mpi_gather( hfile(file)%ti_mlev, count, MPI_DOUBLE_PRECISION, &
                          hfile(file)%gti_mlev, count, MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
	 if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'HST_GATHER: MPI_GATHER failed; error code = ',istat
	    call endrun
#ifdef DEBUG
	 else
	    write(*,*) 'HST_GATHER: MPI_GATHER for ti_mlev done'
#endif
	 end if
      end if
      if( do_timav .and. hfile(file)%tafcnt(1) > 0 ) then
	 count = plon*hfile(file)%tafcnt(1)*platl
#ifdef DEBUG
         write(*,*) 'HST_GATHER: Before ta_slev gather; maxnodes,species,count = ',maxnodes,hfile(file)%tafcnt(1),count
         write(*,*) 'HST_GATHER: Before ta_slev gather; size ta_slev  = ',size( hfile(file)%ta_slev )
         if( io_node ) then
            write(*,*) 'HST_GATHER: Before ta_slev gather; size gta_slev = ',size( hfile(file)%gta_slev )
         end if
#endif
         call mpi_gather( hfile(file)%ta_slev, count, MPI_DOUBLE_PRECISION, &
                          hfile(file)%gta_slev, count, MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
	 if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'HST_GATHER: MPI_GATHER failed; error code = ',istat
	    call endrun
#ifdef DEBUG
	 else
	    write(*,*) 'HST_GATHER: MPI_GATHER for ta_slev done'
#endif
	 end if
      end if
      if( do_timav .and. hfile(file)%tafcnt(2) > 0 ) then
	 count = plon*plev*hfile(file)%tafcnt(2)*platl
#ifdef DEBUG
         write(*,*) 'HST_GATHER: Before ta_slev gather; maxnodes,species,count = ',maxnodes,hfile(file)%tafcnt(2),count
         write(*,*) 'HST_GATHER: Before ta_mlev gather; size ta_mlev  = ',size( hfile(file)%ta_mlev )
         if( io_node ) then
            write(*,*) 'HST_GATHER: Before ta_mlev gather; size gta_mlev = ',size( hfile(file)%gta_mlev )
	 end if
#endif
         call mpi_gather( hfile(file)%ta_mlev, count, MPI_DOUBLE_PRECISION, &
                          hfile(file)%gta_mlev, count, MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
	 if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'HST_GATHER: MPI_GATHER failed; error code = ',istat
	    call endrun
#ifdef DEBUG
	 else
	    write(*,*) 'HST_GATHER: MPI_GATHER for ta_mlev done'
#endif
	 end if
      end if
#ifdef DEBUG
      call date_and_time( cdate, ctime )
      write(*,*) 'HST_GATHER: After buffer gather for file ',file,' @ ',trim(ctime)
#endif
#endif
#undef DEBUG

      end subroutine hst_gather

      subroutine wrnchist( nstep, file, pdate, psec, platl )
!-----------------------------------------------------------------------
! 	... write the node buffers out to the netcdf file
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_calendar, only : newdate, diffdat
      use mo_control,  only : wrestart, delt

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) ::  nstep                        ! Current timestep number
      integer, intent(in) ::  file                         ! file index
      integer, intent(in) ::  pdate                        ! current model date (yyyymmdd)
      integer, intent(in) ::  psec                         ! current seconds in model date
      integer, intent(in) ::  platl

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer            :: j, k, m            ! indicies
      integer            :: istat              ! status
      integer            :: var_id             ! netcdf variable id
      integer            :: ncid               ! netcdf variable id
      real               :: ndys0
      character(len=10)  :: ctime
      character(len=8)   :: cdate

#ifdef DEBUG
      write(*,*) 'wrnchist: Entered subroutine @ node = ',thisnode
      call date_and_time( cdate, ctime )
      write(*,*) 'wrnchist: Before write for file ',file,' @ ',trim(ctime)
#endif
!-----------------------------------------------------------------------
!     	... Write out the buffers
!-----------------------------------------------------------------------
Io_section : &
      if( io_node ) then
!-----------------------------------------------------------------------
!	... check for history restart output
!-----------------------------------------------------------------------
is_history_restart : &
         if( wrestart .and. hfile(file)%partial_ta ) then
            call rhst_create_def( file, hfile(file)%lpath_rhst, hfile(file)%rpath_rhst, pdate, psec, &
                                  ncid, platl )
            call rhst_write( file, ncid, hfile(file)%lpath_rhst, platl )
            call rhst_dispose( hfile(file)%lpath_rhst, hfile(file)%rpath_rhst, .false. )
            if( associated( hfile(file)%gta_slev ) ) then
	       deallocate( hfile(file)%gta_slev )
            end if
            if( associated( hfile(file)%gta_mlev ) ) then
	       deallocate( hfile(file)%gta_mlev )
            end if
         else if( hfile(file)%wrhstts ) then is_history_restart
!-----------------------------------------------------------------------
!     	... Write out the buffers
!-----------------------------------------------------------------------
            write(*,'(1x,''wrnchist: Writing history file '',i2,'' time sample '',i6,'' at date = '' ,i8,'':'',i5)') &
                  file, hfile(file)%hdi(3), pdate, psec
!-----------------------------------------------------------------------
!     	... Form number of days since 0000/01/01
!-----------------------------------------------------------------------
            ndys0 = diffdat( 101, 0, hfile(file)%hdi(26), hfile(file)%hdi(27) )
!-----------------------------------------------------------------------
!     	... open netcdf history file
!-----------------------------------------------------------------------
            call handle_ncerr( nf_open( trim( hfile(file)%lpath ), nf_write, ncid ), &
                               'wrnchist: Failed to open ' // trim( hfile(file)%lpath ) )
            write(*,*) 'wrnchist: Opened file ',trim( hfile(file)%lpath ),' for writing'
            hfile(file)%ncid = ncid
!-----------------------------------------------------------------------
!     	... Write history buffers to netcdf file
!-----------------------------------------------------------------------
            call outncfld( file, platl )
            write(*,*) 'wrnchist: Written buffers'
!-----------------------------------------------------------------------
!     	... On master node write timing information
!-----------------------------------------------------------------------
            call handle_ncerr( nf_inq_varid( ncid, 'date', var_id ), &
                               'wrnchist: Failed to get date variable id' )
            call handle_ncerr( nf_put_var1_int( ncid, var_id, hfile(file)%hdi(3), hfile(file)%hdi(26) ), &
                               'wrnchist: Failed to write date variable' )
            call handle_ncerr( nf_inq_varid( ncid, 'datesec', var_id ), &
                               'wrnchist: Failed to get datesec variable id' )
            call handle_ncerr( nf_put_var1_int( ncid, var_id, hfile(file)%hdi(3), hfile(file)%hdi(27) ), &
                               'wrnchist: Failed to write datesec variable' )
            call handle_ncerr( nf_inq_varid( ncid, 'timestep_index', var_id ), &
                               'wrnchist: Failed to get timestep_index variable id' )
            call handle_ncerr( nf_put_var1_int( ncid, var_id, hfile(file)%hdi(3), nstep ), &
                               'wrnchist: Failed to write timestep_index variable' )
            call handle_ncerr( nf_inq_varid( ncid, 'time', var_id ), &
                               'wrnchist: Failed to get time variable id' )
            call handle_ncerr( nf_put_var1_double( ncid, var_id, hfile(file)%hdi(3), ndys0 ), &
                               'wrnchist: Failed to write date variable' )
            call handle_ncerr( nf_inq_varid( ncid, 'days', var_id ), &
                               'wrnchist: Failed to get days variable id' )
            call handle_ncerr( nf_put_var1_int( ncid, var_id, hfile(file)%hdi(3), hfile(file)%hdi(22) ), &
                               'wrnchist: Failed to write days variable' )
            call handle_ncerr( nf_inq_varid( ncid, 'secs', var_id ), &
                               'wrnchist: Failed to get secs variable id' )
            call handle_ncerr( nf_put_var1_int( ncid, var_id, hfile(file)%hdi(3), hfile(file)%hdi(23) ), &
                               'wrnchist: Failed to write secs variable' )
!-----------------------------------------------------------------------
!     	... Close netcdf history file
!-----------------------------------------------------------------------
            call handle_ncerr( nf_close( ncid ), 'wrnchist: Failed to close file ' // trim( hfile(file)%lpath ) )
            write(*,*) 'wrnchist: Closed file ',trim( hfile(file)%lpath )
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Deallocate the global buffers
!-----------------------------------------------------------------------
            if( associated( hfile(file)%gti_slev) ) then
               deallocate( hfile(file)%gti_slev )
            end if
            if( associated( hfile(file)%gti_mlev) ) then
	       deallocate( hfile(file)%gti_mlev )
            end if
            if( associated( hfile(file)%gta_slev ) ) then
	       deallocate( hfile(file)%gta_slev )
            end if
            if( associated( hfile(file)%gta_mlev ) ) then
	       deallocate( hfile(file)%gta_mlev )
            end if
#endif
         end if is_history_restart
      end if Io_section
#ifdef DEBUG
      call date_and_time( cdate, ctime )
      write(*,*) 'wrnchist: After write for file ',file,' @ ',trim(ctime)
#endif

!-----------------------------------------------------------------------
! 	... Deallocate instantaneous work buffers
!-----------------------------------------------------------------------
         if( associated( hfile(file)%ti_slev) ) then
            deallocate( hfile(file)%ti_slev,stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'wrnchist: Failed to deallocate ti_slev for file ',file,'; error = ',istat
	       call endrun
	    end if
         end if
         if( associated( hfile(file)%ti_mlev) ) then
            deallocate( hfile(file)%ti_mlev,stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'wrnchist: Failed to deallocate ti_mlev for file ',file,'; error = ',istat
	       call endrun
	    end if
         end if
Comp_section : &
      if( comp_node ) then
!-----------------------------------------------------------------------
! 	... Zero time averaging buffers
!-----------------------------------------------------------------------
is_hst_write : &
         if( hfile(file)%wrhstts .and. .not. hfile(file)%partial_ta ) then
            if( hfile(file)%tafcnt(1) > 0 ) then
	       do m = 1,hfile(file)%tafcnt(1)
                  do j = 1,platl
	             hfile(file)%ta_slev(:plon,j,m) = 0.
	          end do
	       end do
            end if
            if( hfile(file)%tafcnt(2) > 0 ) then
	       do m = 1,hfile(file)%tafcnt(2)
	          do k = 1,plev
                     do j = 1,platl
	                hfile(file)%ta_mlev(:plon,j,k,m) = 0.
	             end do
	          end do
	       end do
            end if
         end if is_hst_write
      end if Comp_section
#ifdef DEBUG
      write(*,*) 'wrnchist: Exited subroutine @ node = ',thisnode
#endif

      end subroutine wrnchist

      subroutine rhst_create_def( file, lpath_rhst, rpath_rhst, ncdate, ncsec, &
                                  ncid, platl )
!-----------------------------------------------------------------------
!	... create and define history restart file
!-----------------------------------------------------------------------

      use netcdf
      use mo_grid, only : nodes

      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)             :: file
      integer, intent(in)             :: ncdate
      integer, intent(in)             :: ncsec
      integer, intent(in)             :: platl
      integer, intent(inout)          :: ncid
      character(len=*), intent(inout) :: lpath_rhst
      character(len=*), intent(inout) :: rpath_rhst

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer            :: bndx, endx
      integer            :: varid
      integer            :: lon_id, lat_id, lev_id, slev_id, mlev_id
      integer            :: dims(4)
      character(len=192) :: fullpath

      bndx = index( hfile(file)%lpath, '.h', back=.true. ) + 1
      fullpath = hfile(file)%lpath(:bndx-1) // 'r' // hfile(file)%lpath(bndx:)

      write(*,*) '============================='
      write(*,*) 'rhst_create_def: local full path'
      write(*,*) trim(fullpath)
      write(*,*) '============================='
      lpath_rhst = trim(fullpath)

!-----------------------------------------------------------------------
!     	... create netcdf restart history file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_create( trim( fullpath ), nf_clobber, ncid ), &
                         'rhst_create_def: Failed to create ' // trim( fullpath ) )

!-----------------------------------------------------------------------
!     	... define the dimensions
!-----------------------------------------------------------------------
      call handle_ncerr( nf_def_dim( ncid, 'lon', plon, lon_id ), &
                         'rhst_create_def: Failed to define longitude dimension' )
      call handle_ncerr( nf_def_dim( ncid, 'lat', plat, lat_id ), &
                         'rhst_create_def: Failed to define latitude dimension' )
      call handle_ncerr( nf_def_dim( ncid, 'lev', plev, lev_id ), &
                         'rhst_create_def: Failed to define level dimension' )
      if( hfile(file)%tafcnt(1) > 0 ) then
         call handle_ncerr( nf_def_dim( ncid, 'slev_var_cnt', hfile(file)%tafcnt(1), slev_id ), &
                            'rhst_create_def: Failed to define single level variable count dimension' )
      end if
      if( hfile(file)%tafcnt(2) > 0 ) then
         call handle_ncerr( nf_def_dim( ncid, 'mlev_var_cnt', hfile(file)%tafcnt(2), mlev_id ), &
                            'rhst_create_def: Failed to define multi level variable count dimension' )
      end if

!-----------------------------------------------------------------------
!     	... define the variables
!-----------------------------------------------------------------------
      if( hfile(file)%tafcnt(1) > 0 ) then
         dims(1:3) = (/ lon_id, lat_id, slev_id /)
         call handle_ncerr( nf_def_var( ncid, 'gta_slev', nf_double, 3, dims(1:3), varid ), &
                            'rhst_create_def: Failed to define gta slev variable' )
      end if
      if( hfile(file)%tafcnt(2) > 0 ) then
         dims(:) = (/ lon_id, lat_id, lev_id, mlev_id /)
         call handle_ncerr( nf_def_var( ncid, 'gta_mlev', nf_double, 4, dims, varid ), &
                            'rhst_create_def: Failed to define gta mlev variable' )
      end if

!-----------------------------------------------------------------------
!     	... leave define mode
!-----------------------------------------------------------------------
      call handle_ncerr( nf_enddef( ncid ), 'rhst_create_def: Failed to leave define mode' )

      bndx = index( hfile(file)%rpath, '.h', back=.true. ) + 1
      fullpath = hfile(file)%rpath(:bndx-1) // 'r' // hfile(file)%rpath(bndx:)

      write(*,*) '============================='
      write(*,*) 'rhst_create_def: remote full path'
      write(*,*) trim(fullpath)
      write(*,*) '============================='
      rpath_rhst = trim(fullpath)

      end subroutine rhst_create_def

      subroutine rhst_read( file, rhst_lpath, rhst_rpath, platl )
!-----------------------------------------------------------------------
!	... read history restart file
!-----------------------------------------------------------------------
      
      use netcdf
      use mo_mpi
      use mo_file_utils, only : open_netcdf_file

      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: file
      integer, intent(in) :: platl
      character(len=*), intent(in) :: rhst_lpath
      character(len=*), intent(in) :: rhst_rpath

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: ncid
      integer :: spos
      integer :: varid
      integer :: start(4)
      integer :: cnt(4)
      character(len=64) :: filename
      character(len=64) :: loc_filepath
      character(len=64) :: rem_filepath

!-----------------------------------------------------------------------
!     	... get history restart file to local disk
!-----------------------------------------------------------------------
      spos = index( rhst_lpath, '/', back=.true. )
      filename = rhst_lpath(spos+1:)
      if( spos /= 1 ) then
         loc_filepath = rhst_lpath(:spos)
      else
         loc_filepath = './'
      end if
      spos = index( rhst_rpath, '/', back=.true. )
      if( spos /= 1 ) then
         rem_filepath = rhst_rpath(:spos)
      else
         rem_filepath = './'
      end if
      write(*,*) '$$$$$$$$$$$$$$$$$$$$$$$$$$$$'
      write(*,*) 'filename = ',trim(filename)
      write(*,*) 'loc_filepath = ',trim(loc_filepath)
      write(*,*) 'rem_filepath = ',trim(rem_filepath)
      write(*,*) '$$$$$$$$$$$$$$$$$$$$$$$$$$$$'

      ncid = open_netcdf_file( trim(filename), trim(loc_filepath), trim(rem_filepath) )

      if( comp_node ) then
         if( hfile(file)%tafcnt(1) > 0 ) then
!-----------------------------------------------------------------------
!     	... read the single level history restart variable
!-----------------------------------------------------------------------
            call handle_ncerr( nf_inq_varid( ncid, 'gta_slev', varid ), &
                               'rhst_read: Failed to get gta_slev variable id' )
            start(1:3) = (/ 1, base_lat+1, 1 /)
            cnt(1:3)   = (/ plon, platl, hfile(file)%tafcnt(1) /)
            call handle_ncerr( nf_get_vara_double( ncid, varid, start(1:3), cnt(1:3), hfile(file)%ta_slev ), &
                                  'rhst_read: Failed to read ta_slev' )
         end if
         if( hfile(file)%tafcnt(2) > 0 ) then
!-----------------------------------------------------------------------
!     	... read the multi level history restart variable
!-----------------------------------------------------------------------
            call handle_ncerr( nf_inq_varid( ncid, 'gta_mlev', varid ), &
                               'rhst_read: Failed to get gta_mlev variable id' )
            start = (/ 1, base_lat+1, 1, 1 /)
            cnt   = (/ plon, platl, plev, hfile(file)%tafcnt(2) /)
            call handle_ncerr( nf_get_vara_double( ncid, varid, start, cnt, hfile(file)%ta_mlev ), &
                                  'rhst_read: Failed to read ta_mlev' )
         end if
!-----------------------------------------------------------------------
!  	... close netcdf history restart file
!-----------------------------------------------------------------------
         call handle_ncerr( nf_close( ncid ), 'rhst_read: Failed to close ' // trim( rhst_lpath ) )
         write(*,*) 'rhst_read: closed file ',trim(rhst_lpath)
      end if

      end subroutine rhst_read

      subroutine rhst_write( file, ncid, fullpath, platl )
!-----------------------------------------------------------------------
!	... write history restart file
!-----------------------------------------------------------------------

      use netcdf
      use mo_mpi

      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)             :: file
      integer, intent(in)             :: ncid
      integer, intent(in)             :: platl
      character(len=*), intent(in)    :: fullpath

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer           :: j
      integer           :: varid
      integer           :: var_cnt
      integer           :: astat
      integer           :: node
      real, allocatable :: wrk2d(:,:,:)
      real, allocatable :: wrk3d(:,:,:,:)

!-----------------------------------------------------------------------
!     	... write the variables
!-----------------------------------------------------------------------
      var_cnt = hfile(file)%tafcnt(1)
      if( var_cnt > 0 ) then
         call handle_ncerr( nf_inq_varid( ncid, 'gta_slev', varid ), &
                            'rhst_create_def: Failed to get gta slev variable id' )
#ifdef USE_MPI
         allocate( wrk2d(plon,plat,var_cnt),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'rhst_write: failed to allocate wrk2d; error = ',astat
            call endrun
         end if
         do node = 1,maxnodes
            wrk2d(:,(node-1)*platl+1:node*platl,:) = hfile(file)%gta_slev(:,:,:,node)
         end do
         call handle_ncerr( nf_put_var_double( ncid, varid, wrk2d ), &
                            'rhst_create_def: Failed to write gta slev variable' )
         deallocate( wrk2d )
#else
         call handle_ncerr( nf_put_var_double( ncid, varid, hfile(file)%ta_slev ), &
                            'rhst_create_def: Failed to write gta slev variable' )
#endif
      end if
      var_cnt = hfile(file)%tafcnt(2)
      if( var_cnt > 0 ) then
         call handle_ncerr( nf_inq_varid( ncid, 'gta_mlev', varid ), &
                            'rhst_create_def: Failed to get gta mlev variable id' )
#ifdef USE_MPI
         allocate( wrk3d(plon,plat,plev,var_cnt),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'rhst_write: failed to allocate wrk3d; error = ',astat
            call endrun
         end if
         do node = 1,maxnodes
            wrk3d(:,(node-1)*platl+1:node*platl,:,:) = hfile(file)%gta_mlev(:,:,:,:,node)
         end do
         call handle_ncerr( nf_put_var_double( ncid, varid, wrk3d ), &
                            'rhst_create_def: Failed to write gta slev variable' )
         deallocate( wrk3d )
#else
         call handle_ncerr( nf_put_var_double( ncid, varid, hfile(file)%ta_mlev ), &
                            'rhst_create_def: Failed to write gta slev variable' )
#endif
      end if

!-----------------------------------------------------------------------
!     	... close history restart file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), 'rhst_write: Failed to close file ' // trim(fullpath) )

      end subroutine rhst_write

      subroutine rhst_dispose( lpath_rhst, rpath_rhst, noremove )
!-----------------------------------------------------------------------
!	... dispose history restart file
!-----------------------------------------------------------------------

      use mo_file_utils, only : dispose
      use mo_control, only    : laststep

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: lpath_rhst
      character(len=*), intent(in) :: rpath_rhst
      logical, intent(in)          :: noremove

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      character(len=80) :: com
      logical           :: remove

      if( laststep .or. noremove ) then
         remove = .false.
      else
         remove = rmout
      end if
      com = ' '
      call dispose( trim(lpath_rhst), trim(rpath_rhst), irt, async, com, &
                    msvol, wpasswd, remove, msclass=msclass )

      end subroutine rhst_dispose

      subroutine outncfld( file, platl )
!-----------------------------------------------------------------------
!	... output the fields to the netcdf history file
!-----------------------------------------------------------------------

      use netcdf
      use mo_mpi, only : thisnode, maxnodes

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: file
      integer, intent(in) :: platl

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer, parameter :: single = 1, inst = 1, avrg = 2
      integer :: &
        i, j, k, m, node, base_lat, &
        slen, &
        fldno, &
        lev_flg, &
        acc_flg, &
        var_id, &
        ndx, &
        ncid
      integer, dimension(4) :: &
        start4, &                               ! starting dimension indicies
        count4                                  ! length of each dimension
      integer, dimension(3) :: &
        start3, &                               ! starting dimension indicies
        count3                                  ! length of each dimension
      real :: time_factor
      real :: wrk2d(plon,plat)
      real :: wrk3d(plon,plat,plev)
      logical :: is_lt                          ! local time flag

      character(len=32) :: fname

!-----------------------------------------------------------------------
!  	... Form time average
!-----------------------------------------------------------------------
#ifdef DEBUG
      write(*,*) 'OUTNCFLD: tafcnt = ',hfile(file)%tafcnt
      write(*,*) 'OUTNCFLD: fnorm  = ',hfile(file)%fnorm
#endif

      ncid = hfile(file)%ncid
      start3(:) = (/ 1, 1, hfile(file)%hdi(3) /)
      count3(:) = (/ plon, plat, 1 /)
      start4(:) = (/ 1, 1, 1, hfile(file)%hdi(3) /)
      count4(:4) = (/ plon, plat, plev, 1 /)
Fld_loop: &
      do fldno = 1,hfile(file)%nfld
         fname   = hfile(file)%hdc(88+2*fldno)
         slen    = len_trim( fname )
         if( slen > 3 ) then
            is_lt = fname(slen-2:slen) == '_LT'
         else
            is_lt = .false.
         end if
	 ndx     = 35 + 3*fldno
	 lev_flg = mod( hfile(file)%hdi(ndx),10 ) + 1
	 acc_flg = hfile(file)%hdi(ndx)/10 + 1
	 var_id  = hfile(file)%hdi(ndx+2)
	 ndx     = hfile(file)%hdi(ndx+1)
#ifdef DEBUG
	 write(*,*) 'field #,name = ',fldno,trim(fname),lev_flg,acc_flg,var_id,ndx
#endif
Sng_lev: if( lev_flg == single ) then
!-----------------------------------------------------------------------
!  	... Single level field
!-----------------------------------------------------------------------
#ifdef DEBUG
	    write(*,'(''OUTNCFLD: writing field '',a)') trim( fname)
	    write(*,'(''OUTNCFLD: start3 = '',3i3,'' count3 = '',3i3)') start3, count3
#endif
	    if( acc_flg == inst ) then
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             wrk2d(:,base_lat+j) = hfile(file)%gti_slev(:,j,ndx,node)
#else
	             wrk2d(:,base_lat+j) = hfile(file)%ti_slev(:,j,ndx)
#endif
		  end do
	       end do
	    else
               if( is_lt ) then
                  time_factor = 1.
               else
                  time_factor = hfile(file)%fnorm
               end if
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             wrk2d(:,base_lat+j) = hfile(file)%gta_slev(:,j,ndx,node)*time_factor
#else
	             wrk2d(:,base_lat+j) = hfile(file)%ta_slev(:,j,ndx)*time_factor
#endif
		  end do
	       end do
	    end if
!-----------------------------------------------------------------------
!  	... Output field
!-----------------------------------------------------------------------
            call handle_ncerr( nf_put_vara_double( ncid, var_id, start3, count3, wrk2d ), &
                               'OUTNCFLD: Failed to write ' // trim(fname) // ' variable' )
	 else Sng_lev
!-----------------------------------------------------------------------
!  	... Multi level field
!-----------------------------------------------------------------------
#ifdef DEBUG
	    write(*,'(''OUTNCFLD: writing field '',a)') trim( fname )
	    write(*,'(''OUTNCFLD: start4 = '',4i3,'' count4 = '',4i3)') start4, count4
#endif
	    if( acc_flg == inst ) then
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             wrk3d(:,base_lat+j,:) = hfile(file)%gti_mlev(:,j,:,ndx,node)
#else
	             wrk3d(:,base_lat+j,:) = hfile(file)%ti_mlev(:,j,:,ndx)
#endif
		  end do
	       end do
	    else if( is_lt ) then
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             wrk3d(:,base_lat+j,:) = hfile(file)%gta_mlev(:,j,:,ndx,node)
#else
		     do k = 1,plev
		        do i = 1,plon
	                   wrk3d(i,base_lat+j,k) = hfile(file)%ta_mlev(i,j,k,ndx)
		        end do
		     end do
#endif
		  end do
	       end do
	    else if( acc_flg == avrg ) then
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             wrk3d(:,base_lat+j,:) = hfile(file)%gta_mlev(:,j,:,ndx,node)*hfile(file)%fnorm
#else
		     do k = 1,plev
		        do i = 1,plon
	                   wrk3d(i,base_lat+j,k) = hfile(file)%ta_mlev(i,j,k,ndx)*hfile(file)%fnorm
		        end do
		     end do
#endif
		  end do
	       end do
	    end if
!-----------------------------------------------------------------------
!  	... Output field
!-----------------------------------------------------------------------
#ifdef DEBUG
	    write(*,'(''OUTNCFLD: Before netcdf call for field '',a)') trim( fname )
#endif
            call handle_ncerr( nf_put_vara_double( ncid, var_id, start4, count4, wrk3d ), &
                               'OUTNCFLD: Failed to write ' // trim(fname) // ' variable' )
#ifdef DEBUG
	    write(*,'(''OUTNCFLD: Written field '',a)') trim( fname )
#endif
         end if Sng_lev
      end do Fld_loop

      end subroutine outncfld

      subroutine read_pta_vars( file, platl )
!-----------------------------------------------------------------------
! 	... Read partial time averaged fields into the history accumulators
!-----------------------------------------------------------------------

      use MO_MPI

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: file             ! file index
      integer, intent(in) :: platl

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: count, istat

      if( masternode ) then
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Allocate scatter buffers
!-----------------------------------------------------------------------
         if( hfile(file)%tafcnt(1) > 0 ) then
#ifdef DEBUG
	    write(*,*) 'READ_PTA_VARS: Before gta_slev allocate of ',plon*platl*hfile(file)%tafcnt(1)*maxnodes*8,' bytes'
#endif
	    allocate( hfile(file)%gta_slev(plon,platl,hfile(file)%tafcnt(1),maxnodes), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'READ_PTA_VARS: Failed to allocate local ta_slev; error = ',istat
	       call endrun
	    end if
         end if
         if( hfile(file)%tafcnt(2) > 0 ) then
#ifdef DEBUG
	    write(*,*) 'READ_PTA_VARS: Before gta_mlev allocate of ',plon*platl*plev*hfile(file)%tafcnt(2)*maxnodes*8,' bytes'
#endif
	    allocate( hfile(file)%gta_mlev(plon,platl,plev,hfile(file)%tafcnt(2),maxnodes), stat=istat )
	    if( istat /= 0 ) then
	       write(*,*) 'READ_PTA_VARS: Failed to allocate local ta_mlev; error = ',istat
	       call endrun
	    end if
         end if
         write(*,*) 'READ_PTA_VARS: Allocated scatter buffers'
#endif
	 call rdncfld( file, platl )
      end if
#ifdef USE_MPI
!-----------------------------------------------------------------------
!	... Synchronize
!-----------------------------------------------------------------------
      call mpi_barrier( mpi_comm_comp, istat )
      if( istat /= MPI_SUCCESS ) then
	 write(*,*) 'READ_PTA_VARS: Barrier call failed; error = ',istat
	 call endrun
      end if

!-----------------------------------------------------------------------
!	... Scatter time averaged history variables
!-----------------------------------------------------------------------
      if( hfile(file)%tafcnt(1) > 0 ) then
	 count = plon*hfile(file)%tafcnt(1)*platl
         call mpi_scatter( hfile(file)%gta_slev, count, MPI_DOUBLE_PRECISION, &
                           hfile(file)%ta_slev, count, MPI_DOUBLE_PRECISION, &
                           0, mpi_comm_comp, istat )
	 if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'READ_PTA_VARS: MPI_SCATTER failed; error code = ',istat
	    call endrun
#ifdef DEBUG
	 else
	    write(*,*) 'READ_PTA_VARS: MPI_SCATTER for ta_slev done'
#endif
	 end if
      end if
      if( hfile(file)%tafcnt(2) > 0 ) then
	 count = plon*plev*hfile(file)%tafcnt(2)*platl
         call mpi_scatter( hfile(file)%gta_mlev, count, MPI_DOUBLE_PRECISION, &
                           hfile(file)%ta_mlev, count, MPI_DOUBLE_PRECISION, &
                           0, mpi_comm_comp, istat )
	 if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'READ_PTA_VARS: MPI_SCATTER failed; error code = ',istat
	    call endrun
#ifdef DEBUG
	 else
	    write(*,*) 'READ_PTA_VARS: MPI_SCATTER for ta_mlev done'
#endif
	 end if
      end if
#endif

      if( masternode ) then
         if( associated( hfile(file)%gta_slev ) ) then
            deallocate( hfile(file)%gta_slev )
         end if
         if( associated( hfile(file)%gta_mlev ) ) then
            deallocate( hfile(file)%gta_mlev )
         end if
      end if

      end subroutine read_pta_vars

      subroutine rdncfld( file, platl )
!-----------------------------------------------------------------------
!	... Output the fields to the netcdf history file
!-----------------------------------------------------------------------

      use netcdf
      use MO_MPI, only : maxnodes

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: file
      integer, intent(in) :: platl

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer, parameter :: single = 1, inst = 1, avrg = 2
      integer :: &
        j, k, m, node, base_lat, &
        fldno, &
        lev_flg, &
        acc_flg, &
        var_id, &
        ndx, &
        ncid
      integer, dimension(4) :: &
        start4, &                                 ! starting dimension indicies
        count4                                    ! length of each dimension
      integer, dimension(3) :: &
        start3, &                                 ! starting dimension indicies
        count3                                    ! length of each dimension
      real :: wrk2d(plon,plat)
      real :: wrk3d(plon,plat,plev)

      character(len=32) :: fname

!-----------------------------------------------------------------------
!  	... Form time average
!-----------------------------------------------------------------------
#ifdef DEBUG
      write(*,*) 'RDNCFLD: tafcnt = ',hfile(file)%tafcnt
      write(*,*) 'RDNCFLD: fnorm = ',hfile(file)%fnorm
#endif
      ncid       = hfile(file)%ncid
      start3(:)  = (/ 1, 1, hfile(file)%hdi(3) /)
      count3(:)  = (/ plon, plat, 1 /)
      start4(:)  = (/ 1, 1, 1, hfile(file)%hdi(3) /)
      count4(:4) = (/ plon, plat, plev, 1 /)
      do fldno = 1,hfile(file)%nfld
         fname   = hfile(file)%hdc(88+2*fldno)
	 ndx     = 35 + 3*fldno
	 lev_flg = mod( hfile(file)%hdi(ndx),10 ) + 1
	 acc_flg = hfile(file)%hdi(ndx)/10 + 1
	 var_id  = hfile(file)%hdi(ndx+2)
	 ndx     = hfile(file)%hdi(ndx+1)
#ifdef DEBUG
	 write(*,*) 'field #,name=',fldno,trim(fname),lev_flg,acc_flg,var_id,ndx
#endif
	 if( lev_flg == single ) then
!-----------------------------------------------------------------------
!  	... Single level field
!-----------------------------------------------------------------------
#ifdef DEBUG
	    write(*,'(''RDNCFLD: reading field '',a)') trim( fname)
	    write(*,'(''RDNCFLD: start3 = '',3i3,'' count3 = '',3i3)') start3, count3
#endif
	    if( acc_flg == avrg ) then
!-----------------------------------------------------------------------
!  	... Read field
!-----------------------------------------------------------------------
               call handle_ncerr( nf_get_vara_double( ncid, var_id, start3, count3, wrk2d ), &
                                  'RDNCFLD: Failed to read ' // trim(fname) // ' variable' )
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             hfile(file)%gta_slev(:,j,ndx,node) = wrk2d(:,base_lat+j)
#else
	             hfile(file)%ta_slev(:,j,ndx) = wrk2d(:,base_lat+j)
#endif
		  end do
	       end do
	    end if
	 else
!-----------------------------------------------------------------------
!  	... Multi level field
!-----------------------------------------------------------------------
#ifdef DEBUG
	    write(*,'(''RDNCFLD: reading field '',a)') trim( fname )
	    write(*,'(''RDNCFLD: start4 = '',4i3,'' count4 = '',4i3)') start4, count4
#endif
	    if( acc_flg == avrg ) then
!-----------------------------------------------------------------------
!  	... Read field
!-----------------------------------------------------------------------
               call handle_ncerr( nf_get_vara_double( ncid, var_id, start4, count4, wrk3d ), &
                                  'RDNCFLD: Failed to write ' // trim(fname) // ' variable' )
	       do node = 1,maxnodes
		  base_lat = (node - 1)*platl
	          do j = 1,platl
#ifdef USE_MPI
	             hfile(file)%gta_mlev(:,j,:,ndx,node) = wrk3d(:,base_lat+j,:)
#else
	             hfile(file)%ta_mlev(:,j,:,ndx) = wrk3d(:,base_lat+j,:)
#endif
		  end do
	       end do
	    end if
#ifdef DEBUG
	    write(*,'(''RDNCFLD: Read field '',a)') trim( fname )
#endif
         end if
      end do

      end subroutine rdncfld

      subroutine disphst( laststep, noremove, file )
!-----------------------------------------------------------------------
! 	... Copy the history file to remote storage device.  Remove the
!           local copy if requested.
!-----------------------------------------------------------------------

      use mo_mscomment,  only : ddicom
      use mo_file_utils, only : dispose
      use mo_mpi,        only : io_node

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        file         ! history file index
      logical, intent(in) :: &
        laststep, &  ! true => current timestep is final timestep for run
        noremove     ! true => dont remove history file after disposing.
                     ! (used to override the user specified value)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      logical :: &
        remove      ! true => remove history file after disposing

      character(len=80) :: &
        com         ! comment field for mss

      if( io_node ) then
         if( laststep .or. noremove ) then
            remove = .false.
         else
            remove = rmout
         end if
         call ddicom( hfile(file)%ndcurf, hfile(file)%nscurf, hfile(file)%hdi(22), hfile(file)%hdi(23), &
                      hfile(file)%ncdatf, hfile(file)%ncsecf, hfile(file)%hdi(26), hfile(file)%hdi(27), com )
         call dispose( hfile(file)%lpath, hfile(file)%rpath, irt, async, com, &
                       msvol, wpasswd, remove, msclass=msclass )
      end if

      end subroutine disphst

      integer function qhstc( name, cval, file )
!-----------------------------------------------------------------------
! 	... Query history file module for character data
!           Return value:
!           0 => success
!          -1 => variable name not found
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)           :: file
      character(len=*), intent(in)  :: name      ! variable name of requested character string
      character(len=*), intent(out) :: cval      ! requested character string

      qhstc = 0

      select case( name )
         case( 'LPATH' )
            cval = hfile(file)%lpath
         case( 'RPATH' )
            cval = hfile(file)%rpath
         case( 'MONTHLY' )
            cval = hfile(file)%thtfrq
         case default
            qhstc = -1
      end select

      end function qhstc

      integer function qhsti( name, ival, file )
!-----------------------------------------------------------------------
! 	... Query history file module for integer data
!           Return value:
!           0 => success
!          -1 => variable name not found
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)          :: file
      integer, intent(out)         :: ival      ! requested value
      character(len=*), intent(in) :: name      ! variable name of requested integer data

      qhsti = 0

      select case( name )
         case( 'NTSHST' )                  ! present time point in hist file
            ival = hfile(file)%hdi(3)
         case( 'MXTSHST' )                 ! maximun number time points in hist file
            ival = hfile(file)%mfilt
         case( 'NHTFRQ' )                  ! output frequency in time steps for hist file
            ival = hfile(file)%nhtfrq
         case default
            qhsti = -1
      end select

      end function qhsti

      subroutine getspecres( plon, plat, ntrm, ntrn, ntrk )
!-----------------------------------------------------------------------
! 	... Spectral resolution based on horizontal resolution
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)  :: plon, plat
      integer, intent(out) :: ntrm, ntrn, ntrk

      if( plon == 64 .and. plat == 32 ) then  ! T21
         ntrm = 21
         ntrn = 21
         ntrk = 21
      else if( plon == 128 .and. plat == 64 ) then  ! T42
         ntrm = 42
         ntrn = 42
         ntrk = 42
      else if( plon == 192 .and. plat == 94 ) then ! T62
         ntrm = 62
         ntrn = 62
         ntrk = 62
      else if( plon == 512 .and. plat == 256 ) then ! T170
         ntrm = 170
         ntrn = 170
         ntrk = 170
      else
         ntrm = 0
         ntrn = 0
         ntrk = 0
         write(*,*) 'GETSPECRES: **WARNING** cannot determine spectral'
         write(*,*) '  resolution.  Setting (ntrm,ntrn,ntrk) = (0,0,0)'
      end if

      end subroutine getspecres

      end module MO_HISTOUT
