#include <params.h>

module mo_offline_constits
!---------------------------------------------------------------------
! 	... offline constituents module
!---------------------------------------------------------------------

  use mo_grid,    only : plon, plev, plevp, plat
  use mo_offline_data 

  implicit none

  private
  public :: init_offline_constits
  public :: advance_offline_constits
  public :: mxconstitflds, cnsts_offline, cnsts_names, cnst_data

  save

  type(offline_data) :: cnst_data

  integer :: mxconstitflds

  real, pointer :: cnsts_offline(:,:,:,:,:)           

  character(len=8), pointer :: cnsts_names(:)

contains

  subroutine init_offline_constits( icdate, icsec, offset, lun, mxnbrcnsts, &
                                    offl_cnsts, remove, constitnm, constitsc, &
                                    plonl, platl, pplon )
!---------------------------------------------------------------------------
! 	... initialize offline constituent arrays, and a few other variables.
!           the constituent fields will be linearly interpolated to the initial time (icdate,icsec).
!---------------------------------------------------------------------------

    use mo_control,    only : constit_flsp
    use mo_chem_utls,  only : get_inv_ndx

    implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in) :: plonl
    integer, intent(in) :: platl
    integer, intent(in) :: pplon
    integer, intent(in) :: mxnbrcnsts
    integer, intent(in) :: icdate              ! date of initial conditions in yymmdd format
    integer, intent(in) :: icsec               ! seconds relative to date for initial conditions
    integer, intent(in) :: offset              ! add offset (seconds) to dates read from headers
    integer, intent(in) :: lun                 ! unit number for constituents input
    real, intent(in)    :: constitsc(*)        ! scale factor to convert units in input files to the internal units

    character(len=32), intent(in)   :: constitnm(*)     ! names of constituent fields:
    character(len=8), intent(inout) :: offl_cnsts(*)    ! names of offline source invariants

    logical, intent(in) :: remove              ! true => remove old constituents files

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
    integer              :: astat
    integer              :: m
    integer              :: ndx
    character(len=8)     :: temp_string
    logical, allocatable :: is_invariant(:)

!---------------------------------------------------------------------------
! 	... check that offline constituents are simulation invariants
!---------------------------------------------------------------------------
    allocate( is_invariant(mxnbrcnsts),stat=astat )
    if( astat /= 0 ) then
       write(*,*) 'init_offline_constits: failed to allocate is_invariant; error = ',astat
       call endrun
    end if
    is_invariant(:) = .false.
    do m = 1,mxnbrcnsts
       if( len_trim( offl_cnsts(m) ) > 0 ) then
          ndx = get_inv_ndx( trim(offl_cnsts(m)) )
          is_invariant(m) = ndx > 0
       else
          exit
       end if
    end do

    ndx = 0
    do m = 1,mxnbrcnsts
       if( is_invariant(m) ) then
          ndx = ndx + 1
          if( ndx /= m ) then
             offl_cnsts(ndx) = offl_cnsts(m)
          end if
       end if
    end do

    if( ndx < mxnbrcnsts ) then
       offl_cnsts(ndx+1:mxnbrcnsts) = ' '
    end if

    deallocate( is_invariant )

    call init_offline_data( icdate, icsec, offset, lun, mxnbrcnsts, &
                            offl_cnsts, remove, constitnm, constitsc, cnst_data, &
                            mxconstitflds, cnsts_offline, cnsts_names ,constit_flsp, plonl, &
                            platl, pplon )

  end subroutine init_offline_constits

  subroutine advance_offline_constits( ncdate, ncsec, plonl, platl, pplon )
!---------------------------------------------------------------------------
! advance the offline constituent data -- interpolates, reads more data,
!                                         gets another file if needed 
!                                         time interpolates to ncdate,ncsec
!---------------------------------------------------------------------------

    use mo_control,    only : constit_flsp

    implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in) :: ncdate, ncsec, plonl, platl, pplon

!---------------------------------------------------------------------------
! if there are no offline offline data fields then proceed no farther
!---------------------------------------------------------------------------
    if ( cnst_data%mxflds == 0 ) then 
       return
    endif

!---------------------------------------------------------------------------
! Check that data needed for the specified time is loaded.
! Read the appropriate data, get another file if needed
!---------------------------------------------------------------------------
    call chk_offline_data( ncdate, ncsec, plonl, platl, pplon, cnst_data, constit_flsp )

!---------------------------------------------------------------------------
! time interpolate to ncdate,ncsec
!---------------------------------------------------------------------------
    call interpflds( ncdate, ncsec, plonl, platl, pplon, cnst_data, cnsts_offline )

  end subroutine advance_offline_constits

end module mo_offline_constits
