
module mo_offline_data

  use mo_grid,    only : plon, plev, plevp, plat

  implicit none

  type offline_field
     character(len=32) :: name 
     logical :: mandatory          
     integer :: vid                ! variable id number in netCDF file
     real    :: scfac              ! scale factor
  end type offline_field

  type offline_data

     integer :: mxflds 
     type(offline_field), pointer, dimension(:) :: flds
     logical :: rmfile
     logical :: close_file
     logical :: cosb               ! .true. => cos blocked, cray format
     integer :: lun                ! fortran unit number attached to file
     integer :: dataoffset         ! offset (seconds) added to (date,sec) info
     integer :: ntim               ! number of time samples in current file
     integer :: datevid(2)         ! ids for date and datesec
     logical :: averaged           ! true if time averaged data
     integer :: curr_ts            ! time index for most recently read data

     character(len=168) :: &
          lpath, &                   ! local path
          rpath, &                   ! remote pathname
          filename, &                ! offline data filename
          active_filename, &         ! active data filename
          last_filename              ! prior data filename

     integer :: &
          ncid                    ! netcdf file id

     integer, pointer, dimension(:) :: &
          date, &                    ! dates in input file
          datesec                    ! seconds relative to date
     logical :: initialization
     logical :: force_synch  

     integer :: &
          nlon, &                 ! number of longitudes
          nlat, &                 ! number of latitudes
          nlev, &                 ! number of model layers
          date1, &                ! date previous to current date (in yymmdd format)
          sec1, &                 ! seconds relative to date1
          n1, &                   ! index in data arrays for previous time sample
          date2, &                ! current date (in yymmdd format)
          sec2, &                 ! seconds relative to date2
          n2                      ! index in data arrays for current time sample

     integer :: &
          n_grid                  ! xform grid index

     integer :: lat_lims(2)

     real, pointer :: &
          fldsin(:,:,:,:,:,:)           

  end type offline_data

contains


  subroutine init_offline_data( icdate, icsec, offset, lun, mxnbrflds, &
                                offl_flds, remove, fldsnm, fldssc, data, &
                                nflds, flds_offline, flds_names, flsp, plonl, &
                                platl, pplon )
!---------------------------------------------------------------------------
! 	... initialize offline data arrays, and a few other variables.
!           the fields will be linearly interpolated to the initial time (icdate,icsec).
!---------------------------------------------------------------------------

    use netcdf
    use mo_constants,  only : d2r, r2d, pi, phi, lam
    use mo_regrider,   only : regrid_inti, regrid_lat_limits
    use mo_file_utils, only : open_netcdf_file
    use mo_mpi,        only : masternode, base_lat
    use mo_calendar,   only : addsec2dat, diffdat
    use m_types,       only : filespec
    use mo_charutl,    only : upcase

    implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in)  :: plonl
    integer, intent(in)  :: platl
    integer, intent(in)  :: pplon
    integer, intent(in)  :: icdate                ! date of initial conditions in yymmdd format
    integer, intent(in)  :: icsec                 ! seconds relative to date for initial conditions
    integer, intent(in)  :: offset                ! add offset (seconds) to dates read from headers
    integer, intent(in)  :: lun                   ! unit number for data input
    integer, intent(in)  :: mxnbrflds
    integer, intent(out) :: nflds
    real, intent(in)     :: fldssc(*)             ! scale factor to convert units in input files to the internal units
    real, pointer        :: flds_offline(:,:,:,:,:)


    character(len=8), intent(in)  :: offl_flds(*)    ! names of offline source invariants
    character(len=8), pointer     :: flds_names(:)
    character(len=32), intent(in) :: fldsnm(*)       ! names of data fields:

    logical, intent(in)  :: remove                   ! true => remove old data files

    type(offline_data), intent(out) :: data
    type(filespec), intent(inout)   :: flsp

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
    integer :: i, ii, j, k
    integer :: istat             ! return status
    integer :: astat             ! allocation status
    integer :: vid1   
    integer :: slen   
    integer :: wings             ! number of "wing" latitudes
    integer :: did(4)
    real    :: tmp
    real, allocatable :: data_lats(:)
    real, allocatable :: data_lons(:)
    real, allocatable :: wrk(:)
    character(len=32), allocatable :: tmpstr(:)

!---------------------------------------------------------------------------
! initialize basic offline data stuff
!---------------------------------------------------------------------------
    nflds = 0
    do i = 1,mxnbrflds
       if( len_trim(offl_flds(i)) > 0 ) nflds = nflds+1
    end do

    data%mxflds = nflds

!---------------------------------------------------------------------------
! if no fields then return
!---------------------------------------------------------------------------
    if( nflds == 0 ) then
       return
    end if


    allocate( flds_names( nflds ) )
    allocate( data%flds( data%mxflds ) )
    data%flds(:)%vid = 0

    data%initialization = .true.
    data%force_synch = .false.
    data%close_file = .true.

!---------------------------------------------------------------------------
!     	... set variables in module
!---------------------------------------------------------------------------
!---------------------------------------------------------------------------
!     	... if local path not specified, take filename from remote path
!           and put it in default subdirectory 
!---------------------------------------------------------------------------
    data%lpath     = flsp%local_path
    data%rpath     = flsp%remote_path
    data%filename  = flsp%nl_filename
    data%lun    = lun
    data%rmfile     = remove
    data%n1     = 1
    data%n2     = 2
    data%dataoffset = offset

    allocate( & 
         tmpstr(data%mxflds), &
         stat=astat )

    if( astat /= 0 ) then
       write(*,*) 'init_offline_data: failed to allocate ; error = ',astat
       call endrun
    end if

    do i = 1,data%mxflds

       flds_names(i) = offl_flds(i)

       if( fldsnm(i)(1:1) /= ' ' ) then
          data%flds(i)%name = fldsnm(i)
       end if

       write(*,*) 'init_offline_data: fld name : ',data%flds(i)%name

       call upcase( trim(fldsnm(i)), tmpstr(i) )
!!$       if (  index(  trim(tmpstr), 'AVRG' ) > 0 ) then
!!$          data%flds(i)%averaged=.true.
!!$       else
!!$          data%flds(i)%averaged=.false.
!!$       endif

       data%flds(i)%scfac = fldssc(i)
       
    end do

    data%averaged=.true.
    if ( any( index(  tmpstr, 'AVRG' ) > 0  ) ) then
      if ( all( index(  tmpstr, 'AVRG' ) > 0  ) ) then
         data%averaged=.true.
      else
         write(*,*) 'init_offline_data: if any of the data is averaged then all must be averaged.'
         call endrun
      endif
    endif

    write(*,*) 'init_offline_data: data%averaged : ',data%averaged

    deallocate( tmpstr )

!---------------------------------------------------------------------------
!     	... allocate module arrays
!---------------------------------------------------------------------------

    allocate( & 
!        qin        (plonl,plev,1:platl,pplon,2), &
         data%fldsin(plonl,plev,1:platl,pplon,2,data%mxflds), &
         flds_offline(plonl,plev,1:platl,pplon,data%mxflds), &
         stat=astat )

    if( astat /= 0 ) then
       write(*,*) 'init_offline_data: failed to allocate ; error = ',astat
       call endrun
    end if

    data%fldsin(:,:,:,:,:,:) = 0.

!---------------------------------------------------------------------------
!       ... open netcdf file
!---------------------------------------------------------------------------

    if( masternode ) then
       write(*,*) 'init_offline_data: opening filename=',trim(data%filename)
       write(*,*) 'init_offline_data: data%lpath=',trim(data%lpath)
       write(*,*) 'init_offline_data: data%rpath=',trim(data%rpath)
    end if

    if ( len_trim(data%filename) == 0 .or. data%mxflds == 0 ) then
       write(*,*) '**** No offline data specified. ****'
       data%mxflds = 0
       nflds = 0
       return
    endif

    data%ncid = open_netcdf_file( data%filename, data%lpath, data%rpath )
    if( masternode ) then
       write(*,*) 'init_offline_data: opened netcdf file ',trim(data%lpath) // trim(data%filename)
    end if
    data%active_filename = data%filename

!---------------------------------------------------------------------------
! 	... Set mandatory flag
!---------------------------------------------------------------------------
    do i = 1,data%mxflds
       data%flds(1)%mandatory = .true.
    enddo

!---------------------------------------------------------------------------
!  	... get variable ids for fields required to be in the input data
!---------------------------------------------------------------------------
    do i = 1,data%mxflds
       if( data%flds(i)%mandatory ) then
          call handle_ncerr( nf_inq_varid( data%ncid, data%flds(i)%name, data%flds(i)%vid ), &
               'init_offline_data: '// trim(data%flds(i)%name) // &
               ' not found in data input file' )
       else
          istat = nf_inq_varid( data%ncid, data%flds(i)%name, data%flds(i)%vid )
          if( istat == NF_NOERR ) then
             write(*,*) 'init_offline_data: Found optional variable '// trim(data%flds(i)%name) // &
                  ' in data input file'
          else
             data%flds(i)%vid = 0
          end if
       end if
    end do


!---------------------------------------------------------------------------
!  	... determine dimensions by looking at first data field
!---------------------------------------------------------------------------
    call handle_ncerr( nf_inq_vardimid( data%ncid, data%flds(1)%vid, did ), &
         'init_offline_data: getting dimids for 1st data field' )
    call handle_ncerr( nf_inq_dimlen( data%ncid, did(1), data%nlon ), &
         'init_offline_data: getting data%nlon' )
    call handle_ncerr( nf_inq_dimlen( data%ncid, did(2), data%nlat ), &
         'init_offline_data: getting data%nlat' )
    call handle_ncerr( nf_inq_dimlen( data%ncid, did(3), data%nlev ), &
         'init_offline_data: getting data%nlev' )
    call handle_ncerr( nf_inq_dimlen( data%ncid, did(4), data%ntim ), &
         'init_offline_data: getting data%ntim' )
    write(*,*) 'init_offline_data: data dimensions ',data%nlon,data%nlev,data%nlat,data%ntim

!---------------------------------------------------------------------------
!     	... get date info
!---------------------------------------------------------------------------
    call handle_ncerr( nf_inq_varid( data%ncid, 'date', data%datevid(1) ), &
         'init_offline_data: date not found in data input file' )
    call handle_ncerr( nf_inq_varid( data%ncid, 'datesec', data%datevid(2) ), &
         'init_offline_data: datesec not found in data input file' )
    allocate( data%date(data%ntim), stat=astat )
    if( astat /= 0 ) then
       write(*,*) 'init_offline_data: failed to allocate date vector'
       call endrun
    end if
    allocate( data%datesec(data%ntim), stat=astat )
    if( astat /= 0 ) then
       write(*,*) 'init_offline_data: failed to allocate datesec vector'
       call endrun
    end if
    call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(1), data%date ), 'init_offline_data: getting date' )
    call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(2), data%datesec ), 'init_offline_data: getting datesec' )
    if( masternode ) then
       write(*,*) 'init_offline_data: date'
       write(*,'(10i10)') data%date(:)
       write(*,*) 'init_offline_data: datesec'
       write(*,'(10i10)') data%datesec(:)
    end if

!---------------------------------------------------------------------------
!     	... adjust dates by the specified offset
!---------------------------------------------------------------------------
    do i = 1,data%ntim
       call addsec2dat( data%dataoffset, data%date(i), data%datesec(i) )
    end do
 
   if( masternode ) then
       write(*,*) 'init_offline_data: dataoffset'
       write(*,'(1i10)') data%dataoffset
       write(*,*) 'init_offline_data: date'
       write(*,'(10i10)') data%date(:)
       write(*,*) 'init_offline_data: datesec'
       write(*,'(10i10)') data%datesec(:)
    end if

!---------------------------------------------------------------------------
!     	... find latest date that is earlier or eqaul to (icdate,icsec)
!---------------------------------------------------------------------------

       data%curr_ts = lotim( data, icdate, icsec )

       if ( .not. data%averaged ) then
          if( data%curr_ts == 0 ) then
             write(*,*) 'init_offline_data: first time sample after ic: ', data%date(1), data%datesec(1)
             call endrun
          else
             write(*,*) 'init_offline_data: curr_ts = ',data%curr_ts
          end if
       endif

       write(*,*) 'data%curr_ts = ',data%curr_ts

!---------------------------------------------------------------------------
!     	... allocate data lats, lons and read
!---------------------------------------------------------------------------
    allocate( data_lats(data%nlat), stat=astat )
    if( astat /= 0 ) then
       write(*,*) 'init_offline_data: failed to allocate data_lats vector'
       call endrun
    end if
    allocate( data_lons(data%nlon), stat=astat )
    if( astat /= 0 ) then
       write(*,*) 'init_offline_data: failed to allocate data_lons vector'
       call endrun
    end if
    call handle_ncerr( nf_inq_varid( data%ncid, 'lat', vid1 ), 'init_offline_data: lat dim not in data file' )
    call handle_ncerr( nf_get_var_double( data%ncid, vid1, data_lats ), 'init_offline_data: getting data lats' )
    call handle_ncerr( nf_inq_varid( data%ncid, 'lon', vid1 ), 'init_offline_data: lon dim not in data file' )
    call handle_ncerr( nf_get_var_double( data%ncid, vid1, data_lons ), 'init_offline_data: getting data lons' )

!---------------------------------------------------------------------------
!     	... set the transform from inputs lats to simulation lats
!---------------------------------------------------------------------------
    write(*,*) 'init_offline_data: source lats (deg)'
    write(*,'(10f8.2)') data_lats
    write(*,*) 'init_offline_data: target lats (deg)'
    write(*,'(10f8.2)') phi*r2d
    write(*,*) 'init_offline_data: source lons (deg)'
    write(*,'(10f8.2)') data_lons
    write(*,*) 'init_offline_data: target lons (deg)'
    write(*,'(10f8.2)') lam*r2d
    data_lats(:data%nlat) = d2r * data_lats(:data%nlat)
    data_lons(:data%nlon) = d2r * data_lons(:data%nlon)

    !! ??? fvitt ???
!!$#ifdef USE_MPI
!!$    wings = 4
!!$#else
    wings = 0
!!$#endif


    data%n_grid =  regrid_inti( data%nlat, plat, data%nlon, plon, data_lons, lam, data_lats, phi, wings, platl )

    if( data%n_grid < 0 ) then
       write(*,*) 'init_offline_data: regrid_inti error = ',data%n_grid
       call endrun
    elseif( data%n_grid == 0 ) then
       write(*,*) 'init_offline_data: no regridding needed, regrid_inti using grid number ',data%n_grid
    else
       write(*,*) 'init_offline_data: regrid_inti using grid number ',data%n_grid
    end if

    if( data%n_grid > 0 ) then
       data%lat_lims = regrid_lat_limits( data%n_grid )
    else
       data%lat_lims = (/ base_lat + 1, base_lat + platl /)
    end if
    write(*,*) 'init_offline_data: data%lat_lims = ', data%lat_lims

!---------------------------------------------------------------------------
!     	... read data into time level 1
!---------------------------------------------------------------------------
    data%n1    = 1

    do i = 1,data%mxflds
      if (data%curr_ts > 0) then
         data%date1 = data%date(data%curr_ts)
         data%sec1  = data%datesec(data%curr_ts)

         call readdata( data%curr_ts, data%n1, plonl, platl, pplon, data )
      endif
    enddo

!---------------------------------------------------------------------------
!     	... if next time sample is not in current file then get next file
!---------------------------------------------------------------------------
    if( diffdat( data%date(data%ntim), data%datesec(data%ntim), icdate, icsec )  >= 0. ) then
       data%force_synch   = .true.
       data%last_filename = data%filename
       data%close_file    = .true.
       if( masternode ) then
          write(*,*) 'init_offline_data: calling nextfile with filename = ',trim(data%filename),', force_synch=t'
       end if
       call nextfile(data)
       flsp%nl_filename = data%active_filename
       data%force_synch = .false.
       if( masternode ) then
          write(*,*) 'init_offline_data: back from nextcontit with filename,active_filename = ', &
               trim(data%filename),', ',trim(data%active_filename)
       end if
    else
       data%rmfile      = .false.
       data%close_file = .false.
       if( masternode ) then
          write(*,*) 'init_offline_data: calling nextfile with filename = ',trim(data%filename),', rmfile=f, close_file=f'
       end if
       call nextfile(data)
       if( masternode ) then
          write(*,*) 'init_offline_data: back from nextfile with filename, active_filename = ', &
               trim(data%filename), ', ', trim(data%active_filename)
       end if
       data%close_file = .true.
       data%rmfile      = remove
    end if

! advance time index by one
    data%curr_ts = data%curr_ts + 1
      
    write(*,*) 'init_offline_data: data%curr_ts = ',data%curr_ts

!!$    do i = 1,data%mxflds
!!$       if ( data%flds(i)%averaged ) then
!!$          if ( data%curr_ts == 0 ) then
!!$             data%curr_ts = 1
!!$          else
!!$             data%curr_ts = data%curr_ts + 1
!!$          endif
!!$       else
!!$          data%curr_ts = data%curr_ts + 1
!!$       endif
!!$    enddo

!---------------------------------------------------------------------------
!     	... check that current time sample is an upper bound to the ic time
!---------------------------------------------------------------------------
    if( diffdat( icdate, icsec, data%date(data%curr_ts), data%datesec(data%curr_ts))  < 0. ) then
       write(*,*) 'init_offline_data: ic time not bounded by specified data ' // &
            'input file.  check namelist settings.'
       write(*,*) 'icdate, icsec = ',icdate,icsec
       write(*,*) 'data%date(data%curr_ts), data%datesec(data%curr_ts) = ',data%date(data%curr_ts), data%datesec(data%curr_ts)
       write(*,*) 'data%curr_ts = ',data%curr_ts

       call endrun
    end if

!---------------------------------------------------------------------------
!     	... read data into time level 2
!---------------------------------------------------------------------------
    data%n2    = 2
    data%date2 = data%date(data%curr_ts)
    data%sec2  = data%datesec(data%curr_ts)

    call readdata( data%curr_ts, data%n2, plonl, platl, pplon, data )
 
    data%initialization = .false.

    deallocate( data_lats, data_lons )

  end subroutine init_offline_data

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX


  subroutine readdata( itim, ifield, plonl, platl, pplon, data, lowlimit )

!---------------------------------------------------------------------------
! 	... read the field data for the specified time sample
!---------------------------------------------------------------------------

    use netcdf
    use mo_regrider, only : regrid_3d
    use mo_mpi,      only : masternode, base_lat

    implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
    integer, intent(in) :: &
         itim, &              ! time index for requested record from netcdf file
         ifield, &            ! index in data input array
         plonl, &
         platl, &
         pplon
    
    type(offline_data), intent(inout) :: data

    logical, optional, intent(in) :: lowlimit

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
    integer :: &
         i, j, jl, ju, jll, jul,  m, istat, lats

!!$    integer, dimension(3) :: &
!!$         count2, &
!!$         start2
    integer, dimension(4) :: &
         count, &
         start

    real :: &
         tmp3d(data%nlon,data%lat_lims(1):data%lat_lims(2),plev), &  ! temp 3d field at midpoints or interfaces
!         wrk3d(plon,plev,-3:platl+4)                  ! work 3d field
         wrk3d(plon,plev,platl)                  ! work 3d field

    logical :: read 

!    tmp3d(:,:,:) = 0.0

!!$    logical, save :: entered = .false.
!!$
!!$    if( .not. entered ) then
!!$       count2 = (/ data%nlon, data%nlat, 1 /)
    if( plev < data%nlev ) then
       count = (/ data%nlon, data%nlat, plev, 1 /)
    else
       count = (/ data%nlon, data%nlat, data%nlev, 1 /)
    end if
!!$       count2(2) = data%lat_lims(2) - data%lat_lims(1) + 1
!!$       count(2)  = count2(2)
    count(2)  = data%lat_lims(2) - data%lat_lims(1) + 1
!!$       entered = .true.
!!$    end if

!!$    start2 = (/ 1, data%lat_lims(1), itim /)

    if( plev < data%nlev ) then
       start = (/ 1, data%lat_lims(1), data%nlev-plev+1, itim /)
    else
       start = (/ 1, data%lat_lims(1), 1, itim /)
    end if

#ifdef DEBUG
    write(*,*) 'readdata: diags'
    write(*,*) 'wrk3d dimensions = ',size(wrk3d,dim=1),size(wrk3d,dim=2),size(wrk3d,dim=3)
    write(*,*) 'tmp3d dimensions = ',size(tmp3d,dim=1),size(tmp3d,dim=2),size(tmp3d,dim=3)
!!$    write(*,*) 'start2 = ',start2
!!$    write(*,*) 'count2 = ',count2
    write(*,*) 'start  = ',start
    write(*,*) 'count  = ',count
#endif

    jl = max( 1,base_lat + 1 )
    ju = min( plat,base_lat + platl )
    lats = ju - jl + 1
    
#ifdef DEBUG
    write(*,*) 'jl,ju = ',jl,ju
#endif


!-----------------------------------------------------------------------
!  	 read in each offline data and regrid it
!-----------------------------------------------------------------------

    read = .true.
    if (present(lowlimit)) then
       if( (lowlimit) .and. (data%averaged) ) read = .false.
    endif

    if (read) then 

       do m=1,data%mxflds
          call handle_ncerr( nf_get_vara_double( data%ncid, data%flds(m)%vid, start, count, tmp3d ), &
                             'readdata: getting '// trim( data%flds(m)%name ) )
          call regrid_3d( tmp3d, wrk3d(:,:,1:lats), data%n_grid, jl, ju, .true., data%flds(m)%scfac )
          data%fldsin(:,:,:,:,ifield,m) = reshape( wrk3d, (/plonl,plev,platl,pplon/), order=(/1,4,2,3/) )
       enddo
    endif


    if( masternode ) then
       if (read) write(*,*) 'readdata offline: read time sample for date = ', &
                            data%date(itim),' seconds = ', data%datesec(itim)
    end if

  end subroutine readdata

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

  subroutine chk_offline_data( ncdate, ncsec, plonl, platl, pplon, data, flsp )
!-----------------------------------------------------------------------
! 	... check that date to interpolate the data to is contained
!           in the interval of offline data currently in memory.
!           if not, then read the next time sample from the data file,
!           acquiring the next file if necessary.
!-----------------------------------------------------------------------

    use mo_calendar, only : diffdat
    use mo_mpi,      only : masternode
    use m_types, only : filespec

    implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
    integer, intent(in) :: &
         ncdate, &                   ! interpolate the input data to (ncdate,ncsec)
         ncsec, &
         plonl, &
         platl, &
         pplon

    type(offline_data), intent(inout) :: data
    type(filespec), intent(inout) :: flsp

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
    integer :: &
         istat, lo, hi

    if ( data%averaged ) then
      if( diffdat( ncdate, ncsec, data%date2, data%sec2 ) > 0. ) then
         return
      end if
    else
      if( diffdat( ncdate, ncsec, data%date2, data%sec2 ) >= 0. ) then
         return
      end if
    endif
!-----------------------------------------------------------------------
!     (ncdate,ncsec) is outside of current data time interval.
!     get the time sample(s) for the new time interval.
!
!     make the upper bound of the current time interval the lower bound of the
!     new time interval.  in the "usual" case we won''t need to read new
!     data into this slot.
!-----------------------------------------------------------------------
    data%n1 = data%n2

!-----------------------------------------------------------------------
!     	... is the new upper bound time sample in the current data file?
!-----------------------------------------------------------------------
    lo = lotim( data, ncdate, ncsec )
    hi = lo + 1

    if( hi <= data%ntim ) then   ! all required data is present in current file
!-----------------------------------------------------------------------
!        is the current upper bound time sample the same as the lower bound
!        time sample for the new time interval?  if not we need to acquire
!        new lower bound data.
!-----------------------------------------------------------------------
       if( data%curr_ts == lo ) then
          data%date1 = data%date2
          data%sec1  = data%sec2
       else
          data%date1 = data%date(lo)
          data%sec1  = data%datesec(lo)
!!$          call readdata( lo, data%n1, plonl, &
!!$               platl, pplon, data, lowlimit=.true. )
          call readdata( lo, data%n1, plonl, platl, pplon, data )
       end if
    else
!-----------------------------------------------------------------------
!  	... need next file.  if last time sample in current file is not the
!        current upper bound then read it into the lower bound position
!        before getting next file.
!-----------------------------------------------------------------------
       if( data%curr_ts == data%ntim ) then
          data%date1 = data%date2
          data%sec1  = data%sec2
       else
          data%date1 = data%date(data%ntim)
          data%sec1  = data%datesec(data%ntim)
!!$          call readdata( data%ntim, data%n1, plonl, &
!!$               platl, pplon, data, lowlimit=.true. )
          call readdata( data%ntim, data%n1, plonl, platl, pplon, data )
       end if

       data%last_filename        = data%active_filename
       data%active_filename      = data%filename
       flsp%nl_filename = data%filename
       if( masternode ) then
          write(*,*) 'chk_offline_data: calling nextfile with filename=',trim(data%filename)
       end if
       call nextfile(data)
       if( masternode ) then
          write(*,*) 'chk_offline_data: back from nextfile with filename=',trim(data%filename)
       end if
       lo = lotim( data, ncdate, ncsec )
       hi = lo + 1
!-----------------------------------------------------------------------
!   	... make sure the new upper bound time sample is in this file.
!-----------------------------------------------------------------------
       if( hi > data%ntim ) then
          write(*,*) 'chk_offline_data: current time not bounded by data on next' // &
               ' offline data input file.'
          call endrun
       end if
!-----------------------------------------------------------------------
!        is the current lower bound time sample (i.e., the last time sample
!        from the previous file) the correct one?  if not we need to acquire
!        new lower bound data.
!-----------------------------------------------------------------------
       if( lo > 0 ) then
          data%date1 = data%date(lo)
          data%sec1  = data%datesec(lo)
!!$          call readdata( lo, data%n1, plonl, &
!!$               platl, pplon, data, lowlimit=.true. )
          call readdata( lo, data%n1, plonl, platl, pplon, data, lowlimit=.true. )
       end if
    end if

!-----------------------------------------------------------------------
!     	... read data for new upper bound
!-----------------------------------------------------------------------
    data%n2    = mod( data%n1,2 ) + 1
    data%date2 = data%date(hi)
    data%sec2  = data%datesec(hi)
    call readdata( hi, data%n2, plonl, &
         platl, pplon, data )
    data%curr_ts = hi

  end subroutine chk_offline_data

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

  subroutine interpflds( ncdate, ncsec, plonl, platl, pplon, data, flds_offline )
!---------------------------------------------------------------------------
! 	... interpolate the fields whose values are required at the
!           begining of a timestep.
!---------------------------------------------------------------------------

    use mo_mpi,      only : masternode, lastnode
    use mo_calendar, only : diffdat

    implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in) :: &
         ncdate, &                      ! interpolate the input data to (ncdate,ncsec)
         ncsec, &
         plonl, &
         platl, &
         pplon

    type(offline_data), intent(in) :: data
    real, dimension(:,:,:,:,:), pointer :: flds_offline

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
    integer :: &
         ifld, &
         j, &  ! index
         ip, & ! index
         jl, & ! index
         ju    ! index
    real :: &
         dataday2, & ! (date2,sec2) - (date1,sec1) in days
         intpday    ! (ncdate,ncsec) - (date1,sec1) in days

!!$    do ifld=1,data%mxflds
       if ( .not. data%averaged ) then
          dataday2 = diffdat( data%date1, data%sec1, data%date2, data%sec2 )

          intpday = diffdat( data%date1, data%sec1, ncdate, ncsec )
       endif
!!$    enddo

#ifdef DEBUG
    write(*,*) 'interpcons: timing info'
    write(*,*) 'dataday2 = ',dataday2
    write(*,*) 'intpday = ',intpday
#endif

#ifdef USE_MPI
    jl = -3
    ju = platl + 4
    if( masternode ) then
       jl = 1
    end if
    if( lastnode ) then
       ju = platl
    end if
#else
    jl = 1
    ju = platl
#endif

    if( pplon > 1 ) then
       !$omp parallel do private( ip, j )
       do ip = 1,pplon
          do j = 1,platl
             do ifld=1,data%mxflds
                if ( data%averaged ) then
                   flds_offline(:,:,j,ip,ifld) = data%fldsin(:,:,j,ip,data%n2,ifld)
                else
                   call intp2d( plev, 0., dataday2, intpday, &
                        data%fldsin(:,:,j,ip,data%n1,ifld), data%fldsin(:,:,j,ip,data%n2,ifld), flds_offline(:,:,j,ip,ifld), plonl )
                endif
             enddo
          end do
       end do
       !$omp end parallel do
    else
       !$omp parallel private( j )
       !$omp do
       do j = 1,platl
          do ifld=1,data%mxflds
             if ( data%averaged ) then
                flds_offline(:,:,j,1,ifld) = data%fldsin(:,:,j,1,data%n2,ifld)
             else
                call intp2d( plev, 0., dataday2, intpday, &
                     data%fldsin(:,:,j,1,data%n1,ifld), data%fldsin(:,:,j,1,data%n2,ifld), flds_offline(:,:,j,1,ifld), plonl )
             endif
          enddo
       end do
       !$omp end do
       !$omp end parallel
    end if

  end subroutine interpflds


  subroutine nextfile( data )
!-----------------------------------------------------------------------
! 	... close [and remove] the current offline file.  open the next
!           offline file.  acquire file from mass store if local version
!           doesn''t exist.
!-----------------------------------------------------------------------

    use netcdf
    use mo_file_utils, only : acquire_file, open_netcdf_file
    use mo_calendar,   only : addsec2dat
    use mo_charutl,    only : incstr
#ifdef USE_MPI
    use mo_mpi,        only : masternode, mpi_comm_comp, mpi_success
#else
    use mo_mpi,        only : masternode
#endif

    implicit none

    type(offline_data), intent(inout) :: data

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
    integer :: &
         i, istat, slen, attempts, &
         recdid, nrec
    integer, allocatable, dimension(:) :: &
         next_date, &                    ! dates in input file
         next_datesec                    ! seconds relative to date
    character(len=128) :: &
         ctmp
    character(len=168) :: &
         loc_file
    character(len=10) :: &
         ctime
    character(len=8) :: &
         cdate

    logical :: msread_done, there

!-----------------------------------------------------------------------
! 	... function declarations
!-----------------------------------------------------------------------
    integer :: fsystem

!-----------------------------------------------------------------------
!     	... close current file
!-----------------------------------------------------------------------


    if( data%close_file ) then
       call handle_ncerr( nf_close( data%ncid ), 'nextfile: error closing file ' // &
            trim(data%lpath) // trim(data%filename))
    end if

Master_node_only : &
    if( masternode ) then
!-----------------------------------------------------------------------
!     	... remove if requested
!-----------------------------------------------------------------------
       if( data%rmfile ) then
          loc_file = data%last_filename
          write(*,*) 'nextfile: removing file = ',trim(data%lpath) // trim(loc_file)
          ctmp = 'rm -f ' // trim(data%lpath) // trim(loc_file)
          write(*,*) 'nextfile: fsystem issuing command - '
          write(*,*) trim(ctmp)
          istat = fsystem( trim(ctmp) )
       end if
#ifdef DEBUG
       write(*,*) 'nextfile: checking for next file, initialization=',data%initialization
#endif
       if( .not. data%initialization ) then
#ifdef NCAR
          msread_done = .false.
          do attempts = 1,10
!-----------------------------------------------------------------------
!       ... check to see if file acquired by previous asynch msread is on local disk
!-----------------------------------------------------------------------
             inquire( file = trim( data%lpath ) // trim( data%filename ), exist = there, iostat = istat )
#ifdef DEBUG
             write(*,*) 'nextfile: checking for next file, istat,there,attempts=',istat,there,attempts
#endif
             if( istat /= 0 ) then
                write(*,*) 'nextfile: inquire failed for file ',trim( data%lpath ) // trim( data%filename )
                write(*,*) 'error code = ',istat
                call endrun
             end if
             if( there ) then
                msread_done = .true.
                exit
             else
#ifdef DEBUG
                write(*,*) 'nextfile: waiting for file, starting sleep #',attempts
#endif
                call sleep( 30 )
             end if
          end do
#ifdef DEBUG
          write(*,*) 'nextfile: msread_done=',msread_done
#endif
          if( .not. msread_done ) then
             call date_and_time( cdate, ctime )
             write(*,*) 'nextfile: failed to acquire file ',trim( data%rpath ) // trim(data%filename),' from mass store'
             write(*,*) '         at time ',cdate,' ',ctime
             call endrun
          end if
#else
          inquire( file = trim( data%lpath ) // trim( data%filename ), exist = there, iostat = istat )
          if( istat /= 0 ) then
             write(*,*) 'nextfile: inquire failed for file ',trim( data%lpath ) // trim( data%filename )
             write(*,*) 'error code = ',istat
             call endrun
          end if
          if( .not. there ) then
             call date_and_time( cdate, ctime )
             write(*,*) 'nextfile: file ',trim( data%rpath ) // trim(data%filename),' not found'
             write(*,*) '         at time ',cdate,' ',ctime
             call endrun
          end if
#endif
       end if ! (.not. initialization)
    end if Master_node_only

!---------------------------------------------------------------------------
!     	... all compute nodes wait for master to check last msread
!---------------------------------------------------------------------------
#ifdef USE_MPI
    call mpi_barrier( mpi_comm_comp, istat )
    if( istat /= mpi_success ) then
       write(*,*) 'nextfile: barrier failed; code = ',istat
       call endrun
    end if
#endif
    if( .not. data%initialization ) then
       call handle_ncerr( nf_open( trim(data%lpath) // data%filename, nf_nowrite, data%ncid ), &
            'nextfile: error opening file ' // trim(data%lpath) // trim(data%filename) )
!-----------------------------------------------------------------------
!     	... get time info
!-----------------------------------------------------------------------
       call handle_ncerr( nf_inq_unlimdim( data%ncid, recdid ), &
            'nextfile: inquiring record dimension id' )
       call handle_ncerr( nf_inq_dimlen( data%ncid, recdid, nrec ), &
            'nextfile: inquiring record dimension length' )
       allocate( next_date(nrec), next_datesec(nrec), stat=istat )
       if( istat /= 0 ) then
          write(*,*) 'nextfile: failed to allocate next_date ... next_datsec; error = ',istat
          call endrun
       end if
       call handle_ncerr( nf_inq_varid( data%ncid, 'date', data%datevid(1) ), &
            'nextfile: date not found in input file' )
       call handle_ncerr( nf_inq_varid( data%ncid, 'datesec', data%datevid(2) ), &
            'nextfile: datesec not found in input file' )
       call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(1), next_date ), &
            'nextfile: getting date' )
       call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(2), next_datesec ), &
            'nextfile: getting datesec' )
       data%curr_ts = 0
    end if

!-----------------------------------------------------------------------
!     	... build filenames for next offline file.  netcdf files must
!           have a .nc extension.
!-----------------------------------------------------------------------
    if( masternode ) then
       write(*,*) 'nextfile: attempting to increment filename ',trim( data%filename )
    end if
    if( .not. data%initialization ) then
       call incr_flnm( data%filename, next_date(nrec), next_datesec(nrec), next_datesec(2) - next_datesec(1) )
    else
       call incr_flnm( data%filename, data%date(data%ntim), data%datesec(data%ntim), data%datesec(2) - data%datesec(1) )
    end if
    if( masternode ) then
       write(*,*) 'nextfile: next filename = ',trim( data%filename )
    end if
    if( .not. data%initialization ) then
       deallocate( next_date, next_datesec )
    end if

    if( data%force_synch ) then
       data%ncid = open_netcdf_file( data%filename, data%lpath, data%rpath )
       data%active_filename = data%filename
       data%initialization  = .false.
       data%curr_ts = 0
!-----------------------------------------------------------------------
!     	... get time info
!-----------------------------------------------------------------------
       call handle_ncerr( nf_inq_unlimdim( data%ncid, recdid ), &
            'nextfile: inquiring record dimension id' )
       call handle_ncerr( nf_inq_dimlen( data%ncid, recdid, nrec ), &
            'nextfile: inquiring record dimension length' )
       allocate( next_date(nrec), next_datesec(nrec), stat=istat )
       if( istat /= 0 ) then
          write(*,*) 'nextfile: failed to allocate next_date ... next_datsec; error = ',istat
          call endrun
       end if
       call handle_ncerr( nf_inq_varid( data%ncid, 'date', data%datevid(1) ), &
            'nextfile: date not found in input file' )
       call handle_ncerr( nf_inq_varid( data%ncid, 'datesec', data%datevid(2) ), &
            'nextfile: datesec not found in input file' )
       call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(1), next_date ), &
            'nextfile: getting date' )
       call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(2), next_datesec ), &
            'nextfile: getting datesec' )
!-----------------------------------------------------------------------
!     	... build filenames for next file.  netcdf files must
!           have a .nc extension.
!-----------------------------------------------------------------------
       call incr_flnm( data%filename, next_date(nrec), next_datesec(nrec), next_datesec(2) - next_datesec(1) )
       deallocate( next_date, next_datesec )
    end if

    if( masternode ) then
!-----------------------------------------------------------------------
! 	... asynch acquire next file
!-----------------------------------------------------------------------
       istat = acquire_file( trim(data%lpath) // data%filename, trim(data%rpath) // data%filename, &
            data%lun, .false., data%cosb, .true. )
       if( istat /= 0 ) then
          write(*,*) 'nextfile: acquire_file: returned ', istat
          call endrun
       end if
    end if

    if( .not. data%initialization ) then
!---------------------------------------------------------------------------
!  	... get variable ids for fields required to be in the input data
!---------------------------------------------------------------------------
       do i = 1,data%mxflds
          if( data%flds(i)%mandatory ) then
             call handle_ncerr( nf_inq_varid( data%ncid, data%flds(i)%name, data%flds(i)%vid ), &
                  'nextfile: '// trim(data%flds(i)%name) // &
                  ' not found in offline input file' )
          else
             istat = nf_inq_varid( data%ncid, data%flds(i)%name, data%flds(i)%vid  )
             if( istat == NF_NOERR ) then
                write(*,*) 'nextfile: Found optional variable '// data%flds(i)%name // &
                     ' in offline input file'
             else
                data%flds(i)%vid = 0
             end if
          end if
       end do
!-----------------------------------------------------------------------
!     	... update date info
!-----------------------------------------------------------------------
       call handle_ncerr( nf_inq_unlimdim( data%ncid, recdid ), &
            'nextfile: inquiring record dimension id' )
       call handle_ncerr( nf_inq_dimlen( data%ncid, recdid, nrec ), &
            'nextfile: inquiring record dimension length' )
       if( nrec /= data%ntim ) then
          deallocate( data%date, data%datesec )
          data%ntim = nrec
          allocate( data%date(data%ntim), data%datesec(data%ntim) )
       end if
       call handle_ncerr( nf_inq_varid( data%ncid, 'date', data%datevid(1) ), &
            'nextfile: date not found in input file' )
       call handle_ncerr( nf_inq_varid( data%ncid, 'datesec', data%datevid(2) ), &
            'nextfile: datesec not found in input file' )
       call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(1), data%date ), &
            'nextfile: getting date' )
       call handle_ncerr( nf_get_var_int( data%ncid, data%datevid(2), data%datesec ), &
            'nextfile: getting datesec' )

!-----------------------------------------------------------------------
!     	... adjust dates by the specified offset
!-----------------------------------------------------------------------
       do i = 1,data%ntim
          call addsec2dat( data%dataoffset, data%date(i), data%datesec(i) )
       end do
    end if

  end subroutine nextfile

  subroutine incr_flnm( filename, pdate, psec, delt )
!-----------------------------------------------------------------------
! 	... Increment or decrement a date string withing a filename
!           the filename date section is assumed to be of the form
!           yyyy-dd-mm
!-----------------------------------------------------------------------

    use mo_calendar, only : newdate, caldayr
    use mo_charutl,  only : incstr

    implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
    integer, optional, intent(in)   :: pdate             ! present date (yyyymmdd)
    integer, optional, intent(in)   :: psec              ! present seconds withing date
    integer, optional, intent(in)   :: delt              ! offline dataset time sample increment
    character(len=*), intent(inout) :: filename          ! present dataset filename

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
    integer :: beg_date, end_date, pos, pos1
    integer :: year, month, day, date, secs, istat
    real    :: calday
    character(len=128) :: fn_new
    character(len=6)   :: seconds
    character(len=5)   :: num

    if( filename(1:5) == 'ecmwf' ) then
!-----------------------------------------------------------------------
!	... ecmwf type filename
!-----------------------------------------------------------------------
       if( .not. present( pdate ) .or. .not. present( psec ) .or. .not. present( delt ) ) then
          write(*,*) 'incr_flnm: date, datesec, and delt arguments must be present'
          call endrun
       end if
       fn_new(1:5) = filename(1:5)
       secs = psec + delt
       if( secs >= 86400 ) then
          day  = secs/86400
          secs = mod( secs,86400 )
          date = newdate( pdate, day )
       else
          date = pdate
       end if
       year  = date/10000
       month = mod( date,10000 )/100
       day   = date - (10000*year + 100*month)
       year  = 10000 + year
       write(num,'(i5)') year
       pos = 6
       fn_new(pos:pos+4) = num(2:5)

       month = month + 100
       write(num(1:3),'(i3)') month
       pos = pos + 4
       fn_new(pos:pos+2) = num(2:3)

       day = day + 100
       write(num(1:3),'(i3)') day
       pos = pos + 2
       fn_new(pos:) = num(2:3) // '.nc'
    else
       pos = index( filename, '-', back=.true. )
       if( pos > 0 ) then
!-----------------------------------------------------------------------
!	... cam2 type filename
!-----------------------------------------------------------------------
          if( .not. present( pdate ) .or. .not. present( psec ) .or. .not. present( delt ) ) then
             write(*,*) 'incr_flnm: date, datesec, and delt arguments must be present'
             call endrun
          end if
          pos = index( filename(:pos-1), '-', back=.true. )
          pos = index( filename(:pos-1), '-', back=.true. )
          pos = index( filename(:pos-1), '.', back=.true. )
          fn_new(1:pos) = filename(1:pos)

          secs = psec + delt
          if( secs >= 86400 ) then
             day  = secs/86400
             secs = mod( secs,86400 )
             date = newdate( pdate, day )
          else
             date = pdate
          end if
          year  = date/10000
          month = mod( date,10000 )/100
          day   = date - (10000*year + 100*month)
          year  = 10000 + year
          pos   = pos + 1
          write(num,'(i5)') year
          fn_new(pos:pos+4) = num(2:5) // '-'

          month = month + 100
          write(num(1:3),'(i3)') month
          pos = pos + 5
          fn_new(pos:pos+2) = num(2:3) // '-'

          write(num(1:3),'(i3)') day
          write(seconds,'(i6)') secs + 100000
          pos = pos + 3
          fn_new(pos:) = num(2:3) // '-' // seconds(2:6) // '.nc'
       else
!-----------------------------------------------------------------------
!	... ccm type filename
!-----------------------------------------------------------------------
          pos = len_trim( filename )
          fn_new = filename(:pos)
          write(*,*) 'filename = ',trim(fn_new)
          if( fn_new(pos-2:) == '.nc' ) then
             pos = pos - 3
          end if
          istat = incstr( fn_new(:pos), 1 )
          if( istat /= 0 ) then
             write(*,*) 'incr_flnm: incstr returned ', istat
             write(*,*) '           while trying to decrement ',trim( fn_new )
             call endrun
          end if
       end if
    end if

    write(*,*) 'incr_flnm: new filename = ',trim(fn_new)
    filename = trim(fn_new)

  end subroutine incr_flnm

!!$  subroutine qdata( data, ncdate, ncsec, lpth, rpth, tim_idx )
  subroutine qdata( data, ncdate, ncsec, lpth, rpth )
!-----------------------------------------------------------------------
! 	... query the offline data.
!-----------------------------------------------------------------------

    use mo_calendar,   only : diffdat

    implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
    type(offline_data), intent(in) :: data
    integer, intent(in) :: &
         ncdate, &  ! current simulation date (yyyymmdd)
         ncsec      ! current simulation seconds withing current date (s)
    character(len=*), intent(out) :: &
         lpth, &    ! local path for offline data
         rpth       ! remote path for offline data

    integer :: i
!!$    integer, intent(out) :: &
!!$         tim_idx    ! time index for most recently read data

!!$    do i=1,data%mxflds
!!$       if ( data%flds(i)%averaged ) then
!!$          lpth = trim(data%lpath) // data%active_filename
!!$          rpth = trim(data%rpath) // data%active_filename
!!$          return
!!$       endif
!!$    enddo

    if ( data%mxflds == 0 ) then
       lpth=""
       rpth=""
       return
    endif

    if( diffdat( ncdate, ncsec, data%date(1), data%datesec(1) ) <= 0. ) then
       lpth = trim(data%lpath) // data%active_filename
       rpth = trim(data%rpath) // data%active_filename
    else
!!$       do i=1,data%mxflds
          if ( .not. data%averaged ) then
             lpth = trim(data%lpath) // data%last_filename
             rpth = trim(data%rpath) // data%last_filename
          else
             lpth = trim(data%lpath) // data%active_filename
             rpth = trim(data%rpath) // data%active_filename
          endif
!!$       enddo
    end if


!!$    tim_idx = data%curr_ts

  end subroutine qdata

  subroutine intp2d( nlev, t1, t2, tint, f1, f2, fint, plonl )
!-----------------------------------------------------------------------
! 	... linearly interpolate between f1(t1) and f2(t2) to fint(tint).
!-----------------------------------------------------------------------

    implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
    integer, intent(in) :: &
         nlev, &
         plonl

    real, intent(in) :: &
         t1, &            ! time level of f1
         t2, &            ! time level of f2
         tint             ! interpolant time

    real, dimension(plonl,nlev), intent(in) :: &
         f1, &            ! field at time t1
         f2               ! field at time t2

    real, intent(out) :: &
         fint(plonl,nlev)  ! field at time tint

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
    integer :: k
    real    :: factor

    factor = (tint - t1)/(t2 - t1)

    do k = 1,nlev
       fint(:,k) = f1(:,k) + (f2(:,k) - f1(:,k))*factor
    end do

  end subroutine intp2d

  integer function lotim( data, cdate, csec )
!-----------------------------------------------------------------------
! 	... return the index of the offline data time sample that is the lower
!           bound of the interval that contains the input date.  if
!           (cdate,csec) is earlier than the first time sample then 0 is
!           returned.  if (cdate,csec) is later than the last time sample then
!           that index is returned.  if (cdate,csec) is equal to the date of a
!           offline data time sample then that index is returned.
!-----------------------------------------------------------------------

    use mo_calendar,  only : diffdat

    implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------

    type(offline_data), intent(in) :: data

    integer, intent(in) :: &
         cdate, & ! date in yyyymmdd
         csec     ! seconds relative to date

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
    integer :: i

!-----------------------------------------------------------------------
!     	... find latest date that is earlier than or equal to (date,sec)
!-----------------------------------------------------------------------
    do i = 1,data%ntim
       if( diffdat( cdate, csec, data%date(i), data%datesec(i) ) > 0. ) then
          lotim = i - 1
          exit
       end if
       if( i == data%ntim ) then
          lotim = data%ntim
       end if
    end do

  end function lotim



end module mo_offline_data
