
      module mo_physcan

      private
      public :: physcan

      contains

      subroutine physcan( dtime, calday, ncdate, ncsec, nstep, &
			  nadv, ps, oro, phis, x0, &
			  t1, sh1, u, v, ts, &
			  taux, tauy, hflx, shflx, qrad, &
                          etadot, omga, cgs, kvh, zmu, &
			  zmd, zeu, hketa, hkbeta, cmfdqr, &
			  cldfr, cldtop, cldbot, zm, zint, &
                          concld, nrain, nevapr, cwat, pdmfuen, &
			  pdmfude, pdmfden, pdmfdde, as, x1, &
			  snow, fsds, soilw, precip, ts_avg, &
                          fsds_avg, platl, plonl, pplon, elapsed )
!-----------------------------------------------------------------------
! 	... Latitude loop over physics parameterizations
!-----------------------------------------------------------------------

      use mo_mpi
      use mo_grid,     only : plev, plevp, pcnst, plat
      use mo_physlic,  only : physlic
      use mo_lghtning, only : do_lghtning, lght_diagnostics

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        ncdate, &   ! date at beginning of current timestep in yymmdd format
        ncsec, &    ! seconds relative to ncdate
        nstep, &    ! timestep counter
        platl, &    ! dimension of latitude tile
        plonl, &    ! dimension of longitude tile
        pplon, &    ! number of longitude tiles
        nadv        ! number of misc advected variables

      real, intent(in) :: &
        dtime, &                ! timestep
        calday                  ! calandar day in range [1., 366.)
      real, intent(in) :: &
        x0(plonl,plev,nadv,platl,pplon), &  ! misc advected variables before slt
        u(plonl,plev,-3:platl+4,pplon), &   ! u wind
        v(plonl,plev,-2:platl+3,pplon), &   ! v wind
        etadot(plonl,plevp,platl,pplon), &  ! vertical velocity in hybrid coord
        ps(plonl,-3:platl+4,pplon)          ! surface pressure (Pa)
      real, dimension(plonl,platl,pplon), intent(in) :: &
        oro, &                  ! orography
        phis, &                 ! surface geopotential (m^2/s^2)
        ts, &                   ! surface temperature
        ts_avg, &               ! average surface temperature
        taux, &                 ! x surface stress (n)
        tauy, &                 ! y surface stress (n)
        hflx, &                 ! surface sensible heat flux (w/m^2)
        shflx, &                ! surface water vapor flux (kg/m^2/s)
        snow, &                 ! snow height (m)
        fsds, &                 ! srf direct radiation (w/m^2)
        fsds_avg                ! average srf direct radiation (w/m^2)
      real, dimension(plonl,platl,pplon), intent(inout) :: &
        soilw                   ! soil moisture
      real, dimension(plonl,plev,platl,pplon), intent(in) :: &
        t1, &                   ! temperature (K)
        sh1, &                  ! specific humidity (kg/kg)
        qrad, &                 ! radiative heating tendency (K/s)
        zmu, &                  ! mu2 from conv_ccm, kg/m^2/s
        zmd, &                  ! md2 from conv_ccm, kg/m^2/s
        zeu, &                  ! eu2 from conv_ccm, 1/s
        hketa, &                ! convective mass flux, Hack (kg/m^2/s)
        hkbeta, &               ! overshoot parameter, Hack (fraction)
        pdmfuen, &              
        pdmfude, &              
        pdmfden, &              
        pdmfdde                 
      real, dimension(plonl,plevp,platl,pplon), intent(inout) :: &
        cgs, &                  ! counter-gradient coefficient
        kvh, &                  ! vertical diffusion coefficient
        cldfr, &                ! cloud fraction
        zint                    ! geopotential height above surface at interfaces

      real, dimension(plonl,plev,platl,pplon), intent(inout) :: &
        omga, &                 ! vertical velocity in pressure coord
        cmfdqr, &               ! dq/dt due to convective rainout 
        concld, &               ! convective cloud fraction
        nrain, &                ! rate of release of stratiform precip (1/s)
        nevapr, &               ! rate of evaporation of precipitation (1/s)
        cwat                    ! cloud water (kg/kg)

      real, intent(inout) :: &
        as(plonl,plev,pcnst,platl,pplon), & ! advected species after slt on input,
                                            ! adjustment and tendency physics applied
                                            ! on output.
        x1(plonl,plev,nadv,platl,pplon), &  ! misc advected variables after slt on input.
                                            ! Contrib from physics added on output.
        cldtop(plonl,platl,pplon), &        ! cloud top level index
        cldbot(plonl,platl,pplon), &        ! cloud bottom level index
        zm(plonl,plev,platl,pplon), &       ! geopotential height above surface at midpoints
        precip(plonl,platl,pplon), &        ! srf precipitation (m)
        elapsed(10)                         ! timing

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: ip, j, k, m
      integer :: diagunit, ierr
      real, dimension(plonl,pcnst,platl,pplon) :: vdiffaj, vdiffbj, convj, cloudj
      real, dimension(pcnst,plat) :: vdiffm, convm, cloudm
      real, dimension(pcnst) :: mpi_tmpsum
      character(len=32) :: outfile

      vdiffaj(:,:,:,:) = 0.
      vdiffbj(:,:,:,:) = 0.
      convj(:,:,:,:)   = 0.
      cloudj(:,:,:,:)  = 0.

      if( pplon > 1 ) then
!$omp parallel do private( ip, j ), schedule(dynamic,1)
         do ip = 1,pplon
            do j = 1,platl
               call physlic( ip, j, dtime, calday, ncdate, &
			     ncsec, nstep, ps(1,j,ip), oro(1,j,ip), phis(1,j,ip), &
                             x0(1,1,2,j,ip), x0(1,1,1,j,ip), t1(1,1,j,ip), sh1(1,1,j,ip), u(1,1,j,ip), &
			     v(1,1,j,ip), ts(1,j,ip), taux(1,j,ip), tauy(1,j,ip), hflx(1,j,ip), &
			     shflx(1,j,ip), qrad(1,1,j,ip), etadot(1,1,j,ip), omga(1,1,j,ip), cgs(1,1,j,ip), &
			     kvh(1,1,j,ip), zmu(1,1,j,ip), zmd(1,1,j,ip), zeu(1,1,j,ip), hketa(1,1,j,ip), &
			     hkbeta(1,1,j,ip), cldtop(1,j,ip), cldbot(1,j,ip), zm(1,1,j,ip), zint(1,1,j,ip), &
			     cmfdqr(1,1,j,ip), cldfr(1,1,j,ip), concld(1,1,j,ip), nrain(1,1,j,ip), nevapr(1,1,j,ip), &
			     cwat(1,1,j,ip), pdmfuen(1,1,j,ip), pdmfude(1,1,j,ip), pdmfden(1,1,j,ip), pdmfdde(1,1,j,ip), &
			     as(1,1,1,j,ip), x1(1,1,2,j,ip), x1(1,1,1,j,ip), vdiffbj(1,1,j,ip), vdiffaj(1,1,j,ip), &
			     convj(1,1,j,ip), cloudj(1,1,j,ip), snow(1,j,ip), fsds(1,j,ip), soilw(1,j,ip), &
                             precip(1,j,ip), ts_avg(1,j,ip), fsds_avg(1,j,ip), plonl, platl, elapsed )
            end do
         end do
!$omp end parallel do
      else
!$omp parallel do private( j ), schedule(dynamic,1)
         do j = 1,platl
            call physlic( 1, j, dtime, calday, ncdate, &
			  ncsec, nstep, ps(1,j,1), oro(1,j,1), phis(1,j,1), &
                          x0(1,1,2,j,1), x0(1,1,1,j,1), t1(1,1,j,1), sh1(1,1,j,1), u(1,1,j,1), &
			  v(1,1,j,1), ts(1,j,1), taux(1,j,1), tauy(1,j,1), hflx(1,j,1), &
			  shflx(1,j,1), qrad(1,1,j,1), etadot(1,1,j,1), omga(1,1,j,1), cgs(1,1,j,1), &
			  kvh(1,1,j,1), zmu(1,1,j,1), zmd(1,1,j,1), zeu(1,1,j,1), hketa(1,1,j,1), &
			  hkbeta(1,1,j,1), cldtop(1,j,1), cldbot(1,j,1), zm(1,1,j,1), zint(1,1,j,1), &
			  cmfdqr(1,1,j,1), cldfr(1,1,j,1), concld(1,1,j,1), nrain(1,1,j,1), nevapr(1,1,j,1), &
			  cwat(1,1,j,1), pdmfuen(1,1,j,1), pdmfude(1,1,j,1), pdmfden(1,1,j,1), pdmfdde(1,1,j,1), &
			  as(1,1,1,j,1), x1(1,1,2,j,1), x1(1,1,1,j,1), vdiffbj(1,1,j,1), vdiffaj(1,1,j,1), &
			  convj(1,1,j,1), cloudj(1,1,j,1), snow(1,j,1), fsds(1,j,1), soilw(1,j,1), &
                          precip(1,j,1), ts_avg(1,j,1), fsds_avg(1,j,1), plonl, platl, elapsed )
         end do
!$omp end parallel do
      end if

!-----------------------------------------------------------------------
! 	... check for lightning diagnostics
!-----------------------------------------------------------------------
      if( do_lghtning ) then
         call lght_diagnostics( plonl, platl, pplon )
      end if

#ifdef DEBUG
      do m = 1,pcnst
         do j = 1,platl
            vdiffm(m,j+base_lat) = SUM( vdiffbj(:plonl,m,j,:pplon) )
         end do
      end do
#ifdef USE_MPI
      if( masternode ) then
        call mpi_gather( mpi_in_place, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         vdiffm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      else
        call mpi_gather( vdiffm(:pcnst,base_lat+1:base_lat+platl), pcnst*platl, MPI_DOUBLE_PRECISION, &
                         vdiffm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      endif
      if( ierr /= 0 ) then
         write(*,*) 'physcan: mpi_gather failed for vdiffm'
	 write(*,*) '         Error code = ',ierr
	 call endrun
      end if
#endif
      if( masternode ) then
         do m = 1,pcnst
            mpi_tmpsum(m) = SUM( vdiffm(m,:plat) )
         end do
         write(*,*) 'physcan: Mass before diffusion'
         write(*,*) mpi_tmpsum
      end if
      do m = 1,pcnst
         do j = 1,platl
 	    vdiffm(m,j+base_lat) = SUM( vdiffaj(:plonl,m,j,:pplon) )
         end do
      end do
#ifdef USE_MPI
      if( masternode ) then
        call mpi_gather( mpi_in_place, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         vdiffm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      else
        call mpi_gather( vdiffm(:pcnst,base_lat+1:base_lat+platl), pcnst*platl, MPI_DOUBLE_PRECISION, &
                         vdiffm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      endif
      if( ierr /= 0 ) then
         write(*,*) 'physcan: mpi_gather failed for vdiffm'
	 write(*,*) '         Error code = ',ierr
	 call endrun
      end if
#endif
      if( masternode ) then
         do m = 1,pcnst
            mpi_tmpsum(m) = SUM( vdiffm(m,:plat) )
         end do
         write(*,*) 'physcan: Mass after diffusion'
         write(*,*) mpi_tmpsum
      end if
      do m = 1,pcnst
         do j = 1,platl
            convm(m,j+base_lat)  = SUM( convj(:plonl,m,j,:pplon) )
            cloudm(m,j+base_lat) = SUM( cloudj(:plonl,m,j,:pplon) )
         end do
      end do
#ifdef USE_MPI
      if( masternode ) then
        call mpi_gather( mpi_in_place, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         convm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      else
        call mpi_gather( convm(:pcnst,base_lat+1:base_lat+platl), pcnst*platl, MPI_DOUBLE_PRECISION, &
                         convm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      endif
      if( ierr /= 0 ) then
         write(*,*) 'physcan: mpi_gather failed for convm'
         write(*,*) '         Error code = ',ierr
         call endrun
      end if
#endif
      if( masternode ) then
         do m = 1,pcnst
            mpi_tmpsum(m) = sum( convm(m,:plat) )
         end do
         write(*,*) 'physcan: Mass after convection'
         write(*,*) mpi_tmpsum
      end if
#ifdef USE_MPI
      if( masternode ) then
        call mpi_gather( mpi_in_place, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         cloudm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      else
        call mpi_gather( cloudm(:pcnst,base_lat+1:base_lat+platl), pcnst*platl, MPI_DOUBLE_PRECISION, &
                         cloudm, pcnst*platl, MPI_DOUBLE_PRECISION, &
                         0, MPI_COMM_WORLD, ierr )
      endif
      if( ierr /= 0 ) then
         write(*,*) 'physcan: mpi_gather failed for cloudm'
         write(*,*) '         Error code = ',ierr
         call endrun
      end if
#endif
      if( masternode ) then
         do m = 1,pcnst
            mpi_tmpsum(m) = SUM( cloudm(m,:plat) )
         end do
         write(*,*) 'physcan: Mass after clouds'
         write(*,*) mpi_tmpsum
      end if

      write(*,*) 'CWAT: Min value = ',MINVAL( as(:plonl,:plev,pcnst,:platl,:pplon) )
      write(*,*) 'CWAT: Max value = ',MAXVAL( as(:plonl,:plev,pcnst,:platl,:pplon) )
      write(*,*) 'CWAT: Non_zero count = ', COUNT( as(:plonl,:plev,pcnst,:platl,:pplon) /= 0. )
      do k = 1,plev
         if( ANY( as(:plonl,k,pcnst,:platl,:pplon) /= 0. ) )  then
            exit
         end if
      end do
      write(*,*) 'CWAT: Lowest all zero level = ',k
#endif

      end subroutine physcan

      end module mo_physcan
