
      module mo_regrider
!---------------------------------------------------------------------
!	... general horizontal regriding
!---------------------------------------------------------------------

      implicit none

      integer, private, parameter :: max_conv = 25
      integer, private :: conv_cnt = 0

      type grid_conv

      integer :: nfrom_lats, nto_lats, &
                 nfrom_lons, nto_lons, &
                 latl, latu, lati, &
                 lonl, lonu, loni, &
                 from_min_lat, from_max_lat, &                ! global indicies
                 wings                                        ! number of ghost wings
      integer, pointer, dimension(:) :: interp_lats, &
                                        interp_lons
      real    :: max_lat, min_lat
      real    :: max_lon, min_lon
      real, pointer, dimension(:)    :: lat_del, lon_del
      real, pointer, dimension(:)    :: from_lats, to_lats
      real, pointer, dimension(:)    :: from_lons, to_lons

      logical :: do_lon, do_lat
      logical :: from_lats_mono_pos, from_lons_mono_pos
      logical :: active
      logical :: xparent_xform

      end type grid_conv

      save

      type(grid_conv), private :: converter(max_conv)

      private :: regrid_horiz

      contains

      integer function regrid_inti( from_nlats, to_nlats, &
                                    from_nlons, to_nlons, &
                                    from_lons,  to_lons, &
                                    from_lats,  to_lats, &
				    wing_cnt, platl, &
                                    do_lons,    do_lats )
!---------------------------------------------------------------------
!	... determine indicies and deltas for transform
!           note : it is assumed that the latitude and longitude
!                  arrays are monotonic
!--------------------------------------------------------------------

      use mo_grid,      only : plat
      use mo_mpi,       only : base_lat
      use mo_constants, only : r2d

      implicit none

!---------------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) :: from_nlats, to_nlats, &
                             from_nlons, to_nlons, &
                             wing_cnt                                        ! number of wing terms
      integer, intent(in) :: platl
      real, intent(in)    :: from_lats(from_nlats)
      real, intent(in)    :: to_lats(to_nlats)
      real, intent(in)    :: from_lons(from_nlons)
      real, intent(in)    :: to_lons(to_nlons)
      logical, optional, intent(in) :: do_lons, do_lats

!---------------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------------
      integer :: from_lat, to_lat
      integer :: from_lon, to_lon
      integer :: astat
      integer :: i, j, jglb, m
      integer :: max_ind(1), min_ind(1)
      real    :: target_lat, target_lon
      logical :: match, xparent_xform
      logical :: check_lats, check_lons, check_match
      logical :: lat_xform, lon_xform

!---------------------------------------------------------------------
!	... check if dimension transform is required or requested
!---------------------------------------------------------------------
      check_lats = .not. present( do_lats )
      if( .not. check_lats ) then
         check_lats = do_lats
      end if
      check_lons = .not. present( do_lons )
      if( .not. check_lons ) then
         check_lons = do_lons
      end if

!---------------------------------------------------------------------
!	... no transform requested; leave
!---------------------------------------------------------------------
      if( .not. check_lats .and. .not. check_lons ) then
	 regrid_inti = -1
	 return
      end if

!---------------------------------------------------------------------
!	... check to see if from lat grid == to lat grid
!---------------------------------------------------------------------
      if( check_lats ) then
         lat_xform = from_nlats /= to_nlats
         if( .not. lat_xform ) then
            do j = 1,to_nlats
	       if( from_lats(j) /= to_lats(j) ) then
	          lat_xform = .true.
	          exit
	       end if
	    end do
         end if
      else
	 lat_xform = .false.
      end if
!---------------------------------------------------------------------
!	... check to see if from lon grid == to lon grid
!---------------------------------------------------------------------
      if( check_lons ) then
         lon_xform = from_nlons /= to_nlons
         if( .not. lon_xform ) then
            do i = 1,to_nlons
	       if( from_lons(i) /= to_lons(i) ) then
	          lon_xform = .true.
	          exit
	       end if
	    end do
         end if
      else
	 lon_xform = .false.
      end if
!---------------------------------------------------------------------
!	... transform necessary ?
!---------------------------------------------------------------------
      xparent_xform = .not. lat_xform .and. .not. lon_xform

!---------------------------------------------------------------------
!	... check for match with existing transform
!---------------------------------------------------------------------
      if( conv_cnt > 0 ) then
         do m = 1,conv_cnt
	    if( wing_cnt /= converter(m)%wings ) then
	       match = .false.
	       cycle
	    end if
	    if( .not. xparent_xform .and. .not. converter(m)%xparent_xform  ) then
	       check_match = lat_xform .and. converter(m)%do_lat
	    else if( xparent_xform .and. converter(m)%xparent_xform  ) then
	       check_match = check_lats .and. converter(m)%do_lat
	    else
	       cycle
	    end if
	    if( check_match ) then
	       match = converter(m)%nfrom_lats == from_nlats
	       if( match ) then
	          do j = 1,from_nlats
	             if( converter(m)%from_lats(j) /= from_lats(j) ) then
		        match = .false.
		        exit
		     end if
	          end do
	       end if
	       if( .not. match ) then
	          cycle
	       end if
	       match = converter(m)%nto_lats == to_nlats
	       if( match ) then
	          do j = 1,to_nlats
	             if( converter(m)%to_lats(j) /= to_lats(j) ) then
		        match = .false.
		        exit
		     end if
	          end do
	       end if
            else if( lat_xform .eqv. converter(m)%do_lat ) then
	       match = .true.
	    else
	       match = .false.
	    end if
	    if( .not. match ) then
	       cycle
	    end if
            if( lon_xform .and. converter(m)%do_lon ) then
	       match = converter(m)%nfrom_lons == from_nlons
	       if( match ) then
	          do i = 1,from_nlons
	             if( converter(m)%from_lons(i) /= from_lons(i) ) then
		        match = .false.
		        exit
		     end if
	          end do
	       end if
	       if( .not. match ) then
	          cycle
	       end if
	       match = converter(m)%nto_lons == to_nlons
	       if( match ) then
	          do i = 1,to_nlons
	             if( converter(m)%to_lons(i) /= to_lons(i) ) then
		        match = .false.
		        exit
		     end if
	          end do
	       end if
            else if( lon_xform .eqv. converter(m)%do_lon ) then
	       match = .true.
	    else
	       match = .false.
	    end if
	    if( match ) then
	       exit
	    end if
	 end do
      else
         match = .false.
      end if
      if( match ) then
	 regrid_inti = m
	 return
      else
!---------------------------------------------------------------------
!	... check for conversion count
!---------------------------------------------------------------------
         if( conv_cnt >= max_conv ) then
	    write(*,*) 'regrid_inti: reached max conversion count of ',max_conv
	    regrid_inti = -2
	    return
         end if
         conv_cnt = conv_cnt + 1
      end if

      converter(conv_cnt)%xparent_xform = xparent_xform
      converter(conv_cnt)%wings         = wing_cnt
      if( xparent_xform ) then
         converter(conv_cnt)%do_lat = check_lats
	 if( check_lats ) then
            converter(conv_cnt)%do_lon = .false.
	 else
            converter(conv_cnt)%do_lon = check_lons
	 end if
      else
         converter(conv_cnt)%do_lat = lat_xform
         converter(conv_cnt)%do_lon = lon_xform
      end if

!---------------------------------------------------------------------
!	... new transform; store grids
!---------------------------------------------------------------------
      write(*,*) 'regrid_inti: diagnostics for transform index = ',conv_cnt
      write(*,'(1x,''regrid_inti: from_nlats, to_nlats = '',2i6)') from_nlats, to_nlats
      write(*,'(1x,''regrid_inti: from_nlons, to_nlons = '',2i6)') from_nlons, to_nlons
      write(*,*) 'regrid_inti: lat_xform, lon_xform = ',lat_xform,lon_xform
      if( converter(conv_cnt)%do_lat ) then
         allocate( converter(conv_cnt)%from_lats(from_nlats),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate from_lats array'
	    call endrun
         end if
         converter(conv_cnt)%from_lats(:) = from_lats(:)
         allocate( converter(conv_cnt)%to_lats(to_nlats),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate to_lats array'
	    call endrun
         end if
         write(*,*) 'regrid_inti: size of to_lats = ',size( to_lats )
         write(*,*) 'regrid_inti: size of converter(conv_cnt)%to_lats = ',size( converter(conv_cnt)%to_lats )
         converter(conv_cnt)%to_lats(:) = to_lats(:)
#ifdef debug
         write(*,*) 'regrid_inti: to_lats (deg):'
         write(*,'(10f8.3)') to_lats(:to_nlats)*r2d
#endif
      end if
      if( converter(conv_cnt)%do_lon ) then
         allocate( converter(conv_cnt)%from_lons(from_nlons),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate from_lons array'
	    call endrun
         end if
         converter(conv_cnt)%from_lons(:) = from_lons(:)
         allocate( converter(conv_cnt)%to_lons(to_nlons),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate to_lons array'
	    call endrun
         end if
         write(*,*) 'regrid_inti: size of to_lons = ',size( to_lons )
         write(*,*) 'regrid_inti: size of converter(conv_cnt)%to_lons = ',size( converter(conv_cnt)%to_lons )
#ifdef debug
         write(*,*) 'regrid_inti: to_lons (deg):'
         write(*,'(10f8.3)') to_lons(:to_nlons)*r2d
#endif
         converter(conv_cnt)%to_lons(:) = to_lons(:)
      end if

!---------------------------------------------------------------------
!	... set "module" variables
!---------------------------------------------------------------------
      converter(conv_cnt)%nfrom_lats = from_nlats
      converter(conv_cnt)%nto_lats   = to_nlats
      converter(conv_cnt)%nfrom_lons = from_nlons
      converter(conv_cnt)%nto_lons   = to_nlons
      write(*,*) 'regrid_inti: size of to_lons = ',size( to_lons )

      if( converter(conv_cnt)%do_lat ) then
         max_ind(:) = maxloc( from_lats(:) )
         min_ind(:) = minloc( from_lats(:) )
         converter(conv_cnt)%max_lat = from_lats(max_ind(1))
         converter(conv_cnt)%min_lat = from_lats(min_ind(1))
         if( max_ind(1) >= min_ind(1) ) then
	    converter(conv_cnt)%latl = 1
	    converter(conv_cnt)%latu = from_nlats
	    converter(conv_cnt)%lati = 1
	    converter(conv_cnt)%from_lats_mono_pos = .true.
         else
	    converter(conv_cnt)%latl = from_nlats
	    converter(conv_cnt)%latu = 1
	    converter(conv_cnt)%lati = -1
	    converter(conv_cnt)%from_lats_mono_pos = .false.
         end if
      end if

      if( converter(conv_cnt)%do_lon ) then
         max_ind(:) = maxloc( from_lons(:) )
         min_ind(:) = minloc( from_lons(:) )
         converter(conv_cnt)%max_lon = from_lons(max_ind(1))
         converter(conv_cnt)%min_lon = from_lons(min_ind(1))
         if( max_ind(1) >= min_ind(1) ) then
	    converter(conv_cnt)%lonl = 1
	    converter(conv_cnt)%lonu = from_nlons
	    converter(conv_cnt)%loni = 1
	    converter(conv_cnt)%from_lons_mono_pos = .true.
         else
	    converter(conv_cnt)%lonl = from_nlons
	    converter(conv_cnt)%lonu = 1
	    converter(conv_cnt)%loni = -1
	    converter(conv_cnt)%from_lons_mono_pos = .false.
         end if
      end if

      if( converter(conv_cnt)%do_lat ) then
!---------------------------------------------------------------------
!	... allocate interpolation latitude indicies
!---------------------------------------------------------------------
         allocate( converter(conv_cnt)%interp_lats(to_nlats),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate interp lats array'
	    call endrun
         end if
!---------------------------------------------------------------------
!	... allocate interpolation latitude deltas
!---------------------------------------------------------------------
         allocate( converter(conv_cnt)%lat_del(to_nlats),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate lat del array'
	    call endrun
         end if
!---------------------------------------------------------------------
!	... set interpolation latitude indicies and deltas
!---------------------------------------------------------------------
         if( .not. converter(conv_cnt)%xparent_xform ) then
            do to_lat = 1,converter(conv_cnt)%nto_lats
               target_lat = to_lats(to_lat)
	       if( target_lat <= converter(conv_cnt)%min_lat ) then
	          converter(conv_cnt)%lat_del(to_lat) = 0.
	          converter(conv_cnt)%interp_lats(to_lat) = converter(conv_cnt)%latl
	       else if( target_lat >= converter(conv_cnt)%max_lat ) then
	          converter(conv_cnt)%lat_del(to_lat) = 1.
	          if( converter(conv_cnt)%from_lats_mono_pos ) then
	             converter(conv_cnt)%interp_lats(to_lat) = converter(conv_cnt)%latu - 1
	          else
	             converter(conv_cnt)%interp_lats(to_lat) = converter(conv_cnt)%latu + 1
	          end if
	       else
	          do from_lat = converter(conv_cnt)%latl,converter(conv_cnt)%latu,converter(conv_cnt)%lati
	             if( target_lat < from_lats(from_lat) ) then
		        j = from_lat - converter(conv_cnt)%lati
	                converter(conv_cnt)%interp_lats(to_lat) = min( converter(conv_cnt)%nfrom_lats, max( 1,j ) )
	                converter(conv_cnt)%lat_del(to_lat) = &
                                           (target_lat - from_lats(j))/(from_lats(from_lat) - from_lats(j))
	                exit
	             end if
	          end do
               end if
            end do
	 else
            do to_lat = 1,converter(conv_cnt)%nto_lats
	       converter(conv_cnt)%interp_lats(to_lat) = to_lat
	       converter(conv_cnt)%lat_del(to_lat)     = 0.
            end do
	    if( converter(conv_cnt)%lati == 1 ) then
	       to_lat = converter(conv_cnt)%nto_lats
	    else
	       to_lat = 1
	    end if
	    converter(conv_cnt)%interp_lats(to_lat) = to_lat - converter(conv_cnt)%lati
	    converter(conv_cnt)%lat_del(to_lat)     = 1.
	 end if
	 jglb = min( plat,base_lat + platl + wing_cnt )
	 converter(conv_cnt)%from_max_lat = min( converter(conv_cnt)%nfrom_lats, &
                                                 converter(conv_cnt)%interp_lats(jglb) + converter(conv_cnt)%lati )
	 jglb = max( 1,base_lat - wing_cnt + 1 )
	 converter(conv_cnt)%from_min_lat = max( 1,converter(conv_cnt)%interp_lats(jglb) )
      end if

      if( converter(conv_cnt)%do_lon ) then
!---------------------------------------------------------------------
!	... allocate interpolation longitude indicies
!---------------------------------------------------------------------
         allocate( converter(conv_cnt)%interp_lons(to_nlons),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate interp lon array'
	    call endrun
         end if
!---------------------------------------------------------------------
!	... allocate interpolation longitude deltas
!---------------------------------------------------------------------
         allocate( converter(conv_cnt)%lon_del(to_nlons),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'regrid_inti: failed to allocate lon del array'
	    call endrun
         end if
!---------------------------------------------------------------------
!	... set interpolation longitude indicies and deltas
!---------------------------------------------------------------------
         do to_lon = 1,converter(conv_cnt)%nto_lons
            target_lon = to_lons(to_lon)
	    if( target_lon <= converter(conv_cnt)%min_lon ) then
	       converter(conv_cnt)%lon_del(to_lon) = 0.
	       converter(conv_cnt)%interp_lons(to_lon) = converter(conv_cnt)%lonl
	    else if( target_lon >= converter(conv_cnt)%max_lon ) then
	       converter(conv_cnt)%lon_del(to_lon) = 1.
	       if( converter(conv_cnt)%from_lons_mono_pos ) then
	          converter(conv_cnt)%interp_lons(to_lon) = converter(conv_cnt)%lonu - 1
	       else
	          converter(conv_cnt)%interp_lons(to_lon) = converter(conv_cnt)%lonu + 1
	       end if
	    else
	       do from_lon = converter(conv_cnt)%lonl,converter(conv_cnt)%lonu,converter(conv_cnt)%loni
	          if( target_lon < from_lons(from_lon) ) then
		     i = from_lon - converter(conv_cnt)%loni
	             converter(conv_cnt)%interp_lons(to_lon) = min( converter(conv_cnt)%nfrom_lons, &
                                                                    max( 1,i ) )
	             converter(conv_cnt)%lon_del(to_lon) = &
                                        (target_lon - from_lons(i))/(from_lons(from_lon) - from_lons(i))
	             exit
	          end if
	       end do
            end if
         end do
      end if

      regrid_inti = conv_cnt

      end function regrid_inti

      subroutine regrid_3d( from_field, to_field, index, to_lat_min, to_lat_max, &
                            do_poles, scaling )
!--------------------------------------------------------------------
!	... regrid horizontal data
!           note: this subroutine works on latitude "slices"
!--------------------------------------------------------------------

      use mo_grid, only : plev, plat
      use mo_mpi,  only : base_lat, masternode, lastnode
      use m_adv,   only : has_spole, has_npole

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in)           :: index
      integer, intent(in)           :: to_lat_min, to_lat_max                     ! globals
      real, optional, intent(in)    :: scaling
      real, intent(in)              :: from_field(:,:,:)
      real, intent(out)             :: to_field(:,:,:)
      logical, optional, intent(in) :: do_poles

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: i, j, k, astat
      integer :: nlons, nlats
      real    :: temp, rnlons
      real, allocatable :: wrk(:,:)

      if( index /= 0 ) then
!--------------------------------------------------------------------
!	... check index for validity
!--------------------------------------------------------------------
         if( index < 1 .or. index > conv_cnt ) then
	    write(*,'(''regrid_3d: '',3x,'' is out of range'')') index
	    call endrun
         end if
!--------------------------------------------------------------------
!	... allocate work array
!--------------------------------------------------------------------
	 nlats = to_lat_max - to_lat_min + 1
	 nlons = converter(index)%nto_lons
         allocate( wrk(nlons,nlats),stat=astat )
         if( astat /= 0 ) then
	    write(*,*) 'regrid_3d: failed to allocate work array'
	    call endrun
         end if
!--------------------------------------------------------------------
!	... latitude interp
!--------------------------------------------------------------------
         do k = 1,plev
            call regrid_horiz( from_field(:,:,k), wrk, to_lat_min, to_lat_max, index )
	    do j = 1,nlats
	       do i = 1,nlons
	          to_field(i,k,j) = wrk(i,j)
	       end do
	    end do
         end do
         deallocate( wrk )
      else
!--------------------------------------------------------------------
!	... transparent transform
!--------------------------------------------------------------------
	 nlats = size(from_field,dim=2)
	 nlons = size(from_field,dim=1)
         do k = 1,plev
	    do j = 1,nlats
	       do i = 1,nlons
	          to_field(i,k,j) = from_field(i,j,k)
	       end do
	    end do
	 end do
      end if

      if( present(scaling) ) then
         if( scaling /= 1. ) then
	    do j = 1,nlats
               do k = 1,plev
	          do i = 1,nlons
		     to_field(i,k,j) = scaling * to_field(i,k,j)
	          end do
	       end do
	    end do
         end if
      end if

      if( present(do_poles) ) then
         if( do_poles ) then
	    rnlons = 1./real(nlons)
	    if( has_spole .and. to_lat_min == 1 ) then
               do k = 1,plev
	          temp = sum( to_field(:,k,2) )*rnlons
	          to_field(:,k,1) = temp
	       end do
	    end if
	    if( has_npole .and. to_lat_max == plat ) then
               do k = 1,plev
	          temp = sum( to_field(:,k,nlats-1) )*rnlons
	          to_field(:,k,nlats) = temp
	       end do
	    end if
         end if
      end if

      end subroutine regrid_3d

      subroutine regrid_2d( from_field, to_field, index, to_lat_min, to_lat_max, &
                            do_poles, scaling )
!--------------------------------------------------------------------
!	... regrid horizontal data
!           note: this subroutine works on horizontal "slices"
!--------------------------------------------------------------------

      use mo_grid, only : plon, plat
      use mo_mpi,  only : masternode, lastnode
      use m_adv,   only : has_spole, has_npole

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in)           :: index
      integer, intent(in)           :: to_lat_min, to_lat_max                    ! globals
      real, optional, intent(in)    :: scaling
      real, intent(in)              :: from_field(:,:)
      real, intent(out)             :: to_field(:,:)
      logical, optional, intent(in) :: do_poles

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: i, j
      integer :: nlons, nlats
      real    :: temp, rnlons

      if( index /= 0 ) then
!--------------------------------------------------------------------
!	... check index for validity
!--------------------------------------------------------------------
         if( index < 1 .or. index > conv_cnt ) then
	    write(*,'(''regrid_2d: '',3x,'' is out of range'')') index
	    call endrun
         end if
	 nlats = to_lat_max - to_lat_min + 1
         call regrid_horiz( from_field, to_field, to_lat_min, to_lat_max, index )
	 nlons = converter(index)%nto_lons
      else
!--------------------------------------------------------------------
!	... transparent transform
!--------------------------------------------------------------------
	 nlats = size(from_field,dim=2)
	 nlons = size(from_field,dim=1)
         do j = 1,nlats
            do i = 1,nlons
	       to_field(i,j) = from_field(i,j)
	    end do
	 end do
      end if

      if( present(scaling) ) then
         if( scaling /= 1. ) then
	    do j = 1,nlats
	       do i = 1,nlons
	          to_field(i,j) = scaling * to_field(i,j)
	       end do
	    end do
         end if
      end if

      if( present(do_poles) ) then
         if( do_poles ) then
	    rnlons = 1./real(nlons)
	    if( has_spole .and. to_lat_min == 1 ) then
	       temp = sum( to_field(:,2) )*rnlons
	       to_field(:,1) = temp
	    end if
	    if( has_npole .and. to_lat_max == plat ) then
	       temp = sum( to_field(:,nlats-1) )*rnlons
	       to_field(:,nlats) = temp
	    end if
         end if
      end if

      end subroutine regrid_2d

      subroutine regrid_horiz( from_field, to_field, latl, latu, index )
!--------------------------------------------------------------------
!	... regrid horizontal data
!           note: this subroutine works on horizontal "slices"
!--------------------------------------------------------------------

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: latl, latu                                  ! globals
      integer, intent(in) :: index
      real, intent(in)    :: from_field(:,:)
      real, intent(out)   :: to_field(:,:)

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: j, ji, ji1, jmin, jmax, jloc, jl, ju
      integer :: i, ii, ii1
      integer :: from_latl, from_latu
      real    :: wrk(converter(index)%nto_lons,converter(index)%from_min_lat:converter(index)%from_max_lat)
      logical :: partial_from_lats

      jmax              = size( from_field,dim=2 )
      partial_from_lats = jmax /= converter(index)%nfrom_lats
      jl = converter(index)%from_min_lat
      if( partial_from_lats ) then
         ju = min( jl + jmax - 1,converter(index)%from_max_lat )
         if( jmax /= (ju-jl+1) ) then
	    write(*,'(''regrid_2d: wrk lats = '',i3,'' from lats = '',i3)') ju-jl+1,jmax
	    call endrun
         end if
         from_latl = 1
         from_latu = jmax
      else
         ju        = converter(index)%from_max_lat
         from_latl = converter(index)%from_min_lat
         from_latu = converter(index)%from_max_lat
      end if
!--------------------------------------------------------------------
!	... first longitude interp
!--------------------------------------------------------------------
      if( converter(index)%do_lon ) then
         do i = 1,converter(index)%nto_lons
            ii = converter(index)%interp_lons(i)
	    ii1 = ii + converter(index)%loni
	    wrk(i,jl:ju) = from_field(ii,from_latl:from_latu) &
                         + converter(index)%lon_del(i) * (from_field(ii1,from_latl:from_latu) - from_field(ii,from_latl:from_latu))
         end do
      else
	 wrk(:,jl:ju) = from_field(:,1:jmax)
      end if
!--------------------------------------------------------------------
!	... then latitude interp
!--------------------------------------------------------------------
      if( converter(index)%do_lat ) then
	 jmin = max( 1,latl )
	 jmax = min( converter(index)%nto_lats,latu )
         do j = jmin,jmax
	    ji = converter(index)%interp_lats(j)
	    ji1 = ji + converter(index)%lati
	    jloc = j - jmin + 1
	    to_field(:,jloc) = wrk(:,ji) &
                             + converter(index)%lat_del(j) * (wrk(:,ji1) - wrk(:,ji))
         end do
      else
	 to_field(:,:) = wrk(:,:)
      end if

      end subroutine regrid_horiz

      subroutine regrid_1d( from_field, to_field, index, scaling, do_lat, &
                            to_lat_min, to_lat_max, do_lon, to_lon_min, to_lon_max )
!--------------------------------------------------------------------
!	... regrid horizontal data
!           note: this subroutine works on a horizontal "line"
!--------------------------------------------------------------------

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in)           :: index
      integer, optional, intent(in) :: to_lat_min, to_lat_max
      integer, optional, intent(in) :: to_lon_min, to_lon_max
      real, intent(in)              :: from_field(:)
      real, intent(out)             :: to_field(:)
      real, optional, intent(in)    :: scaling
      logical, optional, intent(in) :: do_lat, do_lon

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: j, ji, ji1, offset
      integer :: jmin, jmax, jloc
      integer :: i, ii, ii1
      integer :: nlons, nlats
      integer :: size_to_field
      integer :: astat
      real, allocatable :: wrk(:)
      logical :: partial_from_field

!--------------------------------------------------------------------
!	... check index for validity
!--------------------------------------------------------------------
      if( index < 0 .or. index > conv_cnt ) then
	 write(*,'(''regrid_1d: '',3x,'' is out of range'')') index
	 call endrun
      end if
      partial_from_field = size(from_field) /= converter(index)%nfrom_lats
      if( present(do_lat) ) then
!--------------------------------------------------------------------
!	... latitude interp
!--------------------------------------------------------------------
         size_to_field = size( to_field )
         if( .not. present( to_lat_min ) .or. .not. present( to_lat_max ) ) then
	    write(*,*) 'regrid_1d: lat xform requires to_lat_min,to_lat_max arguments'
	    call endrun
	 end if
	 nlats = to_lat_max - to_lat_min + 1
         if( size_to_field /= nlats ) then
	    write(*,*) 'regrid_1d: size of to_field does not match to_lat_min,max inputs'
	    call endrun
	 end if
         if( index /= 0 ) then
	    allocate( wrk(nlats),stat=astat )
	    if( astat /= 0 ) then
	       write(*,*) 'regrid_1d: failed to allocate wrk space'
	       call endrun
	    end if
	    jmin = max( 1,to_lat_min )
	    jmax = min( converter(index)%nto_lats,to_lat_max )
            if( partial_from_field ) then
               offset = 1 - converter(index)%from_min_lat
            else
               offset = 0
            end if
            do j = jmin,jmax
               ji   = converter(index)%interp_lats(j) + offset
	       ji1  = ji + converter(index)%lati
	       jloc = j - jmin + 1
	       wrk(jloc) = from_field(ji) &
                         + converter(index)%lat_del(j) * (from_field(ji1) - from_field(ji))
            end do
            to_field(1:nlats) = wrk(1:nlats)
	 else
            if( partial_from_field ) then
	       to_field(1:nlats) = from_field(1:nlats)
            else
	       jmin = max( 1,to_lat_min )
	       jmax = min( converter(index)%nto_lats,to_lat_max )
	       to_field(1:nlats) = from_field(jmin:jmax)
            end if
	 end if
	 if( present(scaling) ) then
	    if( scaling /= 1. ) then
               to_field(1:nlats) = scaling*to_field(1:nlats)
	    end if
	 end if
      else if( present(do_lon ) ) then
         if( index /= 0 ) then
!--------------------------------------------------------------------
!	... check dimensions
!--------------------------------------------------------------------
	    if( .not. converter(index)%do_lon ) then
	       write(*,*) 'regrid_1d: requesting lon interp; not set in intialization'
	       call endrun
	    end if
	    if( size( from_field ) /= converter(index)%nfrom_lons ) then
	       write(*,*) 'regrid_1d: input field does not match module dimension'
	       call endrun
	    end if
	    if( size( to_field ) /= converter(index)%nto_lons ) then
	       write(*,*) 'regrid_1d: output field does not match module dimension'
	       call endrun
	    end if
!--------------------------------------------------------------------
!	... lontitude interp
!--------------------------------------------------------------------
	    nlons = converter(index)%nto_lons
	    allocate( wrk(nlons),stat=astat )
	    if( astat /= 0 ) then
	       write(*,*) 'regrid_1d: failed to allocate wrk space'
	       call endrun
	    end if
            do i = 1,nlons
               ii = converter(index)%interp_lons(i)
	       ii1 = ii + converter(index)%loni
	       wrk(i) = from_field(ii) &
                      + converter(index)%lon_del(i) * (from_field(ii1) - from_field(ii))
            end do
            to_field(1:nlons) = wrk(1:nlons)
	 else
	    nlons = size( from_field )
	    to_field(1:nlons) = from_field(1:nlons)
	 end if
	 if( present(scaling) ) then
	    if( scaling /= 1. ) then
               to_field(1:nlons) = scaling*to_field(1:nlons)
	    end if
	 end if
      end if
      if( allocated( wrk ) ) then
         deallocate( wrk )
      end if

      end subroutine regrid_1d

      function regrid_lat_limits( index )
!--------------------------------------------------------------------
!	... return the from latitude limits
!--------------------------------------------------------------------

      implicit none

!--------------------------------------------------------------------
!	... dummy arguments
!--------------------------------------------------------------------
      integer, intent(in) :: index

!--------------------------------------------------------------------
!	... function declaration
!--------------------------------------------------------------------
      integer :: regrid_lat_limits(2)

      regrid_lat_limits(:2) = (/ converter(index)%from_min_lat, converter(index)%from_max_lat /)

      end function regrid_lat_limits

      subroutine regrid_diagnostics( index )

      implicit none

      integer, intent(in) :: index

      if( index /= 0 ) then
!--------------------------------------------------------------------
!	... check index for validity
!--------------------------------------------------------------------
         if( index < 1 .or. index > conv_cnt ) then
	    write(*,'(''regrid_diagnostics: '',3x,'' is out of range'')') index
	    call endrun
         end if
	 write(*,*) ' '
	 write(*,*) 'Regrid diagnostics for index ',index
	 write(*,*) 'Lon, Lat xform = ',converter(index)%do_lon,converter(index)%do_lat
	 if( converter(index)%do_lat ) then
	    write(*,*) ' '
	    write(*,*) 'Latitude regridding'
	    write(*,*) '-------- ----------'
	    write(*,*) 'Number from lats, to lats = ',converter(index)%nfrom_lats,converter(index)%nto_lats
	    write(*,*) 'latl, latu, lati = ',converter(index)%latl,converter(index)%latu,converter(index)%lati
	    write(*,*) 'From min,max lat = ',converter(index)%from_min_lat,converter(index)%from_max_lat
	    write(*,*) 'Wing count = ',converter(index)%wings
	    write(*,*) 'From lats monotically increasing = ',converter(index)%from_lats_mono_pos
	    write(*,*) 'Lat interp indicies'
	    write(*,'(20i4)') converter(index)%interp_lats(:)
	    write(*,*) 'Lat interp delta'
	    write(*,'(5(1x,g22.15))') converter(index)%lat_del(:)
	 end if
	 if( converter(index)%do_lon ) then
	    write(*,*) ' '
	    write(*,*) 'Longitude regridding'
	    write(*,*) '--------- ----------'
	    write(*,*) 'Number from lons, to lons = ',converter(index)%nfrom_lons,converter(index)%nto_lons
	    write(*,*) 'lonl, lonu, loni = ',converter(index)%lonl,converter(index)%lonu,converter(index)%loni
	    write(*,*) 'From lons monotically increasing = ',converter(index)%from_lons_mono_pos
	    write(*,*) 'Lon interp indicies'
	    write(*,'(20i4)') converter(index)%interp_lons(:)
	    write(*,*) 'Lon interp delta'
	    write(*,'(5(1x,g22.15))') converter(index)%lon_del(:)
	 end if
	 write(*,*) ' '
      end if

      end subroutine regrid_diagnostics

      end module mo_regrider

