
      module mo_srfalb
!-----------------------------------------------------------------------
!! mgs++ : rewrite to use MODIS albedo maps for snow covered/non snow covered surfaces
!!         (27 June 2004)
!! mgs++ : bug fix for fractional orography (18 Aug 2004)
!! tlaepple: ocean albedo set to 0.07 (13 Jun 2005)
!! tlaepple: sea ice albedo of southern hemisphere set to 0.89 (13 June 2005)
!-----------------------------------------------------------------------

      implicit none

      save

      private
      public :: srfalb_inti, srfalb

      real :: polar_limit
      real :: days(12)
      real, allocatable :: agreen(:,:,:,:)                    ! monthly "green" albedo from MODIS
      real, allocatable :: awhite(:,:,:,:)                    ! monthly snow albedo interpolated from MODIS

      contains

      subroutine srfalb_inti( ncfile, lpath, mspath, plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... read albedo data
!-----------------------------------------------------------------------

      use netcdf
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits, regrid_diagnostics
      use mo_grid,       only : plon, plat
      use mo_mpi,        only : base_lat
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file
      use mo_calendar,   only : caldayr

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      character(len=*), intent(in) :: ncfile               ! file name of netcdf file containing data
      character(len=*), intent(in) :: lpath                ! local pathname to ncfile
      character(len=*), intent(in) :: mspath               ! mass store pathname to ncfile

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: gndx
      integer :: ncid
      integer :: nlon
      integer :: nlat
      integer :: nmonth
      integer :: jl
      integer :: ju
      integer :: ierr
      integer :: m
      integer :: vid, dimid_lon, dimid_lat, dimid_month
      integer :: jlim_in(2)
      integer :: start(3)
      integer :: count(3)
      integer :: dates(12) = (/ 116, 214, 316, 415,  516,  615, &
				716, 816, 915, 1016, 1115, 1216 /)
      real    :: wrk2d(plon,platl)
      real, allocatable :: agreen_in(:,:,:)
      real, allocatable :: awhite_in(:,:,:)
      real, allocatable :: lat(:)
      real, allocatable :: lon(:)


!-----------------------------------------------------------------------
!       ... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )

!-----------------------------------------------------------------------
!       ... check number of months
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid_month ), &
                         'srfalb_inti: failed to find dimension time' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_month, nmonth ), &
                         'srfalb_inti: failed to get length of dimension time' )
      if( nmonth /= 12 ) then
         write(*,*) 'srfalb_inti: error! nmonth = ',nmonth,', expecting 12'
         call endrun
      end if
!-----------------------------------------------------------------------
!	... initialize the monthly day of year times
!-----------------------------------------------------------------------
      do m = 1,12
         days(m) = caldayr( dates(m), 0 )
      end do
!-----------------------------------------------------------------------
!	... allocate the module arrays
!-----------------------------------------------------------------------
      allocate( agreen(plonl,platl,pplon,12), &
		awhite(plonl,platl,pplon,12), stat=ierr )
      if( ierr /= 0 ) then
	 write(*,*) 'srflab_inti: failed to allocate module arrays; error = ',ierr
	 call endrun
      end if
      agreen(:,:,:,:) = -9999.
      awhite(:,:,:,:) = -9999.
!-----------------------------------------------------------------------
!       ... get latitude and longitude
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'srfalb_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'srfalb_inti: failed to get length of dimension lat' )
      allocate( lat(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'srfalb_inti: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'srfalb_inti: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'srfalb_inti: failed to read variable lat' )
      lat(:nlat) = lat(:nlat) * d2r

      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                         'srfalb_inti: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'srfalb_inti: failed to get length of dimension lon' )
      allocate( lon(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'srfalb_inti: lon allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'srfalb_inti: failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                         'srfalb_inti: failed to read variable lon' )
      lon(:nlon) = lon(:nlon) * d2r

!-----------------------------------------------------------------------
!       ... get grid interp limits
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, plat, &
                          nlon, plon, &
                          lon,  lam, &
                          lat,  phi, &
                          platl, 0 )
      deallocate( lat, lon, stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'srfalb_inti: failed to deallocate lat,lon; ierr = ',ierr
         call endrun
      end if
      jl = base_lat+1
      ju = base_lat+platl
      if( gndx /= 0 )then
         jlim_in = regrid_lat_limits( gndx )
      else
         jlim_in = (/ jl,ju /)
      end if
      write(*,'(''srfalb_inti: gndx = '',i2,'', grid limits = '',2i4,'', jl,ju = '',2i4)') &
         gndx, jlim_in, jl, ju

!-----------------------------------------------------------------------
!       ... allocate input variables
!-----------------------------------------------------------------------
      allocate( agreen_in(nlon,jlim_in(1):jlim_in(2),12), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'srfalb_inti: agreen_in allocation error = ',ierr
         call endrun
      end if
      allocate( awhite_in(nlon,jlim_in(1):jlim_in(2),12), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'srfalb_inti: awhite_in allocation error = ',ierr
         call endrun
      end if

!-----------------------------------------------------------------------
! 	... read the surface data
!-----------------------------------------------------------------------
      start(:) = (/ 1, jlim_in(1), 1 /)
      count(:) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, nmonth /)
      write(*,*) 'srfalb_inti : start, count = ',start(:),count(:)   !! mgs++
      call handle_ncerr( nf_inq_varid( ncid, 'agreen', vid ), 'srfalb_inti: getting agreen id' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, agreen_in ), &
                         'srfalb_inti: getting agreen' )
      call handle_ncerr( nf_inq_varid( ncid, 'awhite', vid ), 'srfalb_inti: getting awhite id' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, awhite_in ), &
                         'srfalb_inti: getting awhite' )

      call handle_ncerr( nf_close(ncid), 'srfalb_inti: closing netcdf file' )

!-----------------------------------------------------------------------
!	... regrid
!-----------------------------------------------------------------------
      do m = 1,nmonth
         call regrid_2d( agreen_in(:,jlim_in(1):jlim_in(2),m), wrk2d, gndx, jl, ju, do_poles=.true. )
         agreen(:,:,:,m) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
         call regrid_2d( awhite_in(:,jlim_in(1):jlim_in(2),m), wrk2d, gndx, jl, ju, do_poles=.true. )
         awhite(:,:,:,m) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
      end do

      deallocate( agreen_in, awhite_in, stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'srfalb_inti: deallocation error = ',ierr
         call endrun
      end if

      polar_limit = 67. * d2r

      end subroutine srfalb_inti

      subroutine srfalb( lat, ip, albs, calday, tsurf, &
                         oro, snow, plonl )
!-----------------------------------------------------------------------
! 	... compute surface albedo
!-----------------------------------------------------------------------

      use mo_constants, only : phi
      use mo_mpi,       only : base_lat
      use mo_histout,   only : outfld, match_file_cnt
      use mo_grid,      only : plat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) ::  lat          ! lat index for two dimensional data arrays
      integer, intent(in) ::  ip           ! lon tile index
      integer, intent(in) ::  plonl        ! lon tile dim
      real, intent(in)    ::  calday       ! time of year
      real, intent(in)    ::  tsurf(plonl)  ! surface temperature
      real, intent(in)    ::  oro(plonl)   ! orography (0=ocean, 1=land, 2=seaice)
      real, intent(in)    ::  snow(plonl)  ! snow height (only used as yes/no flag
      real, intent(out)   ::  albs(plonl)  ! srf alb

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i, m, last, next            ! longitude index
      integer :: file
      real    :: dels
      real    :: lcagreen(plonl)
      real    :: lcawhite(plonl)

!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      if( calday < days(1) ) then
	 next = 1
	 last = 12
	 dels = (365. + calday - days(12)) / (365. + days(1) - days(12))
      else if( calday >= days(12) ) then
	 next = 1
	 last = 12
	 dels = (calday - days(12)) / (365. + days(1) - days(12))
      else
         do m = 11,1,-1
	    if( calday >= days(m) ) then
	       exit
	    end if
         end do
	 last = m
	 next = m + 1
	 dels = (calday - days(m)) / (days(m+1) - days(m))
      end if
      dels = max( min( 1.,dels ),0. )

!--------------------------------------------------------
!	... interpolate the modis albedo maps in time
!--------------------------------------------------------
      lcagreen(:) = agreen(:,lat,ip,last) &
                  + dels * (agreen(:,lat,ip,next) - agreen(:,lat,ip,last))
      lcawhite(:) = awhite(:,lat,ip,last) &
                  + dels * (awhite(:,lat,ip,next) - awhite(:,lat,ip,last))

      albs(:) = -9999.
      do i = 1,plonl
         albs(i) = .07                   ! open ocean value 
         if ( oro(i) > 1.6 ) then
            if ( phi(base_lat+lat) < 0. ) then  ! ## tlaepple 
                 albs(i) = .89  ! use constant value for antarctic sea ice ##tlaepple
            else 
                 albs(i) = .78  ! use constant value for arctic sea ice ##tlaepple
            endif
         else if( oro(i) > .4 ) then
             if( snow(i) > .008 ) then   ! ### mgs++ ### snow treshold, tested with ECMWF and NCEP 
                albs(i) = lcawhite(i)    ! use snow albedo map
             else
                albs(i) = lcagreen(i)    ! land: assign green vegetation preliminarily
             end if
         end if
      end do

!--------------------------------------------------------------------------------
!       ... output surface albedo to history file
!--------------------------------------------------------------------------------
      do file = 1,match_file_cnt
         call outfld( 'SRFALB', albs(:), plonl, ip, lat, file )
      end do

      end subroutine srfalb

      end module mo_srfalb
