
      module mo_surf
!--------------------------------------------------------------------------
!	... surface data
!--------------------------------------------------------------------------

      implicit none

      save

      real, allocatable, dimension(:,:,:) :: &
               desert, &
               ocean
      real, allocatable, dimension(:,:,:,:) :: &
               seaice

      contains

      subroutine surf_inti( ncfile, lpath, mspath, plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... read surface data database
!-----------------------------------------------------------------------

      use netcdf
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits, regrid_diagnostics
      use mo_grid,       only : plon, plat
      use mo_mpi,        only : base_lat
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl, platl, pplon
      character(len=*), intent(in) :: &
        ncfile, &            ! file name of netcdf file containing data
        lpath, &             ! local pathname to ncfile
        mspath               ! mass store pathname to ncfile

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: gndx, &
                 ncid, &
                 nlon, &
                 nlat, &
                 nmonth, &
                 jlim_in(2), &
                 jl, &
                 ju, &
                 ierr, &
                 m
      integer :: vid, dimid_lon, dimid_lat, dimid_month
      integer, dimension(3) :: start, count
      real :: wrk2d(plon,platl)
      real, allocatable :: &
             desert_in(:,:), &
             ocean_in(:,:), &
             seaice_in(:,:,:), &
             lat(:), &
             lon(:)

!-----------------------------------------------------------------------
!	... allocate the module arrays
!-----------------------------------------------------------------------
      allocate( desert(plonl,platl,pplon), &
		ocean(plonl,platl,pplon), &
		seaice(plonl,platl,pplon,12), stat=ierr )
      if( ierr /= 0 ) then
	 write(*,*) 'mo_surf: failed to allocate module arrays; error = ',ierr
	 call endrun
      end if
      desert(:,:,:)   = z'7ff7ffff7ff7ffff'
      ocean(:,:,:)    = z'7ff7ffff7ff7ffff'
      seaice(:,:,:,:) = z'7ff7ffff7ff7ffff'

!-----------------------------------------------------------------------
!       ... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )

!-----------------------------------------------------------------------
!       ... check number of months
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'month', dimid_month ), &
                         'surf_inti: failed to find dimension month' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_month, nmonth ), &
                         'surf_inti: failed to get length of dimension month' )
      if( nmonth /= 12 ) then
         write(*,*) 'surf_inti: error! nmonth = ',nmonth,', expecting 12'
         call endrun
      end if
!-----------------------------------------------------------------------
!       ... get latitude and longitude
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'surf_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'surf_inti: failed to get length of dimension lat' )
      allocate( lat(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'surf_inti: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'surf_inti: failed to read variable lat' )
      lat(:nlat) = lat(:nlat) * d2r

      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                         'surf_inti: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'surf_inti: failed to get length of dimension lon' )
      allocate( lon(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: lon allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'surf_inti: failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                         'surf_inti: failed to read variable lon' )
      lon(:nlon) = lon(:nlon) * d2r

!-----------------------------------------------------------------------
!       ... get grid interp limits
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, plat, &
                          nlon, plon, &
                          lon,  lam, &
                          lat,  phi, &
                          platl, 0 )
      deallocate( lat, lon, stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: failed to deallocate lat,lon; ierr = ',ierr
         call endrun
      end if
      jl = base_lat+1
      ju = base_lat+platl
      if( gndx /= 0 )then
         jlim_in = regrid_lat_limits( gndx )
	 call regrid_diagnostics( gndx )
      else
         jlim_in = (/ jl,ju /)
      end if
      write(*,'(''surf_inti: gndx='',i2,'', grid limits = '',2i4,'', jl,ju='',2i4)') &
         gndx, jlim_in, jl, ju

!-----------------------------------------------------------------------
!       ... allocate input variables
!-----------------------------------------------------------------------
      allocate( desert_in(nlon,jlim_in(1):jlim_in(2)), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: desert_in allocation error = ',ierr
         call endrun
      end if
      allocate( ocean_in(nlon,jlim_in(1):jlim_in(2)), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: ocean_in allocation error = ',ierr
         call endrun
      end if
      allocate( seaice_in(nlon,jlim_in(1):jlim_in(2),nmonth), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: seaice_in allocation error = ',ierr
         call endrun
      end if

!-----------------------------------------------------------------------
! 	... read the surface data
!-----------------------------------------------------------------------
      start(:) = (/ 1, jlim_in(1), 1 /)
      count(:) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, nmonth /)

      call handle_ncerr( nf_inq_varid( ncid, 'desert', vid ), 'surf_inti: getting desert id' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, start(1:2), count(1:2), desert_in ), &
                         'surf_inti: getting desert' )

      call handle_ncerr( nf_inq_varid( ncid, 'ocean', vid ), 'surf_inti: getting ocean id' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, start(1:2), count(1:2), ocean_in ), &
                         'surf_inti: getting ocean' )

      call handle_ncerr( nf_inq_varid( ncid, 'seaice', vid ), 'surf_inti: getting seaice id' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, seaice_in ), &
                         'surf_inti: getting seaice' )

      call handle_ncerr( nf_close(ncid), 'surf_inti: closing netcdf file' )

!-----------------------------------------------------------------------
!	... due to possible input data errors limit all
!           variables to a the range [0,1]
!lwh 10/00 -- not really necessary when using the netcdf input file.
!             the data in this file are already limited to [0,1].
!-----------------------------------------------------------------------
      desert_in(:,:)   = min( 1., max( 0., desert_in(:,:) ) )
      ocean_in(:,:)    = min( 1., max( 0., ocean_in(:,:) ) )
      seaice_in(:,:,:) = min( 1., max( 0., seaice_in(:,:,:) ) )

!-----------------------------------------------------------------------
!	... regrid
!-----------------------------------------------------------------------
      call regrid_2d( desert_in(:,jlim_in(1):jlim_in(2)), wrk2d, gndx, jl, ju, do_poles=.true. )
      desert = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
      call regrid_2d( ocean_in(:,jlim_in(1):jlim_in(2)), wrk2d, gndx, jl, ju, do_poles=.true. )
      ocean = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
      do m = 1,nmonth
         call regrid_2d( seaice_in(:,jlim_in(1):jlim_in(2),m), wrk2d, gndx, jl, ju, do_poles=.true. )
         seaice(:,:,:,m) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
      end do

      deallocate( desert_in, ocean_in, seaice_in, stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'surf_inti: desert_in,ocean_in,seaice_in deallocation error = ',ierr
         call endrun
      end if

      end subroutine surf_inti

      end module mo_surf
