
      module mo_tropopause
!---------------------------------------------------------------
!	... variables for the tropopause module
!---------------------------------------------------------------

      implicit none

      private
      public  :: trop_inti, tropp_lev

      save

      integer :: gndx = 0
      integer :: jlim_in(2)
      real    :: days(12)
      real, allocatable :: tropp_p(:,:,:,:)

      contains

      subroutine trop_inti( plonl, platl, pplon, ncfile, lpath, mspath )
!------------------------------------------------------------------
!	... initialize upper boundary values
!------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use mo_constants,  only : phi, lam, d2r
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits
      use mo_grid,       only : plong => plon, platg => plat
      use mo_calendar,   only : caldayr
      use mo_mpi,        only : base_lat

      implicit none

!------------------------------------------------------------------
!	... dummy args
!------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      character(len=*), intent(in) :: &
        ncfile, &            ! file name of netcdf file containing data
        lpath, &             ! local pathname to ncfile
        mspath               ! mass store pathname to ncfile

!------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------
      integer :: jl, ju
      integer :: i, j, n
      integer :: ierr
      integer :: ncid
      integer :: dimid
      integer :: vid
      integer :: nlon, nlat, ntimes
      integer :: start(3)
      integer :: count(3)
      integer :: dates(12) = (/ 116, 214, 316, 415,  516,  615, &
                                716, 816, 915, 1016, 1115, 1216 /)
      real    :: wrk2d(plong,platl)
      real, allocatable :: tropp_p_in(:,:,:)
      real, allocatable :: lat(:)
      real, allocatable :: lon(:)

!-----------------------------------------------------------------------
!       ... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )
!-----------------------------------------------------------------------
!       ... get time dimension
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid ), &
                         'trop_inti: failed to find dimension time' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, ntimes ), &
                         'trop_inti: failed to get length of dimension time' )
      if( ntimes /= 12 )then
         write(*,*) 'trop_inti: number of months = ',ntimes,'; expecting 12'
         call endrun
      end if
!-----------------------------------------------------------------------
!       ... get latitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), &
                         'trop_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat ), &
                         'trop_inti: failed to get length of dimension lat' )
      allocate( lat(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'trop_inti: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'trop_inti: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'trop_inti: failed to read variable lat' )
      lat(:nlat) = lat(:nlat) * d2r
!-----------------------------------------------------------------------
!       ... get longitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), &
                         'trop_inti: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon ), &
                         'trop_inti: failed to get length of dimension lon' )
      allocate( lon(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'trop_inti: lon allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'trop_inti: failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                         'trop_inti: failed to read variable lon' )
      lon(:nlon) = lon(:nlon) * d2r

!-----------------------------------------------------------------------
!       ... setup regriding
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, platg, &
                          nlon, plong, &
                          lon,  lam, &
                          lat,  phi, &
                          0, platl, &
                          do_lons=.true.,do_lats=.true. )
      jl = base_lat + 1
      ju = base_lat + platl
      jlim_in = regrid_lat_limits( gndx)
#ifdef DEBUG
	write(*,*) 'lat_in = '
	write(*,'(10f7.1)') lat(jlim_in(1):jlim_in(2))/d2r
	write(*,*) 'lat_out='
	write(*,'(10f7.1)') phi(jl:ju)/d2r
#endif
      deallocate( lat, lon, stat=ierr )

      write(*,'(''trop_inti: gndx = '',i2,'', grid limits = '',2i4,'', jl,ju = '',2i4)') &
         gndx, jlim_in, jl, ju

!------------------------------------------------------------------
!	... allocate arrays
!------------------------------------------------------------------
      allocate( tropp_p_in(nlon,jlim_in(1):jlim_in(2),ntimes), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'trop_inti: tropp_p_in allocation error = ',ierr
         call endrun
      end if
      allocate( tropp_p(plonl,platl,pplon,ntimes), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'trop_inti: tropp_p allocation error = ',ierr
         call endrun
      end if
!------------------------------------------------------------------
!	... read in the tropopause pressure
!------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'trop_p', vid ), 'trop_inti: failed to get trop_p id' )
      start = (/ 1, jlim_in(1), 1 /)
      count = (/ nlon, jlim_in(2) - jlim_in(1) + 1, ntimes /)
      call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, tropp_p_in ), &
                         'trop_inti: getting trop_p' )
!------------------------------------------------------------------
!	... close the netcdf file
!------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), 'trop_inti: failed to close file ' // trim(ncfile) )

!--------------------------------------------------------------------
!	... regrid
!--------------------------------------------------------------------
      do n = 1,ntimes
         call regrid_2d( tropp_p_in(:,jlim_in(1):jlim_in(2),n), wrk2d, gndx, jl, ju, &
                         do_poles=.true. )
         tropp_p(:,:,:,n) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
      end do

      deallocate( tropp_p_in )

!--------------------------------------------------------
!	... initialize the monthly day of year times
!--------------------------------------------------------
      do n = 1,12
         days(n) = caldayr( dates(n), 0 )
      end do

      end subroutine trop_inti

      subroutine tropp_lev( lat, ip, zmid, pmid, tfld, &
                            calday, troplev, plonl, hstout )
!--------------------------------------------------------------------
!	... return tropopause level index
!--------------------------------------------------------------------

      use mo_grid,       only : plev
      use mo_local_time, only : local_time, local_time_char
      use mo_histout,    only : outfld, sim_file_cnt

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in)  :: lat               ! lat index
      integer, intent(in)  :: plonl             ! lon tile dim
      integer, intent(in)  :: ip                ! longitude tile index
      integer, intent(out) :: troplev(plonl)    ! tropopause level vertical index
      real, intent(in)     :: calday            ! day of year including fraction
      real, intent(in)     :: tfld(plonl,plev)  ! (k)
      real, intent(in)     :: zmid(plonl,plev)  ! midpoint height (km)
      real, intent(in)     :: pmid(plonl,plev)  ! midpoint pressure (pa)
      logical, optional, intent(in) :: hstout   ! history output flag

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      real, parameter    :: ztrop_low  = 5.            ! lowest tropopause level allowed (km)
      real, parameter    :: ztrop_high = 20.           ! highest tropopause level allowed (km)
      real, parameter    :: max_dtdz   = 2.            ! max dt/dz for tropopause level (degrees k/km)

      integer  ::  i, k, m
      integer  ::  last
      integer  ::  next
      integer  ::  file
      real     ::  dt
      real     ::  dels
      real     ::  ptrop
      real     ::  wrk(plonl)
      logical  ::  found_trop(plonl)
      character(len=32) :: fldname

!--------------------------------------------------------
!   	... find the tropopause location
!--------------------------------------------------------
      found_trop(:) = .false.
long_loop : &
      do i = 1,plonl
         do k = plev-1,2,-1
	    if( zmid(i,k) < ztrop_low ) then
	       cycle
	    else if( zmid(i,k) > ztrop_high ) then
	       exit
	    end if
            dt = tfld(i,k) - tfld(i,k-1)
            if( 0. < dt .and. dt < max_dtdz*(zmid(i,k-1) - zmid(i,k)) ) then
               troplev(i)    = k
	       found_trop(i) = .true.
	       exit
	    end if
         end do
!        if( .not. found_trop(i) ) then
!           write(*,*) 'set_ub_vals: failed to find tropopause at lat ind=',lat,', lon ind=',i
!        end if
      end do long_loop

!--------------------------------------------------------
!	... check to see if found all tropopause levels
!--------------------------------------------------------
climatology : &
      if( any( .not. found_trop(:) ) ) then
!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
         if( calday < days(1) ) then
	    next = 1
	    last = 12
	    dels = (365. + calday - days(12)) / (365. + days(1) - days(12))
         else if( calday >= days(12) ) then
	    next = 1
	    last = 12
	    dels = (calday - days(12)) / (365. + days(1) - days(12))
         else
            do m = 11,1,-1
	       if( calday >= days(m) ) then
	          exit
	       end if
            end do
	    last = m
	    next = m + 1
	    dels = (calday - days(m)) / (days(m+1) - days(m))
         end if
         dels = max( min( 1.,dels ),0. )
!--------------------------------------------------------
!	... get tropopause level from climatology
!--------------------------------------------------------
long_loop2 : &
         do i = 1,plonl
            if( .not. found_trop(i) ) then
               ptrop = tropp_p(i,lat,ip,last) &
                     + dels * (tropp_p(i,lat,ip,next) - tropp_p(i,lat,ip,last))
               do k = plev,2,-1
                  if( ptrop >= pmid(i,k) ) then
                     troplev(i) = k - 1
                     exit
                  end if
               end do
            end if
         end do long_loop2
      end if climatology

!--------------------------------------------------------
!	... output tropopause height
!--------------------------------------------------------
      if( present( hstout ) ) then
         if( hstout ) then
            do i = 1,plonl
               wrk(i) = zmid(i,troplev(i))
            end do
            do file = 1,sim_file_cnt
               call outfld( 'TROPLEV', wrk,  plonl,  ip, lat, file )
               if( local_time(file) > 0 ) then
                  fldname = 'TROPLEV_' // local_time_char(file) // '_LT'
                  call outfld( fldname, wrk,  plonl,  ip, lat, file )
               end if
            end do
         end if
      end if

      end subroutine tropp_lev

      end module mo_tropopause
