
#include <params.h>

      module MO_ZHANG

      implicit none

      save

      real, parameter :: a  = 21.656
      real, parameter :: b  = 5418.
      real, parameter :: c1 = 6.112
      real, parameter :: c2 = 17.67
      real, parameter :: c3 = 243.5
      real, parameter :: cp = 1004.64
      real, parameter :: rl = 2.5104e6
      real, parameter :: eps1 = .622
      real, parameter :: qmin = 1.e-20
      real, parameter :: tfreez  = 273.16
      real, parameter :: capelmt = 0.

      private
      public  :: CONV_CCM

      CONTAINS

      subroutine CONV_CCM( t, qh, cnst, ncnst, pcpc, &
			   jctop, jcbot, pblh, zm, geos, &
			   zi, qtg, ttg, pap, paph, &
			   dpp, ts, delt, mcon, cme, &
			   nstep, lat, ip, tpert, qpert, &
			   dlf, pflx, zdu, cmfdqr, zmu, &
			   zmd, zeu, plonl )
!-----------------------------------------------------------------------
! 	... MAIN DRIVER FOR ZHANG-MCFARLANE CONVECTION SCHEME
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! This is contributed code not fully standardized by the CCM core group.
! All variables have been typed, where most are identified in comments
! The current procedure will be reimplemented in a subsequent version 
! of the CCM where it will include a more straightforward formulation 
! and will make use of the standard CCM nomenclature
!-----------------------------------------------------------------------

      use mo_grid,       only : plev, plevp, pcnst
      use MO_HACK,       only : limcnv, grav, rgrav, rgas
      use MO_CLDPRP,     only : CLDPRP
      use MO_ZHANG_SUBS, only : CONVTRAN, CLOSURE, BUOYAN, Q1Q2_PJR

      implicit none


!-----------------------------------------------------------------------
! same as conv.up except saturation vapor pressure is calculated
! in a different way.
!
! jul 17/92 - guang jun zhang, m.lazare. calls new buoyan, q1q2
!             and moment (several new work fields added for later).
!
! nov 21/91 - m.lazare. like previous conv except calls new
!                       clpdprp.
! feb 18/91 - guang jun zhang, m.lazare, n.mcfarlane.
!             previous version conv.
! performs deep convective adjustment based on mass-flux closure
! algorithm.
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: ncnst   ! number of passive constituents
      integer, intent(in) :: nstep
      integer, intent(in) :: lat, ip
      integer, intent(in) :: plonl
      real, intent(in)    :: delt

      real, dimension(plonl), intent(in) :: &
          geos, &
          pblh, &
          tpert, &
          qpert, &
          ts
      real, dimension(plonl,plevp), intent(in) :: &
          paph, &
          zi
      real, dimension(plonl,plev), intent(in) :: &
          pap, &
          dpp, &
          zm

      real, intent(inout) :: t(plonl,plev)

      real, dimension(plonl), intent(out) :: &
          jctop, &
          jcbot, &
          pcpc
      real, dimension(plonl,plevp), intent(out) :: &
           pflx                ! scattered precip flux at each level
      real, dimension(plonl,plev), intent(out) :: &
          mcon, &
          dlf, &              ! scattrd version of the detraining cld h2o tend
          cme, &         
          zdu, &         
          cmfdqr, &         
          zmu, &              ! scattered mu2, kg/m2/s
          zmd, &              ! scattered md2, kg/m2/s
          zeu, &              ! scattered eu2, 1/s
          qtg, &
          ttg
      real, intent(inout) :: qh(plonl,plev,1) 
      real, intent(inout) :: cnst(plonl,plev,ncnst)  ! chem constiuents

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer, dimension(plonl) :: &
             lcl, &
             lel, &
             lon, &
             maxi, &
             ideep, &
             itmp, &
             lclg, &
             lelg, &
             maxg
      real :: precip, twodelt, twodelti
      real, dimension(plonl,plev) :: &
          u, &
          v, &
          q, &
          p, &
          z, &
          s, &
          qh0, &
          tp, &
          qstp, &
          pcpck, &
          dlg, &                 ! gathrd version of the detraining cld h2o tend
          cug, &                 ! gathered condensation rate 
          evpg, &                ! gathered evap rate of rain in downdraft
          qg, &
          tg, &
          pg, &
          zg, &
          sg, &
          tpg, &
          qstpg, &
          ug, &
          vg, &
          cmeg, &
          cmfdqrg
      real, dimension(plonl,plevp) :: &
          zf, &
          pf, &
          pflxg, &               ! gather precip flux at each level
          zfg
      real, dimension(plonl) :: &
          cape, &
          tl, &
          sumq, &
          pcpdh, &
          pcpr, &
          pcps, &
          mumax, &
          zs, &
          pblt, &
          paprc, &
          paprs, &
          capeg, &
          tlg

!-----------------------------------------------------------------------
! 	... Work fields arising from gathered calculations
!-----------------------------------------------------------------------
      integer :: &
             i, ii, ind, k, lengath, msg
      integer, dimension(plonl) :: &
             jt, jlcl, j0, jd
      real :: &
          psevap, psheat, psrain, qdifr, sdifr, wrk
      real, dimension(plonl,plev) :: &
          mu, eu, dqdt, dsdt, du, md, ed, &
          alpha, sd, qd, mc, qhat, qu, su, &
          qs, shat, dp, hmn, hsat, ql, &
          ud, vd, mu2, eu2, du2, md2, ed2
      real, dimension(plonl) :: &
          betau, betad, qdb, sdb, dsubcld, &
          mb, totpcp, totevp

      logical, save :: momentm = .false.

!-----------------------------------------------------------------------
! 	... Set internal variable "msg" (convection limit) to "limcnv-1"
!-----------------------------------------------------------------------
      msg      = limcnv - 1
      twodelt  = 2.*delt
      twodelti = 1./twodelt

!-----------------------------------------------------------------------
! 	... Initialize necessary arrays.
!           zero out variables not used in ccm
!-----------------------------------------------------------------------
      paprc(1:plonl) = 0.
      paprs(1:plonl) = 0.
      ideep(:) = 0
      psheat   = 0.
      psevap   = 0.
      psrain   = 0.

!-----------------------------------------------------------------------
! 	... Initialize convective tendencies
!-----------------------------------------------------------------------
      do k = 1,plev
         dqdt(:,k) = 0.
         dsdt(:,k) = 0.
         pcpck(:,k) = 0.
         pflx(:,k) = 0.
         pflxg(:,k) = 0.
         cme(:,k) = 0.
         dlf(:,k) = 0.
         mcon(:,k) = 0.
         cmfdqr(:,k) = 0.
         zdu(:,k) = 0.
         zmu(:,k) = 0.
         zmd(:,k) = 0.
         zeu(:,k) = 0.
      end do
      pflx(:,plevp) = 0
      pflxg(:,plevp) = 0
      if( .not. momentm ) then
         do k = 1,plev
            u(:,k) = 0.
            v(:,k) = 0.
         end do
      end if

      pblt(:) = REAL( plev )
      pcpc(:) = 0.
      pcpr(:) = 0.
      pcps(:) = 0.
      dsubcld(:) = 0.
      sumq(:) = 0.
      pcpdh(:) = rgrav
      jctop(:) = REAL( plev )
      jcbot(:) = 1.

!-----------------------------------------------------------------------
! 	... calculate local pressure (mbs) and height (m) for both interface
!           and mid-layer locations.
!-----------------------------------------------------------------------
      zs(:) = geos(:)*rgrav
      pf(:,plevp) = paph(:,plevp)*.01
      zf(:,plevp) = zi(:,plevp) + zs(:)
      do k = 1,plev
         p(:,k)  = pap(:,k)*.01
         pf(:,k) = paph(:,k)*.01
         z(:,k)  = zm(:,k) + zs(:)
         zf(:,k) = zi(:,k) + zs(:)
      end do

      do k = plev-1,msg+1,-1
         do i = 1,plonl
            if( ABS( z(i,k) - zs(i) - pblh(i) ) < (zf(i,k) - zf(i,k+1))*.5 ) then
	       pblt(i) = REAL( k )
            end if
         end do
      end do

!-----------------------------------------------------------------------
! 	... Store incoming specific humidity field for subsequent calculation
!           of precipitation (through change in storage).
!           convert from specific humidity (bounded by qmin) to mixing ratio.
!           define dry static energy (normalized by cp).
!-----------------------------------------------------------------------
      do k = 1,plev
         qh0(:,k) = qh(:,k,1)
         q(:,k) = MAX( qh(:,k,1),qmin )
         s(:,k) = t(:,k) + (grav/cp)*z(:,k)
         tp(:,k) = 0.
         shat(:,k) = s(:,k)
         qhat(:,k) = q(:,k)
         dp(:,k) = dpp(:,k)*.01
         qg(:,k) = q(:,k)
         tg(:,k) = t(:,k)
         pg(:,k) = p(:,k)
         zg(:,k) = z(:,k)
         sg(:,k) = s(:,k)
         tpg(:,k) = tp(:,k)
         zfg(:,k) = zf(:,k)
         qstpg(:,k) = q(:,k)
         ug(:,k) = u(:,k)
         vg(:,k) = v(:,k)
         dlg(:,k) = 0.
         dlf(:,k) = 0.
      end do
      zfg(:,plevp) = zf(:,plevp)
      capeg(:) = 0.
      lclg(:) = 1
      lelg(:) = plev
      maxg(:) = 1
      tlg(:) = 400.
      dsubcld(:) = 0.
      qdb(:) = 0.
      sdb(:) = 0.
      betau(:) = 0.
      betad(:) = 0.

!-----------------------------------------------------------------------
! 	... Evaluate covective available potential energy (cape).
!-----------------------------------------------------------------------
      call BUOYAN( q, t, p, z, pf, &
                   tp, qstp, tl, cape, pblt, &
		   lcl, lel, lon, maxi, rgas, &
		   msg, nstep, lat, tpert, qpert, plonl )

!-----------------------------------------------------------------------
! 	... Determine whether grid points will undergo some deep convection
!           (ideep=1) or not (ideep=0), based on values of cape,lcl,lel
!           (require cape.gt. 0 and lel<lcl as minimum conditions).
!-----------------------------------------------------------------------
      lengath = COUNT( cape(:plonl) > capelmt )
      if( lengath == 0 ) then
         return
      else
	 ii = 0
	 do i = 1,plonl
	    if( cape(i) > capelmt ) then
	       ii = ii + 1
	       ideep(ii) = i
	    end if
	 end do
      end if
#ifdef CRAY
!DIR  IVDEP
#endif
!-----------------------------------------------------------------------
! 	... Obtain gathered arrays necessary for ensuing calculations.
!-----------------------------------------------------------------------
      do k = 1,plev
         do i = 1,lengath
	    ind = ideep(i)
            dp(i,k) = .01*dpp(ind,k)
            qg(i,k) = q(ind,k)
            tg(i,k) = t(ind,k)
            pg(i,k) = p(ind,k)
            zg(i,k) = z(ind,k)
            sg(i,k) = s(ind,k)
            tpg(i,k) = tp(ind,k)
            zfg(i,k) = zf(ind,k)
            qstpg(i,k) = qstp(ind,k)
            ug(i,k) = u(ind,k)
            vg(i,k) = v(ind,k)
         end do
      end do

      do i = 1,lengath
	 ind = ideep(i)
         zfg(i,plevp) = zf(ind,plevp)
         capeg(i) = cape(ind)
         lclg(i) = lcl(ind)
         lelg(i) = lel(ind)
         maxg(i) = maxi(ind)
         tlg(i) = tl(ind)
      end do

!-----------------------------------------------------------------------
! 	... Calculate sub-cloud layer pressure "thickness" for use in
!           closure and tendency routines.
!-----------------------------------------------------------------------
      do k = msg + 1,plev
         do i = 1,lengath
            if( k >= maxg(i) ) then
               dsubcld(i) = dsubcld(i) + dp(i,k)
            end if
         end do
      end do

!-----------------------------------------------------------------------
! 	... Define array of factors (alpha) which defines interfacial
!           values, as well as interfacial values for (q,s) used in
!           subsequent routines.
!-----------------------------------------------------------------------
      do k = msg + 2,plev
         do i = 1,lengath
            alpha(i,k) = .5
            sdifr = 0.
            qdifr = 0.
            if( sg(i,k) > 0. .or. sg(i,k-1) > 0. ) then 
                sdifr = ABS( (sg(i,k) - sg(i,k-1))/MAX( sg(i,k-1),sg(i,k) ) )
	    end if
            if( qg(i,k) > 0. .or. qg(i,k-1) > 0. ) then
                qdifr = ABS( (qg(i,k) - qg(i,k-1))/MAX( qg(i,k-1),qg(i,k) ) )
	    end if
            if( sdifr > 1.e-6 ) then
               shat(i,k) = LOG(sg(i,k-1)/sg(i,k))*sg(i,k-1)*sg(i,k)/(sg(i,k-1) - sg(i,k))
            else
               shat(i,k) = .5* (sg(i,k) + sg(i,k-1))
            end if
            if( qdifr > 1.e-6 ) then
               qhat(i,k) = LOG(qg(i,k-1)/qg(i,k))*qg(i,k-1)*qg(i,k)/(qg(i,k-1) - qg(i,k))
            else
               qhat(i,k) = .5* (qg(i,k) + qg(i,k-1))
            end if
         end do
      end do

!-----------------------------------------------------------------------
! 	... Cloud properties
!-----------------------------------------------------------------------
      call CLDPRP( qg, tg, ug, vg, pg, &
                   zg, sg, mu, eu, du, &
                   md, ed, sd, qd, ud, &
                   vd, mc, qu, su, zfg, &
                   qs, hmn, hsat, alpha, shat, &
                   ql, totpcp, totevp, cmeg, maxg, &
                   lelg, jt, jlcl, maxg, j0, &
                   jd, 1, lengath, rgas, msg, &
		   nstep, lat, pflxg, evpg, cug, &
		   mu2, eu2, du2, md2, ed2, &
		   cmfdqrg, limcnv, plonl )

!-----------------------------------------------------------------------
! 	... Determine cloud base mass flux.
!-----------------------------------------------------------------------
      do i = 1,lengath
	 ind = maxg(i)
         qdb(i) = qd(i,ind)
         sdb(i) = sd(i,ind)
         betad(i) = md(i,ind)
         betau(i) = mu(i,ind)
      end do

!-----------------------------------------------------------------------
! 	... Convert detrainment from units of "1/m" to "1/mb".
!-----------------------------------------------------------------------
      do k = msg + 1,plev
         do i = 1,lengath
            wrk     = (zfg(i,k) - zfg(i,k+1))/dp(i,k)
            du(i,k) = du(i,k)*wrk
            eu(i,k) = eu(i,k)*wrk
            ed(i,k) = ed(i,k)*wrk
            cug(i,k) = cug(i,k)*wrk
            cmeg(i,k) = cmeg(i,k)*wrk
            cmfdqrg(i,k) = cmfdqrg(i,k)*wrk
            evpg(i,k) = evpg(i,k)*wrk
            du2(i,k) = du2(i,k)*wrk
            eu2(i,k) = eu2(i,k)*wrk
            ed2(i,k) = ed2(i,k)*wrk
         end do
      end do

      call CLOSURE( qg, tg, pg, zg, sg, &
                    tpg, qu, su, mc, du, &
		    mu, md, qd, sd, qhat, &
		    shat, dp, qstpg, zfg, ql, &
		    dsubcld, mb, capeg, tlg, lclg, &
		    lelg, jt, maxg, 1, lengath, &
		    rgas, msg, nstep, lat, plonl )

!-----------------------------------------------------------------------
! 	... Limit cloud base mass flux to theoretical upper bound.
!-----------------------------------------------------------------------
      do i = 1,lengath
        mumax(i) = 0
      end do
      do k = msg + 2,plev
        do i = 1,lengath
          mumax(i) = MAX( mumax(i),mu(i,k)/dp(i,k) )
        end do
      end do
      do i = 1,lengath
        if( mumax(i) > 0. ) then
          mb(i) = MIN( mb(i),.5/(delt*mumax(i)) )
        else
          mb(i) = 0.
        end if
      end do
      do k = msg+1,plev
        do i = 1,lengath
          mu(i,k) = mu(i,k)*mb(i)
          md(i,k) = md(i,k)*mb(i)
          mc(i,k) = mc(i,k)*mb(i)
          du(i,k) = du(i,k)*mb(i)
          eu(i,k) = eu(i,k)*mb(i)
          ed(i,k) = ed(i,k)*mb(i)
          cmeg(i,k) = cmeg(i,k)*mb(i)
          cmfdqrg(i,k) = cmfdqrg(i,k)*mb(i)
          cug(i,k) = cug(i,k)*mb(i)
          evpg(i,k) = evpg(i,k)*mb(i)
          pflxg(i,k+1) = pflxg(i,k+1)*mb(i)*100.*rgrav
          mu2(i,k) = mu2(i,k)*mb(i)
          md2(i,k) = md2(i,k)*mb(i)
          du2(i,k) = du2(i,k)*mb(i)
          eu2(i,k) = eu2(i,k)*mb(i)
          ed2(i,k) = ed2(i,k)*mb(i)
        end do
      end do
      do i = 1,lengath
         betau(i) = betau(i)*mb(i)
         betad(i) = betad(i)*mb(i)
!-----------------------------------------------------------------------
! 	... totpcp from cldprp has the dimension of kg/kg, here it is 
!           converted to kg/(m^2*s), the precipitation rate
!-----------------------------------------------------------------------
         totpcp(i) = totpcp(i)*mb(i)*100.*rgrav
         totevp(i) = totevp(i)*mb(i)*100.*rgrav
      end do

!-----------------------------------------------------------------------
! 	... Compute temperature and moisture changes due to convection.
!-----------------------------------------------------------------------
      call Q1Q2_PJR( dqdt, dsdt, qu, su, du, &
                     qhat, shat, dp, mu, md, &
                     sd, qd, ql, dsubcld, jt, &
		     maxg, twodelt ,1, lengath, msg, &
		     nstep, lat, dlg, evpg, cug, plonl )

!-----------------------------------------------------------------------
! 	... Compute momentum changes due to convection, if desired (i.e
!           if logical switch set).
!-----------------------------------------------------------------------
      if( ncnst > 0 ) then
         call CONVTRAN( cnst, ncnst, mu2, md2, du2, &
			eu2, ed2, dp, dsubcld, jt, &
			maxg, ideep, 1, lengath, nstep, &
			lat, ip, delt, plonl )
      end if

!-----------------------------------------------------------------------
! 	... Gather back temperature and mixing ratio.
!-----------------------------------------------------------------------
!     write(*,*) 'CONV_CCM: lengath, twodelt @ lat,ip = ',lat,ip
!     write(*,*) lengath, twodelt
!     write(*,*) ideep(:lengath)
!     write(*,*) 'CONV_CCM: dqdt'
!     write(*,*) dqdt(1:lengath,12)
!     write(*,*) 'CONV_CCM: dsdt'
!     write(*,*) dsdt(1:lengath,12)
!     write(*,*) 'CONV_CCM: dudt'
!     write(*,*) dudt(1:lengath,12)
!     write(*,*) 'CONV_CCM: dvdt'
!     write(*,*) dvdt(1:lengath,12)
!     write(*,*) 'CONV_CCM: q'
!     write(*,*) q((/ideep(1:lengath)/),12)
      do k = msg + 1,plev
         do i = 1,lengath
!    write(*,*) 'CONV_CCM: Before gather op for k,i,lat,ip = ',k,i,lat,ip
	    ind = ideep(i)
!-----------------------------------------------------------------------
! 	... q is updated to compute net precip, and then reset to old value.
!           the last line is overwritten. so the input basic variables, i.e.
!           q, t, u and v are updated to include convective increments. 
!-----------------------------------------------------------------------
            q(ind,k) = q(ind,k) + twodelt*dqdt(i,k)
            t(ind,k) = t(ind,k) + twodelt*dsdt(i,k)
            cme(ind,k)    = cmeg(i,k)
            cmfdqr(ind,k) = cmfdqrg(i,k)
            zdu(ind,k)    = du2(i,k)
!-----------------------------------------------------------------------
! 	... Convert mass flux from mb/s to kg/m^2/s
!-----------------------------------------------------------------------
            zmu(ind,k)  = mu2(i,k) * 100.*rgrav
            zmd(ind,k)  = md2(i,k) * 100.*rgrav
            zeu(ind,k)  = eu2(i,k)
            mcon(ind,k) = mc(i,k)
            qtg(ind,k)  = dqdt(i,k)
            ttg(ind,k)  = dsdt(i,k)
            dlf(ind,k)  = dlg(i,k)
            pflx(ind,k) = pflxg(i,k)
!    write(*,*) 'CONV_CCM: After gather op for k,i = ',k,i
         end do
      end do

      do i = 1,lengath
         jctop(ideep(i)) = REAL( jt(i) )
         jcbot(ideep(i)) = REAL( maxg(i) )
         pflx(ideep(i),plevp) = pflxg(i,plevp)
         psevap = psevap + totevp(i)
         psrain = psrain + totpcp(i)
      end do

!-----------------------------------------------------------------------
! 	... Convert back to specific humidity from mixing ratio.
!           take into account any moisture added to ensure positiveness
!           of specific humidity at start of routine.
!-----------------------------------------------------------------------
      do k = msg + 1,plev
         do i = 1,plonl
            qh(i,k,1) = q(i,k) - MAX( (qmin - qh0(i,k)),0. )
         end do
      end do
      do k = plev,msg + 1,-1
         do i = 1,plonl
            sumq(i) = sumq(i) - dpp(i,k)* (qh(i,k,1) - qh0(i,k))
!-----------------------------------------------------------------------
! 	... Account for the detraining cloud water in the precip 
!-----------------------------------------------------------------------
            sumq(i) = sumq(i) - dpp(i,k)*dlf(i,k)*twodelt
            pcpck(i,k) = MAX( 0.,sumq(i) )
         end do
      end do

!-----------------------------------------------------------------------
! 	... Obtain final precipitation rate.
!-----------------------------------------------------------------------
      do i = 1,plonl
!-----------------------------------------------------------------------
! 	... Here pcpr and pcps are in units of kg/m^2, ie. precip per time step
!-----------------------------------------------------------------------
         precip = pcpdh(i)*MAX( sumq(i),0. )
         if( ts(i) >= tfreez ) then
           pcpr(i) = precip
           pcps(i) = 0.
         else
           pcpr(i) = 0.
           pcps(i) = precip
         end if
      end do

!-----------------------------------------------------------------------
! 	... Accumulate precipitation, the 1000. is the density of water,
!           paprc and paprs are now in units of meters.
!-----------------------------------------------------------------------
      paprc(:) = paprc(:) + 1.e-3*(pcpr(:) + pcps(:))
      paprs(:) = paprs(:) + 1.e-3*pcps(:)

!-----------------------------------------------------------------------
! 	... Convert precipitation to m/s, ie, precip rate.
!-----------------------------------------------------------------------
      pcpr(:) = 1.e-3 * pcpr(:)*twodelti
      pcps(:) = 1.e-3 * pcps(:)*twodelti
      pcpc(:) = pcpr(:) + pcps(:)
      do k = msg + 1,plev
         pcpck(:,k) = pcpdh(:)*pcpck(:,k)*twodelti
      end do

      end subroutine CONV_CCM


      end module MO_ZHANG
