
      module MO_ZHANG_SUBS

      implicit none

      save

      real, parameter :: c1 = 6.112
      real, parameter :: c2 = 17.67
      real, parameter :: c3 = 243.5
      real, parameter :: cp = 1004.64
      real, parameter :: rl = 2.5104e6
      real, parameter :: eps1 = .622
      real, parameter :: tfreez  = 273.16
      real, parameter :: capelmt = 0.

      CONTAINS

      subroutine CONVTRAN( q, ncnst, mu, md, du, &
			   eu, ed, dp, dsubcld, jt, &
			   mx, ideep, il1g, il2g, nstep, &
			   lat, ip, delt, plonl )
!-----------------------------------------------------------------------
! 	... Convective transport of trace species
!           Note that we are still assuming that the tracers are in
!           a moist mixing ratio this will change soon
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst
      use MO_MPI,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
             ncnst, &          ! number of passive tracers
             il1g, &           ! Gathered min lon indices over which to operate
             il2g, &           ! Gathered max lon indices over which to operate
             lat, &            ! Latitude index
             ip, &             ! lon tile index
             plonl, &          ! lon tile dim
             nstep             ! Time step index
      integer, dimension(plonl), intent(in) :: &
             jt, &             ! Index of cloud top for each column
             mx, &             ! Index of cloud top for each column
             ideep             ! Gathering array
      real, intent(in) :: &
             delt                 ! Time step
      real, dimension(plonl,plev), intent(in) :: &
             mu, &                ! Mass flux up
             md, &                ! Mass flux down
             du, &                ! Mass detraining from updraft
             eu, &                ! Mass entraining from updraft
             ed, &                ! Mass entraining from downdraft
             dp                   ! Delta pressure between interfaces
      real, dimension(plonl), intent(in) :: &
           dsubcld              ! Delta pressure from cloud base to sfc

      real, intent(inout) :: q(plonl,plev,ncnst)  ! Tracer array including moisture

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: small = 1.e-36                ! A small number
      real, parameter :: mbsth = 1.e-15                ! mass flux zero threshold

      integer :: i                 ! Work index
      integer :: k                 ! Work index
      integer :: kbm               ! Highest altitude index of cloud base
      integer :: kk                ! Work index
      integer :: kkp1              ! Work index
      integer :: km1               ! Work index
      integer :: kp1               ! Work index
      integer :: ktm               ! Highest altitude index of cloud top
      integer :: m                 ! Work index

      real :: cabv                 ! Mix ratio of constituent above
      real :: cbel                 ! Mix ratio of constituent below
      real :: cdifr                ! Normalized diff between cabv and cbel
      real :: mupdudp              ! A work variable
      real :: minc                 ! A work variable
      real :: maxc                 ! A work variable
      real :: twodelt              ! A work variable
      real :: fluxin               ! A work variable
      real :: fluxout              ! A work variable
      real :: netflux              ! A work variable
      real, dimension(plonl,plev) :: &
          chat, &              ! Mix ratio in env at interfaces
          cond, &              ! Mix ratio in downdraft at interfaces
          const, &             ! Gathered tracer array 
          conu, &              ! Mix ratio in updraft at interfaces
          dcondt               ! Gathered tend array 

!-----------------------------------------------------------------------
! 	... Find the highest level top and bottom levels of convection
!-----------------------------------------------------------------------
      ktm = plev
      kbm = plev
      do i = il1g,il2g
         ktm = MIN( ktm,jt(i) )
         kbm = MIN( kbm,mx(i) )
      end do

!-----------------------------------------------------------------------
! 	... Loop ever each constituent
!-----------------------------------------------------------------------
species_loop : &
      do m = 1,ncnst
!-----------------------------------------------------------------------
! 	... Gather up the constituent and set tend to zero
!-----------------------------------------------------------------------
         do k = 1,plev
            do i = il1g,il2g
               const(i,k)  = q(ideep(i),k,m)
               dcondt(i,k) = 0.
            end do
         end do

!-----------------------------------------------------------------------
! 	... From now on work only with gathered data
!-----------------------------------------------------------------------
! 	... Interpolate environment tracer values to interfaces
!-----------------------------------------------------------------------
         do k = 1,plev
            km1 = MAX( 1,k-1 )
            do i = il1g,il2g
               minc = MIN( const(i,km1),const(i,k) )
               maxc = MAX( const(i,km1),const(i,k) )
               if( minc < 0 ) then
                  cdifr = 0.
               else
                  cdifr = ABS( const(i,k) - const(i,km1) )/MAX( maxc,small )
               end if
!-----------------------------------------------------------------------
! 	... If the two layers differ significantly use a geometric averaging procedure
!-----------------------------------------------------------------------
               if( cdifr > 1.e-6 ) then
                  cabv = MAX( const(i,km1),maxc*1.e-12 )
                  cbel = MAX( const(i,k),maxc*1.e-12 )
                  chat(i,k) = LOG( cabv/cbel )/(cabv - cbel)*cabv*cbel
               else             ! Small diff, so just arithmetic mean
                  chat(i,k) = .5*(const(i,k) + const(i,km1))
               end if
!-----------------------------------------------------------------------
! 	... Provisional up and down draft values
!-----------------------------------------------------------------------
               conu(i,k) = chat(i,k)
               cond(i,k) = chat(i,k)
            end do
         end do

#ifdef CHECKNEG
         do k = 1,plev
            km1 = MAX( 1,k-1 )
            do i = il1g,il2g
               if( chat(i,k) < 0. ) then
                  write(*,*) 'CONVTRAN: negative chat ', i, k, lat, chat(i,k), const(i,km1), const(i,k)
                  call ENDRUN
               end if
            end do
         end do
#endif


!-----------------------------------------------------------------------
! 	... Do levels adjacent to top and bottom
!-----------------------------------------------------------------------
         k = 2
         km1 = 1
         kk = plev 
         do i = il1g,il2g
            mupdudp = mu(i,kk) + du(i,kk)*dp(i,kk)
            if( mupdudp > mbsth ) then
               conu(i,kk) = (eu(i,kk)*const(i,kk)*dp(i,kk))/mupdudp
            end if
            if( md(i,k) < -mbsth ) then
               cond(i,k) = (-ed(i,km1)*const(i,km1)*dp(i,km1))/md(i,k)
            end if
         end do

!-----------------------------------------------------------------------
! 	... Updraft from bottom to top
!-----------------------------------------------------------------------
         do kk = plev-1,1,-1
            kkp1 = MIN( plev,kk+1 )
            do i = il1g,il2g
               mupdudp = mu(i,kk) + du(i,kk)*dp(i,kk)
               if( mupdudp > mbsth ) then
                  conu(i,kk) = (mu(i,kkp1)*conu(i,kkp1) + eu(i,kk)*const(i,kk)*dp(i,kk))/mupdudp
               end if
            end do
         end do

!-----------------------------------------------------------------------
! 	... Downdraft from top to bottom
!-----------------------------------------------------------------------
         do k = 3,plev
            km1 = MAX( 1,k-1 )
            do i = il1g,il2g
               if( md(i,k) < -mbsth ) then
                  cond(i,k) = (md(i,km1)*cond(i,km1) - ed(i,km1)*const(i,km1)*dp(i,km1))/md(i,k)
               end if
            end do
         end do

         do i = il1g,il2g
            do k = jt(i),mx(i)-1
               km1 = MAX( 1,k-1 )
               kp1 = MIN( plev,k+1 )
!-----------------------------------------------------------------------
! 	... Version 3 limit fluxes outside convection to mass in appropriate layer
!           these limiters are probably only safe for positive definite quantitities
!           it assumes that mu and md already satify a courant number limit of 1
!-----------------------------------------------------------------------
               fluxin = mu(i,kp1)*conu(i,kp1)  &
                        + mu(i,k)*MIN( chat(i,k),const(i,km1) )  &
                        -(md(i,k)*cond(i,k) + md(i,kp1)*MIN( chat(i,kp1),const(i,kp1) ))
               fluxout = mu(i,k)*conu(i,k) + mu(i,kp1)*MIN( chat(i,kp1),const(i,k) ) &
                         -(md(i,kp1)*cond(i,kp1) + md(i,k)*MIN( chat(i,k),const(i,k) ))
               netflux = fluxin - fluxout
               if( ABS(netflux) >= MAX( fluxin,fluxout )*1.e-12 ) then
                  dcondt(i,k) = netflux/dp(i,k)
               end if
            end do
         end do

         do i = il1g,il2g
	    k = mx(i)
            km1 = MAX( 1,k-1 )
!-----------------------------------------------------------------------
! 	... Version 3
!-----------------------------------------------------------------------
            fluxin  = mu(i,k)*MIN( chat(i,k),const(i,km1) ) - md(i,k)*cond(i,k)
            fluxout = mu(i,k)*conu(i,k) - md(i,k)*MIN( chat(i,k),const(i,k) )
            netflux = fluxin - fluxout
            if( ABS(netflux) >= MAX( fluxin,fluxout )*1.e-12 ) then
               dcondt(i,k) = netflux/dp(i,k)
            end if
         end do

!-----------------------------------------------------------------------
! 	... Update and scatter data back to full arrays
!-----------------------------------------------------------------------
	 twodelt = 2.*delt
         do k = 1,plev
            do i = il1g,il2g
               q(ideep(i),k,m) = const(i,k) + dcondt(i,k)*twodelt
            end do
         end do
      end do species_loop

      end subroutine CONVTRAN

      subroutine BUOYAN( q, t, p, z, pf, &
                         tp, qstp, tl, cape, pblt, &
			 lcl, lel, lon, mx, rd, &
			 msg, nstep, lat, tpert, qpert, plonl )
!-----------------------------------------------------------------------
! 	... This is contributed code not fully standardized by the CCM core group.
!
! the documentation has been enhanced to the degree that we are able
! jul 14/92 - guang jun zhang, m.lazare, n.mcfarlane.  as in
!             previous version buoyan except remove pathalogical
!             cases of "zig-zags" in profiles where lel defined
!             far too high (by use of lelten array, which assumes
!             a maximum of five such crossing points).
! feb 18/91 - guang jun zhang, m.lazare, n.mcfarlane.  previous
!             version buoyan.
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst
      use MO_HACK, only : grav

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: nstep
      integer, intent(in) :: lat
      integer, intent(in) :: plonl
      integer, dimension(plonl), intent(out) :: &
             lcl, &            ! 
             lel, &            ! 
             lon, &            ! level of onset of deep convection
             mx                ! level of max moist static energy
      real, dimension(plonl), intent(in) :: &
          pblt, &              ! index of pbl depth
          tpert, &             ! perturbation temperature by pbl processes
          qpert                ! perturbation moisture by pbl processes
      real, dimension(plonl,plev), intent(in) :: &
          q, &                 ! spec. humidity
          t, &                 ! temperature
          p, &                 ! pressure
          z                    ! height
      real, dimension(plonl,plevp), intent(in) :: &
          pf                  ! pressure at interfaces

      real, dimension(plonl), intent(out) :: &
          tl, &                ! parcel temperature at lcl
          cape                 ! convective aval. pot. energy.
      real, dimension(plonl,plev), intent(out) :: &
          tp, &                ! parcel temperature
          qstp                 ! saturation mixing ratio of parcel

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: i
      integer :: k
      integer :: msg
      integer :: n
      integer :: ind
      integer :: knt(plonl) 
      integer :: lelten(plonl,5)

      real :: e
      real :: rd
      real :: capeten(plonl,5)     ! provisional value of cape
      real :: tv(plonl,plev)       ! 
      real :: tpv(plonl,plev)      ! 
      real :: buoy(plonl,plev)

      real :: a1(plonl) 
      real :: a2(plonl) 
      real :: estp(plonl) 
      real :: pl(plonl) 
      real :: plexp(plonl) 
      real :: hmax(plonl) 
      real :: hmn(plonl) 
      real :: y(plonl)

      logical :: plge600(plonl) 

      tp(:,:)   = 0.
      qstp(:,:) = 0.
      do n = 1,5
        do i = 1,plonl
          lelten(i,n) = plev
          capeten(i,n) = 0.
        end do
      end do

      do i = 1,plonl
        lon(i) = plev
        knt(i) = 0
        lel(i) = plev
        mx(i) = lon(i)
        cape(i) = 0.
        hmax(i) = 0.
      end do

!-----------------------------------------------------------------------
! 	... Set "launching" level(mx) to be at maximum moist static energy.
!           search for this level stops at planetary boundary layer top.
!-----------------------------------------------------------------------
      do k = plev,msg + 1,-1
        do i = 1,plonl
          hmn(i) = cp*t(i,k) + grav*z(i,k) + rl*q(i,k)
          if( k >= NINT(pblt(i)) .and. k <= lon(i) .and. hmn(i) > hmax(i) ) then
            hmax(i) = hmn(i)
            mx(i) = k
          end if
        end do
      end do

      do i = 1,plonl
        ind = mx(i)
        lcl(i) = ind
        e = p(i,ind)*q(i,ind)/ (eps1 + q(i,ind))
        tl(i) = 2840./(3.5*LOG( t(i,ind) ) - LOG( e ) - 4.805) + 55.
        if( tl(i) < t(i,ind) ) then
          plexp(i) = 1./(.2854* (1. - .28*q(i,ind)))
          pl(i) = p(i,ind)* (tl(i)/t(i,ind))**plexp(i)
        else
          tl(i) = t(i,ind)
          pl(i) = p(i,ind)
        end if
      end do

!-----------------------------------------------------------------------
! 	... Calculate lifting condensation level (lcl).
!-----------------------------------------------------------------------
      do k = plev,msg + 2,-1
        do i = 1,plonl
          if( k <= mx(i) .and. p(i,k) > pl(i) .and. p(i,k-1) <= pl(i) ) then
            lcl(i) = k - 1
          end if
        end do
      end do

!-----------------------------------------------------------------------
! 	... If lcl is above the nominal level of non-divergence (600 mbs),
!           no deep convection is permitted (ensuing calculations
!           skipped and cape retains initialized value of zero).
!-----------------------------------------------------------------------
      plge600(:plonl) = (pl(:plonl) >= 600. .and. t(:plonl,plev) > tfreez)

!-----------------------------------------------------------------------
! 	... Initialize parcel properties in sub-cloud layer below lcl.
!-----------------------------------------------------------------------
      do k = plev,msg + 1,-1
        do i = 1,plonl
          if( k > lcl(i) .and. k <= mx(i) .and. plge600(i) ) then
	    ind = mx(i)
            tv(i,k) = t(i,k)* (1. + 1.608*q(i,k))/ (1. + q(i,k))
            qstp(i,k) = q(i,ind)
            tp(i,k) = t(i,ind)*(p(i,k)/p(i,ind))**(.2854*(1. - .28*q(i,ind)))
!-----------------------------------------------------------------------
! 	... Buoyancy is increased by 0.5 k as in tiedtke
!-----------------------------------------------------------------------
            tpv(i,k) = (tp(i,k) + tpert(i))*(1. + 1.608*q(i,ind))/(1. + q(i,ind))
            buoy(i,k) = tpv(i,k) - tv(i,k) + 0.5
          end if
        end do
      end do

!-----------------------------------------------------------------------
! 	... Define parcel properties at lcl (i.e. level immediately above pl).
!-----------------------------------------------------------------------
      do k = plev,msg + 1,-1
        do i = 1,plonl
          if( k == lcl(i) .and. plge600(i) ) then
            tv(i,k) = t(i,k)* (1. + 1.608*q(i,k))/(1. + q(i,k))
            qstp(i,k) = q(i,mx(i))
            tp(i,k) = tl(i)*(p(i,k)/pl(i))**(.2854* (1. - .28*qstp(i,k)))
!-----------------------------------------------------------------------
! 	... Use of different formulas for est has about 1 g/kg difference
!           in qs at t= 300k, and 0.02 g/kg at t=263k, with the formula
!           above giving larger qs.
!-----------------------------------------------------------------------
            estp(i) = c1*EXP( (c2*(tp(i,k) - tfreez))/((tp(i,k) - tfreez) + c3) )
            qstp(i,k) = eps1*estp(i)/(p(i,k) - estp(i))
            a1(i) = cp/rl + qstp(i,k)*(1. + qstp(i,k)/eps1)*rl*eps1/(rd*tp(i,k)**2)
            a2(i) = .5* (qstp(i,k)*(1. + 2./eps1*qstp(i,k))* &
                    (1. + qstp(i,k)/eps1)*eps1**2*rl*rl/ &
                    (rd**2*tp(i,k)**4) - qstp(i,k)* &
                    (1. + qstp(i,k)/eps1)*2.*eps1*rl/ &
                    (rd*tp(i,k)**3))
            a1(i) = 1./a1(i)
            a2(i) = -a2(i)*a1(i)**3
            y(i) = q(i,mx(i)) - qstp(i,k)
            tp(i,k) = tp(i,k) + a1(i)*y(i) + a2(i)*y(i)**2
            estp(i) = c1*EXP( (c2*(tp(i,k) - tfreez))/((tp(i,k) - tfreez) + c3) )
            qstp(i,k) = eps1*estp(i)/ (p(i,k) - estp(i))
!-----------------------------------------------------------------------
! 	... Buoyancy is increased by 0.5 k in cape calculation.
!-----------------------------------------------------------------------
            tpv(i,k) = (tp(i,k) + tpert(i))* (1. + 1.608*qstp(i,k))/(1. + q(i,mx(i)))
            buoy(i,k) = tpv(i,k) - tv(i,k) + 0.5
          end if
        end do
      end do

!-----------------------------------------------------------------------
! 	... Main buoyancy calculation.
!-----------------------------------------------------------------------
      do k = plev - 1,msg + 1,-1
        do i = 1,plonl
          if( k < lcl(i) .and. plge600(i) ) then
            tv(i,k) = t(i,k)* (1. + 1.608*q(i,k))/(1. + q(i,k))
            qstp(i,k) = qstp(i,k+1)
            tp(i,k) = tp(i,k+1)* (p(i,k)/p(i,k+1))**(.2854* (1. - .28*qstp(i,k)))
            estp(i) = c1*EXP( (c2*(tp(i,k) - tfreez))/((tp(i,k) - tfreez) + c3) )
            qstp(i,k) = eps1*estp(i)/ (p(i,k) - estp(i))
            a1(i) = cp/rl + qstp(i,k)* (1.+qstp(i,k)/eps1)*rl*eps1/(rd*tp(i,k)**2)
            a2(i) = .5* (qstp(i,k)* (1.+2./eps1*qstp(i,k))* &
                    (1.+qstp(i,k)/eps1)*eps1**2*rl*rl/ &
                    (rd**2*tp(i,k)**4)-qstp(i,k)* &
                    (1.+qstp(i,k)/eps1)*2.*eps1*rl/ &
                    (rd*tp(i,k)**3))
            a1(i) = 1./a1(i)
            a2(i) = -a2(i)*a1(i)**3
            y(i) = qstp(i,k+1) - qstp(i,k)
            tp(i,k) = tp(i,k) + a1(i)*y(i) + a2(i)*y(i)**2
            estp(i) = c1*EXP( (c2*(tp(i,k) - tfreez))/((tp(i,k) - tfreez)+c3) )
            qstp(i,k) = eps1*estp(i)/ (p(i,k) - estp(i))
            tpv(i,k) = (tp(i,k)+tpert(i))* (1. + 1.608*qstp(i,k))/(1. + q(i,mx(i)))
            buoy(i,k) = tpv(i,k) - tv(i,k) + 0.5
          end if
        end do
      end do

      do k = msg + 2,plev
        do i = 1,plonl
          if( k < lcl(i) .and. plge600(i) ) then
            if( buoy(i,k+1) > 0. .and. buoy(i,k) <= 0. ) then
              knt(i) = MIN( knt(i) + 1,5 )
              lelten(i,knt(i)) = k
            end if
          end if
        end do
      end do

!-----------------------------------------------------------------------
! 	... Calculate convective available potential energy (cape).
!-----------------------------------------------------------------------
      do n = 1,5
        do k = msg + 1,plev
          do i = 1,plonl
            if( plge600(i) .and. k <= mx(i) .and. k > lelten(i,n) ) then
              capeten(i,n) = capeten(i,n) + rd*buoy(i,k)*LOG( pf(i,k+1)/pf(i,k) )
            end if
          end do
        end do
      end do

!-----------------------------------------------------------------------
! 	... Find maximum cape from all possible tentative capes from one sounding
!-----------------------------------------------------------------------
      do n = 1,5
        do i = 1,plonl
          if( capeten(i,n) > cape(i) ) then
            cape(i) = capeten(i,n)
            lel(i) = lelten(i,n)
          end if
        end do
      end do

!-----------------------------------------------------------------------
! 	... Put lower bound on cape for diagnostic purposes.
!-----------------------------------------------------------------------
      do i = 1,plonl
        cape(i) = MAX( cape(i), 0. )
      end do
      
      end subroutine BUOYAN

      subroutine CLOSURE( q, t, p, z, s, &
                          tp, qu, su, mc, du, &
			  mu, md, qd, sd, qhat, &
			  shat, dp, qstp, zf, ql, &
			  dsubcld, mb, cape, tl, lcl, &
			  lel, jt, mx, il1g, il2g, &
			  rd, msg, nstep, lat, plonl )
!-----------------------------------------------------------------------
! 	... This is contributed code not fully standardized by the CCM core group.
!
! this code is very much rougher than virtually anything else in the CCM
! We expect to release cleaner code in a future release
!
! the documentation has been enhanced to the degree that we are able
! may 09/91 - guang jun zhang, m.lazare, n.mcfarlane.
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst
      use MO_HACK, only : grav

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: nstep
      integer, intent(in) :: lat
      integer, intent(in) :: plonl
      integer, intent(in) :: msg
      integer, intent(in) :: il1g
      integer, intent(in) :: il2g
      integer, intent(in) :: lcl(plonl)        ! index of lcl
      integer, intent(in) :: lel(plonl)        ! index of launch leve
      integer, intent(in) :: jt(plonl)         ! top of updraft
      integer, intent(in) :: mx(plonl)         ! base of updraft
      real, intent(in)    :: rd
      real, intent(in) ::  &
             q(plonl,plev), &     ! spec humidity
             t(plonl,plev), &     ! temperature
             p(plonl,plev), &     ! pressure (mb)
             z(plonl,plev), &     ! height (m)
             s(plonl,plev), &     ! normalized dry static energy 
             tp(plonl,plev), &    ! parcel temp
             qu(plonl,plev), &    ! updraft spec. humidity
             su(plonl,plev), &    ! normalized dry stat energy of updraft
             mc(plonl,plev), &    ! net convective mass flux 
             du(plonl,plev), &    ! detrainment from updraft
             mu(plonl,plev), &    ! mass flux of updraft
             md(plonl,plev), &    ! mass flux of downdraft
             qd(plonl,plev), &    ! spec. humidity of downdraft
             sd(plonl,plev), &    ! dry static energy of downdraft
             qhat(plonl,plev), &  ! environment spec humidity at interfaces
             shat(plonl,plev), &  ! env. normalized dry static energy at intrfcs
             dp(plonl,plev), &    ! pressure thickness of layers
             qstp(plonl,plev), &  ! spec humidity of parcel
             zf(plonl,plevp), &   ! height of interface levels
             ql(plonl,plev), &    ! liquid water mixing ratio
             dsubcld(plonl), &    ! thickness of subcloud layer
             cape(plonl), &       ! available pot. energy of column
             tl(plonl)

      real, intent(inout) :: mb(plonl)            ! cloud base mass flux

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: i
      integer :: k

      real :: beta
      real :: dadt
      real :: debdt
      real :: dltaa
      real :: eb
      real :: dtpdt(plonl,plev)
      real :: dqsdtp(plonl,plev)
      real :: dtmdt(plonl,plev)
      real :: dqmdt(plonl,plev)
      real :: dboydt(plonl,plev)
      real :: thetavp(plonl,plev)
      real :: thetavm(plonl,plev)
      real :: dtbdt(plonl), dqbdt(plonl), dtldt(plonl)

!-----------------------------------------------------------------------
! 	... tau=4800. were used in canadian climate center. however, when it
!           is used here in echam3 t42, convection is too weak, thus 
!           adjusted to 2400. i.e the e-folding time is 1 hour now.
!-----------------------------------------------------------------------
      real, save :: tau = 7200.

!-----------------------------------------------------------------------
! 	... Change of subcloud layer properties due to convection is
!           related to cumulus updrafts and downdrafts.
!           mc(z)=f(z)*mb, mub=betau*mb, mdb=betad*mb are used
!           to define betau, betad and f(z).
!           note that this implies all time derivatives are in effect
!           time derivatives per unit cloud-base mass flux, i.e. they
!           have units of 1/mb instead of 1/sec.
!-----------------------------------------------------------------------
      do i = il1g,il2g
         mb(i) = 0.
         eb = p(i,mx(i))*q(i,mx(i))/ (eps1 + q(i,mx(i)))
         dtbdt(i) = (1./dsubcld(i))*(mu(i,mx(i))*(shat(i,mx(i)) - su(i,mx(i))) &
                                     + md(i,mx(i))* (shat(i,mx(i)) - sd(i,mx(i))))
         dqbdt(i) = (1./dsubcld(i))*(mu(i,mx(i))*(qhat(i,mx(i)) - qu(i,mx(i))) &
                                     + md(i,mx(i))*(qhat(i,mx(i)) - qd(i,mx(i))))
         debdt = eps1*p(i,mx(i))/(eps1+q(i,mx(i)))**2*dqbdt(i)
         dtldt(i) = -2840.*(3.5/t(i,mx(i))*dtbdt(i) - debdt/eb)/(3.5*LOG(t(i,mx(i))) - LOG(eb) - 4.805)**2
      end do

!-----------------------------------------------------------------------
!   	... Dtmdt and dqmdt are cumulus heating and drying.
!-----------------------------------------------------------------------
      do k = msg + 1,plev
         do i = il1g,il2g
            dtmdt(i,k) = 0.
            dqmdt(i,k) = 0.
         end do
      end do

      do k = msg + 1,plev - 1
         do i = il1g,il2g
            if( k == jt(i) ) then
               dtmdt(i,k) = (1./dp(i,k)) &
                             *(mu(i,k+1)*(su(i,k+1) - shat(i,k+1) - rl/cp*ql(i,k+1)) &
                               + md(i,k+1)*(sd(i,k+1) - shat(i,k+1)))
               dqmdt(i,k) = (1./dp(i,k))*(mu(i,k+1)*(qu(i,k+1) - qhat(i,k+1)+ql(i,k+1)) &
                                          + md(i,k+1)*(qd(i,k+1) - qhat(i,k+1)))
            end if
         end do
      end do

      beta = 0.
      do k = msg + 1,plev - 1
         do i = il1g,il2g
            if( k > jt(i) .and. k < mx(i) ) then
               dtmdt(i,k) = (mc(i,k)*(shat(i,k) - s(i,k)) &
                             + mc(i,k+1)*(s(i,k) - shat(i,k+1)))/dp(i,k) &
                            - rl/cp*du(i,k)*(beta*ql(i,k) + (1. - beta)*ql(i,k+1))
               dqmdt(i,k) = (mu(i,k+1)*(qu(i,k+1) - qhat(i,k+1) + cp/rl*(su(i,k+1) - s(i,k))) &
                             - mu(i,k)*(qu(i,k) - qhat(i,k) + cp/rl*(su(i,k) - s(i,k))) &
                             + md(i,k+1)*(qd(i,k+1) - qhat(i,k+1) + cp/rl*(sd(i,k+1) - s(i,k))) &
                             - md(i,k)*(qd(i,k) - qhat(i,k) + cp/rl*(sd(i,k) - s(i,k))))/dp(i,k) &
                             + du(i,k)*(beta*ql(i,k) + (1. - beta)*ql(i,k+1))
            end if
         end do
      end do

      do k = msg + 1,plev
         do i = il1g,il2g
            if( k >= lel(i) .and. k <= lcl(i) ) then
               thetavp(i,k) = tp(i,k)* (1000./p(i,k))** (rd/cp)*(1. + 1.608*qstp(i,k) - q(i,mx(i)))
               thetavm(i,k) = t(i,k)* (1000./p(i,k))** (rd/cp)*(1. + .608*q(i,k))
               dqsdtp(i,k) = qstp(i,k)* (1.+qstp(i,k)/eps1)*eps1*rl/(rd*tp(i,k)**2)
!-----------------------------------------------------------------------
! 	... dtpdt is the parcel temperature change due to change of
!           subcloud layer properties during convection.
!-----------------------------------------------------------------------
               dtpdt(i,k) = tp(i,k)/(1.+ rl/cp*(dqsdtp(i,k) - qstp(i,k)/tp(i,k))) &
                            *(dtbdt(i)/t(i,mx(i)) + rl/cp*(dqbdt(i)/tl(i) - q(i,mx(i))/tl(i)**2*dtldt(i)))
!-----------------------------------------------------------------------
! 	... dboydt is the integrand of cape change.
!-----------------------------------------------------------------------
               dboydt(i,k) = ((dtpdt(i,k)/tp(i,k) &
                               + 1./(1. + 1.608*qstp(i,k) - q(i,mx(i)))*(1.608*dqsdtp(i,k)*dtpdt(i,k) - dqbdt(i))) &
                               - (dtmdt(i,k)/t(i,k) + .608/(1. + .608*q(i,k))*dqmdt(i,k))) &
                             *grav*thetavp(i,k)/thetavm(i,k)
            end if
         end do
      end do

      do k = msg + 1,plev
         do i = il1g,il2g
            if( k > lcl(i) .and. k < mx(i) ) then
               thetavp(i,k) = tp(i,k)* (1000./p(i,k))** (rd/cp)*(1. + .608*q(i,mx(i)))
               thetavm(i,k) = t(i,k)* (1000./p(i,k))** (rd/cp)*(1. + .608*q(i,k))
!-----------------------------------------------------------------------
! 	... dboydt is the integrand of cape change.
!-----------------------------------------------------------------------
               dboydt(i,k) = (dtbdt(i)/t(i,mx(i)) &
                              + .608/(1. + .608*q(i,mx(i)))*dqbdt(i) &
                              - dtmdt(i,k)/t(i,k) - .608/(1. + .608*q(i,k))*dqmdt(i,k)) &
                             *grav*thetavp(i,k)/thetavm(i,k)
            end if
         end do
      end do

!-----------------------------------------------------------------------
! 	... Buoyant energy change is set to 2/3*excess cape per 3 hours
!-----------------------------------------------------------------------
      do i = il1g,il2g
         dadt = 0.
         do k = lel(i),mx(i) - 1
            dadt = dadt + dboydt(i,k)* (zf(i,k) - zf(i,k+1))
         end do
         dltaa = capelmt - cape(i)
         if( dadt /= 0. ) then
	    mb(i) = MAX( dltaa/(tau*dadt),0. )
	 end if
      end do

      end subroutine CLOSURE

      subroutine Q1Q2_PJR( dqdt, dsdt, qu, su, du, &
                           qhat, shat, dp, mu, md, &
                           sd, qd, ql, dsubcld, jt, &
			   mx, dt, il1g, il2g, msg, &
			   nstep, lat, dl, evp, cu, plonl )
!-----------------------------------------------------------------------
! 	... Rewritten by phil rasch dec 19 1995
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lat, msg, nstep
      integer, intent(in) :: plonl
      integer, intent(in) :: il1g, il2g
      integer, intent(in) :: jt(plonl), &
                             mx(plonl)

      real, intent(in) :: dt
      real, intent(in) :: &
          qu(plonl,plev), &
          su(plonl,plev), &
          du(plonl,plev), &
          qhat(plonl,plev), &
          shat(plonl,plev), &
          dp(plonl,plev), &
          mu(plonl,plev), &
          md(plonl,plev), &
          sd(plonl,plev), &
          qd(plonl,plev), &
          ql(plonl,plev), &
          evp(plonl,plev), &
          cu(plonl,plev), &
          dsubcld(plonl)
      real, intent(inout) :: dqdt(plonl,plev), &
                             dsdt(plonl,plev), &
                             dl(plonl,plev)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i
      integer :: k
      integer :: kbm
      integer :: ktm
      real :: fact
      real :: emc

      do k = msg + 1,plev
         do i = il1g,il2g
            dsdt(i,k) = 0.
            dqdt(i,k) = 0.
            dl(i,k) = 0.
         end do
      end do

!-----------------------------------------------------------------------
! 	... Find the highest level top and bottom levels of convection
!-----------------------------------------------------------------------
      ktm = plev
      kbm = plev
      do i = il1g,il2g
         ktm = MIN( ktm,jt(i) )
         kbm = MIN( kbm,mx(i) )
      end do

      fact = 0.

      do k = ktm,plev-1
         do i = il1g,il2g
            emc = fact*du(i,k)*ql(i,k+1) & ! evaporating cloud detraining to env
                  - cu(i,k)  &             ! condensation in updraft
                  + evp(i,k)               ! evaporating rain in downdraft
            dsdt(i,k) = -rl/cp*emc  &
                        + (mu(i,k+1)*(su(i,k+1) - shat(i,k+1)) &
                           - mu(i,k)*(su(i,k) - shat(i,k)) &
                           + md(i,k+1)*(sd(i,k+1) - shat(i,k+1)) &
                           - md(i,k)*(sd(i,k) - shat(i,k)))/dp(i,k)
            dqdt(i,k) = emc &
                        + (mu(i,k+1)*(qu(i,k+1) - qhat(i,k+1)) &
                           - mu(i,k)*(qu(i,k) - qhat(i,k)) &
                           + md(i,k+1)*(qd(i,k+1) - qhat(i,k+1)) &
                           - md(i,k)*(qd(i,k) - qhat(i,k)))/dp(i,k)
            dl(i,k) = (1. - fact)*du(i,k)*ql(i,k+1)
         end do
      end do

      do k = kbm,plev             
         do i = il1g,il2g
            if( k == mx(i) ) then
               dsdt(i,k) = -(1./dsubcld(i))*(mu(i,k)*(su(i,k) - shat(i,k)) &
                                             + md(i,k)*(sd(i,k) - shat(i,k)))
               dqdt(i,k) = -(1./dsubcld(i))*(mu(i,k)*(qu(i,k) - qhat(i,k)) &
                                             + md(i,k)*(qd(i,k) - qhat(i,k)))
            else if( k > mx(i) ) then
               dsdt(i,k) = dsdt(i,k-1)
               dqdt(i,k) = dqdt(i,k-1)
            end if
         end do
      end do

      end subroutine Q1Q2_PJR

      subroutine ARCONVTRAN( nstep, lat, delt, dp, mu, &
			     md, eu, ncnst, q, plonl )
!-----------------------------------------------------------------------
! 	... This is a setup routine for convective transport using archived mass
!           fluxes from the Zhang scheme.  The setup involves:
!           1. Gather mass flux arrays.
!           2. Calc the mass fluxes that are determined by mass balance.
!           3. Determine top and bottom of convection.
!           4. Call convtran for the actual convective transport calculation.
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst
      use MO_HACK, only : gravit => grav

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
       nstep, &          ! time step index
       lat, &            ! latitude index
       plonl, &          ! lon tile dim
       ncnst             ! number of passive tracers

      real, intent(in) :: &
       delt, &              ! time step (s)
       dp(plonl,plev), &    ! delta pressure between interfaces (Pa)
       mu(plonl,plev), &    ! mass flux up (kg/m2/s)
       md(plonl,plev), &    ! mass flux down (kg/m2/s)
       eu(plonl,plev)       ! mass entraining from updraft (1/s)

      real, intent(inout) :: q(plonl,plev,ncnst)  ! passive tracers (kg tracer/kg moist air)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        i, k, &
       ideep(plonl), lengath, &
       index(plonl), lenpos, &
       jt(plonl), &             ! index of cloud top for each column (gathered)
       mx(plonl)                ! index of cloud bot for each column (gathered)

      real :: &
       sum(plonl), &
       dpg(plonl,plev), &       ! gathered .01*dp
       rdpg(plonl,plev), &      ! gathered 1./(.01*dp)
       mug(plonl,plev), &       ! gathered mu
       mdg(plonl,plev), &       ! gathered md
       eug(plonl,plev), &       ! gathered eu
       dug(plonl,plev), &       ! mass detraining from updraft (gathered)
       edg(plonl,plev), &       ! mass entraining from downdraft (gathered)
       dsubcld(plonl)           ! delta pressure from cloud base to sfc (gathered)

!-----------------------------------------------------------------------
!     	... Gathered array contains all columns with a updraft.
!-----------------------------------------------------------------------
      sum(1:plonl) = 0.
      do k = 1,plev
         do i = 1,plonl
            sum(i) = sum(i) + mu(i,k)
         end do
      end do

      lengath = COUNT( sum(:plonl) > 0. )
      if( lengath == 0 ) then
         return
      else
	 ideep(:plonl) = (/ (i,i=1,plonl) /)
	 ideep(:plonl) = PACK( ideep(:plonl), mask = sum(:plonl) > 0. )
      end if

!-----------------------------------------------------------------------
!     	... Gather input mass fluxes.
!-----------------------------------------------------------------------
      do k = 1,plev
         do i = 1,lengath
            dpg(i,k) = .01*dp(ideep(i),k)         ! convert Pa -> mb
            rdpg(i,k) = 1./dpg(i,k)
            mug(i,k) = mu(ideep(i),k)*gravit*.01  ! convert kg/m2/s -> mb/s
            mdg(i,k) = md(ideep(i),k)*gravit*.01
            eug(i,k) = eu(ideep(i),k)
         end do
      end do

!-----------------------------------------------------------------------
!     	... Calc du and ed.
!-----------------------------------------------------------------------
      do k = 1, plev-1
         do i = 1, lengath
            dug(i,k) = eug(i,k) - (mug(i,k)-mug(i,k+1))*rdpg(i,k)
            edg(i,k) = (mdg(i,k)-mdg(i,k+1))*rdpg(i,k)
         end do
      end do
      do i = 1, lengath
         dug(i,plev) = eug(i,plev) - mug(i,plev)*rdpg(i,plev)
         edg(i,plev) = 0.0
      end do
      do k = 1, plev
         do i = 1, lengath
            if( dug(i,k) < 1.e-7*eug(i,k) ) then
	       dug(i,k) = 0.
	    end if
         end do
      end do

!-----------------------------------------------------------------------
!     	... Find top and bottom layers with updrafts.
!-----------------------------------------------------------------------
      do i = 1,lengath
         jt(i) = plev
         mx(i) = 1
      end do
      do k = 2,plev
	 lenpos = COUNT( mug(:lengath,k) > 0. )
	 if( lenpos > 0 ) then
	    index(:lengath) = (/ (i,i=1,lengath) /)
	    index(:lengath) = PACK( index(:lengath), mask = mug(:lengath,k) > 0. )
            do i = 1,lenpos
               jt(index(i)) = MIN( k-1, jt(index(i)) )
               mx(index(i)) = MAX( k, mx(index(i)) )
            end do
	 end if
      end do

!-----------------------------------------------------------------------
!     	... Calc delta p between srfc and cloud base.
!-----------------------------------------------------------------------
      do i = 1,lengath
         dsubcld(i) = dpg(i,plev)
      end do
      do k = plev-1,2,-1
         do i = 1,lengath
            if( mx(i) <= k ) then
               dsubcld(i) = dsubcld(i) + dpg(i,k)
            end if
         end do
      end do

      end subroutine ARCONVTRAN

      end module MO_ZHANG_SUBS
