
      subroutine limsh( ps, t, q, plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... limit maximum specific humidity to the saturation value.
!-----------------------------------------------------------------------

      use mo_grid,  only : plev, plevp
      use plevs,    only : plevs0
      use eslookup, only : aqsat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, platl, pplon
      real, intent(in) :: &
        ps(plonl,-3:platl+4,pplon), &       ! surface pressure
        t(plonl,plev,platl,pplon)           ! temperature at model levels

      real, intent(inout) :: &
        q(plonl,plev,platl,pplon)           ! specific humidity

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: ip, j, k
      real :: &
        pint(plonl,plevp), &    ! pressure at model interfaces
        pmid(plonl,plev), &     ! pressure at model levels
        pdel(plonl,plev), &     ! layer thickness (pint(k+1) - pint(k))
        es(plonl,plev), &       ! saturation vapor pressure
        qs(plonl,plev)          ! saturation specific humidity

      do ip = 1,pplon
         do j = 1,platl
            call plevs0( ps(1,j,ip), pint, pmid, pdel, plonl )
            call aqsat( t(1,1,j,ip), pmid, es, qs, plonl, &
			plonl, plev, 1, plev)
            do k = 1,plev
               q(:,k,j,ip) = min( q(:,k,j,ip), qs(:,k) )
            end do
         end do
      end do

      end subroutine limsh

      subroutine handle_ncerr( ret, mes )
!-----------------------------------------------------------------------
! 	... check netcdf library function return code.  if error detected 
!           issue error message then abort.
!-----------------------------------------------------------------------

      use netcdf

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: ret            ! return code from netcdf library routine
      character(len=*), intent(in) :: mes   ! message to be printed if error detected

      if( ret /= nf_noerr ) then
         write(*,*) mes
         write(*,*) nf_strerror( ret )
         call endrun
      end if

      end subroutine handle_ncerr

      subroutine moz_chktime( time, nrec )
!-----------------------------------------------------------------------
! 	... make sure the time coordinate looks like calander day,
!           and is increasing.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        nrec
      real, intent(in) :: &
        time(nrec)

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: i

      if( time(1) < 1. .or. time(1) > 366. ) then
         write(*,*) 'moz_chktime: illegal time coordinate ',time(1)
         call endrun
      end if
      do i = 2,nrec
         if( time(i) < 1. .or. time(i) > 366. ) then
            write(*,*) 'moz_chktime: illegal time coordinate ',time(i)
            call endrun
         end if
         if( time(i) <= time(i-1) ) then
            write(*,*)' moz_chktime: time not increasing ', time(i-1), time(i)
            call endrun
         end if
      end do

      end subroutine moz_chktime

      subroutine moz_findplb( x, nx, xval, index )
!-----------------------------------------------------------------------
! 	... find periodic lower bound
! 	search the input array for the lower bound of the interval that
! 	contains the input value.  the returned index satifies:
! 	x(index) .le. xval .lt. x(index+1)
! 	assume the array represents values in one cycle of a periodic coordinate.
! 	so, if xval .lt. x(1), then index=nx.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        nx
      integer, intent(out) :: &
        index
      real, intent(in) :: &
        x(nx), &                         ! strictly increasing array
        xval

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: i

      if( xval < x(1) .or. xval >= x(nx) ) then
         index = nx
         return
      end if

      do i = 2,nx
         if( xval < x(i) ) then
            index = i - 1
            exit
         end if
      end do

      end subroutine moz_findplb

      subroutine moz_linintp( npts, t1, t2, tint, f1, f2, fint )
!-----------------------------------------------------------------------
! 	... linearly interpolate between f1(t1) and f2(t2) to fint(tint)
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        npts
      real, intent(in) :: &
        t1, &            ! time level of f1
        t2, &            ! time level of f2
        tint, &          ! interpolant time
        f1(npts), &      ! field at time t1
        f2(npts)         ! field at time t2
      real, intent(out) :: &
        fint(npts)       ! field at time tint

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i
      real    :: factor

      factor = (tint - t1) / (t2 - t1)

      do i = 1, npts
         fint(i) = f1(i) + ( f2(i) - f1(i) )*factor
      end do

      end subroutine moz_linintp
