
#include <params.h>

      program mozart4
!-----------------------------------------------------------------------
! 	... combined mozart and match
!-----------------------------------------------------------------------

      use mo_mpi
      use mo_control,        only : nestep, laststep, endofrun, wrestart, advqth, itim0, limqin, pdiags, &
                                    ini_date, update_time, xactive_h2o, xactive_prates, xactive_drydep, &
                                    xactive_emissions, use_dust
      use mo_grid,           only : plon, plonl, pplon, plev, plevp, plat, plnplv, pcnst, pcnstm1, nodes
      use mo_ghost_transfer, only : ghost_xfer
      use mo_dyninp,         only : chkdyn, interp1, interp2
      use mo_physcan,        only : physcan
      use mo_restart,        only : wrrst
      use mo_inirun,         only : inirun
      use m_tracname,        only : tracnam
      use mo_ffsldr,         only : ffsldr
      use mo_histout,        only : disphst, hst_gather, wrnchist, nexthist
      use mo_histout,        only : hst_file_max, sim_file_cnt, hfile, moz_file_cnt
      use mo_iniqth,         only : iniqth
      use mo_drymass,        only : adjdry
      use mo_hook,           only : moz_hook
      use mo_calendar,       only : newdate, caldayr
      use mo_timer,          only : sw_second, time_diff, elapsed, cdate, ctime
      use mo_ctlstep,        only : ctlstep
      use mass_diags,        only : advn_flux
      use mo_file_utils,     only : navu, freeunit, close_arch
      use mo_lifetime,       only : set_lifetime
      use mo_local_time,     only : local_time_inti
      use m_adv,             only : do_pressure_fixer, j1, jlim_north, has_npole, has_spole, jle, jue, jl, ju
      use mo_offline_constits, only : advance_offline_constits
      use mo_offline_sources,  only : advance_offline_sources

      implicit none

      integer :: &
        platl, &       ! latitude tile dimension
        astat, &       ! allocation status
        ierr, &        ! status
        file, &        ! file index
        i, ip, j, k, m, &
        nt, &          ! time index for advected species array (current)
        ntp1, &        ! time index for advected species array (forecast)
        idt, &         ! timestep interval in seconds
        ncdate, &      ! date at beginning of current timestep in yymmdd format
        ncsec, &       ! seconds relative to ncdate
        ncdate_end, &  ! date at end of current timestep in yymmdd format
        ncsec_end, &   ! seconds relative to ncdate_end
        mpdate, &      ! date at midpoint of current timestep in yymmdd format
        mpsec, &       ! seconds relative to mpdate
        iday, &        ! used to increment the date
        nstep, &       ! timestep counter
        nadv = 0, &    ! number of misc advected variables
        nx1, &         ! number of elements in 1 time level of x array
        nxt0 = 1, &    ! index for misc advected variables interpolated to the
                       ! beginning of the current time step
        nxt1 = 1, &    ! index for misc advected variables forecast from nxt0
        nxt0off = 1, & ! nxt0 + offset to beginning of actual data
        nxt1off = 1, & ! nxt1 + offset to beginning of actual data
        maxthreads     ! omp thread limit

      integer :: negcnt, mpi_grp_world, mpi_grp_comp
      integer :: longitude_tiles = 1
      integer :: min_ind(4)
      real :: &
        dtime, &       ! timestep interval in seconds
        caldayh, &     ! julian calandar day for current timestep midpoint
        caldayf        ! julian calandar day for current timestep endpoint

      real, allocatable :: &
        u(:,:,:,:), &                      ! u wind (m/s)
        v(:,:,:,:), &                      ! v wind (m/s)
        w(:,:,:,:), &                      ! w wind
        omga(:,:,:,:), &                   ! vertical velocity (omega) (Pa/s)
        as(:,:,:,:,:), &                   ! advected species (kg/kg)
        zm(:,:,:,:), &                     ! potential height above surface at midpoints (m)
        zint(:,:,:,:), &                   ! potential height above surface at interfaces (m)
        t(:,:,:,:,:), &                    ! temperature (K)
        sh(:,:,:,:,:), &                   ! specific humidity (kg/kg)
        shadv(:,:,:,:), &                  ! advected specific humidity (kg/kg)
        ps(:,:,:,:), &                     ! surface pressure (Pa)
        ps_pred(:,:,:), &                  ! predicted surface pressure (Pa)
        ps_diff(:,:,:), &                  ! surface pressure difference (Pa)
        oro(:,:,:), &                      ! orography
        phis(:,:,:), &                     ! surface geopotential (m^2/s^2)
        hw1(:,:), &                        ! constituent mass before advection (kg/m^2)
        hw2(:,:), &                        ! constituent mass after  advection (kg/m^2)
        hw1g(:,:), &                       ! constituent mass before advection (kg/m^2)
        hw2g(:,:), &                       ! constituent mass after  advection (kg/m^2)
        hw1t(:), &                         ! constituent mass before advection (kg/m^2)
        hw2t(:)                            ! constituent mass after  advection (kg/m^2)

      real, allocatable :: x(:,:,:,:,:,:)  ! misc species (kg/kg)

      real, allocatable :: &
        ts(:,:,:), &                       ! surface temperature (K)
        ts_avg(:,:,:), &                   ! average surface temperature (K)
        taux(:,:,:), &                     ! stress in x direction
        tauy(:,:,:), &                     ! stress in y direction
        hflx(:,:,:), &                     ! surface sensible heat flux (W/m^2)
        qflx(:,:,:), &                     ! surface water vapor flux (kg/m2/s)
        cldtop(:,:,:), &                   ! cloud top height level index 
        cldbot(:,:,:), &                   ! cloud bottom height level index
        snow(:,:,:), &                     ! snow height (m)
        fsds(:,:,:), &                     ! direct radiation at srf (W/m^2)
        fsds_avg(:,:,:), &                 ! average direct radiation at srf (W/m^2)
        soilw(:,:,:), &                    ! soil moisture (fraction)
        precip(:,:,:)                      ! precip at srf (m)

      real, allocatable, dimension(:,:,:,:) :: &
        cgs, &                             ! counter-gradient coefficient
        kvh, &                             ! vertical diffusion coefficient (m^2/s)
        zmu, &                             ! mu2 from conv_ccm  (kg/m2/s)
        zmd, &                             ! md2 from conv_ccm  (kg/m2/s)
        zeu, &                             ! eu2 from conv_ccm  (1/s)
        hketa, &                           ! convective mass flux, hack
        hkbeta, &                          ! overshoot parameter, hack
        cmfdqr, &                          ! dq/dt due to convective rainout 
        qrad, &                            ! radiative heating tendency (K/s)
        concld, &                          ! convective cloud fraction
        nrain, &                           ! rate of release of stratiform precip (1/s)
        nevapr, &                          ! rate of evaporation of precipitation (1/s)
        cwat, &                            ! cloud water (kg/kg)
        cldfr, &                           ! cloud fraction
        pdmfuen, &                         ! entrainment into updraft  
        pdmfude, &                         ! detrainment into updraft  
        pdmfden, &                         ! entrainment into downdraft
        pdmfdde                            ! detrainment into downdraft

      real :: etimer, loop_timer
      real :: max_psdel, l2_psdel(plat)
      real :: max_qdel, l2_qdel
      real :: gmax_psdel, gl2_psdel
      real :: temp_elapsed
      character(len=12) :: present_time
      character(len=8)  :: adv_names(2)
      logical, dimension(hst_file_max) :: &
              partial_ta

      namelist / simulation_params / longitude_tiles

#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... initialize mpi
!-----------------------------------------------------------------------
      call mpi_init( astat )
      if( astat /= mpi_success ) then
         write(*,*) 'mozart4: failed to initialize mpi; error = ',astat
         stop
      end if
!-----------------------------------------------------------------------
!     	... get node id
!-----------------------------------------------------------------------
      call mpi_comm_rank( mpi_comm_world, thisnode, astat )
      if( astat /= mpi_success ) then
         write(*,*) 'mozart4: failed to get node rank; error = ',astat
         call endrun
      else
         write(*,*) 'mozart4: this node is number ',thisnode
      end if
!-----------------------------------------------------------------------
!     	... get node count
!-----------------------------------------------------------------------
      call mpi_comm_size( mpi_comm_world, maxnodes, astat )
      if( astat /= mpi_success ) then
         write(*,*) 'mozart4: failed to get node count; error = ',astat
         call endrun
      else
         if( maxnodes < 2 ) then
            write(*,*) 'mozart4: cannot run an mpi simulation with tasks  < 2'
            call endrun
         end if
         if( mod( plat,maxnodes) == 0 ) then
            mpi_comm_comp = mpi_comm_world
            comp_node     = .true.
            io_node       = thisnode == 0
            gather_node   = 0
            ded_io_node   = .false.
         else if( mod( plat,(maxnodes-1) ) == 0 ) then
            maxnodes    = maxnodes - 1
            comp_node   = thisnode < maxnodes
            gather_node = maxnodes
            io_node     = .not. comp_node
            ded_io_node = .true.
            call mpi_comm_group( mpi_comm_world, mpi_grp_world, astat )
            if( astat /= mpi_success ) then
               write(*,*) 'mozart4: failed to get world comm group; error = ',astat
               call endrun
            end if
           call mpi_group_excl( mpi_grp_world, 1, (/ maxnodes /), mpi_grp_comp, astat )
            if( astat /= mpi_success ) then
               write(*,*) 'mozart4: failed to create comp group; error = ',astat
               call endrun
            end if
            call mpi_comm_create( mpi_comm_world, mpi_grp_comp, mpi_comm_comp, astat )
            if( astat /= mpi_success ) then
               write(*,*) 'mozart4: failed to create comp comm; error = ',astat
               call endrun
            end if
         else
            write(*,*) 'mozart4: runtime    nodes = ',maxnodes
            write(*,*) '         simulation nodes = ',nodes
            write(*,*) '         either run on ',nodes,' nodes or'
            write(*,*) '         compile for ',maxnodes, 'nodes'
            call endrun
         end if
         write(*,*) 'mozart4: maxnodes = ',maxnodes,' on node ',thisnode
      end if
      masternode   = thisnode == 0
      lastnode     = thisnode == (maxnodes-1)
      interiornode = .not. masternode .and. .not. lastnode
      platl        = plat/maxnodes
      latspnode    = plat/maxnodes
      base_lat     = thisnode*latspnode
      write(*,*) 'mozart4: mpi diags'
      write(*,*) '         mpi_comm_comp,comp_node,io_node,gather_node,ded_io_node,masternode,lastnode = ',&
                           mpi_comm_comp,comp_node,io_node,gather_node,ded_io_node,masternode,lastnode
#else
      masternode   = .true.
      lastnode     = .true.
      io_node      = .true.
      comp_node    = .true.
      interiornode = .false.
      gather_node  = 0
      ded_io_node  = .false.
      platl        = plat
      latspnode    = plat
      base_lat     = 0
      thisnode     = 0
      maxnodes     = 1
#endif
      nodes = maxnodes
      if( comp_node ) then
         has_spole    = (base_lat - 3) < j1
         has_npole    = (base_lat + platl + 4) > jlim_north
         write(*,*) 'mozart4: has_spole, has_npole = ',has_spole,has_npole
      end if

#ifdef USE_OMP
!-----------------------------------------------------------------------
!     	... read in the longitude tile count (pplon)
!-----------------------------------------------------------------------
      m = navu()
      open( unit = m, file = 'sim.params.nml', iostat = ierr )
      if( ierr /= 0 ) then
         write(*,*) 'mozart4: failed to open sim.params.nml file; error = ',ierr
         call endrun
      end if
      read(m,simulation_params,iostat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'mozart4: failed to read sim.params.nml file; error = ',ierr
         call endrun
      end if
      close( m )
      call freeunit( m )
#endif
      pplon = longitude_tiles

!-----------------------------------------------------------------------
!     	... elementary simulation check
!-----------------------------------------------------------------------
      call chk_sim( pplon, platl, nodes )
      plonl  = plon/pplon
      plnplv = plonl*plev

      call date_and_time( cdate(1), ctime(1) )
      write(*,*) 'start time = ',ctime(1)
      elapsed(10) = sw_second()
!-----------------------------------------------------------------------
!     	... initialize time level indices
!-----------------------------------------------------------------------
      nt   = 1
      ntp1 = 2

!-----------------------------------------------------------------------
!     	... allocate arrays
!-----------------------------------------------------------------------
      allocate( as(plonl,plev,pcnst,platl,pplon), &
                ps(plonl,-3:platl+4,pplon,2), &
		zm(plonl,plev,platl,pplon), &
#if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
                cldtop(plonl,platl,pplon), &
                cldbot(plonl,platl,pplon), &
#endif
                zint(plonl,plevp,platl,pplon),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'mozart4: failed to allocate as ... zint; error = ',astat
	 call endrun
      end if
      if( comp_node ) then
         allocate( snow(plonl,platl,pplon), &
                   fsds(plonl,platl,pplon), &
                   soilw(plonl,platl,pplon), &
                   precip(plonl,platl,pplon),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'mozart4: failed to allocate snow ... precip; error = ',astat
	    call endrun
         end if
         precip(:,:,:) = 0.
         allocate( u(plonl,plev,-3:platl+4,pplon), &
		   v(plonl,plev,-2:platl+3,pplon), &
                   w(plonl,plevp,platl,pplon), &
		   omga(plonl,plev,platl,pplon), &
                   t(plonl,plev,platl,pplon,2), &
		   shadv(plonl,plev,platl,pplon), &
		   sh(plonl,plev,platl,pplon,2), stat=astat )
         if( astat /= 0 ) then
	    write(*,*) 'mozart4: failed to allocate arrays u .. sh; error = ',astat
	    call endrun
         end if
         allocate( ps_pred(plonl,-3:platl+4,pplon), &
		   ps_diff(plonl,platl,pplon), &
		   oro(plonl,platl,pplon), &
		   phis(plonl,platl,pplon), &
                   hw1(pcnst,platl), &
		   hw2(pcnst,platl), &
                   hw1g(pcnst,plat), &
		   hw2g(pcnst,plat), &
		   hw1t(pcnst), &
		   hw2t(pcnst), stat=astat )
         if( astat /= 0 ) then
	    write(*,*) 'mozart4: failed to allocate arrays ps .. hw2t; error = ',astat
	    call endrun
         end if
         allocate( ts(plonl,platl,pplon), &
                   ts_avg(plonl,platl,pplon), &
                   fsds_avg(plonl,platl,pplon), &
                   taux(plonl,platl,pplon), &
                   tauy(plonl,platl,pplon), &
                   hflx(plonl,platl,pplon), &
                   qflx(plonl,platl,pplon), &
#if defined AR_VDIFF || defined DI_VDIFF
                   cgs(plonl,plevp,platl,pplon), &
                   kvh(plonl,plevp,platl,pplon), &
#endif
#if defined AR_CONV_CCM || defined DI_CONV_CCM
                   zmu(plonl,plev,platl,pplon), &
                   zmd(plonl,plev,platl,pplon), &
                   zeu(plonl,plev,platl,pplon), &
                   hketa(plonl,plev,platl,pplon), &
                   hkbeta(plonl,plev,platl,pplon), &
                   cmfdqr(plonl,plev,platl,pplon), &
#endif
#if defined DI_CONV_CCM
                   qrad(plonl,plev,platl,pplon), &
#endif
#if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
                   concld(plonl,plev,platl,pplon), &
                   nrain(plonl,plev,platl,pplon), &
                   nevapr(plonl,plev,platl,pplon), &
                   cwat(plonl,plev,platl,pplon), &
                   cldfr(plonl,plevp,platl,pplon), &
#endif
                   pdmfuen(plonl,plev,platl,pplon), &
                   pdmfude(plonl,plev,platl,pplon), &
                   pdmfden(plonl,plev,platl,pplon), &
                   pdmfdde(plonl,plev,platl,pplon), &
		   stat=astat )
         if( astat /= 0 ) then
	    write(*,*) 'mozart4: failed to allocate arrays ts .. pdmfdde; error = ',astat
	    call endrun
         end if
      end if

!-----------------------------------------------------------------------
!     	... initialize date and field variables
!-----------------------------------------------------------------------
      call inirun( idt, ncdate, ncsec, nadv, u, &
                   v, w, ps(1,-3,1,nt), t(1,1,1,1,nt), sh, &
                   cgs, kvh, zmu, zmd, zeu, &
                   hketa, hkbeta, phis, oro, as, &
                   ts, taux, tauy, hflx, qflx, &
                   qrad, omga, cldtop, cldbot, zm, &
                   zint, cmfdqr, cldfr, concld, nrain, &
                   nevapr, cwat, pdmfuen, pdmfude, pdmfden, &
		   pdmfdde, snow, fsds, soilw, precip, &
                   ts_avg, fsds_avg, plonl, platl, pplon, nodes )

      if( advqth .and. nadv > 0 ) then
!-----------------------------------------------------------------------
!  	... set offset indices and allocate memory for misc advected variables 
!-----------------------------------------------------------------------
	 if( comp_node ) then
	    allocate( x(plonl,plev,nadv,platl,pplon,2),stat=astat )
	    if( astat /= 0 ) then
	       write(*,*) 'mozart4: failed to allocate x; error = ',astat
	       call endrun
	    end if
	 end if
	 adv_names(1) = 'sh'
	 adv_names(2) = 'pt'
         if( xactive_h2o ) then
            do ip = 1,pplon
               do j = 1,platl
                  do k = 1,plev
                     x(:,k,1,j,ip,nt) = sh(:,k,j,ip,2)
                  end do
               end do
            end do
         end if
      else
         xactive_h2o = .false.
      end if

      if( xactive_prates .or. xactive_h2o .or. xactive_drydep .or. xactive_emissions ) then
         write(*,*) ' '
         write(*,*) '--------------------------------------------------------'
         if( xactive_h2o ) then
            write(*,*) ' mozart4 simulation with interactive water vapor'
         end if
         if( xactive_prates ) then
            write(*,*) ' mozart4 simulation with interactive photolysis'
         end if
         if( use_dust ) then
            write(*,*) ' mozart4 simulation with climatological dust'
         end if
         if( xactive_drydep ) then
            write(*,*) ' mozart4 simulation with interactive dry deposition'
         end if
         if( xactive_emissions ) then
            write(*,*) ' mozart4 simulation with interactive no and isoprene emissions'
         end if
         write(*,*) '--------------------------------------------------------'
         write(*,*) ' '
      end if

      call date_and_time( cdate(16), ctime(16) )
      elapsed(16) = time_diff( ctime(1), ctime(16) )
      dtime       = real( idt )
      call ini_date( ncdate, ncsec, idt, itim0, nestep )
      call local_time_inti( moz_file_cnt )
!===========================================================================
!     	... time step loop
!===========================================================================
      call date_and_time( cdate(7), ctime(7) )
model_timestep_loop : &
      do nstep = itim0+1,itim0+nestep
	 call ctlstep( nstep, ncdate, ncsec, idt, platl )
	 call update_time
comp_nodes_only : &
	 if( comp_node ) then
            if( advqth ) then
               call iniqth( ps(1,-3,1,nt), sh(1,1,1,1,nt), t(1,1,1,1,nt), nadv, x(1,1,1,1,1,nt), &
                            plonl, platl, pplon )
            end if
!---------------------------------------------------------------------------
!    	... transfer the ghost points
!---------------------------------------------------------------------------
            call ghost_xfer( as, nadv, x(1,1,1,1,1,nt), plonl, pplon, platl )

!---------------------------------------------------------------------------
!    	... set date at midpoint of timestep
!---------------------------------------------------------------------------
            mpsec  = ncsec + idt/2
            mpdate = ncdate
            if( mpsec >= 86400 ) then
               iday   = mpsec/86400
               mpsec  = mod( mpsec,86400 )
               mpdate = newdate( ncdate, iday )
            end if
            caldayh = caldayr( mpdate, mpsec )

!---------------------------------------------------------------------------
!    	... date at end of timestep
!---------------------------------------------------------------------------
            ncsec_end  = ncsec + idt
            ncdate_end = ncdate
            if( ncsec_end >= 86400 ) then
               iday       = ncsec_end/86400
               ncsec_end  = mod( ncsec_end,86400 )
               ncdate_end = newdate( ncdate, iday )
            end if

!---------------------------------------------------------------------------
! 	... advance offline constituent data to end of timestep
!---------------------------------------------------------------------------
            call advance_offline_constits( ncdate, ncsec, plonl, platl, pplon )

!---------------------------------------------------------------------------
! 	... advance offline sources data to end of timestep
!---------------------------------------------------------------------------
            call advance_offline_sources( ncdate, ncsec, plonl, platl, pplon )

!---------------------------------------------------------------------------
! 	... interpolate data to midpoint of timestep
!---------------------------------------------------------------------------
            call date_and_time( cdate(8), ctime(8) )
            call chkdyn( mpdate, mpsec, phis, oro, plonl, &
			 platl, pplon )
            call interp2( mpdate, mpsec, ps(1,-3,1,ntp1), u, v, &
                          w, cgs, kvh, ts, taux, &
                          tauy, hflx, qflx, qrad, omga, &
                          pdmfuen, pdmfude, pdmfden, pdmfdde, &
			  snow, fsds, soilw, plonl, platl, pplon )

!---------------------------------------------------------------------------
! 	... interpolate data to end of timestep
!---------------------------------------------------------------------------
            call chkdyn( ncdate_end, ncsec_end, phis, oro, plonl, &
			 platl, pplon )
            call interp1( ncdate_end, ncsec_end, ps(1,-3,1,ntp1), t(1,1,1,1,ntp1), sh(1,1,1,1,ntp1), &
                          ts, ts_avg, fsds_avg, plonl, platl, pplon )
            call date_and_time( cdate(20), ctime(20) )
            elapsed(8) = elapsed(8) + time_diff( ctime(8), ctime(20) )
            if( .not. xactive_h2o ) then
!---------------------------------------------------------------------------
!   	... limit input specific humidity to saturation
!---------------------------------------------------------------------------
               if( limqin ) then
                  call limsh( ps(1,-3,1,ntp1), t(1,1,1,1,ntp1), sh(1,1,1,1,ntp1), plonl, platl, pplon )
               end if
!---------------------------------------------------------------------------
!  	... adjust surface pressure to maintain constant dry mass
!---------------------------------------------------------------------------
               call adjdry( ps(1,-3,1,ntp1), sh(1,1,1,1,ntp1), .false., plonl, platl, pplon )
            else
               do ip = 1,pplon
                  do j = 1,platl
                     do k = 1,plev
                        shadv(:,k,j,ip) = x(:,k,1,j,ip,nt)
                     end do
                  end do
               end do
!---------------------------------------------------------------------------
!  	... adjust surface pressure to maintain constant dry mass
!---------------------------------------------------------------------------
               call adjdry( ps(1,-3,1,ntp1), shadv, .false., plonl, platl, pplon )
            end if
!---------------------------------------------------------------------------
!   	... advection
!---------------------------------------------------------------------------
            call date_and_time( cdate(2), ctime(2) )
            call ffsldr( nstep, dtime, u, v, w, &
                         ps(1,-3,1,nt), ps(1,-3,1,ntp1), ps_pred, as, sh(1,1,1,1,nt), &
                         hw1, hw2, nadv, x(1,1,1,1,1,ntp1), plonl, platl, pplon )

	    do m = 1,pcnstm1
	       negcnt = count( as(:,:,m,:,:) < 0. )
               if( negcnt > 0 .and. pdiags%adv ) then
                  min_ind(:) = minloc( as(:,:,m,:,:) )
                  write(*,'(''mozart4: there are '',i8,'' negative values for advected '',a8,'' at '',4i4)') &
                     negcnt, tracnam(m), min_ind
	       end if
	    end do
	    do m = 1,nadv
	       negcnt = count( x(:,:,m,:,:,ntp1) < 0. )
               if( negcnt > 0 .and. pdiags%adv ) then
                  min_ind(:) = minloc( x(:,:,m,:,:,ntp1) )
                  write(*,'(''mozart4: there are '',i8,'' negative values for advected '',a8,'' at '',4i4)') &
                     negcnt, adv_names(m), min_ind
	       end if
	    end do
#ifdef DEBUG
	    max_qdel = 100.*maxval( abs( x(:,:,1,:,:,ntp1) - x(:,:,1,:,:,nt) ) / abs( x(:,:,1,:,:,nt) ) )
            write(*,'(''mozart4: max % diff in advected water vapor = '',1p,e11.3)') max_qdel
	    write(*,*) 'mozart4: done ffsldr on node ',thisnode
#endif
            call date_and_time( cdate(3), ctime(3) )
            elapsed(1) = elapsed(1) + time_diff( ctime(2), ctime(3) )

#ifdef USE_MPI
            call mpi_gather( hw1, pcnst*platl, mpi_double_precision, &
                             hw1g, pcnst*platl, mpi_double_precision, &
                             0, mpi_comm_comp, ierr )
            if( ierr /= 0 ) then
               write(*,*) 'mozart4: mpi_gather failed for hw1; error = ',ierr
               call endrun
            end if
            call mpi_gather( hw2, pcnst*platl, mpi_double_precision, &
                             hw2g, pcnst*platl, mpi_double_precision, &
                             0, mpi_comm_comp, ierr )
            if( ierr /= 0 ) then
               write(*,*) 'mozart4: mpi_gather failed for hw2; error = ',ierr
               call endrun
            end if
#else
            hw1g = hw1
            hw2g = hw2
#endif
            if( masternode ) then
               do m = 1,pcnst
                  hw1t(m) = sum( hw1g(m,:plat) )
                  hw2t(m) = sum( hw2g(m,:plat) )
               end do
            end if

!---------------------------------------------------------------------------
! 	... output mass of tracers to log file
!---------------------------------------------------------------------------
            if( masternode ) then
               write(*,*) ' '
               write(*,*) ' tracer mass before advection (kg/m^2): '
               write(*,'(5(1p,e21.13))') hw1t(:pcnst)
               write(*,*) ' '
               write(*,*) ' advection tracer mass change (%) : '
               write(*,'(5(1p,e21.13))') 100.*(hw2t(:pcnst-1) - hw1t(:pcnst-1))/hw1t(:pcnst-1)
            end if

!---------------------------------------------------------------------------
!   	... date at end of timestep
!---------------------------------------------------------------------------
            ncsec = ncsec + idt
            if( ncsec >= 86400 ) then
               iday   = ncsec/86400
               ncsec  = mod( ncsec,86400 )
               ncdate = newdate( ncdate, iday )
            end if
            caldayf = caldayr( ncdate, ncsec )

            if( .not. do_pressure_fixer ) then
               call date_and_time( cdate(8), ctime(8) )
!---------------------------------------------------------------------------
!  	... interpolate data to end of the timestep
!---------------------------------------------------------------------------
               call chkdyn( ncdate, ncsec, phis, oro, plonl, &
			    platl, pplon )
               call interp1( ncdate, ncsec, ps(1,-3,1,ntp1), t(1,1,1,1,ntp1), sh(1,1,1,1,ntp1), &
                             ts, ts_avg, fsds_avg, plonl, platl, pplon )
               call date_and_time( cdate(20), ctime(20) )
               elapsed(8) = elapsed(8) + time_diff( ctime(8), ctime(20) )
	    end if

            call date_and_time( cdate(19), ctime(19) )
!---------------------------------------------------------------------------
!	... surface pressure diagnostic
!---------------------------------------------------------------------------
	    do ip = 1,pplon
	       do j = jl(5),ju(5)
                  ps_diff(:,j,ip) = (ps_pred(:,j,ip) - ps(:,j,ip,ntp1)) / ps(:,j,ip,ntp1) 
	       end do
	    end do
            max_psdel = maxval( abs( ps_diff(:plonl,jl(5):ju(5),:pplon) ) )
	    do j = jl(5),ju(5)
               l2_psdel(base_lat+j) = sum( ps_diff(:plonl,j,:pplon)**2 )
            end do
#ifdef USE_MPI
            call mpi_reduce( max_psdel, gmax_psdel, 1, mpi_double_precision, &
                             mpi_max, 0, mpi_comm_comp, ierr )
            if( ierr /= 0 ) then
	       write(*,*) 'mozart4: mpi_reduce failed for max_psdel; error = ',ierr
	       call endrun
            end if
	    if( masternode ) then
              call mpi_gather( mpi_in_place, platl, mpi_double_precision, &
                               l2_psdel, platl, mpi_double_precision, &
                               0, mpi_comm_comp, ierr )
            else
              call mpi_gather( l2_psdel(base_lat+1:base_lat+platl), platl, mpi_double_precision, &
                               l2_psdel, platl, mpi_double_precision, &
                               0, mpi_comm_comp, ierr )
            end if
            if( ierr /= 0 ) then
	       write(*,*) 'mozart4: mpi_gather failed for l2_psdel; error = ',ierr
	       call endrun
            end if
#else
            gmax_psdel = max_psdel
#endif
	    if( masternode ) then
               gl2_psdel = sum( l2_psdel(j1:plat-2) )
               write(*,*) ' '
               write(*,*) ' max surface pressure delta = ',100.*gmax_psdel
               write(*,*) ' avg surface pressure delta = ',sqrt( gl2_psdel/real(plon*plat) ) 
	    end if

            call date_and_time( cdate(20), ctime(20) )
            elapsed(19) = elapsed(19) + time_diff( ctime(19), ctime(20) )

!---------------------------------------------------------------------------
!  	... use the predicted pressure from l-r, instead of the met field pressure
!---------------------------------------------------------------------------
            if( do_pressure_fixer ) then
               do ip = 1,pplon
                  do j = jle(7),jue(7)
                     ps(:,j,ip,ntp1) = ps_pred(:,j,ip)
                  end do
               end do
            end if
            if( .not. xactive_h2o ) then
!---------------------------------------------------------------------------
!   	... limit input specific humidity to saturation
!---------------------------------------------------------------------------
               if( limqin ) then
                  call limsh( ps(1,-3,1,ntp1), t(1,1,1,1,ntp1), sh(1,1,1,1,ntp1), plonl, platl, pplon )
               end if
!---------------------------------------------------------------------------
!  	... adjust surface pressure to maintain constant dry mass
!---------------------------------------------------------------------------
               call adjdry( ps(1,-3,1,ntp1), sh(1,1,1,1,ntp1), .false., plonl, platl, pplon )
            else
               do ip = 1,pplon
                  do j = 1,platl
                     do k = 1,plev
                        shadv(:,k,j,ip) = x(:,k,1,j,ip,ntp1)
                     end do
                  end do
               end do
!---------------------------------------------------------------------------
!   	... limit input specific humidity to saturation
!---------------------------------------------------------------------------
               if( limqin ) then
                  call limsh( ps(1,-3,1,ntp1), t(1,1,1,1,ntp1), shadv, plonl, platl, pplon )
                  do ip = 1,pplon
                     do j = 1,platl
                        do k = 1,plev
                           x(:,k,1,j,ip,ntp1) = shadv(:,k,j,ip)
                        end do
                     end do
                  end do
               end if
!---------------------------------------------------------------------------
!  	... adjust surface pressure to maintain constant dry mass
!---------------------------------------------------------------------------
               call adjdry( ps(1,-3,1,ntp1), shadv, .false., plonl, platl, pplon )
            end if
!---------------------------------------------------------------------------
!   	... check for ancillary chemistry inputs
!---------------------------------------------------------------------------
            call moz_hook( ncdate, ncsec, caldayh, caldayf, cldtop, &
                           cldbot, oro, zm, zint, t(1,1,1,1,ntp1), &
                           ts_avg, fsds_avg, plonl, platl, pplon )

!---------------------------------------------------------------------------
!   	... physics/chemistry parameterizations and
!	    write history buffer to disk
!---------------------------------------------------------------------------
            call date_and_time( cdate(2), ctime(2) )
            call physcan( dtime, caldayh, ncdate, ncsec, nstep, &
                          nadv, ps(1,-3,1,ntp1), oro, phis, x(1,1,1,1,1,nt), &
                          t(1,1,1,1,ntp1), sh(1,1,1,1,ntp1), u, v, ts, &
                          taux, tauy, hflx, qflx, qrad, &
                          w, omga, cgs, kvh, zmu, &
                          zmd, zeu, hketa, hkbeta, cmfdqr, &
                          cldfr, cldtop, cldbot, zm, zint, &
                          concld, nrain, nevapr, cwat, pdmfuen, &
                          pdmfude, pdmfden, pdmfdde, as, x(1,1,1,1,1,ntp1), &
                          snow, fsds, soilw, precip, ts_avg, &
                          fsds_avg, platl, plonl, pplon, elapsed )
            call date_and_time( cdate(3), ctime(3) )
            elapsed(2) = elapsed(2) + time_diff( ctime(2), ctime(3) )
!---------------------------------------------------------------------------
!   	... check for lifetime output
!---------------------------------------------------------------------------
            call set_lifetime( plonl, platl, pplon )
         else comp_nodes_only
!---------------------------------------------------------------------------
!   	... date at end of timestep for io node
!---------------------------------------------------------------------------
            ncsec = ncsec + idt
            if( ncsec >= 86400 ) then
               iday   = ncsec/86400
               ncsec  = mod( ncsec,86400 )
               ncdate = newdate( ncdate, iday )
            end if
         end if comp_nodes_only
!---------------------------------------------------------------------------
! 	... gather history file(s) output
!---------------------------------------------------------------------------
	 do file = 1,sim_file_cnt
            if( hfile(file)%wrhstts .or. hfile(file)%partial_ta ) then
               call date_and_time( cdate(18), ctime(18) )
#ifdef DEBUG
               write(*,*) 'mozart4: calling hst_gather for file = ',file,' at nstep =',nstep
#endif
	       call hst_gather( nstep, file, ncdate, ncsec, platl )
#ifdef DEBUG
               write(*,*) 'mozart4: back from hst_gather'
#endif
               call date_and_time( cdate(20), ctime(20) )
               elapsed(18) = elapsed(18) + time_diff( ctime(18), ctime(20) )
	    end if
	 end do
!---------------------------------------------------------------------------
! 	... write history file(s)
!---------------------------------------------------------------------------
	 do file = 1,sim_file_cnt
            if( hfile(file)%wrhstts .or. hfile(file)%partial_ta ) then
               call date_and_time( cdate(17), ctime(17) )
#ifdef DEBUG
               write(*,*) 'mozart4: calling wrnchist for file = ',file,' at nstep =',nstep
#endif
               call wrnchist( nstep, file, ncdate, ncsec, platl )
#ifdef DEBUG
               write(*,*) 'mozart4: back from wrnchist'
#endif
               call date_and_time( cdate(20), ctime(20) )
               elapsed(17) = elapsed(17) + time_diff( ctime(17), ctime(20) )
	    end if
!---------------------------------------------------------------------------
! 	... dispose the history file and open the next one
!---------------------------------------------------------------------------
#ifdef DEBUG
            write(*,*) 'mozart4: closehst, fullhst, laststep =',hfile(file)%closehst, &
                                                                   hfile(file)%fullhst, laststep
#endif
            if( hfile(file)%closehst ) then
               call disphst( laststep, .false., file )
               if( hfile(file)%fullhst .and. .not. hfile(file)%partial_ta ) then
                  call nexthist( laststep, file )
               end if
            end if
         end do

!---------------------------------------------------------------------------
! 	... write restart files and dispose
!---------------------------------------------------------------------------
         if( wrestart ) then
            call date_and_time( cdate(9), ctime(9) )
	    partial_ta(:) = hfile(:)%partial_ta
#if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
            if( .not. xactive_h2o ) then
               if( xactive_drydep .or. xactive_emissions ) then
                  call wrrst( nstep, ncdate, ncsec, ps(1,-3,1,ntp1), as, &
			      partial_ta, plonl, platl, pplon, sh(1,1,1,1,ntp1), & 
                              cldtop, cldbot, zm, zint, precip )
               else
                  call wrrst( nstep, ncdate, ncsec, ps(1,-3,1,ntp1), as, &
			      partial_ta, plonl, platl, pplon, sh(1,1,1,1,ntp1), & 
                              cldtop, cldbot, zm, zint )
               end if
            else
               do ip = 1,pplon
                  do j = 1,platl
                     do k = 1,plev
                        shadv(:,k,j,ip) = x(:,k,1,j,ip,ntp1)
                     end do
                  end do
               end do
               if( xactive_drydep .or. xactive_emissions ) then
                  call wrrst( nstep, ncdate, ncsec, ps(1,-3,1,ntp1), as, &
			      partial_ta, plonl, platl, pplon, shadv, &
                              cldtop, cldbot, zm, zint, precip )
               else
                  call wrrst( nstep, ncdate, ncsec, ps(1,-3,1,ntp1), as, &
			      partial_ta, plonl, platl, pplon, shadv, &
                              cldtop, cldbot, zm, zint )
               end if
            end if
# else
            call wrrst( nstep, ncdate, ncsec, ps, as, &
			partial_ta, plonl, platl, pplon )
# endif
            call date_and_time( cdate(20), ctime(20) )
            elapsed(9) = elapsed(9) + time_diff( ctime(9), ctime(20) )
         end if
!---------------------------------------------------------------------------
!  	... output to summary file
!---------------------------------------------------------------------------
         call date_and_time( cdate(20), ctime(20) )
         write(*,*) ' '
	 present_time = ctime(20)(1:2) // ':' // ctime(20)(3:4) // ':' // ctime(20)(5:10)
         write(*,'(1x,i2,": ",a12,'' done timestep '',i6,''  date = '',i8,'':'',i5)') &
            thisnode, present_time, nstep, ncdate, ncsec
         write(*,*) ' '

!---------------------------------------------------------------------------
!  	... check for end of run
!---------------------------------------------------------------------------
         if( endofrun ) then
!---------------------------------------------------------------------------
!   	... free memory
!---------------------------------------------------------------------------
            if( allocated( x ) ) then
               deallocate( x )
            end if
            exit
         end if

!---------------------------------------------------------------------------
!  	... toggle time level indices
!---------------------------------------------------------------------------
         nt   = ntp1
         ntp1 = mod( nt,2 ) + 1
      end do model_timestep_loop

!---------------------------------------------------------------------------
!  	... check for archive closure
!---------------------------------------------------------------------------
      call close_arch

      call date_and_time( cdate(20), ctime(20) )
      elapsed(7) = time_diff( ctime(7), ctime(20) )

!---------------------------------------------------------------------------
!  	... timing information
!---------------------------------------------------------------------------
      write(*,*) ' '
      write(*,*) thisnode,': init        time = ',elapsed(16)
      write(*,*) thisnode,': advection   time = ',elapsed(1)
      write(*,*) thisnode,': adv vel     time = ',elapsed(12)
      write(*,*) thisnode,': adv msg     time = ',elapsed(13)
      write(*,*) thisnode,': adv xtp     time = ',elapsed(14)
      write(*,*) thisnode,': adv vtp     time = ',elapsed(15)
      write(*,*) thisnode,': diagnostics time = ',elapsed(19)
      write(*,*) thisnode,': physcan     time = ',elapsed(2)
      if( xactive_prates ) then
         write(*,*) thisnode,': rtlink      time = ',elapsed(29)
         write(*,*) thisnode,': jvales      time = ',elapsed(28)
      end if
      write(*,*) thisnode,': loop        time = ',elapsed(7)
      write(*,*) thisnode,': dyn read    time = ',elapsed(8)
      write(*,*) thisnode,': hst gather  time = ',elapsed(18)
      write(*,*) thisnode,': hst write   time = ',elapsed(17)
      write(*,*) thisnode,': hst dispose time = ',elapsed(11)
      write(*,*) thisnode,': rsrt write  time = ',elapsed(9)
      write(*,*) thisnode,': overall     time = ',sw_second() - elapsed(10)

      call date_and_time( cdate(2), ctime(2) )
      write(*,*) ' '
      write(*,*) thisnode,': start time = ',ctime(1)
      write(*,*) thisnode,': stop time  = ',ctime(2)
      temp_elapsed = time_diff( ctime(1), ctime(2) )
      write(*,*) thisnode,': elapsed mozart time = ',temp_elapsed
      call date_and_time( cdate(2), ctime(2) )

      deallocate( as, zm, zint )
      if( comp_node ) then
         deallocate( u, v, w, omga, t, sh )
      end if

#ifdef USE_MPI
      call mpi_finalize( ierr )
#endif

      end program mozart4

      subroutine chk_sim( pplon, platl, nodes )
!-----------------------------------------------------------------------
! 	... basic simulation setup checkout
!-----------------------------------------------------------------------

      use mo_grid,    only : plon, plat, pcnst, plev
      use mo_mpi,     only : masternode

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: pplon
      integer, intent(in) :: platl
      integer, intent(in) :: nodes

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: max_threads

      integer :: omp_get_max_threads

      if( mod( plat,platl ) /= 0 ) then
	 write(*,*) 'chk_sim: plat not a multiple of platl'
	 write(*,*) '         plat, platl = ',plat,platl
	 call endrun
      end if
      if( platl < 4 ) then
	 write(*,*) 'chk_sim: minimum of 4 lats per mpi task'
	 write(*,*) '         platl = ',platl
	 call endrun
      end if
#ifndef USE_OMP
      if( pplon /= 1 ) then
	 write(*,*) 'sim_chk: non openmp simulation requires pplon == 1'
	 call endrun
      end if
#endif
      if( mod( plon,pplon) /= 0 ) then
	 write(*,*) 'sim_chk: pplon is not an even divisor of plon'
	 write(*,*) '         plon, pplon = ',plon,pplon
	 call endrun
      end if

      if( masternode ) then
         write(*,*) ' '
         write(*,*) '--------------------------------------------------------'
#if defined USE_MPI && defined USE_OMP
         write(*,*) 'mozart4: this is a hybrid simulation'
#elif defined USE_MPI
         write(*,*) 'mozart4: this is a pure mpi simulation'
#elif defined USE_OMP
         write(*,*) 'mozart4: this is a pure omp simulation'
#endif
#ifdef USE_MPI
         write(*,'('' mozart4 will use '',i3,'' mpi tasks'')') nodes
#endif
#ifdef USE_OMP
         max_threads = omp_get_max_threads()
#ifdef USE_MPI
         write(*,'('' mozart4 will use '',i3,'' omp threads per mpi task'')') max_threads
#else
         write(*,'('' mozart4 will use '',i3,'' omp threads'')') max_threads
#endif
#endif
         write(*,*) ' '
         write(*,'('' mozart4 horizontal resolution = '',i3,'' x '',i3)') plon,plat
         write(*,'('' mozart4 vertical   levels     = '',i3)') plev
         write(*,'('' mozart4 transported species   = '',i3)') pcnst-1
         write(*,*) '--------------------------------------------------------'
         write(*,*) ' '
      end if

      end subroutine chk_sim
