module mo_aerosols

!-----------------------------------------------------------------
! this module computes the production of ammonium nitrate
! using the formulation by Seinfeld and Pandis (p531, 1998)
! with the simplification of activity coefficients and
! aerosol molality using the parameterizations
! from Metzger et al. (JGR, ACH-16, 107(D16), 2002) 
!
! written by Jean-Francois Lamarque (April 2004)
! updated JFL & LKE (April 2007)
!   NH4 is total ammonium aerosol (nitrate+sulfate+bisulfate+...)
!   NH4NO3 is ammonium nitrate
!   SO4 is total sulfate (ammonium+other...)
!-----------------------------------------------------------------

  private
  public :: aerosols_inti, aerosols_formation
  public :: has_aerosols

  save

  integer, target  :: spc_ndx(5)
  integer, pointer :: nh3_ndx, nh4_ndx, nh4no3_ndx
  integer, pointer :: so4_ndx, hno3_ndx
  integer  :: nu_i(2)
  real     :: zeta_inv
  real     :: z_i(2)
  logical  :: has_aerosols = .true.

  contains

  subroutine aerosols_inti( plonl, platl, plev, pplon )

  use mo_chem_utls, only : get_spc_ndx

  implicit none

!-----------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------
  integer, intent(in) :: plonl, platl, plev, pplon

!-----------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------
  integer :: m
  integer :: astat

  nh3_ndx    => spc_ndx(1)
  nh4_ndx    => spc_ndx(2)
  nh4no3_ndx => spc_ndx(3)
  so4_ndx    => spc_ndx(4)
  hno3_ndx   => spc_ndx(5)

!-----------------------------------------------------------------
! 	... set species index
!-----------------------------------------------------------------
  nh3_ndx    = get_spc_ndx( 'NH3'    )
  nh4_ndx    = get_spc_ndx( 'NH4'    )
  nh4no3_ndx = get_spc_ndx( 'NH4NO3' )
  so4_ndx    = get_spc_ndx( 'SO4'    )
  hno3_ndx   = get_spc_ndx( 'HNO3'   )

  has_aerosols = all( spc_ndx(:) > 0 )
  if( .not. has_aerosols ) then
    write(*,*) '-----------------------------------------'
    write(*,*) 'mozart will NOT do nh4no3'
    write(*,*) 'following species are missing'
    do m = 1,size(spc_ndx)
       if( spc_ndx(m) < 1 ) then
          write(*,*) m
       end if
    end do
    write(*,*) '-----------------------------------------'
    return
  else
    write(*,*) '-----------------------------------------'
    write(*,*) 'mozart will do nh4no3'
    write(*,*) '-----------------------------------------'
  end if

!-----------------------------------------------------------------
! define parameters 
!-----------------------------------------------------------------
!-----------------------------------------------------------------
! ammonium nitrate (NH4NO3)
!-----------------------------------------------------------------
  zeta_inv = .25
  nu_i(1)  = 4
  z_i (1)  = 1.
!-----------------------------------------------------------------
! ammonium sulfate
!-----------------------------------------------------------------
  nu_i(2)  = 4
  z_i (2)  = 0.5

end subroutine aerosols_inti

subroutine aerosols_formation( lat, ip, pmid, tfld, rh, &
                               xhnm, vmr, plonl )

  use mo_grid, only    : plev, pcnstm1
  use chem_mods, only  : adv_mass
  use mo_histout, only : outfld, moz_file_cnt

  implicit none

!-----------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------
  integer, intent(in)                      :: lat                     ! latitude index
  integer, intent(in)                      :: ip                      ! longitude tile index
  integer, intent(in)                      :: plonl                   ! longitude tile dimension
  real, intent(inout)                      :: vmr(plonl,plev,pcnstm1) ! species concentrations (mol/mol)
  real, intent(in)                         :: xhnm(plonl,plev)        ! total atms density (/cm**3)
  real, intent(in)                         :: tfld(plonl,plev)        ! temperature (K)
  real, intent(in)                         :: rh(plonl,plev)          ! relative humidity
  real, intent(in)                         :: pmid(plonl,plev)        ! midpoint pressure (Pa)

!-----------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------
  integer                        :: i, j, k, n
  integer                        :: file
  integer                        :: domain_number               ! concentration domain
  real                           :: sulfate_state               ! fraction of sulfate neutralized by ammonia
  real                           :: log_t
  real                           :: ti
  real                           :: tnh3                        ! total ammonia
  real                           :: fnh3                        ! free ammonia
  real                           :: rhd                         ! relative humidity of deliquescence
  real                           :: gamma                       ! activity coefficient
  real                           :: ssm_nh4no3                  ! single solute molality for NH4NO3
  real                           :: ta                          ! total ammonia
  real                           :: tn                          ! total nitrate
  real                           :: kp                          ! equilibrium constant
  real                           :: nh4no3                      ! ammonium nitrate produced
  real                           :: tso4(plonl,plev)            ! total sulfate
  real                           :: thno3(plonl,plev)           ! total nitric acid

level_loop : &
  do k = 1,plev
long_loop : &
    do i = 1,plonl
!-----------------------------------------------------------------
! 	... compute total concentrations
!-----------------------------------------------------------------
      tnh3      = vmr(i,k,nh3_ndx) + vmr(i,k,nh4_ndx)
      tso4(i,k) = vmr(i,k,so4_ndx)
!-----------------------------------------------------------------
! 	... define concentration domain
!-----------------------------------------------------------------
      if( tnh3 < tso4(i,k) ) then
        domain_number = 4
        sulfate_state = 1.0
      else if( tnh3 < 2.*tso4(i,k) ) then
        domain_number = 3
        sulfate_state = 1.5
      else
        domain_number = 2
        sulfate_state = 2.0
      end if
!-----------------------------------------------------------------
! 	... define free ammonia (ammonia available for ammonium nitrate production)
!-----------------------------------------------------------------
      fnh3 = tnh3 - sulfate_state * tso4(i,k)
      fnh3 = max( 0.,fnh3 )
!-----------------------------------------------------------------
! 	... convert initial concentrations to ppbv
!-----------------------------------------------------------------
      tso4(i,k)  = tso4(i,k) * 1.e9
      tnh3       = tnh3 * 1.e9
      fnh3       = fnh3 * 1.e9
      thno3(i,k) = (vmr(i,k,hno3_ndx) + vmr(i,k,nh4no3_ndx)) * 1.e9

!-----------------------------------------------------------------
! 	... compute relative humidity of deliquescence (%) for NH4NO3
!           (Seinfeld and Pandis, p532)
!-----------------------------------------------------------------
      ti    = 1./tfld(i,k)
      rhd   = 0.01 * exp( 1.6954 + 723.7*ti )
      log_t = log( tfld(i,k)/298. )
      if( rh(i,k) < rhd ) then
!-----------------------------------------------------------------
! 	... crystalline ammonium nitrate
!           compute equilibrium constant
!-----------------------------------------------------------------
        kp = exp( 84.6 - 24220.*ti - 6.1*log_t )
      else
!-----------------------------------------------------------------
! 	... aqueous phase ammonium nitrate
!           compute activity coefficients (from Menzger et al.)
!-----------------------------------------------------------------
        n = domain_number
        gamma = (rh(i,k)**n/(1000/n*(1. - rh(i,k)) + n))**zeta_inv
!-----------------------------------------------------------------
! 	... compute single solute molality for NH4NO3
!-----------------------------------------------------------------
        ssm_nh4no3 = (1000. * 0.81 * nu_i(1) * (1./rh(i,k) - 1.)/80.)**z_i(1)
!-----------------------------------------------------------------
! 	... compute equilibrium constant
!-----------------------------------------------------------------
        kp = (gamma*ssm_nh4no3)**2 * exp( 53.19 - 15850.62*ti + 11.51*log_t )
      end if

!-----------------------------------------------------------------
! 	... calculate production of NH4NO3 (in ppbv) using Seinfeld and Pandis (p534, 1998)
!-----------------------------------------------------------------
      ta = fnh3
      tn = thno3(i,k)
      nh4no3 = 0.5 * (ta + tn - sqrt(max(0.,(ta+tn)**2 - 4.*(ta*tn - kp))));
      nh4no3 = max( 0.,nh4no3 )
!-----------------------------------------------------------------
! 	... reset concentrations according to equilibrium calculation
!-----------------------------------------------------------------
      vmr(i,k,nh4no3_ndx)   = nh4no3
      vmr(i,k,nh3_ndx   )   = max( 0.,(fnh3 - nh4no3) )
      vmr(i,k,nh4_ndx   )   = max( 0.,(tnh3 - (fnh3 - nh4no3)) )
      vmr(i,k,hno3_ndx  )   = max( 0.,(thno3(i,k) - nh4no3) )
      vmr(i,k,so4_ndx   )   =         tso4(i,k)
!-----------------------------------------------------------------
! 	... convert from ppbv to vmr
!-----------------------------------------------------------------
      vmr(i,k,nh4no3_ndx)   = vmr(i,k,nh4no3_ndx)   * 1.e-9
      vmr(i,k,nh3_ndx   )   = vmr(i,k,nh3_ndx   )   * 1.e-9
      vmr(i,k,nh4_ndx   )   = vmr(i,k,nh4_ndx   )   * 1.e-9
      vmr(i,k,hno3_ndx  )   = vmr(i,k,hno3_ndx  )   * 1.e-9
      vmr(i,k,so4_ndx   )   = vmr(i,k,so4_ndx   )   * 1.e-9
    end do long_loop
  end do level_loop

!-----------------------------------------------------------------
! 	... outputs
!-----------------------------------------------------------------
  tso4(:,:)  = tso4(:,:)  * 1.e-9
  thno3(:,:) = thno3(:,:) * 1.e-9
  do file = 1,moz_file_cnt
     call outfld( 'TSO4_VMR' ,tso4, plonl, ip, lat, file )
     call outfld( 'TNO3_VMR' ,thno3, plonl, ip, lat, file )
  end do
  
end subroutine aerosols_formation


end module mo_aerosols
