      module mo_airmas

      private
      public :: airmas

      contains

      subroutine airmas( z, zen, dsdh, nid, cz, &
                         vcol, scol )
!-----------------------------------------------------------------------------
!   purpose:
!   calculate vertical and slant air columns, in spherical geometry, as a
!   function of altitude.
!-----------------------------------------------------------------------------
!   parameters:
!   nz      - integer, number of specified altitude levels in the working (i)
!             grid
!   z       - real, specified altitude working grid (km)                  (i)
!   zen     - real, solar zenith angle (degrees)                          (i)
!   dsdh    - real, slant path of direct beam through each layer crossed  (o)
!             when travelling from the top of the atmosphere to layer i;
!             dsdh(i,j), i = 0..nz-1, j = 1..nz-1
!   nid     - integer, number of layers crossed by the direct beam when   (o)
!             travelling from the top of the atmosphere to layer i;
!             nid(i), i = 0..nz-1
!   vcol    - real, output, vertical air column, molec cm-2, above level iz
!   scol    - real, output, slant air column in direction of sun, above iz
!             also in molec cm-2
!-----------------------------------------------------------------------------

      use mo_params, only : largest
      use mo_grid,   only : plev, plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: nid(0:plev)
      real, intent(in)    :: z(plevp)
      real, intent(in)    :: zen
      real, intent(in)    :: dsdh(0:plev,plev)
      real, intent(in)    :: cz(plevp)
      real, intent(out)   :: vcol(plevp)
      real, intent(out)   :: scol(plevp)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: id, j
      real    :: sum, ssum, vsum, ratio

!-----------------------------------------------------------------------------
! 	... calculate vertical and slant column from each level: work downward
!-----------------------------------------------------------------------------
      vsum = 0.
      ssum = 0.
      do id = 1,plev
         vsum = vsum + cz(plevp-id)
         vcol(plevp-id) = vsum
         sum = 0.
         if( nid(id) < 0 ) then
            sum = largest
         else
!-----------------------------------------------------------------------------
! 	... single pass layers:
!-----------------------------------------------------------------------------
            do j = 1,min( nid(id),id )
               sum = sum + cz(plevp-j)*dsdh(id,j)
            end do
!-----------------------------------------------------------------------------
! 	... double pass layers:
!-----------------------------------------------------------------------------
            do j = min( nid(id),id )+1,nid(id)
               sum = sum + 2.*cz(plevp-j)*dsdh(id,j)
            end do
         end if
         scol(plevp - id) = sum
      end do

!-----------------------------------------------------------------------------
! 	... special section to set scol(plevp)
!-----------------------------------------------------------------------------
      if( scol(plev-1) /= 0. ) then
         ratio       = scol(plev)/scol(plev-1)
	 scol(plevp) = ratio * scol(plev)
      else
         scol(plevp) = 0.
      end if

      end subroutine airmas

      end module mo_airmas
