
      module mo_airplane
!--------------------------------------------------------------------
!	... airplane insitu emission sources
!--------------------------------------------------------------------

      implicit none

      save 

      integer           :: nlev                  ! vertical levels in airplane grid
      real              :: min_alt               ! min level in airplane grid (km)
      real              :: max_alt               ! max level in airplane grid (km)
      real, allocatable :: air_altitude(:)       ! airplane src vertical grid (km)
      real, allocatable :: del_alti(:)           ! airplane src inverse vertical delta (km)
      real, allocatable :: pno(:,:,:,:)          ! airplane no layer production (molec/cm^2/s)
      real, allocatable :: pco(:,:,:,:)          ! airplane co layer production (molec/cm^2/s)
      real, allocatable :: pch4(:,:,:,:)         ! airplane ch4 layer production (molec/cm^2/s)

      private
      public :: airpl_src
      public :: min_alt, max_alt, nlev
      public :: air_altitude, del_alti
      public :: pno, pco, pch4

      contains

      subroutine airpl_src( filename, lpath, mspath, plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... initialize airplane emissions
!	    note: the emissions are read in in units of molecules/cm**2/s
!	          on a vertically resolved grid.
!		  conversion to units of molec/cm**3/s is done in setext
!-----------------------------------------------------------------------

      use netcdf
      use mo_mpi
      use mo_constants,  only : phi, lam, d2r, pi, rearth, latwts
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits
      use mo_grid,       only : plon, plat
      use mo_file_utils, only : open_netcdf_file
      use chem_mods,     only : adv_mass
      use mo_chem_utls,  only : get_spc_ndx

      implicit none

!-----------------------------------------------------------------------
! 	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in)          ::   plonl              ! lon tile dim
      integer, intent(in)          ::   platl              ! lat tile dim
      integer, intent(in)          ::   pplon              ! lon tile count
      character(len=*), intent(in) ::   lpath
      character(len=*), intent(in) ::   mspath
      character(len=*), intent(in) ::   filename

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: ios, k, jl, ju, j
      integer :: jlim(2)
      integer :: nlat, nlon, ndims
      integer :: ncid, vid, ierr
      integer :: dimid_lat, dimid_lon, dimid_lev
      integer :: co_ndx, ch4_ndx
      integer :: gndx
      integer :: dimid(3)
      real, allocatable :: lat(:), lon(:)
      real, allocatable :: pno_in(:,:,:), pco_in(:,:,:), pch4_in(:,:,:)
      real, dimension(plon,platl)     :: wrk2d
      real    :: total(3) = 0., total_wrk
      real    :: seq, sf(plat), factor

!-----------------------------------------------------------------------
!	... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( filename, lpath, mspath )

!-----------------------------------------------------------------------
!       ... get grid dimensions from file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'airpl_src: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'airpl_src: failed to get length of dimension lat' )
      allocate( lat(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'airpl_src: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'airpl_src: failed to read variable lat' )
      lat(:nlat) = lat(:nlat) * d2r
 
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                         'airpl_src: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'airpl_src: failed to get length of dimension lon' )
      allocate( lon(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: lon allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'airpl_src: failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                         'airpl_src: failed to read variable lon' )
      lon(:nlon) = lon(:nlon) * d2r
 
      call handle_ncerr( nf_inq_dimid( ncid, 'altitude', dimid_lev ), &
                         'airpl_src: failed to find dimension altitude' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lev, nlev ), &
                         'airpl_src: failed to get length of dimension altitude' )
      allocate( air_altitude(nlev+1),del_alti(nlev),stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: air_altitude,del_alti allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'altitude', vid ), &
                         'airpl_src: failed to find variable altitude' )
      call handle_ncerr( nf_get_var_double( ncid, vid, air_altitude(1:nlev) ), &
                         'airpl_src: failed to read variable altitude' )
      air_altitude(nlev+1) = air_altitude(nlev) + (air_altitude(nlev) - air_altitude(nlev-1))
      min_alt              = minval( air_altitude(:) )
      max_alt              = maxval( air_altitude(:) )
      del_alti(:)          = 1./(air_altitude(2:nlev+1) - air_altitude(1:nlev))

!-----------------------------------------------------------------------
!       ... set up regridding
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, plat, &
                          nlon, plon, &
                          lon,  lam, &
                          lat,  phi, &
                          0, platl, &
                          do_lons=.true.,do_lats=.true. )
      deallocate( lat, lon, stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: failed to deallocate lat,lon; ierr = ',ierr
         call endrun
      end if
      jl   = base_lat + 1
      ju   = base_lat + platl
      if( gndx /= 0 )then
         jlim = regrid_lat_limits( gndx )
      else
         jlim = (/ jl,ju /)
      end if
      allocate( pno_in( nlon, jlim(1):jlim(2), nlev ), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: pno_in allocation error = ',ierr
         call endrun
      end if
      allocate( pco_in( nlon, jlim(1):jlim(2), nlev ), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: pco_in allocation error = ',ierr
         call endrun
      end if
      allocate( pch4_in( nlon, jlim(1):jlim(2), nlev ), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: pch4_in allocation error = ',ierr
         call endrun
      end if
      allocate( pno( plonl, platl, pplon, nlev ), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: pno allocation error = ',ierr
         call endrun
      end if
      allocate( pco( plonl, platl, pplon, nlev ), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: pco allocation error = ',ierr
         call endrun
      end if
      allocate( pch4( plonl, platl, pplon, nlev ), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'airpl_src: pch4 allocation error = ',ierr
         call endrun
      end if

!-----------------------------------------------------------------------
!	... read emissions
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'nox', vid ), &
                         'airpl_src: failed to get id for variable nox' )
      call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                         'airpl_src: failed to get number of dimensions for variable nox' )
      if ( ndims /= 3 ) then
         write(*,*) 'airpl_src: variable nox has ndims = ',ndims,', expecting 3'
         call endrun
      end if
      call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                         'airpl_src: failed to get dimension ids for variable nox' )
      if( dimid(1) /= dimid_lon  .or. dimid(2) /= dimid_lat .or.  dimid(3) /= dimid_lev ) then
            write(*,*) 'airpl_src: dimensions in wrong order for variable nox'
            write(*,*) '...      expecting (lon, lat, lev)'
            call endrun
      end if
      call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                             (/ 1, jlim(1), 1/), &                   ! start
                                             (/ nlon, jlim(2)-jlim(1)+1, nlev /), &   ! count
                                             pno_in ), &
                         'airpl_src: failed to read variable nox' )

      call handle_ncerr( nf_inq_varid( ncid, 'co', vid ), &
                         'airpl_src: failed to get id for variable co' )
      call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                         'airpl_src: failed to get number of dimensions for variable co' )
      if ( ndims /= 3 ) then
         write(*,*) 'read_sflx: variable co has ndims = ',ndims,', expecting 3'
         call endrun
      end if
      call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                         'airpl_src: failed to get dimension ids for variable co' )
      if ( dimid(1) /= dimid_lon .or. dimid(2) /= dimid_lat .or. dimid(3) /= dimid_lev ) then
            write(*,*) 'airpl_src: dimensions in wrong order for variable co'
            write(*,*) '...      expecting (lon, lat, lev)'
            call endrun
      end if
      call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                             (/ 1, jlim(1), 1/), &                   ! start
                                             (/ nlon, jlim(2)-jlim(1)+1, nlev /), &  ! count
                                             pco_in ), &
                         'airpl_src: failed to read variable co' )
       pch4_in(:,:,:) = 0.

      call handle_ncerr( nf_close( ncid ), 'airpl_src: failed to close netcdf file' )

!-----------------------------------------------------------------------
!	... regrid emissions
!-----------------------------------------------------------------------
      do k = 1,nlev
         call regrid_2d( pno_in(:,jlim(1):jlim(2),k), wrk2d, gndx, jl, ju, do_poles=.true. )
	 pno(:,:,:,k) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/ 1, 3, 2/) )
         call regrid_2d( pco_in(:,jlim(1):jlim(2),k), wrk2d, gndx, jl, ju, do_poles=.true. )
	 pco(:,:,:,k) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/ 1, 3, 2/) )
         call regrid_2d( pch4_in(:,jlim(1):jlim(2),k), wrk2d, gndx, jl, ju, do_poles=.true. )
	 pch4(:,:,:,k) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/ 1, 3, 2/) )
      end do
!-----------------------------------------------------------------------
!       ... get global emission from this source
!-----------------------------------------------------------------------
      seq = 2.*pi*1.e4*rearth**2/real(plon)
      do j = 1,plat
         sf(j) = seq*latwts(j)
      end do

      factor = 86400. * 365. &   ! sec / year
             / 6.022e23 &        ! molec / mole
             * 1.e-12            ! tg / g

      total(:) = 0.
      do j = 1,platl
         total(1) = total(1) &
                  + sum( pno( :plonl, j, :pplon, :nlev ) ) * sf(base_lat+j)
         total(2) = total(2) &
                  + sum( pco( :plonl, j, :pplon, :nlev ) ) * sf(base_lat+j)
         total(3) = total(3) &
                  + sum( pch4( :plonl, j, :pplon, :nlev ) ) * sf(base_lat+j)
      end do
!-----------------------------------------------------------------------
!       ... convert totals from molec cm^-2 s^-1 to tg y^-1
!-----------------------------------------------------------------------
      total(1) = total(1) * 14.00674 * factor
      co_ndx = get_spc_ndx( 'CO' )
      if( co_ndx > 0 ) then
         total(2) = total(2) * adv_mass(co_ndx) * factor
      end if
      ch4_ndx = get_spc_ndx( 'CH4' )
      if( ch4_ndx > 0 ) then
         total(3) = total(3) * adv_mass(ch4_ndx) * factor
      end if
#ifdef USE_MPI
      do j = 1,3
         call mpi_allreduce( total(j), total_wrk, 1, mpi_double_precision, mpi_sum, mpi_comm_comp, ierr )
         if( ierr /= mpi_success ) then
            write(*,*) 'airpl_src: mpi_allreduce for total failed; error = ',ierr
            call endrun
         end if
         total(j) = total_wrk
      end do
#endif
      if (masternode) then
         write(*,'('' airpl_src aircraft emissions: '',a6,'' = '',f10.3,1x,a6)') 'no',total(1),'tgn/y'
	 if( co_ndx > 0 ) then
            write(*,'('' airpl_src aircraft emissions: '',a6,'' = '',f10.3,1x,a6)') 'co',total(2),'tg/y'
	 end if
	 if( ch4_ndx > 0 ) then
            write(*,'('' airpl_src aircraft emissions: '',a6,'' = '',f10.3,1x,a6)') 'ch4',total(3),'tg/y'
	 end if
      end if

      end subroutine airpl_src

      end module mo_airplane
