
      module mo_drydep
!---------------------------------------------------------------------
!       ... Dry deposition velocity input data and code for netcdf input
!---------------------------------------------------------------------
! jfl addition of h2so4, new PAN drydep, fix to season, NH3
! XNOX, O3A, COAA, etc. removed
! removed h2so4 (LKE 5/1/07)
!---------------------------------------------------------------------
 
      use mo_grid, only : pcnst

      implicit none

      save

      interface drydep
         module procedure drydep_table
         module procedure drydep_xactive
      end interface

      private
      public  :: dvel_inti_table, dvel_inti_xactive, drydep, set_soilw, chk_soilw

      real              :: dels
      real, allocatable :: days(:)              ! day of year for soilw
      real, allocatable :: tab_soilw(:,:,:,:)   ! interpolated soil moisture
      real, allocatable :: dvel(:,:,:,:,:), &   ! depvel array interpolated to model grid
                           dvel_interp(:,:,:,:) ! depvel array interpolated to grid and time
      real, allocatable :: sea_ice(:,:,:)       ! sea ice interpolated to correct time
      integer :: last, next                     ! day indicies
      integer :: ndays                          ! # of days in soilw file
      integer :: map(pcnst)                     ! indices for drydep species
      integer :: nspecies                       ! number of depvel species in input file
      integer :: pan_ndx, mpan_ndx, no2_ndx, hno3_ndx, o3_ndx, &
                 h2o2_ndx, onit_ndx, onitr_ndx, ch4_ndx, ch2o_ndx, &
                 ch3ooh_ndx, pooh_ndx, ch3coooh_ndx, c2h5ooh_ndx, &
                 c3h7ooh_ndx, rooh_ndx, ch3cocho_ndx, co_ndx, ch3coch3_ndx, &
                 no_ndx, ho2no2_ndx, glyald_ndx, hyac_ndx, ch3oh_ndx, c2h5oh_ndx, &
                 hydrald_ndx, h2_ndx, Pb_ndx, o3s_ndx, o3inert_ndx, macrooh_ndx, &
                 xooh_ndx, ch3cho_ndx, isopooh_ndx
      integer :: alkooh_ndx, mekooh_ndx, tolooh_ndx, terpooh_ndx, ch3cooh_ndx
      integer :: soa_ndx, so4_ndx, cb1_ndx, cb2_ndx, oc1_ndx, oc2_ndx, nh3_ndx, nh4no3_ndx, &
                 sa1_ndx, sa2_ndx, sa3_ndx, sa4_ndx, nh4_ndx
      logical :: alkooh_dd, mekooh_dd, tolooh_dd, terpooh_dd, ch3cooh_dd
      logical :: soa_dd, so4_dd, cb1_dd, cb2_dd, oc1_dd, oc2_dd, nh3_dd, nh4no3_dd, &
                 sa1_dd, sa2_dd, sa3_dd, sa4_dd, nh4_dd
      integer :: o3_tab_ndx = -1, h2o2_tab_ndx = -1, &
                 ch3ooh_tab_ndx = -1, co_tab_ndx = -1, &
                 ch3cho_tab_ndx = -1
      logical :: pan_dd, mpan_dd, no2_dd, hno3_dd, o3_dd, isopooh_dd, ch4_dd,&
                 h2o2_dd, onit_dd, onitr_dd, ch2o_dd, macrooh_dd, xooh_dd, &
                 ch3ooh_dd, pooh_dd, ch3coooh_dd, c2h5ooh_dd, ch3cho_dd, c2h5oh_dd, &
                 c3h7ooh_dd, rooh_dd, ch3cocho_dd, co_dd, ch3coch3_dd, &
                 glyald_dd, hyac_dd, ch3oh_dd, hydrald_dd, h2_dd, Pb_dd, o3s_dd, o3inert_dd
      logical :: o3_in_tab = .false., h2o2_in_tab = .false., ch3ooh_in_tab = .false., &
                 co_in_tab = .false., ch3cho_in_tab = .false.

      integer, parameter :: n_land_type = 11

      real, parameter    :: small_value = 1.e-36
      real, parameter    :: large_value = 1.e36
      real, parameter    :: diffm       = 1.789e-5
      real, parameter    :: diffk       = 1.461e-5
      real, parameter    :: difft       = 2.060e-5
      real, parameter    :: vonkar      = 0.378
      real, parameter    :: ric         = 0.2
      real, parameter    :: r           = 287.04
      real, parameter    :: cp          = 1004.
      real, parameter    :: grav        = 9.81
      real, parameter    :: p00         = 100000.
      real, parameter    :: wh2o        = 18.0153
      real, parameter    :: ph          = 1.e-5
      real, parameter    :: ph_inv      = 1./ph
      real, parameter    :: rovcp = r/cp

      integer :: so2_ndx
      integer :: mapping(pcnst) = -99
      integer, allocatable :: index_season_lai(:,:,:)

      real    :: crb
      real    :: foxd(pcnst)     = small_value
      real    :: drat(pcnst)     = small_value
      real, allocatable    :: fraction_landuse(:,:,:,:)

      logical :: has_dvel(pcnst) = .false.

      contains

      subroutine dvel_inti_table( plonl, platl, pplon )
!---------------------------------------------------------------------------
!       ... Initialize module, depvel arrays, and a few other variables.
!           The depvel fields will be linearly interpolated to the correct time
!---------------------------------------------------------------------------

      use netcdf
      use mo_constants,  only : d2r, r2d, pi, phi, lam
      use mo_grid,       only : plon, plat
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : dvel_flsp
      use mo_mpi,        only : base_lat, masternode
      use mo_regrider,   only : regrid_inti, regrid_lat_limits, regrid_2d
      use m_tracname,    only : tracnam
      use mo_chem_utls,  only : get_spc_ndx, has_drydep
      use mo_charutl,    only : glc

      implicit none

!---------------------------------------------------------------------------
!       ... Dummy arguments
!---------------------------------------------------------------------------
      integer, intent(in) :: platl
      integer, intent(in) :: plonl
      integer, intent(in) :: pplon

!---------------------------------------------------------------------------
!       ... Local variables
!---------------------------------------------------------------------------
      integer :: ncid, vid, vid_dvel, nlat, nlon, nmonth, ndims
      integer :: dimid_lat, dimid_lon, dimid_species, dimid_time
      integer :: dimid(4), count(4), start(4)
      integer :: jlim(2), jl, ju
      integer :: gndx, m, ispecies, nchar, ierr
      real    :: scale_factor
      real    :: wrk2d(plon,platl)
      real, allocatable :: dvel_lats(:), dvel_lons(:)
      real, allocatable :: dvel_in(:,:,:,:)                          ! input depvel array
      character(len=50) :: units
      character(len=80) :: filename, lpath, rpath
      character(len=20), allocatable :: species_names(:)             ! names of depvel species
      logical :: found

!---------------------------------------------------------------------------
!       ... Setup species maps
!---------------------------------------------------------------------------
      pan_ndx  = get_spc_ndx( 'PAN')
      mpan_ndx = get_spc_ndx( 'MPAN')
      no2_ndx  = get_spc_ndx( 'NO2')
      hno3_ndx = get_spc_ndx( 'HNO3')
      co_ndx   = get_spc_ndx( 'CO')
      o3_ndx   = get_spc_ndx( 'O3')
      if( o3_ndx < 1 ) then
         o3_ndx = get_spc_ndx( 'OX')
      end if
      h2o2_ndx     = get_spc_ndx( 'H2O2')
      onit_ndx     = get_spc_ndx( 'ONIT')
      onitr_ndx    = get_spc_ndx( 'ONITR')
      ch4_ndx      = get_spc_ndx( 'CH4')
      ch2o_ndx     = get_spc_ndx( 'CH2O')
      ch3ooh_ndx   = get_spc_ndx( 'CH3OOH')
      ch3cho_ndx   = get_spc_ndx( 'CH3CHO')
      ch3cocho_ndx = get_spc_ndx( 'CH3COCHO')
      pooh_ndx     = get_spc_ndx( 'POOH')
      ch3coooh_ndx = get_spc_ndx( 'CH3COOOH')
      c2h5ooh_ndx  = get_spc_ndx( 'C2H5OOH')
      c3h7ooh_ndx  = get_spc_ndx( 'C3H7OOH')
      rooh_ndx     = get_spc_ndx( 'ROOH')
      ch3coch3_ndx = get_spc_ndx( 'CH3COCH3')
      no_ndx       = get_spc_ndx( 'NO')
      ho2no2_ndx   = get_spc_ndx( 'HO2NO2')
      glyald_ndx   = get_spc_ndx( 'GLYALD')
      hyac_ndx     = get_spc_ndx( 'HYAC')
      ch3oh_ndx    = get_spc_ndx( 'CH3OH')
      c2h5oh_ndx   = get_spc_ndx( 'C2H5OH')
      macrooh_ndx  = get_spc_ndx( 'MACROOH')
      isopooh_ndx  = get_spc_ndx( 'ISOPOOH')
      xooh_ndx     = get_spc_ndx( 'XOOH')
      hydrald_ndx  = get_spc_ndx( 'HYDRALD')
      h2_ndx       = get_spc_ndx( 'H2')
      Pb_ndx       = get_spc_ndx( 'Pb')
      o3s_ndx      = get_spc_ndx( 'O3S')
      o3inert_ndx  = get_spc_ndx( 'O3INERT')
      alkooh_ndx  = get_spc_ndx( 'ALKOOH')
      mekooh_ndx  = get_spc_ndx( 'MEKOOH')
      tolooh_ndx  = get_spc_ndx( 'TOLOOH')
      terpooh_ndx = get_spc_ndx( 'TERPOOH')
      ch3cooh_ndx = get_spc_ndx( 'CH3COOH')
      soa_ndx     = get_spc_ndx( 'SOA' )
      so4_ndx     = get_spc_ndx( 'SO4' )
      cb1_ndx     = get_spc_ndx( 'CB1' )
      cb2_ndx     = get_spc_ndx( 'CB2' )
      oc1_ndx     = get_spc_ndx( 'OC1' )
      oc2_ndx     = get_spc_ndx( 'OC2' )
      nh3_ndx     = get_spc_ndx( 'NH3' )
      nh4no3_ndx  = get_spc_ndx( 'NH4NO3' )
      sa1_ndx     = get_spc_ndx( 'SA1' )
      sa2_ndx     = get_spc_ndx( 'SA2' )
      sa3_ndx     = get_spc_ndx( 'SA3' )
      sa4_ndx     = get_spc_ndx( 'SA4' )
      nh4_ndx     = get_spc_ndx( 'NH4' )
      alkooh_dd  = has_drydep( 'ALKOOH')
      mekooh_dd  = has_drydep( 'MEKOOH')
      tolooh_dd  = has_drydep( 'TOLOOH')
      terpooh_dd = has_drydep( 'TERPOOH')
      ch3cooh_dd = has_drydep( 'CH3COOH')
      soa_dd     = has_drydep( 'SOA' )
      so4_dd     = has_drydep( 'SO4' )
      cb1_dd     = has_drydep( 'CB1' )
      cb2_dd     = has_drydep( 'CB2' )
      oc1_dd     = has_drydep( 'OC1' )
      oc2_dd     = has_drydep( 'OC2' )
      nh3_dd     = has_drydep( 'NH3' )
      nh4no3_dd  = has_drydep( 'NH4NO3' )
      sa1_dd     = has_drydep( 'SA1' ) 
      sa2_dd     = has_drydep( 'SA2' )
      sa3_dd     = has_drydep( 'SA3' ) 
      sa4_dd     = has_drydep( 'SA4' )
      nh4_dd     = has_drydep( 'NH4' ) 
      pan_dd  = has_drydep( 'PAN')
      mpan_dd = has_drydep( 'MPAN')
      no2_dd  = has_drydep( 'NO2')
      hno3_dd = has_drydep( 'HNO3')
      co_dd   = has_drydep( 'CO')
      o3_dd   = has_drydep( 'O3')
      if( .not. o3_dd ) then
         o3_dd = has_drydep( 'OX')
      end if
      h2o2_dd     = has_drydep( 'H2O2')
      onit_dd     = has_drydep( 'ONIT')
      onitr_dd    = has_drydep( 'ONITR')
      ch4_dd      = has_drydep( 'CH4')
      ch2o_dd     = has_drydep( 'CH2O')
      ch3ooh_dd   = has_drydep( 'CH3OOH')
      ch3cho_dd   = has_drydep( 'CH3CHO')
      c2h5oh_dd   = has_drydep( 'C2H5OH')
      ch3cocho_dd = has_drydep( 'CH3COCHO')
      pooh_dd     = has_drydep( 'POOH')
      ch3coooh_dd = has_drydep( 'CH3COOOH')
      c2h5ooh_dd  = has_drydep( 'C2H5OOH')
      c3h7ooh_dd  = has_drydep( 'C3H7OOH')
      rooh_dd     = has_drydep( 'ROOH')
      ch3coch3_dd = has_drydep( 'CH3COCH3')
      glyald_dd   = has_drydep( 'GLYALD')
      hyac_dd     = has_drydep( 'HYAC')
      ch3oh_dd    = has_drydep( 'CH3OH')
      macrooh_dd  = has_drydep( 'MACROOH')
      isopooh_dd  = has_drydep( 'ISOPOOH')
      xooh_dd     = has_drydep( 'XOOH')
      hydrald_dd  = has_drydep( 'HYDRALD')
      h2_dd       = has_drydep( 'H2')
      Pb_dd       = has_drydep( 'Pb')
      o3s_dd      = has_drydep( 'O3S')
      o3inert_dd  = has_drydep( 'O3INERT')

      write(*,*) 'dvel_inti: diagnostics'
      write(*,'(10i5)') pan_ndx, mpan_ndx, no2_ndx, hno3_ndx, o3_ndx, &
                        h2o2_ndx, onit_ndx, onitr_ndx, ch4_ndx, ch2o_ndx, &
                        ch3ooh_ndx, pooh_ndx, ch3coooh_ndx, c2h5ooh_ndx, &
                        c3h7ooh_ndx, rooh_ndx, ch3cocho_ndx, co_ndx, ch3coch3_ndx, &
                        no_ndx, ho2no2_ndx, glyald_ndx, hyac_ndx, ch3oh_ndx, c2h5oh_ndx, &
                        hydrald_ndx, h2_ndx, Pb_ndx, o3s_ndx, o3inert_ndx, macrooh_ndx, &
                        xooh_ndx, ch3cho_ndx, isopooh_ndx
      write(*,*) pan_dd, mpan_dd, no2_dd, hno3_dd, o3_dd, isopooh_dd, ch4_dd,&
                        h2o2_dd, onit_dd, onitr_dd, ch2o_dd, macrooh_dd, xooh_dd, &
                        ch3ooh_dd, pooh_dd, ch3coooh_dd, c2h5ooh_dd, ch3cho_dd, c2h5oh_dd, &
                        c3h7ooh_dd, rooh_dd, ch3cocho_dd, co_dd, ch3coch3_dd, &
                        glyald_dd, hyac_dd, ch3oh_dd, hydrald_dd, h2_dd, Pb_dd, o3s_dd, o3inert_dd

!---------------------------------------------------------------------------
!       ... Open NetCDF file
!---------------------------------------------------------------------------
      lpath    = dvel_flsp%local_path
      rpath    = dvel_flsp%remote_path
      filename = dvel_flsp%nl_filename
      ncid     = open_netcdf_file( filename, lpath, rpath )
      if( masternode ) then
         write(*,*) 'dvel_inti: Opened netcdf file ',trim(lpath) // trim(filename)
      end if

!---------------------------------------------------------------------------
!       ... Get variable ID for dep vel array
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'dvel', vid_dvel ), 'dvel_inti: dvel not found in depvel input file' )

!---------------------------------------------------------------------------
!       ... Inquire about dimensions
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), 'dvel_inti: getting lon dimension ID' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), 'dvel_inti: getting nlon' )
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), 'dvel_inti: getting lat dimension ID' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), 'dvel_inti: getting nlat' )
      call handle_ncerr( nf_inq_dimid( ncid, 'species', dimid_species ), 'dvel_inti: getting species dimension ID' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_species, nspecies ), 'dvel_inti: getting nspecies' )
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid_time ), 'dvel_inti: getting time dimension ID' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_time, nmonth ), 'dvel_inti: getting nmonth' )
      if( masternode ) then
         write(*,*) 'dvel_inti: dimensions (nlon,nlat,nspecies,nmonth) = ',nlon,nlat,nspecies,nmonth
      end if

!---------------------------------------------------------------------------
!       ... Check dimensions of dvel variable. Must be (lon, lat, species, month).
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_varndims( ncid, vid_dvel, ndims ), &
                         'dvel_inti: Failed to get ndims for dvel' )
      if( ndims /= 4 ) then
         write(*,*) 'dvel_inti: dvel has ',ndims,' dimensions. Expecting 4.'
         call endrun
      end if
      call handle_ncerr( nf_inq_vardimid( ncid, vid_dvel, dimid ), &
                         'dvel_inti: Failed to get dimension IDs for dvel' )
      if( dimid(1) /= dimid_lon .or. dimid(2) /= dimid_lat .or. &
          dimid(3) /= dimid_species .or. dimid(4) /= dimid_time ) then
         write(*,*) 'dvel_inti: Dimensions in wrong order for dvel'
         write(*,*) '...      Expecting (lon, lat, species, month)'
         call endrun
      end if

!---------------------------------------------------------------------------
!       ... Allocate depvel lats, lons and read
!---------------------------------------------------------------------------
      allocate( dvel_lats(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: Failed to allocate dvel_lats vector'
         call endrun
      end if
      allocate( dvel_lons(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: Failed to allocate dvel_lons vector'
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), 'dvel_inti: lat dim not in dvel file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, dvel_lats ), 'dvel_inti: getting dvel lats' )
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), 'dvel_inti: lon dim not in dvel file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, dvel_lons ), 'dvel_inti: getting dvel lons' )

!---------------------------------------------------------------------------
!       ... Set the transform from inputs lats to simulation lats
!---------------------------------------------------------------------------
      if( masternode ) then
         write(*,*) 'dvel_inti: source lats (deg)'
         write(*,'(10F8.2)') dvel_lats
         write(*,*) 'dvel_inti: target lats (deg)'
         write(*,'(10F8.2)') phi*r2d
         write(*,*) 'dvel_inti: source lons (deg)'
         write(*,'(10F8.2)') dvel_lons
         write(*,*) 'dvel_inti: target lons (deg)'
         write(*,'(10F8.2)') lam*r2d
      end if

      dvel_lats(:nlat) = d2r * dvel_lats(:nlat)
      dvel_lons(:nlon) = d2r * dvel_lons(:nlon)
      gndx =  regrid_inti( nlat, plat, nlon, plon, dvel_lons, &
                           lam, dvel_lats, phi, 0, platl )
      write(*,*) 'dvel_inti: REGRID_INTI using grid number ',gndx
      jl = base_lat + 1
      ju = base_lat + platl
      if( gndx /= 0 ) then
         jlim(:) = regrid_lat_limits( gndx )
      else
         jlim(:) = (/ jl,ju /)
      end if
      write(*,*) 'dvel_inti: jlim',jlim(:)
      deallocate( dvel_lats, dvel_lons )

!---------------------------------------------------------------------------
!     	... Allocate dvel and read data from file
!---------------------------------------------------------------------------
      allocate( dvel_in(nlon,jlim(1):jlim(2),nspecies,nmonth), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: Failed to allocate dvel_in'
         call endrun
      end if
      start = (/ 1, jlim(1), 1, 1 /)
      count = (/ nlon, jlim(2)-jlim(1)+1, nspecies, nmonth /)
      call handle_ncerr( nf_get_vara_double( ncid, vid_dvel, start, count, dvel_in ), &
                         'dvel_inti: getting dvel' )

!---------------------------------------------------------------------------
!     	... Check units of deposition velocity. If necessary, convert to cm/s.
!---------------------------------------------------------------------------
      units(:) = ' '
      call handle_ncerr( nf_get_att_text( ncid, vid_dvel, 'units', units  ), 'dvel_inti: dvel units not found' )
      if( units(:GLC(units)) == 'm/s' ) then
#ifdef DEBUG
         write(*,*) 'dvel_inti: depvel units = m/s. Converting to cm/s'
#endif
         scale_factor = 100.
      elseif( units(:GLC(units)) == 'cm/s' ) then
#ifdef DEBUG
         write(*,*) 'dvel_inti: depvel units = cm/s'
#endif
         scale_factor = 1.
      else
#ifdef DEBUG
         write(*,*) 'dvel_inti: Warning! depvel units unknown = ',units(:GLC(units))
         write(*,*) '           ...      proceeding with scale_factor=1'
#endif
         scale_factor = 1.
      end if

!---------------------------------------------------------------------------
!     	... Regrid deposition velocities
!---------------------------------------------------------------------------
      allocate( dvel(plonl,platl,pplon,nspecies,nmonth), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: Failed to allocate dvel'
         call endrun
      end if
      do ispecies = 1,nspecies
         do m = 1,12
            call regrid_2d( dvel_in( :,jlim(1):jlim(2),ispecies,m ), wrk2d, gndx, jl, ju, &
                            do_poles=.true., scaling=scale_factor )
            dvel(:,:,:,ispecies,m) = RESHAPE( wrk2d, (/ plonl, platl, pplon /), order = (/1,3,2/) )
         end do
      end do
      deallocate( dvel_in )

!---------------------------------------------------------------------------
!     	... Read in species names and determine mapping to tracer numbers
!---------------------------------------------------------------------------
      allocate( species_names(nspecies), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: species_names allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'species_name', vid ), &
                         'dvel_inti: Getting species_name id' )
      call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                         'dvel_inti: Getting number of dimensions for species_name' )
      call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                         'dvel_inti: Getting dimensions for species_name' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid(1), nchar ), &
                         'dvel_inti: Getting dimension length' )
      map(:) = 0
      do ispecies = 1,nspecies
         start(:2) = (/ 1, ispecies /)
         count(:2) = (/ nchar, 1 /)
         species_names(ispecies)(:) = ' '
         call handle_ncerr( nf_get_vara_text( ncid, vid, start(:2), count(:2), species_names(ispecies) ), &
                            'dvel_inti: Getting species names' )
         if( species_names(ispecies) == 'O3' ) then
           o3_in_tab  = .true.
           o3_tab_ndx = ispecies
         else if( species_names(ispecies) == 'H2O2' ) then
           h2o2_in_tab  = .true.
           h2o2_tab_ndx = ispecies
         else if( species_names(ispecies) == 'CH3OOH' ) then
           ch3ooh_in_tab  = .true.
           ch3ooh_tab_ndx = ispecies
         else if( species_names(ispecies) == 'CO' ) then
           co_in_tab  = .true.
           co_tab_ndx = ispecies
         else if( species_names(ispecies) == 'CH3CHO' ) then
           ch3cho_in_tab  = .true.
           ch3cho_tab_ndx = ispecies
         end if
         found = .false.
         do m = 1,pcnst
            if( species_names(ispecies) == tracnam(m) .or. &
                (species_names(ispecies) == 'O3' .and. tracnam(m) == 'OX') .or. &
                (species_names(ispecies) == 'HNO4' .and. tracnam(m) == 'HO2NO2') ) then
               map(m) = ispecies
               found = .true.
#ifdef DEBUG
               if( masternode ) then
                  write(*,*) 'dvel_inti: ispecies, m, tracnam = ',ispecies,m,trim(tracnam(m))
               end if
#endif
               exit
            end if
         end do
         if( .not. found ) then
            write(*,*) 'dvel_inti: Warning! DVEL species ',trim(species_names(ispecies)),' not found'
         endif
      end do
      deallocate( species_names )

!---------------------------------------------------------------------------
!     	... Allocate dvel_interp array
!---------------------------------------------------------------------------
      allocate( dvel_interp(plonl,platl,pplon,nspecies),stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: Failed to allocate dvel_interp; error = ',ierr
         call endrun
      end if
      allocate( sea_ice(plonl,platl,pplon),stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dvel_inti: Failed to allocate sea_ice; error = ',ierr
         call endrun
      end if

      end subroutine dvel_inti_table

      subroutine interpdvel( plonl, calday, lat, ip )
!---------------------------------------------------------------------------
! 	... Interpolate the fields whose values are required at the
!           begining of a timestep.
!---------------------------------------------------------------------------

      use mo_calendar, only : caldayr
      use mo_surf,     only : seaice

      implicit none

!---------------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------------
      integer, intent(in) :: &
        plonl, &                          ! latitude tile dimension
        lat, &                            ! local latitude index
        ip                                ! longitude tile index
      real, intent(in) :: &
        calday                            ! Interpolate the input data to calday

!---------------------------------------------------------------------------
! 	... Local variables
!---------------------------------------------------------------------------
      integer :: &
        m, last, next
      integer  ::  dates(12) = (/ 116, 214, 316, 415,  516,  615, &
                                  716, 816, 915, 1016, 1115, 1216 /)
      real :: calday_loc, last_days, next_days
      real, save ::  days(12)
      logical, save  ::  entered = .false.

      if( .not. entered ) then
         do m = 1,12
            days(m) = caldayr( dates(m), 0 )
         end do
         entered = .true.
      end if

      if( calday < days(1) ) then
         next = 1
         last = 12
      else if( calday >= days(12) ) then
         next = 1
         last = 12
      else
         do m = 11,1,-1
            if( calday >= days(m) ) then
               exit
            end if
         end do
         last = m
         next = m + 1
      end if

      last_days  = days( last )
      next_days  = days( next )
      calday_loc = calday

      if( next_days < last_days ) then
         next_days = next_days + 365.
      end if
      if( calday_loc < last_days ) then
         calday_loc = calday_loc + 365.
      end if

      do m = 1,nspecies
         call intp2d( last_days, next_days, calday_loc, &
                      dvel(1:plonl,lat,ip,m,last), &
                      dvel(1:plonl,lat,ip,m,next), &
                      dvel_interp(1:plonl,lat,ip,m) )
      end do

      call intp2d( last_days, next_days, calday_loc, &
                   seaice(1:plonl,lat,ip,last), &
                   seaice(1:plonl,lat,ip,next), &
                   sea_ice(1:plonl,lat,ip) )

      end subroutine interpdvel

      subroutine intp2d( t1, t2, tint, f1, f2, fint )
!-----------------------------------------------------------------------
! 	... Linearly interpolate between f1(t1) and f2(t2) to fint(tint).
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      real, intent(in) :: &
        t1, &            ! time level of f1
        t2, &            ! time level of f2
        tint             ! interpolant time
      real, dimension(:), intent(in) :: &
        f1, &            ! field at time t1
        f2               ! field at time t2

      real, intent(out) :: &
        fint(:) ! field at time tint

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      real :: factor

      factor = (tint - t1)/(t2 - t1)

      fint(:) = f1(:) + (f2(:) - f1(:))*factor

      end subroutine intp2d

      subroutine drydep_table( lat, ip, calday, tsurf, zen_angle, &
                               depvel, dflx, rair, q, p, &
                               tv, plonl )
!--------------------------------------------------------
!       ... Form the deposition velocities for this
!           latitude slice
!--------------------------------------------------------

      use mo_chem_utls, only : get_spc_ndx, has_drydep
      use mo_surf,      only : ocean
      use mo_grid,      only : pcnst, plev
      use mo_srf_emis,  only : npp

      implicit none

!--------------------------------------------------------
!       ... Dummy arguments
!--------------------------------------------------------
      integer, intent(in) ::  plonl               ! longitude tile dimension
      integer, intent(in) ::  lat                 ! incoming latitude index
      integer, intent(in) ::  ip                  ! incoming longitude tile index
      real, intent(in)    ::  rair                ! gas constant of dry air in J/deg/kg
      real, intent(in)    ::  q(plonl,plev,pcnst) ! tracer mmr (kg/kg)
      real, intent(in)    ::  p(plonl)            ! midpoint pressure in surface layer (Pa)
      real, intent(in)    ::  tv(plonl)           ! virtual temperature in surface layer (K)
      real, intent(in)    ::  calday              ! time of year in days
      real, intent(in)    ::  tsurf(plonl)        ! surface temperature (K)
      real, intent(in)    ::  zen_angle(plonl)    ! zenith angle (radians)
      real, intent(inout) ::  dflx(plonl,pcnst)   ! flux due to dry deposition (kg/m^2/sec)
      real, intent(out)   ::  depvel(plonl,pcnst) ! deposition vel (cm/s)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: m, spc_ndx, tmp_ndx
      real, dimension(plonl) :: vel, glace, open_ocean, temp_fac, wrk, tmp
      real, dimension(plonl) :: o3_tab_dvel

!-----------------------------------------------------------------------
!       ... Note the factor 1.e-2 in the wrk array calculation is
!           to transform the incoming dep vel from cm/s to m/s
!-----------------------------------------------------------------------
      wrk(:) =  1.e-2 * p(:) / (rair * tv(:))

!--------------------------------------------------------
!       ... Initialize all deposition velocities to zero
!--------------------------------------------------------
      depvel(:,:) = 0.

!--------------------------------------------------------
!       ... Time interpolate primary depvel array
!           (also seaice and npp)
!--------------------------------------------------------
      call interpdvel( plonl, calday, lat, ip )

     if( o3_in_tab ) then
        o3_tab_dvel(:) = dvel_interp(:,lat,ip,o3_tab_ndx)
     end if
!--------------------------------------------------------
!       ... Set deposition velocities
!--------------------------------------------------------
      do m = 1,pcnst
         if( map(m) /= 0 ) then
            depvel(:,m) = dvel_interp(:,lat,ip,map(m))
            dflx(:,m)   = wrk(:) * depvel(:,m) * q(:,plev,m)
         end if
      end do

!--------------------------------------------------------
!       ... Set some variables needed for some dvel calculations
!--------------------------------------------------------
      temp_fac(:)   = min( 1., max( 0., (tsurf(:) - 268.) / 5. ) )
      open_ocean(:) = max( 0.,ocean(:,lat,ip) - sea_ice(:,lat,ip) )
      glace(:)      = sea_ice(:,lat,ip) + (1. - ocean(:,lat,ip)) * (1. - temp_fac(:))
      glace(:)      = min( 1.,glace(:) )

!--------------------------------------------------------
!       ... Set pan & mpan
!--------------------------------------------------------
      if( o3_in_tab ) then
         tmp(:) = o3_tab_dvel(:) / 3.
      else
         tmp(:) = 0.
      end if
      if( pan_dd ) then
         if( map(pan_ndx) == 0 ) then
            depvel(:,pan_ndx) = tmp(:)
            dflx(:,pan_ndx)   = wrk(:) * tmp(:) * q(:,plev,pan_ndx)
         end if
      end if
      if( mpan_dd ) then
         if( map(mpan_ndx) == 0 ) then
            depvel(:,mpan_ndx) = tmp(:)
            dflx(:,mpan_ndx)   = wrk(:) * tmp(:) * q(:,plev,mpan_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... Set no2 dvel
!--------------------------------------------------------
      if( no2_dd ) then
         if( map(no2_ndx) == 0 .and. o3_in_tab ) then
            depvel(:,no2_ndx) = (.6*o3_tab_dvel(:) + .055*ocean(:,lat,ip)) * .9
            dflx(:,no2_ndx)   = wrk(:) * depvel(:,no2_ndx) * q(:,plev,no2_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... Set hno3 dvel
!--------------------------------------------------------
      tmp(:) = (2. - open_ocean(:)) * (1. - glace(:)) + .05 * glace(:)
      if( hno3_dd ) then
         if( map(hno3_ndx) == 0 ) then
            depvel(:,hno3_ndx) = tmp(:)
            dflx(:,hno3_ndx)   = wrk(:) * tmp(:) * q(:,plev,hno3_ndx)
      else
            tmp(:) = depvel(:,hno3_ndx)
         end if
      end if
      if( onitr_dd ) then
         if( map(onitr_ndx) == 0 ) then
            depvel(:,onitr_ndx) = tmp(:)
            dflx(:,onitr_ndx)   = wrk(:) * tmp(:) * q(:,plev,onitr_ndx)
         end if
      end if
      if( isopooh_dd ) then
         if( map(isopooh_ndx) == 0 ) then
            depvel(:,isopooh_ndx) = tmp(:)
            dflx(:,isopooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,isopooh_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... Set h2o2 dvel
!--------------------------------------------------------
      if( .not. h2o2_in_tab ) then
         if( o3_in_tab ) then
            tmp(:) = .05*glace(:) + ocean(:,lat,ip) - sea_ice(:,lat,ip) &
                                  + (1. - (glace(:) + ocean(:,lat,ip)) + sea_ice(:,lat,ip)) &
                      *max( 1.,1./(.5 + 1./(6.*o3_tab_dvel(:))) )
         else
            tmp(:) = 0.
         end if
      else
         tmp(:) = dvel_interp(:,lat,ip,h2o2_tab_ndx)
      end if
      if( h2o2_dd ) then
         if( map(h2o2_ndx) == 0 ) then
            depvel(:,h2o2_ndx) = tmp(:)
            dflx(:,h2o2_ndx)   = wrk(:) * tmp(:) * q(:,plev,h2o2_ndx)
         end if
      end if
!--------------------------------------------------------
!       ... Set onit
!--------------------------------------------------------
      if( onit_dd ) then
         if( map(onit_ndx) == 0 ) then
            depvel(:,onit_ndx) = tmp(:)
            dflx(:,onit_ndx)   = wrk(:) * tmp(:) * q(:,plev,onit_ndx)
         end if
      end if
      if( ch3cocho_dd ) then
         if( map(ch3cocho_ndx) == 0 ) then
            depvel(:,ch3cocho_ndx) = tmp(:)
            dflx(:,ch3cocho_ndx)   = wrk(:) * tmp(:) * q(:,plev,ch3cocho_ndx)
         end if
      end if
      if( ch3ooh_in_tab ) then
         tmp(:) = dvel_interp(:,lat,ip,ch3ooh_tab_ndx)
      else
         tmp(:) = .5 * tmp(:)
      end if
      if( ch3ooh_dd ) then
         if( map(ch3ooh_ndx) == 0 ) then
            depvel(:,ch3ooh_ndx) = tmp(:)
            dflx(:,ch3ooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,ch3ooh_ndx)
         end if
      end if
      if( pooh_dd ) then
         if( map(pooh_ndx) == 0 ) then
            depvel(:,pooh_ndx) = tmp(:)
            dflx(:,pooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,pooh_ndx)
         end if
      end if
      if( ch3coooh_dd ) then
         if( map(ch3coooh_ndx) == 0 ) then
            depvel(:,ch3coooh_ndx) = tmp(:)
            dflx(:,ch3coooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,ch3coooh_ndx)
         end if
      end if
      if( c2h5ooh_dd ) then
         if( map(c2h5ooh_ndx) == 0 ) then
            depvel(:,c2h5ooh_ndx) = tmp(:)
            dflx(:,c2h5ooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,c2h5ooh_ndx)
         end if
      end if
      if( c3h7ooh_dd ) then
         if( map(c3h7ooh_ndx) == 0 ) then
            depvel(:,c3h7ooh_ndx) = tmp(:)
            dflx(:,c3h7ooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,c3h7ooh_ndx)
         end if
      end if
      if( rooh_dd ) then
         if( map(rooh_ndx) == 0 ) then
            depvel(:,rooh_ndx) = tmp(:)
            dflx(:,rooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,rooh_ndx)
         end if
      end if
      if( macrooh_dd ) then
         if( map(macrooh_ndx) == 0 ) then
            depvel(:,macrooh_ndx) = tmp(:)
            dflx(:,macrooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,macrooh_ndx)
         end if
      end if
      if( xooh_dd ) then
         if( map(xooh_ndx) == 0 ) then
            depvel(:,xooh_ndx) = tmp(:)
            dflx(:,xooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,xooh_ndx)
         end if
      end if
      if( ch3oh_dd ) then
         if( map(ch3oh_ndx) == 0 ) then
            depvel(:,ch3oh_ndx) = tmp(:)
            dflx(:,ch3oh_ndx)   = wrk(:) * tmp(:) * q(:,plev,ch3oh_ndx)
         end if
      end if
      if( c2h5oh_dd ) then
         if( map(c2h5oh_ndx) == 0 ) then
            depvel(:,c2h5oh_ndx) = tmp(:)
            dflx(:,c2h5oh_ndx)   = wrk(:) * tmp(:) * q(:,plev,c2h5oh_ndx)
         end if
      end if
      if( alkooh_dd ) then
         if( map(alkooh_ndx) == 0 ) then
            depvel(:,alkooh_ndx) = tmp(:)
            dflx(:,alkooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,alkooh_ndx)
         end if
      end if
      if( mekooh_dd ) then
         if( map(mekooh_ndx) == 0 ) then
            depvel(:,mekooh_ndx) = tmp(:)
            dflx(:,mekooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,mekooh_ndx)
         end if
      end if
      if( tolooh_dd ) then
         if( map(tolooh_ndx) == 0 ) then
            depvel(:,tolooh_ndx) = tmp(:)
            dflx(:,tolooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,tolooh_ndx)
         end if
      end if
      if( terpooh_dd ) then
         if( map(terpooh_ndx) == 0 ) then
            depvel(:,terpooh_ndx) = tmp(:)
            dflx(:,terpooh_ndx)   = wrk(:) * tmp(:) * q(:,plev,terpooh_ndx)
         end if
      end if

      if( o3_in_tab ) then
         tmp(:) = o3_tab_dvel(:)
      else
         tmp(:) = 0.
      end if
      if( ch2o_dd ) then
         if( map(ch2o_ndx) == 0 ) then
            depvel(:,ch2o_ndx) = tmp(:)
            dflx(:,ch2o_ndx)   = wrk(:) * tmp(:) * q(:,plev,ch2o_ndx)
         end if
      end if
      if( hydrald_dd ) then
         if( map(hydrald_ndx) == 0 ) then
            depvel(:,hydrald_ndx) = tmp(:)
            dflx(:,hydrald_ndx)   = wrk(:) * tmp(:) * q(:,plev,hydrald_ndx)
         end if
      end if
      if( ch3cooh_dd  ) then
         if( map(ch3cooh_ndx) == 0 ) then
            depvel(:,ch3cooh_ndx) = depvel(:,ch2o_ndx)
            dflx(:,ch3cooh_ndx) = wrk(:) * depvel(:,ch3cooh_ndx) * q(:,plev,ch3cooh_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... Set co and related species dep vel
!--------------------------------------------------------
      if( co_in_tab ) then
         tmp(:) = dvel_interp(:,lat,ip,co_tab_ndx)
      else
         tmp(:) = .3333 * npp(:,lat,ip) * ( 1. - glace(:) )
      end if
      if( co_dd ) then
         if( map(co_ndx) == 0 ) then
            depvel(:,co_ndx) = tmp(:)
            dflx(:,co_ndx)   = wrk(:) * tmp(:) * q(:,plev,co_ndx)
         end if
      end if
      if( ch3coch3_dd ) then
         if( map(ch3coch3_ndx) == 0 ) then
            depvel(:,ch3coch3_ndx) = tmp(:)
            dflx(:,ch3coch3_ndx)   = wrk(:) * tmp(:) * q(:,plev,ch3coch3_ndx)
         end if
      end if
      if( hyac_dd ) then
         if( map(hyac_ndx) == 0 ) then
            depvel(:,hyac_ndx) = tmp(:)
            dflx(:,hyac_ndx)   = wrk(:) * tmp(:) * q(:,plev,hyac_ndx)
         end if
      end if
      if( h2_dd ) then
         if( map(h2_ndx) == 0 ) then
            depvel(:,h2_ndx) = tmp(:) * 1.5                ! Hough(1991)
            dflx(:,h2_ndx)   = wrk(:) * depvel(:,h2_ndx) * q(:,plev,h2_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... Set glyald
!--------------------------------------------------------
      if( glyald_dd ) then
         if( map(glyald_ndx) == 0 ) then
            if( ch3cho_dd ) then
               depvel(:,glyald_ndx) = depvel(:,ch3cho_ndx)
            else if( ch3cho_in_tab ) then
               depvel(:,glyald_ndx) = dvel_interp(:,lat,ip,ch3cho_tab_ndx)
            else
               depvel(:,glyald_ndx) = 0.
            end if
            dflx(:,glyald_ndx)   = wrk(:) * depvel(:,glyald_ndx) * q(:,plev,glyald_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... Lead deposition
!--------------------------------------------------------
      if( Pb_dd ) then
         if( map(Pb_ndx) == 0 ) then
            depvel(:,Pb_ndx) = ocean(:,lat,ip)  * .05 + (1. - ocean(:,lat,ip)) * .2 
            dflx(:,Pb_ndx)   = wrk(:) * depvel(:,Pb_ndx) * q(:,plev,Pb_ndx)
         end if
      end if

!--------------------------------------------------------
!       ... diurnal dependence for OX dvel
!--------------------------------------------------------
      if( o3_dd .or. o3s_dd .or. o3inert_dd ) then
         if( o3_dd .or. o3_in_tab ) then
            if( o3_dd ) then
               tmp(:) = max( 1.,sqrt( (depvel(:,o3_ndx) - .2)**3/.27 + 4.*depvel(:,o3_ndx) + .67 ) )
               vel(:) = depvel(:,o3_ndx)
            else if( o3_in_tab ) then
               tmp(:) = max( 1.,sqrt( (o3_tab_dvel(:) - .2)**3/.27 + 4.*o3_tab_dvel(:) + .67 ) )
               vel(:) = o3_tab_dvel(:)
            end if
            where( zen_angle(:) <= 0. )
               vel(:) = vel(:) / tmp(:)
            elsewhere
               vel(:) = vel(:) * tmp(:)
            endwhere
         else
            vel(:) = 0.
         end if
         if( o3_dd ) then
            depvel(:,o3_ndx) = vel(:)
            dflx(:,o3_ndx)   = wrk(:) * vel(:) * q(:,plev,o3_ndx)
         end if
!--------------------------------------------------------
!       ... Set stratospheric O3 deposition
!--------------------------------------------------------
         if( o3s_dd ) then
            depvel(:,o3s_ndx) = vel(:)
            dflx(:,o3s_ndx)   = wrk(:) * vel(:) * q(:,plev,o3s_ndx)
         end if
         if( o3inert_dd ) then
            depvel(:,o3inert_ndx) = vel(:)
            dflx(:,o3inert_ndx)   = wrk(:) * vel(:) * q(:,plev,o3inert_ndx)
         end if
      end if

      end subroutine drydep_table

      subroutine dvel_inti_xactive( plonl, platl, pplon )
!-------------------------------------------------------------------------------------
! 	... intialize interactive drydep
!-------------------------------------------------------------------------------------

      use mo_constants,  only : r2d, phi, lam
      use mo_mpi,        only : masternode, base_lat
      use m_tracname,    only : tracnam
      use chem_mods,     only : adv_mass
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : dvel_flsp, dyn_soilw
      use mo_chem_utls,  only : get_spc_ndx, has_drydep
      use netcdf
      use drydep_input_data_tables

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer :: i, j, ii, jj, jl, ju
      integer :: nlon_veg, nlat_veg, npft_veg
      integer :: nlat_lai, npft_lai, pos_min, imin
      integer :: ncid, vid, dimid
      integer :: m, n, l, id
      integer :: length1, astat
      integer, allocatable :: wk_lai(:,:,:)
      integer :: k, num_max, k_max
      integer :: num_seas(5)

      real              :: spc_mass
      real              :: diff_min, target_lat
      real, allocatable :: vegetation_map(:,:,:)
      real, pointer     :: soilw_map(:,:,:)
      real, allocatable :: work(:,:)
      real, allocatable :: landmask(:,:)
      real, allocatable :: urban(:,:)
      real, allocatable :: lake(:,:)
      real, allocatable :: wetland(:,:)
      real, allocatable :: lon_veg(:)
      real, allocatable :: lon_veg_edge(:)
      real, allocatable :: lat_veg(:)
      real, allocatable :: lat_veg_edge(:)
      real, allocatable :: lat_lai(:)

      character(len=80) :: filename, lpath, rpath
      character(len=32) :: test_name

      logical :: do_soilw

      crb   = (difft/diffm)**.666666
!-------------------------------------------------------------------------------------
! 	... get species indices
!-------------------------------------------------------------------------------------
      h2_ndx       = get_spc_ndx( 'H2' )
      co_ndx       = get_spc_ndx( 'CO' )
      Pb_ndx       = get_spc_ndx( 'Pb' )
      pan_ndx      = get_spc_ndx( 'PAN' )
      mpan_ndx     = get_spc_ndx( 'MPAN' )
      o3_ndx       = get_spc_ndx( 'OX' )
      if( o3_ndx < 0 ) then
         o3_ndx  = get_spc_ndx( 'O3' )
      end if
      so2_ndx     = get_spc_ndx( 'SO2' )
      alkooh_ndx  = get_spc_ndx( 'ALKOOH')
      mekooh_ndx  = get_spc_ndx( 'MEKOOH')
      tolooh_ndx  = get_spc_ndx( 'TOLOOH')
      terpooh_ndx = get_spc_ndx( 'TERPOOH')
      ch3cooh_ndx = get_spc_ndx( 'CH3COOH')
      soa_ndx     = get_spc_ndx( 'SOA' )
      so4_ndx     = get_spc_ndx( 'SO4' )
      cb1_ndx     = get_spc_ndx( 'CB1' )
      cb2_ndx     = get_spc_ndx( 'CB2' )
      oc1_ndx     = get_spc_ndx( 'OC1' )
      oc2_ndx     = get_spc_ndx( 'OC2' )
      nh3_ndx     = get_spc_ndx( 'NH3' )
      nh4no3_ndx  = get_spc_ndx( 'NH4NO3' )
      sa1_ndx     = get_spc_ndx( 'SA1' )
      sa2_ndx     = get_spc_ndx( 'SA2' )
      sa3_ndx     = get_spc_ndx( 'SA3' )
      sa4_ndx     = get_spc_ndx( 'SA4' )
      nh4_ndx     = get_spc_ndx( 'NH4' )
      alkooh_dd  = has_drydep( 'ALKOOH')
      mekooh_dd  = has_drydep( 'MEKOOH')
      tolooh_dd  = has_drydep( 'TOLOOH')
      terpooh_dd = has_drydep( 'TERPOOH')
      ch3cooh_dd = has_drydep( 'CH3COOH')
      soa_dd     = has_drydep( 'SOA' )
      so4_dd     = has_drydep( 'SO4' )
      cb1_dd     = has_drydep( 'CB1' )
      cb2_dd     = has_drydep( 'CB2' )
      oc1_dd     = has_drydep( 'OC1' )
      oc2_dd     = has_drydep( 'OC2' )
      nh3_dd     = has_drydep( 'NH3' )
      nh4no3_dd  = has_drydep( 'NH4NO3' )
      sa1_dd     = has_drydep( 'SA1' ) 
      sa2_dd     = has_drydep( 'SA2' )
      sa3_dd     = has_drydep( 'SA3' ) 
      sa4_dd     = has_drydep( 'SA4' )
      nh4_dd     = has_drydep( 'NH4' ) 
!---------------------------------------------------------------------------
! 	... primary mapping
!---------------------------------------------------------------------------
species_loop : &
      do m = 1,pcnst
        test_name = tracnam(m)
        if( has_drydep( trim(test_name) ) ) then
           if( m == o3_ndx ) then
              test_name = 'OX'
           end if
           length1 = len_trim(test_name)
           do l = 1,n_species_table
              if( test_name(:length1) == trim( species_name_table(l) ) ) then
                 mapping(m)  = l
                 has_dvel(m) = .true.
                 exit
              end if
           end do
           if( .not. has_dvel(m) ) then
              select case( trim(test_name) )
              case( 'H2' )
                 test_name = 'CO'
              case( 'HYAC', 'CH3COOH' )
                 test_name = 'CH2O'
              case( 'O3S', 'O3INERT', 'MPAN' )
                 test_name = 'OX'
              case( 'ISOPOOH', 'MACROOH', 'Pb', 'XOOH' )
                 test_name = 'HNO3'
              case( 'ALKOOH', 'MEKOOH', 'TOLOOH', 'TERPOOH' )
                 test_name = 'CH3OOH'
              case( 'SOA', 'SO4', 'CB1', 'CB2', 'OC1', 'OC2', 'NH3', 'NH4', 'SA1', 'SA2', 'SA3', 'SA4' )
                 test_name = 'OX'  ! this is just a place holder. values are explicitly set below
              case( 'NH4NO3' )
                 test_name = 'HNO3'
              case default
                 test_name = 'blank'
              end select
              if( trim(test_name) /= 'blank' ) then
                 do l = 1,n_species_table
                    length1 = len_trim(test_name)
                    if( test_name(:length1) == trim( species_name_table(l) ) ) then
                       mapping(m)  = l
                       has_dvel(m) = .true.
                       exit
                    end if
                 end do
              else
                 write(*,*) 'dvel_inti: ',trim(tracnam(m)),' not in tables; will have dep vel = 0'
              end if
           end if
        end if
      end do species_loop

      if( all( .not. has_dvel(:) ) ) then
         return
      end if

      do m = 1,pcnst
        if( has_dvel(m) ) then
          l       = mapping(m)
          id      = 6 * (l-1)
          foxd(m) = dfoxd(l)
          if( m /= o3_ndx ) then
             spc_mass = adv_mass(m)
          else
             spc_mass = 47.9982
          end if
          drat(m) = sqrt( spc_mass/wh2o )
        end if
      end do

!---------------------------------------------------------------------------
! 	... allocate module variables
!---------------------------------------------------------------------------
      allocate( fraction_landuse(plonl,platl,pplon,n_land_type),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate fraction_landuse; error = ',astat
         call endrun
      end if
      allocate( index_season_lai(platl,n_land_type,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate index_season_lai; error = ',astat
         call endrun
      end if
!---------------------------------------------------------------------------
! 	... read landuse map
!---------------------------------------------------------------------------
      filename = dvel_flsp%nl_filename
      lpath    = dvel_flsp%local_path
      rpath    = dvel_flsp%remote_path
      ncid     = open_netcdf_file( filename, lpath, rpath )
      if( masternode ) then
         write(*,*) 'dvel_inti: Opened netcdf file ',trim(lpath) // trim(filename)
      end if

!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid ( ncid, 'lon', dimid ), 'dvel_inti: dimension nlon not found' )
      call handle_ncerr( nf_inq_dimlen ( ncid, dimid, nlon_veg ), 'dvel_inti: failed to read nlon' )
      call handle_ncerr( nf_inq_dimid ( ncid, 'lat', dimid ), 'dvel_inti: dimension nlat not found' )
      call handle_ncerr( nf_inq_dimlen ( ncid, dimid, nlat_veg ), 'dvel_inti: failed to read nlat' )
      call handle_ncerr( nf_inq_dimid ( ncid, 'pft', dimid ), 'dvel_inti: dimension nlat not found' )
      call handle_ncerr( nf_inq_dimlen ( ncid, dimid, npft_veg ), 'dvel_inti: failed to read nlat' )
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
      allocate( vegetation_map(nlon_veg,nlat_veg,npft_veg), work(nlon_veg,nlat_veg), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate vegation_map; error = ',astat
         call endrun
      end if
      allocate( urban(nlon_veg,nlat_veg), lake(nlon_veg,nlat_veg), &
                landmask(nlon_veg,nlat_veg), wetland(nlon_veg,nlat_veg), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate vegation_map; error = ',astat
         call endrun
      end if
      allocate( lon_veg(nlon_veg), lat_veg(nlat_veg), &
                lon_veg_edge(nlon_veg+1), lat_veg_edge(nlat_veg+1), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate vegation lon, lat arrays; error = ',astat
 call endrun
      end if
!---------------------------------------------------------------------------
! 	... read the vegetation map and landmask
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid ( ncid, 'PCT_PFT', vid), 'dvel_inti: PCT_PFT not found in input file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, vegetation_map ), 'dvel_inti: getting vegetation_map' )
      call handle_ncerr( nf_inq_varid ( ncid, 'LANDMASK', vid), 'dvel_inti: LANDMASK not found in input file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, landmask ), 'dvel_inti: getting landmask' )
      call handle_ncerr( nf_inq_varid ( ncid, 'PCT_URBAN', vid), 'dvel_inti: PCT_URBAN not found in input file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, urban ), 'dvel_inti: getting urban' )
      call handle_ncerr( nf_inq_varid ( ncid, 'PCT_LAKE', vid), 'dvel_inti: PCT_LAKE not found in input file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lake ), 'dvel_inti: getting lake' )
      call handle_ncerr( nf_inq_varid ( ncid, 'PCT_WETLAND', vid), 'dvel_inti: PCT_WETLAND not found in input file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, wetland ), 'dvel_inti: getting wetland' )
      call handle_ncerr( nf_close( ncid ), 'dvel_inti: error closing vegetation file' )
!---------------------------------------------------------------------------
! scale vegetation, urban, lake, and wetland to fraction
!---------------------------------------------------------------------------
      vegetation_map(:,:,:) = .01 * vegetation_map(:,:,:)
      wetland(:,:)          = .01 * wetland(:,:)
      lake(:,:)             = .01 * lake(:,:)
      urban(:,:)            = .01 * urban(:,:)
#ifdef DEBUG
      write(*,*) 'minmax vegetation_map ',minval(vegetation_map),maxval(vegetation_map)
      write(*,*) 'minmax wetland        ',minval(wetland),maxval(wetland)
      write(*,*) 'minmax landmask       ',minval(landmask),maxval(landmask)
#endif
!---------------------------------------------------------------------------
! 	... define lat-lon of vegetation map (1x1)
!---------------------------------------------------------------------------
      lat_veg(:)      = (/ (-89.5 + (i-1),i=1,nlat_veg  ) /)
      lon_veg(:)      = (/ (  0.5 + (i-1),i=1,nlon_veg  ) /)
      lat_veg_edge(:) = (/ (-90.0 + (i-1),i=1,nlat_veg+1) /)
      lon_veg_edge(:) = (/ (  0.0 + (i-1),i=1,nlon_veg+1) /)
!---------------------------------------------------------------------------
! 	... read soilw table if necessary
!---------------------------------------------------------------------------
      do_soilw = .not. dyn_soilw .and. (has_drydep( 'H2' ) .or. has_drydep( 'CO' ))
      if( do_soilw ) then
         call soilw_inti( 'clim_soilw.nc', trim(dvel_flsp%local_path), trim(dvel_flsp%remote_path), &
                           plonl, platl, pplon, nlon_veg, nlat_veg, soilw_map )
      end if
!---------------------------------------------------------------------------
! 	... regrid to model grid
!---------------------------------------------------------------------------
      call interp_map( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                       lon_veg, lon_veg_edge, landmask, urban, lake, &
                       wetland, vegetation_map, soilw_map, do_soilw, plonl, &
                       platl, pplon )

      deallocate( vegetation_map, work, stat=astat )
      deallocate( lon_veg, lat_veg, lon_veg_edge, lat_veg_edge, stat=astat )
      deallocate( landmask, urban, lake, wetland, stat=astat )
      if( do_soilw ) then
         deallocate( soilw_map, stat=astat )
      end if

!---------------------------------------------------------------------------
! 	... read LAI based season indeces
!---------------------------------------------------------------------------
      filename = 'season_wes.nc'
      lpath    = dvel_flsp%local_path
      rpath    = dvel_flsp%remote_path
      ncid     = open_netcdf_file( filename, lpath, rpath )
      if( masternode ) then
         write(*,*) 'dvel_inti: Opened netcdf file ',trim(lpath) // trim(filename)
      end if

!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), 'dvel_inti: dimension lat not found' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat_lai ), 'dvel_inti: failed to read nlat' )
      call handle_ncerr( nf_inq_dimid( ncid, 'pft', dimid ), 'dvel_inti: dimension pft not found' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, npft_lai ), 'dvel_inti: failed to read nlat' )
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
      allocate( lat_lai(nlat_lai), wk_lai(nlat_lai,npft_lai,12), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate vegation_map; error = ',astat
         call endrun
      end if
      allocate( lon_veg(nlon_veg), lat_veg(nlat_veg), lon_veg_edge(nlon_veg+1), lat_veg_edge(nlat_veg+1), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'dvel_inti: failed to allocate vegation lon, lat arrays; error = ',astat
         call endrun
      end if
!---------------------------------------------------------------------------
! 	... read the latitude and the season indicies
!---------------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid), 'dvel_inti: season_wes not found in input file' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat_lai ), 'dvel_inti: getting season_wes' )
      call handle_ncerr( nf_inq_varid( ncid, 'season_wes', vid), 'dvel_inti: season_wes not found in input file' )
      call handle_ncerr( nf_get_var_int( ncid, vid, wk_lai ), 'dvel_inti: getting season_wes' )
      call handle_ncerr( nf_close( ncid ), 'dvel_inti: error closing season_wes file' )

      jl = base_lat + 1
      ju = base_lat + platl
      imin = 1
      do j = 1,platl
        diff_min = 10.
        pos_min  = -99
        target_lat = phi(base_lat+j)*r2d
        do i = imin,nlat_lai
          if( abs(lat_lai(i) - target_lat) < diff_min ) then
            diff_min = abs(lat_lai(i) - target_lat)
            pos_min  = i
          end if
        end do
        if( pos_min < 0 ) then
          write(*,*) 'dvel_inti: cannot find ',target_lat,' at j,pos_min,diff_min = ',j,pos_min,diff_min
          write(*,*) 'dvel_inti: imin,nlat_lai = ',imin,nlat_lai
          write(*,*) 'dvel_inti: lat_lai'
          write(*,'(1p,10g12.5)') lat_lai(:)
          call endrun
        end if
        imin = pos_min
        index_season_lai(j,:,:) = wk_lai(pos_min,:,:)

!---------------------------------------------------------------------------
! specify the season as the most frequent in the 11 vegetation classes
! this was done to remove a banding problem in dvel (JFL Oct 04)
!---------------------------------------------------------------------------
        do m = 1,12
          num_seas = 0
          do l = 1,11
            do k = 1,5
              if( index_season_lai(j,l,m) == k ) then
                num_seas(k) = num_seas(k) + 1
                exit
              end if
            end do
          end do

          num_max = -1
          do k = 1,5
            if( num_seas(k) > num_max ) then
              num_max = num_seas(k)
              k_max = k
            endif
          end do

          index_season_lai(j,:,m) = k_max
        end do
      end do

      deallocate( lat_lai, wk_lai )

      rgss(:,:) = max( 1.,rgss(:,:) )
      rac(:,:)  = max( small_value,rac(:,:) )

      end subroutine dvel_inti_xactive

      subroutine interp_map( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                             lon_veg, lon_veg_edge, landmask, urban, lake, &
                             wetland, vegetation_map, soilw_map, do_soilw, plonl, &
                             platl, pplon )

      use mo_grid, only      : plon, plat
      use mo_constants, only : r2d, phi, lam
      use mo_control, only   : dyn_soilw
      use m_adv, only        : has_npole
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)      :: plonl
      integer, intent(in)      :: platl
      integer, intent(in)      :: pplon
      integer, intent(in)      :: nlon_veg, nlat_veg, npft_veg
      real, pointer            :: soilw_map(:,:,:)
      real, intent(in)         :: landmask(nlon_veg,nlat_veg)
      real, intent(in)         :: urban(nlon_veg,nlat_veg)
      real, intent(in)         :: lake(nlon_veg,nlat_veg)
      real, intent(in)         :: wetland(nlon_veg,nlat_veg)
      real, intent(in)         :: vegetation_map(nlon_veg,nlat_veg,npft_veg)
      real, intent(in)         :: lon_veg(nlon_veg)
      real, intent(in)         :: lon_veg_edge(nlon_veg+1)
      real, intent(in)         :: lat_veg(nlat_veg)
      real, intent(in)         :: lat_veg_edge(nlat_veg+1)
      logical, intent(in)      :: do_soilw

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer, parameter           :: veg_ext = 10

      integer                      :: i, j, ii, jj, jl, ju, i_ndx, n
      integer, dimension(plon+1)   :: ind_lon
      integer, dimension(platl+1)  :: ind_lat
      real                         :: total_land
      real, dimension(plon+1)      :: lon_edge
      real, dimension(platl+1)     :: lat_edge
      real                         :: lat1, lat2, lon1, lon2
      real                         :: x1, x2, y1, y2, dx, dy
      real                         :: area, total_area
      real, dimension(npft_veg+3)  :: fraction
      real                         :: total_soilw_area
      real                         :: fraction_soilw
      real                         :: total_soilw(12)
      real                         :: soilw_3d(plon,platl,12)
      real, dimension(plon,platl,n_land_type) :: fraction_landuse_3d

      real,    dimension(-veg_ext:nlon_veg+veg_ext) :: lon_veg_edge_ext
      integer, dimension(-veg_ext:nlon_veg+veg_ext) :: mapping_ext

      jl = base_lat + 1
      ju = base_lat + platl

      do i = 1,plon
         lon_edge(i) = lam(i) * r2d - .5*(lam(2) - lam(1)) * r2d
      end do
      lon_edge(plon+1) = lon_edge(plon) + (lam(2) - lam(1)) * r2d
      if( .not. has_npole ) then
        do j = 1,platl+1
           lat_edge(j) = phi(j+base_lat) * r2d - .5*(phi(2) - phi(1)) * r2d
        end do
      else
        do j = 1,platl
           lat_edge(j) = phi(j+base_lat) * r2d - .5*(phi(2) - phi(1)) * r2d
        end do
        lat_edge(platl+1) = lat_edge(platl) + (phi(2) - phi(1)) * r2d
      end if
      do j = 1,platl+1
         lat_edge(j) = min( lat_edge(j), 90. )
         lat_edge(j) = max( lat_edge(j),-90. )
      end do

!-------------------------------------------------------------------------------------
! wrap around the longitudes
!-------------------------------------------------------------------------------------
      do i = -veg_ext,0
        lon_veg_edge_ext(i) = lon_veg_edge(nlon_veg+i) - 360.
        mapping_ext     (i) =              nlon_veg+i
      end do
      do i = 1,nlon_veg
        lon_veg_edge_ext(i) = lon_veg_edge(i)
        mapping_ext     (i) =              i
      end do
      do i = nlon_veg+1,nlon_veg+veg_ext
        lon_veg_edge_ext(i) = lon_veg_edge(i-nlon_veg) + 360.
        mapping_ext     (i) =              i-nlon_veg
      end do
#ifdef DEBUG
      write(*,*) 'interp_map : lon_edge ',lon_edge
      write(*,*) 'interp_map : lat_edge ',lat_edge
      write(*,*) 'interp_map : mapping_ext ',mapping_ext
#endif
      do j = 1,plon+1
        lon1 = lon_edge(j) 
        do i = -veg_ext,nlon_veg+veg_ext
          dx = lon_veg_edge_ext(i  ) - lon1
          dy = lon_veg_edge_ext(i+1) - lon1
          if( dx*dy <= 0. ) then
            ind_lon(j) = i
            exit
          end if
        end do
      end do

      do j = 1,platl+1
        lat1 = lat_edge(j)
        do i = 1,nlat_veg
          dx = lat_veg_edge(i  ) - lat1
          dy = lat_veg_edge(i+1) - lat1
          if( dx*dy <= 0. ) then
            ind_lat(j) = i
            exit
          end if
        end do
      end do
#ifdef DEBUG
      write(*,*) 'interp_map : ind_lon ',ind_lon
      write(*,*) 'interp_map : ind_lat ',ind_lat
#endif
lat_loop : &
      do j = 1,platl
lon_loop : &
        do i = 1,plon
          total_area       = 0.
          fraction         = 0.
          total_soilw(:)   = 0.
          total_soilw_area = 0.
          do jj = ind_lat(j),ind_lat(j+1)
            y1 = max( lat_edge(j),lat_veg_edge(jj) )
            y2 = min( lat_edge(j+1),lat_veg_edge(jj+1) ) 
            dy = (y2 - y1)/(lat_veg_edge(jj+1) - lat_veg_edge(jj))
            do ii =ind_lon(i),ind_lon(i+1)
              i_ndx = mapping_ext(ii)
              x1 = max( lon_edge(i),lon_veg_edge_ext(ii) )
              x2 = min( lon_edge(i+1),lon_veg_edge_ext(ii+1) ) 
              dx = (x2 - x1)/(lon_veg_edge_ext(ii+1) - lon_veg_edge_ext(ii))
              area = dx * dy
              total_area = total_area + area
!-----------------------------------------------------------------
! 	... special case for ocean grid point 
!-----------------------------------------------------------------
              if( nint(landmask(i_ndx,jj)) == 0 ) then
                fraction(npft_veg+1) = fraction(npft_veg+1) + area
              else
                do n = 1,npft_veg
                  fraction(n) = fraction(n) + vegetation_map(i_ndx,jj,n) * area
                end do
                fraction(npft_veg+1) = fraction(npft_veg+1) + area * lake   (i_ndx,jj)
                fraction(npft_veg+2) = fraction(npft_veg+2) + area * wetland(i_ndx,jj)
                fraction(npft_veg+3) = fraction(npft_veg+3) + area * urban  (i_ndx,jj)
!-----------------------------------------------------------------
! 	... check if land accounts for the whole area.
!           If not, the remaining area is in the ocean
!-----------------------------------------------------------------
                total_land = sum(vegetation_map(i_ndx,jj,:)) &
                           + urban  (i_ndx,jj) &
                           + lake   (i_ndx,jj) &
                           + wetland(i_ndx,jj)
                if( total_land < 1. ) then
                  fraction(npft_veg+1) = fraction(npft_veg+1) + (1. - total_land) * area
                end if
!-------------------------------------------------------------------------------------
! 	... compute weighted average of soilw over grid (non-water only)
!-------------------------------------------------------------------------------------
                if( do_soilw ) then
                   fraction_soilw = total_land  - (lake(i_ndx,jj) + wetland(i_ndx,jj))
                   total_soilw_area = total_soilw_area + fraction_soilw * area
                   total_soilw(:)   = total_soilw(:) + fraction_soilw * area * soilw_map(i_ndx,jj,:)
                end if
              end if
            end do
          end do
!-------------------------------------------------------------------------------------
! 	... divide by total area of grid box
!-------------------------------------------------------------------------------------
          fraction(:) = fraction(:)/total_area
!-------------------------------------------------------------------------------------
! 	... make sure we don't have too much or too little
!-------------------------------------------------------------------------------------
          if( abs( sum(fraction) - 1.) > .001 ) then
            fraction(:) = fraction(:)/sum(fraction)
          end if
!-------------------------------------------------------------------------------------
! 	... map to Wesely land classification
!-------------------------------------------------------------------------------------
          fraction_landuse_3d(i,j, 1) =     fraction(20)
          fraction_landuse_3d(i,j, 2) = sum(fraction(16:17))
          fraction_landuse_3d(i,j, 3) = sum(fraction(13:15))
          fraction_landuse_3d(i,j, 4) = sum(fraction( 5: 9))
          fraction_landuse_3d(i,j, 5) = sum(fraction( 2: 4))
          fraction_landuse_3d(i,j, 6) =     fraction(19)
          fraction_landuse_3d(i,j, 7) =     fraction(18)
          fraction_landuse_3d(i,j, 8) =     fraction( 1)
          fraction_landuse_3d(i,j, 9) = 0.
          fraction_landuse_3d(i,j,10) = 0.
          fraction_landuse_3d(i,j,11) = sum(fraction(10:12))
          if( do_soilw ) then
             if( total_soilw_area > 0. ) then
                soilw_3d(i,j,:) = total_soilw(:)/total_soilw_area
             else
                soilw_3d(i,j,:) = -99.
             end if
          end if
        end do lon_loop
      end do lat_loop
!-------------------------------------------------------------------------------------
! 	... make sure there are no negative values
!-------------------------------------------------------------------------------------
      if( any( fraction_landuse_3d(:,:,:) < 0. ) ) then
        write(*,*) 'something is wrong in interpolation of landuse map'
        write(*,*) 'minval(fraction_landuse_3d) = ',minval(fraction_landuse_3d)
        call endrun
      end if

!-------------------------------------------------------------------------------------
! 	... reshape according to lat-lon blocks
!-------------------------------------------------------------------------------------
#ifdef DEBUG
      write(*,*) 'interp_map : jl,ju ',jl,ju,plonl,platl,pplon
      write(*,*) 'interp_map : shape(fraction_landuse) ',shape(fraction_landuse)
      write(*,*) 'interp_map : shape(fraction_landuse_3d) ',shape(fraction_landuse_3d)
#endif
      do i = 1,n_land_type
        fraction_landuse(:,:,:,i) = reshape( fraction_landuse_3d(:,:,i), (/plonl,platl,pplon/), order=(/1,3,2/) )
      end do
    
      if( do_soilw ) then
         do i = 1,12
            tab_soilw(:,:,:,i) = reshape( soilw_3d(:,:,i), (/plonl,platl,pplon/), order=(/1,3,2/) )
         end do
      end if

      end subroutine interp_map

      subroutine drydep_xactive( lat, ip, ncdate, sfc_temp, pressure_sfc,  &
                                 wind_speed, spec_hum, air_temp, pressure_10m, rain, &
                                 snow, solar_flux, dvel, dflx, mmr, &
                                 tv, soilw, rh, plonl )
!-------------------------------------------------------------------------------------
!   code based on wesely (atmospheric environment, 1989, vol 23, p. 1293-1304) for
!   calculation of r_c, and on walcek et. al. (atmospheric enviroment, 1986,
!   vol. 20, p. 949-964) for calculation of r_a and r_b
!
!   as suggested in walcek (u_i)(u*_i) = (u_a)(u*_a)
!   is kept constant where i represents a subgrid environment and a the
!   grid average environment. thus the calculation proceeds as follows:
!   va the grid averaged wind is calculated on dots
!   z0(i) the grid averaged roughness coefficient is calculated
!   ri(i) the grid averaged richardson number is calculated
!   --> the grid averaged (u_a)(u*_a) is calculated
!   --> subgrid scale u*_i is calculated assuming (u_i) given as above
!   --> final deposotion velocity is weighted average of subgrid scale velocities
!
! code written by P. Hess, rewritten in fortran 90 by JFL (August 2000)
! modified by JFL to be used in MOZART-2 (October 2002)
!-------------------------------------------------------------------------------------

      use m_tracname,   only : tracnam
      use mo_constants, only : phi
      use mo_mpi,       only : base_lat
      use mo_grid,      only : plev
      use mo_seasalt,   only : seasalt_sett_vel, has_seasalt, nbin
      use drydep_input_data_tables

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)   :: lat                      ! latitude index
      integer, intent(in)   :: ip                       ! long tile index
      integer, intent(in)   :: plonl                    ! long tile dim
      integer, intent(in)   :: ncdate                   ! present date (yyyymmdd)
      real, intent(in)      :: sfc_temp(plonl)          ! surface temperature (K)
      real, intent(in)      :: pressure_sfc(plonl)      ! surface pressure (Pa)
      real, intent(in)      :: wind_speed(plonl)        ! 10 meter wind speed (m/s)
      real, intent(in)      :: spec_hum(plonl)          ! specific humidity (kg/kg)
      real, intent(in)      :: rh(plonl,1)              ! relative humidity
      real, intent(in)      :: air_temp(plonl)          ! surface air temperature (K)
      real, intent(in)      :: pressure_10m(plonl)      ! 10 meter pressure (Pa)
      real, intent(in)      :: rain(plonl)              
      real, intent(in)      :: snow(plonl)              ! snow height (m)
      real, intent(in)      :: soilw(plonl)             ! soil moisture fraction
      real, intent(in)      :: solar_flux(plonl)        ! direct shortwave radiation at surface (W/m^2)
      real, intent(in)      :: tv(plonl)                ! potential temperature
      real, intent(in)      :: mmr(plonl,plev,pcnst)    ! constituent concentration (kg/kg)
      real, intent(out)     :: dvel(plonl,pcnst)        ! deposition velocity (cm/s)
      real, intent(inout)   :: dflx(plonl,pcnst)        ! deposition flux (/cm^2/s)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      real, parameter :: scaling_to_cm_per_s = 100.
      real, parameter :: scaling_to_m_per_s  = .01
      real, parameter :: tmelt               = 273.16
      real, parameter :: rain_threshold      = 1.e-7  ! of the order of 1cm/day expressed in m/s

      integer :: i, ispec, lt, m
      integer :: sndx
      integer :: month

      real :: slope = 0.
      real :: z0water ! revised z0 over water
      real :: p       ! pressure at midpoint first layer
      real :: pg      ! surface pressure
      real, dimension(plonl)  :: tc      ! temperature in celsius
      real :: es      ! saturation vapor pressure
      real :: ws      ! saturation mixing ratio
      real :: hvar    ! constant to compute xmol
      real :: h       ! constant to compute xmol
      real :: psih    ! stability correction factor
      real, dimension(plonl) :: cts     ! correction to rlu rcl and rgs for frost
      real :: rs      ! constant for calculating rsmx
      real :: rmx     ! resistance by vegetation
      real :: zovl    ! ratio of z to  m-o length
      real :: cvarb   ! cvar averaged over landtypes
      real :: bb      ! b averaged over landtypes
      real :: ustarb  ! ustar averaged over landtypes

!-------------------------------------------------------------------------------------
! local arrays: dependent on location and species
!-------------------------------------------------------------------------------------
      real, dimension(plonl,pcnst) :: heff

!-------------------------------------------------------------------------------------
! local arrays: dependent on location only
!-------------------------------------------------------------------------------------
      integer                :: index_season(plonl,n_land_type)
      real, dimension(plonl) :: tha     ! atmospheric virtual potential temperature
      real, dimension(plonl) :: thg     ! ground virtual potential temperature
      real, dimension(plonl) :: z       ! height of lowest level
      real, dimension(plonl) :: va      ! magnitude of v on cross points
      real, dimension(plonl) :: ribn    ! richardson number
      real, dimension(plonl) :: qs      ! saturation specific humidity
      real                   :: dewm    ! multiplier for rs when dew occurs
      real, dimension(plonl) :: crs     ! multiplier to calculate crs
      real, dimension(plonl) :: rdc     ! part of lower canopy resistance
      real, dimension(plonl) :: uustar  ! u*ustar (assumed constant over grid)
      real, dimension(plonl) :: z0b     ! average roughness length over grid
      real, dimension(plonl) :: wrk     ! work array
      real, dimension(plonl) :: wdiv    ! work array
      real, dimension(plonl) :: term    ! work array
      real, dimension(plonl) :: resc    ! work array
      real, dimension(plonl) :: lnd_tmp ! work array
      logical, dimension(plonl) :: unstable
      logical, dimension(plonl) :: has_rain
      logical, dimension(plonl) :: has_dew 

!-------------------------------------------------------------------------------------
! local arrays: dependent on location and landtype
!-------------------------------------------------------------------------------------
      real, dimension(plonl,n_land_type) :: ra    ! aerodynamic resistance
      real, dimension(plonl,n_land_type) :: rb    ! resistance across sublayer
      real, dimension(plonl,n_land_type) :: rds   ! resistance for deposition of sulfate
      real, dimension(plonl,n_land_type) :: b     ! buoyancy parameter for unstable conditions
      real, dimension(plonl,n_land_type) :: cvar  ! height parameter
      real, dimension(plonl,n_land_type) :: ustar ! friction velocity
      real, dimension(plonl,n_land_type) :: xmol  ! monin-obukhov length

!-------------------------------------------------------------------------------------
! local arrays: dependent on location, landtype and species
!-------------------------------------------------------------------------------------
      real, dimension(plonl,n_land_type,pcnst) :: rsmx  ! vegetative resistance (plant mesophyll)
      real, dimension(plonl,n_land_type,pcnst) :: rclx  ! lower canopy resistance
      real, dimension(plonl,n_land_type,pcnst) :: rlux  ! vegetative resistance (upper canopy)
      real, dimension(plonl,n_land_type,pcnst) :: rgsx  ! ground resistance
      real, dimension(plonl,n_land_type)       :: rlux_o3  ! vegetative resistance (upper canopy) for o3
      real :: pmid(plonl,1)                             ! for seasalt aerosols
      real :: tfld(plonl,1)                             ! for seasalt aerosols
      real :: settling_velocity(plonl,1,nbin)           ! for seasalt aerosols
      real :: rb_part(plonl,nbin,n_land_type)           ! for seasalt aerosols
      real :: fact, vds
      real :: rc                                        ! combined surface resistance
      real :: var_soilw, dv_soil_h2, fact_h2            ! h2 dvel wrking variables
      logical :: fr_lnduse(plonl,n_land_type)           ! wrking array

!-------------------------------------------------------------------------------------
! jfl : mods for PAN
!-------------------------------------------------------------------------------------
      real                  :: dv_pan
      real :: c0_pan(11) = (/ 0.000, 0.006, 0.002, 0.009, 0.015, &
                              0.006, 0.000, 0.000, 0.000, 0.002, 0.002 /)
      real :: k_pan (11) = (/ 0.000, 0.010, 0.005, 0.004, 0.003, &
                              0.005, 0.000, 0.000, 0.000, 0.075, 0.002 /)

!-------------------------------------------------------------------------------------
! initialize
!-------------------------------------------------------------------------------------
      do m = 1,pcnst
         dvel(:,m) = 0.
      end do

      if( all( .not. has_dvel(:) ) ) then
        return
      end if

!-------------------------------------------------------------------------------------
! define species-dependent parameters (temperature dependent)
!-------------------------------------------------------------------------------------
      call set_hcoeff( sfc_temp, heff, plonl )

      do lt = 1,n_land_type
         ra (:,lt)   = 0.
         rb (:,lt)   = 0.
         rds(:,lt)   = 0.
      end do

!-------------------------------------------------------------------------------------
! 	... set month
!-------------------------------------------------------------------------------------
      month = mod( ncdate,10000 )/100

!-------------------------------------------------------------------------------------
! define which season (relative to Northern hemisphere climate)
!-------------------------------------------------------------------------------------
      if( phi(base_lat+lat) >= 0. ) then
        if( month == 12 .or. month < 3 ) then
           index_season(:,:) = 3
        else if( month > 2 .and. month < 6 ) then
           index_season(:,:) = 5
        else if( month > 5 .and. month < 9 ) then
           index_season(:,:) = 1
        else if( month > 8 .and. month < 12 ) then
           index_season(:,:) = 2
        end if
      else
        if( month == 12 .or. month < 3 ) then
           index_season(:,:) = 1
        else if( month > 2 .and. month < 6 ) then
           index_season(:,:) = 2
        else if( month > 5 .and. month < 9 ) then
           index_season(:,:) = 3
        else if( month > 8 .and. month < 12 ) then
           index_season(:,:) = 5
        end if
      end if

!-------------------------------------------------------------------------------------
! define season index based on fixed LAI
!-------------------------------------------------------------------------------------
      do i = 1,plonl
        index_season(i,:) = index_season_lai(lat,:,month)
      end do

!-------------------------------------------------------------------------------------
! special case for snow covered terrain
!-------------------------------------------------------------------------------------
      do i = 1,plonl
        if( snow(i) > .01 ) then
          index_season(i,:) = 4
        end if
      end do

!-------------------------------------------------------------------------------------
! scale rain and define logical arrays
!-------------------------------------------------------------------------------------
      has_rain(:) = rain(:) > rain_threshold

!-------------------------------------------------------------------------------------
! loop over longitude points
!-------------------------------------------------------------------------------------
long_loop : &
      do i = 1,plonl
        p   = pressure_10m(i)
        pg  = pressure_sfc(i)
!-------------------------------------------------------------------------------------
! potential temperature
!-------------------------------------------------------------------------------------
        tha(i) = air_temp(i) * (p00/p )**rovcp * (1. + .61*spec_hum(i))
        thg(i) = sfc_temp(i) * (p00/pg)**rovcp * (1. + .61*spec_hum(i))
!-------------------------------------------------------------------------------------
! height of 1st level
!-------------------------------------------------------------------------------------
        z(i) = - r/grav * air_temp(i) * (1. + .61*spec_hum(i)) * log(p/pg)
!-------------------------------------------------------------------------------------
! wind speed
!-------------------------------------------------------------------------------------
        va(i) = max( .01,wind_speed(i) )
!-------------------------------------------------------------------------------------
! Richardson number
!-------------------------------------------------------------------------------------
        ribn(i) = z(i) * grav * (tha(i) - thg(i))/thg(i) / (va(i)*va(i))
        ribn(i) = min( ribn(i),ric )
        unstable(i) = ribn(i) < 0.
!-------------------------------------------------------------------------------------
! saturation vapor pressure (Pascals)
! saturation mixing ratio
! saturation specific humidity
!-------------------------------------------------------------------------------------
        es    = 611.*exp( 5414.77*(sfc_temp(i) - tmelt)/(tmelt*sfc_temp(i)) )
        ws    = .622*es/(pg - es)
        qs(i) = ws/(1. + ws)
!-------------------------------------------------------------------------------------
! multiplier for rs if rain or dew
!-------------------------------------------------------------------------------------
!       dewm(i) = 1.
!       if( qs(i) <= spec_hum(i) ) then
!         dewm(i) = 3.
!       end if
!-------------------------------------------------------------------------------------
! no dew if < 0C, effect of frost later
!-------------------------------------------------------------------------------------
!       if( sfc_temp(i) < tmelt ) then
!          dewm(i) = 1.
!       end if
!
        has_dew(i) = .false.
        if( qs(i) <= spec_hum(i) ) then
          has_dew(i) = .true.
        end if
        if( sfc_temp(i) < tmelt ) then
           has_dew(i) = .false.
        end if
!
!-------------------------------------------------------------------------------------
! constant in determining rs
!-------------------------------------------------------------------------------------
        tc(i) = sfc_temp(i) - tmelt
        if( sfc_temp(i) > tmelt .and. sfc_temp(i) < 313.15 ) then
           crs(i) = (1. + (200./(solar_flux(i) + .1))**2) * (400./(tc(i)*(40. - tc(i))))
        else
           crs(i) = large_value
        end if
!-------------------------------------------------------------------------------------
! rdc (lower canopy res)
!-------------------------------------------------------------------------------------
        rdc(i) = 100.*(1. + 1000./(solar_flux(i) + 10.))/(1. + 1000.*slope)
      end do long_loop

!-------------------------------------------------------------------------------------
! 	... form working array
!-------------------------------------------------------------------------------------
      do lt = 1,n_land_type
         fr_lnduse(:,lt) = fraction_landuse(:,lat,ip,lt) /= 0.
      end do

!-------------------------------------------------------------------------------------
! find grid averaged z0: z0bar (the roughness length) z_o=exp[S(f_i*ln(z_oi))]
! this is calculated so as to find u_i, assuming u*u=u_i*u_i
!-------------------------------------------------------------------------------------
      z0b(:) = 0.
      do lt = 1,n_land_type
        do i = 1,plonl
          if( fr_lnduse(i,lt) ) then
            z0b(i) = z0b(i) + fraction_landuse(i,lat,ip,lt) * log( z0(index_season(i,lt),lt) )
          end if
        end do
      end do

!-------------------------------------------------------------------------------------
! find the constant velocity uu*=(u_i)(u*_i)
!-------------------------------------------------------------------------------------
      do i = 1,plonl
        z0b(i) = exp( z0b(i) )
        cvarb  = vonkar/log( z(i)/z0b(i) )
!-------------------------------------------------------------------------------------
! unstable and stable cases
!-------------------------------------------------------------------------------------
        if( unstable(i) ) then
          bb = 9.4*(cvarb**2)*sqrt( abs(ribn(i))*z(i)/z0b(i) )
          ustarb = cvarb * va(i) * sqrt( 1. - (9.4*ribn(i)/(1. + 7.4*bb)) )
        else
          ustarb = cvarb * va(i)/(1. + 4.7*ribn(i))
        end if
        uustar(i) = va(i)*ustarb
      end do  

!-------------------------------------------------------------------------------------
! calculate the friction velocity for each land type u_i=uustar/u*_i
!-------------------------------------------------------------------------------------
      do lt = 1,n_land_type
        do i = 1,plonl
          ustar(i,lt) = -99.
          if( fr_lnduse(i,lt) ) then
             if( unstable(i) ) then
                cvar(i,lt)  = vonkar/log( z(i)/z0(index_season(i,lt),lt) )
                b(i,lt)     = 9.4*(cvar(i,lt)**2)* sqrt( abs(ribn(i))*z(i)/z0(index_season(i,lt),lt) )
                ustar(i,lt) = sqrt( cvar(i,lt)*uustar(i)*sqrt( 1. - (9.4*ribn(i)/(1. + 7.4*b(i,lt))) ) )
             else
               cvar(i,lt)  = vonkar/log( z(i)/z0(index_season(i,lt),lt) )
               ustar(i,lt) = sqrt( cvar(i,lt)*uustar(i)/(1. + 4.7*ribn(i)) )
             end if
          end if
        end do
      end do

!-------------------------------------------------------------------------------------
! revise calculation of friction velocity and z0 over water
!-------------------------------------------------------------------------------------
!     lt = 7    
!     do i = 1,plonl
!       ustar(i,lt) = -99.
!       if( fr_lnduse(i,lt) ) then
!          if( unstable(i) ) then
!             z0water     = (.016*(ustar(i,lt)**2)/grav) + diffk/(9.1*ustar(i,lt))
!             cvar(i,lt)  = vonkar/(log( z(i)/z0water ))
!             b(i,lt)     = 9.4*(cvar(i,lt)**2)*sqrt( abs(ribn(i))*z(i)/z0water )
!             ustar(i,lt) = sqrt( cvar(i,lt)*uustar(i)* sqrt( 1. - (9.4*ribn(i)/(1.+ 7.4*b(i,lt))) ) )
!           else
!             z0water     = (.016*(ustar(i,lt)**2)/grav) + diffk/(9.1*ustar(i,lt))
!             cvar(i,lt)  = vonkar/(log(z(i)/z0water))
!             ustar(i,lt) = sqrt( cvar(i,lt)*uustar(i)/(1. + 4.7*ribn(i)) )
!           end if
!       end if
!     end do
!
! jfl : 14 Sep 2006
!
! this code was removed because it lent to very small values of ustar, leading to
! extremely large values of rb_part.  This modification to the code
! has a small impact, increasing deposition
! velocities over the ocean
!
      where ( ustar < 0.01 .and. ustar > 0. )
        ustar = 0.01
      end where

!-------------------------------------------------------------------------------------
! compute monin-obukhov length for unstable and stable conditions/ sublayer resistance
!-------------------------------------------------------------------------------------
      do lt = 1,n_land_type
        do i = 1,plonl
          if( fr_lnduse(i,lt) ) then
             hvar = (va(i)/0.74) * (tha(i) - thg(i)) * (cvar(i,lt)**2)
             if( unstable(i) ) then                      ! unstable
                h = hvar*(1. - (9.4*ribn(i)/(1. + 5.3*b(i,lt))))
             else
                h = hvar/((1.+4.7*ribn(i))**2)
             end if
             xmol(i,lt) = thg(i) * ustar(i,lt) * ustar(i,lt) / (vonkar * grav * h)
           end if
        end do
      end do

!-------------------------------------------------------------------------------------
! psih
!-------------------------------------------------------------------------------------
      do lt = 1,n_land_type
        do i = 1,plonl
          if( fr_lnduse(i,lt) ) then
             if( xmol(i,lt) < 0. ) then
                zovl = z(i)/xmol(i,lt)
                zovl = max( -1.,zovl )
                psih = exp( .598 + .39*log( -zovl ) - .09*(log( -zovl ))**2 )
                vds  = 2.e-3*ustar(i,lt) * (1. + (300/(-xmol(i,lt)))**0.666)
             else
                zovl = z(i)/xmol(i,lt)
                zovl = min( 1.,zovl )
                psih = -5. * zovl
                vds  = 2.e-3*ustar(i,lt)
             end if
             ra (i,lt) = (vonkar - psih*cvar(i,lt))/(ustar(i,lt)*vonkar*cvar(i,lt))
             ra (i,lt) = max(0.,ra (i,lt))
             rb (i,lt) = (2./(vonkar*ustar(i,lt))) * crb
             rds(i,lt) = 1./vds
          end if
!
!         if ( ra(i,lt) < 0. ) then
!           print *,'ra_1 ',i,lt,ra(i,lt),fr_lnduse(i,lt)
!           print *,'ra_2 ',vonkar,psih,cvar(i,lt)
!           print *,'ra_3 ',ustar(i,lt)
!         endif
!
        end do
      end do

!-------------------------------------------------------------------------------------
! surface resistance : depends on both land type and species
! land types are computed seperately, then resistance is computed as average of values
! following wesely rc=(1/(rs+rm) + 1/rlu +1/(rdc+rcl) + 1/(rac+rgs))**-1
!
! compute rsmx = 1/(rs+rm) : multiply by 3 if surface is wet
!-------------------------------------------------------------------------------------
species_loop1 : &
      do ispec = 1,pcnst
has_dvel1 : &
        if( has_dvel(ispec) ) then
           do lt = 1,n_land_type
             do i = 1,plonl
               if( fr_lnduse(i,lt) ) then
                  sndx = index_season(i,lt)
                  if( ispec == o3_ndx .or. ispec == so2_ndx ) then
                     rmx = 0.
                  else
                     rmx = 1./(heff(i,ispec)/3000. + 100.*foxd(ispec))
                  end if
                  cts(i) = 1000.*exp( - tc(i) - 4. )                 ! correction for frost
                  rgsx(i,lt,ispec) = cts(i) + 1./((heff(i,ispec)/(1.e5*rgss(sndx,lt))) + (foxd(ispec)/rgso(sndx,lt)))
!-------------------------------------------------------------------------------------
! special case for H2 and CO
!-------------------------------------------------------------------------------------
                  if( ispec == h2_ndx .or. ispec == co_ndx ) then
                    fact_h2 = 1.
                    if( ispec == co_ndx ) then
                       fact_h2 = 1.5
                    end if
!-------------------------------------------------------------------------------------
! no deposition on snow, ice, desert, and water
!-------------------------------------------------------------------------------------
                    if( lt == 1 .or. lt == 7 .or. lt == 8 .or. sndx == 4 ) then
                      rgsx(i,lt,ispec) = large_value
                    else
                      var_soilw = max( .1,min( soilw(i),.3 ) )
                      if( lt == 3 ) then
                        var_soilw = log( var_soilw )
                      end if
                      dv_soil_h2 = h2_c(lt) + var_soilw*(h2_b(lt) + var_soilw*h2_a(lt))
                      if( dv_soil_h2 > 0. ) then
                        rgsx(i,lt,ispec) = fact_h2/(dv_soil_h2*1.e-4)
                      end if
                    end if
                  end if
                  if( lt == 7 ) then
                     rclx(i,lt,ispec) = large_value
                     rsmx(i,lt,ispec) = large_value
                     rlux(i,lt,ispec) = large_value
                  else
                     rs = ri(sndx,lt)*crs(i)
                     if ( has_dew(i) .or. has_rain(i) ) then
                       dewm = 3.
                     else
                       dewm = 1.
                     end if
                     rsmx(i,lt,ispec) = (dewm*rs*drat(ispec) + rmx)
!-------------------------------------------------------------------------------------
! jfl : special case for PAN
!-------------------------------------------------------------------------------------
                     if( ispec == pan_ndx ) then
                        dv_pan =  c0_pan(lt) * (1. - exp( -k_pan(lt)*(dewm*rs*drat(ispec))*1.e-2 ))
                        if( dv_pan > 0. .and. sndx /= 4 ) then
                          rsmx(i,lt,ispec) = ( 1./dv_pan )
                        end if
                     end if
                     rclx(i,lt,ispec) = cts(i) + 1./((heff(i,ispec)/(1.e5*rcls(sndx,lt))) + (foxd(ispec)/rclo(sndx,lt)))
                     rlux(i,lt,ispec) = cts(i) + rlu(sndx,lt)/(1.e-5*heff(i,ispec) + foxd(ispec))
                  end if
               end if
             end do
           end do
        end if has_dvel1
      end do species_loop1

!-------------------------------------------------------------------------------------
! 	... set o3 variables
!-------------------------------------------------------------------------------------
      do lt = 1,n_land_type
         if( lt /= 7 ) then
            do i = 1,plonl
               rlux_o3(i,lt) = large_value
               if( fr_lnduse(i,lt) ) then
                  sndx = index_season(i,lt)
!-------------------------------------------------------------------------------------
! 	... no effect if sfc_temp < O C
!-------------------------------------------------------------------------------------
                  if( sfc_temp(i) > tmelt ) then
                     if( has_dew(i) ) then
                        rlux_o3(i,lt) = 3000.*rlu(sndx,lt)/(1000. + rlu(sndx,lt))
                     end if
                     if( has_rain(i) ) then
                        rlux_o3(i,lt) = 3000.*rlu(sndx,lt)/(1000. + 3.*rlu(sndx,lt))
                     end if
                  end if
                  if( o3_ndx > 0 ) then
                     rclx(i,lt,o3_ndx) = cts(i) + rclo(index_season(i,lt),lt)
                     rlux(i,lt,o3_ndx) = cts(i) + rlux_o3(i,lt)
                  end if
               end if
            end do
         end if
      end do

species_loop2 : &
      do ispec = 1,pcnst
has_dvel2 : &
        if( has_dvel(ispec) ) then
           if( ispec /= o3_ndx .and. ispec /= so2_ndx ) then
              do lt = 1,n_land_type
                 if( lt /= 7 ) then
                    do i = 1,plonl
                       if( fr_lnduse(i,lt) ) then
!-------------------------------------------------------------------------------------
! no effect if sfc_temp < O C
!-------------------------------------------------------------------------------------
                          if( sfc_temp(i) > tmelt ) then
                             if( has_dew(i) .or. has_rain(i) ) then
                                rlux(i,lt,ispec) = 1./((1./(3.*rlux(i,lt,ispec))) &
                                                     + 1.e-7*heff(i,ispec) + foxd(ispec)/rlux_o3(i,lt))
                             end if
                          end if
                       end if
                    end do
                 end if
              end do
           else if( ispec == so2_ndx ) then
              do lt = 1,n_land_type
                 if( lt /= 7 ) then
                    do i = 1,plonl
                       if( fr_lnduse(i,lt) ) then
!-------------------------------------------------------------------------------------
! no effect if sfc_temp < O C
!-------------------------------------------------------------------------------------
                          if( sfc_temp(i) > tmelt ) then
                             if( has_dew(i) ) then
                                rlux(i,lt,ispec) = 100.
                             end if
                             if( has_rain(i) ) then
!                               rlux(i,lt,ispec) = 1./(2.e-4 + (1./(3.*rlu(index_season(i,lt),lt))))
                                rlux(i,lt,ispec) = 15.*rlu(index_season(i,lt),lt)/(5. + 3.e-3*rlu(index_season(i,lt),lt))
                             end if
                          end if
                          rclx(i,lt,ispec) = cts(i) + rcls(index_season(i,lt),lt)
                          rlux(i,lt,ispec) = cts(i) + rlux(i,lt,ispec)
                       end if
                    end do
                 end if
              end do
              do i = 1,plonl
                 if( fr_lnduse(i,1) .and. (has_dew(i) .or. has_rain(i)) ) then
                    rlux(i,1,ispec) = 50.
                 end if
              end do
           end if
        end if has_dvel2
      end do species_loop2

!-------------------------------------------------------------------------------------
! compute rc
!-------------------------------------------------------------------------------------
      term(:) = 1.e-2 * pressure_10m(:) / (r*tv(:))
      if( has_seasalt ) then
         pmid(:,1) = pressure_sfc(:)
         tfld(:,1) = sfc_temp(:)
         call seasalt_sett_vel( pmid, tfld, rh, settling_velocity, ustar, rb_part )
         settling_velocity(:,:,:) = scaling_to_m_per_s * settling_velocity(:,:,:)
      end if
species_loop3 : &
      do ispec = 1,pcnst
has_dvel3 : &
        if( has_dvel(ispec) ) then
           wrk(:) = 0.
           do lt = 1,n_land_type
              where( fr_lnduse(:,lt) )
                 resc(:) = 1./(1./rsmx(:,lt,ispec) + 1./rlux(:,lt,ispec) &
                               + 1./(rdc(:) + rclx(:,lt,ispec)) &
                               + 1./(rac(index_season(:,lt),lt) + rgsx(:,lt,ispec)))
                 resc(:) = max( 10.,resc(:) )
                 lnd_tmp(:) = fraction_landuse(:,lat,ip,lt)
              endwhere
!-------------------------------------------------------------------------------------
! 	... compute average deposition velocity
!-------------------------------------------------------------------------------------
              select case( tracnam(ispec) )
                 case( 'SO2' )
                    if( lt == 7 ) then
                       where( fr_lnduse(:,lt) )
                          wrk(:) = wrk(:) + lnd_tmp(:)/(ra(:,lt) + rb(:,lt)) ! assume no surface resistance for SO2 over water
                       endwhere
                    else
                       where( fr_lnduse(:,lt) )
                         wrk(:) = wrk(:) + lnd_tmp(:)/(ra(:,lt) + rb(:,lt) + resc(:))
                       endwhere
                    end if
                 case( 'SO4' )
                    where( fr_lnduse(:,lt) )
                       wrk(:) = wrk(:) + lnd_tmp(:)/(ra(:,lt) + rds(:,lt))
                    endwhere
                 case( 'NH4', 'NH4NO3' )
                    where( fr_lnduse(:,lt) )
                       wrk(:) = wrk(:) + lnd_tmp(:)/(ra(:,lt) + 0.5*rds(:,lt))
                    endwhere
                 case( 'SA1', 'SA2', 'SA3', 'SA4' )
                    if( ispec == sa1_ndx ) then
                       m = 1
                    else if( ispec == sa2_ndx ) then
                       m = 2
                    else if( ispec == sa3_ndx ) then
                       m = 3
                    else if( ispec == sa4_ndx ) then
                       m = 4
                    end if
                    where( fr_lnduse(:,lt) )
                       wdiv(:) = rb_part(:,m,lt) + ra(:,lt)*( 1. + rb_part(:,m,lt)*settling_velocity(:,1,m))
                    endwhere
                    where( fr_lnduse(:,lt) .and. wdiv(:) > 0. )
                       wrk(:) = wrk(:) + lnd_tmp(:)*(1./wdiv(:) + settling_velocity(:,1,m))
                    endwhere
                    where( fr_lnduse(:,lt) .and. wdiv(:) == 0. )
                       wrk(:) = wrk(:) + lnd_tmp(:)*settling_velocity(:,1,m)
                    endwhere
!
!                   if ( maxval(wrk) > 1.e2 ) then
!                     print *,'wrk_0 ',lt
!                     print *,'wrk_1 ',wrk
!                     print *,'wrk_2 ',ra(:,lt)
!                     print *,'wrk_3 ',rb_part(:,m,lt)
!                     print *,'wrk_3 ',1./(ra(:,lt)+rb_part(:,m,lt)+ra(:,lt)*rb_part(:,m,lt)*settling_velocity(:,1,m))
!                     print *,'wrk_4 ',settling_velocity(:,1,m)
!                     print *,'wrk_5 ',lnd_tmp
!                     print *,'wrk_6 ',fr_lnduse(:,lt)
!                   endif
!
!-------------------------------------------------------------------------------------
! 	... special case for Pb (for consistency with offline code)
!-------------------------------------------------------------------------------------
                 case( 'Pb' )
                    if( lt == 7 ) then
                       where( fr_lnduse(:,lt) )
                          wrk(:) = wrk(:) + lnd_tmp(:) * 0.05e-2
                       endwhere
                    else
                       where( fr_lnduse(:,lt) )
                         wrk(:) = wrk(:) + lnd_tmp(:) * 0.2e-2
                       endwhere
                    end if
                 case default
                    where( fr_lnduse(:,lt) )
                      wrk(:) = wrk(:) + lnd_tmp(:)/(ra(:,lt) + rb(:,lt) + resc(:))
                    endwhere
              end select
           end do
           select case( trim( tracnam(ispec) ) )
              case( 'CB1', 'CB2', 'OC1', 'OC2' )
                 wrk(:) = 0.10e-2
           end select
           dvel(:,ispec) = wrk(:) * scaling_to_cm_per_s
           dflx(:,ispec) = term(:) * dvel(:,ispec) * mmr(:,plev,ispec)
        end if has_dvel3
      end do species_loop3

!-------------------------------------------------------------------------------------
! 	... special adjustments
!-------------------------------------------------------------------------------------
      if( mpan_ndx > 0 ) then
         if( has_dvel(mpan_ndx) ) then
            dvel(:,mpan_ndx) = dvel(:,mpan_ndx)/3.
            dflx(:,mpan_ndx) = term(:) * dvel(:,mpan_ndx) * mmr(:,plev,mpan_ndx)
         end if
      end if

      end subroutine drydep_xactive

      subroutine set_hcoeff( sfc_temp, heff, plonl )

      use m_tracname, only : tracnam
      use drydep_input_data_tables

      implicit none

!-------------------------------------------------------------------------------------
!         ... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)  :: plonl
      real, intent(in)     :: sfc_temp(plonl)
      real, intent(out)    :: heff(plonl,pcnst)

!-------------------------------------------------------------------------------------
!         ... local variables
!-------------------------------------------------------------------------------------
      real, parameter :: t0 = 298.

      integer :: m, l, id
      real    :: e298, dhr
      real    :: dk1s(plonl)
      real    :: dk2s(plonl)
      real    :: wrk(plonl)

      wrk(:) = (t0 - sfc_temp(:))/(t0*sfc_temp(:))
      do m = 1,pcnst
has_depvel : &
        if( has_dvel(m) ) then
          l    = mapping(m)
          id   = 6*(l - 1)
          e298 = dheff(id+1)
          dhr  = dheff(id+2)
          heff(:,m) = e298*exp( dhr*wrk(:) )
          if( dheff(id+3) /= 0. .and. dheff(id+5) == 0. ) then
             e298 = dheff(id+3)
             dhr  = dheff(id+4)
             dk1s(:) = e298*exp( dhr*wrk(:) )
             where( heff(:,m) /= 0. )
                heff(:,m) = heff(:,m)*(1. + dk1s(:)*ph_inv)
             elsewhere
                heff(:,m) = dk1s(:)*ph_inv
             endwhere
          end if
          if( dheff(id+5) /= 0. ) then
             if( trim(tracnam(m)) == 'CO2' .or. trim(tracnam(m)) == 'NH3' ) then
                e298 = dheff(id+3)
                dhr  = dheff(id+4)
                dk1s(:) = e298*exp( dhr*wrk(:) )
                e298 = dheff(id+5)
                dhr  = dheff(id+6)
                dk2s(:) = e298*exp( dhr*wrk(:) )
                if( trim(tracnam(m)) == 'CO2' ) then
                   heff(:,m) = heff(:,m)*(1. + dk1s(:)*ph_inv)*(1. + dk2s(:)*ph_inv)
                else if( trim( tracnam(m) ) == 'NH3' ) then
                   heff(:,m) = heff(:,m)*(1. + dk1s(:)*ph/dk2s(:))
                else
                   write(*,*) 'error in assigning henrys law coefficients'
                   write(*,*) 'species ',tracnam(m)
                end if
             end if
          end if
        end if has_depvel
      end do

      end subroutine set_hcoeff

      subroutine soilw_inti( ncfile, lpath, mspath, plonl, platl, &
                             pplon, nlon_veg, nlat_veg, soilw_map )
!------------------------------------------------------------------
!	... read primary soil moisture table
!------------------------------------------------------------------

      use netcdf
      use mo_file_utils,   only : open_netcdf_file
      use mo_calendar,     only : caldayr

      implicit none

!------------------------------------------------------------------
!	... dummy args
!------------------------------------------------------------------
      integer, intent(in) :: &
        plonl, &
        platl, &
        pplon, &
        nlon_veg, &
        nlat_veg
      real, pointer :: &
        soilw_map(:,:,:)
      character(len=*), intent(in) :: &
        ncfile, &            ! file name of netcdf file containing data
        lpath, &             ! local pathname to ncfile
        mspath               ! mass store pathname to ncfile

!------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------
      integer :: gndx = 0
      integer :: nlat, &             ! # of lats in soilw file
                 nlon                ! # of lons in soilw file
      integer :: i, ip, k, m
      integer :: j, jl, ju
      integer :: lev, day, ierr
      integer :: ncid, vid
      integer :: dimid_lat, dimid_lon, dimid_time
      integer :: dates(12) = (/ 116, 214, 316, 415,  516,  615, &
                                716, 816, 915, 1016, 1115, 1216 /)

!-----------------------------------------------------------------------
!       ... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )
!-----------------------------------------------------------------------
!       ... get longitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), 'soilw_inti: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'soilw_inti: failed to get length of dimension lon' )
      if( nlon /= nlon_veg ) then
         write(*,*) 'soilw_inti: soil and vegetation lons differ; ',nlon, nlon_veg
         call endrun
      end if
!-----------------------------------------------------------------------
!       ... get latitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'soilw_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'soilw_inti: failed to get length of dimension lat' )
      if( nlat /= nlat_veg ) then
         write(*,*) 'soilw_inti: soil and vegetation lats differ; ',nlat, nlat_veg
         call endrun
      end if
!-----------------------------------------------------------------------
!       ... set times (days of year)
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid_time ), &
                         'soilw_inti: failed to find dimension time' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_time, ndays ), &
                         'soilw_inti: failed to get length of dimension time' )
      if( ndays /= 12 ) then
         write(*,*) 'soilw_inti: dataset not a cyclical year'
         call endrun
      end if
      allocate( days(ndays),stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'soilw_inti: days allocation error = ',ierr
         call endrun
      end if
      do m = 1,min(12,ndays)
         days(m) = caldayr( dates(m), 0 )
      end do

!------------------------------------------------------------------
!	... allocate arrays
!------------------------------------------------------------------
      allocate( soilw_map(nlon,nlat,ndays), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'soilw_inti: soilw_map allocation error = ',ierr
         call endrun
      end if

!------------------------------------------------------------------
!	... read in the soil moisture
!------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'SOILW', vid ), 'soilw_inti: getting species id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, soilw_map ), 'soilw_inti: getting soilw_map' )
!------------------------------------------------------------------
!	... close file
!------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), 'soilw_inti: failed to close file ' // trim(ncfile) )

!-----------------------------------------------------------------------
!	... allocate module variable
!-----------------------------------------------------------------------
      allocate( tab_soilw(plonl,platl,pplon,ndays),stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'soilw_inti: failed to allocate tab_soilw; error = ',ierr
         call endrun
      end if

      end subroutine soilw_inti

      subroutine chk_soilw( calday )
!--------------------------------------------------------------------
!	... check timing for ub values
!--------------------------------------------------------------------

      use mo_constants, only : dayspy

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      real, intent(in)    :: calday

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer  ::  m, upper
      real     ::  numer, denom

!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      if( calday < days(1) ) then
 next = 1
 last = ndays
      else
        if( days(ndays) < dayspy ) then
          upper = ndays
        else
          upper = ndays - 1
        end if
        do m = upper,1,-1
          if( calday >= days(m) ) then
            exit
          end if
        end do
        last = m
        next = mod( m,ndays ) + 1
      end if
      numer = calday - days(last)
      denom = days(next) - days(last)
      if( numer < 0. ) then
    numer = dayspy + numer
      end if
      if( denom < 0. ) then
    denom = dayspy + denom
      end if
      dels = max( min( 1.,numer/denom ),0. )

      end subroutine chk_soilw

      subroutine set_soilw( lat, ip, soilw, plonl )
!--------------------------------------------------------------------
!	... set the soil moisture
!--------------------------------------------------------------------

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: lat               ! lat index
      integer, intent(in) :: ip                ! longitude tile index
      integer, intent(in) :: plonl             ! longitude tile dim
      real, intent(inout) :: soilw(plonl)

      soilw(:) = tab_soilw(:,lat,ip,last) + dels * (tab_soilw(:,lat,ip,next) - tab_soilw(:,lat,ip,last))

      end subroutine set_soilw

      end module mo_drydep
