
      module mo_dust
!---------------------------------------------------------------
!	... annual cycle for sulfur
!---------------------------------------------------------------

      implicit none

      private
      public  :: dust_inti, read_dust, dust_interp, nqdust

      save

      integer, parameter :: nqdust = 4
      integer :: gndx = 0, &
                 ncid = 0, &
                 time_cnt = 0, &
                 lotim = 0, &
                 hitim = 0, &
                 loin = 1, &
                 hiin = 2, &
                 jlim_in(2), &
                 nlevs, &
                 nlon, &
                 nlat
      real    :: P0 = 1.e5
      real, allocatable :: times(:)
      real, allocatable :: ps_in(:,:,:,:)
      real, allocatable :: hyam(:)
      real, allocatable :: hybm(:)
      real, allocatable :: dust_in(:,:,:,:,:,:)
      logical :: cyclical = .true.

      contains 

      subroutine dust_inti( plonl, platl, pplon, ncfile, lpath, mspath )
!-----------------------------------------------------------------------
! 	... open netcdf file containing annual sulfur data.  initialize
!           arrays with the data to be interpolated to the current time.
!
!           it is assumed that the time coordinate is increasing
!           and represents calendar days; range = [1.,366.).
!-----------------------------------------------------------------------

      use netcdf
      use mo_regrider,   only : regrid_inti, regrid_lat_limits, regrid_diagnostics
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file
      use mo_mpi,        only : base_lat, masternode
      use mo_grid,       only : platg=>plat, plong=>plon, plev

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      character(len=*), intent(in) :: &
        ncfile, &            ! file name of netcdf file containing data
        lpath, &             ! local pathname to ncfile
        mspath               ! mass store pathname to ncfile

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: &
        ierr, &     ! return code
        times_id, &
        vid
      integer :: dimid_lat, dimid_lon, dimid_lev
      real, allocatable :: lat(:),lon(:)
      character(len=128) ::  recname
      logical :: there

!-----------------------------------------------------------------------
!     	... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )

!-----------------------------------------------------------------------
!     	... inquire about file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_unlimdim( ncid, times_id ), &
                         'dust_inti: failed to get unlimited dim id' )
      call handle_ncerr( nf_inq_dimlen( ncid, times_id, time_cnt ), &
                         'dust_inti: failed to read time count' )

!-----------------------------------------------------------------------
!     	... allocate space for time coordinate data
!-----------------------------------------------------------------------
      allocate( times(time_cnt), stat = ierr )
      if( ierr /= 0 ) then
	 write(*,*) ' dust_inti : failed to allocate times array; error = ',ierr
	 call endrun
      end if
!-----------------------------------------------------------------------
!     	... get time coordinate
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'time', vid ), 'dust_inti: getting time id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, times ), 'dust_inti: getting times' )
      if( masternode ) then
         write(*,*) 'dust_inti: times' 
         write(*,'(1p,5g15.8)') times
      end if

!-----------------------------------------------------------------------
!     	... make sure the time coordinate looks like calendar day,
!           and is increasing
!-----------------------------------------------------------------------
      call moz_chktime( times, time_cnt )

!-----------------------------------------------------------------------
!     	... get vertical coordinate
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lev', dimid_lev ), &
                         'dust_inti: failed to find lev dim id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lev, nlevs ), &
                         'dust_inti: failed to get length of dimension lev' )
      allocate( hyam(nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: hyam allocation error = ',ierr
         call endrun
      end if
      allocate( hybm(nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: hybm allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'hyam', vid ), 'dust_inti: Getting hyam id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, hyam ), 'dust_inti: reading hyam' )
      call handle_ncerr( nf_inq_varid( ncid, 'hybm', vid ), 'dust_inti: Getting hybm id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, hybm ), 'dust_inti: reading hybm' )
!     call handle_ncerr( nf_inq_varid( ncid, 'P0', vid ), 'dust_inti: Getting P0 id' )
!     call handle_ncerr( nf_get_var_double( ncid, vid, P0 ), 'dust_inti: reading P0' )
      if( masternode ) then
         write(*,*) 'dust_inti: P0 = ',P0
      end if

!-----------------------------------------------------------------------
!     	... get latitude and longitude
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'dust_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'dust_inti: failed to get length of dimension lat' )
      allocate( lat(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'dust_inti: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'dust_inti: failed to read variable lat' )
      lat(:nlat) = lat(:nlat) * d2r
 
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                         'dust_inti: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'dust_inti: failed to get length of dimension lon' )
      allocate( lon(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: lon allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'dust_inti: failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                         'dust_inti: failed to read variable lon' )
      lon(:nlon) = lon(:nlon) * d2r

!-----------------------------------------------------------------------
!     	... get grid interp limits
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, platg, &
                          nlon, plong, &
                          lon,  lam, &
                          lat,  phi, &
                          0, platl )
      deallocate( lat, lon, stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: failed to deallocate lat,lon; ierr = ',ierr
         call endrun
      end if
      if( gndx /= 0 ) then
	 write(*,*) 'dust_inti: non-xparent transform'
	 call regrid_diagnostics( gndx )
         jlim_in = regrid_lat_limits( gndx )
      else
	 write(*,*) 'dust_inti: xparent transform'
         jlim_in = (/ base_lat+1, base_lat+platl /)
      end if

      write(*,'(''dust_inti: gndx ='',i2,'', grid limits = '',2i4,'', jl,ju ='',2i4)') &
         gndx, jlim_in, base_lat+1, base_lat+platl


!-----------------------------------------------------------------------
!     	... allocate primary netcdf input variables
!-----------------------------------------------------------------------
      allocate( dust_in(plonl,nlevs,nqdust,platl,pplon,2), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: dust_in allocation error = ',ierr
         call endrun
      end if

      allocate( ps_in(plonl,platl,pplon,2), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'dust_inti: ps_in allocation error = ',ierr
         call endrun
      end if

      end subroutine dust_inti

      subroutine read_dust( plonl, platl, pplon, calday )
!-----------------------------------------------------------------------
! 	... interpolate annual cycle dust data to the current time.
!           read in new time samples of the input data as necessary.
!-----------------------------------------------------------------------

      use netcdf
      use mo_regrider, only : regrid_2d
      use mo_mpi,      only : base_lat, masternode
      use mo_grid,     only : plon, plev

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      real, intent(in) :: &
        calday  ! current time in calendar days + fraction.

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: &
        i, ip, k, k1, jl, ju, m, &
        oldlotim, oldhitim, &
        vid, ret
      integer :: start(4)
      integer :: count(4)
      real    :: cday
      real, allocatable :: dust_nc(:,:,:)
      real, allocatable :: ps_nc(:,:)
      real    :: wrk(plon,platl)
      character(len=80) :: errcom
      character(len=8)  :: fldname
      character(len=3)  :: numa

      cday = calday  ! convert calday to single precision if necessary

!-----------------------------------------------------------------------
!     	... check to see if model time is still bounded by dataset times.
!-----------------------------------------------------------------------
      oldlotim = lotim
      oldhitim = hitim
      call moz_findplb( times, time_cnt, cday, lotim )
      if( cyclical ) then
         hitim = mod( lotim,time_cnt ) + 1
      else
         hitim = lotim + 1
      end if

      if( hitim /= oldhitim ) then
!-----------------------------------------------------------------------
! 	... read in new hitim data.  replace old lotim data.
!-----------------------------------------------------------------------
         allocate( dust_nc(nlon,jlim_in(1):jlim_in(2),nlevs), &
                   ps_nc(nlon,jlim_in(1):jlim_in(2)), stat=ret )
         if( ret /= 0 ) then
            write(*,*) 'read_dust: failed to allocate dust_nc,ps_nc = ',ret
            call endrun
         end if
	 jl   = base_lat + 1
	 ju   = base_lat + platl
         loin = hiin
         hiin = mod( loin,2 ) + 1
!-----------------------------------------------------------------------
! 	... the surface pressure
!-----------------------------------------------------------------------
         start(:3) = (/ 1, jlim_in(1), hitim /)
         count(:3) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, 1 /)
         call handle_ncerr( nf_inq_varid( ncid, 'PS', vid ), 'read_dust: failed to get PS id' )
         call handle_ncerr( nf_get_vara_double( ncid, vid, start(:3), count(:3), ps_nc ), 'read_dust: failed to read PS' )
         call regrid_2d( ps_nc, wrk, gndx, jl, ju, do_poles=.true. )
         ps_in(:,:,:,hiin) = reshape( wrk, (/plonl,platl,pplon/), order = (/1,3,2/) )
!-----------------------------------------------------------------------
! 	... the dust concentration
!-----------------------------------------------------------------------
         start(:4) = (/ 1, jlim_in(1), 1, hitim /)
         count(:4) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, nlevs, 1 /)
	 do m = 1,nqdust
	    write(numa,'(i3)') 100+m
	    fldname = 'DSTQ' // numa(2:3)
	    errcom = 'read_dust: getting ' // trim(fldname) // ' id'
            call handle_ncerr( nf_inq_varid( ncid, trim(fldname), vid ), trim(errcom) )
	    errcom = 'read_dust: getting ' // trim(fldname)
            call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, dust_nc(1,jlim_in(1),1) ), trim(errcom) )
!-----------------------------------------------------------------------
! 	... regrid cam input dust
!-----------------------------------------------------------------------
	    do k = 1,nlevs
               call regrid_2d( dust_nc(:,:,k), wrk, gndx, jl, ju, do_poles=.true. )
#ifdef DEBUG
               if( k == nlevs ) then
                  write(*,*) 'read_dust: dust_nc :'
                  write(*,'(1p,10e8.1)') dust_nc(1,jlim_in(1):jlim_in(2),k)
                  write(*,*) 'read_dust: wrk_out :'
                  write(*,'(1p,10e8.1)') wrk(1,1:platl)
               end if
#endif
	       do ip = 1,pplon
	          dust_in(:plonl,k,m,:platl,ip,hiin) = wrk((ip-1)*plonl+1:ip*plonl,:platl)
	       end do
	    end do
         end do
         if( masternode ) then
            write(*,*) ' '
            write(*,*) 'read_dust: read data for day ',times(hitim)
         end if
         if( lotim /= oldhitim ) then
!-----------------------------------------------------------------------
!   	... read in new lotim data.  replace old hitim data.
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! 	... the surface pressure
!-----------------------------------------------------------------------
            start(:3) = (/ 1, jlim_in(1), lotim /)
            count(:3) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, 1 /)
            call handle_ncerr( nf_inq_varid( ncid, 'PS', vid ), 'read_dust: failed to get PS id' )
            call handle_ncerr( nf_get_vara_double( ncid, vid, start(:3), count(:3), ps_nc ), 'read_dust: failed to read PS' )
            call regrid_2d( ps_nc, wrk, gndx, jl, ju, do_poles=.true. )
            ps_in(:,:,:,loin) = reshape( wrk, (/plonl,platl,pplon/), order = (/1,3,2/) )
!-----------------------------------------------------------------------
! 	... the dust concentration
!-----------------------------------------------------------------------
            start(:4) = (/ 1, jlim_in(1), 1, lotim /)
            count(:4) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, nlevs, 1 /)
	    do m = 1,nqdust
	       write(numa,'(i3)') 100+m
	       fldname = 'DSTQ' // numa(2:3)
	       errcom = 'read_dust: getting ' // trim(fldname) // ' id'
               call handle_ncerr( nf_inq_varid( ncid, trim(fldname), vid ), trim(errcom) )
	       errcom = 'read_dust: getting ' // trim(fldname)
               call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, dust_nc(1,jlim_in(1),1) ), trim(errcom) )
!-----------------------------------------------------------------------
! 	... regrid cam input dust
!-----------------------------------------------------------------------
	       do k = 1,nlevs
                  call regrid_2d( dust_nc(:,:,k), wrk, gndx, jl, ju, do_poles=.true. )
	          do ip = 1,pplon
	             dust_in(:plonl,k,m,:platl,ip,loin) = wrk((ip-1)*plonl+1:ip*plonl,:platl)
	          end do
	       end do
            end do
            if( masternode ) then
               write(*,*) 'read_dust: read data for day ',times(lotim)
            end if
         end if
         deallocate( dust_nc, ps_nc )
      end if

      end subroutine read_dust

      subroutine dust_interp( lat, ip, pmid, calday, dust, plonl )
!-----------------------------------------------------------------------
! 	... time interpolate dust to current time
!-----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lat                        ! latitude index
      integer, intent(in) :: ip                         ! longitude tile index
      integer, intent(in) :: plonl
      real, intent(in)    :: calday                     ! day of year
      real, intent(in)    :: pmid(plonl,plev)           ! midpoint pressure (Pa)
      real, intent(out)   :: dust(plonl,plev,nqdust)    ! output dust concentration (kg/kg)

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer ::  i, k, m                               ! working indicies
      integer ::  kl, ku                                ! working indicies
      real    ::  tint, dt, dt1                         ! interpolation time
      real    ::  pinterp                               ! model pressure (Pa)
      real    ::  delp                                  ! wrking variable (Pa)
      real    ::  press(plonl,nlevs,2)             ! dust pressure field (Pa)
      real    ::  pint_dust(plonl,plev,2,nqdust)        ! wrking dust concentation (kg/kg)

!-----------------------------------------------------------------------
!	... form the dust pressure field
!-----------------------------------------------------------------------
      do k = 1,nlevs
	 press(:,k,loin) = hyam(k)*P0 + hybm(k)*ps_in(:,lat,ip,loin)
	 press(:,k,hiin) = hyam(k)*P0 + hybm(k)*ps_in(:,lat,ip,hiin)
      end do

!-----------------------------------------------------------------------
! 	... vertical interpolation of dust data to model levels
!-----------------------------------------------------------------------
level_loop : &
      do k = 1,plev
long_loop : &
         do i = 1,plonl
	    pinterp = pmid(i,k)
	    if( pinterp < maxval( press(i,1,:2) ) ) then
	       pint_dust(i,k,loin,:) = 0.
	       pint_dust(i,k,hiin,:) = 0.
	    else
	       if( pinterp >= press(i,nlevs,loin) ) then
		  pint_dust(i,k,loin,:) = dust_in(i,nlevs,:,lat,ip,loin)
	       else
	          do ku = 2,nlevs
		     if( pinterp <= press(i,ku,loin) ) then
			kl = ku - 1
			delp =  log( pinterp/press(i,kl,loin) ) &
                             /  log( press(i,ku,loin)/press(i,kl,loin) )
			pint_dust(i,k,loin,:) = dust_in(i,kl,:,lat,ip,loin) &
                             + delp * (dust_in(i,ku,:,lat,ip,loin) - dust_in(i,kl,:,lat,ip,loin))
		        exit
		     end if
	          end do
	       end if
	       if( pinterp >= press(i,nlevs,hiin) ) then
		  pint_dust(i,k,hiin,:) = dust_in(i,nlevs,:,lat,ip,hiin)
		  cycle
	       else
	          do ku = 2,nlevs
		     if( pinterp <= press(i,ku,hiin) ) then
			kl = ku - 1
			delp = log( pinterp/press(i,kl,hiin) ) &
                             / log( press(i,ku,hiin)/press(i,kl,hiin) )
			pint_dust(i,k,hiin,:) = dust_in(i,kl,:,lat,ip,hiin) &
                             + delp * (dust_in(i,ku,:,lat,ip,hiin) - dust_in(i,kl,:,lat,ip,hiin))
		        exit
		     end if
	          end do
	       end if
	    end if
         end do long_loop
      end do level_loop
!-----------------------------------------------------------------------
!     	... linear interpolation     
!           start by computing the number of days between
!           the lower and upper bounds, and days between
!           the model time and lower bound.
!-----------------------------------------------------------------------
      if( times(hitim) < times(lotim) ) then
         dt = 365. - times(lotim) + times(hitim)
         if( calday <= times(hitim) )then
            dt1 = 365. - times(lotim) + calday
         else
            dt1 = calday - times(lotim)
         end if
      else
         dt = times(hitim) - times(lotim)
         dt1 = calday - times(lotim)
      end if

      tint = dt1 / dt
      do m = 1,nqdust
         call moz_linintp( plonl*plev, 0., 1., tint, &
                           pint_dust(1,1,loin,m), &
                           pint_dust(1,1,hiin,m), &
                           dust(1,1,m) )
      end do

      end subroutine dust_interp

      end module mo_dust
