
module mo_local_time

    use mo_grid, only : plon, plev

    implicit none

    private
    public  :: set_local_values
    public  :: local_time_inti
    public  :: local_time
    public  :: local_time_char

    integer, parameter :: max_files = 10
    integer            :: local_time(max_files) = -1
    character(len=5)   :: local_time_char(max_files)

    save

    real, allocatable    :: beg_lndx(:,:)
    real, allocatable    :: end_lndx(:,:)
    logical, allocatable :: has_grid_pnts(:,:)

  contains

  subroutine local_time_inti( file_cnt )
!---------------------------------------------------------------------------
!	... intialize module
!---------------------------------------------------------------------------

    use mo_mpi, only       : comp_node
    use mo_control, only   : secpday, delt, nspday

!---------------------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in) :: file_cnt

!---------------------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------------------
    integer :: astat
    integer :: file
    integer :: lbeg, lend
    integer :: n
    integer :: ncsec
    integer :: idta
    integer :: intval
    integer :: iwidth
    real    :: beg_ndx
    real    :: end_ndx
    real    :: rplon
    real    :: rsecpday
    real    :: dt
    real    :: width

comp_nodes_only : &
    if( comp_node ) then
       rplon           = real( plon )
       rsecpday        = real( secpday )
       dt              = rsecpday/rplon
       idta            = delt
!---------------------------------------------------------------------------
! last element: fixed value of delt for non-species
! this is the only legal value for non-species;
! (for troplev: other values senseless anyway, since
! averaging might include "missing values=-999.")
!---------------------------------------------------------------------------
       intval          = delt
!---------------------------------------------------------------------------
! next line to be changed by user if width other than delt is desired
!---------------------------------------------------------------------------
! elements 1:nspc: user definable widths for
! individual species in seconds; default value is delt.
!---------------------------------------------------------------------------
       iwidth          = max( idta,intval )
       width           = real(iwidth*plon)/rsecpday    ! width ("swath") as fraction of grid
!---------------------------------------------------------------------------
!	... setup the local time longitude indicies
!---------------------------------------------------------------------------
       allocate( beg_lndx(nspday,file_cnt),end_lndx(nspday,file_cnt),has_grid_pnts(nspday,file_cnt),stat=astat )
       if( astat /= 0 ) then
          write(*,*) 'local_time_init: failed to allocate beg_lndx,end_lndx; error = ',astat
          call endrun
       end if

       write(*,*) '========================================'
       write(*,*) 'local_time_init: diagnostics'
file_loop : &
       do file = 1,file_cnt
has_local_time : &
          if( local_time(file) > 0 ) then
diurnal_loop : &
             do n = 1,nspday
                ncsec = mod( n*delt,secpday )
                end_ndx = rplon*real(local_time(file) - ncsec)/rsecpday
                beg_ndx = rplon*real(local_time(file) - (ncsec-delt))/rsecpday
                if( end_ndx <= 0. ) then
                   end_ndx = rplon + end_ndx
                end if
                if( beg_ndx <= 0. ) then
                   beg_ndx = rplon + beg_ndx
                end if
                end_ndx = end_ndx + 1.
                beg_ndx = beg_ndx + 1.
!---------------------------------------------------------------------------
! 	... check if local_time traversed any grid points during this time step
!---------------------------------------------------------------------------
                lend = ceiling( end_ndx )
                if( end_ndx < beg_ndx ) then
                   lbeg = ceiling( beg_ndx )
                else
                   lbeg = ceiling( beg_ndx+rplon )
                end if
                write(*,'(i3,1p2g15.7,2i4)') n,beg_ndx,end_ndx,lbeg,lend
                has_grid_pnts(n,file) = lend < lbeg
                if( has_grid_pnts(n,file) ) then
                   beg_lndx(n,file) = beg_ndx
                   end_lndx(n,file) = end_ndx
                end if
             end do diurnal_loop
          end if has_local_time
       end do file_loop
    end if comp_nodes_only

    write(*,*) '========================================'

  end subroutine local_time_inti

  subroutine set_local_values( file, hist_buffer, field, fdim, ldim, &
                               plonl, lonp )
!---------------------------------------------------------------------------
!	... put local time values into history buffer
!---------------------------------------------------------------------------

    use mo_mpi, only     : comp_node
    use mo_control, only : nspday, time_step

!---------------------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in)    :: file
    integer, intent(in)    :: fdim
    integer, intent(in)    :: ldim
    integer, intent(in)    :: plonl
    integer, intent(in)    :: lonp
    real, intent(in)       :: field(fdim,*)
    real, intent(inout)    :: hist_buffer(:,:)

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
    real, parameter :: m2km = 1.e-3
    integer :: i, j, k, m, tndx
    integer :: il, ip
    integer :: ih
    integer :: intvl_beg(2)
    integer :: intvl_end(2)
    integer :: lon_lt_beg                   ! first (on grid; last in time) grid point index
                                            ! where time was local time during last time step
    integer :: lon_lt_end                   ! last (on grid; first in time) grid point index
                                            ! where time was local time during last time step

!   write(*,*) 'set_local_values: entered'
!   write(*,*) 'set_local_values: fdim,lonp,time_step,nspday = ',fdim,lonp,time_step,nspday
Compnode : &
    if( comp_node ) then
       tndx = mod( time_step - 1,nspday ) + 1
contains_grid_pnts : &
       if( has_grid_pnts(tndx,file) ) then
          if( mod( end_lndx(tndx,file),1. ) /= 0. ) then 
             lon_lt_beg = int( end_lndx(tndx,file) ) + 1
          else
             lon_lt_beg = int( end_lndx(tndx,file) )
          end if
          lon_lt_beg = mod( lon_lt_beg - 1,plon ) + 1
          if( mod( beg_lndx(tndx,file),1. ) /= 0. ) then
             lon_lt_end = int( beg_lndx(tndx,file) )
          else
             lon_lt_end = int( beg_lndx(tndx,file) ) - 1
          end if
          lon_lt_end = mod( lon_lt_end - 1,plon ) + 1
!---------------------------------------------------------------------------
! 	... check for wrap around plon
!---------------------------------------------------------------------------
          if( lon_lt_beg <= lon_lt_end ) then
             intvl_beg(1) = lon_lt_beg
             intvl_end(1) = lon_lt_end
             intvl_beg(2) = 1                  ! dummy value: loop over second intervall is not executed 
             intvl_end(2) = 0                  ! dummy value
          else
             intvl_beg(1) = lon_lt_beg
             intvl_end(1) = plon
             intvl_beg(2) = 1
             intvl_end(2) = lon_lt_end
          end if
 
!         write(*,*) 'set_local_values: tndx,intvl_beg,intvl_end = ',tndx,intvl_beg(1),intvl_end(1)
!---------------------------------------------------------------------------
! 	... species
!---------------------------------------------------------------------------
!         write(*,*) 'set_local_values: fdim, il, iu,lonp = ',fdim,intvl_beg(1),intvl_end(1),lonp
          do ih = 1,2
             do i = intvl_beg(ih),intvl_end(ih)
                ip = int((i-1)/plonl) + 1
                if( ip == lonp ) then
                   il = i - (ip - 1)*plonl
                   do k = 1,ldim
                      hist_buffer(il,k) = field(il,k)
                   end do
                end if
             end do
          end do
       end if contains_grid_pnts
    end if Compnode
!   write(*,*) 'set_local_values: exited'

  end subroutine set_local_values

end module mo_local_time
