
      module mo_mterps
!---------------------------------------------------------------------
!       ... Interactive calculation of BVOC emissions following Alex
!           Guenther's algorithms
!
! Jean-Francois Lamarque, March 2004
!
! Gabriele Pfister, October 2004
! Updated to new ISO Maps; MEA calculation according to MEGAN V1.0
! test code for monoterpenes emissions
!---------------------------------------------------------------------

      implicit none


      private
      public :: mterps_inti
      public :: mterps_emissions

      save

      integer           :: mterps_ndx
      real, allocatable :: maps_mterps(:,:,:,:)

      contains

      subroutine mterps_inti( plonl, platl, pplon, nlon_veg, nlat_veg, &
                              npft_veg, lon_veg, lon_veg_edge, lat_veg, lat_veg_edge, &
                              vegetation_map, landmask, monthly_lai_clm, monoterpene_ndx )
!-------------------------------------------------------------------------------------
! 	... intialize interactive bvoc emissions
!-------------------------------------------------------------------------------------

      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_double_precision, mpi_comm_comp
#endif
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : emis_flsp
      use netcdf

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: nlon_veg
      integer, intent(in) :: nlat_veg
      integer, intent(in) :: npft_veg
      integer, intent(in) :: monoterpene_ndx
      real, intent(in)    :: lon_veg(nlon_veg)
      real, intent(in)    :: lon_veg_edge(nlon_veg+1)
      real, intent(in)    :: lat_veg(nlat_veg)
      real, intent(in)    :: lat_veg_edge(nlat_veg+1)
      real, intent(in)    :: vegetation_map(nlon_veg,nlat_veg,npft_veg)
      real, intent(in)    :: landmask(nlon_veg,nlat_veg)
      real, intent(in)    :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer, parameter :: veg_types = 3
      integer :: i, j, ii, jj
      integer :: m, n, l, id
      integer :: astat, iret
      integer :: i1, j1, jmin, cnt
      integer :: ncid
      integer :: dimid, vid
      integer :: nlon_data
      integer :: nlat_data

      real              :: lat, avg
      real, allocatable :: maps_mterps_clm(:,:,:)
      real, allocatable :: maps_mterps_wk(:,:)

      character(len=80) :: filename
      character(len=80) :: err_msg
      character(len=4)  :: varname(3) = (/ 'Fmtp', 'Bmtp', 'Smtp' /)

      mterps_ndx = monoterpene_ndx

!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
      allocate( maps_mterps(plonl,platl,pplon,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'mterps_inti: failed to allocate maps_mterps; error = ',astat
         call endrun
      end if
!-------------------------------------------------------------------------------------
! 	... allocate and read monoterpene potential maps from Alex Guenther
!-------------------------------------------------------------------------------------
      allocate( maps_mterps_clm(nlon_veg,nlat_veg,veg_types),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'mterps_inti: failed to allocate maps_mterps_clm; error = ',astat
         call endrun
      end if

!-------------------------------------------------------------------------------------
! 	... map starts at 83.375
!-------------------------------------------------------------------------------------
      jmin = -99
      do j = 1,100
        lat = 89.875 - (j-1) * 0.25
        if( lat == 83.375 ) then
          jmin = j
          exit
        end if
      end do
      if( jmin < 0 ) then
        write(*,*) 'mterps_inti: failed to find jmin'
        call endrun
      else
        write(*,*) 'mterps_inti: jmin = ',jmin
      end if

master_only : &
      if( masternode ) then
!-------------------------------------------------------------------------------------
! 	... open mterps netcdf file
!-------------------------------------------------------------------------------------
         filename = 'bvoc_mterp.nc'
         ncid = open_netcdf_file( filename, trim(emis_flsp%local_path), trim(emis_flsp%remote_path), &
                                  masteronly=.true. )
!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), 'mterps_inti: dimension nlon not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon_data ), 'mterps_inti: failed to read nlon' )
         call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), 'mterps_inti: dimension nlat not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat_data ), 'mterps_inti: failed to read nlat' )
         allocate( maps_mterps_wk(nlon_data,nlat_data),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'mterps_inti: failed to allocate maps_mterps_wk; error = ',astat
            call endrun
         end if
         maps_mterps_wk(:,:) = 0.
!-------------------------------------------------------------------------------------
! 	... read data
!-------------------------------------------------------------------------------------
var_loop : &
         do n = 1,veg_types
            err_msg = 'mterps_inti: failed to find ' // varname(n) // ' id'
            call handle_ncerr( nf_inq_varid( ncid, varname(n), vid ), trim(err_msg) )
            err_msg = 'mterps_inti: failed to read ' // varname(n)
            call handle_ncerr( nf_get_var_double( ncid, vid, maps_mterps_wk ), trim(err_msg) )

            do j = 1,nlat_veg
              j1 = (j-1) * 2 + 1
              do i = 1,nlon_veg
                i1 = (i-1) * 2 + 1
                cnt = 0
                avg = 0.
                do jj = j1,j1+1
                  do ii = i1,i1+1
                    if( maps_mterps_wk(ii,jj) > 0. ) then
                      cnt = cnt + 1
                      avg = avg + maps_mterps_wk(ii,jj)
                    end if
                  end do
                end do
                if( cnt > 0 ) then
                  maps_mterps_clm(i,j,n) = avg / real(cnt)
                else 
                  maps_mterps_clm(i,j,n) = 0.
                end if 
              end do
            end do
          end do var_loop
!---------------------------------------------------------------------------
! 	... close netcdf file
!---------------------------------------------------------------------------
          err_msg = 'mterps_inti: error closing ' // trim(filename)
          call handle_ncerr( nf_close( ncid ), trim(err_msg) )
          deallocate( maps_mterps_wk, stat=astat )
      end if master_only

#ifdef USE_MPI
!------------------------------------------------------------------------------
!       ... bcast arrays
!------------------------------------------------------------------------------
      call mpi_bcast( maps_mterps_clm, nlon_veg*nlat_veg*veg_types , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'mterps_inti: failed to bcast maps_mterps_clm; error = ',iret
         call endrun
      end if
#endif

!---------------------------------------------------------------------------
! 	... regrid to model grid
!---------------------------------------------------------------------------
      call interp_map_bvoc( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                            lon_veg, lon_veg_edge, landmask, vegetation_map, monthly_lai_clm, &
                            maps_mterps_clm, plonl, platl, pplon )

      end subroutine mterps_inti

      subroutine interp_map_bvoc( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                                  lon_veg, lon_veg_edge, landmask, vegetation_map, monthly_lai_clm, &
                                  maps_mterps_clm, plonl, platl, pplon )
!-------------------------------------------------------------------------------------
!	... interpolate monoterpine surface emission from "land" grid
!           to model grid
!-------------------------------------------------------------------------------------

      use mo_grid, only      : plon, plat
      use mo_constants, only : r2d, phi, lam
      use m_adv, only        : has_npole
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)      :: plonl
      integer, intent(in)      :: platl
      integer, intent(in)      :: pplon
      integer, intent(in)      :: nlon_veg, nlat_veg, npft_veg
      real, intent(in)         :: landmask       (nlon_veg,nlat_veg)
      real, intent(in)         :: vegetation_map (nlon_veg,nlat_veg,npft_veg)
      real, intent(in)         :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)
      real, intent(in)         :: maps_mterps_clm   (nlon_veg,nlat_veg,       3   )
      real, intent(in)         :: lon_veg     (nlon_veg)
      real, intent(in)         :: lon_veg_edge(nlon_veg+1)
      real, intent(in)         :: lat_veg     (nlat_veg)
      real, intent(in)         :: lat_veg_edge(nlat_veg+1)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer, parameter :: veg_ext = 20
      real, parameter    :: maps_mterps_grass = 100.
      real, parameter    :: maps_mterps_agric = 200.

      integer                        :: i, j, ii, jj, jl, ju, i_ndx, n
      integer                        :: ind_lon(plon+1)
      integer                        :: ind_lat(platl+1)
      integer                        :: mapping_ext(-veg_ext:nlon_veg+veg_ext)
      real                           :: total_land
      real                           :: lon_edge(plon+1)
      real                           :: lat_edge(platl+1)
      real                           :: lat1, lat2, lon1, lon2
      real                           :: x1, x2, y1, y2, dx, dy
      real                           :: area, total_area
      real                           :: fraction(npft_veg)
      real                           :: total_mterp(12)
      real                           :: maps_mterps_3d(plon,platl,12)
      real                           :: work_mterp(12)
      real                           :: lon_veg_edge_ext(-veg_ext:nlon_veg+veg_ext)
      logical                        :: found

      jl = base_lat + 1
      ju = base_lat + platl

      do i = 1,plon
         lon_edge(i) = lam(i) * r2d - .5*(lam(2) - lam(1)) * r2d
      end do
      lon_edge(plon+1) = lon_edge(plon) + (lam(2) - lam(1)) * r2d
      if( .not. has_npole ) then
        do j = 1,platl+1
           lat_edge(j) = phi(j+base_lat) * r2d - .5*(phi(2) - phi(1)) * r2d
        end do
      else
        do j = 1,platl
           lat_edge(j) = phi(j+base_lat) * r2d - .5*(phi(2) - phi(1)) * r2d
        end do
        lat_edge(platl+1) = lat_edge(platl) + (phi(2) - phi(1)) * r2d
      end if
      do j = 1,platl+1
         lat_edge(j) = min( lat_edge(j), 90. )
         lat_edge(j) = max( lat_edge(j),-90. )
      end do

!-------------------------------------------------------------------------------------
! 	... wrap around the longitudes
!-------------------------------------------------------------------------------------
      do i = -veg_ext,0
        lon_veg_edge_ext(i) = lon_veg_edge(nlon_veg+i) - 360.
        mapping_ext     (i) =              nlon_veg+i
      end do
      do i = 1,nlon_veg
        lon_veg_edge_ext(i) = lon_veg_edge(i)
        mapping_ext     (i) =              i
      end do
      do i = nlon_veg+1,nlon_veg+veg_ext
        lon_veg_edge_ext(i) = lon_veg_edge(i-nlon_veg) + 360.
        mapping_ext     (i) =              i-nlon_veg
      end do
#ifdef DEBUG
      write(*,*) 'interp_map_bvoc : lon_edge ',lon_edge
      write(*,*) 'interp_map_bvoc : lat_edge ',lat_edge
      write(*,*) 'interp_map_bvoc : mapping_ext ',mapping_ext
#endif
      do j = 1,plon+1
        lon1 = lon_edge(j) 
        found = .false.
        do i = -veg_ext,nlon_veg+veg_ext-1
          dx = lon_veg_edge_ext(i  ) - lon1
          dy = lon_veg_edge_ext(i+1) - lon1
          if( dx*dy <= 0. ) then
            ind_lon(j) = i
            found = .true.
            exit
          end if
        end do
        if( .not. found ) then
           write(*,*) ' '
           write(*,*) 'interp_map_bvoc: failed to find interval for j,lon1 = ',j,lon1
           write(*,*) 'interp_map_bvoc: lon_veg_edge_ext(',-veg_ext,':',-veg_ext+1,') = ', &
                      lon_veg_edge_ext(-veg_ext:-veg_ext+1)
           write(*,*) 'interp_map_bvoc: lon_veg_edge_ext(', &
                      nlon_veg+veg_ext-1,':',nlon_veg+veg_ext,') = ', &
                      lon_veg_edge_ext(nlon_veg+veg_ext-1:nlon_veg+veg_ext)
           write(*,*) ' '
           call endrun
        end if
      end do

      do j = 1,platl+1
        lat1 = lat_edge(j)
        found = .false.
        do i = 1,nlat_veg
          dx = lat_veg_edge(i  ) - lat1
          dy = lat_veg_edge(i+1) - lat1
          if( dx*dy <= 0. ) then
            ind_lat(j) = i
            found = .true.
            exit
          end if
        end do
        if( .not. found ) then
           write(*,*) ' '
           write(*,*) 'interp_map_bvoc: failed to find interval for j,lat1 = ',j,lat1
           write(*,*) 'interp_map_bvoc: lat_veg_edge(1:2) = ',lat_veg_edge(1:2)
           write(*,*) 'interp_map_bvoc: lat_veg_edge(',nlat_veg,':',nlat_veg+1,') = ',lat_veg_edge(nlat_veg:nlat_veg+1)
           write(*,*) ' '
           call endrun
        end if
      end do

#ifdef DEBUG
      write(*,*) 'interp_map_bvoc : ind_lon ',ind_lon
      write(*,*) 'interp_map_bvoc : ind_lat ',ind_lat
#endif

lat_loop : &
      do j = 1,platl
lon_loop : &
        do i = 1,plon
          total_area = 0.
          fraction   = 0.
          total_mterp  = 0.
          do jj = ind_lat(j),ind_lat(j+1)
            y1 = max( lat_edge(j),lat_veg_edge(jj) )
            y2 = min( lat_edge(j+1),lat_veg_edge(jj+1) ) 
            dy = (y2 - y1)/(lat_veg_edge(jj+1) - lat_veg_edge(jj))
            do ii =ind_lon(i),ind_lon(i+1)
              i_ndx = mapping_ext(ii)
              x1 = max( lon_edge(i),lon_veg_edge_ext(ii) )
              x2 = min( lon_edge(i+1),lon_veg_edge_ext(ii+1) ) 
              dx = (x2 - x1)/(lon_veg_edge_ext(ii+1) - lon_veg_edge_ext(ii))
              area = dx * dy
              total_area = total_area + area
!-----------------------------------------------------------------
! 	... no emissions for ocean grid point 
!-----------------------------------------------------------------
              if( nint(landmask(i_ndx,jj)) == 0 ) cycle
!-------------------------------------------------------------------------------------
! 	... emissions from fine leaf
!-------------------------------------------------------------------------------------
              work_mterp(:) = fac_lai(monthly_lai_clm(i_ndx,jj, 2,:)) * vegetation_map(i_ndx,jj, 2) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj, 3,:)) * vegetation_map(i_ndx,jj, 3) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj, 4,:)) * vegetation_map(i_ndx,jj, 4)
              total_mterp(:) = total_mterp(:) + work_mterp(:) * maps_mterps_clm(i_ndx,jj,1) * area
!-------------------------------------------------------------------------------------
! 	... emissions from broadleaf
!-------------------------------------------------------------------------------------
              work_mterp(:) = fac_lai(monthly_lai_clm(i_ndx,jj, 5,:)) * vegetation_map(i_ndx,jj, 5) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj, 6,:)) * vegetation_map(i_ndx,jj, 6) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj, 7,:)) * vegetation_map(i_ndx,jj, 7) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj, 8,:)) * vegetation_map(i_ndx,jj, 8) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj, 9,:)) * vegetation_map(i_ndx,jj, 9)
              total_mterp(:) = total_mterp(:) + work_mterp(:) * maps_mterps_clm(i_ndx,jj,2) * area
!-------------------------------------------------------------------------------------
! 	... emissions from shrubs
!-------------------------------------------------------------------------------------
              work_mterp(:) = fac_lai(monthly_lai_clm(i_ndx,jj,10,:)) * vegetation_map(i_ndx,jj,10) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj,11,:)) * vegetation_map(i_ndx,jj,11) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj,12,:)) * vegetation_map(i_ndx,jj,12)
              total_mterp(:) = total_mterp(:) + work_mterp(:) * maps_mterps_clm(i_ndx,jj,3) * area
!-------------------------------------------------------------------------------------
! 	... emissions from grass
!-------------------------------------------------------------------------------------
              work_mterp(:) = fac_lai(monthly_lai_clm(i_ndx,jj,13,:)) * vegetation_map(i_ndx,jj,13) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj,14,:)) * vegetation_map(i_ndx,jj,14) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj,15,:)) * vegetation_map(i_ndx,jj,15)
              total_mterp(:) = total_mterp(:) + work_mterp(:) * maps_mterps_grass * area
!-------------------------------------------------------------------------------------
! 	... emissions from agriculture
!-------------------------------------------------------------------------------------
              work_mterp(:) = fac_lai(monthly_lai_clm(i_ndx,jj,16,:)) * vegetation_map(i_ndx,jj,16) &
                          + fac_lai(monthly_lai_clm(i_ndx,jj,17,:)) * vegetation_map(i_ndx,jj,17)
              total_mterp(:) = total_mterp(:) + work_mterp(:) * maps_mterps_agric * area
            end do
          end do

!-------------------------------------------------------------------------------------
! 	... divide by total grid area
!-------------------------------------------------------------------------------------
          maps_mterps_3d(i,j,:) = total_mterp(:)/(total_area+1.e-30)
        end do lon_loop
      end do lat_loop

!-------------------------------------------------------------------------------------
! 	... reshape according to lat-lon blocks
!-------------------------------------------------------------------------------------
      do i=1,12
        maps_mterps(:,:,:,i) = reshape( maps_mterps_3d(:,:,i), (/plonl,platl,pplon/), order=(/1,3,2/) )
      end do

      end subroutine interp_map_bvoc

      function fac_lai( clai )

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      real, intent(in)  :: clai(:)

!-------------------------------------------------------------------------------------
! 	... function declaration
!       Monthly exchange Ratio according to MEGAN (Version 1.0)
!-------------------------------------------------------------------------------------
      real :: fac_lai(size(clai))   

      fac_lai(:) = .2*clai(:)

      end function fac_lai

      subroutine mterps_emissions( lat, ip, calday, ts, fsds, &
                                 sflx, plonl )

      use mo_grid,   only : pcnst
      use chem_mods, only : adv_mass

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: lat             ! latitude index
      integer, intent(in) :: ip              ! longitude tile index
      integer, intent(in) :: plonl
      real, intent(in)    :: calday          ! julian day + fraction (greenwich)
      real, intent(in)    :: ts(plonl)       ! surface temperature
      real, intent(in)    :: fsds(plonl)     ! surface direct radiation (w/m^2)
      real, intent(inout) :: sflx(plonl)     ! surface flux for advected species

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer, parameter :: days_in_month(12) = (/ 31,28,31,30,31,30,31,31,30,31,30,31 /)
      real, parameter :: alpha  = 0.001
      real, parameter :: ctm1   =  70.
      real, parameter :: ctm2   = 200.
      real, parameter :: topt   = 317.
      real, parameter :: tstd   = 303.15
      real, parameter :: amufac = 1.65979e-23         ! 1.e4* kg / amu

      integer :: month,total_days
      integer :: i,n
      real    :: ppfd, x
      real    :: total_area, total_emis, fac_par, fac_tmp
      real    :: bmterp_flx(plonl)

!-------------------------------------------------------------------------------------
! 	... find month from calendar day
!-------------------------------------------------------------------------------------
      month = 0
      if( calday >= 365. ) then
        month = 12
      else
        do n = 1,12
          total_days = sum( days_in_month(1:n) )
          if( calday <= total_days ) then
            month = n
            exit
          end if
        end do
      end if
      if( month == 0 ) then
        write(*,*) 'mterps_emissions: cannot find month for calday = ',calday
        call endrun
      end if

      do i = 1,plonl
!-------------------------------------------------------------------------------------
! 	... temperature correction
!-------------------------------------------------------------------------------------
         x = ts(i) - tstd
         fac_tmp = exp( .09*x )
         fac_par = 1.
!-------------------------------------------------------------------------------------
! 	... regridded potential emissions, including LAI correction
!-------------------------------------------------------------------------------------
        total_emis = maps_mterps(i,lat,ip,month)
!-------------------------------------------------------------------------------------
! 	... change units from microg/m2/h to mol/cm2/s
!-------------------------------------------------------------------------------------
        bmterp_flx(i) = total_emis * fac_par * fac_tmp * 1.229e8
      end do
!-------------------------------------------------------------------------------------
! 	... change to surface flux units for mozart
!-------------------------------------------------------------------------------------
      sflx(:) = sflx(:) + bmterp_flx(:) * amufac * adv_mass(mterps_ndx)

      end subroutine mterps_emissions

      end module mo_mterps
