
module mo_seasalt
!----------------------------------------------------------------
!	... seasalt aerosol module
!----------------------------------------------------------------
! jfl inclusion of beta
!----------------------------------------------------------------

  private
  public :: seasalt_inti, set_seasalt, seasalt_sett_vel
  public :: has_seasalt, ltop, nbin

  save

  integer, parameter :: nbin = 4
  integer, dimension(nbin), parameter :: ndt   = (/1,       2,       4,       4  /)     !
  real, parameter :: dry_mass = 28.966
  real, parameter :: gs_a     =  1.257           ! gravitational settling constants
  real, parameter :: gs_b     =  0.4             ! from Seinfeld and Pandis (1998)
  real, parameter :: gs_c     =  1.1
  real, parameter :: c1       =  0.7674          ! parameter for growth with humdity (from Gerber, 1985)
  real, parameter :: c2       =  3.0790          ! parameter for growth with humdity
  real, parameter :: c3       =  2.57e-11        ! parameter for growth with humdity
  real, parameter :: c4       = -1.424           ! parameter for growth with humdity
  real, parameter :: z0       =  1.e-4           ! ocean surface roughness (m)
  real, parameter :: dyn_visc =  1.5e-5          ! dyn viscosity (kg/m2/s)   
  real, parameter :: cnst0    =  2./9.
  real, parameter :: top_plev =  1.e-10          ! top level for settling (Pa)
  real, dimension(nbin), parameter :: dr       = (/ 0.05,    0.05,    0.05,    0.05 /)    ! um
  real, dimension(nbin), parameter :: ra       = (/ 0.1,     0.5,     1.5,     5. /)      ! um
  real, dimension(nbin), parameter :: rb       = (/ 0.5,     1.5,     5.0,     10.0 /)    ! um
  real, dimension(nbin), parameter :: ssaltden = (/ 2200.,   2200.,   2200.,   2200. /)   ! kg/m3
  real, dimension(nbin), parameter :: ssaltref = (/ 0.26e-6, 1.19e-6, 2.43e-6, 7.57e-6 /) ! m

  integer           :: ltop                      ! top level of sea-salt calculation 
  integer, target   :: spc_ndx(4)
  integer, pointer  :: sa1_ndx, sa2_ndx, sa3_ndx, sa4_ndx

  real              :: mtv
  real              :: mtvi
  real, allocatable :: salt_source(:)
  logical           :: has_seasalt = .true.

!----------------------------------------------------------------
! jfl : from Paul Ginoux, November 04
!
! Sea salt hygroscopic growth factor from 35 to 99% RH
! We start at the deliquescence point of sea-salt for RH=37%
! Any lower RH doesn't affect dry properties
! Reference: Tang et al., JGR, v102(D19), 23,269-23,275, 1997.
!----------------------------------------------------------------
      integer, parameter   :: nrh = 65
      real :: growth_table(nrh) = (/ 1.000, 1.000, 1.396, &
       1.413, 1.428, 1.441, 1.454, 1.466, 1.478, 1.490, 1.501, 1.512, &
       1.523, 1.534, 1.545, 1.555, 1.566, 1.577, 1.588, 1.599, 1.610, &
       1.621, 1.632, 1.644, 1.655, 1.667, 1.679, 1.692, 1.704, 1.717, &
       1.730, 1.743, 1.757, 1.771, 1.786, 1.801, 1.816, 1.832, 1.849, &
       1.866, 1.884, 1.903, 1.923, 1.944, 1.966, 1.990, 2.014, 2.041, &
       2.069, 2.100, 2.134, 2.170, 2.210, 2.255, 2.306, 2.363, 2.430, &
       2.509, 2.605, 2.723, 2.880, 3.087, 3.402, 3.919, 5.048 /)

!----------------------------------------------------------------
! Sea salt density for 65 RH values from 35% to 99% [g/cm3]
!----------------------------------------------------------------
      real :: rho_table(nrh) = (/ 2.160, 2.160, 1.490, &
       1.475, 1.463, 1.452, 1.441, 1.432, 1.422, 1.414, 1.406, 1.398, &
       1.390, 1.382, 1.375, 1.368, 1.361, 1.354, 1.347, 1.341, 1.334, &
       1.328, 1.322, 1.315, 1.309, 1.303, 1.297, 1.291, 1.285, 1.279, &
       1.273, 1.267, 1.261, 1.255, 1.249, 1.243, 1.237, 1.231, 1.225, &
       1.219, 1.213, 1.207, 1.201, 1.195, 1.189, 1.183, 1.176, 1.170, &
       1.163, 1.156, 1.150, 1.142, 1.135, 1.128, 1.120, 1.112, 1.103, &
       1.094, 1.084, 1.074, 1.063, 1.051, 1.038, 1.025, 1.011 /)

contains

subroutine seasalt_inti( plonl, platl, plev, pplon, oro, ref_pmid )
!----------------------------------------------------------------
!	... initialize sea salt routine
!----------------------------------------------------------------

  use mo_chem_utls, only : get_spc_ndx
  use mo_constants, only : pi
  use chem_mods,    only : adv_mass

  implicit none

!----------------------------------------------------------------
!	... dummy arguments
!----------------------------------------------------------------
  integer, intent(in) :: plonl
  integer, intent(in) :: platl
  integer, intent(in) :: plev
  integer, intent(in) :: pplon
  real,    intent(in) :: oro(plonl,platl,pplon)
  real,    intent(in) :: ref_pmid(plev)

!----------------------------------------------------------------
!	... local variables
!----------------------------------------------------------------
  integer :: j, k, n, nr, ir, ip
  integer :: astat
  real    :: rmid, r0, r1, src
  real    :: const0
  real    :: beta

  beta = growth_table(41)  ! Growth factor at 80% RH
  sa1_ndx => spc_ndx(1)
  sa2_ndx => spc_ndx(2)
  sa3_ndx => spc_ndx(3)
  sa4_ndx => spc_ndx(4)
!----------------------------------------------------------------
! 	... set species index
!----------------------------------------------------------------
  sa1_ndx = get_spc_ndx( 'SA1' )
  sa2_ndx = get_spc_ndx( 'SA2' )
  sa3_ndx = get_spc_ndx( 'SA3' )
  sa4_ndx = get_spc_ndx( 'SA4' )
  has_seasalt = all( spc_ndx(:) > 0 )
  if( .not. has_seasalt ) then
     return
  else
    write(*,*) '-----------------------------------------'
    write(*,*) 'mozart will do seasalt aerosols'
    write(*,*) '-----------------------------------------'
  end if

!----------------------------------------------------------------
! mass conversion factor
!----------------------------------------------------------------
  mtv  = dry_mass/adv_mass(sa4_ndx)
  mtvi = 1./mtv

!----------------------------------------------------------------
! 	... define salt source
!           (will be multiplied by wind speed and timestep in mo_seasalt)
!----------------------------------------------------------------
  allocate( salt_source(nbin),stat=astat )
  if( astat /= 0 ) then
     write(*,*) 'seasalt_inti: failed to allocate salt_source; error = ',astat
     call endrun
  end if
  salt_source(:) = 0.

!----------------------------------------------------------------
!	... find top level for settling
!----------------------------------------------------------------
  do k = plev,1,-1
     if( ref_pmid(k) <= top_plev ) then
        exit
     end if
  end do
  ltop = max( 2,k )
  write(*,*) 'seasalt_inti: seasalt settling will be limited to bottom ',plev-ltop+1, &
             ' model levels. top is ',1.e-2*ref_pmid(ltop),' hpa'

!----------------------------------------------------------------
! formula is from Gong S., L. Barrie, and J.-P. Blanchet, Modeling sea-salt
! aerosols in the atmosphere. 1. model development, JGR, 102, 3805-3818, 1997.
!----------------------------------------------------------------
bin_loop : &
  do n = 1,nbin
    r0 = ra(n)
    r1 = rb(n)
    nr = (r1 - r0)/dr(n)
    const0  = 4./3.*pi*dr(n)*ssaltden(n)/beta**2*1.e-18*1.373
radius_loop : &
    do ir = 1,nr
      rmid = r0 + dr(n) * 0.5 + (ir-1) * dr(n)
      src  = const0*(1. + .057*(beta*rmid)**1.05) &                        ! src in kg(Na)/m^2 over the ocean
                    *10**(1.19*exp( -((0.38 - log10(beta*rmid))/0.65)**2 ) )
      salt_source(n) = salt_source(n) + src
    end do radius_loop
  end do bin_loop

end subroutine seasalt_inti

subroutine set_seasalt( pmid, lat, dtime, ip, tfld, &
                        um1, vm1, rh, pdel, zm, &
                        xhnm, vmr, oro, plonl )
!-----------------------------------------------------------------------      
!      ... seasalt
!-----------------------------------------------------------------------      

  use mo_grid,      only : plev, pcnstm1, plevp
  use mo_chem_utls, only : get_spc_ndx
  use mo_constants, only : rgrav
  use mo_histout,   only : outfld, sim_file_cnt

  implicit none

!-----------------------------------------------------------------------      
!      ... input arguments
!-----------------------------------------------------------------------      
  integer, intent(in)                  :: lat                     ! latitude index
  integer, intent(in)                  :: ip                      ! longitude tile index
  integer, intent(in)                  :: plonl                   ! longitude tile dimension
  real, intent(in)                     :: dtime                   ! time step (sec)
  real, intent(in)                     :: xhnm(plonl,plev)        ! total atms density (/cm**3)
  real, intent(in)                     :: tfld(plonl,plev)        ! temperature (K)
  real, intent(in)                     :: um1(plonl,plev)         ! midpoint wind (m/s)
  real, intent(in)                     :: vm1(plonl,plev)         ! midpoint wind (m/s)
  real, intent(in)                     :: rh(plonl,plev)          ! relative humidity (fraction)
  real, intent(in)                     :: pmid(plonl,plev)        ! midpoint pressure (Pa)
  real, intent(in)                     :: pdel(plonl,plev)        ! delta press across midpoints (Pa)
  real, intent(in)                     :: zm(plonl,plev)          ! abs geopot height at interfa (m)
  real, intent(in)                     :: oro(plonl)              ! orography flag (0 == open ocean)
  real, intent(inout)                  :: vmr(plonl,plev,pcnstm1) ! species concentrations (mol/mol)
      
!-----------------------------------------------------------------------      
!     	... local variables
!-----------------------------------------------------------------------      
      real, parameter :: open_ocean  = 0.
      real, parameter :: small_value = 1.e-20

      integer                          :: i, j, k, l, n, m, ir
      integer                          :: nr
      integer                          :: astat
      integer                          :: file
      real                             :: wrk
      real                             :: vd_cor          ! settling velocity (m/s)
      real, dimension(plonl,plev)      :: delz            ! delz about midpoints (m)
      real, dimension(plonl,plev,nbin) :: tc              ! temp array for sea-salt (kg/kg)
      real, dimension(plonl)           :: airmas          ! surface airmass (kg/m^2)
      real, dimension(plonl)           :: w10m            ! 10 meter wind   (m/s)
      real, dimension(plonl,nbin)      :: salt_src        ! wrk salt src
      real, dimension(nbin) :: dt_settl                   ! split time step for advection (s)

      real :: r0, r1, r, rmid, src
      real :: sa_gsv(plonl,plev,nbin)
      real :: src_tot(plonl,nbin)
      real, allocatable :: settling_velocity(:,:,:)

      sa_gsv(:,:,:) = 0.

!-----------------------------------------------------------------------      
! 	... calculate inverse delz
!-----------------------------------------------------------------------      
  do k = ltop,plev
     delz(:,k) = 1./(zm(:,k-1) - zm(:,k))
  end do

!-----------------------------------------------------------------------      
! find horizontal wind at 10 meter
!-----------------------------------------------------------------------      
  w10m(:) = sqrt( um1(:,plev)*um1(:,plev) + vm1(:,plev)*vm1(:,plev) )   &
            * log(10.0/z0)/log( max( 20.,zm(:,plev) )/z0 )

!-----------------------------------------------------------------------      
! convert from volume mixing ratio to mass mixing ratio 
!-----------------------------------------------------------------------      
  do k = 1,plev
     tc(:,k,1) = vmr(:,k,sa1_ndx) * mtvi
     tc(:,k,2) = vmr(:,k,sa2_ndx) * mtvi
     tc(:,k,3) = vmr(:,k,sa3_ndx) * mtvi
     tc(:,k,4) = vmr(:,k,sa4_ndx) * mtvi
  end do

!-----------------------------------------------------------------------      
! calculate surface air mass (kg/m2) 
!-----------------------------------------------------------------------      
  airmas(:) = pdel(:,plev) *rgrav

!-----------------------------------------------------------------------      
! put source in the lowest layer of the model
!-----------------------------------------------------------------------      
  do n = 1,nbin
     src = salt_source(n)
     where( oro(:) == open_ocean )
	salt_src(:,n) = src
     elsewhere
	salt_src(:,n) = 0.
     endwhere
  end do
  do n = 1,nbin
     tc(:,plev,n) = tc(:,plev,n) + salt_src(:,n)/airmas(:) * dtime * w10m(:)**3.41
     src_tot(:,n) = salt_src(:,n)/airmas(:) * dtime * w10m(:)**3.41
  end do

!============================================
!  settling of sea-salt
!============================================
!-----------------------------------------------------------------------      
! calculate settling velocity
!-----------------------------------------------------------------------      
  allocate( settling_velocity(plonl,ltop-1:plev,nbin),stat=astat )
  if( astat /= 0 ) then
     write(*,*) 'set_seasalt: failed to allocate settling_velocity; error = ',astat
     call endrun
  end if
  call seasalt_sett_vel( pmid(:,ltop-1:plev), tfld(:,ltop-1:plev), &
                         rh(:,ltop-1:plev), settling_velocity )
!-----------------------------------------------------------------------      
! solve the transport equation semi-implicitly
! note that the surface deposition (including the effect
! of the gravitational settling velocity) is done in mo_drydep.F90
!-----------------------------------------------------------------------      
  do m = 1,nbin
    dt_settl(m) = dtime/ndt(m)
    do n = 1,ndt(m)
!-----------------------------------------------------------------------      
! assume no sea-salt coming from above ltop
!-----------------------------------------------------------------------      
      do i = 1,plonl
        vd_cor = 0.5 * (settling_velocity(i,ltop,m) + settling_velocity(i,ltop-1,m))
        tc(i,ltop,m) = tc(i,ltop,m)/(1. + dt_settl(m)*vd_cor*delz(i,ltop)) 
      end do
!-----------------------------------------------------------------------      
! solve for the rest of the domain
!-----------------------------------------------------------------------      
      do k = ltop+1,plev
        do i = 1,plonl
          vd_cor = 0.5 * (settling_velocity(i,k,m) + settling_velocity(i,k-1,m))
          wrk    = dt_settl(m)*vd_cor*delz(i,k)
          tc(i,k,m)     = (tc(i,k,m) + wrk*tc(i,k-1,m))/(1. + wrk)
          sa_gsv(i,k,m) = vd_cor
        end do
      end do
    end do
  end do

!==============================================================
!       ... update the mixing ratios
!==============================================================
  do k = ltop,plev
    vmr(:,k,sa1_ndx) = max( tc(:,k,1) * mtv, small_value )
    vmr(:,k,sa2_ndx) = max( tc(:,k,2) * mtv, small_value )
    vmr(:,k,sa3_ndx) = max( tc(:,k,3) * mtv, small_value )
    vmr(:,k,sa4_ndx) = max( tc(:,k,4) * mtv, small_value )
  end do

  do file = 1,sim_file_cnt
!-----------------------------------------------------------------------      
! output w10m
!-----------------------------------------------------------------------      
     call outfld( 'U_10m',w10m(1), plonl, ip, lat, file )
!-----------------------------------------------------------------------      
! output surface sources
!-----------------------------------------------------------------------      
     call outfld ('SA1_SRC',src_tot(1,1), plonl, ip, lat, file )
     call outfld ('SA2_SRC',src_tot(1,2), plonl, ip, lat, file )
     call outfld ('SA3_SRC',src_tot(1,3), plonl, ip, lat, file )
     call outfld ('SA4_SRC',src_tot(1,4), plonl, ip, lat, file )
!-----------------------------------------------------------------------      
! output gravitational settling velocity
!-----------------------------------------------------------------------      
     call outfld ('SA1_GSV',sa_gsv(1,1,1), plonl, ip, lat, file )
     call outfld ('SA2_GSV',sa_gsv(1,1,2), plonl, ip, lat, file )
     call outfld ('SA3_GSV',sa_gsv(1,1,3), plonl, ip, lat, file )
     call outfld ('SA4_GSV',sa_gsv(1,1,4), plonl, ip, lat, file )
  end do

  deallocate( settling_velocity )

end subroutine set_seasalt

subroutine seasalt_sett_vel( pmid, tfld, rh, sett_vel, ustar, rb_part )
!-----------------------------------------------------------------------      
!	... seasalt settling velocity
!-----------------------------------------------------------------------      

  use mo_constants, only : gravit, pi

  implicit none

!-----------------------------------------------------------------------      
!	... dummy arguments
!-----------------------------------------------------------------------      
  real, intent(in)    :: pmid(:,:)                ! midpoint pressure (Pa)
  real, intent(in)    :: tfld(:,:)                ! temperature (K)
  real, intent(in)    :: rh(:,:)                  ! relative humidity
  real, intent(out)   :: sett_vel(:,:,:)          ! settling velocity (cm/s)
  real, intent(in), optional  :: ustar   (:,:)    ! u*
  real, intent(out), optional :: rb_part (:,:,:)  ! quasi-laminar resistance for particles

!-----------------------------------------------------------------------      
!	... local variables
!-----------------------------------------------------------------------      
  real, parameter    :: Pa2mb = 0.01           ! pascal to mb
  real, parameter    :: m2cm  =  100.          ! meters to centimeters
  real, parameter    :: sb    = 5.671e-8       ! Stefan-Boltzmann constant
  real, parameter    :: rhmin = 1.e-6          ! minimum relative humidity
  real, parameter    :: lnd_pwr = -2./3.       ! rb_part power factor for land
  real, parameter    :: ocn_pwr = -.5          ! rb_part power factor for ocean

  integer :: lons, levs, bins
  integer :: i, k, n, m, n_lt
  integer :: astat
  integer :: irh
  real    :: pres, rhb, rcm
  real    :: rwet,   &       ! wet radius
             rho,    &       ! density of the wet aerosol (kg/m3)
             c_stokes, &     ! dynamic viscosity
             free_path       ! mean free path  
  real    :: c_c
  real    :: diff_aer
  real    :: sc              ! Schmidt number
  real    :: st              ! Stokes number
  real    :: wrk
  real    :: pwr

  lons = size( sett_vel,dim=1 )
  levs = size( sett_vel,dim=2 )
  bins = size( sett_vel,dim=3 )
  if( present(ustar) ) then
     n_lt = size( ustar,dim=2 )
  end if
!-----------------------------------------------------------------------      
! 	... calculate settling velocity
!-----------------------------------------------------------------------      
bin_loop : &
  do n = 1,bins
    rcm = ssaltref(n) * m2cm                                                    ! express radius in cm
    rho = ssaltden(n)                                                           ! sea-salt density
    do k = 1,levs
      do i = 1,lons
         pres      = pmid(i,k)*Pa2mb
         c_stokes  = 1.458e-6 * tfld(i,k)**1.5/(tfld(i,k) + 110.4)              ! dynamic viscosity
         free_path = 1.1e-3/(pres*sqrt( tfld(i,k) ))                            ! mean free path
         rhb       = max( rh(i,k),rhmin )
         rwet      = 0.01*(c1*rcm**c2/(c3*rcm**c4 - log(rhb)) + rcm**3)**0.33   ! particle growth (Gerber, 1985)
         sett_vel(i,k,n) = &                                                    ! settling_velocity is used in drydep
                 cnst0*gravit*rho*rwet**2/c_stokes &
                 * (1. + (free_path/rwet) &                                     ! settling velocity (Seinfeld and Pandis, p 466, 1998)
                  * (gs_a + gs_b*exp( -gs_c*rwet/free_path )))                  ! includes slip correction factor
      end do
    end do

!-----------------------------------------------------------------------      
! compute rb for particles
!-----------------------------------------------------------------------      
    if( present(ustar) ) then
       k = levs
       do i = 1,lons
          pres      = pmid(i,k)*Pa2mb
          c_stokes  = 1.458e-6 * tfld(i,k)**1.5/(tfld(i,k) + 110.4)              ! dynamic viscosity
          free_path = 1.1e-3/(pres*sqrt( tfld(i,k) ))                            ! mean free path
          rhb       = max( rh(i,k),rhmin )
          rwet      = 0.01*(c1*rcm**c2/(c3*rcm**c4 - log(rhb)) + rcm**3)**0.33   ! particle growth (Gerber, 1985)
          c_c       = 1. + free_path/rwet * (1.257+0.4*exp(-1.1*rwet/free_path))
          diff_aer  = sb * tfld(i,k) * c_c / ( 3.*pi*c_stokes*2.*rwet)
!-----------------------------------------------------------------------      
! for the following formulas, see Seinfeld and Pandis (1997) p 964-965
!-----------------------------------------------------------------------      
          do m = 1,n_lt
             if( ustar(i,m) > 0. ) then
                st = sett_vel(i,k,n) * ustar(i,m) * ustar(i,m) / (gravit*c_stokes)   
                sc = c_stokes/diff_aer
                if( m /= 7 ) then
                   pwr = lnd_pwr                                           ! land
                else
                   pwr = ocn_pwr                                           ! ocean
                end if
                rb_part(i,n,m) = 1./(ustar(i,m) * (sc**pwr + 10.**(-3./st)))
             else
                rb_part(i,n,m) = 0.
             end if
          end do
       end do
    end if
  end do bin_loop

end subroutine seasalt_sett_vel

end module mo_seasalt
