
      module mo_seto2

      implicit none

      private
      public :: o2_xsect_inti
      public :: set_o2_xsect

      save

      integer :: nsrc
      integer :: ngast
      integer :: nla
      integer :: nwint
      real, allocatable    :: wlint(:)
      real, allocatable    :: xso2int(:)
      real, allocatable    :: wlla(:)
      real, allocatable    :: wlgast(:)

      contains

      subroutine o2_xsect_inti
!-----------------------------------------------------------------------------
!   purpose:
!   compute equivalent optical depths for o2 absorption, parameterized in
!   the sr bands and the lyman-alpha line.
!-----------------------------------------------------------------------------
!   parameters:
!   nz      - integer, number of specified altitude levels in the working (i)
!             grid
!   z       - real, specified altitude working grid (km)                  (i)
!   nw      - integer, number of specified intervals + 1 in working       (i)
!             wavelength grid
!   wl      - real, vector of lower limits of wavelength intervals in     (i)
!             working wavelength grid
!   cz      - real, number of air molecules per cm^2 at each specified    (i)
!             altitude layer
!   zen     - real, solar zenith angle                                    (i)
!   dto2    - real, optical depth due to o2 absorption at each specified  (o)
!             vertical layer at each specified wavelength
!   xso2    - real, molecular absorption cross section in sr bands at     (o)
!             each specified altitude and wavelength.  includes herzberg
!             continuum.
!-----------------------------------------------------------------------------

      use mo_params,     only : deltax
      use mo_inter,      only : inter2
      use mo_inter,      only : inter_inti
      use mo_wavelen,    only : nw, wl
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : photo_xs_o2_flsp
      use mo_mpi
      use netcdf

      implicit none

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: ncid
      integer :: dimid
      integer :: vid
      integer :: astat
      integer :: iret
      integer :: i, wn, n
      integer :: wrk_ind(4)
      real, allocatable :: x1(:), y1(:)
!-----------------------------------------------------------------------------
! 	... cross section data for use outside the sr-bands (combined from
!           brasseur and solomon and the jpl 1994 recommendation)
!-----------------------------------------------------------------------------

!-----------------------------------------------------------------------------
! 	... read o2 cross section data outside sr-bands
!-----------------------------------------------------------------------------
!	... o2 absorption cross sections:
!           from 116 nm to 245 nm, including schumann-runge continumm
!           from brasseur and solomon 1986.
!-----------------------------------------------------------------------------
master_only : &
      if( masternode ) then
         ncid = open_netcdf_file( photo_xs_o2_flsp%nl_filename, &
                                  photo_xs_o2_flsp%local_path, &
                                  photo_xs_o2_flsp%remote_path, masteronly=.true. )
!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'nosr', dimid ), 'o2_xsect_inti: dimension nsrc not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nsrc ), 'o2_xsect_inti: failed to read nsrc' )
         call handle_ncerr( nf_inq_dimid( ncid, 'ngast', dimid ), 'o2_xsect_inti: dimension ngast not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, ngast ), 'o2_xsect_inti: failed to read ngast' )
         call handle_ncerr( nf_inq_dimid( ncid, 'nla', dimid ), 'o2_xsect_inti: dimension nla not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nla ), 'o2_xsect_inti: failed to read nla' )
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
         allocate( wlint(nsrc), xso2int(nsrc), x1(nsrc), y1(nsrc), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'o2_xsect_inti: failed to allocate wlint ... y1; error = ',astat
            call endrun
         end if
         allocate( wlgast(ngast), wlla(nla), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'o2_xsect_inti: failed to allocate wlgast, wlla; error = ',astat
            call endrun
         end if
!---------------------------------------------------------------------------
! 	... read the wave bin coordinates
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'wl_src', vid ), 'o2_xsect_inti: wl_src not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, x1 ), 'o2_xsect_inti: getting wl_src' )
         call handle_ncerr( nf_inq_varid( ncid, 'xs_src', vid ), 'o2_xsect_inti: xs_src not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, y1 ), 'o2_xsect_inti: getting xs_src' )
         call handle_ncerr( nf_inq_varid( ncid, 'wl_gast', vid ), 'o2_xsect_inti: wl_gast not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, wlgast ), 'o2_xsect_inti: getting wl_gast' )
         call handle_ncerr( nf_inq_varid( ncid, 'wl_lym', vid ), 'o2_xsect_inti: wl_lym not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, wlla ), 'o2_xsect_inti: getting wl_lym' )
         call handle_ncerr( nf_close( ncid ), 'o2_xsect_inti: closing ' // trim(photo_xs_o2_flsp%nl_filename) )
!-----------------------------------------------------------------------------
! 	... put together the internal grid by "pasting" the lyman-alpha grid and 
!           kockarts grid into the combination of brasseur/solomon and jpl grid
!-----------------------------------------------------------------------------
         wlint(1:9) = x1(1:9)
         nwint = 9
         wlint(nwint+1:nwint+2) = wlla(1:2)
         nwint = 11
         wlint(nwint+1:nwint+36) = x1(12:47)
         nwint = 47
         wlint(nwint+1:nwint+ngast) = wlgast(1:ngast)
         nwint = nwint + ngast
         wlint(nwint+1:nwint+41) = x1(65:105)
         nwint = nwint + 41
         wrk_ind(:) = (/ nsrc, ngast, nla, nwint /)
!-----------------------------------------------------------------------------
! 	... initialize interpolation module
!-----------------------------------------------------------------------------
         call inter_inti( nw+1, wl, nsrc, wlint )
!-----------------------------------------------------------------------------
! 	... interpolate brasseur/solomon and jpl data onto internal grid
!-----------------------------------------------------------------------------
         call inter2( nsrc, wlint, xso2int, nsrc, x1, y1, iret )
         deallocate( x1, y1 )
      end if master_only
#ifdef USE_MPI
!-----------------------------------------------------------------------------
! 	... bcast dimensions
!-----------------------------------------------------------------------------
      call mpi_bcast( wrk_ind, 4, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'o2_xsect_inti: failed to bcast dimensions; error = ',iret
         call endrun
      end if
      if( .not. masternode ) then
         nsrc  = wrk_ind(1)
         ngast = wrk_ind(2)
         nla   = wrk_ind(3)
         nwint = wrk_ind(4)
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
         allocate( wlint(nsrc), xso2int(nsrc), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'o2_xsect_inti: failed to allocate wlint .. y1; error = ',astat
            call endrun
         end if
         allocate( wlgast(ngast), wlla(nla), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'o2_xsect_inti: failed to allocate wlgast, wlla; error = ',astat
            call endrun
         end if
      end if
!-----------------------------------------------------------------------------
! 	... bcast arrays
!-----------------------------------------------------------------------------
      call mpi_bcast( wlint, nsrc, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'o2_xsect_inti: failed to bcast wlint; error = ',iret
         call endrun
      end if
      call mpi_bcast( xso2int, nsrc, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'o2_xsect_inti: failed to bcast xso2int; error = ',iret
         call endrun
      end if
      call mpi_bcast( wlgast, ngast, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'o2_xsect_inti: failed to bcast wlgast; error = ',iret
         call endrun
      end if
      call mpi_bcast( wlla, nla, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'o2_xsect_inti: failed to bcast wlla; error = ',iret
         call endrun
      end if
      if( .not. masternode ) then
!-----------------------------------------------------------------------------
! 	... initialize interpolation module
!-----------------------------------------------------------------------------
         call inter_inti( nw+1, wl, nsrc, wlint )
      end if
#endif

      end subroutine o2_xsect_inti

      subroutine set_o2_xsect( lat, z, nw, wl, cz, &
                               vcol, scol, dto2, xso2 )
!-----------------------------------------------------------------------------
!   purpose:
!   compute equivalent optical depths for o2 absorption, parameterized in
!   the sr bands and the lyman-alpha line.
!-----------------------------------------------------------------------------
!   parameters:
!   nz      - integer, number of specified altitude levels in the working (i)
!             grid
!   z       - real, specified altitude working grid (km)
!   nw      - integer, number of specified intervals + 1 in working
!             wavelength grid
!   wl      - real, vector of lower limits of wavelength intervals in
!             working wavelength grid
!   cz      - real, number of air molecules per cm^2 at each specified
!             altitude layer
!   zen     - real, solar zenith angle
!   dto2    - real, optical depth due to o2 absorption at each specified  (o)
!             vertical layer at each specified wavelength
!   xso2    - real, molecular absorption cross section in sr bands at     (o)
!             each specified altitude and wavelength.  includes herzberg
!             continuum.
!-----------------------------------------------------------------------------
!   edit history:
!   02/98  included lyman-alpha parameterization
!   03/97  fix dto2 problem at top level (nz)
!   02/97  changed offset for grid-end interpolation to relative number
!          (x * (1 +- deltax))
!   08/96  modified for early exit, no redundant read of data and smaller
!          internal grid if possible;  internal grid uses user grid points
!          whenever possible
!   07/96  modified to work on internal grid and interpolate final values
!          onto the user-defined grid
!-----------------------------------------------------------------------------

      use mo_params,  only : kw
      use mo_wavelen, only : delw_bin
      use mo_inter,   only : inter3
      use mo_schu,    only : schu
      use mo_lymana,  only : lymana
      use mo_grid,    only : plev, plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: lat
      integer, intent(in) :: nw
      real, intent(in)    :: wl(kw)
      real, intent(in)    :: cz(plevp)
      real, intent(in)    :: z(plevp)
      real, intent(in)    :: vcol(plevp)
      real, intent(in)    :: scol(plevp)
      real, intent(out)   :: dto2(plev,nw)
      real, intent(out)   :: xso2(nw,plevp)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: wn, k, igast
      integer :: astat
      real    :: secchi(plevp)

!-----------------------------------------------------------------------------
! 	... o2 optical depth and equivalent cross section on kockarts grid
!-----------------------------------------------------------------------------
      real, allocatable :: dto2k(:,:)
      real, allocatable :: xso2k(:,:)
!-----------------------------------------------------------------------------
! 	... o2 optical depth and equivalent cross section in the lyman-alpha region
!-----------------------------------------------------------------------------
      real, allocatable :: dto2la(:,:)
      real, allocatable :: xso2la(:,:)
!-----------------------------------------------------------------------------
! 	... temporary one-dimensional storage for optical depth and cross section values
!           xxtmp  - on internal grid
!           xxuser - on user defined grid
!-----------------------------------------------------------------------------
      real, dimension(2*kw) :: dttmp, xstmp
      real, dimension(kw)   :: dtuser, xsuser
      real :: o2col(plevp)

      real :: x, y
      real :: delo2

!-----------------------------------------------------------------------------
!	... allocate local variables
!-----------------------------------------------------------------------------
      allocate( dto2k(plev,ngast-1), xso2k(plevp,ngast-1), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'set_o2_xsect: failed to allocate dto2k,xso2k; error = ',astat
         call endrun
      end if
      allocate( dto2la(plev,nla-1), xso2la(plevp,nla-1), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'set_o2_xsect: failed to allocate dto2k,xso2k; error = ',astat
         call endrun
      end if
!-----------------------------------------------------------------------------
! 	... check, whether user grid is in the o2 absorption band at all...
!           if not, set cross section and optical depth values to zero and return
!-----------------------------------------------------------------------------
      dto2(:plev,:nw)  = 0.
      xso2(:nw,:plevp) = 0.
      if( wl(1) > 243. ) then
         return
      end if

!-----------------------------------------------------------------------------
! 	... sec xhi or chapman calculation
!           for zen > 95 degrees, use zen = 95.  (this is only to compute effective o2
!           cross sections. still, better than setting dto2 = 0. as was done up to 
!           version 4.0) sm 1/2000
!           in future could replace with mu2(iz) (but mu2 is also wavelength-depenedent)
!           or imporved chapman function 
!-----------------------------------------------------------------------------

!-----------------------------------------------------------------------------
! 	... slant o2 column 
!-----------------------------------------------------------------------------
      o2col(1:plevp) = 0.2095 * scol(1:plevp)

!-----------------------------------------------------------------------------
! 	... effective secant of solar zenith angle.  use 2.0 if no direct sun. 
!           for nz, use value at nz-1
!-----------------------------------------------------------------------------
      secchi(1:plev) = scol(1:plev)/vcol(1:plev)
      where( secchi(1:plev) == 0. )
         secchi(1:plev) = 2.
      endwhere
      secchi(plevp) = secchi(plev)

!-----------------------------------------------------------------------------
! 	... if necessary:
!           kockarts parameterization of the sr bands, output values of o2
!           optical depth and o2 equivalent cross section are on his grid
!-----------------------------------------------------------------------------
      if( wl(1) < wlgast(ngast) .and. wl(nw+1) > wlgast(1) ) then
           call schu( o2col, secchi, dto2k, xso2k )
      else
         dto2k(:,:) = 0.
         xso2k(:,:) = 0.
      end if

!-----------------------------------------------------------------------------
! 	... lyman-alpha parameterization, output values of o2 opticaldepth
!           and o2 effective (equivalent) cross section
!-----------------------------------------------------------------------------
      if( wl(1) <= wlla(nla) .and. wl(nw+1) >= wlla(1) ) then
         call lymana( o2col, secchi, dto2la, xso2la )
      else
         dto2la(:,:) = 0.
         xso2la(:,:) = 0.
      end if

!-----------------------------------------------------------------------------
! 	... loop through the altitude levels
!-----------------------------------------------------------------------------
lev_loop1 : &
      do k = 1,plevp
         igast = 0
!-----------------------------------------------------------------------------
! 	... loop through the internal wavelength grid
!-----------------------------------------------------------------------------
wav_loop1 : &
         do wn = 1,nwint-1
!-----------------------------------------------------------------------------
! 	... if outside kockarts grid and outside lyman-alpha, use the 
!           jpl/brasseur+solomon data, if inside
!           kockarts grid, use the parameterized values from the call to schu,
!           if inside lyman-alpha, use the paraemterized values from call to lymana
!-----------------------------------------------------------------------------
            if( wlint(wn+1) <= wlgast(1) .or. wlint(wn) >= wlgast(ngast) ) then
              if( wlint(wn+1) <= wlla(1) .or. wlint(wn) >= wlla(nla) ) then
                 xstmp(wn) = xso2int(wn)
              else
                 xstmp(wn) = xso2la(k,1)
              end if
            else
               igast = igast + 1
               xstmp(wn) = xso2k(k,igast)
            end if
!-----------------------------------------------------------------------------
! 	... compute the area in each bin (for correct interpolation purposes only!)
!-----------------------------------------------------------------------------
            xstmp(wn) = xstmp(wn) * (wlint(wn+1) - wlint(wn))
         end do wav_loop1
!-----------------------------------------------------------------------------
! 	... interpolate o2 cross section from the internal grid onto the user grid
!-----------------------------------------------------------------------------
         call inter3( nw+1, wl, xsuser, nwint, wlint, xstmp )
         xso2(:nw,k) = xsuser(:nw) * delw_bin(:nw)
      end do lev_loop1

lev_loop2 : &
      do k = 1,plev
         igast = 0
         delo2 = .2095 * cz(k)    ! vertical o2 column
!-----------------------------------------------------------------------------
! 	... loop through the internal wavelength grid
!-----------------------------------------------------------------------------
wav_loop2 : &
         do wn = 1,nwint-1
!-----------------------------------------------------------------------------
! 	... if outside kockarts grid and outside lyman-alpha, use the 
!           jpl/brasseur+solomon data, if inside
!           kockarts grid, use the parameterized values from the call to schu,
!           if inside lyman-alpha, use the paraemterized values from call to lymana
!-----------------------------------------------------------------------------
            if( wlint(wn+1) <= wlgast(1) .or. wlint(wn) >= wlgast(ngast) ) then
              if( wlint(wn+1) <= wlla(1) .or. wlint(wn) >= wlla(nla) ) then
                 dttmp(wn) = xso2int(wn) * delo2
              else
                 dttmp(wn) = dto2la(k,1)
              end if
            else
               igast = igast + 1
               dttmp(wn) = dto2k(k,igast)
            end if
!-----------------------------------------------------------------------------
! 	... compute the area in each bin (for correct interpolation purposes only!)
!-----------------------------------------------------------------------------
            dttmp(wn) = dttmp(wn) * (wlint(wn+1) - wlint(wn))
         end do wav_loop2
!-----------------------------------------------------------------------------
! 	... interpolate o2 optical depth from the internal grid onto the user grid
!-----------------------------------------------------------------------------
         call inter3( nw+1, wl, dtuser, nwint, wlint, dttmp )
         dto2(k,:nw) = dtuser(:nw) * delw_bin(:nw)
      end do lev_loop2

      deallocate( dto2k, xso2k, dto2la, xso2la )

      end subroutine set_o2_xsect

      end module mo_seto2
