module mo_setsoa

  use mo_constants, only : avogadro

  private
  public :: soa_inti, setsoa, has_soa

  save

  integer, target  :: spc_ndx(9)
  integer, pointer :: soa_ndx, oc1_ndx, oc2_ndx
  integer, pointer :: c10h16_ndx, o3_ndx, oh_ndx
  integer, pointer :: no3_ndx, bigalk_ndx, toluene_ndx
  integer :: rxn_soa(6)
  integer :: react_ndx(6,2)
  real    :: alpha(6,2)                            ! mass-based stoichiometric coefficients
  real    :: k_om(6,2)                             ! equilibrium gas-particule partition
  real    :: bulk_yield(6)                         ! total yield of condensable compound (ug/m3/ppm)
  real    :: fraction(6)                           ! fraction of VOC used in reaction
  logical :: has_soa = .true.

contains

subroutine soa_inti

  use mo_chem_utls, only : get_spc_ndx, get_rxt_ndx

  implicit none

  soa_ndx     => spc_ndx(1)
  oc1_ndx     => spc_ndx(2)
  oc2_ndx     => spc_ndx(3)
  c10h16_ndx  => spc_ndx(4)
  o3_ndx      => spc_ndx(5)
  oh_ndx      => spc_ndx(6)
  no3_ndx     => spc_ndx(7)
  bigalk_ndx  => spc_ndx(8)
  toluene_ndx => spc_ndx(9)
!-----------------------------------------------------------------------      
! 	... set species index
!-----------------------------------------------------------------------      
  soa_ndx      = get_spc_ndx( 'SOA' )
  oc1_ndx      = get_spc_ndx( 'OC1' )
  oc2_ndx      = get_spc_ndx( 'OC2' )
  c10h16_ndx   = get_spc_ndx( 'C10H16')
  o3_ndx       = get_spc_ndx( 'OX' )
  if( o3_ndx < 1 ) then
     o3_ndx =  get_spc_ndx( 'O3' )
  end if
  oh_ndx       = get_spc_ndx( 'OH' )
  no3_ndx      = get_spc_ndx( 'NO3' )
  bigalk_ndx   = get_spc_ndx( 'BIGALK' )
  toluene_ndx  = get_spc_ndx( 'TOLUENE' )
  
  has_soa = all( spc_ndx(1:9) > 0 )
!-----------------------------------------------------------------------      
! 	... check if this is an aerosol simulation
!-----------------------------------------------------------------------      
  if( .not. has_soa ) then 
    return
  end if

!-----------------------------------------------------------------------      
! 	... set reaction indicies
!-----------------------------------------------------------------------      
  rxn_soa(1) = get_rxt_ndx( 'soa1' )
  rxn_soa(2) = get_rxt_ndx( 'soa2' )
  rxn_soa(3) = get_rxt_ndx( 'soa3' )
  rxn_soa(4) = get_rxt_ndx( 'soa4' )
  rxn_soa(5) = get_rxt_ndx( 'soa4' )
  rxn_soa(6) = get_rxt_ndx( 'soa5' )
  if( all( rxn_soa(:) < 1 ) ) then
     has_soa = .false.
     return
  else
    write(*,*) '-----------------------------------------'
    write(*,*) 'mozart will do soa aerosols'
    write(*,*) '-----------------------------------------'
  end if

!-----------------------------------------------------------------------      
! 	... set reactant indicies
!-----------------------------------------------------------------------      
  react_ndx(1,1) = c10h16_ndx
  react_ndx(1,2) = o3_ndx
  react_ndx(2,1) = c10h16_ndx
  react_ndx(2,2) = oh_ndx
  react_ndx(3,1) = c10h16_ndx
  react_ndx(3,2) = no3_ndx
  react_ndx(4,1) = toluene_ndx
  react_ndx(4,2) = oh_ndx
  react_ndx(5,1) = toluene_ndx
  react_ndx(5,2) = oh_ndx
  react_ndx(6,1) = bigalk_ndx
  react_ndx(6,2) = oh_ndx

  print *,'soa_inti ',c10h16_ndx, o3_ndx, oh_ndx, no3_ndx, bigalk_ndx, toluene_ndx
  print *,'soa_inti ',soa_ndx, oc1_ndx, oc2_ndx
  print *,'soa_inti ',react_ndx


!-----------------------------------------------------------------------      
! 	... define partitioning coefficients for each reaction
!           bulk yields are from Seinfeld and Pandis (1998)
!
!           c10h16 + o3 (from Chung and Seinfeld, JGR, 107, 2002)
!-----------------------------------------------------------------------      

  alpha(1,1)    = 0.067
  alpha(1,2)    = 0.354 
  k_om (1,1)    = 0.184
  k_om (1,2)    = 0.0043
  fraction(1)   = 1.
  bulk_yield(1) = 762.

!-----------------------------------------------------------------------      
! 	... c10h16 + oh (from Chung and Seinfeld, JGR, 107, 2002)
!-----------------------------------------------------------------------      
  alpha(2,1)    = 0.067
  alpha(2,2)    = 0.354 
  k_om (2,1)    = 0.184
  k_om (2,2)    = 0.0043
  fraction(2)   = 1.
  bulk_yield(2) = 762.

!-----------------------------------------------------------------------      
! 	... c10h16 + no3 (from Chung and Seinfeld, JGR, 107, 2002)
!-----------------------------------------------------------------------      
  alpha(3,1)    = 1.000
  alpha(3,2)    = 0.000
  k_om (3,1)    = 0.0163
  k_om (3,2)    = 0.0000 
  fraction(3)   = 1.
  bulk_yield(3) = 762.

!-----------------------------------------------------------------------      
! 	... toluene + oh : toluene (from Odum et al., Environ. Sci. Technol., 1892, 1997)
!-----------------------------------------------------------------------      
  alpha(4,1)    = 0.038
  alpha(4,2)    = 0.167
  k_om (4,1)    = 0.042 
  k_om (4,2)    = 0.0014
  fraction(4)   = 0.7
  bulk_yield(4) = 424.

!-----------------------------------------------------------------------      
! 	... toluene + oh : m-xylene (from Cocker et al., Atmos. Environ., 6079, 2001)
!-----------------------------------------------------------------------      
  alpha(5,1)    = 0.120
  alpha(5,2)    = 0.019
  k_om (5,1)    = 0.060 
  k_om (5,2)    = 0.010 
  fraction(5)   = 0.2
  bulk_yield(5) = 419.

!-----------------------------------------------------------------------      
! 	... bigalk + oh : only for alkanes >= heptane (assume low-yield aromatics as in Lack et al.)
!           (from Odum et al., Environ. Sci. Technol., 1892, 1997)
!-----------------------------------------------------------------------      
  alpha(6,1)    = 0.071
  alpha(6,2)    = 0.138
  k_om (6,1)    = 0.053
  k_om (6,2)    = 0.0019
  fraction(6)   = 0.1
  bulk_yield(6) = 200.

end subroutine soa_inti

subroutine setsoa( dt, reaction_rates, tfld, vmr, xhnm, &
                   lat, ip, plonl )
!-----------------------------------------------------------------------      
! secondary organic aerosol for mozart v2.5
!
! based on Lack et al., JGR, 109, D03203, 2004
!
! rewritten by Jean-Francois Lamarque for updated chemical
! mechanism (March 2004)
!-----------------------------------------------------------------------      

  use mo_grid,      only : plev, pcnstm1
  use chem_mods,    only : adv_mass, rxntot
  use mo_histout,   only : outfld, hst_file_max

  implicit none

!-----------------------------------------------------------------------
!      ... dummy arguments
!-----------------------------------------------------------------------
  integer, intent(in)  :: lat                               ! latitude index  
  integer, intent(in)  :: ip                                ! longitude tile index 
  integer, intent(in)  :: plonl                             ! longitude tile dimension 
  real, intent(in)     :: dt                                ! time step
  real, intent(in)     :: reaction_rates(plonl,plev,rxntot) ! reaction rates
  real, intent(in)     :: tfld(plonl,plev)                  ! temperature (K)
  real, intent(in)     :: xhnm(plonl,plev)                  ! total atms density (mol/cm**3)
  real, intent(inout)  :: vmr(plonl,plev,pcnstm1)           ! xported species ( vmr )

!-----------------------------------------------------------------------
!      ... local variables
!-----------------------------------------------------------------------
  integer :: i, k, n
  real    :: m_0
  real    :: mw_soa, yield, prod, soa_mass

!-----------------------------------------------------------------------
! 	... set molecular weight of SOA
!-----------------------------------------------------------------------
  mw_soa = adv_mass(soa_ndx)

level_loop : &
  do k = 1,plev
long_loop : &
    do i = 1,plonl
!-----------------------------------------------------------------------
! 	... calculate initial mass of organic aerosols from OC1 and OC2 
!           and convert to ug/m3
!-----------------------------------------------------------------------
      m_0 = (vmr(i,k,oc1_ndx) + vmr(i,k,oc2_ndx)) * xhnm(i,k) * adv_mass(oc1_ndx)/avogadro * 1.e12
!-----------------------------------------------------------------------
! 	... switch based on a minimum value of m_0.  The bulk approach is
!           used to initiate the process
!-----------------------------------------------------------------------
      if( m_0 <= 0.2 ) then
!-----------------------------------------------------------------------
! 	... bulk theory
!-----------------------------------------------------------------------
        soa_mass = 0.
        do n = 1,6
          yield = bulk_yield(n)
!-----------------------------------------------------------------------
! 	... define chemical production from gas-phase chemistry
!-----------------------------------------------------------------------
          if( rxn_soa(n) > 0 ) then
             prod  = reaction_rates(i,k,rxn_soa(n)) * fraction(n) &
                     * vmr(i,k,react_ndx(n,1)) * vmr(i,k,react_ndx(n,2)) * dt
          else
             prod  = 0.
          end if
!-----------------------------------------------------------------------
! 	... convert from mixing ratio to ppm
!-----------------------------------------------------------------------
          prod = 1.e6 * prod
!-----------------------------------------------------------------------
! 	... collect into total SOA mass
!-----------------------------------------------------------------------
          soa_mass = soa_mass + yield * prod
        end do
      else
!-----------------------------------------------------------------------
! 	... partitioning theory
!-----------------------------------------------------------------------
        soa_mass = 0.
        do n = 1,6
!-----------------------------------------------------------------------
! 	... define yield from available m_0
!-----------------------------------------------------------------------
          yield = soa_yield( m_0, alpha(n,1:2), k_om(n,1:2) )
!-----------------------------------------------------------------------
! 	... define chemical production from gas-phase chemistry
!-----------------------------------------------------------------------
          if( rxn_soa(n) > 0 ) then
             prod  = reaction_rates(i,k,rxn_soa(n)) * fraction(n) &
                     * vmr(i,k,react_ndx(n,1)) * vmr(i,k,react_ndx(n,2)) * dt
          else
             prod  = 0.
          end if
!-----------------------------------------------------------------------
! 	... convert from mixing ratio to mass (ug/m3)       
!-----------------------------------------------------------------------
          prod = prod * xhnm(i,k) * mw_soa/avogadro * 1.e12
!-----------------------------------------------------------------------
! 	... collect into total SOA mass
!-----------------------------------------------------------------------
          soa_mass = soa_mass + yield * prod
        end do
      end if

!-----------------------------------------------------------------------
! 	... convert from ug/m3 to mixing ratio and update vmr
!-----------------------------------------------------------------------
      vmr(i,k,soa_ndx) = vmr(i,k,soa_ndx) + soa_mass * 1.e-12 * avogadro/(mw_soa*xhnm(i,k))
      if( vmr(i,k,soa_ndx) > 1.e10 ) then
        write(*,*) 'setsoa: i,k,lat,soa_mass,m_0 = ',i,k,lat,soa_mass,m_0
        call endrun
      end if
    end do long_loop
  end do level_loop
  
end subroutine setsoa

real function soa_yield( m_0, alpha, k)

  implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
  real, intent(in)  :: m_0
  real, intent(in)  :: alpha(2)
  real, intent(in)  :: k(2)

  soa_yield = m_0 *((alpha(1)*k(1)/(1. + k(1)*m_0)) + (alpha(2)*k(2)/(1. + k(2)*m_0)))

end function soa_yield

end module mo_setsoa
