
      module mo_setsox

      private
      public :: sox_inti, setsox
      public :: has_sox

      save

      integer, target    ::  spc_ids(8)
      integer, pointer   ::  id_so2, id_so4, id_nh3, id_hno3, id_h2o2, id_ox, id_nh4no3, id_ho2
      logical            ::  has_sox = .true.

      contains

      subroutine sox_inti
!-----------------------------------------------------------------------      
!	... initialize the hetero sox routine
!-----------------------------------------------------------------------      

      use mo_chem_utls, only : get_spc_ndx
      
      implicit none

      id_so2    => spc_ids(1)
      id_so4    => spc_ids(2)
      id_nh3    => spc_ids(3)
      id_hno3   => spc_ids(4)
      id_h2o2   => spc_ids(5)
      id_ox     => spc_ids(6)
      id_nh4no3 => spc_ids(7)
      id_ho2    => spc_ids(8)
!-----------------------------------------------------------------
!       ... get species indicies
!-----------------------------------------------------------------
      id_so2    = get_spc_ndx( 'SO2' )
      id_so4    = get_spc_ndx( 'SO4' )
      id_nh3    = get_spc_ndx( 'NH3' )
      id_hno3   = get_spc_ndx( 'HNO3' )
      id_h2o2   = get_spc_ndx( 'H2O2' )
      id_ox     = get_spc_ndx( 'OX' )
      if( id_ox < 1 ) then
         id_ox  = get_spc_ndx( 'O3' )
      end if
      id_nh4no3 = get_spc_ndx( 'NH4NO3' )
      id_ho2    = get_spc_ndx( 'HO2' )

      has_sox = all( spc_ids(1:8) > 0 )

      if( has_sox ) then
         write(*,*) '-----------------------------------------'
         write(*,*) 'mozart will do sox aerosols'
         write(*,*) '-----------------------------------------'
      end if

      end subroutine sox_inti

      subroutine setsox( press, lat, dtime, ip, tfld, &
                         qfld, rain, evapr, cmfdqr, lwc, &
                         xhnm, vmr, plonl )
!-----------------------------------------------------------------------      
!          ... heterogeneous SOX
!
!       First ...
!           (a) HENRYs law constants
!           (b) PARTIONING
!           (c) PH values
!
!       then ...
!           (a) HENRYs law constants
!           (b) PARTIONING
!           (c) REACTION rates
!           (d) PREDICTION
!-----------------------------------------------------------------------      

      use mo_grid,      only : plev, pcnstm1
      use mo_chem_utls, only : get_spc_ndx
      use mo_histout,   only : outfld, hst_file_max
      use mo_constants, only : avogadro
    
      implicit none

!-----------------------------------------------------------------------      
!      ... dummy arguments
!-----------------------------------------------------------------------      
      integer, intent(in)  ::  lat                       ! latitude index
      integer, intent(in)  ::  ip                        ! longitude tile index
      integer, intent(in)  ::  plonl                     ! longitude tile dimension
      real, intent(in)     ::  dtime                     ! time step (sec)
      real, intent(in)     ::  xhnm(plonl,plev)          ! total atms density (/cm**3)
      real, intent(in)     ::  tfld(plonl,plev), &       ! temperature (K)
                               qfld(plonl,plev), &       ! specific humidity( kg/kg )
                               lwc(plonl,plev),  &       ! cloud liquid water content (kg/kg)
                               cmfdqr(plonl,plev), &     ! dq/dt for convection
                               rain(plonl,plev), &       ! stratoform precip
                               evapr(plonl,plev), &      ! evaporation
                               press(plonl,plev)         ! midpoint pressure (Pa)
      real, intent(inout)  ::  vmr(plonl,plev,pcnstm1)   ! species concentrations (mol/mol)

!-----------------------------------------------------------------------      
!      ... local variables
!
!           xhno3 ... in mixing ratio
!-----------------------------------------------------------------------      
      integer, parameter :: itermax = 20
      real, parameter ::  ph0    = 5.0            ! initial ph value
      real, parameter ::  const0 = 1.e3/avogadro
      real, parameter ::  t0     = 298.
      real, parameter ::  tmelt  = 273.
      real, parameter ::  pa2mb  = .01
      real, parameter ::  xa0 = 11.,   &
                          xb0 = -.1,   &
                          xa1 = 1.053, &
                          xb1 = -4.368,&
                          xa2 = 1.016, &
                          xb2 = -2.54, &
                          xa3 = .816e-32, &
                          xb3 = .259

      real, parameter ::  kh0 = 9.e3, &           ! HO2(g)          -> Ho2(a)
                          kh1 = 2.05e-5, &        ! HO2(a)          -> H+ + O2-
                          kh2 = 8.6e5,   &        ! HO2(a) + ho2(a) -> h2o2(a) + o2
                          kh3 = 1.e8,    &        ! HO2(a) + o2-    -> h2o2(a) + o2
                          Ra = 8314./101325., &   ! universal constant   (atm)/(M-K)
                          xkw = 1.e-14            ! water acidity
      real, parameter :: small_value = 1.e-20

      integer    ::      k, i, iter, file
      real       ::      wrk, delta
      real       ::      xph0, aden, xk, xe, x2
      real       ::      tz, xl, px, qz, pz, es, qs, patm
      real       ::      Eso2, Eso4, Ehno3, Eco2, Eh2o, Enh3
      real       ::      hno3g, nh3g, so2g, h2o2g, co2g, o3g
      real       ::      hno3a, nh3a, so2a, h2o2a, co2a, o3a
      real       ::      rah2o2, rao3, pso4, ccc
      real       ::      xx0, yy1, xkp
      real       ::      cnh3, chno3, com, com1, com2, xra
      real       ::      RH

!-----------------------------------------------------------------------      
!            for Ho2(g) -> H2o2(a) formation 
!            schwartz JGR, 1984, 11589
!-----------------------------------------------------------------------      
      real       ::      kh4    ! kh2+kh3
      real       ::      xam    ! air density /cm3
      real       ::      ho2s   ! ho2s = ho2(a)+o2-
      real       ::      r1h2o2 ! prod(h2o2) by ho2 in mole/L(w)/s
      real       ::      r2h2o2 ! prod(h2o2) by ho2 in mix/s

      real, dimension(plonl,plev)  ::             &
                         xhno3, xh2o2, xso2, xso4,&
                         xnh3, xnh4, xo3,         &
                         xlwc, cfact, xrain,      &
                         xph, xant, xho2,         &
                         hehno3, &            ! henry law const for hno3
                         heh2o2, &            ! henry law const for h2o2
                         heso2,  &            ! henry law const for so2
                         henh3,  &            ! henry law const for nh3
                         heo3,   &            ! henry law const for nh3
                         precip
      real, dimension(plonl)  :: work1
      logical :: converged

!==================================================================
!       ... First set the PH
!==================================================================
!      ... Initial values
!           The values of so2, so4 are after (1) SLT, and CHEM
!-----------------------------------------------------------------
      xph0 = 10.**(-ph0)                      ! initial PH value
      do k = 1,plev
         precip(:,k) = cmfdqr(:,k) + rain(:,k) - evapr(:,k)
      end do

      do k = 1,plev
          cfact(:,k) = xhnm(:,k)            &             ! /cm3(a)  
                            * 1.e6          &             ! /m3(a)
                            * 1.38e-23/287. &             ! Kg(a)/m3(a)
                            * 1.e-3                       ! Kg(a)/L(a)
      end do

      do k = 1,plev
         xph(:,k)   = xph0                              ! initial PH value
         xlwc(:,k)  = lwc(:,k) *cfact(:,k)              ! cloud water  L(water)/L(air)
         xrain(:,k) = rain(:,k) *cfact(:,k)             ! rain  water  L(water)/L(air)
         xhno3(:,k) = vmr(:,k,id_hno3)                  ! mixing ratio
         xh2o2(:,k) = vmr(:,k,id_h2o2)                  ! mixing ratio
         xso2 (:,k) = vmr(:,k,id_so2)                   ! mixing ratio
         xso4 (:,k) = vmr(:,k,id_so4)                   ! mixing ratio
         xnh3 (:,k) = vmr(:,k,id_nh3)                   ! mixing ratio
         xant (:,k) = vmr(:,k,id_nh4no3)                ! mixing ratio
         xo3  (:,k) = vmr(:,k,id_ox)                    ! mixing ratio
         xho2 (:,k) = vmr(:,k,id_ho2)                   ! mixing ratio
      end do 

!-----------------------------------------------------------------
!       ... Temperature dependent Henry constants
!-----------------------------------------------------------------
level_loop : &
      do k = 1,plev                                             !! plev loop for STEP 0
long_loop : &
         do i = 1,plonl
            xl = xlwc(i,k) 
            if( xl >= 1.e-8 ) then
               work1(i) = (t0 - tfld(i,k))/(tfld(i,k)*t0)
!-----------------------------------------------------------------------      
!        ... hno3
!-----------------------------------------------------------------------      
iter_loop :    do iter = 1,itermax
                  xk = 2.1e5 *exp( 8700.*work1(i) )
                  xe = 15.4
                  hehno3(i,k)  = xk*(1. + xe/xph(i,k))
!-----------------------------------------------------------------------      
!         ... h2o2
!-----------------------------------------------------------------------      
                  xk = 7.4e4   *exp( 6621.*work1(i) )
                  xe = 2.2e-12 *exp(-3730.*work1(i) )
                  heh2o2(i,k)  = xk*(1. + xe/xph(i,k))
!-----------------------------------------------------------------------      
!          ... so2
!-----------------------------------------------------------------------      
                  xk = 1.23  *exp( 3120.*work1(i) )
                  xe = 1.7e-2*exp( 2090.*work1(i) )
                  x2 = 6.0e-8*exp( 1120.*work1(i) )
                  wrk = xe/xph(i,k)
                  heso2(i,k)  = xk*(1. + wrk*(1. + x2/xph(i,k)))
!-----------------------------------------------------------------------      
!          ... nh3
!-----------------------------------------------------------------------      
                  xk = 58.   *exp( 4085.*work1(i) )
                  xe = 1.7e-5*exp(-4325.*work1(i) )
                  henh3(i,k)  = xk*(1. + xe*xph(i,k)/xkw)
!-----------------------------------------------------------------
!       ... Partioning and effect of pH 
!-----------------------------------------------------------------
                  pz = pa2mb*press(i,k)       !! pressure in mb
                  tz = tfld(i,k)
                  patm = pz/1013.
                  xam  = press(i,k)/(1.38e-23*tz)  !air density /M3
!-----------------------------------------------------------------
!        ... hno3
!-----------------------------------------------------------------
                  px = hehno3(i,k) * Ra * tz * xl
                  hno3g = xhno3(i,k)/(1. + px)
                  xk = 2.1e5 *exp( 8700.*work1(i) )
                  xe = 15.4
                  Ehno3 = xk*xe*hno3g *patm
!-----------------------------------------------------------------
!          ... so2
!-----------------------------------------------------------------
                  px = heso2(i,k) * Ra * tz * xl
                  so2g =  xso2(i,k)/(1.+ px)
                  xk = 1.23  *exp( 3120.*work1(i) )
                  xe = 1.7e-2*exp( 2090.*work1(i) )
                  Eso2 = xk*xe*so2g *patm
!-----------------------------------------------------------------
!          ... nh3
!-----------------------------------------------------------------
                  px = henh3(i,k) * Ra * tz * xl
                  nh3g = xnh3(i,k)/(1.+ px)
                  xk = 58.   *exp( 4085.*work1(i) )
                  xe = 1.7e-5*exp( -4325.*work1(i) )
                  Enh3 = xk*xe*nh3g/xkw *patm
!-----------------------------------------------------------------
!        ... h2o effects
!-----------------------------------------------------------------
                  Eh2o = xkw
!-----------------------------------------------------------------
!        ... co2 effects
!-----------------------------------------------------------------
                  co2g = 330.e-6                            !330 ppm = 330.e-6 atm
                  xk = 3.1e-2*exp( 2423.*work1(i) )
                  xe = 4.3e-7*exp(-913. *work1(i) )
                  Eco2 = xk*xe*co2g  *patm
!-----------------------------------------------------------------
!        ... PH cal
!-----------------------------------------------------------------
                  com2 = (Eh2o + Ehno3 + Eso2 + Eco2)  &
                       / (1. + Enh3 )
                  com2 = max( com2,1.e-20 )
                  xph(i,k) = SQRT( com2 )
!-----------------------------------------------------------------
!         ... Add so4 effect
!-----------------------------------------------------------------
                  Eso4 = xso4(i,k)*xhnm(i,k)   &         ! /cm3(a)
                        *const0/xl
                  xph(i,k) =  min( 1.e-2,max( 1.e-7,xph(i,k) + 2.*Eso4 ) )
                  if( iter > 1 ) then
                     delta = abs( (xph(i,k) - delta)/delta )
                     converged = delta < .01
                     if( converged ) then
                        exit
                     else if( iter < itermax ) then
                        delta = xph(i,k)
                     else
                        exit
                     end if
                  else
                     delta = xph(i,k)
                  end if
               end do iter_loop
               if( .not. converged ) then
                  write(*,*) 'setsox: pH failed to converge @ (',i,',',k,',',lat,'), % change=', &
                              100.*delta
               end if
            else
               xph(i,k) =  1.e-7
            end if
         end do long_loop
      end do level_loop

      do file = 1,hst_file_max
         call outfld( 'PH', xph,  plonl, ip, lat, file )
      end do

!==============================================================
!          ... Now use the actual PH
!==============================================================
level_loop2 : &
      do k = 1,plev
long_loop2 : &
         do i = 1,plonl
            tz   = tfld(i,k)
            work1(i) = (t0 - tz)/(tz*t0)
            xl   = xlwc(i,k)
            patm = press(i,k)/101300.        ! press is in pascal
            xam  = press(i,k)/(1.38e-23*tz)  ! air density /M3

!-----------------------------------------------------------------
!         ... hno3
!-----------------------------------------------------------------
            xk = 2.1e5 *exp( 8700.*work1(i) )
            xe = 15.4
            hehno3(i,k)  = xk*(1. + xe/xph(i,k))

!-----------------------------------------------------------------
!        ... h2o2
!-----------------------------------------------------------------
            xk = 7.4e4   *exp( 6621.*work1(i) )
            xe = 2.2e-12 *exp(-3730.*work1(i) )
            heh2o2(i,k)  = xk*(1. + xe/xph(i,k))

!-----------------------------------------------------------------
!         ... so2
!-----------------------------------------------------------------
            xk = 1.23  *exp( 3120.*work1(i) )
            xe = 1.7e-2*exp( 2090.*work1(i) )
            x2 = 6.0e-8*exp( 1120.*work1(i) )

            wrk = xe/xph(i,k)
            heso2(i,k)  = xk*(1. + wrk*(1. + x2/xph(i,k)))

!-----------------------------------------------------------------
!          ... nh3
!-----------------------------------------------------------------
            xk = 58.   *exp( 4085.*work1(i) )
            xe = 1.7e-5*exp(-4325.*work1(i) )
            henh3(i,k)  = xk*(1. + xe*xph(i,k)/xkw)

!-----------------------------------------------------------------
!        ... o3
!-----------------------------------------------------------------
            xk = 1.15e-2 *exp( 2560.*work1(i) )
            heo3(i,k) = xk

!------------------------------------------------------------------------
!       ... for Ho2(g) -> H2o2(a) formation 
!           schwartz JGR, 1984, 11589
!------------------------------------------------------------------------
            kh4 = (kh2 + kh3*kh1/xph(i,k)) / ((1. + kh1/xph(i,k))**2)
            ho2s = kh0*xho2(i,k)*patm*(1. + kh1/xph(i,k))  ! ho2s = ho2(a)+o2-
            r1h2o2 = kh4*ho2s*ho2s                         ! prod(h2o2) in mole/L(w)/s
            r2h2o2 = r1h2o2*xlwc(i,k)  &                   ! mole/L(w)/s   * L(w)/fm3(a) = mole/fm3(a)/s
                           *const0     &                   ! mole/fm3(a)/s * 1.e-3       = mole/cm3(a)/s
                           /xam                            ! /cm3(a)/s    / air-den     = mix-ratio/s
            xh2o2(i,k) = xh2o2(i,k) + r2h2o2*dtime         ! updated h2o2 by het production

!-----------------------------------------------
!       ... Partioning 
!-----------------------------------------------
!------------------------------------------------------------------------
!        ... h2o2
!------------------------------------------------------------------------
            px = heh2o2(i,k) * Ra * tz * xl
            h2o2g =  xh2o2(i,k)/(1.+ px)

!------------------------------------------------------------------------
!         ... so2
!------------------------------------------------------------------------
            px = heso2(i,k) * Ra * tz * xl
            so2g =  xso2(i,k)/(1.+ px)

!------------------------------------------------------------------------
!         ... o3 ============
!------------------------------------------------------------------------
            px = heo3(i,k) * Ra * tz * xl
            o3g =  xo3(i,k)/(1.+ px)

!-----------------------------------------------
!       ... Aqueous phase reaction rates
!           SO2 + H2O2 -> SO4
!           SO2 + O3   -> SO4
!-----------------------------------------------
          
!------------------------------------------------------------------------
!       ... S(IV) (HSO3) + H2O2
!------------------------------------------------------------------------
            rah2o2 = 8.e4 * exp( -3650.*work1(i) ) / (.1 + xph(i,k))

!------------------------------------------------------------------------
!        ... S(IV)+ O3
!------------------------------------------------------------------------
            rao3   = 4.39e11 * exp(-4131./tz)  &
                     + 2.56e3  * exp(-996. /tz) /xph(i,k)

!-----------------------------------------------------------------
!       ... Prediction after aqueous phase
!       so4
!       When Cloud is present 
!   
!       S(IV) + H2O2 = S(VI)
!       S(IV) + O3   = S(VI)
!
!       reference:
!           (1) Seinfeld
!           (2) Benkovitz
!
!       S(IV) + H2O2 = S(VI)
!-----------------------------------------------------------------

in_cloud : &
       if( xl >= 1.e-8 ) then                          ! cloud is present
          pso4 = rah2o2 * heh2o2(i,k)*h2o2g  &
                        * heso2(i,k) *so2g             ! [M/s]
          pso4 = pso4       &                          ! [M/s] =  [mole/L(w)/s]
                 * xlwc(i,k)  &                        ! [mole/L(a)/s]
                 / const0     &                        ! [/L(a)/s]
                 / xhnm(i,k)     

          ccc = pso4*dtime
          ccc = max(ccc, 1.e-30)
          if( xh2o2(i,k) > xso2(i,k) ) then
              if( ccc > xso2(i,k) ) then
                  xso4(i,k)  = xso4(i,k) + xso2(i,k)
                  xso2(i,k)  = 1.e-20
                  xh2o2(i,k) = xh2o2(i,k) - xso2(i,k)
              else
                  xso4(i,k)  = xso4(i,k)  + ccc
                  xh2o2(i,k) = xh2o2(i,k) - ccc
                  xso2(i,k)  = xso2(i,k)  - ccc
              end if
          else
               if( ccc > xh2o2(i,k) ) then
                   xso4(i,k)  = xso4(i,k) + xh2o2(i,k)
                   xso2(i,k)  = xso2(i,k) - xh2o2(i,k)
                   xh2o2(i,k) = 1.e-20
               else
                   xso4(i,k)  = xso4(i,k)  + ccc
                   xh2o2(i,k) = xh2o2(i,k) - ccc
                   xso2(i,k)  = xso2(i,k)  - ccc
               end if
          end if

!-----------------------------------------------------------------
!       S(IV) + O3 = S(VI)
!-----------------------------------------------------------------
          pso4 = rao3 * heo3(i,k)*o3g * heso2(i,k)*so2g       ! [M/s]
          pso4 = pso4        &                                ! [M/s] =  [mole/L(w)/s]
                 * xlwc(i,k)   &                              ! [mole/L(a)/s]
                  / const0      &                             ! [/L(a)/s]
                  / xhnm(i,k)                                 ! [mixing ratio/s]

          ccc = pso4*dtime
          ccc = max(ccc, 1.e-30)
          if( ccc > xso2(i,k) ) then
             xso4(i,k) = xso4(i,k) + xso2(i,k)
             xso2(i,k) = 1.e-20
          else
             xso4(i,k) = xso4(i,k)  + ccc
             xso2(i,k) = xso2(i,k)  - ccc
             xso2(i,k) = max(xso2(i,k), 1.e-20)
          end if
       end if in_cloud

!-----------------------------------------------------------------
!       ... Formation of NH4+ + SO4=
!           to balance 1 SO4= should take 2 NH4+
!           According to Dentener and Crutzen (1994) JAC 331
!           the neutralization of sulfuric acid by NH3
!           is (NH4)1.5 H0.5(SO4)
!
!       ... Formation of AMMONIUM NITRID ANT
!           Calculate reaction coefficient NH3(g)+HNO3(g)=NH4(++)NO3(--) 
!                                                   Kp(ppb**2)
!      * Kp is calculated according to
!        Stelson and Seinfeld Atm. Env 16 983, 1982
!        Seinfeld (1986) 
!-----------------------------------------------------------------
            qz = qfld(i,k)             ! H2O mass mxing ratio Kg/Kg
            pz = pa2mb*press(i,k)      ! pressure in mb
 
!-----------------------------------------------------------------
!        ... Calculate RH
!-----------------------------------------------------------------
            wrk = tz - tmelt
            es  = 6.11*10.**(7.63*wrk/(241.9 + wrk))            ! Magnus EQ
            qs  = .622*es/pz                                    ! sat mass mix (H2O)
            RH  = 100.*qz/qs                                    ! relative huminity(%)
            RH  = min( 100.,max( RH,0. ) )
  
            xx0 = xa0 + xb0*RH
            if( RH >= 90. ) then
               yy1 = xa1*exp( xb1/xx0 )
            else
               yy1 = xa2*exp( xb2/xx0 )
            end if            

            xkp = yy1*(xa3*exp( xb3*tz )/.7) &    ! ppb**2
                    * 1.e-18                      ! mixing ratio

            cnh3  = xnh3(i,k)
            chno3 = xhno3(i,k)
            com   = cnh3*chno3
            com1  = (cnh3 + chno3)**2 - 4.*(cnh3*chno3 - xkp)
            com1  = max( com1,1.e-30 )

            if( com >= xkp ) then   ! NH4NO3 is formed
               xra = .5*(cnh3 + chno3 - SQRT(com1))
!-----------------------------------------------------------------
!        ... xra =0.0 for not forming ANT
!-----------------------------------------------------------------
               xra = 0.
               xant(i,k) = max( xant(i,k) + xra, small_value )
               xnh3(i,k) = max( xnh3(i,k) - xra, small_value )
               xhno3(i,k)= max( xhno3(i,k)- xra, small_value )
            end if

!-----------------------------------------------------------------
!      ... Washout SO2, SO4 and NH3
!-----------------------------------------------------------------
            xso4(i,k)  = max( xso4(i,k), small_value )
            xant(i,k)  = max( xant(i,k), small_value )
            xnh3(i,k)  = max( xnh3(i,k), small_value )
            xso2(i,k)  = max( xso2(i,k), small_value )
         end do long_loop2
      end do level_loop2

!==============================================================
!       ... update the mixing ratios
!==============================================================
      do k = 1,plev
         vmr(:,k,id_so2)    = max( xso2(:,k), small_value )
         vmr(:,k,id_so4)    = max( xso4(:,k), small_value )
         vmr(:,k,id_h2o2)   = max( xh2o2(:,k), small_value ) 
         vmr(:,k,id_nh3)    = max( xnh3(:,k), small_value )
         vmr(:,k,id_nh4no3) = max( xant(:,k), small_value )
         vmr(:,k,id_hno3)   = max( xhno3(:,k), small_value )
      end do 
 
      end subroutine setsox

      end module mo_setsox
