
      module mo_soil_no
!---------------------------------------------------------------------
!       ... Interactive calculation of soil NO emissions following
!           Iyenger and Levy (JGR, 11,447-11,464, 1995), combined with estimates
!           of fertilizer input from Lex Bouwman
!
! Jean-Francois Lamarque, March 2004
!---------------------------------------------------------------------

      implicit none

      private
      public :: soil_no_inti
      public :: soil_no_emissions
      public :: soil_no_rdrst
      public :: soil_no_wrrst

      save

      integer              :: no_ndx
      integer              :: npft_veg_soil_no
      integer, allocatable :: is_pulsing(:,:,:)
      integer, allocatable :: growth(:,:,:)
      real,    allocatable :: maps_soil_no_dry(:,:,:,:,:)
      real,    allocatable :: maps_soil_no_wet(:,:,:,:,:)
      real,    allocatable :: maps_soil_no_ftl(:,:,:,:,:)
      real,    allocatable :: soil_wetness(:,:,:)
      real,    allocatable :: time_pulsing(:,:,:)

      contains

      subroutine soil_no_inti( plonl, platl, pplon, nlon_veg, nlat_veg, &
                               npft_veg, lon_veg, lon_veg_edge, lat_veg, lat_veg_edge, &
                               vegetation_map, landmask, monthly_lai_clm, no_index )
!-------------------------------------------------------------------------------------
! 	... intialize interactive soil_no emissions
!-------------------------------------------------------------------------------------

      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_integer, mpi_double_precision, mpi_comm_comp
#endif
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : emis_flsp
      use netcdf

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: nlon_veg
      integer, intent(in) :: nlat_veg
      integer, intent(in) :: npft_veg
      integer, intent(in) :: no_index
      real, intent(in)    :: lon_veg(nlon_veg)
      real, intent(in)    :: lon_veg_edge(nlon_veg+1)
      real, intent(in)    :: lat_veg(nlat_veg)
      real, intent(in)    :: lat_veg_edge(nlat_veg+1)
      real, intent(in)    :: vegetation_map(nlon_veg,nlat_veg,npft_veg)
      real, intent(in)    :: landmask(nlon_veg,nlat_veg)
      real, intent(in)    :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer :: i, j, ii, jj, k, kk, nm1
      integer :: m, n, l, id
      integer :: astat, istat, iret
      integer :: ncid, dimid, varid
      integer :: i1, j1, jmin, cnt
      integer :: nlon_data, nlat_data

      real              :: lai_min, lai_max, delta_lai
      real, allocatable :: lai_za         (:)
      real, allocatable :: no_fertilizer  (:,:,:)

      character(len=80) :: filename
      character(len=80) :: err_msg
      character(len=11) :: varname(3)

      varname(:) = (/ 'NO-grass   ', 'NO-uplcrops', 'NO-wrice   ' /)
      npft_veg_soil_no = npft_veg
      no_ndx           = no_index
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
      allocate( maps_soil_no_dry(plonl,platl,pplon,npft_veg,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate maps_soil_no_dry; error = ',astat
         call endrun
      end if
      allocate( maps_soil_no_wet(plonl,platl,pplon,npft_veg,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate maps_soil_no_wet; error = ',astat
         call endrun
      end if
      allocate( maps_soil_no_ftl(plonl,platl,pplon,5,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate maps_soil_no_ftl; error = ',astat
         call endrun
      end if

!---------------------------------------------------------------------------
! 	... allocate arrays for emission calculation
!---------------------------------------------------------------------------
      allocate( soil_wetness(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate soil_wetness; error = ',astat
         call endrun
      end if 
      allocate( time_pulsing(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate time_pulsing; error = ',astat
         call endrun
      end if 
      allocate( is_pulsing(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate is_pulsing; error = ',astat
         call endrun
      end if 
      allocate( growth(nlat_veg,5,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate growth; error = ',astat
         call endrun
      end if

!---------------------------------------------------------------------------
! 	... initialize arrays
!---------------------------------------------------------------------------
      soil_wetness(:,:,:) = 0.
      time_pulsing(:,:,:) = 0.
      is_pulsing(:,:,:)   = 0

!---------------------------------------------------------------------------
! 	... compute growth of agriculture from LAI
!           similar algorithm used for definition of season
!           in the interactive drydep.  The algorithm
!           is base on discussions with B. Holland (2003)
!---------------------------------------------------------------------------
      allocate( lai_za(12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'soil_no_inti: failed to allocate lai_za; error = ',astat
         call endrun
      end if

      do k = 13,17
        kk = k - 12
        do j = 1,nlat_veg
!---------------------------------------------------------------------------
! 	... calculate zonal average
!---------------------------------------------------------------------------
          do n = 1,12
            lai_za(n) = sum( monthly_lai_clm(:,j,k,n),1 )/nlon_veg
          end do

          lai_max = maxval( lai_za )
          lai_min = minval( lai_za )

          if( lai_min == 0. .and. lai_max == 0. ) then
            growth(j,kk,:) = 0
            cycle
          end if

          do n = 1,12
!---------------------------------------------------------------------------
! 	... define summer
!---------------------------------------------------------------------------
            if( lai_za(n) > .5*lai_max ) then
              growth(j,kk,n) = 1
              cycle
            end if
!---------------------------------------------------------------------------
! 	... define winter
!---------------------------------------------------------------------------
            if( lai_za(n) < (lai_min + .05*(lai_max - lai_min)) ) then
              growth(j,kk,n) = 0
              cycle
            end if

            nm1 = n - 1
            if( nm1 == 0 ) then
               nm1 = 12
            end if
            delta_lai = lai_za(n) - lai_za(nm1)
            if( delta_lai > 0. ) then
              growth(j,kk,n) = 1
            else if( delta_lai < 0. ) then
              growth(j,kk,n) = 0
            else if( delta_lai == 0. ) then
              growth(j,kk,n) = 0
            end if
          end do
        end do
      end do

      deallocate(lai_za)


!---------------------------------------------------------------------------
! 	... read fertilizer input from Lex Bouwman (present-day only)
!---------------------------------------------------------------------------
master_only : &
      if( masternode ) then
!-------------------------------------------------------------------------------------
! 	... open no fertilizer netcdf file
!-------------------------------------------------------------------------------------
         filename = 'no_fertilizer.nc'
         ncid = open_netcdf_file( filename, trim(emis_flsp%local_path), trim(emis_flsp%remote_path), &
                                  masteronly=.true. )
!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), 'soil_no_inti: dimension nlon not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon_data ), 'soil_no_inti: failed to read nlon' )
         call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), 'soil_no_inti: dimension nlat not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat_data ), 'soil_no_inti: failed to read nlat' )
         if( nlon_data /= nlon_veg .or. nlat_data /= nlat_veg ) then
            write(*,*) 'soil_no_inti: model and data dimensions do not match'
            write(*,*) '              data  nlon,nlat = ',nlon_data,nlat_data
            write(*,*) '              model nlon,nlat = ',nlon_veg,nlat_veg
         end if
         allocate( no_fertilizer(nlon_data,nlat_data,3),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'soil_no_inti: failed to allocate no_fertilizer; error = ',astat
            call endrun
         end if
!-------------------------------------------------------------------------------------
! 	... read data
!-------------------------------------------------------------------------------------
var_loop : &
         do n = 1,3
            err_msg = 'soil_no_inti: failed to find ' // varname(n) // ' id'
            call handle_ncerr( nf_inq_varid( ncid, varname(n), varid ), trim(err_msg) )
            err_msg = 'soil_no_inti: failed to read ' // varname(n)
            call handle_ncerr( nf_get_var_double( ncid, varid, no_fertilizer(:,:,n) ), trim(err_msg) )
         end do var_loop
!---------------------------------------------------------------------------
! 	... close netcdf file
!---------------------------------------------------------------------------
         err_msg = 'soil_no_inti: error closing ' // trim(filename)
         call handle_ncerr( nf_close( ncid ), trim(err_msg) )
      end if master_only

#ifdef USE_MPI
!------------------------------------------------------------------------------
!       ... bcast dimensions
!------------------------------------------------------------------------------
      call mpi_bcast( nlon_data, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'soil_no_inti: failed to bcast nlon_data; error = ',iret
         call endrun
      end if
      call mpi_bcast( nlat_data, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'soil_no_inti: failed to bcast nlat_data; error = ',iret
         call endrun
      end if
      if( .not. masternode ) then
         allocate( no_fertilizer(nlon_data,nlat_data,3),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'soil_no_inti: failed to allocate no_fertilizer; error = ',astat
            call endrun
         end if
      end if
!------------------------------------------------------------------------------
!       ... bcast arrays
!------------------------------------------------------------------------------
      call mpi_bcast( no_fertilizer, nlon_data*nlat_data*3 , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'soil_no_inti: failed to bcast no_fertilizer; error = ',iret
         call endrun
      end if
#endif

!---------------------------------------------------------------------------
! 	... scale fertilizer from kg/ha/yr to ng/m2/s
!---------------------------------------------------------------------------
      no_fertilizer(:,:,:) = no_fertilizer(:,:,:) * 3.17

!---------------------------------------------------------------------------
! 	... regrid to model grid
!---------------------------------------------------------------------------
      write(*,*) 'soil_no_inti: calling interp_map_soil_no'
      call interp_map_soil_no( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                               lon_veg, lon_veg_edge, landmask, no_fertilizer, &
                               vegetation_map, monthly_lai_clm, &
                               plonl, platl, pplon )

      deallocate( no_fertilizer, stat=astat )

      end subroutine soil_no_inti

      subroutine interp_map_soil_no( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                                  lon_veg, lon_veg_edge, landmask, no_fertilizer, &
                                  vegetation_map, monthly_lai_clm, &
                                  plonl, platl, pplon )

      use mo_grid, only      : plon, plat
      use mo_constants, only : r2d, phi, lam
      use m_adv, only        : has_npole
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)      :: plonl
      integer, intent(in)      :: platl
      integer, intent(in)      :: pplon
      integer, intent(in)      :: nlon_veg, nlat_veg, npft_veg
      real, intent(in)         :: landmask(nlon_veg,nlat_veg)
      real, intent(in)         :: vegetation_map (nlon_veg,nlat_veg,npft_veg)
      real, intent(in)         :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)
      real, intent(in)         :: no_fertilizer(nlon_veg,nlat_veg,3)
      real, intent(in)         :: lon_veg(nlon_veg)
      real, intent(in)         :: lon_veg_edge(nlon_veg+1)
      real, intent(in)         :: lat_veg(nlat_veg)
      real, intent(in)         :: lat_veg_edge(nlat_veg+1)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
!     integer, parameter       :: veg_ext = 10
      integer, parameter       :: veg_ext = 20
      real, parameter          :: ks      = 8.75
      real, parameter          :: kc      = 0.24

      integer                  :: i, j, ii, jj, jl, ju, i_ndx, n, k
      integer                  :: ind_lon(plon+1)
      integer                  :: ind_lat(platl+1)
      integer                  :: mapping_ext(-veg_ext:nlon_veg+veg_ext)
      real                     :: total_land
      real                     :: lat1, lat2, lon1, lon2
      real                     :: x1, x2, y1, y2, dx, dy
      real                     :: area, total_area
      real                     :: lon_edge(plon+1)
      real                     :: lat_edge(platl+1)
      real                     :: total_no_dry(npft_veg,12)
      real                     :: total_no_wet(npft_veg,12)
      real                     :: total_no_ftl(5,12)
      real                     :: aw(npft_veg)
      real                     :: ad(npft_veg)
      real                     :: sai(npft_veg)
      real                     :: clai(12), csai(12),  crf(12)
      real                     :: maps_soil_no_dry_4d(plon,platl,npft_veg,12)
      real                     :: maps_soil_no_wet_4d(plon,platl,npft_veg,12)
      real                     :: maps_soil_no_ftl_4d(plon,platl,5,12)
      real                     :: lon_veg_edge_ext(-veg_ext:nlon_veg+veg_ext)
      logical                  :: found

!---------------------------------------------------------------------------
! 	... define wet and dry emission factors
!---------------------------------------------------------------------------
      aw( 1) = 0.00 ! Ice/Ocean/Desert/Water
      aw( 2) = 0.03 ! Needleleaf evergreen temperate trees
      aw( 3) = 0.03 ! Needleleaf evergreen boreal tree
      aw( 4) = 0.03 ! Needeleaf deciduous boreal tree
      aw( 5) = 0.03 ! Broadleaf evergreen tropical trees
      aw( 6) = 0.03 ! Broadleaf evergreen temperate trees
      aw( 7) = 0.03 ! Broadleaf deciduous tropical trees
      aw( 8) = 0.06 ! Broadleaf deciduous temperate trees
      aw( 9) = 0.03 ! Broadleaf deciduous boreal trees
      aw(10) = 0.03 ! Broadleaf evergreen shrub
      aw(11) = 0.06 ! Broadleaf deciduous temperate shrub
      aw(12) = 0.03 ! Broadleaf deciduous boreal shrub
      aw(13) = 0.05 ! C3 arctic grass
      aw(14) = 0.36 ! C3 non-arctic grass
      aw(15) = 0.36 ! C4 grass
      aw(16) = 0.36 ! Corn
      aw(17) = 0.36 ! Wheat

      ad( 1) = 0.00 ! Ice/Ocean/Desert/Water
      ad( 2) = 0.22 ! Needleleaf evergreen temperate trees
      ad( 3) = 0.22 ! Needleleaf evergreen boreal tree
      ad( 4) = 0.22 ! Needeleaf deciduous boreal tree
      ad( 5) = 0.22 ! Broadleaf evergreen tropical trees
      ad( 6) = 0.22 ! Broadleaf evergreen temperate trees
      ad( 7) = 0.22 ! Broadleaf deciduous tropical trees
      ad( 8) = 0.40 ! Broadleaf deciduous temperate trees
      ad( 9) = 0.22 ! Broadleaf deciduous boreal trees
      ad(10) = 0.22 ! Broadleaf evergreen shrub
      ad(11) = 0.40 ! Broadleaf deciduous temperate shrub
      ad(12) = 0.22 ! Broadleaf deciduous boreal shrub
      ad(13) = 0.37 ! C3 arctic grass
      ad(14) = 2.65 ! C3 non-arctic grass
      ad(15) = 2.65 ! C4 grass
      ad(16) = 2.65 ! Corn
      ad(17) = 2.65 ! Wheat

!---------------------------------------------------------------------------
! 	... define stomatal area index
!---------------------------------------------------------------------------
      sai( 1) = 0.000 ! Ice/Ocean/Desert/Water
      sai( 2) = 0.036 ! Needleleaf evergreen temperate trees
      sai( 3) = 0.036 ! Needleleaf evergreen boreal tree
      sai( 4) = 0.025 ! Needeleaf deciduous boreal tree
      sai( 5) = 0.120 ! Broadleaf evergreen tropical trees
      sai( 6) = 0.036 ! Broadleaf evergreen temperate trees
      sai( 7) = 0.120 ! Broadleaf deciduous tropical trees
      sai( 8) = 0.036 ! Broadleaf deciduous temperate trees
      sai( 9) = 0.036 ! Broadleaf deciduous boreal trees
      sai(10) = 0.020 ! Broadleaf evergreen shrub
      sai(11) = 0.020 ! Broadleaf deciduous temperate shrub
      sai(12) = 0.020 ! Broadleaf deciduous boreal shrub
      sai(13) = 0.010 ! C3 arctic grass
      sai(14) = 0.018 ! C3 non-arctic grass
      sai(15) = 0.020 ! C4 grass
      sai(16) = 0.32  ! Corn
      sai(17) = 0.32  ! Wheat

      jl = base_lat + 1
      ju = base_lat + platl

      do i = 1,plon
         lon_edge(i) = lam(i) * r2d - .5*(lam(2) - lam(1)) * r2d
      end do
      lon_edge(plon+1) = lon_edge(plon) + (lam(2) - lam(1)) * r2d
      if( .not. has_npole ) then
        do j = 1,platl+1
           lat_edge(j) = phi(j+base_lat) * r2d - .5*(phi(2) - phi(1)) * r2d
        end do
      else
        do j = 1,platl
           lat_edge(j) = phi(j+base_lat) * r2d - .5*(phi(2) - phi(1)) * r2d
        end do
        lat_edge(platl+1) = lat_edge(platl) + (phi(2) - phi(1)) * r2d
      end if
      do j = 1,platl+1
         lat_edge(j) = min( lat_edge(j), 90. )
         lat_edge(j) = max( lat_edge(j),-90. )
      end do

!-------------------------------------------------------------------------------------
! 	... wrap around the longitudes
!-------------------------------------------------------------------------------------
      do i = -veg_ext,0
        lon_veg_edge_ext(i) = lon_veg_edge(nlon_veg+i) - 360.
        mapping_ext     (i) =              nlon_veg+i
      end do
      do i = 1,nlon_veg
        lon_veg_edge_ext(i) = lon_veg_edge(i)
        mapping_ext     (i) =              i
      end do
      do i = nlon_veg+1,nlon_veg+veg_ext
        lon_veg_edge_ext(i) = lon_veg_edge(i-nlon_veg) + 360.
        mapping_ext     (i) =              i-nlon_veg
      end do
#ifdef DEBUG
      write(*,*) 'interp_map_soil_no : lon_edge ',lon_edge
      write(*,*) 'interp_map_soil_no : lat_edge ',lat_edge
      write(*,*) 'interp_map_soil_no : mapping_ext ',mapping_ext
#endif
      do j = 1,plon+1
        lon1 = lon_edge(j) 
        found = .false.
        do i = -veg_ext,nlon_veg+veg_ext-1
          dx = lon_veg_edge_ext(i  ) - lon1
          dy = lon_veg_edge_ext(i+1) - lon1
          if( dx*dy <= 0. ) then
            ind_lon(j) = i
            found = .true.
            exit
          end if
        end do
        if( .not. found ) then
           write(*,*) ' '
           write(*,*) 'interp_map_soil_no: failed to find interval for j,lon1 = ',j,lon1
           write(*,*) 'interp_map_soil_no: lon_veg_edge_ext(',-veg_ext,':',-veg_ext+1,') = ', &
                      lon_veg_edge_ext(-veg_ext:-veg_ext+1)
           write(*,*) 'interp_map_soil_no: lon_veg_edge_ext(', &
                      nlon_veg+veg_ext-1,':',nlon_veg+veg_ext,') = ', &
                      lon_veg_edge_ext(nlon_veg+veg_ext-1:nlon_veg+veg_ext)
           write(*,*) ' '
           call endrun
        end if
      end do

      do j = 1,platl+1
        lat1 = lat_edge(j)
        found = .false.
        do i = 1,nlat_veg
          dx = lat_veg_edge(i  ) - lat1
          dy = lat_veg_edge(i+1) - lat1
          if( dx*dy <= 0. ) then
            ind_lat(j) = i
            found = .true.
            exit
          end if
        end do
        if( .not. found ) then
           write(*,*) ' '
           write(*,*) 'interp_map_soil_no: failed to find interval for j,lat1 = ',j,lat1
           write(*,*) 'interp_map_soil_no: lat_veg_edge(1:2) = ',lat_veg_edge(1:2)
           write(*,*) 'interp_map_soil_no: lat_veg_edge(',nlat_veg,':',nlat_veg+1,') = ', &
                       lat_veg_edge(nlat_veg:nlat_veg+1)
           write(*,*) ' '
           call endrun
        end if
      end do

#ifdef DEBUG
      write(*,*) 'interp_map_soil_no : ind_lon ',ind_lon
      write(*,*) 'interp_map_soil_no : ind_lat ',ind_lat
#endif
lat_loop : &
      do j = 1,platl
lon_loop : &
        do i = 1,plon
          total_area   = 0.
          total_no_dry = 0.
          total_no_wet = 0.
          total_no_ftl = 0.
          do jj = ind_lat(j),ind_lat(j+1)
            y1 = max( lat_edge(j),lat_veg_edge(jj) )
            y2 = min( lat_edge(j+1),lat_veg_edge(jj+1) ) 
            dy = (y2 - y1)/(lat_veg_edge(jj+1) - lat_veg_edge(jj))
            do ii =ind_lon(i),ind_lon(i+1)
              i_ndx = mapping_ext(ii)
              x1 = max( lon_edge(i),lon_veg_edge_ext(ii) )
              x2 = min( lon_edge(i+1),lon_veg_edge_ext(ii+1) ) 
              dx = (x2 - x1)/(lon_veg_edge_ext(ii+1) - lon_veg_edge_ext(ii))
              area = dx * dy
              total_area = total_area + area
!-----------------------------------------------------------------
! 	... no emissions for ocean grid point 
!-----------------------------------------------------------------
              if( nint(landmask(i_ndx,jj)) == 0 ) cycle
              do k = 2,npft_veg
                if( vegetation_map(i_ndx,jj,k) == 0 ) cycle
                clai = monthly_lai_clm(i_ndx,jj,k,:)
                csai = sai(k)
                crf  = .5 * (exp( -ks*csai ) + exp( -kc*clai ))
                total_no_dry(k,:) = total_no_dry(k,:) + ad(k) * crf(:) * vegetation_map(i_ndx,jj,k) * area
                total_no_wet(k,:) = total_no_wet(k,:) + aw(k) * crf(:) * vegetation_map(i_ndx,jj,k) * area
!---------------------------------------------------------------------------
! 	... fertilizer impact, including seasonal cycle of application based on growth
!---------------------------------------------------------------------------
                if( k >= 13 .and. k <= 15 ) then
                  do n = 1,12
                    if( growth(jj,k-12,n) > 0 ) then
                      total_no_ftl(k-12,n) = total_no_ftl(k-12,n) &
                                           + no_fertilizer(i_ndx,jj,1) * 12./sum(growth(jj,k-12,1:12)) &
                                           * crf(n) * vegetation_map(i_ndx,jj,k) * area
                    end if
                  end do
                elseif ( k > 15 ) then
                  do n = 1,12
                    if( growth(jj,k-12,n) > 0 ) then
                      total_no_ftl(k-12,n) = total_no_ftl(k-12,n)  &
                                           + sum(no_fertilizer(i_ndx,jj,2:3)) * 12./sum(growth(jj,k-12,1:12)) &
                                           * crf(n) * vegetation_map(i_ndx,jj,k) * area
                    end if
                  end do
                end if
              end do
            end do
          end do
!---------------------------------------------------------------------------
! 	... divide by total grid area
!---------------------------------------------------------------------------
          maps_soil_no_dry_4d(i,j,:,:) = total_no_dry/(total_area+1.e-30)
          maps_soil_no_wet_4d(i,j,:,:) = total_no_wet/(total_area+1.e-30)
          maps_soil_no_ftl_4d(i,j,:,:) = total_no_ftl/(total_area+1.e-30)
        end do lon_loop
      end do lat_loop

!-------------------------------------------------------------------------------------
! 	... reshape according to lat-lon blocks
!-------------------------------------------------------------------------------------
      do i = 1,12
        do ii = 1,npft_veg
          maps_soil_no_dry(:,:,:,ii,i) = reshape( maps_soil_no_dry_4d(:,:,ii,i), (/plonl,platl,pplon/), order=(/1,3,2/) )
          maps_soil_no_wet(:,:,:,ii,i) = reshape( maps_soil_no_wet_4d(:,:,ii,i), (/plonl,platl,pplon/), order=(/1,3,2/) )
        end do
      end do

      do i = 1,12
        do ii = 1,5
          maps_soil_no_ftl(:,:,:,ii,i) = reshape( maps_soil_no_ftl_4d(:,:,ii,i), (/plonl,platl,pplon/), order=(/1,3,2/) )
        end do
      end do
!
!     print *,'maps_soil_no_dry ',maxval(maps_soil_no_dry),maxval(maps_soil_no_dry_4d)
!     print *,'maps_soil_no_wet ',maxval(maps_soil_no_wet),maxval(maps_soil_no_wet_4d)
!     print *,'maps_soil_no_ftl ',maxval(maps_soil_no_ftl),maxval(maps_soil_no_ftl_4d)
!
      end subroutine interp_map_soil_no

      subroutine soil_no_emissions( lat, ip, calday, deltat, ts, &
                                    precip, qflx, sflx, plonl )

      use mo_grid,    only : pcnst
      use chem_mods,  only : adv_mass
      use mo_histout, only : outfld

      implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
      integer, intent(in) :: lat            ! latitude index
      integer, intent(in) :: ip             ! longitude tile index
      integer, intent(in) :: plonl          ! longitude tile dimension
      real, intent(in)    :: calday         ! julian day + fraction (greenwich)
      real, intent(in)    :: deltat         ! time step (s) 
      real, intent(in)    :: ts(plonl)      ! surface temperature (K)
      real, intent(in)    :: precip(plonl)  ! total instantaneous precipitation
      real, intent(in)    :: qflx(plonl)    ! surface moisture flux
      real, intent(inout) :: sflx(plonl)    ! surface flux for advected species

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
      real, parameter :: amufac               = 1.65979e-23                    ! conversion from mol/cm2 to kg/m2
      real, parameter :: pulse_decay(0:3)     = (/ 0.000, 0.805, 0.384, 0.208 /)
      real, parameter :: pulse_amplitude(0:3) = (/ 1.000, 11.19, 14.68, 18.46 /)
      real, parameter :: pulse_duration(0:3)  = (/ 0.000, 3.000, 7.000, 14.00 /)

      integer :: month,total_days
      integer :: i,k,n
      integer :: days_in_month(12) = (/ 31,28,31,30,31,30,31,31,30,31,30,31 /)
      real    :: total_emis,pulse,emis_factor,ts_C,t_soil,pulse_factor,precip_cm_per_day
      real    :: evaporation
      real    :: ta(npft_veg_soil_no)
      real    :: tb(npft_veg_soil_no)
      logical :: dry

      ta( 1) = 0.00 ! Ice/Ocean/Desert/Water
      ta( 2) = 0.84 ! Needleleaf evergreen temperate trees
      ta( 3) = 0.84 ! Needleleaf evergreen boreal tree
      ta( 4) = 0.84 ! Needeleaf deciduous boreal tree
      ta( 5) = 0.84 ! Broadleaf evergreen tropical trees
      ta( 6) = 0.84 ! Broadleaf evergreen temperate trees
      ta( 7) = 0.84 ! Broadleaf deciduous tropical trees
      ta( 8) = 0.84 ! Broadleaf deciduous temperate trees
      ta( 9) = 0.84 ! Broadleaf deciduous boreal trees
      ta(10) = 0.84 ! Broadleaf evergreen shrub
      ta(11) = 0.84 ! Broadleaf deciduous temperate shrub
      ta(12) = 0.84 ! Broadleaf deciduous boreal shrub
      ta(13) = 0.66 ! C3 arctic grass
      ta(14) = 0.66 ! C3 non-arctic grass
      ta(15) = 0.66 ! C4 grass
      ta(16) = 0.72 ! Corn
      ta(17) = 1.03 ! Wheat

      tb( 1) = 0.00 ! Ice/Ocean/Desert/Water
      tb( 2) = 3.60 ! Needleleaf evergreen temperate trees
      tb( 3) = 3.60 ! Needleleaf evergreen boreal tree
      tb( 4) = 3.60 ! Needeleaf deciduous boreal tree
      tb( 5) = 3.60 ! Broadleaf evergreen tropical trees
      tb( 6) = 3.60 ! Broadleaf evergreen temperate trees
      tb( 7) = 3.60 ! Broadleaf deciduous tropical trees
      tb( 8) = 3.60 ! Broadleaf deciduous temperate trees
      tb( 9) = 3.60 ! Broadleaf deciduous boreal trees
      tb(10) = 3.60 ! Broadleaf evergreen shrub
      tb(11) = 3.60 ! Broadleaf deciduous temperate shrub
      tb(12) = 3.60 ! Broadleaf deciduous boreal shrub
      tb(13) = 8.80 ! C3 arctic grass
      tb(14) = 8.80 ! C3 non-arctic grass
      tb(15) = 8.80 ! C4 grass
      tb(16) = 5.80 ! Corn
      tb(17) = 2.90 ! Wheat

!---------------------------------------------------------------------------
! 	... find month from calendar day
!---------------------------------------------------------------------------
      month = 0
      if ( calday >= 365 ) then
        month = 12
      else
        do n = 1,12
          total_days = sum( days_in_month(1:n) )
          if ( calday <= total_days ) then
            month = n
            exit
          end if
        end do
      end if
      if( month == 0 ) then
        write(*,*) 'soil_no_emissions: cannot find month for calday = ',calday
        call endrun
      end if

long_loop : &
      do i = 1,plonl
!---------------------------------------------------------------------------
! 	... update bucket (make sure we don't get a runaway dryness)
!---------------------------------------------------------------------------
        evaporation = qflx(i)
        soil_wetness(i,lat,ip) = soil_wetness(i,lat,ip) + (precip(i)*1.e3 - evaporation)*deltat
        soil_wetness(i,lat,ip) = max( 0.,soil_wetness(i,lat,ip) )
!---------------------------------------------------------------------------
! 	... define dry soils
!---------------------------------------------------------------------------
        dry = soil_wetness(i,lat,ip) == 0.
!---------------------------------------------------------------------------
! 	... is it already pulsing ?
!---------------------------------------------------------------------------
        if( is_pulsing(i,lat,ip) > 0 ) then
          time_pulsing(i,lat,ip) = time_pulsing(i,lat,ip) + deltat/86400.
          if( time_pulsing(i,lat,ip) > pulse_duration(is_pulsing(i,lat,ip)) ) then
            is_pulsing(i,lat,ip) = 0
          end if
        end if
!---------------------------------------------------------------------------
! 	... change units from m/s to cm/day
!---------------------------------------------------------------------------
        precip_cm_per_day = precip(i) * 86400. * 100.
!---------------------------------------------------------------------------
! 	... is there a new pulse
!---------------------------------------------------------------------------
        if( is_pulsing(i,lat,ip) == 0 .and. dry ) then
          if( precip_cm_per_day <= 0.1 ) then
            is_pulsing  (i,lat,ip) = 0
            time_pulsing(i,lat,ip) = 1.
          else if( precip_cm_per_day <= 0.5 ) then
            is_pulsing  (i,lat,ip) = 1
            time_pulsing(i,lat,ip) = 1.
          else if( precip_cm_per_day <= 1.5 ) then
            is_pulsing  (i,lat,ip) = 2
            time_pulsing(i,lat,ip) = 1.
          else
            is_pulsing  (i,lat,ip) = 3
            time_pulsing(i,lat,ip) = 1.
          end if
        end if
        pulse_factor = pulse_amplitude(is_pulsing(i,lat,ip)) * exp(-pulse_decay(is_pulsing(i,lat,ip))*time_pulsing(i,lat,ip))

!---------------------------------------------------------------------------
! 	... no emission if ground is frozen
!---------------------------------------------------------------------------
        ts_C = ts(i) - 273.16
        if( ts_C <= 0. ) then
           cycle
        end if

!---------------------------------------------------------------------------
! 	... loop over vegetation types
!---------------------------------------------------------------------------
        total_emis = 0.
veg_type_loop : &
        do k = 2,npft_veg_soil_no
          if( maps_soil_no_dry(i,lat,ip,k,month) /= 0. ) then
!---------------------------------------------------------------------------
! 	... switch between dry and wet soils
!---------------------------------------------------------------------------
             if( dry ) then
               t_soil = ts_C + 5.
               if( t_soil < 30. ) then
                 emis_factor = maps_soil_no_dry(i,lat,ip,k,month) * t_soil/30.
               else
                 emis_factor = maps_soil_no_dry(i,lat,ip,k,month)
               end if
             else
               t_soil = ta(k) * ts_C + tb(k)
               if( t_soil <= 10. ) then
                 emis_factor = 0.28 * maps_soil_no_wet(i,lat,ip,k,month) * t_soil
               else if( t_soil <= 30. ) then
                 emis_factor = maps_soil_no_wet(i,lat,ip,k,month) * exp(0.103*t_soil)
               else
                 emis_factor = 21.97 * maps_soil_no_wet(i,lat,ip,k,month)
               end if
             end if

!---------------------------------------------------------------------------
! 	... fertilization effect
!---------------------------------------------------------------------------
             if( k >= 13 .and. k <= 15 ) then
               emis_factor = emis_factor + maps_soil_no_ftl(i,lat,ip,k-12,month)
             else if ( k > 15 ) then
               emis_factor = emis_factor + maps_soil_no_ftl(i,lat,ip,k-12,month)
             end if

             total_emis = total_emis + emis_factor * pulse_factor
          end if
        end do veg_type_loop

!---------------------------------------------------------------------------
! 	... scale emissions from ngN/m2/s to mol/cm2/s
!---------------------------------------------------------------------------
        total_emis = total_emis * (6.02e10/14.)
!---------------------------------------------------------------------------
! 	... scale emissions to MOZART units
!---------------------------------------------------------------------------
        total_emis = total_emis * amufac * adv_mass(no_ndx)
!---------------------------------------------------------------------------
! 	... add to other emissions of NO
!---------------------------------------------------------------------------
        sflx(i) = sflx(i) + total_emis
      end do long_loop

      call outfld( 'SOIL_WETNESS', soil_wetness(1,lat,ip), plonl, ip, lat, 1 )
      call outfld( 'QFLX'        , qflx        (1)       , plonl, ip, lat, 1 )

      end subroutine soil_no_emissions

      subroutine soil_no_rdrst( plonl, platl, pplon, ncid )
!---------------------------------------------------------------------- 
!	... read soil no restart variables
!---------------------------------------------------------------------- 
      
      use netcdf
      use mo_mpi,  only : base_lat
      use mo_grid, only : plon

      implicit none

!---------------------------------------------------------------------- 
!	... dummy arguments
!---------------------------------------------------------------------- 
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncid                          ! netcdf dataset id

!---------------------------------------------------------------------- 
!	... local variables
!---------------------------------------------------------------------- 
      integer              :: j
      integer              :: var_id
      integer              :: start(2)
      integer              :: cnt(2)
      integer              :: iwrk2d(plon,platl)
      real                 :: wrk2d(plon,platl)

      start(:) = (/ 1, base_lat+1 /)
      cnt(:)   = (/ plon, platl /)
!-----------------------------------------------------------------------
!     	... read is_pulsing
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'is_pulsing', var_id ), &
                         'soil_no_rdrst: Failed to get is_pulsing variable id' )
      call handle_ncerr( nf_get_vara_int( ncid, var_id, start, cnt, iwrk2d ), &
                         'soil_no_rdrst: failed to read is_pulsing variable' )
      do j = 1,platl
	 is_pulsing(:,j,:) = reshape( iwrk2d(:,j), (/ plonl, pplon /) )
      end do
      write(*,*) 'soil_no_rdrst: read is_pulsing'
!-----------------------------------------------------------------------
!     	... read soil_wetness
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'soil_wetness', var_id ), &
                         'soil_no_rdrst: Failed to get soil_wetness variable id' )
      call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk2d ), &
                         'soil_no_rdrst: failed to read soil_wetness variable' )
      do j = 1,platl
	 soil_wetness(:,j,:) = reshape( wrk2d(:,j), (/ plonl, pplon /) )
      end do
      write(*,*) 'soil_no_rdrst: read soil_wetness'
!-----------------------------------------------------------------------
!     	... read time_pulsing
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'time_pulsing', var_id ), &
                         'soil_no_rdrst: Failed to get time_pulsing variable id' )
      call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk2d ), &
                         'soil_no_rdrst: failed to read time_pulsing variable' )
      do j = 1,platl
	 time_pulsing(:,j,:) = reshape( wrk2d(:,j), (/ plonl, pplon /) )
      end do
      write(*,*) 'soil_no_rdrst: time_pulsing soil_wetness'

      end subroutine soil_no_rdrst

      subroutine soil_no_wrrst( plonl, platl, pplon, ncid )
!---------------------------------------------------------------------- 
!	... write soil no restart variables
!---------------------------------------------------------------------- 
      
      use netcdf
      use mo_mpi
      use mo_grid, only : plon, plat

      implicit none

!---------------------------------------------------------------------- 
!	... dummy arguments
!---------------------------------------------------------------------- 
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncid                          ! netcdf dataset id

!---------------------------------------------------------------------- 
!	... local variables
!---------------------------------------------------------------------- 
      integer              :: astat
      integer              :: istat
      integer              :: j
      integer              :: var_id
      integer              :: count
      integer              :: node
      integer              :: nodes
      integer              :: offset
      integer              :: iwrk2d(plon,plat)
      real                 :: wrk2d(plon,plat)
      integer, allocatable :: igather_2d(:,:,:,:)
      real   , allocatable :: gather_2d(:,:,:,:)

#ifdef USE_MPI
      if( ded_io_node ) then
	 nodes = maxnodes + 1
      else
	 nodes = maxnodes
      end if
!-----------------------------------------------------------------------
!     	... gather is_pulsing
!-----------------------------------------------------------------------
      count = plon*platl
      if( io_node ) then
         allocate( igather_2d(plonl,platl,pplon,nodes),stat=astat )
      else
         allocate( igather_2d(1,1,1,1),stat=astat )
      end if
      if( astat /= 0 ) then
	 write(*,*) 'soil_no_wrrst: failed to allocate igather_2d; error = ',astat
	 call endrun
      end if
      call mpi_gather( is_pulsing, count, mpi_integer, &
	               igather_2d, count, mpi_integer, &
		       gather_node, mpi_comm_world, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'soil_no_wrrst: mpi_gather failed for is_pulsing; error code = ',istat
	 call endrun
      end if
#endif
!-----------------------------------------------------------------------
!     	... write is_pulsing
!-----------------------------------------------------------------------
      if( io_node ) then
         call handle_ncerr( nf_inq_varid( ncid, 'is_pulsing', var_id ), &
                            'soil_no_wrrst: failed to get is_pulsing variable id' )
#ifdef USE_MPI
	 do node = 1,maxnodes
	    offset = (node-1)*platl
	    do j = 1,platl
	       iwrk2d(:,j+offset) = reshape( igather_2d(:,j,:,node), (/ plon /) )
	    end do
	 end do
#else
	 do j = 1,platl
	    iwrk2d(:,j) = reshape( is_pulsing(:,j,:), (/ plon /) )
	 end do
#endif
         call handle_ncerr( nf_put_var_int( ncid, var_id, iwrk2d ), &
                            'soil_no_wrrst: failed to write is_pulsing variable' )
         write(*,*) 'soil_no_wrrst: is_pulsing written'
      end if
      if( allocated( igather_2d ) ) then
         deallocate( igather_2d )
      end if

!-----------------------------------------------------------------------
!     	... gather soil_wetness
!-----------------------------------------------------------------------
#ifdef USE_MPI
      if( io_node ) then
         allocate( gather_2d(plonl,platl,pplon,nodes),stat=astat )
      else
         allocate( gather_2d(1,1,1,1),stat=astat )
      end if
      if( astat /= 0 ) then
	 write(*,*) 'soil_no_wrrst: failed to allocate gather_2d; error = ',astat
	 call endrun
      end if
      call mpi_gather( soil_wetness, count, mpi_double_precision, &
	               gather_2d, count, mpi_double_precision, &
		       gather_node, mpi_comm_world, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'soil_no_wrrst: mpi_gather failed for soil_wetness; error code = ',istat
	 call endrun
      end if
#endif
!-----------------------------------------------------------------------
!     	... write soil_wetness
!-----------------------------------------------------------------------
      if( io_node ) then
         call handle_ncerr( nf_inq_varid( ncid, 'soil_wetness', var_id ), &
                            'soil_no_wrrst: failed to get soil_wetness variable id' )
#ifdef USE_MPI
	 do node = 1,maxnodes
	    offset = (node-1)*platl
	    do j = 1,platl
	       wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
	    end do
	 end do
#else
	 do j = 1,platl
	    wrk2d(:,j) = reshape( soil_wetness(:,j,:), (/ plon /) )
	 end do
#endif
         call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d ), &
                            'soil_no_wrrst: failed to write soil_wetness variable' )
         write(*,*) 'soil_no_wrrst: soil_wetness written'
      end if

!-----------------------------------------------------------------------
!     	... gather time_pulsing
!-----------------------------------------------------------------------
#ifdef USE_MPI
      call mpi_gather( time_pulsing, count, mpi_double_precision, &
	               gather_2d, count, mpi_double_precision, &
		       gather_node, mpi_comm_world, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'soil_no_wrrst: mpi_gather failed for time_pulsing; error code = ',istat
	 call endrun
      end if
#endif
!-----------------------------------------------------------------------
!     	... write time_pulsing
!-----------------------------------------------------------------------
      if( io_node ) then
         call handle_ncerr( nf_inq_varid( ncid, 'time_pulsing', var_id ), &
                            'soil_no_wrrst: failed to get time_pulsing variable id' )
#ifdef USE_MPI
	 do node = 1,maxnodes
	    offset = (node-1)*platl
	    do j = 1,platl
	       wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
	    end do
	 end do
#else
	 do j = 1,platl
	    wrk2d(:,j) = reshape( time_pulsing(:,j,:), (/ plon /) )
	 end do
#endif
         call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d ), &
                            'soil_no_wrrst: failed to write time_pulsing variable' )
         write(*,*) 'soil_no_wrrst: time_pulsing written'
      end if
      if( allocated( gather_2d ) ) then
         deallocate( gather_2d )
      end if

      end subroutine soil_no_wrrst

      end module mo_soil_no
