
      module mo_strato_sad
!---------------------------------------------------------------
! 	... sad module
!---------------------------------------------------------------

      use mo_grid,   only : plon, plat, plev, plevp
      use m_types,   only : time_ramp

      implicit none

      private
      public  :: strato_sad_inti
      public  :: strato_sad_timestep_init
      public  :: strato_sad_set

      save

      integer, parameter    :: time_span = 4

      integer               :: ntimes
      integer               :: nlon
      integer               :: nlat
      integer               :: nlev
      integer               :: gndx = 0
      integer               :: tim_ndx(2)
      integer               :: jlim(2)
      integer, allocatable  :: dates(:)
      real, allocatable     :: times(:)
      real, allocatable     :: sad_lats(:)
      real, allocatable     :: sad_lons(:)
      real, allocatable     :: sad_levs(:)
      real, allocatable     :: sage_sad(:,:,:,:,:)
      character(len=80)     :: filename
      character(len=80)     :: lpath
      character(len=80)     :: rpath

      type(time_ramp)   :: strato_sad_timing

      contains

      subroutine strato_sad_inti( plonl, platl, pplon, ncdate, ncsec, &
                                  ncfile, loc_path, rem_path, strato_sad_timing_in )
!-----------------------------------------------------------------------
! 	... initialize the strato sad module
!-----------------------------------------------------------------------

      use mo_constants,  only : phi, lam, d2r
      use mo_regrider,   only : regrid_inti, regrid_lat_limits
      use mo_file_utils, only : open_netcdf_file
      use mo_charutl,    only : upcase
      use mo_mpi,        only : base_lat, masternode
      use netcdf

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate
      integer, intent(in) :: ncsec
      character(len=*), intent(in) :: ncfile
      character(len=*), intent(in) :: loc_path
      character(len=*), intent(in) :: rem_path
      type(time_ramp), intent(in)  :: strato_sad_timing_in

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      real, parameter :: hPa2Pa = 100.
      integer :: astat
      integer :: j, l, m, n                           ! Indices
      integer :: t1, t2
      integer :: ncid
      integer :: dimid
      integer :: varid
      integer :: yr, wrk_date, wrk_sec
      real    :: seq
      real    :: wrk_time
      character(len=8)  :: time_type

!-----------------------------------------------------------------------
! 	... check timing
!-----------------------------------------------------------------------
      strato_sad_timing = strato_sad_timing_in
      call upcase( strato_sad_timing%type, time_type )
      strato_sad_timing%type = time_type
      if( time_type /= 'SERIAL' .and. time_type /= 'CYCLICAL' &
                                .and. time_type /= 'FIXED' ) then
         write(*,*) 'strato_sad_inti: time type ',trim(time_type),' is not SERIAL,CYCLICAL, or FIXED'
         call endrun
      end if

      wrk_sec  = ncsec
      if( time_type == 'SERIAL' ) then
	 wrk_date = ncdate + strato_sad_timing%yr_offset*10000
      else if( time_type == 'CYCLICAL' ) then
         wrk_date = (strato_sad_timing%date/10000)*10000 + mod(ncdate,10000)
      else
         wrk_date = strato_sad_timing%date
         wrk_sec  = 0
      end if
      wrk_time = days0( wrk_date, wrk_sec )
      write(*,*) 'strato_sad_inti: wrk_date,wrk_sec,wrk_time = ',wrk_date,wrk_sec,wrk_time

      lpath    = loc_path
      rpath    = rem_path
      filename = ncfile

!-----------------------------------------------------------------------
! 	... diagnostics
!-----------------------------------------------------------------------
      write(*,*) ' '
      write(*,*) 'strato_sad_inti: diagnostics'
      write(*,*) ' '
      write(*,*) 'strato sad timing specs'
      write(*,*) 'type = ',strato_sad_timing%type
      if( time_type == 'SERIAL' ) then
         write(*,*) 'year offset = ',strato_sad_timing%yr_offset
      else if( time_type == 'CYCLICAL' ) then
         write(*,*) 'year = ',strato_sad_timing%date/10000
      else
         write(*,*) 'date = ',strato_sad_timing%date
      end if

!-----------------------------------------------------------------------
! 	... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( filename, lpath, rpath )
!-----------------------------------------------------------------------
! 	... get timing information, allocate arrays, and read in dates
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid ), &
                         'strato_sad_inti: failed to find time dimension' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, ntimes ), &
                         'strato_sad_inti: failed to get size of time dimension' )
      allocate( dates(ntimes),stat=astat )
      if( astat/= 0 ) then
	 write(*,*) 'strato_sad_inti: failed to allocate dates array; error = ',astat
	 call endrun
      end if
      allocate( times(ntimes),stat=astat )
      if( astat/= 0 ) then
	 write(*,*) 'strato_sad_inti: failed to allocate times array; error = ',astat
	 call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'date', varid ), &
                         'strato_sad_inti: Failed to find date variable' )
      call handle_ncerr( nf_get_var_int( ncid, varid, dates ), &
                         'strato_sad_inti: Failed to read date variable' )
      do n = 1,ntimes
         times(n) = days0( dates(n), 0 )
      end do
      if( time_type /= 'CYCLICAL' ) then
         if( wrk_time < times(1) .or. wrk_time > times(ntimes) ) then
            write(*,*) 'strato_sad_inti: time out of bounds for dataset = ',trim(filename)
            call endrun
         end if
         do n = 2,ntimes
            if( wrk_time <= times(n) ) then
               exit
            end if
         end do
         tim_ndx(1) = n - 1
      else
         yr = strato_sad_timing%date/10000
         do n = 1,ntimes
            if( yr == dates(n)/10000 ) then
               exit
            end if
         end do
         if( n >= ntimes ) then
            write(*,*) 'strato_sad_inti: time out of bounds for dataset = ',trim(filename)
            call endrun
	 end if
         tim_ndx(1) = n
      end if
      select case( time_type )
         case( 'FIXED' )
            tim_ndx(2) = n
         case( 'CYCLICAL' )
            do n = tim_ndx(1),ntimes
               if( yr /= dates(n)/10000 ) then
                  exit
               end if
            end do
            tim_ndx(2) = n - 1
	    if( (tim_ndx(2) - tim_ndx(1)) < 2 ) then
               write(*,*) 'strato_sad_inti: cyclical lb conds require at least two time points'
               call endrun
            end if
         case( 'SERIAL' )
            tim_ndx(2) = min( ntimes,tim_ndx(1) + time_span )
      end select
      t1 = tim_ndx(1)
      t2 = tim_ndx(2)

      write(*,*) ' '
      write(*,*) 'strato_sad time cnt = ',ntimes
      write(*,*) 'strato_sad times'
      write(*,'(10i10)') dates(:)
      write(*,'(1p,5g15.7)') times(:)
      write(*,*) 'strato_sad time indicies = ',tim_ndx(:)
      write(*,'(10i10)') dates(tim_ndx(1):tim_ndx(2))
      write(*,*) ' '

!-----------------------------------------------------------------------
!     	... inquire about latitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), &
                         'strato_sad_inti: failed to get latitude dimension id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat  ), &
                         'strato_sad_inti: failed to get latitude dimension size' )
!-----------------------------------------------------------------------
!     	... allocate space for latitude coordinates
!-----------------------------------------------------------------------
      allocate( sad_lats(nlat),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) ' strato_sad_inti : failed to allocate latitudes array; error = ',astat
         call endrun
      end if
!-----------------------------------------------------------------------
!     	... get latitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'lat', varid ), 'strato_sad_inti: getting latitudes id' )
      call handle_ncerr( nf_get_var_double( ncid, varid, sad_lats ), 'strato_sad_inti: getting latitudes' )

!-----------------------------------------------------------------------
!     	... inquire about longitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), &
                         'strato_sad_inti: failed to get longitude dimension id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon  ), &
                         'strato_sad_inti: failed to get longitude dimension size' )
!-----------------------------------------------------------------------
!     	... allocate space for longitude coordinates
!-----------------------------------------------------------------------
      allocate( sad_lons(nlon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) ' strato_sad_inti : failed to allocate longitudes array; error = ',astat
         call endrun
      end if
!-----------------------------------------------------------------------
!     	... get longitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'lon', varid ), 'strato_sad_inti: getting longitudes id' )
      call handle_ncerr( nf_get_var_double( ncid, varid, sad_lons ), 'strato_sad_inti: getting longitudes' )

!-----------------------------------------------------------------------
!     	... convert to radians and setup regridding
!-----------------------------------------------------------------------
      sad_lats(:) = d2r * sad_lats(:)
      sad_lons(:) = d2r * sad_lons(:)
      gndx = regrid_inti( nlat, plat, nlon, plon, sad_lons, &
                          lam, sad_lats, phi, 0, platl, do_lons=.false., do_lats=.true. )
      if( gndx < 0 ) then
         write(*,*) 'strato_sad_inti: regrid_inti error = ',gndx
         call endrun
      else
         write(*,*) 'strato_sad_inti: regrid_inti index = ',gndx
      end if

      if( gndx /= 0 ) then
         jlim = regrid_lat_limits( gndx )
      else
         jlim(:) = (/ base_lat+1, base_lat+platl /)
      end if
      write(*,*) 'strato_sad_inti: jlim = ',jlim

!-----------------------------------------------------------------------
!     	... inquire about levels
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lev', dimid ), &
                         'strato_sad_inti: failed to get lev dimension id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlev  ), &
                         'strato_sad_inti: failed to get lev dimension size' )
!-----------------------------------------------------------------------
!     	... allocate space for level coordinates
!-----------------------------------------------------------------------
      allocate( sad_levs(nlev),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) ' strato_sad_inti : failed to allocate levels array; error = ',astat
         call endrun
      end if
!-----------------------------------------------------------------------
!     	... get levels
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'lev', varid ), 'strato_sad_inti: getting levels id' )
      call handle_ncerr( nf_get_var_double( ncid, varid, sad_levs ), 'strato_sad_inti: getting levels' )
      sad_levs(:) = hPa2Pa * sad_levs(:)

      write(*,*) ' '
      write(*,*) 'strato_sad_inti: sad_levs'
      write(*,'(10g12.5)') sad_levs(:)
      write(*,*) ' '

!-----------------------------------------------------------------------
!     	... allocate module sad variable
!-----------------------------------------------------------------------
      allocate( sage_sad(plonl,nlev,platl,pplon,t1:t2),stat=astat )
      if( astat /= 0 ) then
         write(*,*) ' strato_sad_inti : failed to allocate sage_sad array; error = ',astat
	 call endrun
      end if
!-----------------------------------------------------------------------
! 	... read and interpolate sage sad data
!-----------------------------------------------------------------------
      call strato_sad_get( plonl, platl, pplon, ncid )
!-----------------------------------------------------------------------
! 	... close netcdf file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), &
                         'strato_sad_inti: Failed to close NetCDF file =' // trim(filename) )

      end subroutine strato_sad_inti

      subroutine strato_sad_timestep_init( plonl, platl, pplon, ncdate, ncsec )
!-----------------------------------------------------------------------
!       ... check serial case for time span
!-----------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate              ! present date (yyyymmdd)
      integer, intent(in) :: ncsec               ! seconds in present date

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: m
      integer                     :: t1, t2, tcnt
      integer                     :: astat
      integer                     :: ncid
      real                        :: wrk_time

      if( strato_sad_timing%type == 'SERIAL' ) then
         wrk_time = days0( ncdate + strato_sad_timing%yr_offset*10000, ncsec )
         if( wrk_time > times(tim_ndx(2)) ) then
            tcnt = tim_ndx(2) - tim_ndx(1)
            tim_ndx(1) = tim_ndx(2)
            tim_ndx(2) = min( ntimes,tim_ndx(1) + time_span )
            t1 = tim_ndx(1)
            t2 = tim_ndx(2)
!-----------------------------------------------------------------------
! 	... allocate array
!-----------------------------------------------------------------------
            if( allocated( sage_sad ) ) then
               deallocate( sage_sad,stat=astat )
               if( astat/= 0 ) then
                  write(*,*) 'strato_sad_timestep_init: failed to deallocate strato_sad vmr; error = ',astat
	          call endrun
               end if
	    end if
            allocate( sage_sad(plonl,nlev,platl,pplon,t1:t2),stat=astat )
            if( astat/= 0 ) then
	       write(*,*) 'strato_sad_timestep_init: failed to allocate sage_sad; error = ',astat
               call endrun
            end if
!-----------------------------------------------------------------------
! 	... open netcdf file
!-----------------------------------------------------------------------
            ncid = open_netcdf_file( filename, lpath, rpath )
!-----------------------------------------------------------------------
! 	... read and interpolate sage sad data
!-----------------------------------------------------------------------
            call strato_sad_get( plonl, platl, pplon, ncid )
!-----------------------------------------------------------------------
! 	... close netcdf file
!-----------------------------------------------------------------------
            call handle_ncerr( nf_close( ncid ), &
                               'strato_sad_timestep_init: failed to close netcdf file = ' // trim(filename) )
         end if
      end if

      end subroutine strato_sad_timestep_init

      subroutine strato_sad_get( plonl, platl, pplon, ncid )
!-----------------------------------------------------------------------
!       ... read lower bndy values
!-----------------------------------------------------------------------

      use mo_mpi,        only : base_lat
      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use mo_regrider,   only : regrid_1d

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)           :: plonl
      integer, intent(in)           :: platl
      integer, intent(in)           :: pplon
      integer, intent(in)           :: ncid

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer            :: j
      integer            :: jl, ju
      integer            :: k, n                            ! Indices
      integer            :: t1, t2, tcnt
      integer            :: ierr
      integer            :: vid
      real, allocatable  :: sage_sad_in(:,:,:)
      real, allocatable  :: wrk2d(:,:)

!-----------------------------------------------------------------------
!       ... read sage data
!-----------------------------------------------------------------------
      t1 = tim_ndx(1)
      t2 = tim_ndx(2)
      tcnt = t2 - t1 + 1
!-----------------------------------------------------------------------
!       ... allocate local wrk arrays
!-----------------------------------------------------------------------
      allocate( sage_sad_in(jlim(1):jlim(2),nlev,t1:t2), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'strato_sad_get: wrk allocation error = ',ierr
         call endrun
      end if
      allocate( wrk2d(platl,nlev),stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'read_aero: failed to allocate wrk2d; error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'sad_sage', vid ), &
                         'strato_sad_get: failed to find sad_sage id' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                             (/ jlim(1), 1, t1/), &                        ! start
                                             (/ jlim(2)-jlim(1)+1, nlev, tcnt /), &        ! count
                                             sage_sad_in ), &
                         'strato_sad_get: failed to read sad_sage ' )
      jl = base_lat + 1
      ju = base_lat + platl
      do n = t1,t2
         if( gndx /= 0 ) then
            do k = 1,nlev
               call regrid_1d( sage_sad_in(jlim(1):jlim(2),k,n), wrk2d(:,k), gndx, do_lat=.true., to_lat_min=jl, to_lat_max=ju )
             end do
         else
             do k = 1,nlev
                wrk2d(:platl,k) = sage_sad_in(jlim(1):jlim(2),k,n)
             end do
         end if
         do k = 1,nlev
            do j = 1,platl
               sage_sad(:,k,j,:,n) = wrk2d(j,k)
            end do
        end do
      end do
      deallocate( sage_sad_in, wrk2d, stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'strato_sad_get: failed to deallocate sage_sad_in,wrk2d, ierr = ',ierr
         call endrun
      end if

      end subroutine strato_sad_get

      subroutine strato_sad_set( pmid, lat, ip, ncdate, ncsec, sad, plonl )
!--------------------------------------------------------
!	... set the lower bndy values
!--------------------------------------------------------

      use mo_mpi,       only : base_lat
      use m_tracname,   only : tracnam

      implicit none

!--------------------------------------------------------
!	... dummy arguments
!--------------------------------------------------------
      integer, intent(in) ::   plonl
      integer, intent(in) ::   lat                 ! latitude index
      integer, intent(in) ::   ip                  ! longitude tile index
      integer, intent(in) ::   ncdate              ! present date (yyyymmdd)
      integer, intent(in) ::   ncsec               ! seconds in present date
      real, intent(in)    ::   pmid(plonl,plev)    ! midpoint pressure (Pa)
      real, intent(inout) ::   sad(plonl,plev)     ! stratospheric aerosol surface area density (1/cm)

!--------------------------------------------------------
!	... local variables
!--------------------------------------------------------
      integer  ::  i, k, n
      integer  ::  last, next
      integer  ::  wrk_date, wrk_sec
      integer  ::  tcnt
      integer  ::  astat
      real     ::  dels
      real     ::  wrk_time
      real     ::  sad_int(plonl,plev,2)

!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      wrk_sec  = ncsec
      select case( strato_sad_timing%type )
         case( 'SERIAL' )
            wrk_date = ncdate + strato_sad_timing%yr_offset*10000
         case( 'CYCLICAL' )
            wrk_date = (strato_sad_timing%date/10000)*10000 + mod( ncdate,10000 )
         case( 'FIXED' )
            wrk_date = strato_sad_timing%date
            wrk_sec  = 0
      end select
      wrk_time = days0( wrk_date, wrk_sec )

!--------------------------------------------------------
!	... set time interpolation factor
!--------------------------------------------------------
      if( strato_sad_timing%type /= 'CYCLICAL' ) then
         do n = tim_ndx(1)+1,tim_ndx(2)
            if( wrk_time <= times(n) ) then
               last = n - 1
               next = n
               exit
            end if
         end do
         if( n > ntimes ) then
            write(*,*) 'lbc_set: interp time is out of bounds'
            call endrun
         end if
         dels = (wrk_time - times(last))/(times(next) - times(last))
!        write(*,*) ' '
!        write(*,*) 'strato_sad_set: last,next,dels,ncdate,ncsec = ',last,next,dels,ncdate,ncsec
      else
         tcnt = tim_ndx(2) - tim_ndx(1) + 1
         call moz_findplb( times(tim_ndx(1)), tcnt, wrk_time, n )
         if( n < tcnt ) then
            last = tim_ndx(1) + n - 1
            next = last + 1
            dels = (wrk_time - times(last))/(times(next) - times(last))
         else
            next = tim_ndx(1)
            last = tim_ndx(2)
            dels = wrk_time - times(last)
            if( dels < 0. ) then
               dels = 365. + dels
            end if
            dels = dels/(365. + times(next) - times(last))
         end if
      end if

      dels = max( min( 1.,dels ),0. )

#ifdef DEBUG
      write(*,*) ' '
      write(*,*) 'strato_sad_set: last,next,dels,ncdate,ncsec = ',last,next,dels,ncdate,ncsec
      write(*,*) 'strato_sad_set: dates(last),dates(next)       = ',dates(last),dates(next)
#endif

      call vinterp( pmid, sage_sad(1,1,lat,ip,last), sad_int, plonl )
      call vinterp( pmid, sage_sad(1,1,lat,ip,next), sad_int(1,1,2), plonl )

#ifdef DEBUG
      write(*,*) 'strato_sad_set: pmid(1,:)'
      write(*,'(1p,5g15.7)') pmid(1,:)
      write(*,*) 'strato_sad_set: sad_levs'
      write(*,'(1p,5g15.7)') sad_levs(:)
      write(*,*) 'strato_sad_set: sage_sad(last)'
      write(*,'(1p,5g15.7)') sage_sad(1,:,lat,ip,last)
      write(*,*) 'strato_sad_set: sage_sad(next)'
      write(*,'(1p,5g15.7)') sage_sad(1,:,lat,ip,next)
#endif

      do k = 1,plev
         sad(:,k) = sad_int(:,k,1) + dels * (sad_int(:,k,2) - sad_int(:,k,1))
      end do

#ifdef DEBUG
      write(*,*) 'strato_sad_set: sad'
      write(*,'(1p,5g15.7)') sad(1,:)
      write(*,*) ' '
      call endrun 
#endif

      end subroutine strato_sad_set

      subroutine vinterp( pmid, sad_src, sad_int, plonl )
!-----------------------------------------------------------------------
!   	... vertically interpolate input data
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!   	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in)    :: pmid(plonl,plev)
      real, intent(in)    :: sad_src(plonl,nlev)
      real, intent(out)   :: sad_int(plonl,plev)

!-----------------------------------------------------------------------
!   	... local variables
!-----------------------------------------------------------------------
      integer :: i
      integer :: k, kl, ku
      real    :: delp, pinterp

level_loop : &
      do k = 1,plev
long_loop : &
         do i = 1,plonl
            pinterp = pmid(i,k)
            if( pinterp <= sad_levs(1) ) then
               sad_int(i,k) = sad_src(i,1)
            else if( pinterp > sad_levs(nlev) ) then
               sad_int(i,k) = sad_src(i,nlev)
            else
               do ku = 2,nlev
                  if( pinterp <= sad_levs(ku) ) then
                     kl = ku - 1
                     delp = log( pinterp/sad_levs(kl) ) &
                            / log( sad_levs(ku)/sad_levs(kl) )
                     sad_int(i,k) = sad_src(i,kl) + delp * (sad_src(i,ku) - sad_src(i,kl))
                     exit
                  end if
               end do
            end if
         end do long_loop
      end do level_loop

      end subroutine vinterp


      real function days0( ncdate, ncsec )
!----------------------------------------------------------------------- 
! Purpose: Convert date and seconds of day to floating point days since
!          00/01/01
! 
! Method: Use table of days per month to do conversion
! 
! Author: CCM Core Group
!-----------------------------------------------------------------------


      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)   :: ncdate      ! Current date as yymmdd or yyyymmdd
      integer, intent(in)   :: ncsec       ! Seconds of day for current date

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer  :: year        ! year
      integer  :: mnth        ! Month number
      integer  :: mday        ! Day number of month
      integer  :: jdcon(12) & ! Starting day number for each month
                       = (/ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 /)
      real     :: doy

!-----------------------------------------------------------------------
! 	... decode month and day
!-----------------------------------------------------------------------
      year = ncdate/10000
      mnth = mod( ncdate,10000 )/100
      if( mnth < 1 .or. mnth > 12) then
         write(*,*) 'days0: Bad month index = ', mnth
         call endrun
      end if
      mday = mod(ncdate,100)
      doy  = jdcon(mnth) + mday + ncsec/86400.

      if( doy < 1. .or. doy > 366. ) then
         write(*,*) 'days0: bad day of year = ',doy
         call endrun
      end if

      days0 = 365.*year + doy

      end function days0

      end module mo_strato_sad
