
      module mo_surface

      implicit none

      save

      real    :: rair, gravit
      logical :: table_soilw

      contains

      subroutine inisflx( xrair, xgravit )

      use mo_control,   only : xactive_drydep
      use mo_chem_utls, only : has_drydep

      implicit none

!----------------------------------------------------------------------------
! 	... dummy arguments:
!----------------------------------------------------------------------------

      real, intent(in) :: &
        xrair, &    ! gas constant for dry air
        xgravit     ! gravitational acceleration

!----------------------------------------------------------------------------
!  	... set the physical constants for surface flux routines
!----------------------------------------------------------------------------
      rair   = xrair
      gravit = xgravit
      table_soilw = xactive_drydep .and. (has_drydep( 'H2' ) .or. has_drydep( 'CO' ))

      end subroutine inisflx

      subroutine sflxdr( lat, ip, dtime, calday, ncdate, &
                         ncsec, ioro, as, pmid, rpdel, &
                         tv, zi, ts, sflx, snow, &
                         fsds, p_srf, wind_speed, spec_hum, precip, &
                         air_temp, soilw, shflx, ts_avg, fsds_avg, plonl )
!-----------------------------------------------------------------------
! 	... set surface fluxes or make adjustment to field due to
!           surface flux. dry deposition routines are called from
!           here because they work by setting surface fluxes.
!-----------------------------------------------------------------------

      use chem_mods,    only : drydep_cnt
      use mo_grid,      only : plev, plevp, pcnst
      use mo_constants, only : latwts, pi, rearth
      use mo_control,   only : delt, xactive_drydep, dyn_soilw, xactive_emissions
      use mo_mpi,       only : base_lat
      use mass_diags,   only : hsa_fac
      use mo_histout,   only : outfld, hfile, moz_file_cnt
      use mo_srf_emis,  only : srf_emis_set
      use mo_drydep,    only : set_soilw
      use mo_drydep,    only : drydep
      use mo_photo,     only : diurnal_geom
      use mo_xemis,     only : isop_ndx, mterps_ndx, no_ndx
      use mo_xemis,     only : has_xemis_isop, has_xemis_no, has_xemis_mterps
      use mo_bvoc,      only : megan_iso_emis
      use mo_bvoc,      only : megan_mterps_emis
      use mo_bvoc,      only : megan_cnt, megan_species
      use mo_soil_no,   only : soil_no_emissions
      use eslookup,     only : aqsat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        plonl, &
        lat, &                 ! latitude index
        ip, &                  ! longitude tile index
        ncdate, &              ! date at begin of current timestep in yymmdd format
        ncsec, &               ! seconds of current timestep
        ioro(plonl)            ! orography 

      real, intent(in) :: &
        dtime, &               ! timestep size (s)
        calday, &              ! julian day + fraction (greenwich)
        pmid(plonl,plev), &    ! pressure at layer midpoints
        rpdel(plonl,plev), &   ! 1/pdel
        tv(plonl,plev), &      ! virtual temperature
        ts(plonl), &           ! surface temperature (K)
        ts_avg(plonl), &       ! average surface temperature (K)
        snow(plonl), &         ! snow height (m)
        fsds(plonl), &         ! surface direct radiation (w/m^2)
        fsds_avg(plonl), &     ! average surface direct radiation (w/m^2)
        precip(plonl), &       ! surface precipitation (m/s)
        wind_speed(plonl), &   ! surface wind speed (m/s)
        spec_hum(plonl), &     ! surface specific humidity (kg/kg)
        p_srf(plonl), &        ! surface pressure (Pa)
        air_temp(plonl), &     ! lowest level temperature (K)
        zi(plonl,plevp), &     ! potential height above surface at interfaces
        shflx(plonl)           ! surface water vapor flux (kg/m2/s)
      real, intent(inout) :: &
        soilw(plonl)           ! soil moisture fraction

      real, intent(inout) :: &
        as(plonl,plev,pcnst)   ! advected species

      real, intent(out) :: &
        sflx(plonl,pcnst)      ! surface flux for advected species

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer, parameter :: inst = 1
      integer, parameter :: avrg = 2

      integer :: i, m, n, base, file
      real    :: sunon                ! sunrise longitude (radians)
      real    :: sunoff               ! sunset longitude (radians)
      real    :: zen_angle(plonl)     ! zenith angle as function of longitude (radians)
      real    :: loc_angle(plonl)     ! time angle as function of longitude (radians)
      real    :: depvel(plonl,pcnst)  ! deposition velocity ( cm/s )
      real    :: dflx(plonl,pcnst)    ! deposition flux ( kg/m^2/s )
      real    :: dep_flx(plonl)       ! deposition flux ( kg/m^2/s )
      real    :: satv(plonl)          ! wrk array for relative humidity
      real    :: satq(plonl)          ! wrk array for relative humidity
      real    :: rh(plonl,1)          ! relative humidity
      logical :: polar_night          ! wrk flag for diurnal_geom
      logical :: polar_day            ! wrk flag for diurnal_geom
      character(len=32) :: fldname
      character(len=3)  :: num

      do m = 1,pcnst
         sflx(:,m) = 0.
         dflx(:,m) = 0.
      end do

!-----------------------------------------------------------------------
!	... diurnal geometry
!-----------------------------------------------------------------------
      call diurnal_geom( ip, lat, calday, polar_night, polar_day, &
                         sunon, sunoff, loc_angle, zen_angle, plonl )
!-----------------------------------------------------------------------
!	... the surface emissions
!-----------------------------------------------------------------------
      call srf_emis_set( lat, ip, ncdate, ncsec, sflx, &
                         ioro, loc_angle, polar_night, polar_day, sunon, &
                         sunoff, plonl )
!-----------------------------------------------------------------------
!	... megan interactive surface emissions
!-----------------------------------------------------------------------
      if( megan_cnt > 0 ) then
species_loop : &
         do m = 1,megan_cnt
            select case( megan_species(m) )
            case( 'ISOP' )
               call megan_iso_emis( lat, ip, calday, ts, fsds, &
                                    sflx(1,isop_ndx), ts_avg, fsds_avg, plonl, zen_angle )
            case( 'C10H16' )
               call megan_mterps_emis( lat, ip, calday, ts, sflx(1,mterps_ndx), plonl )
            end select
         end do species_loop
      end if

      if( xactive_emissions ) then
         if( has_xemis_no ) then
            call soil_no_emissions( lat, ip, calday, dtime, ts, &
                                    precip, shflx, sflx(1,no_ndx), plonl )
         end if
      end if

!-----------------------------------------------------------------------
!	... surface emissions to history files
!-----------------------------------------------------------------------
      do file = 1,moz_file_cnt
         do i = inst,avrg
            if( i == inst .and. .not. hfile(file)%wrhstts ) then
               cycle
            end if
            if( hfile(file)%histout_cnt(3,i) > 0 ) then
               base = hfile(file)%histout_ind(3,i) - 1
               do m = 1,hfile(file)%histout_cnt(3,i)
                  if( i == inst ) then
                     n       = hfile(file)%inst_map(base+m)
                     fldname = hfile(file)%hist_inst(base+m)
                  else if( i == avrg ) then
                     n       = hfile(file)%timav_map(base+m)
                     fldname = hfile(file)%hist_timav(base+m)
                  end if
                  call outfld( fldname, sflx(1,n), plonl, ip, lat, file )
               end do
            end if
            if( hfile(file)%histout_cnt(20,avrg) > 0 ) then
               base = hfile(file)%histout_ind(20,avrg) - 1
               do m = 1,hfile(file)%histout_cnt(20,avrg)
                  n       = mod( hfile(file)%timav_map(base+m),10000 )
                  fldname = hfile(file)%hist_timav(base+m)
                  if( index( fldname, '_emis' ) > 0 ) then
                     call outfld( fldname, sflx(1,n), plonl, ip, lat, file )
                  end if
               end do
            end if
         end do
      end do
!-----------------------------------------------------------------------
!	... dry deposition
!-----------------------------------------------------------------------
      if( drydep_cnt > 0 ) then
         if( .not. xactive_drydep ) then
            call drydep( lat, ip, calday, ts, zen_angle, &
                         depvel, dflx, rair, as, pmid(:,plev), &
                         tv(:,plev), plonl )
         else
            if( .not. dyn_soilw .and. table_soilw ) then
               call set_soilw( lat, ip, soilw, plonl )
            end if
!-----------------------------------------------------------------
!	... compute the relative humidity
!-----------------------------------------------------------------
            call aqsat( ts, p_srf, satv, satq, plonl, &
                        plonl, 1, 1, 1 )
            rh(:,1) = spec_hum(:) / satq(:)
            rh(:,1) = max( 0.,min( 1.,rh(:,1) ) )
            call drydep( lat, ip, ncdate, ts, p_srf, &
                         wind_speed, spec_hum, air_temp, pmid(:,plev), precip, &
                         snow, fsds, depvel, dflx, as, &
                         tv(:,plev), soilw, rh, plonl )
         end if
      end if
!-----------------------------------------------------------------------
!	... dry deposition velocity to history files
!-----------------------------------------------------------------------
      do file = 1,moz_file_cnt
         do i = inst,avrg
            if( i == inst .and. .not. hfile(file)%wrhstts ) then
               cycle
            end if
            if( hfile(file)%histout_cnt(4,i) > 0 ) then
               base = hfile(file)%histout_ind(4,i) - 1
               do m = 1,hfile(file)%histout_cnt(4,i)
                  if( i == inst ) then
                     n       = hfile(file)%inst_map(base+m)
                     fldname = hfile(file)%hist_inst(base+m)
                  else if( i == avrg ) then
                     n       = hfile(file)%timav_map(base+m)
                     fldname = hfile(file)%hist_timav(base+m)
                  end if
                  call outfld( fldname, depvel(1,n), plonl, ip, lat, file )
               end do
            end if
         end do
      end do
!-----------------------------------------------------------------------
!	... dry deposition flux to history files
!-----------------------------------------------------------------------
      do file = 1,moz_file_cnt
         do i = inst,avrg
            if( i == inst .and. .not. hfile(file)%wrhstts ) then
               cycle
            end if
            if( hfile(file)%histout_cnt(17,i) > 0 ) then
               base = hfile(file)%histout_ind(17,i) - 1
               do m = 1,hfile(file)%histout_cnt(17,i)
                  if( i == inst ) then
                     n       = hfile(file)%inst_map(base+m)
                     fldname = hfile(file)%hist_inst(base+m)
                  else if( i == avrg ) then
                     n       = hfile(file)%timav_map(base+m)
                     fldname = hfile(file)%hist_timav(base+m)
                  end if
                  dep_flx(:) = hsa_fac*latwts(base_lat+lat)*delt*dflx(:,n)
                  call outfld( fldname, dep_flx, plonl, ip, lat, file )
               end do
            end if
         end do
      end do

!-----------------------------------------------------------------------
!	... form surface flux
!-----------------------------------------------------------------------
      do m = 1,pcnst
         sflx(:,m) = sflx(:,m) - dflx(:,m)
      end do

      end subroutine sflxdr

      end module mo_surface
