
      module mo_ub_vals
!---------------------------------------------------------------
!	... variables for the upper boundary values
!---------------------------------------------------------------

      implicit none

      private
      public  :: ub_inti
      public  :: set_ub_vals
      public  :: set_ub_h2o

      save

      real, parameter :: taurelax = 864000.         ! 10 days
      integer :: gndx = 0
      integer :: table_nox_ndx = -1
      integer :: table_h2o_ndx = -1
      integer :: table_ox_ndx  = -1
      integer :: no_ndx
      integer :: no2_ndx
      integer :: h2o_ndx
      integer :: ox_ndx
      integer :: o3s_ndx
      integer :: o3inert_ndx
      integer :: synoz_ndx
      integer :: o3rad_ndx
      integer :: jlim_in(2)
      real    :: facrelax
      real    :: days(12)
      real, allocatable       :: ub_plevs(:)         ! table midpoint pressure levels (Pa)
      real, allocatable       :: ub_plevse(:)        ! table edge pressure levels (Pa)
      integer                 :: ub_nlevs            ! # of levs in ubc file
      integer                 :: ub_nlat             ! # of lats in ubc file
      integer                 :: ub_nspecies         ! # of species in ubc file
      integer                 :: ub_nmonth           ! # of months in ubc file
      real, allocatable       :: mr_ub(:,:,:,:)      ! vmr
      integer, allocatable    :: map(:,:)            ! species indices for ubc species
      logical :: has_no_ubc
      logical :: has_no2_ubc
      logical :: has_nox_ubc

      contains

      subroutine ub_inti( platl, ncfile, lpath, mspath, dtime )
!------------------------------------------------------------------
!	... initialize upper boundary values
!------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use mo_constants,  only : phi, lam, d2r
      use mo_regrider,   only : regrid_inti, regrid_1d, regrid_lat_limits
      use chem_mods,     only : fbc_cnt, fubc_lst, fubc_alias_lst
      use mo_grid,       only : plong => plon, platg => plat, pcnstm1
      use m_tracname,    only : tracnam
      use mo_chem_utls,  only : get_spc_ndx, get_inv_ndx, has_fixed_ubc
      use mo_calendar,   only : caldayr
      use mo_mpi,        only : base_lat, masternode, lastnode

      implicit none

!------------------------------------------------------------------
!	... dummy args
!------------------------------------------------------------------
      integer, intent(in)          :: platl
      integer, intent(in)          :: dtime                ! model time step (s)
      character(len=*), intent(in) :: ncfile               ! file name of netcdf file containing data
      character(len=*), intent(in) :: lpath                ! local pathname to ncfile
      character(len=*), intent(in) :: mspath               ! mass store pathname to ncfile

!------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------
      real, parameter :: mb2pa = 100.

      integer :: jl, ju, m
      integer :: i, j, nchar
      integer :: spcno, lev, month, ierr
      integer :: ncid, vid, ndims
      integer :: dimid_lat, dimid_lev, dimid_species, dimid_month
      integer :: dimid(4)
      integer :: start(4)
      integer :: count(4)
      integer :: dates(12) = (/ 116, 214, 316, 415,  516,  615, &
                                716, 816, 915, 1016, 1115, 1216 /)
      real, allocatable :: mr_ub_in(:,:,:,:)
      real, allocatable :: lat(:)
      character(len=80) :: attribute
      character(len=16) :: wrk_name
      character(len=25), allocatable :: ub_species_names(:)
      logical :: die

!-----------------------------------------------------------------------
!       ... get species indicies
!-----------------------------------------------------------------------
      no_ndx      = get_spc_ndx( 'NO' )
      no2_ndx     = get_spc_ndx( 'NO2' )
      has_no_ubc  = has_fixed_ubc( 'NO' )
      has_no2_ubc = has_fixed_ubc( 'NO2' )
      has_nox_ubc = has_no_ubc .or. has_no2_ubc
      ox_ndx      = get_spc_ndx( 'OX' )
      if( ox_ndx < 1 ) then
         ox_ndx = get_spc_ndx( 'O3' )
      end if
      o3s_ndx     = get_spc_ndx( 'O3S' )
      o3inert_ndx = get_spc_ndx( 'O3INERT' )
      o3rad_ndx   = get_spc_ndx( 'O3RAD' )
      synoz_ndx   = get_spc_ndx( 'SYNOZ' )
      h2o_ndx     = get_spc_ndx( 'H2O' )
      if( h2o_ndx < 0 ) then
         h2o_ndx  = get_inv_ndx( 'H2O' )
      end if

!-----------------------------------------------------------------------
!       ... open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )
!-----------------------------------------------------------------------
!       ... get latitude
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'ub_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, ub_nlat ), &
                         'ub_inti: failed to get length of dimension lat' )
      allocate( lat(ub_nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'ub_inti: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'ub_inti: failed to read variable lat' )
      lat(:ub_nlat) = lat(:ub_nlat) * d2r

!-----------------------------------------------------------------------
!       ... get grid interp limits
!-----------------------------------------------------------------------
      gndx = regrid_inti( ub_nlat, platg, &
                          plong, plong, &
                          lam,  lam, &
                          lat,  phi, &
                          0, platl, &
                          do_lons=.false.,do_lats=.true. )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: failed to deallocate lat; ierr = ',ierr
         call endrun
      end if
      jl = base_lat + 1
      ju = base_lat + platl
      jlim_in = regrid_lat_limits( gndx)
#ifdef DEBUG
	write(*,*) 'lat_in='
	write(*,'(10f7.1)') lat(jlim_in(1):jlim_in(2))/d2r
	write(*,*) 'lat_out='
	write(*,'(10f7.1)') phi(jl:ju)/d2r
#endif
      deallocate( lat, stat=ierr )

      write(*,'(''ub_inti: gndx='',i2,'', grid limits = '',2i4,'', jl,ju='',2i4)') &
         gndx,jlim_in,jl,ju

!-----------------------------------------------------------------------
!       ... get vertical coordinate (if necessary, convert units to pa)
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lev', dimid_lev ), &
                         'ub_inti: failed to find dimension lev' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lev, ub_nlevs ), &
                         'ub_inti: failed to get length of dimension lev' )
      allocate( ub_plevs(ub_nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: ub_plevs allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lev', vid ), &
                         'ub_inti: getting lev id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, ub_plevs ), &
                         'ub_inti: getting ub_plevs' )
      attribute(:) = ' '
      ierr = nf_get_att_text( ncid, vid, 'units', attribute )
      if( ierr == nf_noerr )then
         if( trim(attribute) == 'mb' .or. trim(attribute) == 'hpa' )then
            write(*,*) 'ub_inti: units for lev = ',trim(attribute),'... converting to pa'
            ub_plevs(:) = mb2pa * ub_plevs(:)
         else if( trim(attribute) /= 'pa' .and. trim(attribute) /= 'pa' )then
            write(*,*) 'ub_inti: unknown units for lev, units=*',trim(attribute),'*'
            write(*,*) 'ub_inti: ',attribute=='mb',trim(attribute)=='mb',attribute(1:2)=='mb'
            call endrun
         end if
      else
            write(*,*) 'ub_inti: warning! units attribute for lev missing, assuming mb'
            ub_plevs(:) = mb2pa * ub_plevs(:)
      end if
!-----------------------------------------------------------------------
!       ... get time and species dimensions
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'month', dimid_month ), &
                         'ub_inti: failed to find dimension month' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_month, ub_nmonth ), &
                         'ub_inti: failed to get length of dimension month' )
      if( ub_nmonth /= 12 )then
         write(*,*) 'ub_inti: error! number of months = ',ub_nmonth,', expecting 12'
         call endrun
      end if
      call handle_ncerr( nf_inq_dimid( ncid, 'species', dimid_species ), &
                         'ub_inti: failed to find dimension species' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_species, ub_nspecies ), &
                         'ub_inti: failed to get length of dimension species' )

!------------------------------------------------------------------
!	... allocate arrays
!------------------------------------------------------------------
      allocate( mr_ub_in(jlim_in(1):jlim_in(2),ub_nspecies,ub_nmonth,ub_nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: mr_ub_in allocation error = ',ierr
         call endrun
      end if
      allocate( mr_ub(platl,ub_nspecies,ub_nmonth,ub_nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: mr_ub allocation error = ',ierr
         call endrun
      end if

!------------------------------------------------------------------
!	... read in the species names
!------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'specname', vid ), &
                         'ub_inti: getting specname id' )
      call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                         'ub_inti: getting number of dimensions for specname' )
      call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                         'ub_inti: getting dimensions for vmr' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid(1), nchar ), &
                         'ub_inti: getting dimension length' )
      allocate( ub_species_names(ub_nspecies), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: ub_species_names allocation error = ',ierr
         call endrun
      end if
      allocate( map(fbc_cnt(2),2), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ub_inti: map allocation error = ',ierr
         call endrun
      end if
      do i = 1,ub_nspecies
         start(:2) = (/ 1, i /)
         count(:2) = (/ nchar, 1 /)
         ub_species_names(i)(:) = ' '
         call handle_ncerr( nf_get_vara_text( ncid, vid, start(:2), count(:2), ub_species_names(i) ), &
                            'ub_inti: getting species names' )
         if( trim(ub_species_names(i)) == 'NOX' ) then
            table_nox_ndx = i
         else if( trim(ub_species_names(i)) == 'H2O' ) then
            table_h2o_ndx = i
         else if( trim(ub_species_names(i)) == 'OX' ) then
            table_ox_ndx = i
         end if
      end do
!------------------------------------------------------------------
!	... map ubc solution species to ubc file
!------------------------------------------------------------------
      map(:,:) = 0
table_loop : &
      do i = 1,fbc_cnt(2)
         map(i,1) = get_spc_ndx( trim(fubc_lst(i)) )
         if( fubc_alias_lst(i) == ' ' ) then
            wrk_name = trim( fubc_lst(i) )
         else
            wrk_name = trim( fubc_alias_lst(i) )
         end if
         do j = 1,ub_nspecies
            if( trim(ub_species_names(j)) == trim(wrk_name) ) then
               map(i,2) = j
               exit
            end if
         end do
         if( map(i,2) == 0 ) then
            if( fubc_lst(i) /= 'NO' .and. fubc_lst(i) /= 'NO2' ) then
               write(*,*) 'ub_inti: ubc species ',trim(fubc_lst(i)), ' not found'
               call endrun
            end if
         end if
      end do table_loop

      if( masternode ) then
!------------------------------------------------------------------
!	... check for ox, nox, h2o table entries
!------------------------------------------------------------------
         die = .false.
         if( has_nox_ubc .and. table_nox_ndx < 1 ) then
            write(*,*) 'ub_inti: simulation has nox ubc but nox not in ubc table'
            die = .true.
         end if
         if( h2o_ndx > 0 .and. table_h2o_ndx < 1 ) then
            write(*,*) 'ub_inti: simulation has h2o ubc but h2o not in ubc table'
            die = .true.
         end if
         if( die ) then
            call endrun
         end if
         write(*,*) ' '
         if( fbc_cnt(2) > 0 ) then
            write(*,*) 'Species with specified upper boundary values'
            do m = 1,fbc_cnt(2)
               if( fubc_alias_lst(m) == ' ' ) then
                  write(*,*) trim( fubc_lst(m) )
               else
                  write(*,*) trim( fubc_lst(m) ) // ' -> ' // trim( fubc_alias_lst(m) )
               end if
            end do
         else
            write(*,*) 'There are no species with specified upper boundary values'
         end if
         write(*,*) ' '
         write(*,*) 'ub_inti: nox flags'
         write(*,*) has_no_ubc, has_no2_ubc, has_nox_ubc
         write(*,*) ' '
         write(*,*) 'ub_inti: table indicies for ox,nox,h2o'
         write(*,*) table_ox_ndx, table_nox_ndx, table_h2o_ndx
         if( fbc_cnt(2) > 0 ) then
            write(*,*) 'ub_inti: map'
            do i = 1,2
               write(*,'(10i5)') map(:,i)
            end do
         else
            write(*,*) 'ub_inti: no solution species with ubc'
         end if
         write(*,*) ' '
      end if

!------------------------------------------------------------------
!	... check dimensions for vmr variable
!------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'vmr', vid ), &
                         'ub_inti: getting vmr id' )
      call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                         'ub_inti: getting number of dimensions for vmr' )
      if( ndims /= 4 ) then
         write(*,*) 'ub_inti: error! variable vmr has ndims = ',ndims,', expecting 4'
         call endrun
      end if
      call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                         'ub_inti: getting dimensions for vmr' )
      if( dimid(1) /= dimid_lat .or. dimid(2) /= dimid_species .or. &
          dimid(3) /= dimid_month .or. dimid(4) /= dimid_lev )then
         write(*,*) 'ub_inti: error! dimensions in wrong order for variable vmr,'// &
                    'expecting (lat,species,month,lev)'
         call endrun
      end if

!------------------------------------------------------------------
!	... read in the ub mixing ratio values
!------------------------------------------------------------------
      start = (/ jlim_in(1), 1, 1, 1 /)
      count = (/ jlim_in(2) - jlim_in(1) + 1, ub_nspecies, 12, ub_nlevs /)

      call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, mr_ub_in ), &
                         'ub_inti: getting vmr' )
      call handle_ncerr( nf_close( ncid ), 'ub_inti: failed to close file ' // trim(ncfile) )
!--------------------------------------------------------------------
!	... regrid
!--------------------------------------------------------------------
      do lev = 1,ub_nlevs
         do month = 1,ub_nmonth
	    do spcno = 1,ub_nspecies
	       call regrid_1d( mr_ub_in(:,spcno,month,lev), mr_ub(:,spcno,month,lev), &
                               gndx, do_lat=.true., to_lat_min=jl, to_lat_max=ju )
#ifdef DEBUG
	       if( lev == 25 .and. month == 1 .and. spcno == 1 ) then
	          write(*,*) 'mr_ub_in='
		  write(*,'(10f7.1)') mr_ub_in(:,spcno,month,lev)*1.e9
		  write(*,*) 'mr_ub='
		  write(*,'(10f7.1)') mr_ub(:,spcno,month,lev)*1.e9
	       end if
#endif
	       if( masternode ) then
	          mr_ub(1,spcno,month,lev) = mr_ub(2,spcno,month,lev)
	       end if
	       if( lastnode ) then
	          mr_ub(platl,spcno,month,lev) = mr_ub(platl-1,spcno,month,lev)
	       end if
	    end do
	 end do
      end do

!--------------------------------------------------------
!	... initialize the monthly day of year times
!--------------------------------------------------------
      do month = 1,12
         days(month) = caldayr( dates(month), 0 )
      end do

!--------------------------------------------------------
!   	... set up the relaxation for lower stratosphere
!--------------------------------------------------------
! 	... taurelax = relaxation timescale (in sec)
!           facrelax = fractional relaxation towards ubc
!            1 => use ubc
!            0 => ignore ubc, use model concentrations
!--------------------------------------------------------
      facrelax = 1. - exp( -real(dtime)/taurelax )

!--------------------------------------------------------
! 	... setup conserving interp for OX
!--------------------------------------------------------
      if( table_ox_ndx > 0 ) then
         allocate( ub_plevse(ub_nlevs-1), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'ub_inti: ub_plevse allocation error = ',ierr
            call endrun
         end if
         ub_plevse(1:ub_nlevs-1) = .5*(ub_plevs(1:ub_nlevs-1) + ub_plevs(2:ub_nlevs))
      end if

      end subroutine ub_inti

      subroutine set_ub_vals( lat, ip, vmr, pmid, pint, &
                              zmid, tfld, ltrop, calday, plonl )
!--------------------------------------------------------------------
!	... set the upper boundary values for :
!           ox, nox, hno3, ch4, co, n2o, n2o5 & stratospheric o3
!--------------------------------------------------------------------

      use mo_control,    only : delt
      use mo_grid,       only : plev, plevp, pcnstm1
      use chem_mods,     only : fbc_cnt

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: lat               ! lat index
      integer, intent(in) :: plonl             ! lon tile dim
      integer, intent(in) :: ip                ! longitude tile index
      integer, intent(in) :: ltrop(plonl)      ! tropopause vertical index
      real, intent(in)    :: calday            ! day of year including fraction
      real, intent(in)    :: tfld(plonl,plev)  ! (K)
      real, intent(in)    :: zmid(plonl,plev)  ! midpoint height (km)
      real, intent(in)    :: pmid(plonl,plev)  ! midpoint pressure (Pa)
      real, intent(in)    :: pint(plonl,plevp) ! interface pressure (Pa)
      real, intent(inout) :: vmr(plonl,plev,pcnstm1) ! species concentrations (mol/mol)

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer, parameter :: zlower = plev
      real, parameter    :: synoz_thres = 100.e-9      ! synoz threshold
      real, parameter    :: o3rad_relax = .5*86400.    ! 1/2 day relaxation constant
      real, parameter    :: synoz_relax = 2.*86400.    ! 2 day relaxation constant

      integer  ::  m, last, next, i, k, k1, km
      integer  ::  astat
      integer  ::  spc_ndx, tab_ndx
      integer  ::  kmax(plonl)
      integer  ::  levrelax
      integer  ::  kl(plonl,zlower)
      integer  ::  ku(plonl,zlower)
      real     ::  vmrrelax
      real     ::  fac_relax
      real     ::  pinterp
      real     ::  nox_ubc, xno, xno2, rno
      real     ::  dels
      real     ::  delp(plonl,zlower)
      real     ::  pint_vals(2)
      real, allocatable :: table_ox(:)
      logical  ::  found_trop


!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      if( calday < days(1) ) then
          next = 1
          last = 12
          dels = (365. + calday - days(12)) / (365. + days(1) - days(12))
      else if( calday >= days(12) ) then
          next = 1
          last = 12
          dels = (calday - days(12)) / (365. + days(1) - days(12))
      else
         do m = 11,1,-1
            if( calday >= days(m) ) then
               exit
            end if
         end do
         last = m
         next = m + 1
         dels = (calday - days(m)) / (days(m+1) - days(m))
      end if
      dels = max( min( 1.,dels ),0. )

!--------------------------------------------------------
!	... setup the pressure interpolation
!--------------------------------------------------------
      do k = 1,zlower
         do i = 1,plonl
            if( pmid(i,k) <= ub_plevs(1) ) then
               kl(i,k) = 1
               ku(i,k) = 1
               delp(i,k) = 0.
            else if( pmid(i,k) >= ub_plevs(ub_nlevs) ) then
               kl(i,k) = ub_nlevs
               ku(i,k) = ub_nlevs
               delp(i,k) = 0.
            else
              pinterp = pmid(i,k)
              do k1 = 2,ub_nlevs
                 if( pinterp <= ub_plevs(k1) ) then
                    ku(i,k) = k1
                    kl(i,k) = k1 - 1
                    delp(i,k) = log( pinterp/ub_plevs(kl(i,k)) ) &
                                / log( ub_plevs(ku(i,k))/ub_plevs(kl(i,k)) )
                    exit
                 end if
              end do
            end if
         end do
      end do

!--------------------------------------------------------
!	... find max level less than 50 mb
!           fix UB vals from top of model to this level
!--------------------------------------------------------
      do i = 1,plonl
         do k = 2,plev
           if( pmid(i,k) > 50.e2 ) then
             kmax(i) = k
             exit
           end if
         end do
      end do

!--------------------------------------------------------
!	... setup for ox conserving interp
!--------------------------------------------------------
      if( table_ox_ndx > 0 ) then
         if( any( map(:,2) == table_ox_ndx ) ) then
            allocate( table_ox(ub_nlevs-2),stat=astat )
            if( astat /= 0 ) then
               write(*,*) 'set_ub_vals: table_ox allocation error = ',astat
               call endrun
            end if
#ifdef UB_DEBUG
            write(*,*) ' '
            write(*,*) 'set_ub_vals: ub_nlevs = ',ub_nlevs
            write(*,*) 'set_ub_vals: ub_plevse'
            write(*,'(1p5g15.7)') ub_plevse(:)
            write(*,*) ' '
#endif
         end if
      end if

!--------------------------------------------------------
!	... set the mixing ratios at upper boundary
!--------------------------------------------------------
species_loop : &
      do m = 1,fbc_cnt(2)
         tab_ndx = map(m,2)
ub_overwrite : &
         if( tab_ndx > 0 ) then
            spc_ndx = map(m,1)
            if( tab_ndx == table_ox_ndx ) then
               table_ox(1:ub_nlevs-2) = mr_ub(lat,tab_ndx,last,2:ub_nlevs-1) &
                                      + dels*(mr_ub(lat,tab_ndx,next,2:ub_nlevs-1) &
                                              - mr_ub(lat,tab_ndx,last,2:ub_nlevs-1))
#ifdef UB_DEBUG
               write(*,*) 'set_ub_vals: table_ox @ lat = ',lat
               write(*,'(1p5g15.7)') table_ox(:)
               write(*,*) ' '
#endif
               do i = 1,plonl
                  km = kmax(i)
#ifdef UB_DEBUG
                  write(*,*) 'set_ub_vals: pint with km = ',km
                  write(*,'(1p5g15.7)') pint(i,:km+1)
                  write(*,*) ' '
                  write(*,*) 'set_ub_vals: pmid with km = ',km
                  write(*,'(1p5g15.7)') pmid(i,:km)
                  write(*,*) ' '
#endif
                  call rebin( ub_nlevs-2, km, ub_plevse, pint(i,:km+1), table_ox, vmr(i,:km,spc_ndx) )
#ifdef UB_DEBUG
                  write(*,*) 'set_ub_vals: ub o3 @ lat = ',lat
                  write(*,'(1p5g15.7)') vmr(i,:km,spc_ndx)
#endif
               end do
               cycle species_loop
            end if
            do i = 1,plonl
                do k = 1,kmax(i)
                   pint_vals(1) = mr_ub(lat,tab_ndx,last,kl(i,k)) &
                                 + delp(i,k) &
                                * (mr_ub(lat,tab_ndx,last,ku(i,k)) &
                                   - mr_ub(lat,tab_ndx,last,kl(i,k)))
                   pint_vals(2) = mr_ub(lat,tab_ndx,next,kl(i,k)) &
                                 + delp(i,k) &
                                * (mr_ub(lat,tab_ndx,next,ku(i,k)) &
                                   - mr_ub(lat,tab_ndx,next,kl(i,k)))
                   vmr(i,k,spc_ndx) = pint_vals(1) &
                                    + dels * (pint_vals(2) - pint_vals(1))
                end do
            end do
         end if ub_overwrite
      end do species_loop
      if( allocated( table_ox ) ) then
        deallocate( table_ox )
      endif
!--------------------------------------------------------
!	... check for nox species
!--------------------------------------------------------
      if( has_nox_ubc ) then
         do i = 1,plonl
             do k = 1,kmax(i)
                pint_vals(1) = mr_ub(lat,table_nox_ndx,last,kl(i,k)) &
                              + delp(i,k) &
                             * (mr_ub(lat,table_nox_ndx,last,ku(i,k)) &
                                - mr_ub(lat,table_nox_ndx,last,kl(i,k)))
                pint_vals(2) = mr_ub(lat,table_nox_ndx,next,kl(i,k)) &
                              + delp(i,k) &
                             * (mr_ub(lat,table_nox_ndx,next,ku(i,k)) &
                                - mr_ub(lat,table_nox_ndx,next,kl(i,k)))
                nox_ubc = pint_vals(1) + dels * (pint_vals(2) - pint_vals(1))
                if( has_no_ubc ) then
                   xno  = vmr(i,k,no_ndx)
                else
                   xno  = 0.
                end if
                if( has_no2_ubc ) then
                   xno2 = vmr(i,k,no2_ndx)
                else
                   xno2 = 0.
                end if
                rno  = xno / (xno + xno2)
                if( has_no_ubc ) then
                   vmr(i,k,no_ndx)  = rno * nox_ubc
                end if
                if( has_no2_ubc ) then
                   vmr(i,k,no2_ndx) = (1. - rno) * nox_ubc
                end if
             end do
         end do
      end if

!---------------------------------------------------------------
!	... now temporally "relax" ubc species to ubc
!           values from the tropopause to ~50 mb
!---------------------------------------------------------------
long_loop2 : &
      do i = 1,plonl
!--------------------------------------------------------
! 	... relax lower stratosphere to extended ubc
!           check to make sure ubc is not being imposed too low
!           levrelax = lowest model level (highest pressure)
!                      in which to relax to ubc
!--------------------------------------------------------
        levrelax = ltrop(i)
        do while( pmid(i,levrelax) > ub_plevs(ub_nlevs) )
           levrelax = levrelax - 1
        end do
#ifdef DEBUG
        if( levrelax /= ltrop(i) ) then
           write(*,*) 'warning -- raised ubc: ',lat,i,
                      ltrop(i)-1,nint(pmid(i,ltrop(i)-1)/mb2pa),'mb -->',
                      levrelax,nint(pmid(i,levrelax)/mb2pa),'mb'
        end if
#endif
        do m = 1,fbc_cnt(2)
           tab_ndx = map(m,2)
           if( tab_ndx > 0 ) then
              spc_ndx = map(m,1)
level_loop2 : do k = kmax(i)+1,levrelax
                 pint_vals(1) = mr_ub(lat,tab_ndx,last,kl(i,k)) &
                                + delp(i,k) &
                                  * (mr_ub(lat,tab_ndx,last,ku(i,k)) &
                                     - mr_ub(lat,tab_ndx,last,kl(i,k)))
                 pint_vals(2) = mr_ub(lat,tab_ndx,next,kl(i,k)) &
                                + delp(i,k) &
                                 * (mr_ub(lat,tab_ndx,next,ku(i,k)) &
                                    - mr_ub(lat,tab_ndx,next,kl(i,k)))
                 vmrrelax = pint_vals(1) + dels * (pint_vals(2) - pint_vals(1))
                 vmr(i,k,spc_ndx) = vmr(i,k,spc_ndx) &
                                  + (vmrrelax - vmr(i,k,spc_ndx)) * facrelax
              end do level_loop2
           end if
        end do
        if( has_nox_ubc ) then
           do k = kmax(i)+1,levrelax
              if( has_no_ubc ) then
                 xno  = vmr(i,k,no_ndx)
              else
                 xno  = 0.
              end if
              if( has_no2_ubc ) then
                 xno2 = vmr(i,k,no2_ndx)
              else
                 xno2 = 0.
              end if
              rno     = xno / (xno + xno2)
              nox_ubc = xno + xno2
              pint_vals(1) = mr_ub(lat,table_nox_ndx,last,kl(i,k)) &
                             + delp(i,k) &
                               * (mr_ub(lat,table_nox_ndx,last,ku(i,k)) &
                                  - mr_ub(lat,table_nox_ndx,last,kl(i,k)))
              pint_vals(2) = mr_ub(lat,table_nox_ndx,next,kl(i,k)) &
                             + delp(i,k) &
                              * (mr_ub(lat,table_nox_ndx,next,ku(i,k)) &
                                 - mr_ub(lat,table_nox_ndx,next,kl(i,k)))
              vmrrelax = pint_vals(1) + dels * (pint_vals(2) - pint_vals(1))
              nox_ubc  = nox_ubc + (vmrrelax - nox_ubc) * facrelax
              if( has_no_ubc ) then
                 vmr(i,k,no_ndx)  = rno * nox_ubc
              end if
              if( has_no2_ubc ) then
                 vmr(i,k,no2_ndx) = (1. - rno) * nox_ubc
              end if
           end do
        end if

has_synoz : &
        if( synoz_ndx > 0 ) then
!--------------------------------------------------------
! 	... special assignments if synoz is present
!           update ox, o3s, o3inert in the stratosphere
!--------------------------------------------------------
           if( ox_ndx > 0 ) then
              do k = 1,levrelax
                 if( vmr(i,k,synoz_ndx) >= synoz_thres ) then
                    vmr(i,k,ox_ndx) = vmr(i,k,synoz_ndx)
                 end if
              end do
           end if
           if( o3s_ndx > 0 ) then
              do k = 1,levrelax
                 if( vmr(i,k,synoz_ndx) >= synoz_thres ) then
                    vmr(i,k,o3s_ndx) = vmr(i,k,synoz_ndx)
                 end if
              end do
           end if
           if( o3rad_ndx > 0 .and. o3inert_ndx > 0 ) then
              vmr(i,:ltrop(i),o3inert_ndx) = vmr(i,:ltrop(i),o3rad_ndx)
           end if
!--------------------------------------------------------
! 	... O3RAD is relaxed to climatology in the stratosphere
!           (done above) and OX in the troposphere
!--------------------------------------------------------
           if( o3rad_ndx > 0 .and. ox_ndx > 0 ) then
              fac_relax = 1. - exp( -real(delt) / o3rad_relax )
              do k = levrelax+1,plev
                 vmr(i,k,o3rad_ndx) = vmr(i,k,o3rad_ndx) &
                                 + (vmr(i,k,ox_ndx) - vmr(i,k,o3rad_ndx)) * fac_relax
              end do
           end if
!--------------------------------------------------------
! 	... relax synoz to 25 ppbv in lower troposphere
!           (p > 500 hPa) with an e-fold time of 2 days
!           (Mc Linden et al., JGR, p14,660, 2000)
!--------------------------------------------------------
           fac_relax = 1. - exp( -real(delt) / synoz_relax )
           vmrrelax = 25.e-9
           do k = levrelax+2,plev
              if( pmid(i,k) >= 50000. ) then
                 vmr(i,k,synoz_ndx) = vmr(i,k,synoz_ndx) &
                                      + (vmrrelax - vmr(i,k,synoz_ndx)) * fac_relax
              end if
           end do
        else has_synoz
!--------------------------------------------------------
!       ... set O3S and O3INERT to OX when no synoz
!--------------------------------------------------------
           if( ox_ndx > 0 ) then
             if( o3s_ndx > 0 ) then
                vmr(i,:ltrop(i),o3s_ndx)     = vmr(i,:ltrop(i),ox_ndx)
             end if
             if( o3inert_ndx > 0 ) then
                vmr(i,:ltrop(i),o3inert_ndx) = vmr(i,:ltrop(i),ox_ndx)
             end if
          end if
        end if has_synoz
      end do long_loop2

      end subroutine set_ub_vals

      subroutine set_ub_h2o( lat, ip, h2o, pmid, zmid, &
                             tfld, ltrop, calday, plonl )
!--------------------------------------------------------------------
!	... set the h2o upper boundary values
!--------------------------------------------------------------------

      use mo_grid,       only : plev, pcnstm1

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: lat               ! lat index
      integer, intent(in) :: plonl             ! lon tile dim
      integer, intent(in) :: ip                ! longitude tile index
      integer, intent(in) :: ltrop(plonl)      ! tropopause vertical index
      real, intent(in)    :: calday            ! day of year including fraction
      real, intent(in)    :: tfld(plonl,plev)  ! (K)
      real, intent(in)    :: zmid(plonl,plev)  ! midpoint height (km)
      real, intent(in)    :: pmid(plonl,plev)  ! midpoint pressure (Pa)
      real, intent(inout) :: h2o(plonl,plev)   ! h2o concentration (mol/mol)

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer, parameter :: zlower = plev

      integer  ::  m, last, next, i, k, k1
      integer  ::  kmax(plonl)
      integer  ::  levrelax
      integer  ::  kl(plonl,zlower)
      integer  ::  ku(plonl,zlower)
      real     ::  vmrrelax
      real     ::  fac_relax
      real     ::  pinterp
      real     ::  dels
      real     ::  delp(plonl,zlower)
      real     ::  pint_vals(2)
      logical  ::  found_trop

h2o_overwrite : &
      if( h2o_ndx > 0 .and. table_h2o_ndx > 0 ) then
!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
         if( calday < days(1) ) then
             next = 1
             last = 12
             dels = (365. + calday - days(12)) / (365. + days(1) - days(12))
         else if( calday >= days(12) ) then
             next = 1
             last = 12
             dels = (calday - days(12)) / (365. + days(1) - days(12))
         else
            do m = 11,1,-1
               if( calday >= days(m) ) then
                  exit
               end if
            end do
            last = m
            next = m + 1
            dels = (calday - days(m)) / (days(m+1) - days(m))
         end if
         dels = max( min( 1.,dels ),0. )

!--------------------------------------------------------
!	... setup the pressure interpolation
!--------------------------------------------------------
         do k = 1,zlower
            do i = 1,plonl
               if( pmid(i,k) <= ub_plevs(1) ) then
                  kl(i,k) = 1
                  ku(i,k) = 1
                  delp(i,k) = 0.
               else if( pmid(i,k) >= ub_plevs(ub_nlevs) ) then
                  kl(i,k) = ub_nlevs
                  ku(i,k) = ub_nlevs
                  delp(i,k) = 0.
               else
                 pinterp = pmid(i,k)
                 do k1 = 2,ub_nlevs
                    if( pinterp <= ub_plevs(k1) ) then
                       ku(i,k) = k1
                       kl(i,k) = k1 - 1
                       delp(i,k) = log( pinterp/ub_plevs(kl(i,k)) ) &
                                   / log( ub_plevs(ku(i,k))/ub_plevs(kl(i,k)) )
                       exit
                    end if
                 end do
               end if
            end do
         end do

!--------------------------------------------------------
!	... Find max level less than 50 mb
!           fix UB vals from top of model to this level
!--------------------------------------------------------
         do i = 1,plonl
            do k = 2,plev
              if( pmid(i,k) > 50.e2 ) then
                kmax(i) = k
                exit
              end if
            end do
         end do
!--------------------------------------------------------
!	... set the mixing ratio at upper boundary
!--------------------------------------------------------
         m = table_h2o_ndx
         do i = 1,plonl
            do k = 1,kmax(i)
                pint_vals(1) = mr_ub(lat,m,last,kl(i,k)) &
                               + delp(i,k) &
                                  * (mr_ub(lat,m,last,ku(i,k)) &
                                   - mr_ub(lat,m,last,kl(i,k)))
                pint_vals(2) = mr_ub(lat,m,next,kl(i,k)) &
                              + delp(i,k) &
                             * (mr_ub(lat,m,next,ku(i,k)) &
                                   - mr_ub(lat,m,next,kl(i,k)))
                h2o(i,k) = pint_vals(1) &
                           + dels * (pint_vals(2) - pint_vals(1))
            end do
         end do

long_loop2 : &
         do i = 1,plonl
!--------------------------------------------------------
! 	... relax lower stratosphere to extended ubc
!           check to make sure ubc is not being imposed too low
!           levrelax = lowest model level (highest pressure)
!                      in which to relax to ubc
!--------------------------------------------------------
           levrelax = ltrop(i)
           do while( pmid(i,levrelax) > ub_plevs(ub_nlevs) )
              levrelax = levrelax - 1
           end do
#ifdef DEBUG
           if( levrelax /= ltrop(i) ) then
              write(*,*) 'warning -- raised ubc: ',lat,i,
                         ltrop(i)-1,nint(pmid(i,ltrop(i)-1)/100.),'mb -->',
                         levrelax,nint(pmid(i,levrelax)/100.),'mb'
           end if
#endif
           do k = kmax(i)+1,levrelax
              pint_vals(1) = mr_ub(lat,m,last,kl(i,k)) &
                             + delp(i,k) &
                               * (mr_ub(lat,m,last,ku(i,k)) &
                                  - mr_ub(lat,m,last,kl(i,k)))
              pint_vals(2) = mr_ub(lat,m,next,kl(i,k)) &
                             + delp(i,k) &
                              * (mr_ub(lat,m,next,ku(i,k)) &
                                 - mr_ub(lat,m,next,kl(i,k)))
              vmrrelax = pint_vals(1) &
                         + dels * (pint_vals(2) - pint_vals(1))
              h2o(i,k) = h2o(i,k) + (vmrrelax - h2o(i,k)) * facrelax
           end do
         end do long_loop2
      end if h2o_overwrite

      end subroutine set_ub_h2o

      subroutine rebin( nsrc, ntrg, src_x, trg_x, src, trg )
!---------------------------------------------------------------
!	... rebin src to trg
!---------------------------------------------------------------

      implicit none

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in)   :: nsrc                  ! dimension source array
      integer, intent(in)   :: ntrg                  ! dimension target array
      real, intent(in)      :: src_x(nsrc+1)         ! source coordinates
      real, intent(in)      :: trg_x(ntrg+1)         ! target coordinates
      real, intent(in)      :: src(nsrc)             ! source array
      real, intent(out)     :: trg(ntrg)             ! target array

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: i, l
      integer  :: si, si1
      integer  :: sil, siu
      real     :: y
      real     :: sl, su
      real     :: tl, tu

!---------------------------------------------------------------
!	... check interval overlap
!---------------------------------------------------------------
!     if( trg_x(1) < src_x(1) .or. trg_x(ntrg+1) > src_x(nsrc+1) ) then
!        write(*,*) 'rebin: target grid is outside source grid'
!        write(*,*) '       target grid from ',trg_x(1),' to ',trg_x(ntrg+1)
!        write(*,*) '       source grid from ',src_x(1),' to ',src_x(nsrc+1)
!        call endrun
!     end if

      do i = 1,ntrg
        tl = trg_x(i)
        if( tl < src_x(nsrc+1) ) then
           do sil = 1,nsrc+1
              if( tl <= src_x(sil) ) then
                 exit
              end if
           end do
           tu = trg_x(i+1)
           do siu = 1,nsrc+1
              if( tu <= src_x(siu) ) then
                 exit
              end if
           end do
           y   = 0.
           sil = max( sil,2 )
           siu = min( siu,nsrc+1 )
           do si = sil,siu
              si1 = si - 1
              sl  = max( tl,src_x(si1) )
              su  = min( tu,src_x(si) )
              y   = y + (su - sl)*src(si1)
           end do
           trg(i) = y/(trg_x(i+1) - trg_x(i))
        else
           trg(i) = 0.
        end if
      end do

      end subroutine rebin

      end module mo_ub_vals
