
      module mo_xsections

      implicit none

      save

      real, allocatable :: a(:)
      real, allocatable :: b(:)
      real, allocatable :: suma(:)
      real, allocatable :: sumb(:)

      contains

      subroutine r01( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide the product of (cross section) x (quantum yield) for the two
!   o3 photolysis reactions:
!              (a) o3 + hv -> o2 + o(1d)
!              (b) o3 + hv -> o2 + o(3p)
!   cross section:  combined data from wmo 85 ozone assessment (use 273k
!                   value from 175.439-847.5 nm) and data from molina and
!                   molina (use in hartley and huggins bans (240.5-350 nm)
!   quantum yield:  choice between
!                    (1) data from michelsen et al, 1994
!                    (2) jpl 87 recommendation
!                    (3) jpl 90/92 recommendation (no "tail")
!                    (4) data from shetter et al., 1996
!                    (5) jpl 97 recommendation
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i)
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i)
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i)
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i)
!   tlev   - real, temperature (k) at each specified altitude level       (i)
!   airlev - real, air density (molec/cc) at each altitude level          (i)
!   j      - integer, counter for number of weighting functions defined  (io)
!   sq     - real, cross section x quantum yield (cm^2) for each          (o)
!            photolysis reaction defined, at each defined wavelength and
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o)
!            defined
!-----------------------------------------------------------------------------
!   edit history:
!   05/98  original, adapted from former jspec1 subroutine
!-----------------------------------------------------------------------------
!  this program is free software;  you can redistribute it and/or modify
!  it under the terms of the gnu general public license as published by the
!  free software foundation;  either version 2 of the license, or (at your
!  option) any later version.
!  the tuv package is distributed in the hope that it will be useful, but
!  without any warranty;  without even the implied warranty of merchantibi-
!  lity or fitness for a particular purpose.  see the gnu general public
!  license for more details.
!  free software foundation, inc., 675 mass ave, cambridge, ma 02139, usa.
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_waveo3
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real, parameter ::     c0 = 12./19.
      real, parameter ::     a1=0.887
      real, parameter ::     a2=2.35
      real, parameter ::     a3=57.0
      real, parameter ::     wc1=302.
      real, parameter ::     wc2=311.1
      real, parameter ::     wc3=313.9
      real, parameter ::     v2=820.0
      real, parameter ::     v3=1190.0
      real, parameter ::     w1=7.9
      real, parameter ::     w2=2.2
      real, parameter ::     w3=7.4
      real, parameter ::     xk=0.695

      integer :: i, iw
      integer :: myld
      real    :: qy1d, qy3p, so3, wrk, tinv
      logical :: tl

!-----------------------------------------------------------------------------
!       NEW O3 qy 2002 
!       Matsumi et al., 2002
!-----------------------------------------------------------------------------
      integer, parameter :: kmats = 7

      myld  = kmats

level_loop : &
      do i = 1,plevp
         tinv = 1./tlev(i)
         tl   = tlev(i) < 263.
         if( tl ) then
            wrk = (tlev(i) - 226.)/(263. - 226.)
         else
            wrk = (tlev(i) - 263.)/(298. - 263.)
         end if
wave_loop : &
         do iw = 1,nw
            if( wl(iw) > 240.5  .and. wl(iw+1) < 350. ) then
               if( tl ) then
                  so3 = s226(iw) + (s263(iw) - s226(iw)) * wrk
               else
                  so3 = s263(iw) + (s298(iw) - s263(iw)) * wrk
               end if
            else
               so3 = xso3(iw)
            end if
!-----------------------------------------------------------------------------
! 	... from jpl97
!-----------------------------------------------------------------------------
             if( wc(iw) < 271. ) then
                qy1d = .87
             else if( wc(iw) >= 271. .and. wc(iw) < 290. ) then
                qy1d = .87 + (wc(iw) - 271.)*c0
             else if( wc(iw) >= 290. .and. wc(iw) < 305. ) then
                qy1d = .95
             else if( wc(iw) >= 305. .and. wc(iw) <= 325. ) then
                qy1d = r01g1(iw) * exp ( -r01g2(iw)*tinv )
             else
                qy1d = 0.
             end if
!-------------------------------------------------------------------------------
!	... from jpl2000
!-------------------------------------------------------------------------------
             if( wc(iw) < 300. ) then
                qy1d = 0.95
             else if( wc(iw) >= 300. .and. wc(iw) < 331. ) then
                qy1d = a1*exp( -((wc(iw) - wc1 )/w1)**4 ) &
                     + a2*(tlev(i)/300.)**4*exp( -v2/xk*tinv ) &
                    * exp( -((wc(iw) - wc2)/w2)**2 ) &
                     + a3*exp( -v3/xk*tinv ) * exp( -((wc(iw) - wc3)/w3)**2 ) &
                     + 0.06
             else if( wc(iw) >= 331. .and. wc(iw) <= 345. ) then
                qy1d = 0.06
             else
                qy1d = 0.
             end if

             if( myld == kmats ) then
                qy1d = fo3qy( wc(iw), tlev(i) )
             end if

             if( trim( jlabel ) == 'jo1d' ) then
                xs(iw,i) = qy1d*so3
             else
                qy3p     = 1. - qy1d
                xs(iw,i) = qy3p*so3
             end if
         end do wave_loop
      end do level_loop

      end subroutine r01

      function fo3qy( w, t )
!-----------------------------------------------------------------------------
!   PURPOSE:
! function to calculate the quantum yield O3 + hv -> O(1D) + O2,
! according to:                                                             
! Matsumi, Y., F. J. Comes, G. Hancock, A. Hofzumanhays, A. J. Hynes,
! M. Kawasaki, and A. R. Ravishankara, QUantum yields for production of O(1D)
! in the ultraviolet photolysis of ozone:  Recommendation based on evaluation
! of laboratory data, J. Geophys. Res., 107, 10.1029/2001JD000510, 2002.
!-----------------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      real, intent(in)  :: w
      real, intent(in)  :: t

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real :: kt
      real :: q1
      real :: q2 
      real :: a(3)  = (/ 0.8036, 8.9061, 0.1192 /)
      real :: x(3)  = (/ 304.225, 314.957, 310.737 /)
      real :: om(3) = (/ 5.576, 6.601, 2.187 /)

!-----------------------------------------------------------------------------
!	... function declarations
!-----------------------------------------------------------------------------
      real :: fo3qy
      
      fo3qy = 0.
      kt = 0.695 * t
      q1 = 1.
      q2 = exp( -825.518/kt )
      
      if( w <= 305. ) then
         fo3qy = .90
      else if( w > 305. .and. w <= 328. ) then
         fo3qy = 0.0765 &
                + a(1)*             (q1/(q1+q2))*exp( -((x(1)-w)/om(1))**4 ) &
                + a(2)*(t/300.)**2 *(q2/(q1+q2))*exp( -((x(2)-w)/om(2))**2 ) &
                + a(3)*(t/300.)**1.5            *exp( -((x(3)-w)/om(3))**2 )
      else if( w > 328. .and. w <= 340. ) then
         fo3qy = 0.08
      else if( w > 340. ) then
         fo3qy = 0.
      end if

      end function fo3qy

      subroutine r04( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product of (cross section) x (quantum yiels) for n2o5 photolysis
!   reactions:
!        (a) n2o5 + hv -> no3 + no + o(3p)
!        (b) n2o5 + hv -> no3 + no2
!   cross section from jpl97: use tabulated values up to 280 nm, use expon.
!                             expression for >285nm, linearly interpolate
!                             between s(280) and s(285,t) in between
!   quantum yield: analysis of data in jpl94 (->dataj1/yld/n2o5.qy)
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i)
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i)
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i)
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i)
!   tlev   - real, temperature (k) at each specified altitude level       (i)
!   airlev - real, air density (molec/cc) at each altitude level          (i)
!   j      - integer, counter for number of weighting functions defined  (io)
!   sq     - real, cross section x quantum yield (cm^2) for each          (o)
!            photolysis reaction defined, at each defined wavelength and
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o)
!            defined
!-----------------------------------------------------------------------------
!   edit history:
!   05/98  original, adapted from former jspec1 subroutine
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real, parameter :: xs280 = 1.16999993e-19

      integer :: i, iw
      real :: qy, qy1
      real :: xsect, xst285
      real :: t

!-----------------------------------------------------------------------------
!	... n2o5 photodissociation
!-----------------------------------------------------------------------------
! 	... cross section from jpl97, table up to 280 nm
!           quantum yield : see dataj1/yld/n2o5.qy for explanation
!           correct for t-dependence of cross section
!-----------------------------------------------------------------------------
      do i = 1,plevp
!-----------------------------------------------------------------------------
! 	... temperature dependence only valid for 225 - 300 k.
!-----------------------------------------------------------------------------
         t = 1./max( 225.,min( tlev(i),300. ) )
         do iw = 1,nw
            qy  = max( 0.,min( 1., 3.832441 - 0.012809638 * wc(iw) ) )
            qy1 = 1. - qy
!-----------------------------------------------------------------------------
! 	... evaluate exponential
!-----------------------------------------------------------------------------
            if( wl(iw) >= 285. .and. wl(iw+1) <= 380. ) then
               xs(iw,i) = qy1 * 1.e-20*exp( 2.735 + (4728.5 - 17.127*wc(iw)) * t )
!-----------------------------------------------------------------------------
! 	... between 280 and 285, interpolate between temperature evaluated exponential
!           at 285 nm and the tabulated value at 280 nm.
!-----------------------------------------------------------------------------
            else if( wl(iw) >= 280. .and. wl(iw+1) <= 286. ) then
               xst285 = 1.e-20* exp( 2.735 + (4728.5 - 17.127*286.)*t )
               xsect  = xs280 + (wc(iw) - 280.)*(xst285 - xs280)/(286.-280.)
               xs(iw,i) = qy1 * xsect 
!-----------------------------------------------------------------------------
! 	... use tabulated values
!-----------------------------------------------------------------------------
            else if (wl(iw) <= 280. ) then
               xs(iw,i) = qy1 * r04g(iw)
!-----------------------------------------------------------------------------
! 	... beyond 380 nm, set to zero
!-----------------------------------------------------------------------------
            else
               xs(iw,i) = 0.
            end if
         end do
      end do

      end subroutine r04

      subroutine r44_inti( nw, wc )
!-----------------------------------------------------------------------------
!	... initialize subroutine r44
!-----------------------------------------------------------------------------

      use mo_params, only : kw

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: nw
      real, intent(in)    :: wc(kw)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
!-----------------------------------------------------------------------------
!	... cross sections according to jpl97 recommendation (identical to 94 rec.)
!           see file dataj1/abs/n2o_jpl94.abs for detail
!-----------------------------------------------------------------------------
      real, parameter :: a0 = 68.21023                
      real, parameter :: a1 = -4.071805               
      real, parameter :: a2 = 4.301146e-02            
      real, parameter :: a3 = -1.777846e-04           
      real, parameter :: a4 = 2.520672e-07

      real, parameter :: b0 = 123.4014
      real, parameter :: b1 = -2.116255
      real, parameter :: b2 = 1.111572e-02
      real, parameter :: b3 = -1.881058e-05

      integer :: iw, astat
      real    :: lambda

      allocate( a(nw), b(nw), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'r44_inti: a,b allocate failed; error = ',astat
         call endrun()
      end if
      do iw = 1,nw
         lambda = wc(iw)   
         if( lambda >= 173. .and. lambda <= 240. ) then
            a(iw) = (((a4*lambda + a3)*lambda + a2)*lambda + a1)*lambda + a0
            b(iw) = ((b3*lambda + b2)*lambda + b1)*lambda + b0
         end if
      end do

      end subroutine r44_inti

      subroutine r44( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product (cross section) x (quantum yield) for n2o photolysis:
!               n2o + hv -> n2 + o(1d)
!   cross section: from jpl 97 recommendation
!   quantum yield: assumed to be unity, based on greenblatt and ravishankara
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i)
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i)
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i)
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i)
!   tlev   - real, temperature (k) at each specified altitude level       (i)
!   airlev - real, air density (molec/cc) at each altitude level          (i)
!   j      - integer, counter for number of weighting functions defined  (io)
!   sq     - real, cross section x quantum yield (cm^2) for each          (o)
!            photolysis reaction defined, at each defined wavelength and
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o)
!            defined
!-----------------------------------------------------------------------------
!   edit history:
!   05/98  original, adapted from former jspec1 subroutine
!-----------------------------------------------------------------------------

      use mo_params
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: i
      real    :: t

!-----------------------------------------------------------------------------
!	... n2o photodissociation
!-----------------------------------------------------------------------------
!	... quantum yield of n(4s) and no(2pi) is less than 1% (greenblatt and
!           ravishankara), so quantum yield of o(1d) is assumed to be unity
!-----------------------------------------------------------------------------
      do i = 1,plevp
         t = max( 194.,min( tlev(i),320. ) ) - 300.
         where( wc(:nw) >= 173. .and. wc(:nw) <= 240. )
            xs(:nw,i) = exp( a(:nw) + t*exp( b(:nw) ) )
         elsewhere
            xs(:nw,i) = 0.
         endwhere
      end do

      end subroutine r44

      subroutine r08_inti( nw, wl, wc )
!-----------------------------------------------------------------------------
!	... initialize subroutine r08
!-----------------------------------------------------------------------------

      use mo_params, only : kw

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: nw
      real, intent(in)    :: wl(kw), wc(kw)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real, parameter :: a0 = 6.4761e+04            
      real, parameter :: a1 = -9.2170972e+02        
      real, parameter :: a2 = 4.535649              
      real, parameter :: a3 = -4.4589016e-03        
      real, parameter :: a4 = -4.035101e-05         
      real, parameter :: a5 = 1.6878206e-07
      real, parameter :: a6 = -2.652014e-10
      real, parameter :: a7 = 1.5534675e-13

      real, parameter :: b0 = 6.8123e+03
      real, parameter :: b1 = -5.1351e+01
      real, parameter :: b2 = 1.1522e-01
      real, parameter :: b3 = -3.0493e-05
      real, parameter :: b4 = -1.0924e-07

      integer :: iw, astat
      real    :: lambda

      allocate( suma(nw), sumb(nw), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'r08_inti: suma,sumb allocate failed; error = ',astat
         call endrun()
      end if
      do iw = 1,nw
         if( wl(iw) >= 260. .and. wl(iw) < 350. ) then
            lambda = wc(iw)
            suma(iw) = ((((((a7*lambda + a6)*lambda + a5)*lambda + a4)*lambda +a3)*lambda + a2)*lambda + a1)*lambda + a0
            sumb(iw) = (((b4*lambda + b3)*lambda + b2)*lambda + b1)*lambda + b0
         end if
      end do

      end subroutine r08_inti

      subroutine r08( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product of (cross section) x (quantum yield) for h2o2 photolysis
!          h2o2 + hv -> 2 oh
!   cross section:  from jpl97, tabulated values @ 298k for <260nm, t-depend.
!                   parameterization for 260-350nm
!   quantum yield:  assumed to be unity
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i)
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i)
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i)
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i)
!   tlev   - real, temperature (k) at each specified altitude level       (i)
!   airlev - real, air density (molec/cc) at each altitude level          (i)
!   j      - integer, counter for number of weighting functions defined  (io)
!   sq     - real, cross section x quantum yield (cm^2) for each          (o)
!            photolysis reaction defined, at each defined wavelength and
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o)
!            defined
!-----------------------------------------------------------------------------
!   edit history:
!   05/98  original, adapted from former jspec1 subroutine
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: i

      real :: t
      real :: chi

!-----------------------------------------------------------------------------
!	... h2o2 photodissociation
!           cross section from lin et al. 1978
!-----------------------------------------------------------------------------
      do i = 1,plevp
         t = 1./min( max( tlev(i),200. ),400. )            
         chi = 1./(1. + exp( -1265.*t ))
!-----------------------------------------------------------------------------
! 	... parameterization (jpl94)
!           range 260-350 nm; 200-400 k
!-----------------------------------------------------------------------------
         where( wl(:nw) > 260. .and. wl(:nw) < 350. )
            xs(:nw,i) = (chi * suma(:nw) + (1. - chi)*sumb(:nw))*1.e-21
         elsewhere
            xs(:nw,i) = r08g(:nw)
         endwhere
      end do

      end subroutine r08

      subroutine r06( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product of (cross section) x (quantum yield) for hno3 photolysis 
!         hno3 + hv -> oh + no2
!   cross section: burkholder et al., 1993
!   quantum yield: assumed to be unity
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i)
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i)
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i)
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i)
!   tlev   - real, temperature (k) at each specified altitude level       (i)
!   airlev - real, air density (molec/cc) at each altitude level          (i)
!   j      - integer, counter for number of weighting functions defined  (io)
!   sq     - real, cross section x quantum yield (cm^2) for each          (o)
!            photolysis reaction defined, at each defined wavelength and
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o)
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp
      
      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw)
      real, intent(in)       :: wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: k
      real    :: wrk

!-----------------------------------------------------------------------------
!	... hno3 photodissociation
!-----------------------------------------------------------------------------
! 	... hno3 cross section parameters from burkholder et al. 1993
!           quantum yield = 1
!           correct for temperature dependence
!-----------------------------------------------------------------------------
      do k = 1,plevp
         wrk = 1.e-3*(tlev(k) - 298.)
         xs(:nw,k) = r06g1(:nw) * 1.e-20 * exp( r06g2(:nw)*wrk )
      end do

      end subroutine r06

      subroutine r10( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product of (cross section) x (quantum yield) for ch2o photolysis *
!         (a) ch2o + hv -> h + hco
!         (b) ch2o + hv -> h2 + co
!   cross section: choice between
!                  1) bass et al., 1980 (resolution: 0.025 nm)
!                  2) moortgat and schneider (resolution: 1 nm)
!                  3) cantrell et al. (orig res.) for > 301 nm,
!                     iupac 92, 97 elsewhere
!                  4) cantrell et al. (2.5 nm res.) for > 301 nm,
!                     iupac 92, 97 elsewhere
!                  5) rogers et al., 1990
!                  6) new ncar recommendation, based on averages of
!                     cantrell et al., moortgat and schneider, and rogers
!                     et al.
!   quantum yield: choice between
!                  1) evaluation by madronich 1991 (unpublished)
!                  2) iupac 89, 92, 97
!                  3) madronich, based on 1), updated 1998.
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   j      - integer, counter for number of weighting functions defined  (io) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o)
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw)
      real, intent(in)       :: wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel


!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer, parameter :: mopt1 = 6
      integer, parameter :: mopt2 = 1
      real, parameter :: c0 = 1./70.

      integer :: i, iw
      real    :: phi1, phi2, phi20, ak300, akt
      real    :: qy, qy1, qy2, qy3, t, t1
      real    :: sigma, sig, slope

!-----------------------------------------------------------------------------
!	... ch2o photodissociatation
!-----------------------------------------------------------------------------
! 	... combine
!           y1 = xsect
!           y2 = xsect(223), cantrell et al.
!           y3 = xsect(293), cantrell et al.
!           y4 = qy for radical channel
!           y5 = qy for molecular channel
!           pressure and temperature dependent for w > 330.
!-----------------------------------------------------------------------------
      do i = 1,plevp
         t = max( 223.15,min( tlev(i),293.15 ) )
         t1 = airlev(i)
         do iw = 1,nw
            if( mopt1 == 6 ) then
               sig = r10g2(iw)
            else
               sig = r10g1(iw)
            end if
!-----------------------------------------------------------------------------
! 	... correct cross section for temperature dependence for > 301. nm
!-----------------------------------------------------------------------------
            if( wl(iw) >= 301. ) then 
               if( mopt1 == 3 .or. mopt1 == 6 ) then
                  sig = r10g2(iw) + r10g3(iw) * (t - 273.15)
               else if( mopt1 == 4 ) then
                  slope = (r10g3(iw) - r10g2(iw)) * c0
                  slope = (r10g3(iw) - r10g2(iw)) * c0
                  sig = r10g2(iw) + slope * (t - 223.)
               end if
            end if
            sig = max( sig,0. )
!-----------------------------------------------------------------------------
! 	... quantum yields:
!           temperature and pressure dependence beyond 330 nm
!-----------------------------------------------------------------------------
            qy1 = r10g4(iw)
            if( trim(jlabel) == 'jch2o_a' ) then
               xs(iw,i) = sig * qy1
            else
               if( wc(iw) >= 330. .and. r10g5(iw) > 0. ) then
                  phi1 = r10g4(iw)
                  phi2 = r10g5(iw)
                  phi20 = 1. - phi1
                  ak300 = ((1./phi2) - (1./phi20))/2.54e+19
                  akt = ak300*(1. + 61.69*(1. - tlev(i)/300.)*(wc(iw)/329. - 1.))
                  qy2 = 1. / ((1./phi20) + t1*akt)
               else
                  qy2 = r10g5(iw)
               end if
               qy2 = max( 0.,qy2 )
               qy2 = min( 1.,qy2 )
               xs(iw,i) = sig * qy2
            end if
         end do
      end do

      end subroutine r10

      subroutine r11( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product (cross section) x (quantum yield) for ch3cho photolysis:
!       (a)  ch3cho + hv -> ch3 + hco
!       (b)  ch3cho + hv -> ch4 + co
!       (c)  ch3cho + hv -> ch3co + h
!   cross section:  choice between
!                    (1) iupac 97 data, from martinez et al.
!                    (2) calvert and pitts
!                    (3) martinez et al., table 1 scanned from paper
!                    (4) kfa tabulations
!   quantum yields: choice between
!                    (1) iupac 97, pressure correction using horowith and
!                                  calvert, 1982
!                    (2) ncar data file, from moortgat, 1986
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   j      - integer, counter for number of weighting functions defined  (io) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and      
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o) 
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw)
      real, intent(in)       :: wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer, parameter :: mabs = 3
      integer, parameter :: myld = 1
      real, parameter    :: c0 = 1./2.465e19

      integer :: k, wn
      real    :: qy1
      real    :: sig, t

!-----------------------------------------------------------------------------
!	... ch3cho photolysis
!           1:  ch3 + hco
!           2:  ch4 + co
!           3:  ch3co + h
!-----------------------------------------------------------------------------
! 	... options
!           mabs for cross sections
!           myld for quantum yields
!
!           absorption:
!           1:  iupac-97 data, from martinez et al.
!           2:  calvert and pitts
!           3:  martinez et al., table 1 scanned from paper
!           4:  kfa tabulations, 6 choices, see file open statements
!
!           quantum yield
!           1:  dataj1/ch3cho/ch3cho_iup.yld
!               pressure correction using horowitz and calvert 1982, 
!               based on slope/intercept of stern-volmer plots
!
!           2:  ncar data file, from moortgat 1986.
!-----------------------------------------------------------------------------

      do k = 1,plevp
         t = airlev(k)*c0
         do wn = 1,nw
            sig = r11g(wn)
!-----------------------------------------------------------------------------
! 	... pressure correction for channel 1, ch3 + cho
!           based on horowitz and calvert 1982.
!-----------------------------------------------------------------------------
            if( trim( jlabel ) == 'jch3cho_a' ) then
               qy1 = r11g1(wn) * (1. + r11g4(wn))/(1. + r11g4(wn)*t)
               qy1 = min( 1.,max( 0.,qy1 ) )
               xs(wn,k) = sig * qy1
            else if( trim( jlabel ) == 'jch3cho_b' ) then
               xs(wn,k) = sig * r11g2(wn)
            else if( trim( jlabel ) == 'jch3cho_c' ) then
               xs(wn,k) = sig * r11g3(wn)
            end if
         end do
      end do

      end subroutine r11

      subroutine r14( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide the product (cross section) x (quantum yield) for ch3cocho
!   photolysis:
!            ch3cocho + hv -> products
!
!   cross section: choice between
!                   (1) from meller et al., 1991, as tabulated by iupac 97
!                          5 nm resolution (table 1) for < 402 nm
!                          2 nm resolution (table 2) for > 402 nm
!                   (2) average at 1 nm of staffelbach et al., 1995, and
!                       meller et al., 1991
!                   (3) plum et al., 1983, as tabulated by kfa
!                   (4) meller et al., 1991 (0.033 nm res.), as tab. by kfa
!                   (5) meller et al., 1991 (1.0 nm res.), as tab. by kfa
!                   (6) staffelbach et al., 1995, as tabulated by kfa
!   quantum yield: choice between
!                   (1) plum et al., fixed at 0.107
!                   (2) plum et al., divided by 2, fixed at 0.0535
!                   (3) staffelbach et al., 0.45 for < 300 nm, 0 for > 430 nm
!                       linear interp. in between
!                   (4) koch and moortgat, prv. comm., 1997
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   j      - integer, counter for number of weighting functions defined  (io) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and      
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o) 
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real, parameter :: sig2 = .45
      real, parameter :: sig3 = 1./130.

      integer :: i, iw
      real    :: qy
      real    :: phi0, kq

!-----------------------------------------------------------------------------
!	... ch3cocho photolysis
!-----------------------------------------------------------------------------
! 	... options
!           mabs for cross sections
!           myld for quantum yields
!
!           absorption:
!           1:  from meller et al. (1991), as tabulated by iupac-97
!               for wc < 402, use coarse data (5 nm, table 1)
!               for wc > 402, use finer data (2 nm, table 2)
!           2: average at 1nm of  staffelbach et al. 1995 and meller et al. 1991
!               cross section from kfa tables:
!           3: ch3cocho.001 - plum et al. 1983
!           4: ch3cocho.002 - meller et al. 1991, 0.033 nm resolution
!           5: ch3cocho.003 - meller et al. 1991, 1.0   nm resolution
!           6: ch3cocho.004 - staffelbach et al. 1995
!
!           quantum yield
!           1:  plum et al., 0.107
!           2:  plum et al., divided by two = 0.0535
!           3:  staffelbach et al., 0.45 at wc .le. 300, 0 for wc .gt. 430, linear 
!               interp in between
!           4:  koch and moortgat, prv. comm. 1997. - pressure-dependent
!         * 5:  Chen, Y., W. Wang, and L. Zhu, Wavelength-dependent photolysis of methylglyoxal
!         *      in the 290-440 nm region, J Phys Chem A, 104, 11126-11131, 2000
!-----------------------------------------------------------------------------
      do i = 1,plevp
         do iw = 1,nw
            phi0 = 1. - (wc(iw) - 380.)/60.
            phi0 = max( 0.,min( phi0,1. ) )
            kq = 1.36e8 * exp( -8793./wc(iw) )
            if( phi0 > 0. ) then
               if( wc(iw) >= 380. .and. wc(iw) <= 440. ) then
                  qy = phi0 / (phi0 + kq * airlev(i) * 760./2.456e19)
               else
                  qy = phi0
               end if
            else
               qy = 0.
            end if
            xs(iw,i) = r14g(iw) * qy
         end do
      end do

      end subroutine r14

      subroutine r15( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product (cross section) x (quantum yield) for ch3coch3 photolysis
!           ch3coch3 + hv -> products
!
!   cross section:  choice between
!                    (1) calvert and pitts
!                    (2) martinez et al., 1991, alson in iupac 97
!                    (3) noaa, 1998, unpublished as of 01/98
!   quantum yield:  choice between
!                    (1) gardiner et al, 1984
!                    (2) iupac 97
!                    (3) mckeen et al., 1997
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   j      - integer, counter for number of weighting functions defined  (io) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and      
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o) 
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw)
      real, intent(in)       :: wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer, parameter :: mabs = 2
      integer, parameter :: myld = 3

      integer :: i, iw
      real    :: qy
      real    :: sig
      real    :: a, b, t, t1
      real    :: m, fco, fac, w

!-----------------------------------------------------------------------------
!	... ch3coch3 photodissociation
!-----------------------------------------------------------------------------
! 	... options
!           mabs for cross sections
!           myld for quantum yields
!
!           absorption:
!           1:  cross section from calvert and  pitts
!           2:  martinez et al. 1991, also in iupac97
!           3:  noaa 1998, unpublished as of jan 98.
!
!           quantum yield
!           1:  gardiner et al. 1984
!           2:  iupac 97
!           3:  mckeen, s. a., t. gierczak, j. b. burkholder, p. o. wennberg, t. f. hanisco,
!               e. r. keim, r.-s. gao, s. c. liu, a. r. ravishankara, and d. w. fahey, 
!               the photochemistry of acetone in the upper troposphere:  a source of 
!               odd-hydrogen radicals, geophys. res. lett., 24, 3177-3180, 1997.
!           4:  Blitz, M. A., D. E. Heard, M. J. Pilling, S. R. Arnold, and M. P. Chipperfield 
!              (2004), Pressure and temperature-dependent quantum yields for the 
!               photodissociation of acetone between 279 and 327.5 nm, Geophys. 
!               Res. Lett., 31, L06111, doi:10.1029/2003GL018793.
!
!-----------------------------------------------------------------------------
      do i = 1,plevp
         m = airlev(i)
         t = tlev(i)
         do iw = 1,nw
            sig = r15g(iw)
            w   = wc(iw)
            call qyacet( w, t, m, fco, fac )
            qy = min( 1.,max( 0.,fac ) )
            xs(iw,i) = sig*qy
         end do
      end do

      end subroutine r15

      subroutine qyacet( w, t, m, fco, fac )
!-----------------------------------------------------------------------------
! Compute acetone quantum yields according to the parameterization of:
! Blitz, M. A., D. E. Heard, M. J. Pilling, S. R. Arnold, and M. P. Chipperfield 
!       (2004), Pressure and temperature-dependent quantum yields for the 
!       photodissociation of acetone between 279 and 327.5 nm, Geophys. 
!       Res. Lett., 31, L06111, doi:10.1029/2003GL018793.
!-----------------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      real, intent(in)  :: w            ! w = wavelength (nm)
      real, intent(in)  :: t            ! T = temperature (K)
      real, intent(in)  :: m            ! m = air number density (molec/cm^3)
      real, intent(out) :: fco          ! fco = quantum yield for product CO
      real, intent(out) :: fac          ! fac = quantum yield for product CH3CO (acetyl radical)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real :: a0, a1, a2, a3, a4
      real :: b0, b1, b2, b3, b4
      real :: c3
      real :: cA0, cA1, cA2, cA3, cA4

!-----------------------------------------------------------------------------
!** set out-of-range values:
! use low pressure limits for shorter wavelengths
! set to zero beyound 327.5
!-----------------------------------------------------------------------------
      if( w < 279. ) then
         fco = 0.05
         fac = 0.95
      else if( w > 327.5 ) then
         fco = 0.
         fac = 0.
      else
!-----------------------------------------------------------------------------
!	... CO (carbon monoxide) quantum yields
!-----------------------------------------------------------------------------
         a0  = .350 * (t/295.)**(-1.28)
         b0  = .068 * (t/295.)**(-2.65)
         cA0 = exp( b0*(w - 248.) ) * a0 / (1. - a0)
         fco = 1. / (1. + cA0)
!-----------------------------------------------------------------------------
!	... CH3CO (acetyl radical) quantum yields:
!-----------------------------------------------------------------------------
         if( w >= 279. .and. w < 302. ) then
            a1  = 1.600e-19 * (t/295.)**(-2.38)
            b1  = 0.55e-3   * (t/295.)**(-3.19)
            cA1 = a1 * exp( -b1*((1.e7/w) - 33113.) )
            fac = (1. - fco) / (1. + cA1 * m)
         else if( w >= 302. .and. w < 327.5 ) then
            a2  = 1.62e-17 * (t/295.)**(-10.03)
            b2  = 1.79e-3  * (t/295.)**(-1.364)
            cA2 = a2 * exp( -b2*((1.e7/w) - 30488.) )
            a3  = 26.29   * (t/295.)**(-6.59)
            b3  = 5.72e-7 * (t/295.)**(-2.93)
            c3  = 30006   * (t/295.)**(-0.064)
            ca3 = a3 * exp( -b3*((1.e7/w) - c3)**2 )
            a4  = 1.67e-15 * (t/295.)**(-7.25)
            b4  = 2.08e-3  * (t/295.)**(-1.16)
            cA4 = a4 * exp( -b4*((1.e7/w) - 30488.) )
            fac = (1. - fco) * (1. + cA3 + cA4 * m) &
                  /((1. + cA3 + cA2 * M)*(1. + cA4 * m))
         end if
      end if

      end subroutine qyacet

      subroutine r17( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product (cross section) x (quantum yield) for ch3ono2
!   photolysis:
!           ch3ono2 + hv -> ch3o + no2
!
!   cross section: choice between
!                   (1) calvert and pitts, 1966
!                   (2) talukdar, burkholder, hunter, gilles, roberts,
!                       ravishankara, 1997
!                   (3) iupac 97, table of values for 198k
!                   (4) iupac 97, temperature-dependent equation
!                   (5) taylor et al, 1980
!                   (6) fit from roberts and fajer, 1989
!                   (7) rattigan et al., 1992
!                   (8) libuda and zabel, 1995
!   quantum yield: assumed to be unity
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   j      - integer, counter for number of weighting functions defined  (io) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and      
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o) 
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw)
      real, intent(in)       :: wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: i, iw
      real    :: qy, t, sig

!-----------------------------------------------------------------------------
!	... ch3ono2 photodissociation
!-----------------------------------------------------------------------------
! 	... mabs: absorption cross section options:
!           1:  calvert and  pitts 1966
!           2:  talukdar, burkholder, hunter, gilles, roberts, ravishankara, 1997.
!           3:  iupac-97, table of values for 298k.
!           4:  iupac-97, temperature-dependent equation
!           5:  taylor et al. 1980
!           6:  fit from roberts and fajer, 1989
!           7:  rattigan et al. 1992
!           8:  libuda and zabel 1995
!-----------------------------------------------------------------------------
      do i = 1,plevp
         t = tlev(i) - 298.
         xs(:nw,i) = r17g(:nw) * exp( r17g1(:nw)*t )
      end do

      end subroutine r17

      subroutine r18( nw, wl, wc, tlev, airlev, jlabel, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product (cross section) x (quantum yield) for pan photolysis:
!        pan + hv -> products
!
!   cross section: from talukdar et al., 1995
!   quantum yield: assumed to be unity
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   j      - integer, counter for number of weighting functions defined  (io) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and      
!            at each defined altitude level
!   jlabel - character*40, string identifier for each photolysis reaction (o) 
!            defined
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)
      character(len=*), intent(in) :: jlabel

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: i
      real    :: t

!-----------------------------------------------------------------------------
!	... pan photodissociation
!-----------------------------------------------------------------------------
! 	... cross section from senum et al., 1984, j.phys.chem. 88/7, 1269-1270
!           quantum yield
!           yet unknown, but assumed to be 1.0 (talukdar et al., 1995)
!-----------------------------------------------------------------------------
      do i = 1,plevp
         t = tlev(i) - 298.
         xs(:nw,i) = r18g(:nw) * exp( r18g2(:nw)*t )
      end do 

      end subroutine r18

      subroutine xs_mvk( nw, wl, wc, tlev, airlev, xs )
!-----------------------------------------------------------------------------
!   purpose:
!   provide product (cross section) x (quantum yield) for mvk photolysis:
!        mvk + hv -> products
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working        (i) 
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i) 
!            working wavelength grid
!   wc     - real, vector of center points of wavelength intervals in     (i) 
!            working wavelength grid
!   nz     - integer, number of altitude levels in working altitude grid  (i) 
!   tlev   - real, temperature (k) at each specified altitude level       (i) 
!   airlev - real, air density (molec/cc) at each altitude level          (i) 
!   sq     - real, cross section x quantum yield (cm^2) for each          (o) 
!            photolysis reaction defined, at each defined wavelength and      
!            at each defined altitude level
!-----------------------------------------------------------------------------

      use mo_params
      use mo_waveall
      use mo_grid,    only : plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nw
      real, intent(in)       :: wl(kw), wc(kw)
      real, intent(in)       :: tlev(plevp)
      real, intent(in)       :: airlev(plevp)
      real, intent(inout)    :: xs(:,:)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: k
      integer :: w
      real    :: denomi
      real    :: qy(nw)

      do k = 1,plevp
         denomi = 1./(5.5 + 9.2e-19*airlev(k))
         qy(:)  = exp( -.055*(wc(:nw) - 308.) )*denomi
         xs(:nw,k) = min( qy(:nw),1. ) * xs(:nw,k)
      end do

      end subroutine xs_mvk

      end module mo_xsections
