
      module mass_diags
!--------------------------------------------------------------------
!	... general purpose mozart2 diagnostic module
!--------------------------------------------------------------------

      use mo_control, only : date_time
      use mo_grid,    only : plon, plev, plevp, plat
      use mo_histout, only : hst_file_max

      implicit none

      save

      integer, parameter :: init = 0,  advection = 1, &
                            fixer = 2, diffusion = 3 , &
                            convection = 4, chem_diags = 5, &
                            ubc = 6

!--------------------------------------------------------------------
!	... volume mass definition
!--------------------------------------------------------------------
      type conservation
	 integer :: index                                ! species index
	 integer :: ofreq                                ! output frequency in time steps
	 integer :: ndates                               ! count of individual output dates
	 integer, dimension(2) :: longitudes, latitudes
	 integer, dimension(2,plat) :: levels
         real :: cvol                                    ! constant of volume integrals
         real :: area                                    ! region area in m^2
	 real :: initial_mass
	 real :: diurnal_mass
	 real :: step_delta, diurnal_delta, overall_delta
	 real, dimension(1:6) :: diurnal_acc_delta, &
                                 overall_acc_delta, &
                                 delta_mass
	 real, dimension(0:6) :: mass
	 real, dimension(plat,0:6) :: massj
         character(len=8)  :: species
         character(len=64) :: region
         character(len=64) :: filespec
	 type(date_time), pointer :: odates(:)
      end type conservation

!--------------------------------------------------------------------
!	... layer mass definition
!--------------------------------------------------------------------
      type layer
	 integer :: index                                ! species index
	 integer :: ofreq                                ! output frequency in time steps
	 integer :: ndates                               ! count of individual output dates
	 real    :: mass_factor
	 real, dimension(plev,1:6) :: diurnal_acc_delta, &
                                      overall_acc_delta, &
                                      delta_mass
	 real, dimension(plev,0:6) :: mass
	 real, dimension(plat,0:6,plev) &
                                   ::  massj
         character(len=8)          :: species
         character(len=8)          :: mass_units
	 type(date_time), pointer  :: odates(:)
      end type layer

!--------------------------------------------------------------------
!	... vertical flux definition
!--------------------------------------------------------------------
      type adv_flux
	 integer :: index                                ! species index
	 integer :: ofreq                                ! output frequency in time steps
	 integer :: ndates                               ! count of individual output dates
	 real    :: mass_factor
	 real    :: time_factor
	 real, dimension(plevp) :: diurnal_acc, &
                                   overall_acc
	 real, dimension(plevp) :: flux
	 real, dimension(plevp,plat) :: fluxj
	 real, pointer, dimension(:,:,:,:) :: &
                                   adv_mass, &
                                   mass_delta, &
                                   hist_vflx, &
                                   hist_xflx, &
                                   hist_yflx
         character(len=8)       :: species
         character(len=8)       :: mass_units
         character(len=8)       :: time_units
	 type(date_time), pointer  :: odates(:)
      end type adv_flux

      integer :: ndiags = 0
      type(conservation), allocatable :: moz_diags(:)
      integer :: ldiags = 0
      type(layer), allocatable :: layer_mass(:)
      integer, dimension(hst_file_max) :: fdiags = 0
      type(adv_flux), allocatable :: advn_flux(:,:)

      real :: cwava, rearthsq
      real :: sa_earth, sa_fac
      real :: hsa_fac, y_fac
      integer :: flux_indx = 0

      contains

      subroutine inidiags( file )
!--------------------------------------------------------------------
!	... initialize the diagnostic variables
!--------------------------------------------------------------------

      use mo_constants, only : gravit, rearth, pi, latwts, dayspy

      implicit none

!--------------------------------------------------------------------
!	... dummy arguments
!--------------------------------------------------------------------
      integer, intent(in) :: file

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: m

      if( file == 1 ) then
         cwava    = 1. / (real(plon)*gravit)
         rearthsq = rearth * rearth
         sa_fac   = 2. * pi * rearthsq
         sa_earth = 2. * sa_fac
         hsa_fac  = sa_fac / real(plon)
         y_fac    = 2. * pi * rearth / real(plon)
         do m = 1,ndiags
            moz_diags(m)%cvol = .5 * sa_earth * cwava
            moz_diags(m)%area = &
                  2. * pi * rearth * rearth &
                     * real(moz_diags(m)%longitudes(2) &
                            - moz_diags(m)%longitudes(1)) &
                     * sum( latwts(moz_diags(m)%latitudes(1): &
                                   moz_diags(m)%latitudes(2)) ) / real(plon)
            moz_diags(m)%mass(:) = 0.
            moz_diags(m)%diurnal_acc_delta(:) = 0.
            moz_diags(m)%overall_acc_delta(:) = 0.
         end do
         do m = 1,ldiags
            layer_mass(m)%delta_mass(:,:)        = 0.
            layer_mass(m)%diurnal_acc_delta(:,:) = 0.
            layer_mass(m)%overall_acc_delta(:,:) = 0.
         end do
      end if
      do m = 1,fdiags(file)
         advn_flux(m,file)%flux(:)        = 0.
         advn_flux(m,file)%diurnal_acc(:) = 0.
         advn_flux(m,file)%overall_acc(:) = 0.
      end do

      end subroutine inidiags

      integer function do_flux( target, file )
!--------------------------------------------------------------------
!	... check for flux diagnostics
!--------------------------------------------------------------------

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: target
      integer, intent(in) :: file            ! history file index

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: m

      do_flux = 0
      do m = 1,fdiags(file)
         if( target == advn_flux(m,file)%index ) then
	    do_flux = m
	    exit
	 end if
      end do

      end function do_flux

      subroutine hist_vflux( index, flux, file, plonl, platl, pplon )
!--------------------------------------------------------------------
!	... put vertical or horizontal flux into holding variable
!--------------------------------------------------------------------

      use mo_constants, only : rgrav, latwts
      use mo_control,   only : delt
      use mo_mpi,       only : base_lat

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: index
      integer, intent(in) :: file             ! history file index
      integer, intent(in) :: plonl            ! lon tile dim
      integer, intent(in) :: platl            ! lat tile dim
      integer, intent(in) :: pplon            ! lon tile count
      real, intent(in)    :: flux(:,:,:)

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: ip, j, k, beg_lat
      real    :: z_fac

      beg_lat = (size( flux,dim=2 ) - platl)/2
      do j = 1,platl
	 z_fac = rgrav*hsa_fac*latwts(base_lat+j)
	 do k = 2,plevp
	    do ip = 1,pplon
               advn_flux(index,file)%hist_vflx(:plonl,k-1,j,ip) = &
			 -flux((ip-1)*plonl+1:ip*plonl,j+beg_lat,k) * z_fac
	    end do
	 end do
      end do

      end subroutine hist_vflux

      subroutine hist_yflux( index, flux, initial, file, plonl, platl, pplon )
!--------------------------------------------------------------------
!	... put vertical or horizontal flux into holding variable
!--------------------------------------------------------------------

      use mo_constants, only : rgrav, latwts
      use mo_mpi,       only : base_lat

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: index
      integer, intent(in) :: file             ! history file index
      integer, intent(in) :: plonl            ! lon tile dim
      integer, intent(in) :: platl            ! lat tile dim
      integer, intent(in) :: pplon            ! lon tile count
      real, intent(in)    :: flux(:,:,:)
      logical, intent(in) :: initial

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: ip, j, k, beg_lat
      real    :: yfac

      beg_lat = (size( flux,dim=2 ) - platl)/2
      if( initial ) then
	 do j = 1,platl
	    yfac = rgrav*hsa_fac*latwts(base_lat+j)
	    do k = 1,plev
	       do ip = 1,pplon
                  advn_flux(index,file)%hist_yflx(:plonl,k,j,ip) = &
                     flux((ip-1)*plonl+1:ip*plonl,beg_lat+j,k) * yfac
	       end do
	    end do
	 end do
      else
	 do j = 1,platl
	    yfac = rgrav*hsa_fac*latwts(base_lat+j)
	    do k = 1,plev
	       do ip = 1,pplon
                  advn_flux(index,file)%hist_yflx(:plonl,k,j,ip) = &
                     advn_flux(index,file)%hist_yflx(:plonl,k,j,ip) &
                     + flux((ip-1)*plonl+1:ip*plonl,beg_lat+j,k) * yfac
	       end do
	    end do
	 end do
      end if

      end subroutine hist_yflux

      subroutine hist_xflux( index, flux, initial, file, plonl, platl, pplon )
!--------------------------------------------------------------------
!	... put vertical or horizontal flux into holding variable
!--------------------------------------------------------------------

      use mo_constants, only : rgrav, latwts
      use mo_mpi,       only : base_lat

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in) :: index
      integer, intent(in) :: file             ! history file index
      integer, intent(in) :: plonl            ! lon tile dim
      integer, intent(in) :: platl            ! lat tile dim
      integer, intent(in) :: pplon            ! lon tile count
      real, intent(in)    :: flux(:,:,:)
      logical, intent(in) :: initial

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      integer :: ip, j, k, beg_lat
      real    :: xfac

      beg_lat = (size( flux,dim=2 ) - platl)/2
      if( initial ) then
	 do j = 1,platl
	    xfac = rgrav*hsa_fac*latwts(base_lat+j)
	    do k = 1,plev
	       do ip = 1,pplon
                  advn_flux(index,file)%hist_xflx(:plonl,k,j,ip) = &
      	             flux((ip-1)*plonl+1:ip*plonl,beg_lat+j,k) * xfac
	       end do
	    end do
	 end do
      else
	 do j = 1,platl
	    xfac = rgrav*hsa_fac*latwts(base_lat+j)
	    do k = 1,plev
	       do ip = 1,pplon
                  advn_flux(index,file)%hist_xflx(:plonl,k,j,ip) = &
                     advn_flux(index,file)%hist_xflx(:plonl,k,j,ip) &
                     + flux((ip-1)*plonl+1:ip*plonl,beg_lat+j,k) * xfac
	       end do
	    end do
	 end do
      end if

      end subroutine hist_xflux

      subroutine diags_inti( assym, plonl, platl, pplon )
!------------------------------------------------------------------------
!	... initialize the diagnostics
!------------------------------------------------------------------------

      use mo_constants, only : dayspy
      use mo_control,   only : secpday
      use mo_histout,   only : hfile, sim_file_cnt
      use mo_grid,      only : pcnstm1

      implicit none

!------------------------------------------------------------------------
!	... dummy args
!------------------------------------------------------------------------
      integer, intent(in) :: plonl, platl, pplon
      character(len=8), intent(in) :: assym(pcnstm1)

!------------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------------
      integer :: d, n, astat, file

!------------------------------------------------------------------------
!	... mass delta and flux diagnostics
!------------------------------------------------------------------------
      if( any( hfile(:sim_file_cnt)%histout_cnt(16,2) > 0 ) ) then
         do file = 1,sim_file_cnt
            fdiags(file) = hfile(file)%histout_cnt(16,2)/8
	 end do
	 n = maxval( fdiags(:sim_file_cnt) )
         allocate( advn_flux(n,sim_file_cnt), stat=astat )
         if( astat /= 0 ) then
            write(*,*) ' diags_inti: failed to allocate advn_flux'
	    call endrun
         end if
      end if
      do file = 1,sim_file_cnt
         if( fdiags(file) > 0 ) then
	    d = hfile(file)%histout_ind(16,2)
	    write(*,*) 'diag_inti: file,d,index = ',file,d,hfile(file)%timav_map(d)
	    do n = 1,fdiags(file)
               allocate( advn_flux(n,file)%adv_mass(plonl,plev,platl,pplon), &
                         advn_flux(n,file)%mass_delta(plonl,plev,platl,pplon), &
                         advn_flux(n,file)%hist_vflx(plonl,plev,platl,pplon), &
                         advn_flux(n,file)%hist_xflx(plonl,plev,platl,pplon), &
                         advn_flux(n,file)%hist_yflx(plonl,plev,platl,pplon), &
	                 stat=astat )
               if( astat /= 0 ) then
                  write(*,*) ' diags_inti: failed to allocate advn_flux arrays: error =',astat
	          call endrun
               end if
               advn_flux(n,file)%index   = hfile(file)%timav_map(d+8*(n-1))
               advn_flux(n,file)%species = assym(advn_flux(n,file)%index)
               advn_flux(n,file)%ofreq   = 72
               advn_flux(n,file)%ndates  = 0
               advn_flux(n,file)%mass_factor = 1.e-9                   ! kg -> tg
               advn_flux(n,file)%time_factor = dayspy * real(secpday)  ! seconds per year
               advn_flux(n,file)%mass_units  = 'tg'
               advn_flux(n,file)%time_units  = 'yr'
	    end do
      end if

      call inidiags( file )

      end do

      end subroutine diags_inti

      end module mass_diags
