
      module mo_adv

      use m_adv

      private
      public :: ffsl_inti, advectxy, advectz, convert_winds, set_cross_terms

      contains

      subroutine ffsl_inti( nadv, diff_adv, xdo_pressure_fixer, platl )
!------------------------------------------------------------------------
! 	... Initialize Lin and Rood advection
!------------------------------------------------------------------------

      use MO_MPI, only : thisnode, masternode, lastnode, base_lat
#ifdef USE_MPI
      use MO_MPI, only : slab, region, xregion, yslab, yregion,  &
                         mpi_comm_comp, MPI_PROC_NULL, MPI_DOUBLE_PRECISION, MPI_SUCCESS,   &
                         persist_recv, persist_send
#endif
      use PLEVS,        only : ps0
      use MO_CONSTANTS, only : pi
      use MO_CONTROL,   only : delt
      use MO_GRID,      only : plon, plat, plev, plevp, pcnstm1
      use PLEVS,        only : ap => hyai, bp => hybi

      implicit none

!------------------------------------------------------------------------
! 	... Dummy arguments
!------------------------------------------------------------------------
      integer, intent(in) :: nadv                     ! number of advected species
      integer, intent(in) :: platl                    ! lat tile dim
      logical, intent(in) :: diff_adv                 ! diffusive advection flag ( true or false )
      logical, intent(in) :: xdo_pressure_fixer       ! diffusive advection flag ( true or false )

!------------------------------------------------------------------------
! 	... Local variables
!------------------------------------------------------------------------
      integer :: i, jglob, dest, ierr
      integer :: jlg(13), jug(13)
      real    :: dl, dp, agle

      do_pressure_fixer = xdo_pressure_fixer
      j2                = platl - j1 + 1

#ifdef USE_MPI
!------------------------------------------------------------------------
! 	... MPI defined type for communication
!------------------------------------------------------------------------
      call MPI_TYPE_VECTOR( plon*4, 1, 1, MPI_DOUBLE_PRECISION, slab, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_VECTOR failed for slab'
	 write(*,*) '           Error code = ',ierr
	 call ENDRUN
      end if

      call MPI_TYPE_COMMIT( slab, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_COMMIT failed for slab'
	 write(*,*) '           Error code = ',ierr
	 call ENDRUN
      end if

      call MPI_TYPE_HVECTOR( plev*nadv, 1, 8*plon*(platl+8), slab, region, ierr )
      if( ierr /= MPI_SUCCESS ) then
	 write(*,*) 'FFSL_INTI: MPI_TYPE_HVECTOR failed for region'
	 write(*,*) '           Error code = ',ierr
	 call ENDRUN
      end if

      call MPI_TYPE_COMMIT( region, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_COMMIT failed for region'
	 write(*,*) '           Error code = ',ierr
	 call ENDRUN
      end if

      call MPI_TYPE_HVECTOR( plev, 1, 8*plon*(platl+8), slab, xregion, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_HVECTOR failed for xregion'
         write(*,*) '           Error code = ',ierr
         call ENDRUN
      end if

      call MPI_TYPE_COMMIT( xregion, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_COMMIT failed for xregion'
         write(*,*) '           Error code = ',ierr
         call ENDRUN
      end if

      call MPI_TYPE_VECTOR( plon*3, 1, 1, MPI_DOUBLE_PRECISION, yslab, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_VECTOR failed for yslab'
         write(*,*) '           Error code = ',ierr
         call ENDRUN
      end if

      call MPI_TYPE_COMMIT( yslab, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_COMMIT failed for yslab'
         write(*,*) '           Error code = ',ierr
         call ENDRUN
      end if

      call MPI_TYPE_HVECTOR( plev, 1, 8*plon*(platl+6), yslab, yregion, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_HVECTOR failed for yregion'
         write(*,*) '           Error code = ',ierr
         call ENDRUN
      end if

      call MPI_TYPE_COMMIT( yregion, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'FFSL_INTI: MPI_TYPE_COMMIT failed for yregion'
         write(*,*) '           Error code = ',ierr
         call ENDRUN
      end if
#endif

      ALLOCATE( mmr(plon,-3:platl+4,plev,nadv), &
                jtt(plon,-1:platl+2,plev),      &
                jtn(plon,-1:platl+2,plev),      &
                stat=ierr )
      if( ierr /= 0 ) then
        write(*,*) 'FFSL_INTI: Failed to allocate mmr, jtt, jtn; error = ',ierr
        call ENDRUN
      end if

#ifdef USE_MPI
!------------------------------------------------------------------------
! 	... MPI persistent non-blocking comm handles
!------------------------------------------------------------------------
      if( .not. lastnode ) then
	 dest = thisnode + 1
      else
	 dest = MPI_PROC_NULL
      end if
      call MPI_SEND_INIT( mmr(1,platl-3,1,1), 1, region, dest, 1, mpi_comm_comp, persist_send(2), ierr )
      if( ierr /= 0 ) then
         write(*,*) 'FFSL_INTI: Failed to initialize persistent isend; error = ',ierr
         call ENDRUN
      end if
      call MPI_RECV_INIT( mmr(1,platl+1,1,1), 1, region, dest, 0, mpi_comm_comp, persist_recv(2), ierr )
      if( ierr /= 0 ) then
         write(*,*) 'FFSL_INTI: Failed to initialize persistent isend; error = ',ierr
         call ENDRUN
      end if
      if( .not. masternode ) then
	 dest = thisnode - 1
      else
	 dest = MPI_PROC_NULL
      end if
      call MPI_SEND_INIT( mmr(1,1,1,1), 1, region, dest, 0, mpi_comm_comp, persist_send(1), ierr )
      if( ierr /= 0 ) then
         write(*,*) 'FFSL_INTI: Failed to initialize persistent isend; error = ',ierr
         call ENDRUN
      end if
      call MPI_RECV_INIT( mmr(1,-3,1,1), 1, region, dest, 1, mpi_comm_comp, persist_recv(1), ierr )
      if( ierr /= 0 ) then
         write(*,*) 'FFSL_INTI: Failed to initialize persistent irecv; error = ',ierr
         call ENDRUN
      end if
#endif

      imh       = plon/2
      if( diff_adv ) then
         iord = 3
	 jord = 3
	 mfct = .false.
      end if

!-----------------------------------------------------------------------
! 	... Initialization
!-----------------------------------------------------------------------
      if( masternode ) then
         if( plev < 6 ) then
            write(*,*) 'TPCORE: plev must be >= 6'
            call ENDRUN
         end if

         write(*,*) ' '
         write(*,*) 'FFSL TransPort Core version 7m'
         write(*,*) ' '
#ifdef USE_MPI
         write(*,*) 'TPCORE was compiled for MPI multitasking'
#endif
#ifdef USE_OMP
         write(*,*) 'TPCORE was compiled for OMP multitasking'
#endif
#if defined(DEC)
         write(*,*) 'TPCORE was compiled for multitasking on Compaq Alpha'
#elif defined(SGI)
         write(*,*) 'TPCORE was compiled for multitasking on SGI O2K'
#elif defined(CRAY)
         write(*,*) 'TPCORE was compiled for multitasking on CRAY PVP'
#elif defined(IBM)
         write(*,*) 'TPCORE was compiled for multitasking on IBM SP'
#endif
         if( mfct ) then
            write(*,*) 'TPCORE: MFCT option is on'
         end if
 
         write(*,'('' TPCORE: Transporting '',i3,'' compounds'')') nadv
         write(*,'('' TPCORE: x, y, z order = '',3i3)') iord, jord, kord
         write(*,'('' TPCORE: Time step = '',i6)') delt
         if( do_pressure_fixer ) then
            write(*,*) 'TPCORE: Using pressure fixer'
         else
            write(*,*) 'TPCORE: Not using pressure fixer'
         end if
      end if
 
      dap(:plev) = (ap(2:plevp) - ap(:plev))*ps0
      dbk(:plev) =  bp(2:plevp) - bp(:plev)
 
      dl = 2.*pi / REAL(plon)
      dp =    pi / REAL(plat-1)
!------------------------------------------------------------------------
! 	... Local node lat limits
!------------------------------------------------------------------------
      jle((/1,6,7/)) = -3
      jue((/1,6,7/)) = platl + 4
      jle(2:4)       = -2
      jue(2:4)       = platl + 3
      jle((/5,8/))   = (/ -1, -2 /)
      jue((/5,8/))   = platl + 2
      jle(9:13)      = (/ -1, 0, -1, 0, -1 /)
      jue(9:13)      = (/ platl+3, platl+2, platl+2, platl+1, platl+2 /)

      jlg(1:2)   = 2
      jlg(3:7:2) = 1
      jlg(4:6:2) = j1
      jlg(8)     = jlg(2)
      jlg(9:13)  = j1
      jlg(11)    = j1 - 1
!------------------------------------------------------------------------
! 	... Modify lat lower limits for south pole
!------------------------------------------------------------------------
      do i = 1,13
	 jglob  = MAX( base_lat + jle(i),jlg(i) )
	 jle(i) = jglob - base_lat
      end do

      jug(1:2)   = plat - 1
      jug(3:7:2) = plat
      jug(4:6:2) = jlim_north
      jug(8)     = jug(2)
      jug(9:11)  = jlim_north + 1
      jug(12:13) = jlim_north
!------------------------------------------------------------------------
! 	... Modify lat upper limits for north pole
!------------------------------------------------------------------------
      do i = 1,13
	 jglob  = MIN( base_lat + jue(i),jug(i) )
	 jue(i) = jglob - base_lat
      end do

      jl(:4)   = -1
      ju(:4)   = platl + 2
      jl(5:8)  = (/ 1, -1, 1, -1 /)
      ju(5:8)  = (/ platl, platl + 3, platl, platl + 2 /)
      jl(9:10) = (/ 0, -1 /)
      ju(9:10) = (/ platl+1, platl+2 /)

      jlg(1)    = 1
      jlg(2:5)  = j1
      jlg(6:8)  = (/ 2, j1 + 1, 2 /)
      jlg(9:10) = 1
!------------------------------------------------------------------------
! 	... Modify lat lower limits for south pole
!------------------------------------------------------------------------
      do i = 1,10
	 jglob  = MAX( base_lat + jl(i),jlg(i) )
	 jl(i)  = jglob - base_lat
      end do

      jug(1)    = plat
      jug(2:5)  = jlim_north
      jug(3)    = jlim_north + 1
      jug(6:8)  = (/ plat, jlim_north -1, plat - 1 /)
      jug(9:10) = plat
!------------------------------------------------------------------------
! 	... Modify lat upper limits for north pole
!------------------------------------------------------------------------
      do i = 1,10
	 jglob  = MIN( base_lat + ju(i),jug(i) )
	 ju(i)  = jglob - base_lat
      end do
 
!------------------------------------------------------------------------
! 	... Output node advection limits
!------------------------------------------------------------------------
      write(*,*) '-------------------------'
      write(*,*) 'Node lat limits diagnostic'
      write(*,'(''jle = '',20i4)') jle
      write(*,'(''jue = '',20i4)') jue
      write(*,'(''jl  = '',20i4)') jl
      write(*,'(''ju  = '',20i4)') ju
      write(*,*) '-------------------------'

!-----------------------------------------------------------------------
! 	... Compute analytic cosine at cell edges
!-----------------------------------------------------------------------
      call COSA( cosp, cose, dp, platl )
 
      acosp(2:plat-1) = 1./cosp(2:plat-1)
!-----------------------------------------------------------------------
! 	... Inverse of the Scaled polar cap area
!-----------------------------------------------------------------------
      agle = (REAL(j1) - 1.5)*dp
      rcap  = dp / ( REAL(plon)*(1. - COS(agle)) )
      acosp(1)  = rcap
      acosp(plat) = rcap
      jvan = MAX( 1,plat/20 )
      j1vl = j1 + jvan
      j2vl = plat - j1vl + 1

      end subroutine FFSL_INTI

      subroutine ADVECTXY( ic, delp, delp1, u, v, &
                           pu, crx, cry, xmass, ymass, &
                           dq, fx, fy, Tmin, Tmax, &
			   Bmin, Bmax, plonl, platl, pplon )
!-----------------------------------------------------------------------
!	... Advect species
!-----------------------------------------------------------------------

      use MO_MPI, only     :  masternode, lastnode, base_lat
      use MO_GRID, only    :  plon, plev, plevp, plat
      use MASS_DIAGS, only :  DO_FLUX, HIST_XFLUX, HIST_YFLUX
      use MO_HISTOUT, only :  moz_file_cnt

      implicit none

!-----------------------------------------------------------------------
!	... Dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: ic
      integer, intent(in) :: plonl, platl, pplon
      real, intent(out)   :: Tmin, Tmax, Bmin, Bmax
      real, dimension(plon,-1:platl+2,plev), intent(out) :: dq
      real, dimension(plon,-3:platl+4,plev), intent(in)  :: u, crx
      real, dimension(plon,-3:platl+4,plev), intent(in)  :: delp, delp1, pu, xmass
      real, dimension(plon,-2:platl+3,plev), intent(in)  :: v, cry, ymass
      real, dimension(plon+1,-1:platl+2,plev), intent(out) :: fx
      real, dimension(plon,-2:platl+3,plev), intent(out)   :: fy

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, jt, jb, k, m, file
      real :: sum1, sum2
      real :: bad_mem
      real, dimension(plon,-3:platl+4) :: wk1
      real, dimension(plon,-1:platl+2) :: wk2
      real, dimension(plon,-1:platl+2,plev) :: fxp
      real, dimension(plon,-2:platl+3,plev) :: fyp
      real, dimension(plon,platl,plev)      :: hflx

#ifdef INIT_TEST
      bad_mem   = z'7ff7ffff7ff7ffff'
      wk1(:,:)  = bad_mem
      wk2(:,:)  = bad_mem
      dq(:,:,:) = bad_mem
#endif

      do k = 1,plev
         if( large_cap ) then
	    if( has_spole ) then
	       jb = 1 - base_lat
               mmr(:plon,jb+1,k,ic) = mmr(:plon,jb,k,ic)
	    end if
	    if( has_npole ) then
	       jb = plat - base_lat
               mmr(:plon,jb-1,k,ic) = mmr(:plon,jb,k,ic)
	    end if
         end if
!-----------------------------------------------------------------------
! 	... Initialize dq
!-----------------------------------------------------------------------
         do j = jle(5),jue(5)
#ifdef DEBUG
	    write(*,*) 'advectxy : @ j,k,ic = ',j,k,ic
	    write(*,*) 'mmr'
	    write(*,'(1p,10e15.8)') mmr(:,j,k,ic)
	    write(*,*) 'delp1'
	    write(*,'(1p,10e15.8)') delp1(:,j,k)
	    if( any( mmr(:plon,j,k,ic) == bad_mem ) ) then
	       write(*,*) 'advectxy : some mmr bad at j,k,ic = ',j,k,ic
	       call endrun
	    end if
	    if( any( delp1(:plon,j,k) == bad_mem ) ) then
	       write(*,*) 'advectxy : some delp1 bad at j,k = ',j,k
	       call endrun
	    end if
#endif
            dq(:plon,j,k) = mmr(:plon,j,k,ic)*delp1(:plon,j,k)
         end do

!-----------------------------------------------------------------------
! 	... East-West advective cross term
!-----------------------------------------------------------------------
         call XADV( mmr(1,-3,k,ic), u(1,-3,k), wk1, k, ic, platl )
         do j = jle(7),jue(7)
            wk1(:plon,j) = mmr(:plon,j,k,ic) + .5*wk1(:plon,j)
         end do
 
!-----------------------------------------------------------------------
! 	... North-South advective cross term
!-----------------------------------------------------------------------
         do j = jl(4),ju(4)
            do i = 1,plon
               jt = jtt(i,j,k)
               wk2(i,j) = .5*(v(i,j,k) * (mmr(i,jt,k,ic) - mmr(i,jt+1,k,ic))) + mmr(i,j,k,ic)
            end do
         end do

!-----------------------------------------------------------------------
! 	... Flux in  East-West direction
!-----------------------------------------------------------------------
         call XTP( k, pu(1,-3,k), dq(1,-1,k),  &
                   wk2, crx(1,-3,k), fx(1,-1,k), fxp(1,-1,k), xmass(1,-3,k), &
		   platl )
!-----------------------------------------------------------------------
! 	... Flux in  North-South direction
!-----------------------------------------------------------------------
         call YTP( ic, k, dq(1,-1,k), wk1, &
                   cry(1,-2,k), ymass(1,-2,k), fyp(1,-2,k), fy(1,-2,k), &
		   platl )
      end do

!-----------------------------------------------------------------------
! 	... Save intermediate flux
!-----------------------------------------------------------------------
      do file = 1,moz_file_cnt
         m = DO_FLUX( ic, file )
         if( m > 0 ) then
	   if( masternode ) then
	      fxp(:,:j1-1,:) = 0.
	      fyp(:,:j1-1,:) = 0.
	   end if
	   if( lastnode ) then
	      fxp(:,j2+1:platl,:) = 0.
	      fyp(:,j2+1:platl,:) = 0.
	   end if
	   call HIST_XFLUX( m, fxp, .true., file, plonl, platl, pplon )
	   do j = 1,platl
	      hflx(:plon,j,:) = fyp(:plon,j,:)*acosp(j+base_lat)
	   end do
	   call HIST_YFLUX( m, hflx, .true., file, plonl, platl, pplon )
         end if
      end do

!-----------------------------------------------------------------------
! 	... Limiting value used in later routine FZPPM
!-----------------------------------------------------------------------
      do j = 1,platl
         wk1(:plon,j) = dq(:plon,j,1)/delp(:plon,j,1)
      end do
      Tmin = MINVAL( wk1(:plon,1:platl) )
      Tmax = MAXVAL( wk1(:plon,1:platl) )
      do j = 1,platl
         wk1(:plon,j) = dq(:plon,j,plev)/delp(:plon,j,plev)
      end do
      Bmin = MINVAL( wk1(:plon,1:platl) )
      Bmax = MAXVAL( wk1(:plon,1:platl) )

      end subroutine ADVECTXY

      subroutine ADVECTZ( ic, delp, delp2, u, w, &
                          dq, fx, fy, &
			  Tmin, Tmax, Bmin, Bmax, plonl, &
			  platl, pplon )
!-----------------------------------------------------------------------
!	... Vertical advection
!-----------------------------------------------------------------------

      use MO_MPI, only     :  masternode, lastnode, base_lat
      use MO_GRID, only    :  plon, plev, plevp, plat
      use MASS_DIAGS, only :  DO_FLUX, HIST_XFLUX, HIST_YFLUX, HIST_VFLUX
      use MO_HISTOUT, only :  moz_file_cnt

      implicit none

!-----------------------------------------------------------------------
!	... Dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: ic
      integer, intent(in) :: plonl, platl, pplon
      real, intent(in)    :: Tmin, Tmax, Bmin, Bmax
      real, dimension(plon,-1:platl+2,plev), intent(inout) :: dq
      real, dimension(plon,-3:platl+4,plev), intent(in)    :: u
      real, dimension(plon,-1:platl+2,plev), intent(in)    :: w
      real, dimension(plon,-3:platl+4,plev), intent(in)    :: delp, delp2
      real, dimension(plon+1,-1:platl+2,plev), intent(inout) :: fx
      real, dimension(plon,-2:platl+3,plev), intent(inout)   :: fy

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: j, jb, k, m, file
      real :: sum1, sum2, bad_mem
      real :: fz(plon,0:platl+1,plevp)
      real :: fzp(plon,platl,plevp)
      real :: qz(plon,-1:platl+2,plev)
      real :: qlow(plon,-1:platl+2,plev)

#ifdef INIT_TEST
      bad_mem     = z'7ff7ffff7ff7ffff'
      fz(:,:,:)   = bad_mem
      qz(:,:,:)   = bad_mem
      qlow(:,:,:) = bad_mem
#endif

      if( zcross ) then
!-----------------------------------------------------------------------
! 	... qz is the horizontal advection modified value for input to the
!           vertical transport operator FZPPM
!           Note: DQ contains only first order upwind contribution.
!-----------------------------------------------------------------------
         do k = 1,plev
            do j = jl(1),ju(1)
               qz(:plon,j,k) = dq(:plon,j,k) / delp(:plon,j,k)
            end do
         end do
      else
         do k = 1,plev
            do j = jl(1),ju(1)
               qz(:plon,j,k) = mmr(:plon,j,k,ic)
            end do
         end do
      end if
 
!-----------------------------------------------------------------------
! 	... Flux in the vertical direction
!-----------------------------------------------------------------------
      call FZPPM( qz, fz, fzp, dq, w, &
                  delp, Tmin, Tmax, Bmin, Bmax, platl )
 
      if( mfct ) then
!-----------------------------------------------------------------------
! 	... qlow is the low order "monotonic" solution
!-----------------------------------------------------------------------
         do k = 1,plev
            do j = jl(1),ju(1)
               qlow(:plon,j,k) = dq(:plon,j,k) / delp2(:plon,j,k)
            end do
            if( large_cap ) then
	       if( masternode ) then
		  jb = 1 - base_lat
                  qlow(:plon,jb+1,k) = qlow(:plon,jb,k)
	       end if
	       if( lastnode ) then
		  jb = plat - base_lat
                  qlow(:plon,jb-1,k) = qlow(:plon,jb,k)
	       end if
            end if
         end do
         call FCT3D( mmr(1,-3,1,ic), qlow, fx, fy, fz, &
		     delp2, u, ic, platl )
      end if

!-----------------------------------------------------------------------
! 	... Final update
!-----------------------------------------------------------------------
      do k = 1,plev
         do j = jl(5),ju(5)
            dq(:plon,j,k) = dq(:plon,j,k) +  fx(:plon,j,k) - fx(2:plon+1,j,k) &
                          + (fy(:plon,j,k) - fy(:plon,j+1,k))*acosp(j+base_lat) &
                          +  fz(:plon,j,k) - fz(:plon,j,k+1)
         end do
!-----------------------------------------------------------------------
! 	... Poles
!-----------------------------------------------------------------------
	 if( masternode ) then
	    jb = j1 - base_lat
            sum1 = SUM( fy(:plon,jb,k) )
	    jb = 1 - base_lat
            dq(1,jb,k) = dq(1,jb,k) - sum1*rcap + fz(1,jb,k) - fz(1,jb,k+1)
            dq(2:plon,jb,k) = dq(1,jb,k)
	 end if
	 if( lastnode ) then
	    jb = jlim_north + 1 - base_lat
            sum2 = SUM( fy(:plon,jb,k) )
	    jb = plat - base_lat
            dq(1,jb,k) = dq(1,jb,k) + sum2*rcap + fz(1,jb,k) - fz(1,jb,k+1)
            dq(2:plon,jb,k) = dq(1,jb,k)
	 end if
      end do
 
      do k = 1,plev
         do j = jl(5),ju(5)
            mmr(:plon,j,k,ic) = dq(:plon,j,k) / delp2(:plon,j,k)
         end do
	 if( masternode ) then
	    jb = 1 - base_lat
            mmr(:plon,jb,k,ic) = dq(:plon,jb,k) / delp2(:plon,jb,k)
            if( large_cap ) then
               mmr(:plon,jb+1,k,ic) = mmr(:plon,jb,k,ic)
	    end if
	 end if
	 if( lastnode ) then
	    jb = plat - base_lat
            mmr(:plon,jb,k,ic) = dq(:plon,jb,k) / delp2(:plon,jb,k)
            if( large_cap ) then
               mmr(:plon,jb-1,k,ic) = mmr(:plon,jb,k,ic)
	    end if
	 end if
      end do     

!-----------------------------------------------------------------------
! 	... Save final flux
!-----------------------------------------------------------------------
      do file = 1,moz_file_cnt
         m = DO_FLUX( ic, file )
         if( m > 0 ) then
	   if( masternode ) then
	      fx(:,:j1-1,:) = 0.
	      fy(:,:j1-1,:) = 0.
	   end if
	   if( lastnode ) then
	      fx(:,j2+1:platl,:) = 0.
	      fy(:,j2+1:platl,:) = 0.
	   end if
	   call HIST_XFLUX( m, fx, .false., file, plonl, platl, pplon )
	   do j = 1,platl
	      fy(:plon,j,:) = fy(:plon,j,:)*acosp(j+base_lat)
	   end do
	   call HIST_YFLUX( m, fy, .false., file, plonl, platl, pplon )
	   fzp(:plon,1:platl,:plevp) = fz(:plon,1:platl,:plevp) + fzp(:plon,1:platl,:plevp)
	   call HIST_VFLUX( m, fzp, file, plonl, platl, pplon )
         end if
      end do

      end subroutine ADVECTZ

      subroutine XADV( p, ua, adx, k, ic, platl )

      use MO_MPI,  only :  base_lat
      use MO_GRID, only : plon, plat

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: k, ic
      integer, intent(in) :: platl
      real, intent(in)    :: p(plon,-3:platl+4), &
                             ua(plon,-3:platl+4)
      real, intent(out)   :: adx(plon,-3:platl+4)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: i, indx, iu, iiu, ip, j, jglob, jmr, jb
      real :: ru, bad_mem
      real :: qtmp(-plon:2*plon)

#ifdef INIT_TEST
      bad_mem  = z'7ff7ffff7ff7ffff'
      adx(:,:) = bad_mem
      qtmp(:)  = bad_mem
#endif
      if( large_cap ) then
	 indx = 6
      else
	 indx = 1
      endif
      do j = jle(indx),jue(indx)
	 jglob = base_lat + j
         if( jglob > js(k) .and. jglob < jn(k) ) then
!-----------------------------------------------------------------------
! 	... Eulerian upwind
!-----------------------------------------------------------------------
            qtmp(1:plon) = p(:plon,j)
            qtmp(0)     = p(plon,j)
            qtmp(plon+1) = p(1,j)
            do i = 1,plon
               ip = INT( REAL(i) - ua(i,j) )
               adx(i,j) = ua(i,j)*(qtmp(ip) - qtmp(ip+1))
            end do
	 else
            qtmp(1:plon) = p(:plon,j)
            do i = -iml,0
               qtmp(i)       = p(plon+i,j)
               qtmp(plon+1-i) = p(1-i,j)
            end do

            do i = 1,plon
               iu = INT( ua(i,j) )
               ru = ua(i,j) - REAL(iu)
               iiu = i - iu
               if( ua(i,j) >= 0. ) then
                  adx(i,j) = qtmp(iiu) + ru*(qtmp(iiu-1) - qtmp(iiu))
               else
                  adx(i,j) = qtmp(iiu) + ru*(qtmp(iiu) - qtmp(iiu+1))
               end if
            end do
            adx(:plon,j) = adx(:plon,j) - p(:plon,j)
         end if
      end do

!-----------------------------------------------------------------------
! 	... Set cross term due to x-adv adjacent to poles to zero
!-----------------------------------------------------------------------

      if( large_cap ) then
	 if( has_spole ) then
	    jb = 2 - base_lat
            adx(:plon,jb) = 0.
	 end if
	 if( has_npole ) then
	    jb = plat - 1 - base_lat
            adx(:plon,jb) = 0.
	 end if
      end if

!-----------------------------------------------------------------------
! 	... Set cross term due to x-adv at the poles to zero
!-----------------------------------------------------------------------
      if( has_spole ) then
	 jb = 1 - base_lat
         adx(:plon,jb) = 0.
      end if
      if( has_npole ) then
	 jb = plat - base_lat
         adx(:plon,jb) = 0.
      end if

      end subroutine XADV

      subroutine XMIST( p, dc )

      use MO_GRID, only : plon

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      real, intent(in)    :: p(-iml:plon+1+iml)
      real, intent(inout) :: dc(-iml:plon+1+iml)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i
      real :: tmp, Pmin, Pmax

!-----------------------------------------------------------------------
! 	... 2nd order version
!-----------------------------------------------------------------------
      do i = 1,plon
         tmp = .25*(p(i+1) - p(i-1))
         Pmax = MAX( p(i-1), p(i), p(i+1) ) - p(i)
         Pmin = p(i) - MIN( p(i-1), p(i), p(i+1) )
         dc(i) = SIGN( MIN( ABS(tmp),Pmax,Pmin ),tmp )
      end do

      end subroutine XMIST

      subroutine YMIST( k, ic, p, dc, platl )

      use MO_GRID, only : plon, plat
      use MO_MPI,  only : masternode, lastnode, base_lat

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: k, ic
      integer, intent(in) :: platl
      real, intent(in)  :: p(plon,-3:platl+4)
      real, intent(out) :: dc(plon,-2:platl+3)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: j, jm1, jp1, jb
      real, dimension(plon) :: tmp, Pmax, Pmin

#ifdef INIT_TEST
      dc(:,:) = z'7ff7ffff7ff7ffff'
#endif
!-----------------------------------------------------------------------
! 	... 2nd order version for scalars
!-----------------------------------------------------------------------
      do j = jle(2),jue(2)
	 jm1 = j - 1
	 jp1 = j + 1
         tmp(:)  = .25*(p(:,jp1) - p(:,jm1))
         Pmax(:) = MAX( p(:,jm1),p(:,j),p(:,jp1) ) - p(:,j)
         Pmin(:) = p(:,j) - MIN( p(:,jm1),p(:,j),p(:,jp1) )
         dc(:,j) = SIGN( MIN( ABS(tmp(:)),Pmin(:),Pmax(:) ),tmp(:) )
      end do

!-----------------------------------------------------------------------
! 	... Poles
!-----------------------------------------------------------------------
      if( large_cap ) then
	 if( masternode ) then
	    jb = 1 - base_lat
            dc(:plon,jb) = 0.
	 end if
	 if( lastnode ) then
	    jb = plat - base_lat
            dc(:plon,jb) = 0.
	 end if
      end if

      end subroutine YMIST

      subroutine LMTPPM( dc, a6, ar, al, p, lmt )
!-----------------------------------------------------------------------
! 	... a6 =  CURVATURE OF THE TEST PARABOLA
!           ar =  RIGHT EDGE VALUE OF THE TEST PARABOLA
!           al =  LEFT  EDGE VALUE OF THE TEST PARABOLA
!           dc =  0.5 * MISMATCH
!           p  =  CELL-AVERAGED VALUE
!
!           Options:
!
!           lmt = 0: FULL MONOTONICITY
!           lmt = 1: SEMI-MONOTONIC CONSTRAINT (NO UNDERSHOOTS)
!           lmt = 2: POSITIVE-DEFINITE CONSTRAINT
!-----------------------------------------------------------------------

      use MO_GRID, only : plon

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lmt
      real, dimension(plon), intent(in)    :: p, dc
      real, dimension(plon), intent(inout) :: a6, ar, al

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: R12 = 1./12.

      integer :: i
      real :: da1, da2, a6da, fmin

      if( lmt == 0 ) then
!-----------------------------------------------------------------------
! 	... Full constraint
!-----------------------------------------------------------------------
         do i = 1,plon
            if( dc(i) == 0. ) then
               ar(i) = p(i)
               al(i) = p(i)
               a6(i) = 0.
            else
               da1  = ar(i) - al(i)
               da2  = da1**2
               a6da = a6(i)*da1
               if( a6da < -da2 ) then
                  a6(i) = 3.*(al(i) - p(i))
                  ar(i) = al(i) - a6(i)
               else if( a6da > da2 ) then
                  a6(i) = 3.*(ar(i) - p(i))
                  al(i) = ar(i) - a6(i)
               end if
            end if
         end do
      else if( lmt == 1 ) then
!-----------------------------------------------------------------------
! 	... Semi-monotonic constraint
!-----------------------------------------------------------------------
         do i = 1,plon
            if( ABS( ar(i) - al(i) ) >=  -a6(i) ) then
	       cycle
	    end if
            if( p(i) < ar(i) .and. p(i) < al(i) ) then
               ar(i) = p(i)
               al(i) = p(i)
               a6(i) = 0.
            else if( ar(i) > al(i) ) then
               a6(i) = 3.*(al(i) - p(i))
               ar(i) = al(i) - a6(i)
            else
               a6(i) = 3.*(ar(i) - p(i))
               al(i) = ar(i) - a6(i)
            end if
         end do
      else if( lmt == 2 ) then
         do i = 1,plon
            if( ABS( ar(i) - al(i) ) >=  -a6(i) ) then
	       cycle
	    end if
            fmin = p(i) + .25*(ar(i) - al(i))**2/a6(i) + a6(i)*R12
            if( fmin >= 0. ) then
	       cycle
	    end if
            if( p(i) < ar(i) .and. p(i) < al(i) ) then
               ar(i) = p(i)
               al(i) = p(i)
               a6(i) = 0.
            else if( ar(i) > al(i) ) then
               a6(i) = 3.*(al(i) - p(i))
               ar(i) = al(i) - a6(i)
            else
               a6(i) = 3.*(ar(i) - p(i))
               al(i) = ar(i) - a6(i)
            end if
         end do
      end if

      end subroutine LMTPPM

      subroutine HILO( q, qmax, qmin, flag, jl, ju, platl )

      use MO_MPI,  only :  masternode, lastnode, base_lat
      use MO_GRID, only : plon, plat

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: flag, jl, ju
      integer, intent(in) :: platl
      real, intent(in)    :: q(plon,-(2*flag+1):platl+2*(flag+1))
      real, intent(out)   :: Qmax(plon,-flag:platl+flag+1)
      real, intent(out)   :: Qmin(plon,-flag:platl+flag+1)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, im1, jb, jb1
      real    :: Pmin, Pmax, bad_mem
      real    :: bt(plon,-flag:platl+flag+1), bd(plon,-flag:platl+flag+1)
 
#ifdef INIT_TEST
      bad_mem   = z'7ff7ffff7ff7ffff'
      bt(:,:)   = bad_mem
      bd(:,:)   = bad_mem
      Qmax(:,:) = bad_mem
      Qmin(:,:) = bad_mem
#endif
!-----------------------------------------------------------------------
! 	... Y-sweep
!-----------------------------------------------------------------------
      do j = jl,ju
         bt(:plon,j) = MAX( q(:plon,j-1),q(:plon,j),q(:plon,j+1) )
         bd(:plon,j) = MIN( q(:plon,j-1),q(:plon,j),q(:plon,j+1) )
      end do
!-----------------------------------------------------------------------
! 	... X-sweep
!-----------------------------------------------------------------------
      im1 = plon - 1
      do j = jl,ju
         do i = 2,im1
            Qmax(i,j) = MAX( bt(i-1,j),bt(i,j),bt(i+1,j) )
            Qmin(i,j) = MIN( bd(i-1,j),bd(i,j),bd(i+1,j) )
         end do
      end do

      do j = jl,ju
!-----------------------------------------------------------------------
!     	... i = 1
!-----------------------------------------------------------------------
         Qmax(1,j) = MAX( bt(plon,j),bt(1,j),bt(2,j) )
         Qmin(1,j) = MIN( bd(plon,j),bd(1,j),bd(2,j) )
!-----------------------------------------------------------------------
!     	... i = im
!-----------------------------------------------------------------------
         Qmax(plon,j) = MAX( bt(im1,j),bt(plon,j),bt(1,j) )
         Qmin(plon,j) = MIN( bd(im1,j),bd(plon,j),bd(1,j) )
      end do
!-----------------------------------------------------------------------
! 	... North Pole
!-----------------------------------------------------------------------
      if( lastnode ) then
	 jb  = plat - base_lat
	 jb1 = jlim_north - base_lat
         Pmax = MAX( q(1,jb),MAXVAL( q(:,jb1) ) )
         Pmin = MIN( q(1,jb),MINVAL( q(:,jb1) ) )
         Qmax(:plon,jb) = Pmax
         Qmin(:plon,jb) = Pmin
         if( large_cap ) then
	    jb  = plat - 1 - base_lat
            Qmax(:plon,jb) = Qmax(:plon,jb+1)
            Qmin(:plon,jb) = Qmin(:plon,jb+1)
         end if
      end if

!-----------------------------------------------------------------------
! 	... South Pole
!-----------------------------------------------------------------------
      if( masternode ) then
	 jb  = 1 - base_lat
	 jb1 = j1 - base_lat
         Pmax = MAX( q(1,jb),MAXVAL( q(:,jb1) ) )
         Pmin = MIN( q(1,jb),MINVAL( q(:,jb1) ) )
         Qmax(:plon,jb) = Pmax
         Qmin(:plon,jb) = Pmin
         if( large_cap ) then
	    jb  = 1 - base_lat
            Qmax(:plon,jb+1) = Qmax(:plon,jb)
            Qmin(:plon,jb+1) = Qmin(:plon,jb)
         end if
      end if

      end subroutine HILO

      subroutine HILO3D( p, Pmax, Pmin, flag, platl )
 
      use MO_GRID, only : plon, plev

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: flag
      integer, intent(in) :: platl
      real, intent(in)    :: p(plon,-(2*flag+1):platl+2*(flag+1),plev)
      real, intent(out)   :: Pmax(plon,-flag:platl+1+flag,plev)
      real, intent(out)   :: Pmin(plon,-flag:platl+1+flag,plev)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: j, k, km1, km2, ji, jl_loc, ju_loc
      real    :: bad_mem
      real    :: Qmax(plon,-flag:platl+flag+1,plev)
      real    :: Qmin(plon,-flag:platl+flag+1,plev)
 
#ifdef INIT_TEST
      bad_mem     = z'7ff7ffff7ff7ffff'
      Pmax(:,:,:) = bad_mem
      Pmin(:,:,:) = bad_mem
#endif

      km1 = plev - 1
      km2 = plev - 2
      jl_loc = jle(flag+12)
      ju_loc = jue(flag+12)
      ji = -(2*flag+1)

      do k = 1,plev
         call HILO( p(1,ji,k), Qmax(1,-flag,k), Qmin(1,-flag,k), flag, jl_loc, ju_loc, platl )
      end do

      jl_loc = jl(flag+9)
      ju_loc = ju(flag+9)

      do j = jl_loc,ju_loc
!-----------------------------------------------------------------------
! 	... k=1 and k=plev
!-----------------------------------------------------------------------
         Pmax(:plon,j, 1) = MAX( Qmax(:plon,j,2),Qmax(:plon,j,1) )
         Pmin(:plon,j, 1) = MIN( Qmin(:plon,j,2),Qmin(:plon,j,1) )
         Pmax(:plon,j,plev) = MAX( Qmax(:plon,j,km1),Qmax(:plon,j,plev) )
         Pmin(:plon,j,plev) = MIN( Qmin(:plon,j,km1),Qmin(:plon,j,plev) )
!-----------------------------------------------------------------------
! 	... k=2 and k=km1
!-----------------------------------------------------------------------
         Pmax(:plon,j,  2) = MAX( Qmax(:plon,j,3),Pmax(:plon,j,1) )
         Pmin(:plon,j,  2) = MIN( Qmin(:plon,j,3),Pmin(:plon,j,1) )
         Pmax(:plon,j,km1) = MAX( Qmax(:plon,j,km2),Pmax(:plon,j,plev) )
         Pmin(:plon,j,km1) = MIN( Qmin(:plon,j,km2),Pmin(:plon,j,plev) )
      end do
 
      do k = 3,km2
         do j = jl_loc,ju_loc
            Pmax(:plon,j,k) = MAX( Qmax(:plon,j,k-1),Qmax(:plon,j,k),Qmax(:plon,j,k+1) )
            Pmin(:plon,j,k) = MIN( Qmin(:plon,j,k-1),Qmin(:plon,j,k),Qmin(:plon,j,k+1) )
         end do
      end do

      end subroutine HILO3D

      subroutine FXPPM( ut, p, dc, fx1, fx2 )

      use MO_GRID,      only : plon

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      real, intent(out) :: fx1(plon+1), fx2(plon+1)
      real, intent(in)  :: ut(plon)
      real, intent(in)  :: p(-iml:plon+iml+1), dc(-iml:plon+iml+1)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: R3 = 1./3., R23 = 2./3. 

      integer :: i, lmt
      real    :: bad_mem
      real, dimension(-1:plon) :: ar, al, a6

#ifdef INIT_TEST
      bad_mem = z'7ff7ffff7ff7ffff'
      ar(:)   = bad_mem
      al(:)   = bad_mem
      a6(:)   = bad_mem
      fx1(:)  = bad_mem
      fx2(:)  = bad_mem
#endif

      lmt = iord - 3

      al(1:plon)   = .5*(p(0:plon-1) + p(1:plon)) + (dc(0:plon-1) - dc(1:plon))*R3
      ar(1:plon-1) = al(2:plon)
      ar(plon)     = al(1)
      a6(1:plon)   = 3.*(2.*p(1:plon) - (al(1:plon) + ar(1:plon)))

      if( lmt <= 2 ) then
         call LMTPPM( dc(1), a6(1), ar(1), al(1), p(1), lmt )
      end if

      al(0) = al(plon)
      ar(0) = ar(plon)
      a6(0) = a6(plon)

!-----------------------------------------------------------------------
! 	... Abs(UT(i)) < 1
!-----------------------------------------------------------------------
      do i = 1,plon
         if( ut(i) > 0. ) then
            fx1(i) = p(i-1)
            fx2(i) = ar(i-1) + .5*ut(i)*(al(i-1) - ar(i-1) + a6(i-1)*(1. - R23*ut(i)))
         else
            fx1(i) = p(i)
            fx2(i) = al(i) - .5*ut(i)*(ar(i) - al(i) + a6(i)*(1. + R23*ut(i)))
         end if
      end do

      do i = 1,plon
         fx2(i) = fx2(i) - fx1(i)
      end do

      end subroutine FXPPM

      subroutine FYPPM( k, ic, c, p, dc, &
			fy1, fy2, platl )

      use MO_GRID, only : plon, plat
      use MO_MPI,  only : masternode, lastnode, base_lat

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: k, ic
      integer, intent(in) :: platl
      real, intent(in)  :: c(plon,-2:platl+3), &
                           p(plon,-3:platl+4), &
			   dc(plon,-2:platl+3)
      real, intent(out) :: fy1(plon,-2:platl+3), &
                           fy2(plon,-2:platl+3)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: R3 = 1./3., R23 = 2./3.

      integer :: i, j, jm1, jb, lmt
      real    :: bad_mem
      real, dimension(plon,-1:platl+2) :: ar, al, a6

#ifdef INIT_TEST
      bad_mem  = z'7ff7ffff7ff7ffff'
      fy1(:,:) = bad_mem
      fy2(:,:) = bad_mem
      ar(:,:)  = bad_mem
      al(:,:)  = bad_mem
      a6(:,:)  = bad_mem
#endif

      lmt     = jord - 3

      do j = jle(11),jue(11)
         jm1 = j - 1
         al(:plon,j) = .5*(p(:plon,jm1) + p(:plon,j)) + (dc(:plon,jm1) - dc(:plon,j))*r3
      end do
      do j = jle(11)+1,jue(11)
         jm1 = j - 1
         ar(:plon,jm1) = al(:plon,j)
      end do
      ar(:plon,jue(11)) = .5*(p(:plon,jue(11)) + p(:plon,jue(11)+1)) &
                          + (dc(:plon,jue(11)) - dc(:plon,jue(11)+1))*r3

!-----------------------------------------------------------------------
! 	... Poles
!-----------------------------------------------------------------------
      if( masternode ) then
	 jb = 1 - base_lat
         do i = 1,imh
            al(i,jb) = al(i+imh,jb+1)
            al(i+imh,jb) = al(i,jb+1)
         end do
      end if
      if( lastnode ) then
	 jb = plat - base_lat
         do i = 1,imh
            ar(i,jb) = ar(i+imh,jb-1)
            ar(i+imh,jb) = ar(i,jb-1)
         end do
      end if

      do j = jle(11),jue(11)
         a6(:plon,j) = 3.*(2.*p(:plon,j) - (al(:plon,j) + ar(:plon,j)))
      end do

      if( lmt <= 2 ) then
	 do j = jle(11),jue(11)
	    call LMTPPM( dc(1,j), a6(1,j), ar(1,j), al(1,j), p(1,j), lmt )
	 end do
      end if

      do j = jle(9),jue(9)
	 jm1 = j - 1
         where( c(:plon,j) > 0. )
            fy1(:plon,j) = p(:plon,jm1)
         elsewhere
            fy1(:plon,j) = p(:plon,j)
         endwhere
      end do
      do j = jle(10),jue(10)
	 jm1 = j - 1
         where( c(:plon,j) > 0. )
            fy2(:plon,j) = ar(:plon,jm1) &
                          + .5*c(:plon,j)*(al(:plon,jm1) - ar(:plon,jm1) &
                                            + a6(:plon,jm1)*(1. - r23*c(:plon,j)))
         elsewhere
            fy2(:plon,j) = al(:plon,j) &
                          - .5*c(:plon,j)*(ar(:plon,j) - al(:plon,j) &
                                            + a6(:plon,j)*(1. + r23*c(:plon,j)))
         endwhere
         fy2(:plon,j) = fy2(:plon,j) - fy1(:plon,j)
      end do

      end subroutine FYPPM

      subroutine FCT3D( p, plow, fx, fy, fz, &
			delp, crx, ic, platl )
!-----------------------------------------------------------------------
!	... MFCT Limiter
!           plow: low order solution matrix
!           p: current solution matrix
!-----------------------------------------------------------------------

      use MO_MPI,  only : masternode, lastnode, base_lat
      use MO_GRID, only : plon, plat, plev, plevp

      implicit none
 
!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: ic
      integer, intent(in) :: platl
      real, intent(in) :: &
              p(plon,-3:platl+4,plev), &
              plow(plon,-1:platl+2,plev), &
              crx(plon,-3:platl+4,plev), &
              delp(plon,-3:platl+4,plev)
      real, intent(inout) :: &
              fx(plon+1,-1:platl+2,plev), &
              fy(plon,-2:platl+3,plev), &
              fz(plon,0:platl+1,plevp)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: esl = 1.e-30

      integer ::  i, it, j, jt, jb,  k, flag
      real    ::  wkx, wkn, btop, bdon
      real    ::  ps1, ps2, pn1, pn2
      real    ::  Ain, Aou, Bin, Bou, Cin, Cou
      real, allocatable ::  Qmax(:,:,:), Qmin(:,:,:), adx(:,:,:), ady(:,:,:)
      integer :: astat

      ALLOCATE( Qmax(plon,0:platl+1,plev), &
                Qmin(plon,0:platl+1,plev), &
                adx(plon,-1:platl+2,plev), &
		ady(plon,-1:platl+2,plev), &
                stat=astat )
      if( astat /= 0 ) then
         write(*,*)  &
            'FCT3D: failed to allocate local arrays; error code = ',astat
         call ENDRUN
      end if

!-----------------------------------------------------------------------
! 	... Find local min/max of the low-order monotone solution
!-----------------------------------------------------------------------
      call HILO3D( p, adx, ady, 1, platl )
      call HILO3D( plow, Qmax, Qmin, 0, platl )
 
      do k = 1,plev
         do j = jle(12),jue(12)
            do i = 1,plon
               it = NINT( REAL(i) - crx(i,j,k) )
!-----------------------------------------------------------------------
! 	... Wrap around in E-W
!-----------------------------------------------------------------------
               if( it < 1 ) then
                  it = plon + it
               else if( it  >  plon ) then
                  it = it - plon
               end if
               jt = jtn(i,j,k)
               Qmax(i,j,k) = MAX( Qmax(i,j,k), adx(it,jt,k) )
               Qmin(i,j,k) = MIN( Qmin(i,j,k), ady(it,jt,k) )
            end do
         end do
!-----------------------------------------------------------------------
! 	... Poles
!-----------------------------------------------------------------------
	 if( masternode ) then
	    jb = 1 - base_lat
            ps1 = MAX( Qmax(1,jb,k), adx(1,jb,k) )
            ps2 = MIN( Qmin(1,jb,k), ady(1,jb,k) )
            Qmax(:plon,jb,k) = ps1
            Qmin(:plon,jb,k) = ps2
	 end if
	 if( lastnode ) then
	    jb = plat - base_lat
            pn1 = MAX( Qmax(1,jb,k), adx(1,jb,k) )
            pn2 = MIN( Qmin(1,jb,k), ady(1,jb,k) )
            Qmax(:plon,jb,k) = pn1
            Qmin(:plon,jb,k) = pn2
	 end if
      end do

 
!-----------------------------------------------------------------------
! 	... Flux Limiter
!-----------------------------------------------------------------------
      do k = 1,plev
         do j = jle(12),jue(12)
            do i = 1,plon
               if( fx(i,j,k) > 0. ) then
                  Ain = fx(i,j,k)
                  Aou = 0.
               else
                  Ain = 0.
                  Aou = -fx(i,j,k)
               end if
 
               if( fx(i+1,j,k) > 0. ) then
                  Aou = Aou + fx(i+1,j,k)
               else
                  Ain = Ain - fx(i+1,j,k)
               end if
 
               if( fy(i,j,k) > 0. ) then
                  Bin = fy(i,j,k)
                  Bou = 0.
               else
                  Bin = 0.
                  Bou = -fy(i,j,k)
               end if
 
               if( fy(i,j+1,k) > 0. ) then
                  Bou = Bou + fy(i,j+1,k)
               else
                  Bin = Bin - fy(i,j+1,k)
               end if
 
               if( fz(i,j,k) > 0. ) then
                  Cin = fz(i,j,k)
                  Cou = 0.
               else
                  Cin = 0.
                  Cou = -fz(i,j,k)
               end if
 
               if( fz(i,j,k+1) > 0. ) then
                  Cou = Cou + fz(i,j,k+1)
               else
                  Cin = Cin - fz(i,j,k+1)
               end if
 
               wkx        = Ain + Bin*acosp(j+base_lat) + Cin
               wkn        = Aou + Bou*acosp(j+base_lat) + Cou
               adx(i,j,k) = delp(i,j,k)*(Qmax(i,j,k) - plow(i,j,k))/(wkx + esl)
               ady(i,j,k) = delp(i,j,k)*(plow(i,j,k) - Qmin(i,j,k))/(wkn + esl)
            end do
         end do
 
!-----------------------------------------------------------------------
! 	... South Pole
!-----------------------------------------------------------------------
	 if( masternode ) then
	    jb = j1 - base_lat
            Ain = 0.
            Aou = 0.
            do i = 1,plon
               if( fy(i,jb,k) > 0. ) then
                  Aou = Aou + fy(i,jb,k)
               else
                  Ain = Ain + fy(i,jb,k)
               end if
            end do
            Ain = -Ain * rcap
            Aou =  Aou * rcap
!-----------------------------------------------------------------------
! 	... Add vertical contribution
!-----------------------------------------------------------------------
	    jb = 1 - base_lat
            if( fz(1,jb,k) > 0. ) then
               Cin = fz(1,jb,k)
               Cou = 0.
            else
               Cin = 0.
               Cou = -fz(1,jb,k)
            end if
 
            if( fz(1,jb,k+1) > 0. ) then
               Cou = Cou + fz(1,jb,k+1)
            else
               Cin = Cin - fz(1,jb,k+1)
            end if
 
            btop = delp(1,jb,k)*(Qmax(1,jb,k) - plow(1,jb,k))/(Ain + Cin + esl)
            bdon = delp(1,jb,k)*(plow(1,jb,k) - Qmin(1,jb,k))/(Aou + Cou + esl)
            adx(:plon,jb,k) = btop
            ady(:plon,jb,k) = bdon
	 end if

!-----------------------------------------------------------------------
! 	... North Pole
!-----------------------------------------------------------------------
	 if( lastnode ) then
	    jb = jlim_north + 1 - base_lat
            Ain = 0.
            Aou = 0.
            do i = 1,plon
               if( fy(i,jb,k) > 0. ) then
                  Ain = Ain + fy(i,jb,k)
               else
                  Aou = Aou + fy(i,jb,k)
               end if
            end do
            Ain =  Ain * rcap
            Aou = -Aou * rcap
!-----------------------------------------------------------------------
! 	... Add vertical contribution
!-----------------------------------------------------------------------
	    jb = plat - base_lat
            if( fz(1,jb,k) > 0. ) then
               Cin = fz(1,jb,k)
               Cou = 0.
            else
               Cin = 0.
               Cou = -fz(1,jb,k)
            end if
 
            if( fz(1,jb,k+1) > 0. ) then
               Cou = Cou + fz(1,jb,k+1)
            else
               Cin = Cin - fz(1,jb,k+1)
            end if
 
            btop = delp(1,jb,k)*(Qmax(1,jb,k) - plow(1,jb,k))/(Ain + Cin + esl)
            bdon = delp(1,jb,k)*(plow(1,jb,k) - Qmin(1,jb,k))/(Aou + Cou + esl)
            adx(:plon,jb,k) = btop
            ady(:plon,jb,k) = bdon
	 end if
 
         if( large_cap ) then
!-----------------------------------------------------------------------
! 	... South Pole
!-----------------------------------------------------------------------
	    if( masternode ) then
	       jb = 1 - base_lat
               adx(:plon,jb+1,k) = adx(:plon,jb,k)
               ady(:plon,jb+1,k) = ady(:plon,jb,k)
	    end if
!-----------------------------------------------------------------------
! 	... North Pole
!-----------------------------------------------------------------------
	    if( lastnode ) then
	       jb = plat - base_lat
               adx(:plon,jb-1,k) = adx(:plon,jb,k)
               ady(:plon,jb-1,k) = ady(:plon,jb,k)
	    end if
         end if
      end do
 
!-----------------------------------------------------------------------
! 	... Correct the fluxes
!           First the zonal flux
!-----------------------------------------------------------------------
      do k = 1,plev
         do j = jl(5),ju(5)
            do i = 2,plon
               if( fx(i,j,k) > 0. ) then
                  fx(i,j,k) = MIN( 1.,ady(i-1,j,k),adx(i,j,k) )*fx(i,j,k)
               else
                  fx(i,j,k) = MIN( 1.,adx(i-1,j,k),ady(i,j,k) )*fx(i,j,k)
               end if
            end do
         end do
!-----------------------------------------------------------------------
! 	... For i=1
!-----------------------------------------------------------------------
         do j = jl(5),ju(5)
            if( fx(1,j,k) > 0. ) then
               fx(1,j,k) = MIN( 1.,ady(plon,j,k),adx(1,j,k) )*fx(1,j,k)
            else
               fx(1,j,k) = MIN( 1.,adx(plon,j,k),ady(1,j,k) )*fx(1,j,k)
            end if
            fx(plon+1,j,k) = fx(1,j,k)
         end do
 
!-----------------------------------------------------------------------
! 	... The meridional flux
!-----------------------------------------------------------------------
         do j = jl(5),ju(5)+1
            do i = 1,plon
               if( fy(i,j,k) > 0. ) then
                  fy(i,j,k) = MIN( 1.,ady(i,j-1,k),adx(i,j,k) )*fy(i,j,k)
               else
                  fy(i,j,k) = MIN( 1.,adx(i,j-1,k),ady(i,j,k) )*fy(i,j,k)
               end if
            end do
         end do

!-----------------------------------------------------------------------
! 	... The vertical flux
!-----------------------------------------------------------------------
         if( k /= 1 ) then
            do j = 1,platl
               do i = 1,plon
                  if( fz(i,j,k) > 0. ) then
                     fz(i,j,k) = MIN( 1.,ady(i,j,k-1),adx(i,j,k) )*fz(i,j,k)
                  else
                     fz(i,j,k) = MIN( 1.,adx(i,j,k-1),ady(i,j,k) )*fz(i,j,k)
                  end if
               end do
            end do
         end if
      end do
 
      DEALLOCATE( Qmax, Qmin, adx, ady, stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'FCT3D: failed to deallocate local arrays; error code = ',astat
         call ENDRUN
      end if

      end subroutine FCT3D
 
      subroutine FZPPM( p, fz, fzp, dq, wz, &
                        delp, Tmin, Tmax, Bmin, Bmax, platl )
!-----------------------------------------------------------------------
!       ... Vertical flux
!-----------------------------------------------------------------------
 
      use MO_GRID, only :      plon, plev, plevp

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)    :: Tmax, Tmin, Bmax, Bmin
      real, intent(in)    :: wz(plon,-1:platl+2,plev)
      real, intent(in)    :: p(plon,-1:platl+2,plev)
      real, intent(in)    :: delp(plon,-3:platl+4,plev)
      real, intent(inout) :: dq(plon,-1:platl+2,plev)
      real, intent(out)   :: fz(plon,0:platl+1,plevp)
      real, intent(out)   :: fzp(plon,platl,plevp)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: R23 = 2./3., R3 = 1./3.

      integer :: i, j, k, km1, lmt
      real :: a1, a2, c1, c2, c3, d1, d2, dp
      real :: cmin, cmax, qmin, qmax, qm, qmp
      real :: lac, tmp, bad_mem
      real, dimension(plon,plev) :: ar, al, a6, delq, dc, cf
 
#ifdef INIT_TEST
      bad_mem    = z'7ff7ffff7ff7ffff'
      fz(:,:,:)  = bad_mem
      fzp(:,:,:) = bad_mem
#endif

      km1 = plev - 1
      lmt = MAX( kord - 3,0 )

      do j = jl(1),ju(1)
#ifdef INIT_TEST
         ar(:,:) = bad_mem
         al(:,:) = bad_mem
         a6(:,:) = bad_mem
         delq(:,:) = bad_mem
         dc(:,:) = bad_mem
         cf(:,:) = bad_mem
#endif
         do k = 2,plev
            a6(:plon,k) = delp(:plon,j,k-1) + delp(:plon,j,k)
         end do

         do k = 1,km1
            delq(:plon,k) = p(:plon,j,k+1) - p(:plon,j,k)
         end do
 
         do k = 2,km1
            do i = 1,plon
               c1 = (delp(i,j,k-1) + .5*delp(i,j,k))/a6(i,k+1)
               c2 = (delp(i,j,k+1) + .5*delp(i,j,k))/a6(i,k)
               tmp = delp(i,j,k)*(c1*delq(i,k) + c2*delq(i,k-1)) / (a6(i,k) + delp(i,j,k+1))
               Qmax = MAX( p(i,j,k-1),p(i,j,k),p(i,j,k+1) ) - p(i,j,k)
               Qmin = p(i,j,k) - MIN( p(i,j,k-1),p(i,j,k),p(i,j,k+1) )
               dc(i,k) = SIGN( MIN( ABS(tmp),Qmax,Qmin ),tmp )
            end do
         end do
!-----------------------------------------------------------------------
! 	... Compute the first guess at cell interface
!           First guesses are required to be continuous.
!-----------------------------------------------------------------------
!           Interior
!-----------------------------------------------------------------------
         do k = 3,km1
            do i = 1,plon
               c1 = delq(i,k-1)*delp(i,j,k-1) / a6(i,k)
               a1 = a6(i,k-1) / (a6(i,k) + delp(i,j,k-1))
               a2 = a6(i,k+1) / (a6(i,k) + delp(i,j,k))
               al(i,k) = p(i,j,k-1) + c1 &
                       + 2./(a6(i,k-1) + a6(i,k+1))  &
                         * (delp(i,j,k)*(c1*(a1 - a2) + a2*dc(i,k-1)) - delp(i,j,k-1)*a1*dc(i,k))
            end do
         end do
!-----------------------------------------------------------------------
! 	... Area preserving cubic with 2nd deriv. = 0 at the boundaries
!           Top
!-----------------------------------------------------------------------
         do i = 1,plon
            d1 = delp(i,j,1)
            d2 = delp(i,j,2)
            qm = (d2*p(i,j,1) + d1*p(i,j,2)) / (d1 + d2)
            dp = 2.*(p(i,j,2) - p(i,j,1)) / (d1 + d2)
            c1 = 4.*(al(i,3) - qm - d2*dp) / (d2*(2.*d2*d2 + d1*(d2+3.*d1)))
            c3 = dp - .5*c1*(d2*(5.*d1 + d2) - 3.*d1**2)
            al(i,2) = qm - .25*c1*d1*d2*(d2 + 3.*d1)
            al(i,1) = d1*(2.*c1*d1**2 - c3) + al(i,2)
            dc(i,1) =  p(i,j,1) - al(i,1)
!-----------------------------------------------------------------------
! 	... No over- and undershoot condition
!-----------------------------------------------------------------------
            al(i,1) = MAX( Tmin,al(i,1) )
            al(i,1) = MIN( Tmax,al(i,1) )
            Cmax    = MAX( p(i,j,1),p(i,j,2) )
            Cmin    = MIN( p(i,j,1),p(i,j,2) )
            al(i,2) = MAX( Cmin,al(i,2) )
            al(i,2) = MIN( Cmax,al(i,2) )
         end do
 
!-----------------------------------------------------------------------
! 	... Bottom
!-----------------------------------------------------------------------
         do i = 1,plon
            d1 = delp(i,j,plev)
            d2 = delp(i,j,km1)
            qm = (d2*p(i,j,plev) + d1*p(i,j,km1)) / (d1 + d2)
            dp = 2.*(p(i,j,km1) - p(i,j,plev)) / (d1 + d2)
	    c1 = 4.*(al(i,km1) -(qm + d2*dp)) / (d2*(2.*d2*d2 + d1*(d2 + 3.*d1)))
	    c3 = dp - .5*c1*(d2*(5.*d1 + d2) - 3.*d1**2)
            al(i,plev) = qm - .25*c1*d1*d2*(d2 + 3.*d1)
            ar(i,plev) = d1*(2.*c1*d1**2 - c3) + al(i,plev)
            dc(i,plev) = ar(i,plev) -  p(i,j,plev)
!-----------------------------------------------------------------------
! 	... No over- and undershoot condition
!-----------------------------------------------------------------------
            Cmax       = MAX( p(i,j,plev),p(i,j,km1) )
            Cmin       = MIN( p(i,j,plev),p(i,j,km1) )
            al(i,plev) = MAX( Cmin,al(i,plev) )
            al(i,plev) = MIN( Cmax,al(i,plev) )
            ar(i,plev) = MAX( Bmin,ar(i,plev) )
            ar(i,plev) = MIN( Bmax,ar(i,plev) )
         end do

         do k = 1,km1
            ar(:plon,k) = al(:plon,k+1)
         end do
 
!-----------------------------------------------------------------------
! f(s) = AL + s*[(AR-AL) + A6*(1-s)]         ( 0 <= s  <= 1 )
! Top 2 layers
!-----------------------------------------------------------------------
         do k = 1,2
            a6(:plon,k) = 3.*(2.*p(:plon,j,k) - (al(:plon,k) + ar(:plon,k)))
            call LMTPPM( dc(1,k), a6(1,k), ar(1,k), al(1,k), p(1,j,k), 0 )
         end do

!-----------------------------------------------------------------------
! 	... Interior
!-----------------------------------------------------------------------
         if( lmt <= 2 ) then
            do k = 3,plev-2
               a6(:plon,k) = 3.*(2.*p(:plon,j,k) - (al(:plon,k) + ar(:plon,k)))
               call LMTPPM( dc(1,k), a6(1,k), ar(1,k), al(1,k), p(1,j,k), lmt )
            end do
         else if( lmt == 4 ) then
!-----------------------------------------------------------------------
! 	... Huynhs 2nd constraint
!-----------------------------------------------------------------------
            do k = 2,km1
               dc(:plon,k) = delq(:plon,k) - delq(:plon,k-1)
            end do

            do k = 3,plev-2
               do i = 1,plon
!-----------------------------------------------------------------------
! 	... Right edges
!-----------------------------------------------------------------------
                  qmp   = p(i,j,k)                 + 2.*delq(i,k-1)
                  lac   = p(i,j,k) + 1.5*dc(i,k-1) + .5*delq(i,k-1)
                  qmin  = MIN( p(i,j,k), qmp, lac )
                  qmax  = MAX( p(i,j,k), qmp, lac )
                  ar(i,k) = MIN( MAX( ar(i,k),qmin ),qmax )
!-----------------------------------------------------------------------
! 	... Left  edges
!-----------------------------------------------------------------------
                  qmp   = p(i,j,k)                 - 2.*delq(i,k)
                  lac   = p(i,j,k) + 1.5*dc(i,k+1) - .5*delq(i,k)
                  qmin  = MIN( p(i,j,k), qmp, lac )
                  qmax  = MAX( p(i,j,k), qmp, lac )
                  al(i,k) = MIN( MAX( al(i,k),qmin ),qmax )
!-----------------------------------------------------------------------
! 	... Recompute a6
!-----------------------------------------------------------------------
                  a6(i,k) = 3.*(2.*p(i,j,k) - (ar(i,k) + al(i,k)))
               end do
            end do
         end if

!-----------------------------------------------------------------------
! 	... Bottom 2 layers
!-----------------------------------------------------------------------
         do k = km1,plev
            a6(:plon,k) = 3.*(2.*p(:plon,j,k) - (al(:plon,k) + ar(:plon,k)))
            call LMTPPM( dc(1,k), a6(1,k), ar(1,k), al(1,k), p(1,j,k), 0 )
         end do
 
         do k = 2,plev
	    where( wz(:plon,j,k-1) > 0. )
               cf(:plon,k) = wz(:plon,j,k-1) / delp(:plon,j,k-1)
               dc(:plon,k) = p(:plon,j,k-1)
            elsewhere
               cf(:plon,k) = wz(:plon,j,k-1) / delp(:plon,j,k)
               dc(:plon,k) = p(:plon,j,k)
            endwhere
         end do
	 if( j >= 0 .and. j <= platl+1 ) then
            do k = 2,plev
	       where( wz(:plon,j,k-1) > 0. )
                  fz(:plon,j,k) = ar(:plon,k-1) &
                                + .5*cf(:plon,k)*(al(:plon,k-1) - ar(:plon,k-1) + a6(:plon,k-1)*(1. - R23*cf(:plon,k)))
               elsewhere
                  fz(:plon,j,k) = al(:plon,k) &
                                + .5*cf(:plon,k)*(al(:plon,k) - ar(:plon,k) - a6(:plon,k)*(1. + R23*cf(:plon,k)))
               endwhere
               fz(:plon,j,k)  = wz(:plon,j,k-1) * (fz(:plon,j,k) - dc(:plon,k))
            end do
            fz(:plon,j,1)      = 0.
            fz(:plon,j,plevp)  = 0.
	 end if
         do k = 2,plev
            dc(:plon,k)    = wz(:plon,j,k-1) * dc(:plon,k)
         end do
         do k = 2,km1
            dq(:plon,j,k)  = dq(:plon,j,k) + dc(:plon,k) - dc(:plon,k+1)
         end do
         dq(:plon,j,1)      = dq(:plon,j,1) - dc(:plon,2)
         dq(:plon,j,plev)   = dq(:plon,j,plev) + dc(:plon,plev)
	 if( j >= 1 .and. j <= platl ) then
	    do k = 2,plev
               fzp(:plon,j,k) = dc(:plon,k)
	    end do
            fzp(:plon,j,1)     = 0.
            fzp(:plon,j,plevp) = 0.
	 end if
      end do

      end subroutine FZPPM
 
      subroutine XTP( k, pu, dq, q, c, &
		      fx2, fxp, xmass, platl )
!-----------------------------------------------------------------------
!	... Zonal transport
!-----------------------------------------------------------------------

      use MO_MPI,  only : base_lat
      use MO_GRID, only : plon

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: k
      integer, intent(in) :: platl
      real, intent(in)    :: c(plon,-3:platl+4), &
                             xmass(plon,-3:platl+4)
      real, intent(in)    :: pu(plon,-3:platl+4), &
                             q(plon,-1:platl+2)
      real, intent(inout) :: dq(plon,-1:platl+2)
      real, intent(out)   :: fx2(plon+1,-1:platl+2)
      real, intent(out)   :: fxp(plon,-1:platl+2)

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: i, imp, j, jglob, jbl, jbu
      integer :: itmp, ist, iue, iu, iuw
      integer :: isave(plon)
      real    :: rut, bad_mem
      real, dimension(-iml:iml+plon+1)  :: dc, qtmp
      real :: fx1(plon+1)

#ifdef INIT_TEST
      bad_mem = z'7ff7ffff7ff7ffff'
      dc(:)   = bad_mem
      qtmp(:) = bad_mem
#endif

      imp = plon + 1
      jbl = j1 - base_lat
      jbu = jlim_north - base_lat
      do j = jl(4),ju(4)
	 jglob = base_lat + j
         qtmp(1:plon) = q(:plon,j)
         if( jglob > js(k) .and. jglob < jn(k) ) then
!-----------------------------------------------------------------------
! 	... Eulerian
!-----------------------------------------------------------------------
            qtmp(0)     = q(plon,j)
            qtmp(-1)    = q(plon-1,j)
            qtmp(imp)   = q(1,j)
            qtmp(imp+1) = q(2,j)
            if( iord == 1 .or. (has_spole .and. j == jbl) .or. (has_npole .and. j == jbu) ) then
               do i = 1,plon
                  iu = INT( REAL(i) - c(i,j) )
                  fx1(i) = qtmp(iu)
               end do
!-----------------------------------------------------------------------
! 	... Zero high order contribution
!-----------------------------------------------------------------------
               fx2(:plon,j) = 0.
            else
               call XMIST( qtmp, dc )
               dc(0) = dc(plon)
               if( iord == 2 .or. jglob <= j1vl .or. jglob >= j2vl ) then
                  do i = 1,plon
                     iu = INT( REAL(i) - c(i,j) )
                     fx1(i  ) = qtmp(iu)
                     fx2(i,j) = dc(iu)*(SIGN( 1.,c(i,j) ) - c(i,j))
                  end do
               else
                  call FXPPM( c(1,j), qtmp, dc, fx1, fx2(1,j) )
               end if
            end if
            fx1(:plon  ) = fx1(:plon  )*xmass(:plon,j)
            fx2(:plon,j) = fx2(:plon,j)*xmass(:plon,j)
         else
!-----------------------------------------------------------------------
! 	... Conservative (flux-form) Semi-Lagrangian transport
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! 	... ghost zone for the western edge
!-----------------------------------------------------------------------
            iuw = INT( -c(1,j) )
            iuw = MIN( 0,iuw )
            do i = iuw,0
               qtmp(i) = q(plon+i,j)
            end do
!-----------------------------------------------------------------------
! 	... ghost zone for the eastern edge
!-----------------------------------------------------------------------
            iue = INT( REAL(imp) - c(plon,j) )
            iue = MAX( imp,iue )

            do i = imp,iue
               qtmp(i) = q(i-plon,j)
            end do

            if( iord == 1 .or. (has_spole .and. j == jbl) .or. (has_npole .and. j == jbu) ) then
               do i = 1,plon
                  iu = INT( c(i,j) )
                  if( c(i,j) <= 0. ) then
                     itmp = i - iu
                     isave(i) = itmp - 1
                  else
                     itmp = i - iu - 1
                     isave(i) = itmp + 1
                  end if
                  fx1(i) = (c(i,j) - REAL(iu)) * qtmp(itmp)
               end do
!-----------------------------------------------------------------------
! 	... Zero high order contribution
!-----------------------------------------------------------------------
               fx2(:plon,j) = 0.
            else
               call XMIST( qtmp, dc )

               do i = iuw,0
                  dc(i) = dc(plon+i)
               end do

               do i = imp,iue
                  dc(i) = dc(i-plon)
               end do

               do i = 1,plon
                  iu  = INT( c(i,j) )
                  rut = c(i,j) - REAL( iu )
                  if( c(i,j) <= 0. ) then
                     itmp = i - iu
                     isave(i) = itmp - 1
                     fx2(i,j) = -rut*dc(itmp)*(1. + rut)
                  else
                     itmp = i - iu - 1
                     isave(i) = itmp + 1
                     fx2(i,j) = rut*dc(itmp)*(1. - rut)
                  end if
                  fx1(i) = rut*qtmp(itmp)
               end do
            end if
 
            do i = 1,plon
               if( c(i,j) > 1. ) then
#ifdef CRAY
!DIR$ NOVECTOR
#endif
                  do ist = isave(i),i-1
                     fx1(i) = fx1(i) + qtmp(ist)
                  end do
               else if( c(i,j) < -1. ) then
#ifdef CRAY
!DIR$ NOVECTOR
#endif
                  do ist = i,isave(i)
                     fx1(i) = fx1(i) - qtmp(ist)
                  end do
               end if
            end do

#ifdef CRAY
!DIR$ VECTOR
#endif
            fx1(:plon)   = pu(:plon,j)*fx1(:plon)
            fx2(:plon,j) = pu(:plon,j)*fx2(:plon,j)
         end if
 
         fx1(imp  ) = fx1(1  )
         fx2(imp,j) = fx2(1,j)
!-----------------------------------------------------------------------
! 	... Update using low order fluxes
!-----------------------------------------------------------------------
         dq(:plon,j) =  dq(:plon,j) + fx1(:plon) - fx1(2:plon+1)
	 fxp(:plon,j) = fx1(:plon)
      end do

      end subroutine XTP
 
      subroutine YTP( ic, k, dq, p, c, &
		      ymass, fy1, fy2, platl )
!-----------------------------------------------------------------------
! 	... Meridional transport
!-----------------------------------------------------------------------

      use MO_MPI,  only :  masternode, lastnode, base_lat
      use MO_GRID, only :  plon, plat

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) ::  ic
      integer, intent(in) ::  k
      integer, intent(in) ::  platl
      real, intent(in)    ::  p(plon,-3:platl+4), &
                              c(plon,-2:platl+3), &
			      ymass(plon,-2:platl+3)
      real, intent(inout) ::  dq(plon,-1:platl+2)
      real, intent(inout) ::  fy1(plon,-2:platl+3)
      real, intent(inout) ::  fy2(plon,-2:platl+3)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: j, jmr, jb, jb1
      real    :: sum1, sum2
      real    :: dc2(plon,-2:platl+3)

#ifdef INIT_TEST
      dc2(:,:) = z'7ff7ffff7ff7ffff'
#endif

      call YMIST( k, ic, p, dc2, platl )
      call FYPPM( k, ic, c, p, dc2, &
		  fy1, fy2, platl )

      do j = jle(13),jue(13)+1
         fy1(:plon,j) = fy1(:plon,j)*ymass(:plon,j)
      end do
      do j = jle(10),jue(10)
         fy2(:plon,j) = fy2(:plon,j)*ymass(:plon,j)
      end do

      do j = jle(13),jue(13)
         dq(:plon,j) = dq(:plon,j) + &
                       (fy1(:plon,j) - fy1(:plon,j+1)) * acosp(j+base_lat)
      end do

!-----------------------------------------------------------------------
! 	... Poles
!-----------------------------------------------------------------------
      if( masternode ) then
	 jb  = 1 - base_lat
	 jb1 = j1 - base_lat
         sum1 = dq(1,jb) - SUM( fy1(:plon,jb1) )*rcap
         dq(:plon,jb) = sum1
         if( large_cap ) then
            dq(:plon,jb+1) = sum1
         end if
      end if
      if( lastnode ) then
	 jb  = plat - base_lat
	 jb1 = jlim_north + 1 - base_lat
         sum2 = dq(1,jb) + SUM( fy1(:plon,jb1) )*rcap
         dq(:plon,jb) = sum2
         if( large_cap ) then
            dq(:plon,jb-1) = sum2
         end if
      end if

      end subroutine YTP

      subroutine COSA( cosp, cose, dp, platl )

      use MO_GRID,      only : plat
      use MO_CONSTANTS, only : pi
      use MO_MPI,       only : base_lat

      implicit none

!------------------------------------------------------------------------
!	... Dummy arguments
!------------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)    :: dp
      real, dimension(plat), intent(out) :: cosp     ! cosine of gridbox centers
      real, dimension(plat), intent(out) :: cose     ! cosine of gridbox edges

!------------------------------------------------------------------------
!	... Local variables
!------------------------------------------------------------------------
      integer :: jl, ju, j, jb
      real    :: ph5
      real    :: sine(plat+1)
 
      do j = 1,plat+1
         ph5  = -.5*pi + (REAL(j-1) - .5)*dp
         sine(j) = SIN( ph5 )
      end do
 
      do j = 1,plat
         cosp(j) = (sine(j+1) - sine(j))/dp
      end do
 
!------------------------------------------------------------------------
! 	... Define cosine at edges
!------------------------------------------------------------------------
      cosp(1)    = 0.
      cosp(plat) = 0.

      do j = 2,plat
         cose(j) = .5 * (cosp(j-1) + cosp(j))
      end do
      cose(1) = 0.

#ifdef DEBUG
	write(*,*) 'cose=',cose
	write(*,*) 'cosp=',cosp
#endif
 
      end subroutine COSA

      subroutine CONVERT_WINDS( ndt, u, v, crx, cry, platl )
!-----------------------------------------------------------------------
!       ... Convert winds on A-Grid to Courant # on C-Grid.
!-----------------------------------------------------------------------

      use MO_MPI,       only : masternode, base_lat
      use MO_GRID,      only : plev, plevp, plat, plon
      use MO_CONSTANTS, only : pi, ae => rearth, r2d
      use MO_TIMER,     only : TIME_DIFF, elapsed

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: ndt                              ! timestep (s)
      integer, intent(in) :: platl                            ! lat tile dim
      real, intent(in)    :: u(plon,-3:platl+4,plev), &       ! u wind component (m/s)
                             v(plon,-2:platl+3,plev)          ! v wind component (m/s)
      real, intent(out)   :: crx(plon,-3:platl+4,plev), &     ! Courant number in E-W direction
                             cry(plon,-2:platl+3,plev)        ! Courant number in N-S direction
!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k, ic, thread_no
      integer :: jb, jb1, jglob, jm1, ierr, astat
      real    :: dl, dp, maxdt, d5, agle, ztc, dt, cr1, elapsed_time
      character(len=8)    :: cdate(2)
      character(len=10)   :: ctime(2)
      logical :: found

      if( ndt0 /= ndt ) then
         dt   = ndt
         ndt0 = ndt

         dl = 2.*pi / REAL(plon)
         dp =    pi / REAL(plat-1)
         cr1  = ABS( Umax*dt )/(dl*ae)
         MaxDT = dp*ae / ABS( Umax)  + .5

         if( masternode ) then
            write(*,*) ' '
            write(*,'('' TPCORE: Largest time step for max(V) = '',1p,e10.3,'' (m/s) is '',e10.3,'' s'')') &
               Umax, MaxDT   
            if( MaxDT < ABS( ndt ) ) then
               write(*,*) 'TPCORE: Warning: delt maybe too large!'
            end if
            write(*,*) ' '
         end if

         if( cr1 >= .95 ) then
            js0 = 0
            jn0 = plat
            iml = plon - 2
            ztc = 0.
         else
            ztc = ACOS( cr1 ) * r2d
            js0 = REAL(plat-1)*(90. - ztc)/180. + 2
            js0 = MAX( js0,j1+1 )
            iml = MIN( 6*js0/(j1 - 1) + 2,4*plon/5 )
            jn0 = plat - js0 + 1
         end if
 
#ifdef DEBUG
         if( masternode ) then
            write(*,*) 'ZTC = ',ztc,' JS = ',js0,' JN = ',jn0,' IML = ',iml
         end if
#endif
         allocate( dtdx(-3:platl+4),dtdx5(-3:platl+4),stat=astat)
         if( astat /= 0 ) then
            write(*,*) 'ffsl_inti: failed to allocate dtdx,dtdx5; error = ',astat
            call endrun
         end if
         dtdx(jle(1):jue(1))  = dt / ( dl*ae*cosp(jle(1)+base_lat:jue(1)+base_lat) )
         dtdx5(jle(1):jue(1)) = .5*dtdx(jle(1):jue(1))
         dtdy  = dt /(ae*dp)
         dtdy5 = .5*dtdy
      end if

!-----------------------------------------------------------------------
!       ... Compute Courant number
!-----------------------------------------------------------------------
#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, jb, jb1, k, jglob, found, d5 ), schedule(dynamic,1)
#endif
      do k = 1,plev
         do j = jle(6),jue(6)
            do i = 2,plon
               crx(i,j,k) = dtdx5(j)*(u(i,j,k) + u(i-1,j,k))
            end do
         end do
!-----------------------------------------------------------------------
!       ... For i=1
!-----------------------------------------------------------------------
         do j = jle(6),jue(6)
            crx(1,j,k) = dtdx5(j)*(u(1,j,k) + u(plon,j,k))
         end do

         do j = jle(9),jue(9)
            do i = 1,plon
               cry(i,j,k) = dtdy5*(v(i,j,k) + v(i,j-1,k))
            end do
         end do
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine CONVERT_WINDS

      subroutine SET_CROSS_TERMS( crx, cry, u, v, platl )
!-----------------------------------------------------------------------
!       ... Compute Courant number at cell center
!-----------------------------------------------------------------------

      use MO_GRID, only : plev, plat, plon
      use MO_MPI,  only : masternode, lastnode, base_lat

      implicit none
!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: crx(plon,-3:platl+4,plev), &    ! Courant number in E-W direction
                           cry(plon,-2:platl+3,plev)       ! Courant number in N-S direction
      real, intent(out) :: u(plon,-3:platl+4,plev), &      ! E-W Courant number at cell center
                           v(plon,-2:platl+3,plev)         ! N-S Courant number at cell center
!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k, jb, jb1

#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, k, jb, jb1 ), schedule(dynamic,1)
#endif
      do k = 1,plev
         do j = jle(6),jue(6)
            do i = 1,plon-1
               if( crx(i,j,k)*crx(i+1,j,k) > 0. ) then
                  if( crx(i,j,k) > 0. ) then
                     u(i,j,k) = crx(i,j,k)
                  else
                     u(i,j,k) = crx(i+1,j,k)
                  end if
               else
                  u(i,j,k) = 0.
               end if
            end do
            if( crx(plon,j,k)*crx(1,j,k) > 0. ) then
               if( crx(plon,j,k) > 0. ) then
                  u(plon,j,k) = crx(plon,j,k)
               else
                  u(plon,j,k) = crx(1,j,k)
               end if
            else
               u(plon,j,k) = 0.
            end if
         end do
 
         do j = jle(13),jue(13)
            do i = 1,plon
               if( cry(i,j,k)*cry(i,j+1,k) > 0. ) then
                  if( cry(i,j,k) > 0. ) then
                     v(i,j,k) = cry(i,j,k)
                  else
                     v(i,j,k) = cry(i,j+1,k)
                  end if
               else
                  v(i,j,k) = 0.
               end if
               jtt(i,j,k) = INT( REAL(base_lat+j) - v(i,j,k) ) - base_lat
               if( ABS( jtt(i,j,k) - j ) > 1 ) then
                  write(*,*) 'SET_CROSS_TERMS: jtt failed courant at ', &
                             i,j,jtt(i,j,k),k,v(i,j,k)
                  call ENDRUN
               end if
               jtn(i,j,k) = NINT( REAL(base_lat+j) - v(i,j,k) ) - base_lat
               if( ABS( jtn(i,j,k) - j ) > 1 ) then
                  write(*,*) 'SET_CROSS_TERMS: jtn failed courant at ', &
                             i,j,jtn(i,j,k),k,v(i,j,k)
                  call ENDRUN
               end if
            end do
         end do
 
!        if( masternode ) then
!           jb  = 1 - base_lat
!           jb1 = jb + 1
!           do i = 1,imh
!              v(i,    jb,k) = .5*(cry(i,jb1,k) - cry(i+imh,jb1,k))
!              v(i+imh,jb,k) = -v(i,jb,k)
!           end do
!        end if
!        if( lastnode ) then
!           jb  = plat - base_lat
!           jb1 = jb - 1
!           do i = 1,imh
!              v(i,jb,k) = .5*(cry(i,jb1,k) - cry(i+imh,jb1,k))
!              v(i+imh,jb,k) = -v(i,jb,k)
!           end do
!        end if
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine SET_CROSS_TERMS

      end module MO_ADV
