
      module mo_cldprp

      implicit none

      save

      real, parameter :: c1 = 6.112
      real, parameter :: c2 = 17.67
      real, parameter :: c3 = 243.5
      real, parameter :: cp = 1004.64
      real, parameter :: rl = 2.5104e6
      real, parameter :: eps1 = .622
      real, parameter :: tfreez  = 273.16

      contains

      subroutine cldprp( q, t, u, v, p, &
                         z, s, mu, eu, du, &
                         md, ed, sd, qd, ud, &
                         vd, mc, qu, su, zf, &
                         qst, hmn, hsat, alpha, shat, &
                         ql, totpcp, totevp, cmeg, jb, &
                         lel, jt, jlcl, mx, j0, &
                         jd, il1g, il2g, rd, msg, &
                         nstep, lat, pflx, evp, cu, &
                         mu2, eu2, du2, md2, ed2, &
                         cmfdqr, limcnv, plonl )
!------------------------------------------------------------------------------
! 	... This is contributed code not fully standardized by the CCM core group.
!
! this code is very much rougher than virtually anything else in the CCM
! there are debug statements left strewn about and code segments disabled
! these are to facilitate future development. We expect to release a
! cleaner code in a future release
!
! the documentation has been enhanced to the degree that we are able
!
! Original version:  G. Zhang and collaborators
! Standardized:      Core group staff, 1994 and 195
! Reviewed:          P. Rasch, April 1996
!
!**** PLEASE NOTE ****
!
! we are aware of a specific problem in this code 
! (identified by the string ---> PROBLEM ONE)
! during the calculation of the updraft cloud properties,
! rather than adding a perturbation to the updraft temperature of 
! half a degree, (there was an inadvertant addition of cp*0.5) degrees
! or about 500 degrees. (This problem was in the code prior to its 
! contribution to the NCAR effort)

! Fortunately, the erroneous values
! are overwritten later in the code. The problem is quite subtle.
! The erroneous values would persist between cloud base and the lifting 
! condensation level. The addition of the very high perturbation to the updraft
! temperature causes the saturation mixing ratio to be set to zero, 
! and later the lcl to be set to one level above cloud base.
! There are therefore no levels between cloud base and the lcl. Therefore
! all erroneous values are overwritten.

! The only manifestation we are aware of with respect to this problem
! is that the lifting condensation level is constrained to be one level above
! cloud base.

! We discovered the problem after too much had been invested in
! very long integrations (in terms of computer time)
! to allow for a modification and model retuning. It is our expectation that
! this problem will be fixed in the next release of the model.
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! nov 20/92 - guang jun zhang,m.lazare. now has deeper (more
!             realistic) downdrafts.
! jul 14/92 - guang jun zhang,m.lazare. add shallow mixing
!             formulation.
! nov 21/91 - m.lazare. like previous cldprop except minimum "f"
!                       now 0.0004 instead of 0.001 (more
!                       realistic with more deep).
! may 09/91 - guang jun zhang, m.lazare, n.mcfarlane.
!             original version cldprop.
!------------------------------------------------------------------------------

      use mo_grid, only : plev, plevp
      use mo_hack, only : grav

      implicit none

!------------------------------------------------------------------------------
! 	... Dummy arguments
!------------------------------------------------------------------------------
      integer, intent(in) :: msg               ! missing moisture vals (always 0)
      integer, intent(in) :: nstep             ! time step index
      integer, intent(in) :: lat               ! lat index
      integer, intent(in) :: plonl             ! lon tile dim
      integer, intent(in) :: il1g              ! beginning longitude index
      integer, intent(in) :: il2g              ! ending    longitude index
      integer, intent(in) :: &
             jb(plonl), &      ! updraft base level
             lel(plonl), &     ! updraft launch level
             mx(plonl), &      ! updraft base level (same is jb)
             limcnv            ! convection limiting level
      integer, intent(inout) :: &
             j0(plonl), &      ! level where updraft begins detraining
             jt(plonl), &      ! updraft plume top
             jd(plonl), &      ! level of downdraft
             jlcl(plonl)       ! updraft lifting cond level
      real, intent(in) :: rd   ! gas constant for dry air
      real, intent(in) :: &
          q(plonl,plev), &     ! spec. humidity of env
          t(plonl,plev), &     ! temp of env
          p(plonl,plev), &     ! pressure of env
          z(plonl,plev), &     ! height of env
          s(plonl,plev), &     ! normalized dry static energy of env
          zf(plonl,plevp), &   ! height of interfaces
          u(plonl,plev), &     ! zonal velocity of env
          v(plonl,plev), &     ! merid. velocity of env
          shat(plonl,plev)     ! interface values of dry stat energy

      real, intent(inout) :: &
          alpha(plonl,plev), &    !
          cmfdqr(plonl,plev), &   ! rate of production of precip at that layer
          du(plonl,plev), &       ! detrainement rate of updraft
          ed(plonl,plev), &       ! entrainment rate of downdraft
          eu(plonl,plev), &       ! entrainment rate of updraft
          hmn(plonl,plev), &      ! moist stat energy of env
          hsat(plonl,plev), &     ! sat moist stat energy of env
          mc(plonl,plev), &       ! net mass flux
          md(plonl,plev), &       ! downdraft mass flux
          mu(plonl,plev), &       ! updraft mass flux
          pflx(plonl,plevp), &    ! precipitation flux thru layer
          qd(plonl,plev), &       ! spec humidity of downdraft
          ql(plonl,plev), &       ! liq water of updraft
          qst(plonl,plev), &      ! saturation spec humidity of env.
          qu(plonl,plev), &       ! spec hum of updraft
          sd(plonl,plev), &       ! normalized dry stat energy of downdraft
          su(plonl,plev), &       ! normalized dry stat energy of updraft
          ud(plonl,plev), &       ! downdraft u
          vd(plonl,plev)          ! downdraft v

      real, intent(out) :: mu2(plonl,plev)      ! updraft mass flux
      real, intent(out) :: eu2(plonl,plev)      ! updraft entrainment
      real, intent(out) :: du2(plonl,plev)      ! updraft detrainment
      real, intent(out) :: md2(plonl,plev)      ! downdraft mass flux
      real, intent(out) :: ed2(plonl,plev)      ! downdraft entrainment

!------------------------------------------------------------------------------
! 	... Local workspace
!------------------------------------------------------------------------------
      real, parameter :: small  = 1.e-20
      real, parameter :: c0     = 2.e-3
      real, parameter :: cu_eps = 5.e-6

      integer :: khighest
      integer :: klowest  
      integer :: kount 
      integer :: i, ind, k, km1, kp1

      real :: beta
      real :: ql1
      real :: weight
      real :: tu
      real :: estu
      real :: qstu
      real :: mdt  
      real :: cu2
      real :: est
      real :: arg1, arg2, tsteps
      real :: wrk

      real, dimension(plonl,plev) :: &
             gamma, dz, iprm, hu, hd, eps, &
             f, k1, i2, ihat, i3, idag, i4, &
             qsthat, hsthat, gamhat, cu, evp, &
             cmeg, qds
      real, dimension(plonl) :: &
             hmin, expdif, expnum, ftemp, eps0, &
             rmue, zuef, zdef, epsm, ratmjb, &
             totpcp, totevp, alfa

      logical, dimension(plonl) :: &
              doit, done

      ftemp(:il2g)  = 0.
      expnum(:il2g) = 0.
      expdif(:il2g) = 0.

!------------------------------------------------------------------------------
!   	... Change from msg+1 to 1 to prevent blowup
!------------------------------------------------------------------------------
      do k = 1,plev
         dz(:il2g,k) = zf(:il2g,k) - zf(:il2g,k+1)
      end do

!------------------------------------------------------------------------------
! 	... Initialize many output and work variables to zero
!------------------------------------------------------------------------------
      do k = msg + 1,plev
         k1(:il2g,k) = 0.
         i2(:il2g,k) = 0.
         i3(:il2g,k) = 0.
         i4(:il2g,k) = 0.
         mu(:il2g,k) = 0.
         f(:il2g,k) = 0.
         eps(:il2g,k) = 0.
         eu(:il2g,k) = 0.
         du(:il2g,k) = 0.
         ql(:il2g,k) = 0.
         cu(:il2g,k) = 0.
         evp(:il2g,k) = 0.
         cmeg(:il2g,k) = 0.
         md(:il2g,k) = 0.
         ed(:il2g,k) = 0.
         mc(:il2g,k) = 0.
         mu2(:il2g,k) = 0.
         eu2(:il2g,k) = 0.
         du2(:il2g,k) = 0.
         md2(:il2g,k) = 0.
         ed2(:il2g,k) = 0.
         pflx(:il2g,k) = 0.
         cmfdqr(:il2g,k) = 0.
         qds(:il2g,k) = q(:il2g,k)
         sd(:il2g,k) = s(:il2g,k)
         qd(:il2g,k) = q(:il2g,k)
         ud(:il2g,k) = u(:il2g,k)
         vd(:il2g,k) = v(:il2g,k)
         qu(:il2g,k) = q(:il2g,k)
         su(:il2g,k) = s(:il2g,k)
      end do

      do k = msg + 1,plev
        do i = 1,il2g
          est = c1*EXP( (c2*(t(i,k) - tfreez))/((t(i,k) - tfreez) + c3) )
          if( p(i,k) - est > 0. ) then
             qst(i,k) = eps1*est/(p(i,k) - est)
          else
             qst(i,k) = 1.
          end if
          gamma(i,k) = qst(i,k)*(1. + qst(i,k)/eps1)*eps1*rl/(rd*t(i,k)**2)*rl/cp
          hmn(i,k) = cp*t(i,k) + grav*z(i,k) + rl*q(i,k)
          hsat(i,k) = cp*t(i,k) + grav*z(i,k) + rl*qst(i,k)
          hu(i,k) = hmn(i,k)
          hd(i,k) = hmn(i,k)
        end do
      end do

!------------------------------------------------------------------------------
! 	... Set to zero things which make this routine blow up
!------------------------------------------------------------------------------
      do k = 1,msg
         cmfdqr(:il2g,k) = 0.
         mu2(:il2g,k) = 0.
         eu2(:il2g,k) = 0.
         du2(:il2g,k) = 0.
         md2(:il2g,k) = 0.
         ed2(:il2g,k) = 0.
      end do

!------------------------------------------------------------------------------
! 	... Interpolate the layer values of qst, hsat and gamma to layer interfaces
!------------------------------------------------------------------------------
      hsthat(:il2g,msg+1) = hsat(:il2g,msg+1)
      qsthat(:il2g,msg+1) = qst(:il2g,msg+1)
      gamhat(:il2g,msg+1) = gamma(:il2g,msg+1)
      totpcp(:il2g) = 0.
      totevp(:il2g) = 0.
      do k = msg + 2,plev
        do i = 1,il2g
          if( ABS( qst(i,k-1) - qst(i,k) ) > 1.e-6 ) then
            qsthat(i,k) = LOG( qst(i,k-1)/qst(i,k) )*qst(i,k-1)*qst(i,k)/(qst(i,k-1) - qst(i,k))
          else
            qsthat(i,k) = qst(i,k)
          end if
          hsthat(i,k) = cp*shat(i,k) + rl*qsthat(i,k)
          if( ABS( gamma(i,k-1) - gamma(i,k) ) > 1.e-6 ) then
            gamhat(i,k) = LOG( gamma(i,k-1)/gamma(i,k) )*gamma(i,k-1)*gamma(i,k)/(gamma(i,k-1) - gamma(i,k))
          else
            gamhat(i,k) = gamma(i,k)
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Initialize cloud top to highest plume top.
!           changed hard-wired 4 to limcnv+1 (not to exceed plev)
!------------------------------------------------------------------------------
      do i = 1,il2g
        jt(i) = MAX( lel(i),limcnv+1 )
        jt(i) = MIN( jt(i),plev )
        jd(i) = plev
        jlcl(i) = lel(i)
        hmin(i) = 1.e6
      end do

!------------------------------------------------------------------------------
! 	... Find the level of minimum hsat, where detrainment starts
!------------------------------------------------------------------------------
      do k = msg + 1,plev
        do i = 1,il2g
          if( hsat(i,k) <= hmin(i) .and. k >= jt(i) .and. k <= jb(i) ) then
            hmin(i) = hsat(i,k)
            j0(i) = k
          end if
        end do
      end do
      do i = 1,il2g
        j0(i) = MIN( j0(i),jb(i)-2 )
        j0(i) = MAX( j0(i),jt(i)+2 )
!------------------------------------------------------------------------------
! 	... Fix from Guang Zhang to address out of bounds array reference
!------------------------------------------------------------------------------
        j0(i) = MIN( j0(i),plev )
      end do

!------------------------------------------------------------------------------
! 	... Initialize certain arrays inside cloud
!------------------------------------------------------------------------------
      do k = msg + 1,plev
        do i = 1,il2g
          if( k >= jt(i) .and. k <= jb(i) ) then
            hu(i,k) = hmn(i,mx(i)) + cp*.5
            su(i,k) = s(i,mx(i)) + .5
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Compute Taylor series for approximate eps(z) below
!------------------------------------------------------------------------------
      do k = plev - 1,msg + 1,-1
        kp1 = k + 1
        do i = 1,il2g
          if( k < jb(i) .and. k >= jt(i) ) then
            k1(i,k) = k1(i,kp1) + (hmn(i,mx(i)) - hmn(i,k))*dz(i,k)
            ihat(i,k) = .5*(k1(i,kp1) + k1(i,k))
            i2(i,k) = i2(i,kp1) + ihat(i,k)*dz(i,k)
            idag(i,k) = .5*(i2(i,kp1) + i2(i,k))
            i3(i,k) = i3(i,kp1) + idag(i,k)*dz(i,k)
            iprm(i,k) = .5*(i3(i,kp1) + i3(i,k))
            i4(i,k) = i4(i,kp1) + iprm(i,k)*dz(i,k)
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Re-initialize hmin array for ensuing calculation.
!------------------------------------------------------------------------------
      do i = 1,il2g
        hmin(i) = 1.e6
      end do
      do k = msg + 1,plev
        do i = 1,il2g
          if( k >= j0(i) .and. k <= jb(i) .and. hmn(i,k) <= hmin(i) ) then
            hmin(i) = hmn(i,k)
            expdif(i) = hmn(i,mx(i)) - hmin(i)
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Compute approximate eps(z) using above Taylor series
!------------------------------------------------------------------------------
      do k = msg + 2,plev
        do i = 1,il2g
          expnum(i) = 0.
          ftemp(i) = 0.
          if( k < jt(i) .or. k >= jb(i) ) then
            k1(i,k) = 0.
            expnum(i) = 0.
          else
            expnum(i) = hmn(i,mx(i)) - (hsat(i,k-1)*(zf(i,k) - z(i,k)) &
                        + hsat(i,k)* (z(i,k-1) - zf(i,k)))/(z(i,k-1) - z(i,k))
          end if
          if( expdif(i) > 100. .and. expnum(i) > 0. .and. k1(i,k) > expnum(i)*dz(i,k) ) then
            ftemp(i) = expnum(i)/k1(i,k)
            f(i,k) = ftemp(i) + i2(i,k)/k1(i,k)*ftemp(i)**2 &
                     + (2.*i2(i,k)**2 - k1(i,k)*i3(i,k))/k1(i,k)**2*ftemp(i)**3  &
                     + (-5.*k1(i,k)*i2(i,k)*i3(i,k) + 5.*i2(i,k)**3+k1(i,k)**2*i4(i,k)) &
                        /k1(i,k)**3*ftemp(i)**4
            f(i,k) = MAX( f(i,k),0. )
            f(i,k) = MIN( f(i,k),.0002 )
          end if
        end do
      end do
      do i = 1,il2g
        if( j0(i) < jb(i) ) then
          if( f(i,j0(i)) < 1.e-6 .and. f(i,j0(i)+1) > f(i,j0(i)) ) then
            j0(i) = j0(i) + 1
          end if
        end if
      end do
      do k = msg + 2,plev
        do i = 1,il2g
          if( k >= jt(i) .and. k <= j0(i) ) then
            f(i,k) = MAX( f(i,k),f(i,k-1) )
          end if
        end do
      end do
      do i = 1,il2g
        eps0(i) = f(i,j0(i))
        eps(i,jb(i)) = eps0(i)
      end do

!------------------------------------------------------------------------------
! 	... Right now I have set this to do it the nflux41way as I want to match it
!           but it is probably better to disable it soon
!------------------------------------------------------------------------------
#define PJRWAY
#ifdef PJRWAY
      do k = plev,msg + 1,-1
         do i = 1,il2g
            if( k >= j0(i) .and. k <= jb(i) ) then
               eps(i,k) = f(i,j0(i))
            end if
         end do
      end do
      do k = plev,msg + 1,-1
         do i = 1,il2g
            if( k < j0(i) .and. k >= jt(i) ) then
               eps(i,k) = f(i,k)
            end if
         end do
      end do
#else
      do k = plev,msg+1,-1
        do i = 1,il2g
          if( k >= j0(i) ) then
            if( k <= jb(i) ) then
               eps(i,k) = f(i,j0(i))
            end if
          else
            if( k >= jt(i) ) then
               eps(i,k) = f(i,k)
            end if
          end if
        end do
      end do
#endif

!------------------------------------------------------------------------------
! 	... Specify the updraft mass flux mu, entrainment eu, detrainment du
!           and moist static energy hu.
!           here and below mu, eu,du, md and ed are all normalized by mb
!------------------------------------------------------------------------------
      do i = 1,il2g
        if( eps0(i) > 0. ) then
          ind = jb(i)
          mu2(i,ind) = 1.
          mu(i,ind)  = 1.
          eu2(i,ind) = 1./dz(i,ind)
          eu(i,ind)  = eu2(i,ind)
        end if
      end do
      do k = plev,msg + 1,-1
        kp1 = k + 1
        do i = 1,il2g
           if( eps0(i) > 0. .and. k >= jt(i) .and. k < jb(i) ) then
            zuef(i) = zf(i,k) - zf(i,jb(i))
            rmue(i) = (1./eps0(i))* (EXP( eps(i,kp1)*zuef(i)) - 1. )/zuef(i)
            mu(i,k) = (1./eps0(i))* (EXP( eps(i,k)*zuef(i)) - 1. )/zuef(i)
            eu(i,k) = (rmue(i) - mu(i,kp1))/dz(i,k)
            du(i,k) = (rmue(i) - mu(i,k))/dz(i,k)
            mu2(i,k) = mu(i,k)
            eu2(i,k) = eu(i,k)
            du2(i,k) = du(i,k)
          end if
        end do
      end do

      khighest = MINVAL( lel(:il2g) )
      klowest  = MAXVAL( jb(:il2g) )
      do k = klowest-1,khighest,-1
#ifdef CRAY
!dir  ivdep
#endif
        do i = 1,il2g
          if( k <= jb(i)-1 .and. k >= lel(i) .and. eps0(i) > 0. ) then
            if( mu(i,k) < .01 ) then
              hu(i,k) = hu(i,jb(i))
              mu(i,k)  = 0.
              mu2(i,k) = 0.
              eu(i,k)  = 0.
              eu2(i,k) = 0.
              du2(i,k) = mu2(i,k+1)/dz(i,k)
              du(i,k)  = du2(i,k)
            else
              hu(i,k) = mu(i,k+1)/mu(i,k)*hu(i,k+1) &
                        + dz(i,k)/mu(i,k)*(eu(i,k)*hmn(i,k) - du(i,k)*hsat(i,k))
            end if
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Reset cloud top index beginning from two layers above the
!           cloud base (i.e. if cloud is only one layer thick, top is not reset
!------------------------------------------------------------------------------
!**pjr there are diffs here, and I hope to god they dont matter
!------------------------------------------------------------------------------
      doit(:il2g) = .true.
      do k = klowest-2,khighest-1,-1
        do i = 1,il2g
          if( doit(i) .and. k <= jb(i)-2 .and. k >= lel(i)-1 ) then
            if( hu(i,k  ) <= hsthat(i,k) .and. hu(i,k+1) > hsthat(i,k+1) .and. mu(i,k) >= .02 ) then
              if( hu(i,k) - hsthat(i,k) < -2000. ) then
                jt(i) = k + 1
                doit(i) = .false.
              else
                jt(i) = k
                if( eps0(i) <= 0. ) then
                   doit(i) = .false.
                end if
              end if
            else if( hu(i,k) > hu(i,jb(i)) .or. mu(i,k) < .01 ) then
              jt(i) = k + 1
              doit(i) = .false.
            end if
          end if
        end do
      end do
      do k = plev,msg + 1,-1
#ifdef CRAY
!dir  ivdep
#endif
        do i = 1,il2g
          if( eps0(i) > 0. ) then
             if( k >= lel(i) .and. k <= jt(i) ) then
                eu2(i,k) = 0.
                mu2(i,k) = 0.
                eu(i,k) = 0.
                mu(i,k) = 0.
                du(i,k)  = 0.
                du2(i,k) = 0.
                hu(i,k) = hu(i,jb(i))
             end if
             if( k == jt(i) ) then
                eu2(i,k) = 0.
                mu2(i,k) = 0.
                eu(i,k) = 0.
                mu(i,k) = 0.
                du(i,k)  = mu(i,k+1)/dz(i,k)
                du2(i,k) = mu2(i,k+1)/dz(i,k)
              end if
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Specify downdraft properties (no downdrafts if jd.ge.jb).
!           scale down downward mass flux profile so that net flux
!           (up-down) at cloud base in not negative.
!------------------------------------------------------------------------------
      do i = 1,il2g
!------------------------------------------------------------------------------
! 	... Normal downdraft strength run alfa=0.2.  In test4 alfa=0.1
!------------------------------------------------------------------------------
        alfa(i) = .1
        jt(i) = MIN( jt(i),jb(i)-1 )
        jd(i) = MAX( j0(i),jt(i)+1 )
        jd(i) = MIN( jd(i),jb(i) )
        ind = jd(i)
        hd(i,ind) = hmn(i,ind-1)
        ud(i,ind) = u(i,ind-1)
        vd(i,ind) = v(i,ind-1)
        if( jd(i) < jb(i) .and. eps0(i) > 0. ) then
          epsm(i) = eps0(i)
          md(i,ind) = -alfa(i)*epsm(i)/eps0(i)
          md2(i,ind) = md(i,ind)
        end if
      end do
      do k = msg + 1,plev
        do i = 1,il2g
          if( k > jd(i) .and. k <= jb(i) .and. eps0(i) > 0. ) then
            zdef(i) = zf(i,jd(i)) - zf(i,k)
            md(i,k) = -alfa(i)/ (2.*eps0(i))*(EXP( 2.*epsm(i)*zdef(i) ) - 1.)/zdef(i)
            md2(i,k) = md(i,k)
          end if
        end do
      end do
      do k = msg + 1,plev
#ifdef CRAY
!dir  ivdep
#endif
        do i = 1,il2g
          if( k >= jt(i) .and. k <= jb(i) .and. eps0(i) > 0. .and. jd(i) < jb(i) ) then
            ratmjb(i) = MIN( ABS( mu2(i,jb(i))/md2(i,jb(i)) ),1. )
            md2(i,k) = md2(i,k)*ratmjb(i)
               md(i,k) = md2(i,k)
          end if
        end do
      end do

      do k = msg + 1,plev
        km1 = k - 1
        do i = 1,il2g
          if( k >= jt(i) .and. k <= plev .and. eps0(i) > 0. ) then
            ed2(i,km1) = (md2(i,km1) - md2(i,k))/dz(i,km1)
            ed(i,km1) = ed2(i,km1)
            mdt = MIN( md2(i,k),-small )
            hd(i,k) = (md(i,km1)*hd(i,km1) - dz(i,km1)*ed(i,km1)*hmn(i,km1))/mdt
          end if
        end do
      end do

!------------------------------------------------------------------------------
! 	... Calculate updraft and downdraft properties.
!------------------------------------------------------------------------------
      do k = msg + 2,plev
        do i = 1,il2g
          if( k >= jd(i) .and. k <= jb(i) .and. eps0(i) > 0. .and. jd(i) < jb(i) ) then
            qds(i,k) = qsthat(i,k) + gamhat(i,k)*(hd(i,k) - hsthat(i,k))/(rl*(1. + gamhat(i,k)))
          end if
        end do
      end do

      do i = 1,il2g
         done(i) = .false.
      end do
      kount = 0
      do k = plev,msg + 2,-1
        kp1 = k + 1
        do i = 1,il2g
          if( .not.done(i) .and. k > jt(i) .and. k < jb(i) .and. eps0(i) > 0. ) then
            su(i,k) = mu(i,kp1)/mu(i,k)*su(i,kp1) &
                      + dz(i,k)/mu(i,k)* (eu(i,k) - du(i,k))*s(i,k)
            qu(i,k) = mu(i,kp1)/mu(i,k)*qu(i,kp1) &
                      + dz(i,k)/mu(i,k)* (eu(i,k)*q(i,k) - du(i,k)*qst(i,k))
            tu = su(i,k) - grav/cp*zf(i,k)
            estu = c1*EXP( (c2*(tu - tfreez))/((tu - tfreez) + c3) )
            qstu = eps1*estu/ ((p(i,k) + p(i,k-1))/2. - estu)
            if( qu(i,k) >= qstu ) then
              jlcl(i) = k
              kount = kount + 1
              done(i) = .true.
            end if
          end if
        end do
        if( kount >= il2g ) then
           exit
        end if
      end do

      do k = msg + 2,plev
        do i = 1,il2g
          if( k == jb(i) .and. eps0(i) > 0. ) then
            qu(i,k) = q(i,mx(i))
            su(i,k) = (hu(i,k) - rl*qu(i,k))/cp
          end if
          if( k > jt(i) .and. k <= jlcl(i) .and. eps0(i) > 0. ) then
            su(i,k) = shat(i,k) + (hu(i,k) - hsthat(i,k))/(cp*(1. + gamhat(i,k)))
            qu(i,k) = qsthat(i,k) + gamhat(i,k)*(hu(i,k) - hsthat(i,k))/(rl*(1. + gamhat(i,k)))
          end if
        end do
      end do

      do k = plev,msg + 2,-1
        do i = 1,il2g
          if( k >= jt(i) .and. k < jb(i) .and. eps0(i) > 0. ) then
!           cu(i,k) = ((mu(i,k)*su(i,k) - mu(i,k+1)*su(i,k+1))/ &
!                    dz(i,k) - (eu(i,k) - du(i,k))*s(i,k))/(rl/cp)
            arg1   = (mu(i,k)*su(i,k) - mu(i,k+1)*su(i,k+1))/dz(i,k)
            arg2   = (eu(i,k) - du(i,k))*s(i,k)
            tsteps = cu_eps*abs( max( arg1,arg2 ) )
            if( (arg1 - arg2) > tsteps ) then
               cu(i,k) = cp*(arg1 - arg2)/rl
            else
               cu(i,k) = 0.
            end if
            if( k == jt(i) ) then
               cu(i,k) = 0.
            end if
            cu(i,k) = MAX( 0.,cu(i,k) )
          end if
        end do
      end do

      beta = 0.
      do k = plev,msg + 2,-1
        do i = 1,il2g
          cmfdqr(i,k) = 0.
!------------------------------------------------------------------------------
! 	... This modification is for test3 run, modified on 6/20/1995
!------------------------------------------------------------------------------
          if( k >= jt(i) .and. k < jb(i) .and. eps0(i) > 0. .and. mu(i,k) >= 0. ) then
            if( mu(i,k) > 0. ) then
              ql1 = 1./mu(i,k)* (mu(i,k+1)*ql(i,k+1)- &
                   dz(i,k)*du(i,k)*ql(i,k+1)+dz(i,k)*cu(i,k))
              ql(i,k) = ql1/ (1. + dz(i,k)*c0)
            else
              ql(i,k) = 0.
            end if
            totpcp(i) = totpcp(i) + dz(i,k)*(cu(i,k) - du(i,k)*(beta*ql(i,k) + (1. - beta)*ql(i,k+1)))
            cmfdqr(i,k) = c0*mu(i,k)*ql(i,k)
          end if
        end do
      end do

      do i = 1,il2g
        qd(i,jd(i)) = qds(i,jd(i))
        sd(i,jd(i)) = (hd(i,jd(i)) - rl*qd(i,jd(i)))/cp
      end do

      do k = msg + 2,plev
        kp1 = k + 1
        do i = 1,il2g
          if( k >= jd(i) .and. k < jb(i) .and. eps0(i) > 0. ) then
            qd(i,kp1) = qds(i,kp1)
            evp(i,k) = MAX( -ed(i,k)*q(i,k) + (md(i,k)*qd(i,k) - md(i,kp1)*qd(i,kp1))/dz(i,k),0. )
            mdt = MIN( md(i,kp1),-small )
            sd(i,kp1) = ((rl/cp*evp(i,k) - ed(i,k)*s(i,k))*dz(i,k) + md(i,k)*sd(i,k))/mdt
            totevp(i) = totevp(i) - dz(i,k)*ed(i,k)*q(i,k)
          end if
        end do
      end do
      do i = 1,il2g
        totevp(i) = totevp(i) + md(i,jd(i))*qd(i,jd(i)) - md(i,jb(i))*qd(i,jb(i))
      end do
      do i = 1,il2g
        ind = jb(i)
        if( eps0(i) > 0. ) then
          evp(i,ind) = MAX( -ed(i,ind)*q(i,ind) + (md(i,ind)*qd(i,ind))/dz(i,ind),0. )
          totevp(i) = totevp(i) - dz(i,ind)*ed(i,ind)*q(i,ind)
        end if
      end do

      totpcp(:il2g) = MAX( totpcp(:il2g),0. )
      totevp(:il2g) = MAX( totevp(:il2g),0. )

      weight = 1.0
      doit(:il2g) = totevp(:il2g) > 0. .and. totpcp(:il2g) > 0.
      do k = msg + 2,plev
        do i = 1,il2g
          if( doit(i) ) then
            wrk = weight*totpcp(i)/(totevp(i) + weight*totpcp(i))
            md2(i,k) = md2(i,k)*MIN( 1.,wrk )
            ed2(i,k) = ed2(i,k)*MIN( 1.,wrk )
            evp(i,k) = evp(i,k)*MIN( 1.,wrk )
          else
            md2(i,k) = 0.
            ed2(i,k) = 0.
            evp(i,k) = 0.
          end if
          md(i,k) = md2(i,k)
          ed(i,k) = ed2(i,k)
!------------------------------------------------------------------------------
!  	... cmeg is the cloud water condensed - rain water evaporated
!           cmfdqr  is the cloud water converted to rain - (rain evaporated)
!------------------------------------------------------------------------------
          cmeg(i,k) = cu(i,k) - evp(i,k)
          cmfdqr(i,k) = cmfdqr(i,k) - evp(i,k)
        end do
      end do
      do k = 2,plevp
         km1 = k - 1
         pflx(:il2g,k) = pflx(:il2g,km1) + cmfdqr(:il2g,km1)*dz(:il2g,km1)
      end do
      where( doit(:il2g) )
         totevp(:il2g) = totevp(:il2g)*MIN( 1.,weight*totpcp(:il2g)/(totevp(:il2g) + weight*totpcp(:il2g)) )
      elsewhere
         totevp(:il2g) = 0.
      endwhere

      do k = msg + 1,plev
         mc(:il2g,k) = mu(:il2g,k) + md(:il2g,k)
      end do
      
      end subroutine cldprp

      end module mo_cldprp
