
      module mo_ctlstep

      contains

      subroutine ctlstep( nstep, pdate, psec, mdelt, platl )
!-----------------------------------------------------------------------
! 	... set control variables for this timestep.
!-----------------------------------------------------------------------

      use mo_control,  only : endofrun, wrestart, laststep, itim0, begstep, nestep
      use mo_grid,     only : plon, plev
      use mo_restart,  only : qrsti, qrstc
      use mo_histout,  only : qhsti, qhstc, sim_file_cnt, hfile
      use mo_calendar, only : newdate
      use mo_mpi,      only : comp_node

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        nstep, &    ! current timestep
        pdate, &    ! current model date (yyyymmdd)
        psec, &     ! current model seconds in date
        mdelt, &    ! model timestep (seconds)
        platl       ! latitude tile dimension

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: &
        itim, &     ! time index at the end of timestep nstep
        istat, &    ! return status
        astat, &    ! allocation return status
        nhtfrq, &   ! frequency of history time sample writes in timesteps
        ntshst, &   ! number of time samples currently in history file
        mxtshst, &  ! maximum number of time samples in history file
        rstfrq, &   ! frequency of restart file writes in timesteps
        file, &     ! file index
        ndate, &    ! next model date
        nsec, &     ! model seconds in next date
        pmonth, &   ! present model month
        nmonth, &   ! next model month
        iday        ! wrk variable for time
      character(len=1) :: &
         monthly
      logical :: &
        shutdown, & ! true => shut down run this timestep
        do_alloc, & ! true => allocate inst output buffers
        pure_inst   ! true => history file has no time averaged fields

      itim = nstep

      if( nstep /= itim0+1 ) then
         begstep = .false.
      else
         begstep = .true.
      end if

      if( nstep /= itim0+nestep ) then
         laststep = .false.
      else
         laststep = .true.
      end if

!-----------------------------------------------------------------------
!     	... endofrun is set true either because all requested timesteps
!           have been completed, or because the job needs to shut down
!           for external reasons (e.g. operator could set a switch
!           requesting the job to shut down, or the job might check to
!           see if enough cpu time remains to allow another timestep to complete)
!-----------------------------------------------------------------------
      shutdown = .false.
      if( laststep .or. shutdown ) then
         endofrun = .true.
      else
         endofrun = .false.
      end if

!-----------------------------------------------------------------------
! 	... time to write a restart file?
!-----------------------------------------------------------------------
is_endofrun : &
      if( endofrun ) then
         wrestart = .true.
      else is_endofrun
         istat = qrstc( 'MONTHLY', monthly )
         if( istat /= 0 ) then
            write(*,*) 'ctlstep: qrstc returned error for monthly'
            call endrun
         end if
rst_monthly : &
         if( monthly /= 'm' ) then
            istat = qrsti( 'RSTFRQ', rstfrq )
            if( istat /= 0 ) then
               write(*,*) 'ctlstep: qrsti returned error for rstfrq'
               call endrun
            end if
            if( mod( itim,rstfrq ) == 0 .or. endofrun ) then
               wrestart = .true.
            else
               wrestart = .false.
            end if
         else rst_monthly
            ndate = pdate
            nsec  = psec + mdelt
            if( nsec >= 86400 ) then
               iday  = nsec/86400
               nsec  = mod( nsec,86400 )
               ndate = newdate( pdate, iday )
            end if
            pmonth = mod( pdate,10000 )/100
            nmonth = mod( ndate,10000 )/100
            if( nmonth /= pmonth ) then
               wrestart = .true.
            else
               wrestart = .false.
            end if
         end if rst_monthly
      end if is_endofrun

files_loop : &
      do file = 1,sim_file_cnt
!-----------------------------------------------------------------------
!     	... is it time to write a time sample to the history files ?
!-----------------------------------------------------------------------
         pure_inst               = sum( hfile(file)%tafcnt(:) ) == 0
         hfile(file)%wrhstts     = .false.
         do_alloc                = .false.
         hfile(file)%force_inst  = .false.
         hfile(file)%partial_ta  = .false.
         nullify( hfile(file)%ti_slev )
         nullify( hfile(file)%ti_mlev )
         istat = qhstc( 'MONTHLY', monthly, file )
         if( istat /= 0 ) then
            write(*,*) 'ctlstep: qhstc returned error for monthly'
            call endrun
         end if
         istat = qhsti( 'NHTFRQ', nhtfrq, file )
         if( istat /= 0 ) then
            write(*,*) 'ctlstep: qhsti returned error for nhtfrq'
            call endrun
         end if
hst_monthly : &
         if( monthly /= 'm' ) then
            if( mod( itim,hfile(file)%nhtfrq ) == 0 ) then
               hfile(file)%wrhstts    = .true.
               do_alloc               = .true.
            else if( endofrun ) then
               hfile(file)%partial_ta = any( hfile(file)%tafcnt(:) > 0 )
               if( hfile(file)%partial_ta ) then
                  hfile(file)%wrhstts     = .true.
                  do_alloc                = .true.
                  hfile(file)%fnorm       = 1.
               else if( endofrun .and. pure_inst .and. hfile(file)%hdi(3) == 0 ) then
                  hfile(file)%wrhstts    = .true.
                  do_alloc               = .true.
                  hfile(file)%force_inst = .true.
               end if
            end if
         else hst_monthly
            ndate = pdate
            nsec  = psec + mdelt
            if( nsec >= 86400 ) then
               iday  = nsec/86400
               nsec  = mod( nsec,86400 )
               ndate = newdate( pdate, iday )
            end if
            pmonth = mod( pdate,10000 )/100
            nmonth = mod( ndate,10000 )/100
            if( nmonth /= pmonth ) then
               hfile(file)%wrhstts    = .true.
               do_alloc               = .true.
            else if( endofrun ) then
               hfile(file)%partial_ta = any( hfile(file)%tafcnt(:) > 0 )
               if( hfile(file)%partial_ta ) then
                  hfile(file)%wrhstts     = .true.
                  do_alloc                = .true.
                  hfile(file)%fnorm   = 1.
               else if( endofrun .and. pure_inst .and. hfile(file)%hdi(3) == 0 ) then
                  hfile(file)%wrhstts    = .true.
                  do_alloc               = .true.
                  hfile(file)%force_inst = .true.
               end if
            end if
         end if hst_monthly

         if( do_alloc ) then
!-----------------------------------------------------------------------
!     	... allocate memory for instantaneous buffers
!-----------------------------------------------------------------------
            if( hfile(file)%tifcnt(1) > 0 ) then
               allocate( hfile(file)%ti_slev(plon,platl,hfile(file)%tifcnt(1)),stat=astat )
               if( astat /= 0 ) then
                  write(*,*) 'ctlstep: failed to allocate ti_slev for file = ',file,'; error = ',astat
	          call endrun
               end if
            end if
            if( hfile(file)%tifcnt(2) > 0 ) then
               allocate( hfile(file)%ti_mlev(plon,platl,plev,hfile(file)%tifcnt(2)),stat=astat )
               if( astat /= 0 ) then
                  write(*,*) 'ctlstep: failed to allocate ti_mlev for file = ',file,'; error = ',astat
	          call endrun
               end if
            end if
         end if
!-----------------------------------------------------------------------
!     	... is it time to close the history file?  only set this flag when a
!           time sample is to be written.
!-----------------------------------------------------------------------
         istat = qhsti( 'NTSHST', ntshst, file )
         if( istat /= 0 ) then
            write(*,*) 'ctlstep: qhsti returned error for ntshst'
            call endrun
         end if
         istat = qhsti( 'MXTSHST', mxtshst, file )
         if( istat /= 0 ) then
            write(*,*) 'ctlstep: qhsti returned error for mxtshst'
            call endrun
         end if
         if( hfile(file)%wrhstts .and. ntshst == mxtshst-1 ) then
            hfile(file)%fullhst = .true.
         else
            hfile(file)%fullhst = .false.
         end if
	 if( hfile(file)%fullhst .or. endofrun ) then
            hfile(file)%closehst = .true.
	 else
            hfile(file)%closehst = .false.
	 end if
      end do files_loop

      end subroutine ctlstep

      end module mo_ctlstep
