
      module MO_DRYMASS

      implicit none

      save

      real :: &
        mdryatm      ! atmospheric dry mass (kg/m^2)

      CONTAINS

      subroutine INIDRY( ps, sh, plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... Initialize /drymass/
!-----------------------------------------------------------------------

      use PLEVS,   only : PLEVS0
      use mo_grid, only : plev, plevp, plat
      use MO_MPI,  only : thisnode, base_lat
#ifdef USE_MPI
      use MO_MPI,  only : MPI_DOUBLE_PRECISION, MPI_SUCCESS, mpi_comm_comp, mpi_in_place
#endif
      use MO_MASS, only : QMASSA, GATMASS

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      real, intent(in) :: &
        ps(plonl,-3:platl+4,pplon), &    ! surface pressure (pascals)
        sh(plonl,plev,platl,pplon)       ! specific humidity

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: ip, j, ierr
      real :: tmass, qmass
      real, dimension(plat) :: tmass_sum, qmass_sum
      real, dimension(plonl,platl,pplon) :: &
        tmassj, &       ! total atm mass
        qmassj          ! total mass of moisture
      real, dimension(plonl,plev) :: &
        pmid, &                ! pressure at layer midpoints
        pdel                   ! pressure difference across layers
      real, dimension(plonl,plevp) :: &
        pint                   ! pressure at layer interfaces

      write(*,*) 'inidry: diagnostics'
      write(*,*) 'INIDRY: min ps = ',MINVAL( ps(:plonl,1:platl,:pplon) ),' on node = ',thisnode
      write(*,*) 'INIDRY: max ps = ',MAXVAL( ps(:plonl,1:platl,:pplon) ),' on node = ',thisnode

      if( pplon > 1 ) then
!$OMP PARALLEL DO private( ip, j, pmid, pint, pdel )
         do ip = 1,pplon
            do j = 1,platl
               call PLEVS0( ps(1,j,ip), pint, pmid, pdel, plonl )
               call GATMASS( j, ps(1,j,ip), tmassj(1,j,ip), plonl )
               call QMASSA( j, 1, sh(1,1,j,ip), pdel, qmassj(1,j,ip), plonl )
            end do
         end do
!$OMP END PARALLEL DO
      else
!$OMP PARALLEL DO private( j, pmid, pint, pdel )
         do j = 1,platl
            call PLEVS0( ps(1,j,1), pint, pmid, pdel, plonl )
            call GATMASS( j, ps(1,j,1), tmassj(1,j,1), plonl )
            call QMASSA( j, 1, sh(1,1,j,1), pdel, qmassj(1,j,1), plonl )
         end do
!$OMP END PARALLEL DO
      end if

      do j = 1,platl
         tmass_sum(j+base_lat) = SUM( tmassj(:,j,:) )
         qmass_sum(j+base_lat) = SUM( qmassj(:,j,:) )
      end do

#ifdef USE_MPI
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          tmass_sum, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'INIDRY: MPI_ALLGATHER failed; Error code = ',ierr
         call endrun
      end if
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          qmass_sum, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'INIDRY: MPI_ALLGATHER failed; Error code = ',ierr
         call endrun
      end if
#endif

      tmass   = SUM( tmass_sum )
      qmass   = SUM( qmass_sum )
      mdryatm = tmass - qmass

      write(*,*) 'INIDRY: tmass = ',tmass 
      write(*,*) 'INIDRY: qmass = ',qmass 
      write(*,*) 'INIDRY: dry mass of atmosphere will be held at ',mdryatm, ' kg/m^2'

      end subroutine INIDRY

      subroutine ADJDRY( ps, sh, diagprt, plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... Adjust ps so that the total dry mass remains constant
!-----------------------------------------------------------------------

      use PLEVS,   only : PDELAB
      use mo_grid, only : plev, plat
      use MO_MPI,  only : masternode, lastnode, base_lat
#ifdef USE_MPI
      use MO_MPI,  only : MPI_DOUBLE_PRECISION, MPI_SUCCESS, mpi_comm_comp, mpi_in_place
#endif
      use MO_MASS, only : QMASSA, GATMASS

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      real, intent(in) :: &
        sh(plonl,plev,platl,pplon)    ! specific humidity
      logical, intent(in) :: &
        diagprt
      real, intent(inout) :: &
        ps(plonl,-3:platl+4,pplon) ! adjusted surface pressure (pascals)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i, ip, j, jl, ju, ierr
      real ::  tmass, aqmass, bqmass, fac
      real, dimension(plat) :: tmass_sum, aqmass_sum, bqmass_sum
      real, dimension(plonl,platl,pplon) :: &
        tmassj, &              ! total atm mass
        aqmassj, &             ! A contribution to mass of moisture
        bqmassj                ! B contribution to mass of moisture
      real, dimension(plonl,plev) :: &
        pdela, &               ! A contribution to pressure diff across layers
        pdelb                  ! B contribution to pressure diff across layers

      if( pplon > 1 ) then
!$OMP PARALLEL DO private( ip, j, pdela, pdelb )
         do ip = 1,pplon
            do j = 1,platl
               call PDELAB( ps(1,j,ip), pdela, pdelb, plonl )
               call GATMASS( j, ps(1,j, ip), tmassj(1,j,ip), plonl )
               call QMASSA( j, 1, sh(1,1,j,ip), pdela, aqmassj(1,j,ip), plonl )
               call QMASSA( j, 1, sh(1,1,j,ip), pdelb, bqmassj(1,j,ip), plonl )
            end do
         end do
!$OMP END PARALLEL DO
      else
!$OMP PARALLEL DO private( j, pdela, pdelb )
         do j = 1,platl
            call PDELAB( ps(1,j,1), pdela, pdelb, plonl )
            call GATMASS( j, ps(1,j, 1), tmassj(1,j,1), plonl )
            call QMASSA( j, 1, sh(1,1,j,1), pdela, aqmassj(1,j,1), plonl )
            call QMASSA( j, 1, sh(1,1,j,1), pdelb, bqmassj(1,j,1), plonl )
         end do
!$OMP END PARALLEL DO
      end if
      do j = 1,platl
         tmass_sum(j+base_lat)  = SUM( tmassj(:,j,:) )
         aqmass_sum(j+base_lat) = SUM( aqmassj(:,j,:) )
         bqmass_sum(j+base_lat) = SUM( bqmassj(:,j,:) )
      end do

#ifdef USE_MPI
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          tmass_sum, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'ADJDRY: MPI_ALLGATHER failed; Error code = ',ierr
         call endrun
      end if
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          aqmass_sum, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'ADJDRY: MPI_ALLGATHER failed; Error code = ',ierr
         call endrun
      end if
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          bqmass_sum, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'ADJDRY: MPI_ALLGATHER failed; Error code = ',ierr
         call endrun
      end if
#endif

      tmass  = SUM( tmass_sum )
      aqmass = SUM( aqmass_sum )
      bqmass = SUM( bqmass_sum )
      fac = (mdryatm + aqmass) / (tmass - bqmass)
#ifdef DEBUG
      write(*,*) 'ADJDRY: fac=',fac
#endif

#ifdef USE_MPI
      jl = -3
      ju = platl+4
      if( masternode ) then
        jl = 1
      end if
      if( lastnode ) then
        ju = platl
      end if
#else
      jl = 1
      ju = platl
#endif
      if( pplon > 1 ) then
!$OMP PARALLEL DO private( ip, i, j )
         do ip = 1,pplon
            do j = jl,ju
               do i = 1,plonl
                  ps(i,j,ip) = ps(i,j,ip) * fac
               end do
            end do
         end do
!$OMP END PARALLEL DO
      else
!$OMP PARALLEL DO private( i, j )
         do j = jl,ju
            do i = 1,plonl
               ps(i,j,1) = ps(i,j,1) * fac
            end do
         end do
!$OMP END PARALLEL DO
      end if

      if( diagprt ) then
         write(*,*) 'ADJDRY: dry mass of atmosphere will be held at ', mdryatm, ' kg/m^2'
      end if

      end subroutine ADJDRY

      subroutine MR2WET( ntrac, xdry, sh, xwet, plonl )
!-----------------------------------------------------------------------
! 	... Convert tracer mixing ratios from a dry to a wet basis.
! ... Obsolete?
!-----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        ntrac, &                  ! number of tracers
        plonl                     ! lon tile dim
      real, intent(in) :: &
        xdry(plonl,plev,ntrac), & ! Tracer concentration (kg tracer/kg dry air)
        sh(plonl,plev)            ! Specific humidity (kg water vapor/kg moist air)

      real, intent(out) :: &
       xwet(plonl,plev,ntrac)     ! Tracer concentration (kg tracer/kg moist air)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i, k, m

      do m = 1,ntrac
         do k = 1,plev
            do i = 1,plonl
               xwet(i,k,m) = xdry(i,k,m) * (1. - sh(i,k))
            end do
         end do
      end do

      end subroutine MR2WET

      subroutine MR2DRY( ntrac, xwet, sh, xdry, plonl )
!-----------------------------------------------------------------------
! 	... Convert tracer mixing ratios from a wet to a dry basis
! ... Obsolete?
!-----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        ntrac, &                  ! number of tracers
        plonl                     ! lon tile dim
      real, intent(in) :: &
        xwet(plonl,plev,ntrac), & ! Tracer concentration (kg tracer/kg moist air)
        sh(plonl,plev)            ! Specific humidity (kg water vapor/kg moist air)

      real, intent(out) :: &
       xdry(plonl,plev,ntrac)     ! Tracer concentration (kg tracer/kg dry air)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i, k, m

      do m = 1,ntrac
         do k = 1,plev
            do i = 1,plonl
               xdry(i,k,m) = xwet(i,k,m) / (1. - sh(i,k))
            end do
         end do
      end do

      end subroutine MR2DRY

      end module MO_DRYMASS
