
      module mo_ffsladv

      private
      public :: advection, adjust_press

      contains

      subroutine ADVECTION( pctm1, pctm2, pmet2, xmass, ymass, &
                            w, ndt, nc, ap, bp, &
			    crx, cry, plonl, platl, pplon )
!-----------------------------------------------------------------------
!       ... Lin and Rood Flux Form SL Transport
!-----------------------------------------------------------------------
 
#ifdef USE_MPI
      use MO_MPI,        only : persist_recv, persist_send, status,             &
                                MPI_SUCCESS, MPI_DOUBLE_PRECISION, MPI_INTEGER, &
                                MPI_MIN, MPI_MAX, mpi_comm_comp, base_lat
#else
      use MO_MPI,        only : base_lat
#endif
      use m_adv,         only : jle, jue, js, jn, jwrk, mmr, do_pressure_fixer
      use MO_ADV,        only : ADVECTXY, ADVECTZ, SET_CROSS_TERMS
      use MO_GRID,       only : plon, plev, plevp, plat
      use MO_CONSTANTS,  only : pi, ae => rearth, r2d
      use MO_TIMER,      only : TIME_DIFF, elapsed

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl                             ! lon tile dim
      integer, intent(in) :: platl                             ! lat tile dim
      integer, intent(in) :: pplon                             ! lon tile count
      integer, intent(in) :: nc, &                             ! number of advected components
                             ndt                               ! timestep (s)
      real, intent(in)    :: ap(plevp), bp(plevp)              ! hybrid grid a, b factors
      real, intent(in)    :: pctm1(plon,-3:platl+4), &         ! surface pressure at t(n)
                             pmet2(plon,-3:platl+4), &         ! surface pressure at t(n+1) from met fields
                             pctm2(plon,-3:platl+4)            ! predicted surface pressure at t(n+1)
      real, intent(in)    :: xmass(plon,-3:platl+4,plev), &    ! zonal mass flux
                             ymass(plon,-2:platl+3,plev)       ! meridional mass flux
      real, intent(out)   :: w(plon,-1:platl+2,plev)           ! vertical mass flux
      real, intent(inout) :: crx(plon,-3:platl+4,plev)
      real, intent(inout) :: cry(plon,-2:platl+3,plev)

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k, ic, thread_no
      integer :: jb, jb1, jglob, jm1, ierr, astat
      real    :: dl, dp, maxdt, d5, agle, ztc, dt, cr1, elapsed_time
      real    :: bad_mem
      real, allocatable, dimension(:,:,:,:) ::  dq, fx, fy
      real, allocatable, dimension(:,:,:) ::   &
           delp, delpm, delp1, delp2, pu, dpi, &
           delp2_junk,                         &
           ua, va
      real, allocatable, dimension(:,:) :: dps_ctm
      real, dimension(nc) :: Tmin, Tmax, Bmin, Bmax, Ttmp
      character(len=8)    :: cdate(2)
      character(len=10)   :: ctime(2)
      logical :: found

!-----------------------------------------------------------------------
!       ... Allocate locals
!-----------------------------------------------------------------------
      ALLOCATE( delp(plon,-3:platl+4,plev), &
                delpm(plon,-3:platl+4,plev), &
		delp1(plon,-3:platl+4,plev), &
                delp2(plon,-3:platl+4,plev), &
		pu(plon,-3:platl+4,plev), &
                delp2_junk(plon,-3:platl+4,plev), &
                dpi(plon,-3:platl+4,plev), &
		dq(plon,-1:platl+2,plev,nc), &
                fx(plon+1,-1:platl+2,plev,nc), &
		fy(plon,-2:platl+3,plev,nc), &
                dps_ctm(plon,-3:platl+4), &
                ua(plon,-3:platl+4,plev), &
		va(plon,-2:platl+3,plev), &
                stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'ADVECTION: failed to allocate local arrays; error code = ',astat
	 call ENDRUN
      end if

#ifdef INIT_TEST
      bad_mem         = Z'7FF7FFFF7FF7FFFF'
      delp(:,:,:)     = bad_mem
      delpm(:,:,:)    = bad_mem
      delp1(:,:,:)    = bad_mem
      delp2(:,:,:)    = bad_mem
      pu(:,:,:)       = bad_mem
      dpi(:,:,:)      = bad_mem
      dq(:,:,:,:)     = bad_mem
      fx(:,:,:,:)     = bad_mem
      fy(:,:,:,:)     = bad_mem
      dps_ctm(:,:)    = bad_mem
      ua(:,:,:)       = bad_mem
      va(:,:,:)       = bad_mem
#endif

      call DATE_AND_TIME( cdate(1), ctime(1) )
      call CALC_DIVERGENCE( 4, xmass, ymass, dpi, platl )

      do j = jle(5),jue(5)
         dps_ctm(:,j) = pctm2(:,j) - pctm1(:,j)
      end do

      call SET_PRESS_TERMS( pctm1, pctm2, delp1, delp2, delpm, &
			    pu, platl )
      call SET_PRESS_TERMS( pctm1, pmet2, delp1, delp2_junk, delpm, &
			    pu, platl )

      do k = 1,plev
         do j = jle(5),jue(5)
            delp(:,j,k) = delp1(:,j,k) + dpi(:,j,k)
	 end do
      end do

      call CALC_COURANT( delpm, pu, xmass, ymass, crx, cry, platl )
      call SET_CROSS_TERMS( crx, cry, ua, va, platl )
      call CALC_VERT_MASS_FLUX( dpi, dps_ctm, w, platl )

      call DATE_AND_TIME( cdate(2), ctime(2) )
      elapsed(12) = elapsed(12) + TIME_DIFF( ctime(1), ctime(2) )

      call DATE_AND_TIME( cdate(1), ctime(1) )
!------------------------------------------------------------------------
!       ... Check and clear prior constituent isends and irecvs
!------------------------------------------------------------------------
#ifdef USE_MPI
      call MPI_WAITALL( 2, persist_recv, status, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'ADVECTION: MPI_WAITALL recv error = ',ierr
         call ENDRUN
      end if
      call MPI_WAITALL( 2, persist_send, status, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'ADVECTION: MPI_WAITALL send error = ',ierr
         call ENDRUN
      end if
#endif
      call DATE_AND_TIME( cdate(2), ctime(2) )
      elapsed(13) = elapsed(13) + TIME_DIFF( ctime(1), ctime(2) )
!     write(*,*) 'q recv time = ',elapsed(13)

!------------------------------------------------------------------------
!       ... Set jn and js from inidividual nodes
!------------------------------------------------------------------------
#ifdef USE_MPI
      call MPI_ALLREDUCE( js, jwrk, plev, MPI_INTEGER, MPI_MAX, mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: MPI_ALLREDUCE failed'
         write(*,*) '           Error code = ',ierr
	 call ENDRUN
      end if
      js(:) = jwrk(:)
      call MPI_ALLREDUCE( jn, jwrk, plev, MPI_INTEGER, MPI_MIN, mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: MPI_ALLREDUCE failed'
         write(*,*) '           Error code = ',ierr
	 call ENDRUN
      end if
      jn(:) = jwrk(:)
#endif

      call DATE_AND_TIME( cdate(1), ctime(1) )
!-----------------------------------------------------------------------
!       ... Task advection over tracers
!-----------------------------------------------------------------------
#ifdef USE_OMP
!$OMP PARALLEL DO private( ic ), schedule(dynamic,1)
#endif
      do ic = 1,nc
         call ADVECTXY( ic, delp, delp1, ua, va,             &
                        pu, crx, cry, xmass, ymass,                  &
                        dq(1,-1,1,ic), fx(1,-1,1,ic), fy(1,-2,1,ic), &
                        Tmin(ic), Tmax(ic), Bmin(ic), Bmax(ic), plonl, &
			platl, pplon )
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

#ifdef USE_MPI
      call MPI_BARRIER( mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: BARRIER error = ',ierr
	 call ENDRUN
      end if
      call MPI_ALLREDUCE( Tmin, Ttmp, nc, MPI_DOUBLE_PRECISION, MPI_MIN, mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: MPI_ALLREDUCE failed; Error code = ',ierr
	 call ENDRUN
      end if
      Tmin(:) = Ttmp(:)
      call MPI_ALLREDUCE( Bmin, Ttmp, nc, MPI_DOUBLE_PRECISION, MPI_MIN, mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: MPI_ALLREDUCE failed; Error code = ',ierr
	 call ENDRUN
      end if
      Bmin(:) = Ttmp(:)
      call MPI_ALLREDUCE( Tmax, Ttmp, nc, MPI_DOUBLE_PRECISION, MPI_MAX, mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: MPI_ALLREDUCE failed; Error code = ',ierr
	 call ENDRUN
      end if
      Tmax(:) = Ttmp(:)
      call MPI_ALLREDUCE( Bmax, Ttmp, nc, MPI_DOUBLE_PRECISION, MPI_MAX, mpi_comm_comp, ierr )
      if( ierr /= 0 ) then
         write(*,*) 'ADVECTION: MPI_ALLREDUCE failed; Error code = ',ierr
	 call ENDRUN
      end if
      Bmax(:) = Ttmp(:)
#endif
      call DATE_AND_TIME( cdate(2), ctime(2) )
      elapsed(14) = elapsed(14) + TIME_DIFF( ctime(1), ctime(2) )
      call DATE_AND_TIME( cdate(1), ctime(1) )
#ifdef USE_OMP
!$OMP PARALLEL DO private( ic ), schedule(dynamic,1)
#endif
      do ic = 1,nc
         call ADVECTZ( ic, delp, delp2, ua, w, &
                       dq(1,-1,1,ic), fx(1,-1,1,ic), fy(1,-2,1,ic), &
                       Tmin(ic), Tmax(ic), Bmin(ic), Bmax(ic), plonl, &
		       platl, pplon )
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif
      call DATE_AND_TIME( cdate(2), ctime(2) )
      elapsed(15) = elapsed(15) + TIME_DIFF( ctime(1), ctime(2) )

      DEALLOCATE( delp, delpm, delp1, delp2, pu, dpi, dq, fx, fy, &
                  delp2_junk, &
                  dps_ctm, ua, va, &
                  stat=astat )
      if( astat /= 0 ) then
        write(*,*) 'ADVECTION: Failed to deallocate local arrays; error = ',astat
        call ENDRUN
      end if

      end subroutine ADVECTION

      subroutine ADJUST_PRESS( pctm1, pmet2, pctm2, u, v, &
                               xmass, ymass, ndt, crx, cry, &
			       plonl, platl, pplon )
!-----------------------------------------------------------------------
!       ... Adjust pressures for Lin-Rood Transport
!-----------------------------------------------------------------------
 
      use MO_GRID,    only : plon, plev
      use MO_HISTOUT, only : OUTFLD, match_file_cnt
      use MO_TIMER,   only : TIME_DIFF, elapsed
      use M_ADV,      only : jl, ju, jue, j1, do_pressure_fixer
      use MO_ADV,     only : CONVERT_WINDS
      use MO_MPI,     only : thisnode

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: ndt                             ! timestep (s)
      integer, intent(in) :: plonl                           ! lon time dim
      integer, intent(in) :: platl                           ! lat time dim
      integer, intent(in) :: pplon                           ! lon tile count
      real, intent(in)    :: pctm1(plon,-3:platl+4), &       ! surface pressure at t(n)
                             pmet2(plon,-3:platl+4)          ! surface pressure at t(n+1) from met fields
      real, intent(out)   :: pctm2(plon,-3:platl+4)          ! predicted surface pressure at t(n+1)
      real, intent(in)    :: u(plon,-3:platl+4,plev), &      ! u wind component (m/s)
                             v(plon,-2:platl+3,plev)         ! v wind component (m/s)
      real, intent(out)   :: xmass(plon,-3:platl+4,plev), &  ! zonal mass flux
                             ymass(plon,-2:platl+3,plev)     ! meridional mass flux
      real, intent(out)   :: crx(plon,-3:platl+4,plev), &
                             cry(plon,-2:platl+3,plev)

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k, ip
      integer :: jb, jb1, jglob, jm1, ierr, astat
      integer :: file
      real    :: dl, dp, maxdt, d5, agle, ztc, dt, cr1, elapsed_time
      real    :: bad_mem
      real, allocatable, dimension(:,:,:) ::   &
           delpm, delp1, delp2, pu, dpi
      real, allocatable, dimension(:,:) :: dps, dps_ctm
      character(len=8)    :: cdate(2)
      character(len=10)   :: ctime(2)
#ifdef DEBUG
      real, dimension(plon,-3:platl+4) :: ddps
#endif

!-----------------------------------------------------------------------
!       ... Allocate locals
!-----------------------------------------------------------------------
      ALLOCATE( delpm(plon,-3:platl+4,plev), &
                delp1(plon,-3:platl+4,plev), &
                delp2(plon,-3:platl+4,plev), &
		pu(plon,-3:platl+4,plev), &
                dpi(plon,-3:platl+4,plev), &
                dps(plon,-3:platl+4), &
		dps_ctm(plon,-3:platl+4), &
                stat=astat )
      if( astat /= 0 ) then
         write(*,*)  &
            'ADJUST_PRESS: failed to allocate local arrays; error code = ',astat
         call ENDRUN
      end if

#ifdef INIT_TEST
      bad_mem         = Z'7FF7FFFF7FF7FFFF'
      delpm(:,:,:)    = bad_mem
      delp1(:,:,:)    = bad_mem
      delp2(:,:,:)    = bad_mem
      pu(:,:,:)       = bad_mem
      dpi(:,:,:)      = bad_mem
      dps_ctm(:,:)    = bad_mem
      dps(:,:)        = bad_mem
#endif

      call DATE_AND_TIME( cdate(1), ctime(1) )

      call SET_PRESS_TERMS( pctm1, pmet2, delp1, delp2, delpm, &
			    pu, platl )
      call CONVERT_WINDS( ndt, u, v, crx, cry, platl )
      call CALC_HORIZ_MASS_FLUX( delpm, pu, crx, cry, xmass, ymass, platl )
#ifdef DEBUG
      call CALC_DPS( pctm1, pmet2, dps, platl )
#endif
      if( do_pressure_fixer ) then
#ifndef DEBUG
         call CALC_DPS( pctm1, pmet2, dps, platl )
#else
         do ip = 1,pplon
            do j = 1,platl
               do file = 1,match_file_cnt
                  call OUTFLD( 'DPS-MET', dps((ip-1)*plonl+1,j), plonl, ip, j, file )
               end do
            end do
         end do
#endif
         call CALC_DIVERGENCE( 5, xmass, ymass, dpi, platl )
         call CALC_DPS2( dpi, dps_ctm, platl )
#ifdef DEBUG
         do j = 1,platl
            ddps(:,j) = dps_ctm(:,j) - dps(:,j)
         end do
         do ip = 1,pplon
            do j = 1,platl
               do file = 1,match_file_cnt
                  call OUTFLD( 'DPS-CTM0', dps_ctm((ip-1)*plonl+1,j), plonl, ip, j, file )
                  call OUTFLD( 'DDPS0', ddps((ip-1)*plonl+1,j), plonl, ip, j, file )
               end do
            end do
         end do
#endif
         call DO_PRESS_FIX_LLNL( dps, dps_ctm, xmass, ymass, platl )
      end if

      call CALC_DIVERGENCE( 5, xmass, ymass, dpi, platl )
      call CALC_DPS2( dpi, dps_ctm, platl )

#ifdef DEBUG
      do j = 1,platl
         ddps(:,j) = dps_ctm(:,j) - dps(:,j)
      end do
      do ip = 1,pplon
         do j = 1,platl
            do file = 1,match_file_cnt
               call OUTFLD( 'DPS-CTM', dps_ctm((ip-1)*plonl+1,j), plonl,  ip, j, file )
               call OUTFLD( 'DDPS', ddps((ip-1)*plonl+1,j), plonl, ip, j, file )
            end do
         end do
      end do
#endif

!     do j = jl(1),ju(1)
      do j = 1,platl
         pctm2(:,j) = pctm1(:,j) + dps_ctm(:,j)
      end do

      call DATE_AND_TIME( cdate(2), ctime(2) )
      elapsed(12) = elapsed(12) + TIME_DIFF( ctime(1), ctime(2) )

      DEALLOCATE( delpm, delp1, delp2, &
                  pu, dpi, dps, dps_ctm, &
                  stat=astat )
      if( astat /= 0 ) then
        write(*,*)  &
           'ADJUST_PRESS: Failed to deallocate local arrays; error = ',astat
        call ENDRUN
      end if

      end subroutine ADJUST_PRESS

      subroutine CALC_HORIZ_MASS_FLUX( delp2, pu, crx, cry, xmass, ymass, platl )
!-----------------------------------------------------------------------
!       ... Calculate the horizontal mass flux
!-----------------------------------------------------------------------

      use MO_GRID, only : plev, plon
      use M_ADV,   only : jl, ju, jle, jue, cose
      use MO_MPI,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: delp2(plon,-3:platl+4,plev), &  ! pressure thickness at t(n+1)
                           pu(plon,-3:platl+4,plev)        ! pressure thickness at edges (in lon)
      real, intent(in)  :: crx(plon,-3:platl+4,plev), &    ! Courant number in E-W direction
                           cry(plon,-2:platl+3,plev)       ! Courant number in N-S direction
      real, intent(out) :: xmass(plon,-3:platl+4,plev), &  ! zonal mass flux
                           ymass(plon,-2:platl+3,plev)     ! meridional mass flux
!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k
      real    :: d5

#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, d5 ), schedule(dynamic,1)
#endif
      do k = 1,plev
!-----------------------------------------------------------------------
!       ... North-South component
!-----------------------------------------------------------------------
         do j = jl(3),jue(2)
            d5 = .5 * cose(j+base_lat)
            do i = 1,plon
               ymass(i,j,k) = cry(i,j,k)*d5*(delp2(i,j,k) + delp2(i,j-1,k))
            end do
         end do
!-----------------------------------------------------------------------
!       ... East-West component
!-----------------------------------------------------------------------
         do j = jl(4),ju(4)
            do i = 1,plon
               xmass(i,j,k) = pu(i,j,k)*crx(i,j,k)
            end do
         end do
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine CALC_HORIZ_MASS_FLUX
 
      subroutine CALC_DIVERGENCE( jndx, xmass, ymass, dpi, platl )
!-----------------------------------------------------------------------
!       ... Calculate horizontal mass flux divergence
!-----------------------------------------------------------------------

      use MO_GRID, only : plev, plat, plon
      use M_ADV,   only : j1, jl, ju, rcap, acosp, jlim_north, large_cap
      use MO_MPI,  only : masternode, lastnode, base_lat

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: jndx
      integer, intent(in) :: platl
      real, intent(in)    :: xmass(plon,-3:platl+4,plev), &    ! zonal mass flux
                             ymass(plon,-2:platl+3,plev)       ! meridional mass flux
      real, intent(inout) :: dpi(plon,-3:platl+4,plev)         ! horizontal mass flux divergence

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k, jb, jb1
      real    :: d5

#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, k, jb, jb1, d5 ), schedule(dynamic,1)
#endif
      do k = 1,plev
!-----------------------------------------------------------------------
!       ... North-South component
!-----------------------------------------------------------------------
         do j = jl(jndx),ju(jndx)
            do i = 1,plon
               dpi(i,j,k) = (ymass(i,j,k) - ymass(i,j+1,k)) * acosp(j+base_lat)
            end do
         end do
!-----------------------------------------------------------------------
!       ... Poles
!-----------------------------------------------------------------------
         if( masternode ) then
            jb  = 1 - base_lat
            jb1 = j1 - base_lat
            d5  = -SUM( ymass(:plon,jb1,k) ) * rcap
            dpi(:plon,jb,k) = d5
            if( large_cap ) then                         ! Enlarged polar cap
               dpi(:plon,jb+1,k) = d5
            end if
         end if
         if( lastnode ) then
            jb  = plat - base_lat
            jb1 = jlim_north + 1 - base_lat
            d5  = SUM( ymass(:plon,jb1,k) ) * rcap
            dpi(:plon,jb,k) = d5
            if( large_cap ) then                         ! Enlarged polar cap
               dpi(:plon,jb-1,k) = d5
            end if
         end if
!-----------------------------------------------------------------------
!       ... East-West component
!-----------------------------------------------------------------------
         do j = jl(jndx),ju(jndx)
            do i = 1,plon-1
               dpi(i,j,k)    = dpi(i,j,k) + xmass(i,j,k) - xmass(i+1,j,k)
            end do
            dpi(plon,j,k) = dpi(plon,j,k) + xmass(plon,j,k) - xmass(1,j,k)
         end do
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine CALC_DIVERGENCE

      subroutine SET_PRESS_TERMS( ps1, ps2, delp1, delp2, delpm, &
				  pu, platl )
!-----------------------------------------------------------------------
!       ... Set pressure thicknesses of model levels
!-----------------------------------------------------------------------

      use MO_GRID, only : plev, plevp, plat, plon
      use M_ADV,   only : jle, jue, jl, ju, dap, dbk

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: ps1(plon,-3:platl+4), &              ! surface pressure at t(n)
                           ps2(plon,-3:platl+4)                 ! surface pressure at t(n+1)
      real, intent(out) :: delpm(plon,-3:platl+4,plev), &       ! pressure thickness at t(n+1/2)
                           delp1(plon,-3:platl+4,plev), &       ! pressure thickness at t(n)
                           delp2(plon,-3:platl+4,plev), &       ! pressure thickness at t(n+1)
                           pu(plon,-3:platl+4,plev)             ! pressure thickness at edges (in lon)

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k

!-----------------------------------------------------------------------
!       ... Compute horizontal mass fluxes
!           delpm = pressure thickness: the psudo-density in a hydrostatic system
!                   at t1+tdt/2 (mb)
!-----------------------------------------------------------------------
#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, k ), schedule(dynamic,1)
#endif
      do k = 1,plev
         do j = jle(7),jue(7)
            do i = 1,plon
               delp1(i,j,k) = dap(k) + dbk(k)*ps1(i,j)
               delp2(i,j,k) = dap(k) + dbk(k)*ps2(i,j)
               delpm(i,j,k) = dap(k) + dbk(k)*.5*( ps1(i,j) + ps2(i,j) )
            end do
            pu(1,j,k) = .5 * (delpm(1,j,k) + delpm(plon,j,k))
            do i = 2,plon
               pu(i,j,k) = .5 * (delpm(i,j,k) + delpm(i-1,j,k))
            end do
         end do
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine SET_PRESS_TERMS

      subroutine CALC_VERT_MASS_FLUX( dpi, dps_ctm, w, platl )
!-----------------------------------------------------------------------
!       ... Compute vertical mass flux (same dimensional unit as PS)
!           from continuity equation
!-----------------------------------------------------------------------

      use M_ADV,   only : dbk, jle, jue
      use MO_GRID, only : plev, plat, plon

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: dpi(plon,-3:platl+4,plev)  ! horizontal mass flux divergence
      real, intent(in)  :: dps_ctm(plon,-3:platl+4)   ! surface pressure change
      real, intent(out) :: w(plon,-1:platl+2,plev)    ! vertical mass flux

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k

      w(:,:,:) = 0.

#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, k )
#endif
      do j = jle(5),jue(5)
         do i = 1,plon
!-----------------------------------------------------------------------
!       ... Compute vertical mass flux from mass conservation principle.
!-----------------------------------------------------------------------
            w(i,j,1) = dpi(i,j,1) - dbk(1)*dps_ctm(i,j)
            w(i,j,plev) = 0.
         end do
         do k = 2,plev-1
            do i = 1,plon
               w(i,j,k) = w(i,j,k-1) + dpi(i,j,k) - dbk(k)*dps_ctm(i,j)
            end do
         end do
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine CALC_VERT_MASS_FLUX

      subroutine CALC_DPS( ps1, ps2, dps, platl )
!-----------------------------------------------------------------------
!       ... Calculate difference between surface pressures
!-----------------------------------------------------------------------

      use MO_GRID, only : plon

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: ps1(plon,-3:platl+4), &   ! surface pressure at t(n)
                           ps2(plon,-3:platl+4)      ! surface pressure at t(n+1)
      real, intent(out) :: dps(plon,-3:platl+4)      ! surface pressure change from met fields

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: j

      do j = 1,platl
         dps(:,j) = ps2(:,j) - ps1(:,j)
      end do

      end subroutine CALC_DPS

      subroutine CALC_DPS2( dpi, dps_ctm, platl )
!-----------------------------------------------------------------------
!       ... Calculate surface pressure change from model level divergence
!-----------------------------------------------------------------------

      use MO_GRID, only : plev, plon

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: dpi(plon,-3:platl+4,plev)    ! divergence at a grid point
      real, intent(out) :: dps_ctm(plon,-3:platl+4)     ! surface pressure change calc from L-R

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j

      do j = 1,platl
         do i = 1,plon
            dps_ctm(i,j) = sum( dpi(i,j,:plev) )
         end do
      end do

      end subroutine CALC_DPS2

      subroutine CALC_COURANT( delpm, pu, xmass, ymass, crx, cry, platl )
!-----------------------------------------------------------------------
!       ... Calculate courant numbers from the horizontal mass fluxes
!-----------------------------------------------------------------------

      use MO_MPI,  only : base_lat
      use MO_GRID, only : plev, plat, plon
      use M_ADV,   only : jl, ju, jle, jue, cose, j1, jlim_north, js0, jn0, js, jn

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)  :: delpm(plon,-3:platl+4,plev), &
                           pu(plon,-3:platl+4,plev)
      real, intent(in)  :: xmass(plon,-3:platl+4,plev), &
                           ymass(plon,-2:platl+3,plev) 
      real, intent(out) :: crx(plon,-3:platl+4,plev),   &
                           cry(plon,-2:platl+3,plev)

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, k, jglob

#ifdef USE_OMP
!$OMP PARALLEL DO private( i, j, k, jglob ), schedule(dynamic,1)
#endif
      do k = 1,plev
!-----------------------------------------------------------------------
!       ... Calculate E-W horizontal mass flux
!-----------------------------------------------------------------------
         do j = jle(6),jue(6)
            crx(:plon,j,k) = xmass(:plon,j,k) / pu(:plon,j,k)
         end do
!-----------------------------------------------------------------------
!       ... Calculate N-S horizontal mass flux
!-----------------------------------------------------------------------
!        do j = jle(4),jue(2)
         do j = jl(3),jue(2)
            cry(:plon,j,k) = ymass(:plon,j,k) &
                             /(.5*cose(j+base_lat) * (delpm(:plon,j,k) + delpm(:plon,j-1,k)))
         end do
!-----------------------------------------------------------------------
!       ... Find jn and js
!-----------------------------------------------------------------------
         js(k) = j1
         jn(k) = jlim_north
         do j = ju(7),jl(7),-1
            jglob = base_lat + j
            if( jglob > j1 .and. jglob <= js0 ) then
               if( ANY( ABS( crx(:plon,j,k) ) > 1. ) ) then
                  js(k) = jglob
                  exit
               end if
            end if
         end do

         do j = jl(7),ju(7)
            jglob = base_lat + j
            if( jglob >= jn0 .and. jglob < jlim_north ) then
               if( ANY( ABS( crx(:plon,j,k) ) > 1. ) ) then
                  jn(k) = jglob
                  exit
               end if
            end if
         end do
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine CALC_COURANT

      subroutine DO_PRESS_FIX_LLNL( dps, dps_ctm, xmass, ymass, platl )
!-----------------------------------------------------------------------
!       ... Fix the mass fluxes to match the met field pressure tendency
!-----------------------------------------------------------------------

      use MO_GRID,      only : plev, plat, plon
      use MO_CONSTANTS, only : latwts
      use M_ADV,        only : dbk, jl, ju, jle, jue, cosp, rcap
      use MO_MPI,       only : base_lat, masternode
#ifdef USE_MPI
      use MO_MPI,       only : MPI_DOUBLE_PRECISION, mpi_comm_comp, MPI_SUCCESS, mpi_in_place
#endif

      implicit none

!-----------------------------------------------------------------------
!       ... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: platl
      real, intent(in)    :: dps(plon,-3:platl+4)                 ! surface pressure change from met fields
      real, intent(in)    :: dps_ctm(plon,-3:platl+4)             ! vert. sum of dpi from original mass fluxes
      real, intent(inout) :: xmass(plon,-3:platl+4,plev), &       ! zonal mass flux
                             ymass(plon,-2:platl+3,plev)          ! meridional mass flux

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: i, j, jglob, k, astat, ierr
      real    :: dpress_g                     ! global pressure error
      real    :: fxmean, factor
      real    :: ddps(plon,-3:platl+4)        ! surface pressure change error
      real    :: dpresslat(plat)
      real    :: mmfd(plat)
      real    :: mmf(plat+1)
      real    :: fxintegral(plon+1)
      real    :: xcolmass_fix(plon,-3:platl+4)

#ifdef DEBUG
      ddps(:,:)         = Z'7FF0000000000001'
      xcolmass_fix(:,:) = Z'7FF0000000000001'
#endif

      do j = 1,platl
         ddps(:plon,j) = dps(:plon,j) - dps_ctm(:plon,j)
      end do
      factor = .5/real(plon)
      do j = 1,platl
         dpresslat(j+base_lat) = sum( ddps(:plon,j) ) * latwts(j+base_lat) * factor
      end do

#ifdef USE_MPI
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          dpresslat, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'DO_PRESS_FIX_LLNL: MPI_ALLGATHER failed; Error code = ',ierr
         call ENDRUN
      end if
#endif
      dpress_g = sum( dpresslat(:) )
      if( masternode ) then
         write(*,*) 'do_press_fix_llnl: dpress_g = ',dpress_g
      end if

!-----------------------------------------------------------------------
!     Calculate mean meridional flux divergence (df/dy).
!     Note that mmfd is actually the zonal mean pressure change,
!     which is related to df/dy by geometrical factors.
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     	... Handle non-Pole regions.
!-----------------------------------------------------------------------
      factor = 1./real(plon)
      do j = 1,platl
        mmfd(j+base_lat) = dpress_g - sum( ddps(:plon,j) ) * factor
      end do

#ifdef USE_MPI
      call MPI_ALLGATHER( mpi_in_place, platl, MPI_DOUBLE_PRECISION, &
                          mmfd, platl, MPI_DOUBLE_PRECISION, mpi_comm_comp, ierr )
      if( ierr /= MPI_SUCCESS ) then
         write(*,*) 'DO_PRESS_FIX_LLNL: MPI_ALLGATHER failed; Error code = ',ierr
         call ENDRUN
      end if
#endif

!-----------------------------------------------------------------------
!     Calculate mean meridional fluxes (cosp*fy).
!     NB: This calculation is being done for global lats, i.e., (1,plat)
!-----------------------------------------------------------------------

      mmf(1) = 0.
      mmf(2) = sum( mmfd(1:2)*latwts(1:2) ) / (sum( latwts(1:2) )*rcap*REAL(plon))
      mmf(3) = mmf(2)

      do j = 3,plat-2
        mmf(j+1) = mmf(j) + mmfd(j) * cosp(j)
      end do

      mmf(plat)   = mmf(plat-1)
      mmf(plat+1) = 0.

!-----------------------------------------------------------------------
!     Fix latitude bands.
!     Note that we do not need to worry about geometry here because
!     all boxes in a latitude band are identical.
!     Note also that fxintegral(plon+1) should equal fxintegral(1),
!     i.e., zero.
!-----------------------------------------------------------------------
#ifdef USE_OMP
!$OMP PARALLEL DO private(i,j,k,fxmean,fxintegral)
#endif
      do j = jl(5),ju(5)
         fxintegral(:) = 0.
         do i = 1,plon
            fxintegral(i+1) = fxintegral(i) - (ddps(i,j) - dpress_g) - mmfd(j+base_lat)
         end do
         fxintegral(1) = fxintegral(plon+1)
         fxmean        = sum( fxintegral(:plon) ) * factor
         xcolmass_fix(:plon,j) = fxintegral(:plon) - fxmean
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

!-----------------------------------------------------------------------
!     	... Distribute colmass_fix in vertical
!-----------------------------------------------------------------------
#ifdef USE_OMP
!$OMP PARALLEL DO private(k,j), schedule(dynamic,1)
#endif
      do k = 1,plev
         do j = jl(5),ju(5)
           xmass(:plon,j,k) = xmass(:plon,j,k) + xcolmass_fix(:plon,j) * dbk(k)
           ymass(:plon,j,k) = ymass(:plon,j,k) + mmf(j+base_lat) * dbk(k)
         end do
         ymass(:plon,ju(5)+1,k) = ymass(:plon,ju(5)+1,k) + mmf(ju(5)+1+base_lat) * dbk(k)
      end do
#ifdef USE_OMP
!$OMP END PARALLEL DO
#endif

      end subroutine DO_PRESS_FIX_LLNL

      end module MO_FFSLADV
