
      module mo_file_utils

      integer :: arch_unit
      character(len=128) :: arch_file = ' '
      logical :: arch_file_opened = .false.
      logical :: do_archive       = .false.

      private
      public  :: acquire_file, attach, dispose, close_arch
      public  :: navu, open_netcdf_file, freeunit, make_filename
      public  :: arch_file

      contains

      integer function acquire_file( lpath, rpath, lun, binary, cosb, asynch )
!-----------------------------------------------------------------------
! 	... Check to see if specified local file exists.  If not,
!           acquire it from remote storage.  Then open as an unformatted
!           file (if binary) or a formatted file (if not binary).
!
! 	    This routine contains locally written NCAR specific code for:
! 1) Acquiring files that dont reside on the local disk from the NCAR Mass
!    Storage System (msrcp).
!
!           Return values:
!           0 => successful return
!-----------------------------------------------------------------------
      
      use mo_charutl, only : lastsl

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: &
        lpath, &   ! pathname for local file
        rpath      ! pathname for remote file

      logical, intent(in) :: &
        binary, &  ! .true. => file is binary (unformatted), else assume ascii (formatted).
        asynch     ! .true. => asynch mscp

      integer, intent(inout) :: &
           lun     ! the unit number (input) to open file on.

      logical, intent(out) :: &
        cosb       ! true => file is COS blocked

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        slash, &   ! position of last slash in a pathname
        istat, &   ! return value from ishell
        i

      logical :: &
        there      ! test for existence of file
      
      character(len=256) :: &
        ctmp, &    ! tmp string for commands and file names
        opts       ! mscp options string
      character(len=8) :: &
        pass, &    ! msrcp passwrd string
        cdate      ! date string
      character(len=10) :: &
        ctime      ! time string

!-----------------------------------------------------------------------
!  	... Function declarations
!-----------------------------------------------------------------------
      integer :: fsystem

!-----------------------------------------------------------------------
!  	... Check for the existence of the local file
!-----------------------------------------------------------------------
      inquire( file=trim(lpath), exist=there )

      if( there ) then
         write(*,*) 'acquire_file: using local disk copy of ',trim(lpath)
      else
#ifdef NCAR
!-----------------------------------------------------------------------
!        Create the local directory if necessary.
!-----------------------------------------------------------------------
         slash = lastsl( lpath )
         if( slash > 1 ) then
            ctmp = 'mkdir -p ' // lpath(:slash)
            istat = fsystem( ctmp )
!-----------------------------------------------------------------------
!     	... Ignore error return... directory may already exist.
!-----------------------------------------------------------------------
         end if
!-----------------------------------------------------------------------
!  	... Acquire remote file from NCAR mass store.
!           This code will need modification at non-NCAR sites.
!-----------------------------------------------------------------------
         opts = ' '
         if( asynch ) then
            opts = '-a'
         end if
         ctmp  = 'msrcp ' // trim(opts) // ' mss:' // trim(rpath) // ' ' // trim(lpath)
         call date_and_time( cdate, ctime )
         write(*,*) 'acquire_file: fsystem issuing command: at time ',cdate,' ',ctime
         write(*,*) '       ',trim(ctmp)
         ctmp  = trim(ctmp) // char(0)
         istat = fsystem( trim(ctmp) )
         if( istat /= 0 ) then
            write(*,*) 'acquire_file: msrcp error = ',istat
            call endrun
         end if
#elif defined(GFDL)
!-----------------------------------------------------------------------
!        Create the local directory if necessary.
!-----------------------------------------------------------------------
         slash = lastsl( lpath )
         if( slash > 1 ) then
            ctmp = 'mkdir -p ' // lpath(:slash)
            istat = fsystem( ctmp )
!-----------------------------------------------------------------------
!     	... Ignore error return... directory may already exist.
!-----------------------------------------------------------------------
         end if
!-----------------------------------------------------------------------
!  	... Acquire remote file from GFDL archive
!-----------------------------------------------------------------------
         ctmp  = ' '
         ctmp  = 'dmcopy ' // trim(rpath) // ' ' // trim(lpath)
         if( asynch ) then
            ctmp = trim(ctmp) // ' &'
         end if
         call date_and_time( cdate, ctime )
         write(*,*) 'acquire_file: fsystem issuing command: at time ',cdate,' ',ctime
         write(*,*) '       ',trim(ctmp)
         istat = fsystem( ctmp ) 
         if( istat /= 0 ) then
            write(*,*) 'acquire_file: error = ',istat
            call endrun
         end if
#else
         write(*,*) 'acquire_file: file ' // trim(lpath) // ' not found ... aborting'
         call endrun
#endif

      end if

      if( .not. asynch ) then
         acquire_file = open_file( lpath, lun, binary, cosb )
      else
         acquire_file = 0
      end if

      end function acquire_file
      
      integer function open_file( lpath, lun, binary, cosb )
!-----------------------------------------------------------------------
! 	... Open local file
!-----------------------------------------------------------------------
      
      use mo_charutl, only : lastsl

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: &
        lpath      ! pathname for local file

      logical, intent(in) :: &
        binary     ! .true. => file is binary (unformatted), else assume ascii (formatted).

      integer, intent(inout) :: &
           lun  ! the unit number (input) to open file on.

      logical, intent(out) :: &
        cosb    ! true => file is COS blocked

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        slash, &   ! position of last slash in a pathname
        istat, &   ! return value from ishell
        i

      logical :: &
        there      ! test for existence of file
      
      character(len=256) :: &
        ctmp, &    ! tmp string for commands and file names
        opts       ! msrcp options string
      character(len=8) :: &
        pass       ! msrcp passwrd string

!-----------------------------------------------------------------------
!  	... Function declarations
!-----------------------------------------------------------------------
      integer :: fsystem

      open_file = 0

      if( binary ) then
         ctmp = trim(lpath) // char(0)
         cosb = .false.
!-----------------------------------------------------------------------
!    	... Open binary file
!-----------------------------------------------------------------------
         write(*,*) 'open_file: opening binary file'
         open( unit = lun, &
               file = trim(lpath), &
               status = 'OLD', &
               form = 'UNFORMATTED', &
#ifdef DEC
               convert = 'BIG_ENDIAN', &
#endif
               iostat = istat )
         if( istat /= 0 ) then
            write(*,*) 'open_file: Failed to open file ',trim(lpath)
            call endrun
         end if
      else
!-----------------------------------------------------------------------
!    	... Open ASCII file.
!-----------------------------------------------------------------------
         write(*,*) 'open_file: opening ascii file'
         open( unit = lun, &
               file = trim(lpath), &
               status = 'OLD', &
               iostat = istat )
         if( istat /= 0 ) then
            write(*,*) 'open_file: Failed to open file ',trim(lpath)
            call endrun
         end if
      end if

      end function open_file

      integer function open_netcdf_file( filename, lpath, mspath, use_io_node, abort, masteronly )
!-----------------------------------------------------------------------
!       ... obtain and open a NetCDF data file
!-----------------------------------------------------------------------

      use netcdf, only : nf_open, nf_nowrite
#ifdef USE_MPI
      use mo_mpi, only : masternode, io_node, mpi_comm_comp, mpi_comm_world, mpi_success
#else
      use mo_mpi, only : masternode
#endif

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in)           :: filename, lpath
      character(len=*), optional, intent(in) :: mspath
      logical, optional, intent(in)          :: use_io_node
      logical, optional, intent(in)          :: abort
      logical, optional, intent(in)          :: masteronly

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer :: ierr, ncid, communicator
      logical :: there
      logical :: use_masteronly
      character(len=256) :: ctmp

!-----------------------------------------------------------------------
!       ... Function declarations
!-----------------------------------------------------------------------
      integer :: fsystem

#ifdef USE_MPI
      if( present( masteronly ) ) then
         use_masteronly = masteronly
      else
         use_masteronly = .false.
      end if
      if( .not. present( use_io_node ) ) then
         communicator = mpi_comm_comp
      else
         communicator = MPI_COMM_WORLD
      end if
master_only : &
      if( (communicator == mpi_comm_comp .and. masternode) .or. &
          (communicator == MPI_COMM_WORLD .and. io_node) ) then
#else
      use_masteronly = .true.
master_only : &
      if( masternode ) then
#endif
         inquire( file=trim( lpath ) // trim( filename ), exist=there )
file_exists : &
         if( there ) then
            write(*,*) 'open_netcdf_file: using local disk copy of ' // &
                       trim( lpath ) // trim( filename )
         else file_exists
#ifdef NCAR
!-----------------------------------------------------------------------
!        Acquire remote file from ncar mass store.
!        This code will need modification at non-ncar sites.
!-----------------------------------------------------------------------
            if( present( mspath ) ) then
               ctmp = ' '
               ctmp = 'msrcp mss:' // trim( mspath ) // trim( filename ) // ' ' // &
                      trim( lpath ) // trim( filename )
               write(*,*) 'open_netcdf_file: fsystem issuing command --'
               write(*,*) trim( ctmp )
               ctmp = trim(ctmp) // char(0)
               ierr = fsystem( ctmp )
               if ( ierr /= 0 ) then
                  if( present( abort ) ) then
                     if( .not. abort ) then
                        write(*,*) 'open_netcdf_file: fsystem can''t read file from mass store'
                        open_netcdf_file = -1
                        return
                     end if
                  end if
                  write(*,*) 'open_netcdf_file: fsystem can''t read file from mass store'
                  call endrun
               end if
            else
               if( present( abort ) ) then
                  if( .not. abort ) then
                     write(*,*) 'open_netcdf_file: file ' // trim(lpath) // trim(filename) &
                                                          // ' not found'
                     open_netcdf_file = -1
                     return
                  end if
               end if
               write(*,*) 'open_netcdf_file: file ' // trim(lpath) // trim(filename) &
                          // ' not found ... aborting'
               call endrun
            end if
#elif defined(GFDL)
!-----------------------------------------------------------------------
!        Acquire remote file from GFDL archive.
!-----------------------------------------------------------------------
            if( present( mspath ) ) then
               ctmp = ' '
               ctmp = 'dmcopy ' // trim( mspath ) // trim( filename ) // ' ' // &
                      trim( lpath ) // trim( filename )
               write(*,*) 'open_netcdf_file: fsystem issuing command --'
               write(*,*) trim( ctmp )
               ierr = fsystem( ctmp )
               if( ierr /= 0 ) then
                  if( present( abort ) ) then
                     if( .not. abort ) then
                        write(*,*) 'open_netcdf_file: fsystem can''t read file from archive'
                        open_netcdf_file = -1
                        return
                     end if
                  end if
                  write(*,*) 'open_netcdf_file: fsystem can''t read file from archive'
                  call endrun
               end if
            else
               if( present( abort ) ) then
                  if( .not. abort ) then
                     write(*,*) 'open_netcdf_file: file ' // trim(lpath) // trim(filename) &
                                                          // ' not found'
                     open_netcdf_file = -1
                     return
                  end if
               end if
               write(*,*) 'open_netcdf_file: file ' // trim(lpath) // trim(filename) &
                          // ' not found ... aborting'
               call endrun
            end if
#else
            if( present( abort ) ) then
               if( .not. abort ) then
                  write(*,*) 'open_netcdf_file: file ' // trim(lpath) // trim(filename) &
                                                       // ' not found'
                  open_netcdf_file = -1
                  return
               end if
            end if
            write(*,*) 'open_netcdf_file: file ' // trim(lpath) // trim(filename) &
                       // ' not found ... aborting'
            call endrun
#endif
         end if file_exists
      end if master_only
#ifdef USE_MPI
      if( .not. use_masteronly ) then
!---------------------------------------------------------------------------
!     	... All nodes wait for master to acquire the netcdf file
!---------------------------------------------------------------------------
         call mpi_barrier( communicator, ierr )
         if( ierr /= mpi_success ) then
            write(*,*) 'open_netcdf_file: Barrier failed; code = ',ierr
            call endrun
         end if
      end if
#endif
      if( use_masteronly ) then
         if( masternode ) then
!---------------------------------------------------------------------------
!     	... master node opens NetCDF file
!---------------------------------------------------------------------------
            call handle_ncerr( nf_open( trim( lpath ) // trim( filename ), nf_nowrite, ncid ), &
                               'open_netcdf_file: Failed to nf_open ' &
                               // trim( lpath ) // trim( filename ) )
         open_netcdf_file = ncid
         end if
      else
!---------------------------------------------------------------------------
!     	... All nodes open NetCDF file
!---------------------------------------------------------------------------
         call handle_ncerr( nf_open( trim( lpath ) // trim( filename ), nf_nowrite, ncid ), &
                            'open_netcdf_file: Failed to nf_open ' &
                            // trim( lpath ) // trim( filename ) )
         open_netcdf_file = ncid
      end if
      if( masternode ) then
         write(*,*) 'open_netcdf_file: Opened NetCDF file ',trim( lpath ) // trim( filename )
      end if

      end function open_netcdf_file

      integer function attach( lpath, rpath, lun, binary, cosb )
!-----------------------------------------------------------------------
! 	... Check to see if specified local file exists.  If not,
!           acquire it from remote storage.  Then open as an unformatted
!           file (if binary) or a formatted file (if not binary).
!
!           This routine contains locally written NCAR specific code for:
! 1) Acquiring files that dont reside on the local disk from the NCAR Mass
!    Storage System (msrcp).
!
!           Return values:
!           0 => successful return
!          -2 => error reading remote file
!-----------------------------------------------------------------------
      
      use mo_charutl, only : lastsl, glc

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: &
        lpath, &   ! pathname for local file
        rpath      ! pathname for remote file

      logical, intent(in) :: &
        binary     ! .true. => file is binary (unformatted), else assume ascii (formatted).

#if defined(CRAY)
      integer, intent(in) :: &
#else
      integer, intent(inout) :: &
#endif
           lun    ! the unit number (input) to open file on.

      logical, intent(out) :: &
        cosb      ! true => file is COS blocked


!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        slash, &   ! position of last slash in a pathname
        istat, &   ! return value from ishell
        i

      logical :: &
        there      ! test for existence of file
      
      character(len=256) :: &
        ctmp       ! tmp string for commands and file names

!-----------------------------------------------------------------------
!  	... Function declarations
!-----------------------------------------------------------------------
      integer :: fsystem

!-----------------------------------------------------------------------
!  	... Check for the existence of the local file
!-----------------------------------------------------------------------
      inquire( file=trim(lpath), exist=there )

      if( there ) then
         write(*,*) 'attach: using local disk copy of ',trim(lpath)
      else
#ifdef NCAR
!-----------------------------------------------------------------------
!        Create the local directory if necessary.
!-----------------------------------------------------------------------
         slash = lastsl( lpath )
         if( slash > 1 ) then
            ctmp(:) = ' '
            ctmp = 'mkdir -p ' // lpath(:slash)
            istat = fsystem( ctmp )
!-----------------------------------------------------------------------
!     	... Ignore error return... directory may already exist.
!-----------------------------------------------------------------------
         end if

!-----------------------------------------------------------------------
!        Acquire remote file from NCAR mass store.
!        This code will need modification at non-NCAR sites.
!-----------------------------------------------------------------------
         ctmp(:) = ' '
         ctmp  = 'msrcp mss:' // trim(rpath) // ' ' // trim(lpath)
         write(*,*) 'attach: fsystem issuing command - '
         write(*,*) trim(ctmp)
         ctmp = trim(ctmp) // char(0)
         istat = fsystem( ctmp )
         if( istat /= 0 ) then
            write(*,*) 'attach - fsystem: can''t acquire file from mass store'
            attach = -2
            return
         end if
#elif defined(GFDL)
!-----------------------------------------------------------------------
!        Create the local directory if necessary.
!-----------------------------------------------------------------------
         slash = lastsl( lpath )
         if( slash > 1 ) then
            ctmp(:) = ' '
            ctmp = 'mkdir -p ' // lpath(:slash)
            istat = fsystem( ctmp )
!-----------------------------------------------------------------------
!     	... Ignore error return... directory may already exist.
!-----------------------------------------------------------------------
         end if

!-----------------------------------------------------------------------
!        Acquire remote file from GFDL archive
!-----------------------------------------------------------------------
         ctmp(:) = ' '
         ctmp  = 'dmcopy ' // trim(rpath) // ' ' // trim(lpath) // char(0)
         write(*,*) 'attach: fsystem issuing command - '
         write(*,*) ctmp(:glc(ctmp))
         istat = fsystem( ctmp )
         if( istat /= 0 ) then
            write(*,*) 'attach - fsystem: can''t read file from archive'
            attach = -2
            return
         end if
#else
         write(*,*)'attach: file ',trim(lpath),' not found ... aborting'
         call endrun
#endif
      end if

      if( binary ) then
!-----------------------------------------------------------------------
!  	... Check for COS blocked file
!-----------------------------------------------------------------------
         do i = 1,len(ctmp)
            ctmp(i:i) = ' '
         end do
         ctmp = lpath(:glc(lpath)) // char(0)
         cosb = .false.

!-----------------------------------------------------------------------
!    	... Open binary file
!-----------------------------------------------------------------------
#ifdef CRAY
         if( cosb ) then
            write(*,*) 'attach: opening COS blocked file'
         else
            write(*,*) 'attach: opening binary file'
            call asnunit( lun, '-F f77 -N ieee', istat )
         end if
         open( unit = lun, &
               file = lpath(:glc(lpath)), &
               status = 'OLD', &
               form = 'UNFORMATTED', &
               iostat = istat )
	 if( istat /= 0 ) then
	    write(*,*) 'attach: Failed to open file ',lpath(:glc(lpath))
	    call endrun
	 end if
#else
         if( .not. cosb ) then
            write(*,*) 'attach: opening binary file'
            open( unit = lun, &
                  file = lpath(:glc(lpath)), &
                  status = 'OLD', &
                  form = 'UNFORMATTED', &
#ifdef DEC
                  convert = 'BIG_ENDIAN', &
#endif
                  iostat = istat )
	    if( istat /= 0 ) then
	       write(*,*) 'attach: Failed to open file ',lpath(:glc(lpath))
	       call endrun
	    end if
         end if
#endif
      else
!-----------------------------------------------------------------------
!    	... Open ASCII file.
!-----------------------------------------------------------------------
         write(*,*) 'attach: opening ascii file'
         open( unit = lun, &
               file = lpath(:glc(lpath)), &
               status = 'OLD', &
               iostat = istat )
	 if( istat /= 0 ) then
	    write(*,*) 'attach: Failed to open file ',lpath(:glc(lpath))
	    call endrun
	 end if
      end if

      attach = 0

      end function attach

      integer function navu( )
!-----------------------------------------------------------------------
!	... Utility function to return index of next
!	    available Fortran unit number.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: unit      ! Index
      logical :: inuse(99) = .false.

      do unit = 20,99
         if( .not. inuse(unit) ) then
            navu = unit
            inuse(unit) = .true.
            return
         end if
      end do

      write(*,*) 'navu: Ran out of Fortran unit numbers'
      call endrun

      end function navu 

      subroutine freeunit( unit_no )
!-----------------------------------------------------------------------
!	... Utility function to free unit
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: unit_no             ! unit number

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: unit      ! Index
      logical :: inuse(99) = .false.

      if( unit_no >= 20 .and. unit_no <= 99 ) then
         if( inuse(unit_no) ) then
            inuse(unit_no) = .false.
         end if
      end if

      end subroutine freeunit 

      subroutine dispose( locpath, mspath, irt, async, comment, &
                          mvn, pass, remove, lun, msclass )
!-----------------------------------------------------------------------
! 	... close history file, link if requested, dispose to mass store
!           if irt > 0, and remove if requested.
!-----------------------------------------------------------------------
      use mo_charutl, only : lastsl

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: irt                     !  Mass Store retention time
      integer, optional, intent(in) :: lun           !  unit number

      character(len=*), intent(in)  :: locpath       !  Local filename
      character(len=*), intent(in)  :: mspath        !  Mass Store full pathname
      character(len=*), intent(in)  :: comment       !  Mass Store comment field
      character(len=*), intent(in)  :: mvn           !  Mass Store cartridge name
      character(len=*), intent(in)  :: pass          !  write password
      character(len=*), optional, intent(in) :: msclass  !  mass store class

      logical, intent(in) :: async                   !  If true, use "NOWAIT,NOMAIL" option
      logical, intent(in) :: remove                  !  If true, rm local file after dispose

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: ier, slash
      character(len=256) :: &
        opts, &     !  Options for mswrite
        cmd, &      !  Command string
        cmdtem      !  Temporary for command string

!-----------------------------------------------------------------------
! 	... function declarations
!-----------------------------------------------------------------------
      integer :: fsystem

      if( present( lun ) ) then
         close( lun )
      end if

#ifdef NCAR
!-----------------------------------------------------------------------
!     	... dispose dataset to Mass Store only if irt.gt.0
!-----------------------------------------------------------------------
dispose_file : &
      if( irt /= 0 ) then
         write(*,*) '-------------------------------------------------------------'
         write(*,*) 'dispose: locpath = ',trim(locpath)
         write(*,*) 'dispose: mspath  = ',trim(mspath)
         write(*,*) '-------------------------------------------------------------'
         if( .not. arch_file_opened ) then
            arch_unit = navu()
            arch_file_opened = .true.
            open( unit=arch_unit,file='arch/'//trim(arch_file),iostat=ier )
            if( ier /= 0 ) then
               write(*,*) '**********************************************************************************'
               write(*,*) 'dispose: failed to open archive intermediate file ',trim(arch_file),'; error = ',ier 
               write(*,*) '**********************************************************************************'
            else
               do_archive = .true.
               write(*,*) 'dispose: opened archive intermediate file ',trim(arch_file),' on unit ',arch_unit
            end if
         end if
         if( do_archive ) then
            write(*,*) 'dispose: writing archive intermediate file ',trim(arch_file),' on unit ',arch_unit
            write(arch_unit,'(i4)') min( irt,9999 )
            if( present(msclass) ) then
               if( msclass == 'EC' ) then
                  write(arch_unit,'(a)') 'EC'
               else
                  write(arch_unit,'(a)') 'ST'
               end if
            else
               write(arch_unit,'(a)') 'ST'
            end if
            write(arch_unit,'(a)') trim(locpath)
            write(arch_unit,'(a)') trim(mspath)
            if( remove ) then
               write(arch_unit,'(a)') '1'
            else
               write(arch_unit,'(a)') '0'
            end if
         end if
      end if dispose_file
#elif defined(GFDL)
!-----------------------------------------------------------------------
!     	... Dispose dataset to archive only if irt.gt.0
!-----------------------------------------------------------------------
      if( irt /= 0 ) then
!-----------------------------------------------------------------------
!        Create the archive directory if necessary.
!-----------------------------------------------------------------------
         slash = lastsl( mspath )
         if( slash > 1 ) then
            cmd = ' '
            cmd = 'mkdir -p ' // mspath(:slash)
            ier = fsystem( cmd )
!-----------------------------------------------------------------------
!     	... Ignore error return... directory may already exist.
!-----------------------------------------------------------------------
         end if

!-----------------------------------------------------------------------
!	... Build a shell command for the write to archive.
!           Include a cp to archive and a dmput command. (Except for
!           rstrt info file, which cannot be dmput b/c of small size.)
!-----------------------------------------------------------------------
         cmd = ' '
         cmd = 'cp ' // trim(locpath) // ' ' // trim(mspath)
         cmdtem = ' '
         if( locpath( lastsl(locpath)+1 : len_trim(locpath) ) /= 'rstrt' ) then
            cmdtem = 'dmput'
            if( remove ) then
               cmdtem = trim(cmdtem) // ' -r'
            end if
            cmdtem = trim(cmdtem) // ' ' // trim(mspath)
            cmd = trim(cmd) // ' ; ' // trim(cmdtem)
         end if
         if( async ) then
            cmd = trim(cmd) // ' &'
         end if
         write(*,*) 'dispose: writing to archive as follows:'
         write(*,*) trim(cmd)
         ier = fsystem( cmd )
         if( ier /= 0 ) then
            write(*,*) 'dispose: error = ',ier,' writing ',trim(locpath),' to archive'
         end if
      end if
!-----------------------------------------------------------------------
!     	... remove disk copy if requested
!-----------------------------------------------------------------------
      if( remove ) then
         cmd = 'rm -f ' // trim(locpath)
         write(*,*) 'dispose: fsystem issuing command - '
         write(*,*) trim(cmd)
         ier = fsystem( cmd )
      end if
#endif

      end subroutine dispose

      subroutine close_arch
!-----------------------------------------------------------------------
!     	... if required close archive intermediate file
!-----------------------------------------------------------------------

      implicit none

      if( do_archive ) then
         write(*,*) ' '
         write(*,*) 'close_arch: closing unit ',arch_unit
         close( arch_unit )
      end if

      end subroutine close_arch

      subroutine make_filename( date, sec, fileno, is_monthly, case, fullpath )
!-----------------------------------------------------------------------
!	... form restart history full filename
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)             :: date
      integer, intent(in)             :: sec
      integer, intent(in)             :: fileno
      character(len=*), intent(in)    :: case
      character(len=*), intent(out)   :: fullpath
      logical, intent(in)             :: is_monthly

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer            :: yr, mnth, day
      character(len=6)   :: time_string
      character(len=20)  :: fl_suffix

      if( fileno /= 0 ) then
        fullpath  = trim( case ) // '.mz4.h'
        write(fullpath(len_trim(fullpath)+1:),'(i1)') fileno-1
      else
        fullpath  = trim( case ) // '.mz4.r'
      end if

      yr = date/10000 + 10000
      write(time_string(1:5),'(i5)') yr
      fl_suffix = time_string(2:5) // '-'

      mnth = mod( date,10000 )/100 + 100
      write(time_string(1:3),'(i3)') mnth

      if( .not. is_monthly ) then
        fl_suffix = trim( fl_suffix ) // time_string(2:3) // '-'
        day = mod( mod( date,1000 ),100 ) + 100
        write(time_string(1:3),'(i3)') day
        fl_suffix = trim( fl_suffix ) // time_string(2:3) // '-'
        write(time_string,'(i6)') 100000 + sec
        fl_suffix = trim( fl_suffix ) // time_string(2:6)
      else
        fl_suffix = trim( fl_suffix ) // time_string(2:3)
      end if

      fullpath  = trim( fullpath ) // '.' // trim( fl_suffix ) // '.nc'

      end subroutine make_filename

      end module mo_file_utils
