
      module MO_GHOST_TRANSFER

      private
      public :: GHOST_XFER

      CONTAINS

      subroutine GHOST_XFER( as, nadv, x1, plonl, pplon, platl )
!---------------------------------------------------------------------
!	... Ghost point mpi transfer
!---------------------------------------------------------------------

      use MO_MPI
      use MO_GRID, only : plev, pcnst
      use M_ADV,   only : mmr

      implicit none

!---------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) :: &
        nadv         ! number of misc advected variables
      integer, intent(in) :: &
        plonl, pplon, platl
      real, intent(in) :: &
        as(plonl,plev,pcnst,platl,pplon), &    ! species mass mixing ratio before advection
        x1(plonl,plev,nadv,platl,pplon)        ! miscellaneous variables before advection

!---------------------------------------------------------------------
! 	... Local variables
!---------------------------------------------------------------------
      integer :: ip, j, k, m, ierr, off

!---------------------------------------------------------------------
! 	... Reorder variables for advection
!---------------------------------------------------------------------
#ifdef DEBUG
      mmr = Z'7FF0000000000001'
#endif
      if( pplon > 1 ) then
!$OMP PARALLEL DO private( ip, j, k, m, off )
         do ip = 1,pplon
	    off = (ip - 1)*plonl
            do m = 1,pcnst-1
               do j = 1,platl
	          do k = 1,plev
	             mmr(off+1:off+plonl,j,k,m) = as(:plonl,k,m,j,ip)
	          end do
	       end do
            end do
            if( nadv > 0 ) then
               do m = 1,nadv
                  do j = 1,platl
	             do k = 1,plev
	                mmr(off+1:off+plonl,j,k,m+pcnst-1) = x1(:plonl,k,m,j,ip)
	             end do
	          end do
               end do
            end if
         end do
!$OMP END PARALLEL DO
      else
         do m = 1,pcnst-1
            do j = 1,platl
	       do k = 1,plev
	          mmr(:plonl,j,k,m) = as(:plonl,k,m,j,1)
	       end do
	    end do
         end do
         if( nadv > 0 ) then
            do m = 1,nadv
               do j = 1,platl
	          do k = 1,plev
	             mmr(:plonl,j,k,m+pcnst-1) = x1(:plonl,k,m,j,1)
	          end do
	       end do
            end do
         end if
      end if

!---------------------------------------------------------------------
! 	... Average advected variables at polar caps
!---------------------------------------------------------------------
      call AVERAGE_CONST_POLES( pcnst - 1 + nadv, platl )

#ifdef USE_MPI
!---------------------------------------------------------------------
! 	... Send ghost tracer lats
!---------------------------------------------------------------------
      call MPI_STARTALL( 2, persist_recv, ierr )
      if( ierr /= 0 ) then
	 write(*,*) 'GHOST_XFER: MPI_STARTALL recv error = ',ierr
	 call ENDRUN
      end if
      call MPI_STARTALL( 2, persist_send, ierr )
      if( ierr /= 0 ) then
	 write(*,*) 'GHOST_XFER: MPI_STARTALL send error = ',ierr
	 call ENDRUN
      end if
#endif

      end subroutine GHOST_XFER

      subroutine AVERAGE_CONST_POLES( nadvected, platl )
!-----------------------------------------------------------------------
!       ... Average tracers at the poles.
!           When the enlarged polar cap is used, make tracer
!           concentrations at the last two latitudes equal
!-----------------------------------------------------------------------

      use MO_MPI,       only : masternode, lastnode, base_lat
      use MO_GRID,      only : plon, plev
      use MO_CONSTANTS, only : latwts
      use M_ADV,        only : mmr

      implicit none

!---------------------------------------------------------------------
! 	... Dummy arguments
!---------------------------------------------------------------------
      integer, intent(in) :: &
        platl
      integer, intent(in) :: &
        nadvected                                      ! number of advected variables

!-----------------------------------------------------------------------
!       ... Local variables
!-----------------------------------------------------------------------
      integer :: j, k, n
      real    :: meanmmr

      if( masternode ) then
         do n = 1,nadvected
            do k = 1,plev
               meanmmr  = sum( mmr(:,1,k,n) ) * latwts(base_lat + 1) &
                          + sum( mmr(:,2,k,n) ) * latwts(base_lat + 2)
               meanmmr  = meanmmr / (real(plon) * sum( latwts(base_lat+1:base_lat+2) ))
	       do j = 1,2
                  mmr(:,j,k,n) = meanmmr
	       end do
            end do
         end do
      end if
      if ( lastnode ) then
         do n = 1,nadvected
            do k = 1,plev
               meanmmr  = sum( mmr(:,platl-1,k,n) ) * latwts(base_lat + platl - 1) &
                          + sum( mmr(:,platl,k,n) ) * latwts(base_lat + platl)
               meanmmr  = meanmmr / (real(plon) * sum( latwts(base_lat+platl-1:base_lat+platl) ))
	       do j = platl-1,platl
                  mmr(:,j,k,n) = meanmmr
	       end do
            end do
         end do
      end if

      end subroutine AVERAGE_CONST_POLES

      end module MO_GHOST_TRANSFER
