
#include <params.h>

!-----------------------------------------------------------------------
!++bee, 4 March 1996
! Hack moist convection codes from omega0_10_1 modified for MATCH as follows:
! * use pmgrid.com from MATCH rather than CCM3s pmgrid.h
! * change qpert(plonl,pcnst) to qpert(plonl) since thats all we get from
!   vdiff.  (Its also all that is used in the Zhang scheme.)
! * split q(plonl,plev,pcnst) into q1(plonl,plev) and q(plonl,plev,ncnst)
!   where ncnst is the number of non-water tracers.
! * pass number of passive tracers through the argument list.
! * output eta and beta 
!--bee
!-----------------------------------------------------------------------

      module MO_HACK

      implicit none

      private
      public  :: MFINTI, CMFCMA, CMFADJ
      public  :: limcnv, grav, rgrav, rgas

      save

      integer :: &
           iloc, &     ! longitude location for diagnostics
           jloc, &     ! latitude  location for diagnostics
           nsloc, &    ! nstep for which to produce diagnostics
           limcnv      ! top interface level limit for convection

      real :: &
           cp, &       ! specific heat of dry air
           hlat, &     ! latent heat of vaporization
           grav, &     ! gravitational constant       
           c0, &       ! rain water autoconversion coefficient
           betamn, &   ! minimum overshoot parameter
           rhlat, &    ! reciprocal of hlat
           rcp, &      ! reciprocal of cp
           rgrav, &    ! reciprocal of grav
           cmftau, &   ! characteristic adjustment time scale
           rhoh2o, &   ! density of liquid water (STP)
           rgas, &     ! gas constant for dry air
           dzmin       ! minimum convective depth for precipitation
      real :: &
           tiny, &     ! arbitrary small num used in transport estimates
           eps, &      ! convergence criteria (machine dependent)
           tpmax, &    ! maximum acceptable t perturbation (degrees C)
           shpmax      ! maximum acceptable q perturbation (g/g)           

      logical :: &
           rlxclm      ! logical to relax column versus cloud triplet


      CONTAINS

      subroutine MFINTI( hypi, rair, cpair, gravit, latvap, rhowtr )
!-----------------------------------------------------------------------
! 	... Initialize moist convective mass flux procedure
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      real, intent(in) :: &
           rair, &           ! gas constant for dry air
           cpair, &          ! specific heat of dry air
           gravit, &         ! acceleration due to gravity
           latvap, &         ! latent heat of vaporization
           rhowtr            ! density of liquid water (STP)
      real, intent(in) :: &
           hypi(plevp)       ! ref press at interfaces

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: k

!-----------------------------------------------------------------------
! 	... Initialize physical constants for moist convective mass flux procedure
!-----------------------------------------------------------------------
      cp     = cpair         ! specific heat of dry air     
      hlat   = latvap        ! latent heat of vaporization  
      grav   = gravit        ! gravitational constant       
      rgas   = rair          ! gas constant for dry air
      rhoh2o = rhowtr        ! density of liquid water (STP)

!-----------------------------------------------------------------------
! 	... Initialize free parameters for moist convective mass flux procedure
!-----------------------------------------------------------------------
      c0     = 1.0e-4        ! rain water autoconversion coeff (1/m)
      dzmin  = 0.0           ! minimum cloud depth to precipitate (m)
      betamn = 0.10          ! minimum overshoot parameter
      cmftau = 3600.         ! characteristic adjustment time scale
!-----------------------------------------------------------------------
! 	... Limit convection to below 40 mb
!-----------------------------------------------------------------------
      if( hypi(1) >= 4.e3 ) then
	 limcnv = 1
      else
	 limcnv = plevp
         do k = 1,plev
	    if( hypi(k) < 4.e3 .and. hypi(k+1) >= 4.e3 ) then
	       limcnv = k
	       exit
	    end if
	 end do
      end if
      write(*,*) 'MFINTI: Convection will be capped at interface level ',limcnv
      write(*,*) '        which is ',1.e-2*hypi(limcnv),' mb'

      tpmax  = 1.50          ! maximum acceptable t perturbation (deg C)
      shpmax = 1.50e-3       ! maximum acceptable q perturbation (g/g)
      rlxclm = .true.        ! logical variable to specify that relaxation
                             ! time scale should applied to column as 
                             ! opposed to triplets individually 

!-----------------------------------------------------------------------
! 	... Initialize miscellaneous (frequently used) constants
!-----------------------------------------------------------------------
      rhlat  = 1.0/hlat      ! reciprocal latent heat of vaporization  
      rcp    = 1.0/cp        ! reciprocal specific heat of dry air     
      rgrav  = 1.0/grav      ! reciprocal gravitational constant       

!-----------------------------------------------------------------------
! 	... Initialize diagnostic location information for moist convection scheme
!-----------------------------------------------------------------------
      iloc   = 1             ! longitude point for diagnostic info
      jloc   = 1             ! latitude  point for diagnostic info
      nsloc  = 1             ! nstep value at which to begin diagnostics

!-----------------------------------------------------------------------
! 	... Initialize other miscellaneous parameters
!-----------------------------------------------------------------------
      tiny   = 1.e-36       ! arbitrary small number (scalar transport)
      eps    = 1.e-13       ! convergence criteria (machine dependent) 
      
      end subroutine MFINTI

      subroutine CMFCMA(lat     ,nstep   ,tdt     ,pmid    ,pdel, &
                        rpdel   ,gz      ,tpert   ,qpert   ,phis, &
                        pblht   ,t       ,q1      ,q       ,ncnst, &
                        coneta  ,conbeta ,cmfdt   ,cmfdq, &
                        cmfmc   ,cmfdqr  ,cmfsl   ,cmflq   ,precc, &
                        qc      ,cnt     ,cnb     ,plonl     )
!-----------------------------------------------------------------------
! 	... Moist convective mass flux procedure
!           If stratification is unstable to nonentraining parcel ascent,
!           complete an adjustment making successive use of a simple cloud model
!           consisting of three layers (sometimes referred to as a triplet)
! 
!           Code generalized to allow specification of parcel ("updraft")
!           properties, as well as convective transport of an arbitrary
!           number of passive constituents (see q array).  The code
!           is written so the water vapor field is passed independently
!           in the calling list from the block of other transported
!           constituents, even though as currently designed, it is the
!           first component in the constituents field. 
!-----------------------------------------------------------------------

      use mo_grid,  only : plev, plevp, pcnst
      use ESLOOKUP, only : AQSATD, VQSATD

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
           lat, &                  ! latitude index (S->N)
           plonl, &                ! lon tile dim
           nstep, &                ! current time step index
           ncnst                   ! number of passive tracers
      real, intent(in) :: &
           tdt                     ! 2 delta-t (seconds)
      real, dimension(plonl,plev), intent(in) :: &
           pmid, &                 ! pressure
           pdel, &                 ! delta-p
           rpdel, &                ! 1./pdel
           gz                      ! geopotential
      real, dimension(plonl), intent(in) :: &
           tpert, &                ! PBL perturbation theta
           qpert, &                ! PBL perturbation specific humidity 
           phis, &                 ! surface geopotential
           pblht                   ! PBL height (provided by PBL routine)

      real, intent(inout) :: &
           t(plonl,plev), &        ! temperature (t bar)
           q1(plonl,plev), &       ! specific humidity (sh bar)
           q(plonl,plev,ncnst)     ! passive tracers

       real, intent(out) :: &
        coneta(plonl,plev), &      ! save eta
        conbeta(plonl,plev)        ! save beta 
       real, intent(out) :: &
           cmfdt(plonl,plev), &    ! dt/dt due to moist convection
           cmfdq(plonl,plev), &    ! dq/dt due to moist convection
           cmfmc(plonl,plev ), &   ! moist convection cloud mass flux
           cmfdqr(plonl,plev), &   ! dq/dt due to convective rainout 
           cmfsl(plonl,plev ), &   ! convective lw static energy flux
           cmflq(plonl,plev ), &   ! convective total water flux
           precc(plonl), &         ! convective precipitation rate
           qc(plonl,plev), &       ! dq/dt due to rainout terms
           cnt(plonl), &           ! top level of convective activity   
           cnb(plonl)              ! bottom level of convective activity

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      real, parameter :: ssfac = 1.001                 ! supersaturation bound (detrained air)

      integer :: &
           i,k, &                  ! longitude, level indices
           ii, &                   ! index on "gathered" vectors
           len1, &                 ! vector length of "gathered" vectors
           m, &                    ! constituent index
           ktp, &                  ! temporary index used to track top of convective layer
           km1, kp1
      real :: &
	   gam(plonl,plev), & ! 1/cp (d(qsat)/dT)
           sb(plonl,plev), &  ! dry static energy (s bar)
           hb(plonl,plev), &  ! moist static energy (h bar)
           shbs(plonl,plev), &! sat. specific humidity (sh bar star)
           hbs(plonl,plev), & ! sat. moist static energy (h bar star)
           shbh(plonl,plevp),&! specific humidity on interfaces
           sbh(plonl,plevp),& ! s bar on interfaces
           hbh(plonl,plevp),& ! h bar on interfaces
           cmrh(plonl,plevp),&! interface constituent mixing ratio 
           prec(plonl), &     ! instantaneous total precipitation
           dzcld(plonl), &    ! depth of convective layer (m)
           beta(plonl), &     ! overshoot parameter (fraction)
           betamx(plonl), &   ! local maximum on overshoot
           eta(plonl), &      ! convective mass flux (kg/m^2 s)
           etagdt(plonl), &   ! eta*grav*dt
           cldwtr(plonl), &   ! cloud water (mass)
           rnwtr(plonl)       ! rain water  (mass)
      real :: &
	   sc  (plonl), &     ! dry static energy   ("in-cloud")
           shc (plonl), &     ! specific humidity   ("in-cloud")
           hc  (plonl), &     ! moist static energy ("in-cloud")
           cmrc(plonl)        ! constituent mix rat ("in-cloud")
      real :: &
	   dq1(plonl), &      ! shb  convective change (lower lvl)
           dq2(plonl), &      ! shb  convective change (mid level)
           dq3(plonl), &      ! shb  convective change (upper lvl)
           ds1(plonl), &      ! sb   convective change (lower lvl)
           ds2(plonl), &      ! sb   convective change (mid level)
           ds3(plonl), &      ! sb   convective change (upper lvl)
           dcmr1(plonl), &    ! q convective change (lower lvl)
           dcmr2(plonl), &    ! q convective change (mid level)
           dcmr3(plonl), &    ! q convective change (upper lvl)
           estemp(plonl,plev), &! saturation vapor pressure (scratch)
           vtemp1(2*plonl), & ! intermediate scratch vector
           vtemp2(2*plonl), & ! intermediate scratch vector
           vtemp3(2*plonl), & ! intermediate scratch vector
           vtemp4(2*plonl)    ! intermediate scratch vector
      integer :: &
           indx1(plonl)       ! longitude indices for condition true
      logical :: &
           etagt0             ! true if eta > 0.0
      real :: &
	   sh1, &             ! dummy arg in qhalf statement func.
           sh2, &             ! dummy arg in qhalf statement func.
           shbs1, &           ! dummy arg in qhalf statement func.
           shbs2, &           ! dummy arg in qhalf statement func.
           cats, &            ! modified characteristic adj. time
           rtdt, &            ! 1./tdt
           qprime, &          ! modified specific humidity pert.
           tprime, &          ! modified thermal perturbation
           pblhgt, &          ! bounded pbl height (max[pblh,1m])
           fac1, &            ! intermediate scratch variable
           shprme, &          ! intermediate specific humidity pert.
           qsattp, &          ! sat mix rat for thermally pert PBL parcels 
           dz, &              ! local layer depth
           temp1, &           ! intermediate scratch variable
           b1, &              ! bouyancy measure in detrainment lvl
           b2, &              ! bouyancy measure in condensation lvl
           temp2, &           ! intermediate scratch variable
           temp3              ! intermediate scratch variable
      real :: &
	   g, &               ! bounded vertical gradient of hb
           tmass, &           ! total mass available for convective exch
           denom, &           ! intermediate scratch variable
           qtest1, &          ! used in negative q test (middle lvl) 
           qtest2, &          ! used in negative q test (lower lvl) 
           fslkp, &           ! flux lw static energy (bot interface)
           fslkm, &           ! flux lw static energy (top interface)
           fqlkp, &           ! flux total water (bottom interface)
           fqlkm, &           ! flux total water (top interface)
           botflx, &          ! bottom constituent mixing ratio flux
           topflx, &          ! top constituent mixing ratio flux
           efac1, &           ! ratio q to convectively induced chg (btm lvl)
           efac2, &           ! ratio q to convectively induced chg (mid lvl)
           efac3              ! ratio q to convectively induced chg (top lvl)
      real :: &
	   tb(plonl,plev), &  ! working storage for temp (t bar)
           shb(plonl,plev),&  ! working storage for spec hum (sh bar)
           adjfac             ! adjustment factor (relaxation related)

!-----------------------------------------------------------------------
!	... Function declarations
!-----------------------------------------------------------------------
      real :: QHALF
      QHALF(sh1,sh2,shbs1,shbs2) = min( max( sh1,sh2 ),(shbs2*sh1 + shbs1*sh2)/(shbs1+shbs2) )

!-----------------------------------------------------------------------
! 	... Ensure that characteristic adjustment time scale (cmftau) assumed
!           in estimate of eta isnt smaller than model time scale (tdt)
!           The time over which the convection is assumed to act (the adjustment
!           time scale) can be applied with each application of the three-level
!           cloud model, or applied to the column tendencies after a "hard"
!           adjustment (i.e., on a 2-delta t time scale) is evaluated
!-----------------------------------------------------------------------
      if( rlxclm ) then
         cats   = tdt                     ! relaxation applied to column
         adjfac = tdt/(max( tdt,cmftau) )
      else
         cats   = max( tdt,cmftau )       ! relaxation applied to triplet
         adjfac = 1.
      endif
      rtdt  = 1.0/tdt
      do k = 1,plev
         coneta(:,k)  = 0.
         conbeta(:,k) = 0.
         cmfdt(:,k)   = 0.
         cmfdq(:,k)   = 0.
         cmfmc(:,k)   = 0.
         cmfdqr(:,k)  = 0.
         cmfsl(:,k)   = 0.
         cmflq(:,k)   = 0.
      end do

!-----------------------------------------------------------------------
! 	... Move temperature and moisture into working storage
!-----------------------------------------------------------------------
      do k = limcnv,plev
         tb(:,k)  = t(:,k)
         shb(:,k) = q1(:,k)
      end do

!-----------------------------------------------------------------------
! 	... Compute sb,hb,shbs,hbs
!-----------------------------------------------------------------------
      call AQSATD( tb      ,pmid    ,estemp ,shbs    ,gam , &
                   plonl   ,plonl    ,plev   ,limcnv ,plev )

      do k = limcnv,plev
         sb (:,k) = cp*tb(:,k) + gz(:,k)
         hb (:,k) = sb(:,k) + hlat*shb(:,k)
         hbs(:,k) = sb(:,k) + hlat*shbs(:,k)
      end do

!-----------------------------------------------------------------------
! 	... Compute sbh, shbh
!-----------------------------------------------------------------------
      do k = limcnv+1,plev
        km1 = k - 1
        do i = 1,plonl
          sbh (i,k) = .5*(sb(i,km1) + sb(i,k))
          shbh(i,k) = QHALF( shb(i,km1), shb(i,k), shbs(i,km1), shbs(i,k) )
          hbh (i,k) = sbh(i,k) + hlat*shbh(i,k)
        end do
      end do

!-----------------------------------------------------------------------
! 	... Specify properties at top of model (not used, but filling anyway)
!-----------------------------------------------------------------------
      sbh (:,limcnv) = sb(:,limcnv)
      shbh(:,limcnv) = shb(:,limcnv)
      hbh (:,limcnv) = hb(:,limcnv)

!-----------------------------------------------------------------------
! 	... Zero vertically independent control, tendency & diagnostic arrays
!-----------------------------------------------------------------------
      prec(:)  = 0.
      dzcld(:) = 0.
      cnb(:)   = 0.
      cnt(:)   = real(plevp)

!-----------------------------------------------------------------------
! 	... Begin moist convective mass flux adjustment procedure.
!           Formalism ensures that negative cloud liquid water can never occur
!-----------------------------------------------------------------------
level_loop : &
      do k = plev-1,limcnv+1,-1
        km1 = k - 1
        kp1 = k + 1
        do i = 1,plonl
          etagdt(i) = 0.
          eta(i)    = 0.
          beta(i)   = 0.
          ds1(i)    = 0.
          ds2(i)    = 0.
          ds3(i)    = 0.
          dq1(i)    = 0.
          dq2(i)    = 0.
          dq3(i)    = 0.
!-----------------------------------------------------------------------
! 	... Specification of "cloud base" conditions
!-----------------------------------------------------------------------
          qprime    = 0.
          tprime    = 0.
!-----------------------------------------------------------------------
! 	... Assign tprime within the PBL to be proportional to the quantity
!           tpert (which will be bounded by tpmax), passed to this routine by 
!           the PBL routine.  Dont allow perturbation to produce a dry 
!           adiabatically unstable parcel.  Assign qprime within the PBL to be 
!           an appropriately modified value of the quantity qpert (which will be 
!           bounded by shpmax) passed to this routine by the PBL routine.  The 
!           quantity qprime should be less than the local saturation value 
!           (qsattp=qsat[t+tprime,p]).  In both cases, tpert and qpert are
!           linearly reduced toward zero as the PBL top is approached.
!-----------------------------------------------------------------------
          pblhgt = max( pblht(i),1. )
          if( (gz(i,kp1) - phis(i))*rgrav <= pblhgt .and. dzcld(i) == 0. ) then
            fac1   = max( .0,1. - (gz(i,kp1) - phis(i))*rgrav/pblhgt )
            tprime = min( tpert(i),tpmax )*fac1
            qsattp = shbs(i,kp1) + cp*rhlat*gam(i,kp1)*tprime
            shprme = min( min( qpert(i),shpmax )*fac1,max( qsattp - shb(i,kp1),0. ) )
            qprime = max( qprime,shprme )
          else
            tprime = 0.
            qprime = 0.
          end if
!-----------------------------------------------------------------------
! 	... Specify "updraft" (in-cloud) thermodynamic properties
!-----------------------------------------------------------------------
          sc (i)    = sb (i,kp1) + cp*tprime
          shc(i)    = shb(i,kp1) + qprime
          hc (i)    = sc (i) + hlat*shc(i)
          vtemp4(i) = hc(i) - hbs(i,k)
          dz        = pdel(i,k)*rgas*tb(i,k)*rgrav/pmid(i,k)
          if( vtemp4(i) > 0. ) then
            dzcld(i) = dzcld(i) + dz
          else
            dzcld(i) = 0.
          end if
        end do

!-----------------------------------------------------------------------
! 	... Check on moist convective instability
!           Build index vector of points where instability exists
!-----------------------------------------------------------------------
	len1 = count( vtemp4(:plonl) > 0. )
        if( len1 <= 0 ) then
	   cycle
	else
	   ii = 0
	   do i = 1,plonl
	      if( vtemp4(i) > 0. ) then
		 ii = ii + 1
		 indx1(ii) = i
	      end if
	   end do
	end if
!-----------------------------------------------------------------------
! 	... Current level just below top level => no overshoot
!-----------------------------------------------------------------------
        if( k <= limcnv+1 ) then
          do ii = 1,len1
            i = indx1(ii)
            temp1     = vtemp4(i)/(1.0 + gam(i,k))
            cldwtr(i) = max( .0,(sb(i,k) - sc(i) + temp1) )
            beta(i)   = 0.
            vtemp3(i) = (1. + gam(i,k))*(sc(i) - sbh(i,k))
          end do
        else
!-----------------------------------------------------------------------
! 	... First guess at overshoot parameter using crude buoyancy closure
!           10% overshoot assumed as a minimum and 1-c0*dz maximum to start
!           If pre-existing supersaturation in detrainment layer, beta=0
!           cldwtr is temporarily equal to hlat*l (l=> liquid water)
!-----------------------------------------------------------------------
#ifdef CRAY
!DIR  IVDEP
#endif
          do ii = 1,len1
            i = indx1(ii)
            temp1     = vtemp4(i)/(1.0 + gam(i,k))
            cldwtr(i) = max( .0,(sb(i,k) - sc(i) + temp1) )
            betamx(i) = 1. - c0*max( .0,(dzcld(i) - dzmin) )
            b1        = (hc(i) - hbs(i,km1))*pdel(i,km1)
            b2        = (hc(i) - hbs(i,k  ))*pdel(i,k  )
            beta(i)   = max( betamn,min( betamx(i),1.  + b1/b2 ) )
            if( hbs(i,km1) <= hb(i,km1)) then
	       beta(i) = 0.
	    end if
!-----------------------------------------------------------------------
! 	... Bound maximum beta to ensure physically realistic solutions
!           First check constrains beta so that eta remains positive
!           (assuming that eta is already positive for beta equal zero)
!-----------------------------------------------------------------------
            vtemp1(i) = -(hbh(i,kp1) - hc(i))*pdel(i,k)*rpdel(i,kp1) &
                        + (1. + gam(i,k))*(sc(i) - sbh(i,kp1) + cldwtr(i))
            vtemp2(i) = (1. + gam(i,k))*(sc(i) - sbh(i,k))
            vtemp3(i) = vtemp2(i)
            if ((beta(i)*vtemp2(i) - vtemp1(i)) > 0. ) then
              betamx(i) = .99*(vtemp1(i)/vtemp2(i))
              beta(i)   = max( .0,min( betamx(i),beta(i) ) )
            end if
          end do

!-----------------------------------------------------------------------
! 	... Second check involves supersaturation of "detrainment layer"
!           small amount of supersaturation acceptable (by ssfac factor)
!-----------------------------------------------------------------------
#ifdef CRAY
!DIR  IVDEP
#endif
          do ii = 1,len1
            i = indx1(ii)
            if( hb(i,km1) < hbs(i,km1) ) then
              vtemp1(i) = vtemp1(i)*rpdel(i,k)
              temp2 = gam(i,km1)*(sbh(i,k) - sc(i) + cldwtr(i)) - hbh(i,k) + hc(i) - sc(i) + sbh(i,k)
              temp3 = vtemp3(i)*rpdel(i,k)
              vtemp2(i) = (tdt/cats)*(hc(i) - hbs(i,k))*temp2/(pdel(i,km1)*(hbs(i,km1) - hb(i,km1))) + temp3
              if( (beta(i)*vtemp2(i) - vtemp1(i)) > 0. ) then
                betamx(i) = ssfac*(vtemp1(i)/vtemp2(i))
                beta(i)   = max( .0,min( betamx(i),beta(i) ) )
              end if
            else 
              beta(i) = 0.
            end if
          end do

!-----------------------------------------------------------------------
! 	... Third check to avoid introducing 2 delta x thermodynamic
!           noise in the vertical ... constrain adjusted h (or theta e)
!           so that the adjustment doesnt contribute to "kinks" in h
!-----------------------------------------------------------------------
#ifdef CRAY
!DIR  IVDEP
#endif
          do ii = 1,len1
            i = indx1(ii)
            g = min( 0.,hb(i,k) - hb(i,km1) )
            temp1 = (hb(i,k) - hb(i,km1) - g)*(cats/tdt)/(hc(i) - hbs(i,k))
            vtemp1(i) = temp1*vtemp1(i) + (hc(i) - hbh(i,kp1))*rpdel(i,k)
            vtemp2(i) = temp1*vtemp3(i)*rpdel(i,k) &
                        + (hc(i) - hbh(i,k) - cldwtr(i)) * (rpdel(i,k) + rpdel(i,kp1))
            if( (beta(i)*vtemp2(i) - vtemp1(i)) > 0. ) then
              if( vtemp2(i) /= 0. ) then
                betamx(i) = vtemp1(i)/vtemp2(i)
              else
                betamx(i) = 0.
              end if
              beta(i) = max( .0,min( betamx(i),beta(i) ) )
            end if
          end do
        end if

!-----------------------------------------------------------------------
! 	... Calculate mass flux required for stabilization.
!           Ensure that the convective mass flux, eta, is positive by
!           setting negative values of eta to zero..
!           Ensure that estimated mass flux cannot move more than the
!           minimum of total mass contained in either layer k or layer k+1.
!           Also test for other pathological cases that result in non-
!           physical states and adjust eta accordingly.
!-----------------------------------------------------------------------
#ifdef CRAY
!DIR  IVDEP
#endif
        do ii = 1,len1
          i = indx1(ii)
          beta(i) = max( .0,beta(i) )
          temp1 = hc(i) - hbs(i,k)
          temp2 = ((1. + gam(i,k))*(sc(i) - sbh(i,kp1) + cldwtr(i)) - beta(i)*vtemp3(i))*rpdel(i,k) &
                  - (hbh(i,kp1) - hc(i))*rpdel(i,kp1)
          eta(i) = temp1/(temp2*grav*cats)
          tmass = min( pdel(i,k),pdel(i,kp1) )*rgrav
          if( eta(i) > tmass*rtdt .or. eta(i) <= 0. ) then
	     eta(i) = 0.
	  end if

!-----------------------------------------------------------------------
! 	... Check on negative q in top layer (bound beta)
!-----------------------------------------------------------------------
          if( shc(i) - shbh(i,k) < 0. .and. beta(i)*eta(i) /= 0. ) then
            denom = eta(i)*grav*tdt*(shc(i) - shbh(i,k))*rpdel(i,km1)
            beta(i) = max( 0.,min( -.999*shb(i,km1)/denom,beta(i) ) )
          end if
!-----------------------------------------------------------------------
! 	... Check on negative q in middle layer (zero eta)
!-----------------------------------------------------------------------
          qtest1 = shb(i,k) + eta(i)*grav*tdt*((shc(i) - shbh(i,kp1)) &
                     - (1. - beta(i))*cldwtr(i)*rhlat &
                     - beta(i)*(shc(i) - shbh(i,k)))*rpdel(i,k)
          if( qtest1 <= 0. ) then
	     eta(i) = 0.
	  end if
!-----------------------------------------------------------------------
! 	... Check on negative q in lower layer (bound eta)
!-----------------------------------------------------------------------
          fac1 = -(shbh(i,kp1) - shc(i))*rpdel(i,kp1)
          qtest2 = shb(i,kp1) - eta(i)*grav*tdt*fac1
          if( qtest2  < 0. ) then
            eta(i) = .99*shb(i,kp1)/(grav*tdt*fac1)
          end if
          etagdt(i) = eta(i)*grav*tdt
          coneta(i,k)  = eta(i)
          conbeta(i,k) = beta(i)
        end do

!-----------------------------------------------------------------------
! 	... Calculate cloud water, rain water, and thermodynamic changes
!-----------------------------------------------------------------------
#ifdef CRAY
!DIR  IVDEP
#endif
        do ii = 1,len1
          i = indx1(ii)
          cldwtr(i) = etagdt(i)*cldwtr(i)*rhlat*rgrav
          rnwtr(i) = (1. - beta(i))*cldwtr(i)
          ds1(i) = etagdt(i)*(sbh(i,kp1) - sc(i))*rpdel(i,kp1)
          dq1(i) = etagdt(i)*(shbh(i,kp1) - shc(i))*rpdel(i,kp1)
          ds2(i) = (etagdt(i)*(sc(i) - sbh(i,kp1)) &
                    + hlat*grav*cldwtr(i) - beta(i)*etagdt(i)*(sc(i) - sbh(i,k)))*rpdel(i,k)
          dq2(i) = (etagdt(i)*(shc(i) - shbh(i,kp1)) &
                    - grav*rnwtr(i) - beta(i)*etagdt(i)*(shc(i) - shbh(i,k)))*rpdel(i,k)
          ds3(i) = beta(i)*(etagdt(i)*(sc(i) - sbh(i,k)) - hlat*grav*cldwtr(i))*rpdel(i,km1)
          dq3(i) = beta(i)*etagdt(i)*(shc(i) - shbh(i,k))*rpdel(i,km1)
!-----------------------------------------------------------------------
! 	... Isolate convective fluxes for later diagnostics
!-----------------------------------------------------------------------
          fslkp = eta(i)*(sc(i) - sbh(i,kp1))
          fslkm = beta(i)*(eta(i)*(sc(i) - sbh(i,k)) - hlat*cldwtr(i)*rtdt)
          fqlkp = eta(i)*(shc(i) - shbh(i,kp1))
          fqlkm = beta(i)*eta(i)*(shc(i) - shbh(i,k))
!-----------------------------------------------------------------------
! 	... Update thermodynamic profile (update sb, hb, & hbs later)
!-----------------------------------------------------------------------
          tb (i,kp1) = tb(i,kp1)  + ds1(i)*rcp
          tb (i,k  ) = tb(i,k)    + ds2(i)*rcp
          tb (i,km1) = tb(i,km1)  + ds3(i)*rcp
          shb(i,kp1) = shb(i,kp1) + dq1(i)
          shb(i,k  ) = shb(i,k)   + dq2(i)
          shb(i,km1) = shb(i,km1) + dq3(i)
!-----------------------------------------------------------------------
! 	... Update diagnostic information for final budget
!           Tracking precipitation, temperature & specific humidity tendencies,
!           rainout term, convective mass flux, convective liquid
!           water static energy flux, and convective total water flux
!           The variable afac makes the necessary adjustment to the
!           diagnostic fluxes to account for adjustment time scale based on
!           how relaxation time scale is to be applied (column vs. triplet)
!-----------------------------------------------------------------------
          prec(i)    = prec(i) + (rnwtr(i)/rhoh2o)*adjfac
!-----------------------------------------------------------------------
! 	... The following variables have units of "units"/second
!-----------------------------------------------------------------------
          cmfdt (i,kp1) = cmfdt (i,kp1) + ds1(i)*rcp*rtdt*adjfac
          cmfdt (i,k  ) = cmfdt (i,k  ) + ds2(i)*rcp*rtdt*adjfac
          cmfdt (i,km1) = cmfdt (i,km1) + ds3(i)*rcp*rtdt*adjfac
          cmfdq (i,kp1) = cmfdq (i,kp1) + dq1(i)*rtdt*adjfac
          cmfdq (i,k  ) = cmfdq (i,k  ) + dq2(i)*rtdt*adjfac
          cmfdq (i,km1) = cmfdq (i,km1) + dq3(i)*rtdt*adjfac
          qc    (i,k  ) = (grav*rnwtr(i)*rpdel(i,k))*rtdt*adjfac
          cmfdqr(i,k  ) = cmfdqr(i,k  ) + qc(i,k)
          cmfmc (i,kp1) = cmfmc (i,kp1) + eta(i)*adjfac
          cmfmc (i,k  ) = cmfmc (i,k  ) + beta(i)*eta(i)*adjfac
!-----------------------------------------------------------------------
! 	... The following variables have units of w/m**2
!-----------------------------------------------------------------------
          cmfsl (i,kp1) = cmfsl (i,kp1) + fslkp*adjfac
          cmfsl (i,k  ) = cmfsl (i,k  ) + fslkm*adjfac
          cmflq (i,kp1) = cmflq (i,kp1) + hlat*fqlkp*adjfac
          cmflq (i,k  ) = cmflq (i,k  ) + hlat*fqlkm*adjfac
        end do

!-----------------------------------------------------------------------
! 	... Next, convectively modify passive constituents
!           For now, when applying relaxation time scale to thermal fields after 
!           entire column has undergone convective overturning, constituents will 
!           be mixed using a "relaxed" value of the mass flux determined above
!           Although this will be inconsistant with the treatment of the thermal
!           fields, its computationally much cheaper, no more-or-less justifiable,
!           and consistent with how the history tape mass fluxes would be used in
!           an off-line mode (i.e., using an off-line transport model)
!-----------------------------------------------------------------------
species_loop : &
        do m = 1,ncnst
#ifdef CRAY
!DIR  IVDEP
#endif
          do ii = 1,len1
            i = indx1(ii)
!-----------------------------------------------------------------------
! 	... If any of the reported values of the constituent is negative in
!           the three adjacent levels, nothing will be done to the profile
!-----------------------------------------------------------------------
            if( q(i,kp1,m) < 0. .or. q(i,k,m) < 0. .or. q(i,km1,m) < 0. ) then
	       cycle
	    end if
!-----------------------------------------------------------------------
! 	... Specify constituent interface values (linear interpolation)
!-----------------------------------------------------------------------
            cmrh(i,k  ) = .5*(q(i,km1,m) + q(i,k  ,m))
            cmrh(i,kp1) = .5*(q(i,k  ,m) + q(i,kp1,m))
            cmrc(i) = q(i,kp1,m)
!-----------------------------------------------------------------------
! 	... Determine fluxes, flux divergence => changes due to convection
!           Logic must be included to avoid producing negative values. A bit
!           messy since there are no a priori assumptions about profiles.
!           Tendency is modified (reduced) when pending disaster detected.
!-----------------------------------------------------------------------
            botflx   = etagdt(i)*(cmrc(i) - cmrh(i,kp1))*adjfac
            topflx   = beta(i)*etagdt(i)*(cmrc(i)-cmrh(i,k))*adjfac
            dcmr1(i) = -botflx*rpdel(i,kp1)
            efac1    = 1.
            efac2    = 1.
            efac3    = 1.
            if( q(i,kp1,m)+dcmr1(i)  < 0. ) then
              efac1 = max( tiny,ABS(q(i,kp1,m)/dcmr1(i)) - eps )
            end if
            if( efac1 == tiny .or. efac1 > 1. ) then
	      efac1 = 0.
	    end if
            dcmr1(i) = -efac1*botflx*rpdel(i,kp1)
            dcmr2(i) = (efac1*botflx - topflx)*rpdel(i,k)
            if( q(i,k,m)+dcmr2(i) <  0. ) then
              efac2 = max( tiny,ABS( q(i,k,m)/dcmr2(i) ) - eps )
            end if
            if( efac2 == tiny .or. efac2 > 1. ) then
	      efac2 = 0.
	    end if
            dcmr2(i) = (efac1*botflx - efac2*topflx)*rpdel(i,k)
            dcmr3(i) = efac2*topflx*rpdel(i,km1)
            if( q(i,km1,m)+dcmr3(i)  < 0. ) then
              efac3 = max( tiny,ABS( q(i,km1,m)/dcmr3(i) ) - eps )
            end if
            if( efac3 == tiny .or. efac3 > 1. ) then
	      efac3 = 0.
	    end if
            efac3    = min( efac2,efac3 )
            dcmr2(i) = (efac1*botflx - efac3*topflx)*rpdel(i,k)
            dcmr3(i) = efac3*topflx*rpdel(i,km1)
            q(i,kp1,m) = q(i,kp1,m) + dcmr1(i)
            q(i,k  ,m) = q(i,k  ,m) + dcmr2(i)
            q(i,km1,m) = q(i,km1,m) + dcmr3(i)
          end do
        end do species_loop

!-----------------------------------------------------------------------
! 	... Constituent modifications complete
!-----------------------------------------------------------------------
        if( k /= limcnv+1 ) then
!-----------------------------------------------------------------------
! 	... Complete update of thermodynamic structure at integer levels
!           gather/scatter points that need new values of shbs and gamma
!-----------------------------------------------------------------------
           do ii = 1,len1
              i = indx1(ii)
              vtemp1(ii     ) = tb(i,k)
              vtemp1(ii+len1) = tb(i,km1)
              vtemp2(ii     ) = pmid(i,k)
              vtemp2(ii+len1) = pmid(i,km1)
            end do
            call VQSATD( vtemp1, vtemp2, estemp, vtemp3, vtemp4, 2*len1 )    ! using estemp as extra long vector
#ifdef CRAY
!DIR  IVDEP
#endif
            do ii = 1,len1
              i = indx1(ii)
              shbs(i,k  ) = vtemp3(ii     )
              shbs(i,km1) = vtemp3(ii+len1)
              gam(i,k  ) = vtemp4(ii     )
              gam(i,km1) = vtemp4(ii+len1)
              sb (i,k  ) = sb(i,k  ) + ds2(i)
              sb (i,km1) = sb(i,km1) + ds3(i)
              hb (i,k  ) = sb(i,k  ) + hlat*shb(i,k  )
              hb (i,km1) = sb(i,km1) + hlat*shb(i,km1)
              hbs(i,k  ) = sb(i,k  ) + hlat*shbs(i,k  )
              hbs(i,km1) = sb(i,km1) + hlat*shbs(i,km1)
            end do

!-----------------------------------------------------------------------
! 	... Update thermodynamic information at half (i.e., interface) levels
!-----------------------------------------------------------------------
#ifdef CRAY
!DIR  IVDEP
#endif
            do ii = 1,len1
              i = indx1(ii)
              sbh (i,k) = .5*(sb(i,k) + sb(i,km1))
              shbh(i,k) = QHALF( shb(i,km1), shb(i,k), shbs(i,km1), shbs(i,k) )
              hbh (i,k) = sbh(i,k) + hlat*shbh(i,k)
              sbh (i,km1) = 0.5*(sb(i,km1) + sb(i,k-2))
              shbh(i,km1) = QHALF( shb(i,k-2), shb(i,km1), shbs(i,k-2), shbs(i,km1) )
              hbh (i,km1) = sbh(i,km1) + hlat*shbh(i,km1)
            end do
        end if

!-----------------------------------------------------------------------
! 	... Ensure that dzcld is reset if convective mass flux zero
!           specify the current vertical extent of the convective activity
!           top of convective layer determined by size of overshoot param.
!-----------------------------------------------------------------------
        do i = 1,plonl
          etagt0 = eta(i) > 0.
          if( .not. etagt0 ) then
	     dzcld(i) = 0.
	  end if
          if( etagt0 .and. beta(i) > betamn ) then
            ktp = k-1
          else
            ktp = k
          end if
          if( etagt0 ) then
            cnt(i) = min( cnt(i),real(ktp) )
            cnb(i) = max( cnb(i),real(k) )
          end if
        end do
      end do level_loop

!-----------------------------------------------------------------------
! 	... Apply final thermodynamic tendencies
!-----------------------------------------------------------------------
      do k = limcnv,plev
         t (:,k) = t (:,k) + cmfdt(:,k)*tdt
         q1(:,k) = q1(:,k) + cmfdq(:,k)*tdt
      end do

!-----------------------------------------------------------------------
! 	... Kludge to prevent cnb-cnt from being zero (in the event
!           someone decides that they want to divide by this quantity)
!-----------------------------------------------------------------------
      do i = 1,plonl
        if( cnb(i) /= 0. .and. cnb(i) == cnt(i) ) then
          cnt(i) = cnt(i) - 1.
        end if
      end do

      precc(:) = prec(:)*rtdt

      end subroutine CMFCMA

      subroutine CMFADJ( tdt, rpdel, eta, beta, q, plonl )
!-----------------------------------------------------------------------
! 	... Compute the convective mass flux adjustment to all tracers
!           using the convective mass fluxes and overshoot parameters
!           for the Hack scheme.
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl
      real, intent(in) :: &
        tdt, &                  ! 2 delta-t (seconds)
        rpdel(plonl,plev), &    ! 1./pdel
        eta(plonl,plev), &      ! convective mass flux (kg/m^2 s)
        beta(plonl,plev)        ! overshoot parameter (fraction)

      real, intent(inout) :: &
        q(plonl,plev,pcnst)     ! passive tracers

!-----------------------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------------------
      integer :: &
           i,k, &                  ! longitude, level indices
           ii, &                   ! index on "gathered" vectors
           len1, &                 ! vector length of "gathered" vectors
           m                       ! constituent index
      integer :: &
        indx1(plonl)       ! longitude indices for condition true
      real :: &
        adjfac, &          ! adjustment factor (relaxation related)
        etagdt(plonl), &   ! eta*grav*dt
        cmrh(plonl,plevp), & ! interface constituent mixing ratio 
        cmrc(plonl), &     ! constituent mix rat ("in-cloud")
        dcmr1(plonl), &    ! q convective change (lower lvl)
        dcmr2(plonl), &    ! q convective change (mid level)
        dcmr3(plonl), &    ! q convective change (upper lvl)
        botflx, &          ! bottom constituent mixing ratio flux
        topflx, &          ! top constituent mixing ratio flux
        efac1, &           ! ratio q to convectively induced chg (btm lvl)
        efac2, &           ! ratio q to convectively induced chg (mid lvl)
        efac3              ! ratio q to convectively induced chg (top lvl)

!-----------------------------------------------------------------------
! 	... Ensure that characteristic adjustment time scale (cmftau) assumed
!           in estimate of eta isnt smaller than model time scale (tdt)
!           The time over which the convection is assumed to act (the adjustment
!           time scale) can be applied with each application of the three-level
!           cloud model, or applied to the column tendencies after a "hard"
!           adjustment (i.e., on a 2-delta t time scale) is evaluated
!-----------------------------------------------------------------------
      if( rlxclm ) then
         adjfac = tdt/max( tdt,cmftau )
      else
         adjfac = 1.
      end if

!-----------------------------------------------------------------------
! 	... Begin moist convective mass flux adjustment procedure.
!           Formalism ensures that negative cloud liquid water can never occur
!-----------------------------------------------------------------------
      do k = plev-1,limcnv+1,-1
        len1 = 0
        do i = 1,plonl
           if( eta(i,k) /= 0. ) then
              etagdt(i) = eta(i,k)*grav*tdt
              len1 = len1 + 1
              indx1(len1) = i
           else
              etagdt(i) = 0.
           end if
        end do
        if( len1 <= 0 ) then
	   cycle
	end if

!-----------------------------------------------------------------------
! 	... Next, convectively modify passive constituents
!           For now, when applying relaxation time scale to thermal fields after 
!           entire column has undergone convective overturning, constituents will 
!           be mixed using a "relaxed" value of the mass flux determined above
!           Although this will be inconsistant with the treatment of the thermal
!           fields, its computationally much cheaper, no more-or-less justifiable,
!           and consistent with how the history tape mass fluxes would be used in
!           an off-line mode (i.e., using an off-line transport model)
!-----------------------------------------------------------------------
        do m = 1,pcnst
#ifdef CRAY
!DIR  IVDEP
#endif
          do ii = 1,len1
            i = indx1(ii)
!-----------------------------------------------------------------------
! 	... If any of the reported values of the constituent is negative in
!           the three adjacent levels, nothing will be done to the profile
!-----------------------------------------------------------------------
            if( q(i,k+1,m) < 0. .or. q(i,k,m) < 0. .or. q(i,k-1,m) < 0. ) then
	       cycle
	    end if
!-----------------------------------------------------------------------
! 	... Specify constituent interface values (linear interpolation)
!-----------------------------------------------------------------------
            cmrh(i,k  ) = .5*(q(i,k-1,m) + q(i,k  ,m))
            cmrh(i,k+1) = .5*(q(i,k  ,m) + q(i,k+1,m))
            cmrc(i) = q(i,k+1,m)
!-----------------------------------------------------------------------
! 	... Determine fluxes, flux divergence => changes due to convection
!           Logic must be included to avoid producing negative values. A bit
!           messy since there are no a priori assumptions about profiles.
!           Tendency is modified (reduced) when pending disaster detected.
!-----------------------------------------------------------------------
            botflx   = etagdt(i)*(cmrc(i) - cmrh(i,k+1))*adjfac
            topflx   = beta(i,k)*etagdt(i)*(cmrc(i)-cmrh(i,k))*adjfac
            dcmr1(i) = -botflx*rpdel(i,k+1)
            efac1    = 1.
            efac2    = 1.
            efac3    = 1.
            if( q(i,k+1,m)+dcmr1(i)  < 0. ) then
              efac1 = max( tiny,ABS( q(i,k+1,m)/dcmr1(i) ) - eps )
            end if
            if( efac1 == tiny .or. efac1 > 1. ) then
	      efac1 = 0.
	    end if
            dcmr1(i) = -efac1*botflx*rpdel(i,k+1)
            dcmr2(i) = (efac1*botflx - topflx)*rpdel(i,k)
            if( q(i,k,m)+dcmr2(i)  < 0. ) then
              efac2 = max( tiny,ABS( q(i,k,m)/dcmr2(i) ) - eps )
            end if
            if( efac2 == tiny .or. efac2 > 1. ) then
	      efac2 = 0.
	    end if
            dcmr2(i) = (efac1*botflx - efac2*topflx)*rpdel(i,k)
            dcmr3(i) = efac2*topflx*rpdel(i,k-1)
            if( q(i,k-1,m)+dcmr3(i)  < 0. ) then
              efac3 = max( tiny,ABS( q(i,k-1,m)/dcmr3(i) ) - eps )
            end if
            if( efac3 == tiny .or. efac3 > 1. ) then
	      efac3 = 0.
	    end if
            efac3    = min( efac2,efac3 )
            dcmr2(i) = (efac1*botflx - efac3*topflx)*rpdel(i,k)
            dcmr3(i) = efac3*topflx*rpdel(i,k-1)
            q(i,k+1,m) = q(i,k+1,m) + dcmr1(i)
            q(i,k  ,m) = q(i,k  ,m) + dcmr2(i)
            q(i,k-1,m) = q(i,k-1,m) + dcmr3(i)
          end do
        end do
      end do

      end subroutine CMFADJ

      end module MO_HACK
