
#include <params.h>

      module mo_inirun

      implicit none

      private :: hst_files

      contains

      subroutine inirun( xdtime, ncdate, ncsec, nadv, &
                         u, v, w, ps, t, q, &
                         cgs, kvh, zmu, zmd, zeu, &
                         hketa, hkbeta, phis, oro, as, &
                         ts, taux, tauy, hflx, qflx, &
                         qrad, omga, cldtop, cldbot, zm, &
                         zint, cmfdqr, cldfr, concld, nrain, &
                         nevapr, cwat, pdmfuen, pdmfude, pdmfden, &
                         pdmfdde, snow, fsds, soilw, precip, &
                         ts_avg, fsds_avg, plonl, platl, pplon, nodes )
!---------------------------------------------------------------------
! 	... initialize common blocks, dynamics and trace specie arrays.
!---------------------------------------------------------------------

      use m_types,         only : filespec, hst_list, hst_usr_lst, hstdim, time_ramp
      use m_tracname,      only : tracnam
      use mo_mpi
      use mo_grid,         only : plon, plat, plev, plevp, pcnst
      use mo_vgrid,        only : hycoef
      use mo_constants,    only : pi, rearth, phi, latwts, lam, coslam, sinlam, &
                                  mo_constants_inti
      use mo_control
      use mo_dyninp,       only : chkdyn, inidyn, interp1, interp2
      use mo_dyninp,       only : mxdynflds, dyn_flnm_prefix, dyn_flnm_suffix, dyn_flnm_date_frmt
      use plevs,           only : iniplevs
      use mo_surface,      only : inisflx
      use mo_restart,      only : inirest, rdrst
      use mo_adv,          only : ffsl_inti
      use eslookup,        only : esinti
      use mo_tracinp,      only : initrac
      use mo_vdiff,        only : vdinti
      use mo_cloud,        only : inimc, inimland, inicld
      use mo_histout,      only : inihist, addfld, hst_file_max, sim_file_cnt, match_file_cnt, hfile
      use mo_hack,         only : mfinti
      use mo_virtem,       only : inivtem
      use mo_z0,           only : iniz
      use mo_qneg,         only : iniqneg
      use mo_drydep,       only : dvel_inti_table, dvel_inti_xactive
      use mo_xemis,        only : xemis_inti
      use mo_setsox,       only : sox_inti
      use mo_setsoa,       only : soa_inti
      use mo_seasalt,      only : seasalt_inti
      use mo_aerosols,     only : aerosols_inti
      use mo_pottemp,      only : inipottem
      use mo_calendar,     only : inicalendar, caldayr, addsec2dat
      use mo_hook,         only : moz_hook_inti
      use mo_lghtning,     only : lght_inti
      use mo_mass,         only : inimass
      use mo_drymass,      only : inidry, adjdry, mdryatm
      use mo_charutl,      only : upcase, glc
      use mo_init_species, only : initr0
      use mo_file_utils,   only : navu, arch_file, make_filename
      use mo_sim_dat,      only : set_sim_dat
      use mo_read_sim_chm, only : read_sim_chm
      use mo_chemini,      only : chemini
      use mo_chemdr,       only : chemdr_inti
      use chem_mods,       only : chem_mods_inti, srfems_cnt, drydep_cnt, xactive_srf_flx_map
      use mo_lifetime,     only : lifetime_inti
      use mo_imp_sol,      only : imp_slv_lt_inti
      use mo_offline_constits, only : init_offline_constits, advance_offline_constits, mxconstitflds
      use mo_offline_sources,  only : init_offline_sources,  advance_offline_sources,  mxsrcsflds
      use mo_chem_utls,    only : iniele

      implicit none

!---------------------------------------------------------------------
!	... dummy args
!---------------------------------------------------------------------
      integer, intent(in)  :: platl, plonl, pplon, nodes
      integer, intent(out) :: nadv
      integer, intent(inout) :: &
        xdtime, &   ! timestep interval in seconds
        ncdate, &   ! current date in yymmdd format
        ncsec       ! current seconds relative to ncdate

      real, intent(inout) :: &
        u(plonl,plev,-3:platl+4,pplon), &   ! u wind (time interpolant)
        v(plonl,plev,-2:platl+3,pplon), &   ! v wind (time interpolant)
        w(plonl,plevp,platl,pplon), &       ! vertical wind (time interpolant)
        as(plonl,plev,pcnst,platl,pplon), & ! advected species (initial conditions)
        precip(plonl,platl,pplon)           ! precip at the ground (m)
      real, dimension(plonl,-3:platl+4,pplon), intent(inout) :: &
        ps                     ! surface pressure (time interpolant)
      real, dimension(plonl,platl,pplon), intent(inout) :: &
        oro, &                 ! orography
        phis, &                ! surface geopotential (m**2/s**2)
        ts, &                  ! surface temperature
        ts_avg, &              ! surface temperature
        fsds_avg, &            ! surface temperature
        taux, &                ! stress in x direction
        tauy, &                ! stress in y direction
        hflx, &                ! surface sensible heat flux (w/m2)
        qflx, &                ! surface water vapor flux (kg/m2/s)
        cldtop, &              ! cloud top level index
        cldbot, &              ! cloud bottom level index
        snow, &                ! snow height (m)
        fsds, &                ! downward direct radiation at ground (w/m^2)
        soilw                  ! soil moisture fraction
      real, dimension(plonl,plev,platl,pplon,2), intent(inout) :: &
        q                      ! specific humidity (time interpolant) (kg/kg)
      real, dimension(plonl,plev,platl,pplon), intent(inout) :: &
        t, &                   ! temperature (time interpolant)
        zmu, &                 ! mu2 from conv_ccm, kg/m2/s
        zmd, &                 ! md2 from conv_ccm, kg/m2/s
        zeu, &                 ! eu2 from conv_ccm, 1/s
        hketa, &               ! interpolated convective mass flux
        hkbeta, &              ! interpolated overshoot parameter
        qrad, &                ! radiative heating tendency (K/s)
        omga, &                ! vertical velocity (omega) (Pa/s)
        cmfdqr, &              ! dq/dt due to convective rainout 
        concld, &              ! convective cloud fraction
        zm, &                  ! potential height above surface at midpoints
        nrain, &               ! rate of release of stratiform precip (1/s)
        nevapr, &              ! rate of evaporation of precipitation (1/s)
        cwat                   ! cloud water (kg/kg)
      real, dimension(plonl,plevp,platl,pplon), intent(inout) :: &
        cgs, &                 ! interpolated counter-gradient coefficient
        kvh, &                 ! interpolated vertical diffusion coefficient
        cldfr, &               ! cloud fraction
        zint                   ! potential height above surface at interfaces

      real, dimension(plonl,plev,platl,pplon), intent(inout) :: &
        pdmfuen, &                 ! Tiedtke entrainment into updraft
        pdmfude, &                 ! Tiedtke detrainment into updraft
        pdmfden, &                 ! Tiedtke entrainment into downdraft
        pdmfdde                    ! Tiedtke detrainment into downdraft

!---------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------
      integer, parameter :: maxnmbrcnsts = 10
      integer, parameter :: maxnmbrsrcs  = 10

      integer :: &
        i, ip, j, k, m, n, &      ! general purpose indices
        file, &                   ! file index
        iret , &                  ! function return value
        inerror, &                ! keep track of input errors
        wrk_date
      integer :: &
        lundyn, &                 ! unit number for dynamics input
        luntra, &                 ! unit number for tracer initial conditions input
        lunrst0, &                ! unit number for restart info file
        lunrst1, &                ! unit number for restart data file
        itimrst, &                ! restart time index
        lunconstit, &             ! unit number for offline constituents input
        lunsrcs, &                ! unit number for offline sources input
        hst_user_fld_cnt

      integer :: &
        nprlev          ! index of lowest pure pressure interface

      integer, dimension(hst_file_max) :: &
        ntshst      = 0, &      ! number of time samples in initial history file
        nhtfrq      = 0, &      ! freq of hist file writes in timesteps (pos) or hours (neg)
        mfilt       = 1, &      ! max number of time samples written to hist tape
        rst_date    = 0, &      ! date retrieved from rstrt file (yyyymmdd) 
        rst_datesec = 0, &      ! seconds in date retrieved from rstrt file (s) 
        rst_days    = 0, &      ! simulation elapsed days retrieved from rstrt file
        rst_secs    = 0         ! seconds in simulation elapsed days retrieved from rstrt file

      real :: &
        calday, &       ! calandar day for current timestep
        delphi2, &      ! delta phi/2
        lat_fac, &      ! latitude factor
        tmp             ! work variable

      real, dimension(plevp) :: &
        hyai, &         ! p0 component of hybrid coordinate - interfaces
        hybi, &         ! ps component of hybrid coordinate - interfaces
        hypi, &         ! reference pressure at interface levels
        etaint, wrk2    ! hyai + hybi
      real, dimension(plev) :: &
        hyam, &         ! p0 component of hybrid coordinate - midpoints
        hybm, &         ! ps component of hybrid coordinate - midpoints
        hybd, &         ! difference of the hybis
        etamid, wrk1    ! hyam + hybm
      real :: &
        wrk_2d(plon,plat)

      character(len=80), dimension(hst_file_max) :: &
        rst_lnhstf      ! path of first hst file for this case
      character(len=128) :: &
        nml_file        ! namelist input file name
      character(len=32) :: &
        case, &         ! case name from the environment variable CASE
        logname, &      ! users name from the environment variable LOGNAME
        calendar        ! calendar type; 365_days or gregorian
      character(len=16) :: &
        up_case, &      ! upper case wrk variable
        sim_type, &     ! simulation type (initial or restart)
        sim_duration, & ! simulation duration
        timestep, &     ! simulation time step
        hst_rt, &       ! history files mass store retention time
        rst_rt          ! restart files mass store retention time
      character(len=16), dimension(hst_file_max) :: &
        hstfrq, &
        hst_prefix      ! history file filename prefix
      character(len=16) :: &
        restart_frq, &
        wrk_time
      character(len=4) :: &
        trnuma          ! for converting integers to character strings
      character(len=3) :: &
        trnum           ! for converting integers to character strings
      character(len=3) :: &
        msclass, &      ! for converting integers to character strings
        uc_msclass      ! for converting integers to character strings
      character(len=1) :: &
        rst_token = ' '

!---------------------------------------------------------------------
!     Control variables not stored in control.com (they are passed to the
!     individual modules).
!---------------------------------------------------------------------
      logical :: &
        rdt               = .false., &    ! read T from the dynamics files
        rdoro             = .true., &     ! read ORO from the dynamics files
        rdphis            = .false., &    ! read PHIS from dynamics files
        negtrc_diagprnt   = .false., &    ! negtrc diagnostic print
        imp_slv_diagprnt  = .false., &    ! imp_slv diagnostic print
        adv_diagprnt      = .false., &    ! advection diagnostic print
        diff_adv          = .true., &     ! diffusive advection
        do_pressure_fixer = .true.        ! pressure fixer in advection ?

!---------------------------------------------------------------------
! 	... Arrays to pass any user specified names or scale factors
!           for the dynamics input fields to inidyn.
!---------------------------------------------------------------------
      character(len=32) :: &
        sim_data_file       ! name of the simulation data file; defaults to sim.dat
      character(len=32) :: &
        dynnm(mxdynflds)    ! alternative names of dynamics fields
      real :: &
        dynsc(mxdynflds)    ! scale factor to convert units in input files to internal units.

!---------------------------------------------------------------------
! 	... Namelist inputs
!---------------------------------------------------------------------
!     namelist variables stored in /control0/
!      integer
!        rstflg     ! 0 for initial run, 1 for restart run, 2 for branch run
!        nestep     ! ending timestep (pos) or day (neg)
!      logical
!        fixmas      ! t => apply mass fixer
!        dowetdep    ! t => apply wet deposition parameterization
!        addqrad     ! t => add radiative heating tendency to the temperature
!                           field supplied to the convection routine
!        limqin      ! t => limit input Q s.t. it doesnt exceed saturation
!---------------------------------------------------------------------

      integer :: &
        dtime, &     ! timestep interval in seconds
        icdate, &    ! date of initial dynamic state in yymmdd format
        icsec, &     ! seconds relative to icdate
        trdate, &    ! date of as and nas initial conditions in yymmdd format
        trsec, &     ! seconds relative to trdate
        dynoffset, & ! offset (seconds) added to (date,sec) for dynamics data
        constitoffset, & ! offset (seconds) added to (date,sec) for constit data
        srcsoffset, & ! offset (seconds) added to (date,sec) for sources data
        stday, &     ! starting day for initial run
        stsec, &     ! seconds relative to stday for initial run
        ndens, &     ! write density for output history tapes
        irt, &       ! retention period in days on mass store for history files.
                     ! If 0 (default) then output files not transfered.
        rstfrq, &    ! freq of restart file writes in timesteps (pos) or day (neg)
        rstrrt       ! retention period in days on mass store for restart files.

      logical :: &
        filtedot, &   ! true => apply filter to calculated etadot
        esice, &      ! true => saturation vapor pressures account for ice phase
        tracfrht, &   ! true => initialize tracers from a history file
        async, &      ! true => output files transferred to ms asynchronously
        linkhst, &    ! true => link completed hist files to linkdir/linkfn
        rmconstits, & ! true => remove old constituents files
        rmsrcs, &     ! true => remove old sources files
        rmout, &      ! true => remove hist file after linking and/or transferring to mass store
        rmdyn, &      ! true => remove old dynamics files
        rmtra, &      ! true => remove tracer initial conditions file
        rmrst         ! true => remove restart data files

!---------------------------------------------------------------------
!     	... Names and scale factors for dynamics input data.
!---------------------------------------------------------------------
      character(len=32) :: &
        unm, &         !  1) zonal wind component                             
        vnm, &         !  2) meridional wind component                        
        psnm, &        !  3) surface pressure                                 
        oronm, &       !  4) orography                                        
        phisnm, &      !  5) surface geopotential                             
        etadotnm, &    !  6) etadot (vertical wind component, d(eta)/dt )     
        omeganm, &     !  7) omega  (vertical wind component, d(pressure)/dt )
        tnm, &         !  8) temperature                                      
        qnm, &         !  9) specific humidity                                
        tsnm, &        ! 10) surface temperature (K)
        tauxnm, &      ! 11) surface zonal stress (n/m2)                      
        tauynm, &      ! 12) surface meridional stress (n/m2)                 
        shflxnm, &     ! 13) surface heat flux (w/m2)                         
        qflxnm, &      ! 14) surface moisture flux (kg/m2/s)                  
        kvhnm, &       ! 15) vertical diffusion coefficient                   
        cgsnm, &       ! 16) counter-gradient coefficient                     
        zmunm, &       ! 17) zhang mu2 from conv_ccm, kg/m2/s                 
        zmdnm, &       ! 18) zhang md2 from conv_ccm, kg/m2/s                 
        zeunm, &       ! 19) zhang eu2 from conv_ccm, 1/s                     
        hketanm, &     ! 20) hack convective mass flux (kg/m^2/s)             
        hkbetanm, &    ! 21) hack overshoot parameter (fraction)              
        qradnm, &      ! 22) radiative heating tendency (k/s)                 
        cmfdqrnm, &    ! 23) dq/dt due to convective rainout (kg/kg/s)
        cldfrnm, &     ! 24) total cloud fraction
        concldnm, &    ! 25) convective cloud fraction
        nrainnm, &     ! 26) dq/dt due to stratiform precip (kg/kg/s)
        nevaprnm, &    ! 27) rate of evaporation of strat precip (kg/kg/s)
        gbcwnm, &      ! 28) grid box cloud water (kg/kg)
        pdmfuennm, &   ! 29) tiedtke entrainment for updroaft                 
        pdmfudenm, &   ! 30) tiedtke detrainment for updroaft                 
        pdmfdennm, &   ! 31) tiedtke entrainment for downdroaft               
        pdmfddenm, &   ! 32) tiedtke detrainment for downdroaft               
        snownm, &      ! 33) snow height
        fsdsnm, &      ! 34) direct shortwave surface radiation
        soilwnm, &     ! 35) soil moisture
        icenm, &       ! 36) sea ice
        ts_avgnm, &    ! 37) monthly average ts
        fsds_avgnm     ! 38) monthly average fsds
      real :: &
        usc, vsc, pssc, orosc, phissc, etadotsc, omegasc, tsc, qsc, tssc, &
        tauxsc, tauysc, shflxsc, qflxsc, kvhsc, cgssc, zmusc, zmdsc, &
        zeusc, hketasc, hkbetasc, qradsc, &
        cmfdqrsc, cldfrsc, concldsc, nrainsc, nevaprsc, gbcwsc, &
        pdmfuensc, pdmfudesc, pdmfdensc, pdmfddesc, &
        snowsc, fsdssc, soilwsc, icesc, ts_avgsc, fsds_avgsc, &
        lght_no_prd_factor = 1.

      real :: qmin(pcnst)                                 ! minimum tracer mixing ratio values.

      type(hst_list)    :: hstinst(hst_file_max)          ! names of time instantaneous namelist output fields
      type(hst_list)    :: hsttimav(hst_file_max)         ! names of time averaged namelist output fields
      type(hst_usr_lst) :: hst_user_flds(hstdim)          ! names of user defined history fields

      character(len=16) :: rrootd            ! Default remote root directory for output history and
                                             ! restart files.  Defaults to the environment variable LOGNAME
      character(len=80)  :: title             ! case title
      character(len=8)   :: msvol             ! mass store virtual volume name
      character(len=8)   ::  wpasswd          ! MS write password

      character(len=128) :: local_data_path  = ' '
      character(len=128) :: remote_data_path = ' '

      character(len=168) ::  ldynpath, rdynpath, dynfile           ! dynamic input 
      character(len=168) ::  licpath, ricpath, icfile              ! initial conditions file
      character(len=168) ::  ljvalpath, rjvalpath, jvalfile        ! jval path info
      character(len=168) ::  lemipath, remipath                    ! emission files (surface and airplane)
      character(len=168) ::  lsurfpath, rsurfpath                  ! surface data
      character(len=168) ::  lsulfpath, rsulfpath                  ! sulfur data
      character(len=168) ::  ldepvelpath, rdepvelpath, depvelfile  ! deposition velocity files
      character(len=168) ::  lubpath, rubpath, ubfile              ! upper boundary condition
      character(len=168) ::  lrestpath, rrestpath, restfile        ! restart files
      character(len=168) ::  rstifile                              ! restart info file
      character(len=256) ::  rhst_lpath(hst_file_max)
      character(len=256) ::  rhst_rpath(hst_file_max)

      type(filespec)     ::  hst_flsp(hst_file_max)                ! history files
      type(filespec)     ::  branch_rest_flsp                      ! branch restart info file filespec
      type(filespec)     ::  aux_flsp(10)                          ! auxilliary file specs

      character(len=30)  :: emires                                 ! resolution string for emissions
      character(len=30)  :: surfres                                ! resolution string for surface data
 
      logical            :: partial_ta(hst_file_max) = .false.     ! history files partial time average flag

      type(time_ramp)    ::  emis_timing                           ! emission timing type
      type(time_ramp)    ::  frc_timing                            ! insitu frcing timing type
      type(time_ramp)    ::  flbc_timing                           ! lbc timing type
      type(time_ramp)    ::  sad_timing                            ! sad timing type

      character(len=8)  :: offline_constituents(maxnmbrcnsts) = ''
      character(len=8)  :: offline_sources(maxnmbrsrcs)       = ''
      character(len=32) :: constits_nm(maxnmbrcnsts) = ''
      character(len=32) :: srcs_nm(maxnmbrsrcs)      = ''
      real :: constits_sc(maxnmbrcnsts) = 1.
      real :: srcs_sc(maxnmbrsrcs)      = 1.

!---------------------------------------------------------------------
!     	... Constants for use in parameterizations
!---------------------------------------------------------------------
      real, parameter :: &
        cpwv   = 1.81e3, &
        cpair  = 1004.64, &
        rair   = 287.04, &
        rh2o   = 461., &
        zvir   = rh2o/rair - 1., &
        gravit = 9.80616, &
        ra     = 1./rearth, &
        epsilo = 0.622, &
        latvap = 2.5104e06, &
        latice = 3.336e5, &
        cappa  = rair/cpair, &
        rhoh2o = 1.e3, &
        tfh2o  = 273.16

!---------------------------------------------------------------------
!     	... Function declarations
!---------------------------------------------------------------------
      integer :: fgetenv

      namelist / nlist / &
        fixmas, filtedot, esice, limqin, arch_file, &
        sim_type, dtime, timestep, icdate, icsec, trdate, trsec, &
        dynoffset, calendar, hst_prefix, rrootd, &
        stday, stsec, tracfrht, qmin, dyn_flnm_prefix, dyn_flnm_date_frmt, &
        hst_flsp, hstinst, hsttimav, hst_user_flds, dyn_flnm_suffix, &
        title, hstfrq, sim_duration, ndens, nhtfrq, mfilt, hst_rt, irt, async, &
        msvol, wpasswd, linkhst, rmout, rmdyn, rmtra, &
        rstfrq, rmrst, rstrrt, rst_rt, msclass, restart_frq, &
        chembgt, dowetdep, rdoro, addqrad, &
        negtrc_diagprnt, adv_diagprnt, imp_slv_diagprnt, diff_adv, do_pressure_fixer, &
        unm, vnm, psnm, oronm, phisnm, etadotnm, omeganm, tnm, qnm, tsnm, &
        tauxnm, tauynm, shflxnm, qflxnm, kvhnm, cgsnm, zmunm, zmdnm, &
        zeunm, hketanm, hkbetanm, qradnm, &
        cmfdqrnm, cldfrnm, concldnm, nrainnm, nevaprnm, gbcwnm, &
        pdmfuennm, pdmfudenm, pdmfdennm, pdmfddenm, &
        snownm, fsdsnm, soilwnm, icenm, &
        usc, vsc, pssc, orosc, phissc, etadotsc, omegasc, tsc, qsc, tssc, &
        tauxsc, tauysc, shflxsc, qflxsc, kvhsc, cgssc, zmusc, zmdsc, &
        zeusc, hketasc, hkbetasc, qradsc, use_dust, case, &
        cmfdqrsc, cldfrsc, concldsc, nrainsc, nevaprsc, gbcwsc, &
        pdmfuensc, pdmfudesc, pdmfdensc, pdmfddesc, sim_data_file, &
        snowsc, fsdssc, soilwsc, icesc, xactive_emissions, &
        flbc_timing, emis_timing, frc_timing, sad_timing, &
        lght_no_prd_factor, xactive_h2o, xactive_prates, xactive_drydep, &
        dyn_flsp, ic_flsp, lbc_flsp, ubc_flsp, photo_flsp, emis_flsp, frc_flsp, &
        photo_xs_long_flsp, photo_rsf_flsp, sad_flsp, &
        photo_xs_short_flsp, photo_xs_o2_flsp, solar_parms_flsp, woods_flsp, neckel_flsp, &
        sulf_flsp, surf_flsp, dvel_flsp, rest0_flsp, rest1_flsp, &
        aux_flsp, branch_rest_flsp, dust_flsp, local_data_path, remote_data_path, &
        constit_flsp, constitoffset, rmconstits, &
        srcs_flsp,    srcsoffset,    rmsrcs, &
        offline_constituents, offline_sources, &
        constits_nm, srcs_nm, constits_sc, srcs_sc

!---------------------------------------------------------------------
!     	... Initial data for namelist variables.
!---------------------------------------------------------------------
      data &
        unm, vnm, psnm, oronm, phisnm, etadotnm, omeganm, tnm, qnm, tsnm, &
        tauxnm, tauynm, shflxnm, qflxnm, kvhnm, cgsnm, zmunm, zmdnm, &
        zeunm, hketanm, hkbetanm, qradnm, &
        cmfdqrnm, cldfrnm, concldnm, nrainnm, nevaprnm, gbcwnm, &
        pdmfuennm, pdmfudenm, pdmfdennm, pdmfddenm, &
        snownm, fsdsnm, soilwnm, icenm, ts_avgnm, fsds_avgnm &
        / mxdynflds*' ' /
      data &
        usc, vsc, pssc, orosc, phissc, etadotsc, omegasc, tsc, qsc, tssc, &
        tauxsc, tauysc, shflxsc, qflxsc, kvhsc, cgssc, zmusc, zmdsc, &
        zeusc, hketasc, hkbetasc, qradsc, &
        cmfdqrsc, cldfrsc, concldsc, nrainsc, nevaprsc, gbcwsc, &
        pdmfuensc, pdmfudesc, pdmfdensc, pdmfddesc, ts_avgsc, fsds_avgsc, &
        snowsc, fsdssc, soilwsc, icesc &
        / mxdynflds*1.0 /

!---------------------------------------------------------------------
!     	... Set defaults for namelist variables.
!           Initialize local variables.
!---------------------------------------------------------------------
      inerror = 0
      call mo_constants_inti

      dyn_flsp   = filespec( "NONE/", "NONE/", "NONE", " " )
      ic_flsp    = filespec( "NONE", "NONE", "NONE", " " )
      photo_flsp = filespec( "NONE", "NONE", "NONE", " " )
      photo_xs_long_flsp  = filespec( "NONE", "NONE", "temp_prs_GT200nm.nc", " " )
      photo_rsf_flsp      = filespec( "NONE", "NONE", "RSF_GT200nm.nc", " " )
      photo_xs_short_flsp = filespec( "NONE", "NONE", "tuv_xsect.nc", " " )
      photo_xs_o2_flsp    = filespec( "NONE", "NONE", "o2src.nc", " " )
      solar_parms_flsp    = filespec( "NONE", "NONE", "solar_parms.nc", " " )
      woods_flsp          = filespec( "NONE", "NONE", "woods.nc", " " )
      neckel_flsp         = filespec( "NONE", "NONE", "neckel.nc", " " )
      dvel_flsp           = filespec( "NONE", "NONE", "regrid_vegetation.nc", " " )
      emis_flsp           = filespec( "NONE", "NONE", "NONE", "T42LR" )
      surf_flsp           = filespec( "NONE", "NONE", "NONE", "t42" )
      sulf_flsp           = filespec( "NONE", "NONE", "sulfate.M1.nc", " " )
      ubc_flsp            = filespec( "NONE", "NONE", "ubvals_mz4.nc", " " )
      lbc_flsp            = filespec( "NONE", "NONE", "lbc_mz4_1985-2007.nc", " " )
      dust_flsp           = filespec( "NONE", "NONE", "dust.nc", " " )
      sad_flsp            = filespec( "NONE", "NONE", "SAD_SULF_1850-2100_1.9x2.5.nc", " " )
      frc_flsp            = filespec( "NONE", "NONE", "NONE", "T42LR" )
      srcs_flsp           = filespec( "NONE/", "NONE/", "NONE", " " )
      constit_flsp        = filespec( "NONE/", "NONE/", "NONE", " " )

      rest0_flsp = filespec( "rest/", " ", "rstrt", " " )
      branch_rest_flsp = filespec( "rest/", " ", "rstrt", " " )
      rest1_flsp  = filespec( "rest/", " ", "r0000.nc", " " )

      emis_timing = time_ramp( "SERIAL", 0, 0 )
      frc_timing  = time_ramp( "CYCLICAL", 19990101, 0 )
      flbc_timing = time_ramp( "SERIAL", 0, 0 )
      sad_timing  = time_ramp( "CYCLICAL", 19970101, 0 )
      rstflg    = 0
      itim0     = 0
      nestep    = 0
      icdate    = 0
      icsec     = 0
      fixmas    = .true.
      filtedot  = .true.
      esice     = .true.
      limqin    = .true.
      xactive_h2o       = .true.                              ! non-interactive water vapor
      xactive_prates    = .true.                              ! table driven photorates
      xactive_drydep    = .true.                              ! table driven dry dep velocities
      xactive_emissions = .true.                              ! non-interactive no,isoprene srf emissions
      use_dust          = .true.                              ! use specified dust
      stday     = 0
      stsec     = 0
      tracfrht  = .true.
      trdate    = -1
      trsec     = -1
      dynoffset = 0
      constitoffset = 0
      srcsoffset = 0
      calendar  = '365_days'
      sim_type  = 'initial'
      hst_prefix(:) = ' '

      do m = 1,pcnst
         qmin(m) = 0.
      end do
      do m = 1,hst_file_max
         hstinst(m)%list(:)  = ' '
         hsttimav(m)%list(:) = ' '
      end do

      rrootd   = '                '
      title    = 'Mozart4 off-line driver'
      ndens    = 1
      irt      = 0
      rstrrt   = 0
#ifdef NCAR
      async    = .true.
#else
      async    = .false.
#endif
      msvol    = '        '
      wpasswd  = '        '
      linkhst  = .false.
      rmout    = .false.
      rmdyn    = .false.
      rmconstits = .false.
      rmsrcs   = .false.
      rmtra    = .false.
      rmrst    = .false.
      rstfrq   = 0

!---------------------------------------------------------------------
!     	... Logical unit numbers for Fortran I/O
!---------------------------------------------------------------------
      lundyn     = navu()
      luntra     = navu()
      lunrst0    = navu()
      lunrst1    = navu()
      lunconstit = navu()
      lunsrcs    = navu()

!---------------------------------------------------------------------
!     	... Initialize control variables
!---------------------------------------------------------------------
      divdiff   = .false.
      arvdiff   = .false.
      diconvccm = .false.
      arconvccm = .false.
      dicldphys = .false.
      arcldphys = .false.
      ditiedtke = .false.
      artiedtke = .false.
      stratchem = .false.
      tropchem  = .false.
      chembgt   = .false.
      vdiffuse  = .false.
      convect   = .false.
      chemistry = .false.
      physics   = .false.
      calcedot  = .false.
      advqth    = .false.
      rdomega   = .false.
      satvap    = .false.
      dowetdep  = .false.
      addqrad   = .false.
#ifdef DI_VDIFF
      divdiff   = .true.
      advqth    = .true.
#endif
#ifdef AR_VDIFF
      arvdiff   = .true.
#endif
#ifdef DI_CONV_CCM
      diconvccm = .true.
      advqth    = .true.
      satvap    = .true.
#endif
#ifdef AR_CONV_CCM
      arconvccm = .true.
#endif
#ifdef DI_CLOUD_PHYS
      dicldphys = .true.
      advqth    = .true.
      tracnam(pcnst) = 'CWAT    '
#endif
#ifdef AR_CLOUD_PHYS
      arcldphys = .true.
      tracnam(pcnst) = 'CWAT    '
#endif
#ifdef DI_TIEDTKE
      ditiedtke = .true.
      advqth    = .true.
      satvap    = .true.
#endif
#ifdef AR_TIEDTKE
      artiedtke = .true.
      satvap    = .true.
#endif
#ifdef STRAT_CHEM
      stratchem = .true.
#endif
#ifdef TROP_CHEM
      tropchem  = .true.
      chembgt   = .true.
      satvap    = .true.
      dowetdep  = .true.
#endif
#ifdef CALC_ETADOT
      calcedot  = .true.
#endif
#ifdef READ_OMGA
      rdomega   = .true.
#endif

      vdiffuse  = divdiff .or. arvdiff
      convect   = diconvccm .or. arconvccm .or. ditiedtke .or. artiedtke
      chemistry = stratchem .or. tropchem
      if( vdiffuse .or. convect .or. chemistry ) then
         physics = .true.
         rdt     = .true.
         rdphis  = .true.
      end if

!---------------------------------------------------------------------
!     	... Check for inconsistent options
!---------------------------------------------------------------------
      if( diconvccm .and. .not. divdiff ) then
         write(*,*) 'inirun: diagnosed CCM convection uses pblh, tpert'
         write(*,*) '  and qpert calculated by diagnosed vertical'
         write(*,*) '  diffusion scheme.  Must define DI_VDIFF.'
         inerror = inerror + 1
      end if

      if( dicldphys .and. .not. diconvccm ) then
         write(*,*) 'inirun: diagnosed cloud physics requires diagnosed'
         write(*,*) '  convection.  Must define DI_CONV_CCM'
         inerror = inerror + 1
      end if

      if( dowetdep .and. (.not. dicldphys .and. .not. arcldphys)) then
         write(*,*) 'inirun: wet deposition requires cloud physics.'
         write(*,*) '  Must define DI_CLOUD_PHYS or AR_CLOUD_PHYS.'
         inerror = inerror + 1
      end if

      if( dicldphys .and. pcnst .eq. 1 ) then
         write(*,*) 'inirun: diagnosed cloud physics uses tracer pcnst'
         write(*,*) '  for the cloud water.  No space has been allocated'
         write(*,*) '  for user defined tracers.  Must define PCNST > 1'
         inerror = inerror + 1
      end if

      case          = ' '
      sim_duration  = ' '
      timestep      = ' '
      hst_rt        = ' '
      msclass       = 'EC '
      rst_rt        = ' '
      restart_frq   = ' '
      hstfrq(:)     = ' '
      sim_data_file = ' '
      hst_user_flds(:)%name   = ' '
      hst_user_flds(:)%levs   = ' '
      hst_user_flds(:)%units  = ' '
      hst_flsp(:)%local_path  = ' '
      hst_flsp(:)%remote_path = ' '
      hst_flsp(:)%nl_filename = ' '
      aux_flsp(:)%nl_filename = ' '
!--------------------------------------------------------------------
!   	... Get namelist filespec
!---------------------------------------------------------------------
      nml_file(:) = ' '
      call getarg( 1, nml_file )
!---------------------------------------------------------------------
!   	... Read namelist from standard input
!---------------------------------------------------------------------
      file = navu()
      open( unit = file, file = trim(nml_file),iostat=iret )
      if( iret /= 0 ) then
         write(*,*) 'inirun: Failed to open namelist file'
         call endrun
      end if
      read(file,nlist,iostat=iret)
      if( iret /= 0 ) then
         write(*,*) 'inirun: Failed to read namelist file ',trim(nml_file),'; error = ',iret
         call endrun
      end if
      close( file )

!---------------------------------------------------------------------
!	... check and setup calendar
!---------------------------------------------------------------------
      if( calendar(1:8) /= '365_days' .and. &
          calendar(1:9) /= 'gregorian' ) then
         write(*,*) 'inirun: Calendar must be 365_days or gregorian'
         write(*,*) '        Calendar input = ',calendar
         inerror = inerror + 1
      end if
!---------------------------------------------------------------------
!	... Initialize the calendar calculations
!---------------------------------------------------------------------
      if( inerror == 0 ) then
        if( calendar == '365_days' ) then
          calendar = '365'
        end if
        call inicalendar( calendar )
      end if

!---------------------------------------------------------------------
!	... diagnostics
!---------------------------------------------------------------------
      pdiags%negtrc  = negtrc_diagprnt
      pdiags%imp_slv = imp_slv_diagprnt
      pdiags%adv     = adv_diagprnt
!---------------------------------------------------------------------
!	... simulation time step
!---------------------------------------------------------------------
         if( timestep /= ' ' ) then
            call time_conversion( timestep, dtime, valid_timec='SMH' )
         end if
!---------------------------------------------------------------------
!	... check user defined fields
!---------------------------------------------------------------------
      do i = 0,hstdim-1
         if( hst_user_flds(i+1)%name == ' ' ) then
            exit
         end if
         call upcase( hst_user_flds(i+1)%levs, up_case )
         if( up_case /= 'SINGLE' .and. up_case /= 'MULTMID' ) then
            write(*,*) 'inirun: Hst user field lev type not SINGLE or MULTMID'
            write(*,*) '        hst_user_flds = ',hst_user_flds(i+1),up_case
            call endrun
         end if
      end do
      hst_user_fld_cnt = i
!---------------------------------------------------------------------
!	... history output timing
!---------------------------------------------------------------------
      do file = 1,hst_file_max
        if( hstfrq(file) == ' ' ) then
          cycle
        end if
        wrk_time = hstfrq(file)
        call time_conversion( wrk_time, nhtfrq(file), hstfrq(file), dtime )
        if( masternode ) then
          write(*,*) 'inirun: file,hstfrq,nhtfrq = ', file,hstfrq(file),nhtfrq(file)
        end if
      end do
!---------------------------------------------------------------------
!     	... set default history local filepaths and names
!---------------------------------------------------------------------
      write(*,*) 'inirun: history filespecs'
      do file = 1,hst_file_max
        if( hstfrq(file) /= ' ' ) then
          if( hst_flsp(file)%local_path == ' ' ) then
            hst_flsp(file)%local_path   = 'hist/'
          end if
          if( hstfrq(file) /= 'm' ) then
            ncdate = icdate
            ncsec  = icsec
            call addsec2dat( dtime*nhtfrq(file), ncdate, ncsec )
            call make_filename( ncdate, ncsec, file, .false., case, hst_flsp(file)%nl_filename )
          else
            call make_filename( icdate, icsec, file, .true., case, hst_flsp(file)%nl_filename )
          end if
          if( masternode ) then
            write(*,*) trim( hst_flsp(file)%nl_filename )
          end if
        end if
      end do
      if( sim_data_file == ' ' ) then
         sim_data_file = 'sim.dat'
      end if
!---------------------------------------------------------------------
!     	... Set internal rstflg from the namelist sim_type
!---------------------------------------------------------------------
      call upcase( sim_type, up_case )
      if( up_case /= 'INITIAL' .and. up_case /= 'RESTART' .and. up_case /= 'BRANCH' ) then
         write(*,*) 'inirun: Simulation type ',trim( up_case ),' must be INITIAL, RESTART, or BRANCH'
         call endrun
      else if( up_case == 'RESTART' ) then
         rstflg = 1
      else if( up_case == 'BRANCH' ) then
         rstflg = 2
      end if
      write(*,*) ' '
      write(*,*) '---------------------------------------'
      write(*,*) 'inirun: Simulation is ',trim( up_case )
      write(*,*) '---------------------------------------'
      write(*,*) ' '
      if( any( offline_constituents(:maxnmbrsrcs) /= ' ' ) ) then 
         write(*,*) 'inirun: offline_constituents'
         do i = 1,maxnmbrcnsts
            write(*,*) i,' : ', offline_constituents(i),'  nm : ',constits_nm(i),'  sc : ',constits_sc(i)
         end do
      else
         write(*,*) 'inirun: there are no offline constituents'
      end if

      if( any( offline_sources(:maxnmbrsrcs) /= ' ' )  ) then
         write(*,*) 'inirun: offline_sources'
         do i = 1,maxnmbrsrcs
            write(*,*) i,' : ', offline_sources(i),'  nm : ',srcs_nm(i),'  sc : ',srcs_sc(i)
         end do
      end if
!---------------------------------------------------------------------
!     	... Initialize dynamics input scale factors
!---------------------------------------------------------------------
         dynsc(1) = usc      
         dynsc(2) = vsc      
         dynsc(3) = pssc     
         dynsc(4) = orosc    
         dynsc(5) = phissc   
         dynsc(6) = etadotsc 
         dynsc(7) = omegasc  
         dynsc(8) = tsc      
         dynsc(9) = qsc      
         dynsc(10) = tssc
         dynsc(11) = tauxsc   
         dynsc(12) = tauysc   
         dynsc(13) = shflxsc  
         dynsc(14) = qflxsc   
         dynsc(15) = kvhsc    
         dynsc(16) = cgssc    
         dynsc(17) = zmusc    
         dynsc(18) = zmdsc    
         dynsc(19) = zeusc    
         dynsc(20) = hketasc 
         dynsc(21) = hkbetasc
         dynsc(22) = qradsc   
         dynsc(23) = cmfdqrsc
         dynsc(24) = cldfrsc
         dynsc(25) = concldsc
         dynsc(26) = nrainsc
         dynsc(27) = nevaprsc
         dynsc(28) = gbcwsc
         dynsc(29) = pdmfuensc
         dynsc(30) = pdmfudesc
         dynsc(31) = pdmfdensc
         dynsc(32) = pdmfddesc
         dynsc(33) = snowsc   
         dynsc(34) = fsdssc   
         dynsc(35) = soilwsc   
         dynsc(36) = icesc   
         dynsc(37:38) = 1.
!---------------------------------------------------------------------
!     	... Initialize dynamics input data names
!---------------------------------------------------------------------
         dynnm(1) = unm      
         dynnm(2) = vnm      
         dynnm(3) = psnm     
         dynnm(4) = oronm    
         dynnm(5) = phisnm   
         dynnm(6) = etadotnm 
         dynnm(7) = omeganm  
         dynnm(8) = tnm      
         dynnm(9) = qnm      
         dynnm(10) = tsnm
         dynnm(11) = tauxnm   
         dynnm(12) = tauynm   
         dynnm(13) = shflxnm  
         dynnm(14) = qflxnm   
         dynnm(15) = kvhnm    
         dynnm(16) = cgsnm    
         dynnm(17) = zmunm    
         dynnm(18) = zmdnm    
         dynnm(19) = zeunm    
         dynnm(20) = hketanm 
         dynnm(21) = hkbetanm
         dynnm(22) = qradnm   
         dynnm(23) = cmfdqrnm
         dynnm(24) = cldfrnm
         dynnm(25) = concldnm
         dynnm(26) = nrainnm
         dynnm(27) = nevaprnm
         dynnm(28) = gbcwnm
         dynnm(29) = pdmfuennm
         dynnm(30) = pdmfudenm
         dynnm(31) = pdmfdennm
         dynnm(32) = pdmfddenm
         dynnm(33) = snownm   
         dynnm(34) = fsdsnm   
         dynnm(35) = soilwnm   
         dynnm(36) = icenm   
         dynnm(37:38) = ' '

!---------------------------------------------------------------------
!	... process timing controls
!---------------------------------------------------------------------
!---------------------------------------------------------------------
!	... get simulation duration
!---------------------------------------------------------------------
         if( sim_duration == ' ' ) then
           write(*,*) 'inirun: must specify simulation duration via "sim_duration"'
           write(*,*) '        in simulation namelist file'
           call endrun
         end if
         if( rstflg == 0 ) then
           nestep = get_sim_duration( icdate, icsec, sim_duration, dtime )
         end if
         if( arch_file /= ' ' ) then
            write(*,*) 'inirun: simulation will archive output file(s)'
         end if
!---------------------------------------------------------------------
!	... history files mass store retention time
!---------------------------------------------------------------------
         if( hst_rt /= ' ' ) then
            m = len_trim( hst_rt )
            call upcase( hst_rt, up_case )
            select case( up_case(m:m) )
               case( 'D', 'Y' )
                  n = m - 1
               case default
                  n = m
            end select
            read(hst_rt(1:n),*,iostat=iret) irt
            if( iret /= 0 ) then
               write(*,*) 'inirun: irt read failed; error = ',iret
               call endrun
            end if
            irt = abs( irt )
            if( up_case(m:m) == 'Y' ) then                              ! years
               irt = irt * 365
            end if
         end if
!---------------------------------------------------------------------
!	... restart files mass store retention time
!---------------------------------------------------------------------
         if( rst_rt /= ' ' ) then
            m = len_trim( rst_rt )
            call upcase( rst_rt, up_case )
            select case( up_case(m:m) )
               case( 'D', 'Y' )
                  n = m - 1
               case default
                  n = m
            end select
            read(rst_rt(1:n),*,iostat=iret) rstrrt
            if( iret /= 0 ) then
               write(*,*) 'inirun: rstrrt read failed; error = ',iret
               call endrun
            end if
            rstrrt = abs( rstrrt )
            if( up_case(m:m) == 'Y' ) then                              ! years
               rstrrt = rstrrt * 365
            end if
         end if
!---------------------------------------------------------------------
!	... restart output timing
!---------------------------------------------------------------------
         if( restart_frq /= ' ' ) then
            call time_conversion( restart_frq, rstfrq, rst_token, dtime )
            write(*,*) 'inirun: restart_frq,rstfrq,rst_token = ', trim(restart_frq),rstfrq,rst_token
         end if
      call hst_files( hstinst, hsttimav )
!---------------------------------------------------------------------
!	... Enroll the user defined fields
!---------------------------------------------------------------------
      if( hst_user_fld_cnt > 0 ) then
         do i = 1,hst_user_fld_cnt
            call addfld( hst_user_flds(i)%name, hst_user_flds(i)%levs, hst_user_flds(i)%units )
         end do
      end if
      if( masternode ) then
         write(*,*) title
         write(*,*) 'primary hstinst'
         write(*,*) hstinst(1)%list(:10)
         write(*,*) 'primary hsttimav'
         write(*,*) hsttimav(1)%list(:10)
      end if


      xdtime  = dtime
      write(*,*) 'inirun: There are ',match_file_cnt,' namelist history files'

!-----------------------------------------------------------------------
!	... Initialize the chem module
!-----------------------------------------------------------------------
      call chem_mods_inti
!---------------------------------------------------------------------
!     	... All nodes read the simulation chemistry input
!---------------------------------------------------------------------
      call read_sim_chm( sim_data_file, hstinst, hsttimav, sim_file_cnt )
      call set_sim_dat

!---------------------------------------------------------------------
!     	... Cast "nestep", "nhtfrq" and "rstfrq" in terms of time step
!---------------------------------------------------------------------
      if( nestep < 0 ) then
         nestep = -nestep*secpday/dtime
      end if
      if( rstfrq < 0 ) then
         rstfrq = -rstfrq*secpday/dtime
      end if
      do file = 1,sim_file_cnt
         if( nhtfrq(file) < 0 .and. hstfrq(file) == ' ' ) then
            nhtfrq(file) = -nhtfrq(file)*3600/dtime
         end if
      end do

!---------------------------------------------------------------------
!     	... check for invalid namelist input
!---------------------------------------------------------------------
      if( rstflg == 0 .and. icdate == 0 ) then
         write(*,*) 'inirun: Must specify date for initial dynamic'
         write(*,*) '         state.  Use the ICDATE namelist variable.'
         inerror = inerror + 1
      end if
      if( trdate == -1 ) then
         trdate = icdate
         trsec  = icsec
      end if
      if( rstflg == 0 .and. nestep == 0 ) then
         write(*,*) 'inirun: simulation duration must be > 0'
         inerror = inerror + 1
      end if
      do file = 1,sim_file_cnt
         if( nhtfrq(file) == 0 ) then
            write(*,*) 'inirun: Must specify history output'
            write(*,*) '        frequency using the NHTFRQ namelist variable'
            inerror = inerror + 1
         end if
      end do
      if( rstfrq == 0 ) then
         rstfrq = mfilt(1)*nhtfrq(1)
      end if
      if( io_node ) then
         if( rstflg == 0 .and. dyn_flsp%local_path(1:1) == ' ' .and. &
	                       dyn_flsp%remote_path(1:1) == ' ' ) then
            write(*,*) 'inirun: Must specify dynamics datasets using the'
            write(*,*) '        dyn_flsp%local_path and/or dyn_flsp%remote_path namelist variables'
            inerror = inerror + 1
         end if
         if( dvel_flsp%local_path(1:1) == ' ' .and. dvel_flsp%remote_path(1:1) == ' ' ) then
            write(*,*) 'inirun: Must specify depvel datasets using the'
            write(*,*) '        dvel_flsp%local_path and/or dvel_flsp%remote_path namelist variables'
            inerror = inerror + 1
         end if
      end if
      if( rstflg == 0 .and. tracfrht .and. io_node ) then
         if( ic_flsp%local_path(1:1) == ' ' .and. ic_flsp%remote_path(1:1) == ' ' &
                                            .and. ic_flsp%nl_filename(1:1) == ' ' ) then
            write(*,*) 'inirun: Must specify tracer initial condition dataset'
            write(*,*) '        using the ic_flsp namelist variable'
            inerror = inerror + 1
         end if
      end if

#if !defined(NCAR)
      if( async .and. io_node ) then
         write(*,*) 'inirun: async=.true. only available on NCAR SCD machines'
         async = .false.
      end if
#endif

!---------------------------------------------------------------------
!     	... Get environment variables for CASE and LOGNAME
!---------------------------------------------------------------------
      if( case == ' ' ) then
         iret = fgetenv( 'CASE', case )
         if( iret /= 0 ) then
            write(*,*) 'inirun: CASE not found.  Set to debug'
            case = 'debug'
         end if
      end if
      write(*,*) 'CASE = ',case(:glc(case))
      logname = ' '
      iret = fgetenv( 'LOGNAME', logname )
      if( iret /= 0 ) then
         write(*,*) 'inirun: LOGNAME not set'
      else
         if( rrootd(1:1) == ' ' ) then
            call upcase( logname, rrootd )
         end if
      end if
      write(*,*) 'LOGNAME = ',logname(:glc(logname))
      if( irt /= 0 .and. rstflg == 0 ) then
         if( rrootd(1:1) == ' ' ) then
            write(*,*) 'inirun: Don''t have remote path for disposing'
            write(*,*) '        history files.  Use namelist variable RROOTD'
            inerror = inerror + 1
         end if
      end if

!---------------------------------------------------------------------
!     	... Exit if input error(s)
!---------------------------------------------------------------------
      if( inerror > 0 ) then
         call endrun
      end if


!---------------------------------------------------------------------
!	... check and set filespecs
!---------------------------------------------------------------------
      call check_filespecs( local_data_path )

!---------------------------------------------------------------------
!     	... Initialize restart (on restart ncdate, ncsec are overwritten)
!---------------------------------------------------------------------
      ncdate  = icdate
      ncsec   = icsec
      call inirest( rstflg, lunrst0, lunrst1, rstrrt, rstfrq, &
                    rrootd, case, async, rmrst, &
                    stratchem, tropchem, diconvccm, dicldphys, ncdate, &
                    ncsec, itimrst, ntshst, partial_ta, hst_flsp, &
                    rst_date, rst_datesec, rst_days, rst_secs, rst_lnhstf, &
		    branch_rest_flsp, rst_token, rhst_lpath, rhst_rpath )
      if( rstflg /= 0 ) then
        nestep = get_sim_duration( ncdate, ncsec, sim_duration, dtime )
        if( nestep == 0 ) then
          write(*,*) 'inirun: simulation duration must be > 0'
          inerror = inerror + 1
        end if
      end if
      if( io_node ) then
	 do file = 1,sim_file_cnt
            if( hfile(file)%rpath(1:1) == ' ' ) then
               write(*,*) 'inirun: Don''t have remote path for disposing'
               write(*,*) '        history file ',file,'  Use namelist variable RHISTPATH'
               inerror = inerror + 1
	       exit
            end if
	 end do
         if( inerror > 0 ) then
	    call endrun
         end if
      end if
      write(*,*) 'inirun: restart initialized'
      itim0 = itimrst
      write(*,'('' inirun: Start time index = '',i8)') itim0

      begin_time%date = ncdate
      begin_time%secs = ncsec
      end_time%date = ncdate
      end_time%secs = ncsec
      call addsec2dat( dtime*nestep, end_time%date, end_time%secs )

!---------------------------------------------------------------------
!     	... Set FFSL grid and weigths
!---------------------------------------------------------------------
      lat_fac = 1./real(plat-1)
      write(*,*) 'inirun: lat_fac (deg) = ',lat_fac*180.
      do j = 0,plat-1
	 phi(j+1) = (real(j)*lat_fac -.5)*pi
      end do
      lat_fac = real(2*(plat-1))
      do j = 2,plat-1
	 latwts(j) = sin( phi(j) + pi/lat_fac ) - sin( phi(j) - pi/lat_fac )
      end do
      latwts(1)    = 1. - sin( phi(plat) - pi/lat_fac )
      latwts(plat) = latwts(1)
      write(*,*) 'inirun: latwts(1:plat)'
      write(*,'(1p,10e10.3)') latwts(:plat)
      lam(:plon)    = (/ (real(i)*2.*pi/real(plon),i=0,plon-1) /)
      sinlam(:plon) = sin( lam(:) )
      coslam(:plon) = cos( lam(:) )

      if( comp_node ) then
!---------------------------------------------------------------------
!     	... Initialize dynamics data and get coordinate data.
!---------------------------------------------------------------------
         call upcase( dyn_flnm_date_frmt, up_case )
         dyn_flnm_date_frmt = up_case
         call inidyn( plon, plat, plev, plevp, &
                      ncdate, ncsec, dynoffset, lundyn, &
                      rmdyn, rdt, rdoro, rdphis, calcedot, rdomega, &
                      arvdiff, divdiff, arconvccm, addqrad, &
                      arcldphys, artiedtke, ditiedtke, dynnm, dynsc, &
                      hyam, hybm, hyai, hybi, phis, &
		      oro, plonl, platl, pplon )
         if( masternode .and. .not. rdoro ) then
            write(*,*) 'inirun: ****WARNING**** the oro field has not been'
            write(*,*) '        initialized.  It is required by the surface'
            write(*,*) '        flux and diagnosed cloud physics modules'
         end if
         write(*,*) ' inirun: dynamics initialized'
!---------------------------------------------------------------------
!     	... Initialize offline constituent data
!---------------------------------------------------------------------
         call init_offline_constits( ncdate, ncsec, constitoffset, lunconstit, maxnmbrcnsts, &
                                     offline_constituents, rmconstits, constits_nm, constits_sc, &
                                     plonl, platl, pplon )
         write(*,*) ' inirun: finished init_offline_constits'
!---------------------------------------------------------------------
!     	... Initialize offline constituent data
!---------------------------------------------------------------------
         call init_offline_sources( ncdate, ncsec, srcsoffset, lunsrcs, maxnmbrsrcs, &
                                    offline_sources, rmsrcs,  srcs_nm, srcs_sc, &
                                    plonl, platl, pplon )
         write(*,*) ' inirun: finished init_offline_sources'
      end if
#ifdef USE_MPI
      if( ded_io_node ) then
         call mpi_bcast( hyam, plev, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, iret )
         if( iret /= MPI_SUCCESS ) then
            write(*,*) 'inirun: Bcast for hyam failed; code = ',iret
            call endrun
         end if
         call mpi_bcast( hybm, plev, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, iret )
         if( iret /= MPI_SUCCESS ) then
	    write(*,*) 'inirun: Bcast for hybm failed; code = ',iret
	    call endrun
         end if
         call mpi_bcast( hyai, plevp, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, iret )
         if( iret /= MPI_SUCCESS ) then
	    write(*,*) 'inirun: Bcast for hyai failed; code = ',iret
	    call endrun
         end if
         call mpi_bcast( hybi, plevp, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, iret )
         if( iret /= MPI_SUCCESS ) then
	    write(*,*) 'inirun: Bcast for hybi failed; code = ',iret
	    call endrun
         end if
      end if
#endif

!---------------------------------------------------------------------
!     	... Initialize vertical coordinates
!---------------------------------------------------------------------
      call hycoef( hyai, hyam, hybi, hybm, hypi, &
                   hybd, nprlev, etamid, etaint )
      write(*,*) 'inirun: vertical coordinates set'

!---------------------------------------------------------------------
!     	... Initialize pressure levels calculation
!---------------------------------------------------------------------
      call iniplevs( hyai, hyam, hybi, hybm )
      write(*,*) 'inirun: pressure levels initialized'

Comp_nodes_only : &
      if( comp_node ) then
!---------------------------------------------------------------------
!     	... Check that input data interval contains initial time.
!           If not, read time samples until the interval is found.
!---------------------------------------------------------------------
         call chkdyn( ncdate, ncsec, phis, oro, plonl, &
		      platl, pplon )
!---------------------------------------------------------------------
!     	... Interpolate dynamics to initial time
!---------------------------------------------------------------------
         call interp1( ncdate, ncsec, ps, t, q, &
                       ts, ts_avg, fsds_avg, plonl, platl, pplon )
         call interp2( ncdate, ncsec, ps, &
                       u, v, w, cgs, kvh, &
                       ts, taux, tauy, hflx, qflx, qrad, &
                       omga, pdmfuen, pdmfude, pdmfden, pdmfdde, &
		       snow, fsds, soilw, plonl, platl, pplon )

!---------------------------------------------------------------------
!   	... Initialize saturation vapor pressure tables
!---------------------------------------------------------------------
         if( satvap .or. limqin ) then
            call esinti( epsilo, latvap, latice, rh2o, cpair, esice )
            write(*,*) ' inirun: saturation vapor pressure initialized'
         end if

!---------------------------------------------------------------------
!   	... Limit input specific humidity to saturation
!---------------------------------------------------------------------
         write(*,*) 'inirun: min sh = ',minval( q(:,:,:,:,1) ),' on node ',thisnode
         write(*,*) 'inirun: max sh = ',maxval( q(:,:,:,:,1) ),' on node ',thisnode
         if( limqin ) then
            call limsh( ps, t, q, plonl, platl, pplon )
            write(*,*) 'inirun: after limsh, min sh = ',minval( q(:,:,:,:,1) ),' on node ',thisnode
            write(*,*) 'inirun: after limsh, max sh = ',maxval( q(:,:,:,:,1) ),' on node ',thisnode
         end if

!---------------------------------------------------------------------
!     	... Initialize mass integral calculations
!---------------------------------------------------------------------
         call inimass( gravit, latwts, etamid )
         write(*,*) 'inirun: mass integrals intialized'

!---------------------------------------------------------------------
!     	... Initialize virtual temperature calculation
!---------------------------------------------------------------------
         call inivtem( zvir )
         write(*,*) 'inirun: virtual temperature initialized'

!---------------------------------------------------------------------
!     	... Initialize potential temperature calculation.
!---------------------------------------------------------------------
         call inipottem( cappa )
         write(*,*) 'inirun: potential temperature initialized'

!---------------------------------------------------------------------
!     	... Initialize geopotential height calculation.
!---------------------------------------------------------------------
         call iniz( rair, gravit )
         write(*,*) 'inirun: geopotential height initialized'

!---------------------------------------------------------------------
!     	... Initialize surface flux calculation.
!---------------------------------------------------------------------
         call inisflx( rair, gravit )
         write(*,*) 'inirun: surface flux initialized'

!---------------------------------------------------------------------
!     	... Initialize minimum mixing ratio value enforcer.
!---------------------------------------------------------------------
         call iniqneg( qmin )
         write(*,*) 'inirun: water vapor floor initialized'

!---------------------------------------------------------------------
!  	... Initialize Hacks convective mass flux adjustment parameterization.
!---------------------------------------------------------------------
         if( arconvccm .or. diconvccm ) then
            call mfinti( hypi, rair, cpair, gravit, latvap, rhoh2o )
            write(*,*) 'inirun: convection initialized'
         end if

!---------------------------------------------------------------------
!   	... Initialize cloud physics calculations.
!---------------------------------------------------------------------
         if( dicldphys ) then
            call inicld( phi, cappa, gravit, rair, latvap, &
                         cpair, rhoh2o*1.e-3, tfh2o, etamid )
            write(*,*) 'inirun: clouds intialized'
            call inimc( 1.e5*etamid )
            write(*,*) 'inirun: inimc initialized'
            call inimland( oro, phi, plonl, platl, pplon, nodes )
            write(*,*) 'inirun: land mask initialized'
         end if

!---------------------------------------------------------------------
!  	... Initialize vertical diffusion.
!---------------------------------------------------------------------
         if( vdiffuse ) then
            call vdinti( cpwv, cpair, gravit, rair, qmin, 1.e5*etamid )
            write(*,*) 'inirun: vertical diffusion initialized'
         end if

!---------------------------------------------------------------------
!  	... Initialize ffsl_adv
!---------------------------------------------------------------------
         if( advqth ) then
            nadv = 2
         end if
         call ffsl_inti( pcnst + nadv - 1, diff_adv, do_pressure_fixer, platl )
         write(*,*) 'inirun: advection initialized'

!---------------------------------------------------------------------
!    	... initialize tropospheric chemistry calculation
!---------------------------------------------------------------------
         call chemini( calday, plonl, platl, pplon, ncdate, &
                       ncsec, emis_timing, frc_timing, flbc_timing, sad_timing, &
                       dtime )
         write(*,*) 'inirun: chemistry initialized'

!---------------------------------------------------------------------
!     	... initialize aerosols
!---------------------------------------------------------------------
         call sox_inti
         call soa_inti
         call seasalt_inti( plonl, platl, plev, pplon, oro, 1.e5*etamid )
         call aerosols_inti( plonl, platl, plev, pplon )

!---------------------------------------------------------------------
!     	... initialize deposition velocities
!---------------------------------------------------------------------
         if( drydep_cnt > 0 ) then
            if( .not. xactive_drydep ) then
               call dvel_inti_table( plonl, platl, pplon )
            else
               call dvel_inti_xactive( plonl, platl, pplon )
            end if
            write(*,*) 'inirun: dry deposition initialized'
         else
            write(*,*) 'inirun: there are no species with dry deposition'
         end if

         if( .not. xactive_emissions ) then
            if( srfems_cnt > 0 ) then
               xactive_emissions = any( xactive_srf_flx_map(:srfems_cnt) )
            else
               xactive_emissions = .false.
            end if
         end if
         if( xactive_emissions ) then
!---------------------------------------------------------------------
!       ... initialize bvoc interactive emissions
!---------------------------------------------------------------------
            call xemis_inti( ncdate, ts_avg, fsds_avg, plonl, platl, pplon )
         end if
         call chemdr_inti
      end if Comp_nodes_only

!---------------------------------------------------------------------
!    	... Initialize history output
!---------------------------------------------------------------------
      call upcase( msclass, uc_msclass )
      call inihist( rrootd, case, ntshst, nestep, nhtfrq, &
                    hstfrq, ndens, mfilt, stday, stsec, &
                    hst_flsp, icdate, icsec, dtime, rstflg, &
                    title, phi, latwts, irt, async, &
                    rmout, msvol, wpasswd, stratchem, tropchem, &
                    partial_ta, rst_date, rst_datesec, rst_days, rst_secs, &
                    rst_lnhstf, uc_msclass, rhst_lpath, rhst_rpath, plonl, platl )

Comp_nodes_only_1 : &
      if( comp_node ) then
!---------------------------------------------------------------------
!   	... initialize tracers
!---------------------------------------------------------------------
         if( rstflg == 0 ) then
            if( tracfrht ) then
               call initrac( trdate, trsec, luntra, rmtra, as, &
                             ps, plonl, pplon, platl )
            else
               call initr0( as, ps, phi, plonl, platl, pplon )
            end if
!---------------------------------------------------------------------
!  	... initialize dry mass adjustment calculation
!---------------------------------------------------------------------
            call inidry( ps, q, plonl, platl, pplon )
            write(*,*) 'inirun: dry mass adjustment initialized'
!-------------------------------------------------------------------
!	... initialize cloud top index
!-------------------------------------------------------------------
            if( arcldphys .or. dicldphys ) then
               cldtop(:,:,:) = real(plev)
               cldbot(:,:,:) = real(plev)
            end if
            if( xactive_h2o ) then
               do ip = 1,pplon
                  do j = 1,platl
                     do k = 1,plev
                        q(:,k,j,ip,2) = q(:,k,j,ip,1)
                     end do
                  end do
               end do
            end if
         else if( rstflg == 1 .or. rstflg == 2 ) then
!-------------------------------------------------------------------
!   	... read restart data file
!-------------------------------------------------------------------
# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
            if( .not. xactive_h2o ) then
               if( xactive_drydep .or. xactive_emissions ) then
                  call rdrst( as, cldtop, cldbot, zm, zint, &
                              rstflg, plonl, platl, pplon, ps, &
                              q, precip )
               else
                  call rdrst( as, cldtop, cldbot, zm, zint, &
                              rstflg, plonl, platl, pplon, ps, q )
               end if
            else
               if( xactive_drydep .or. xactive_emissions ) then
                  call rdrst( as, cldtop, cldbot, zm, zint, &
                              rstflg, plonl, platl, pplon, ps, &
                              q(1,1,1,1,2), precip )
               else
                  call rdrst( as, cldtop, cldbot, zm, zint, &
                              rstflg, plonl, platl, pplon, ps, q(1,1,1,1,2) )
               end if
            end if
# else
            call rdrst( as, rstflg, plonl, platl, pplon, ps, q )
# endif
            write(*,*) 'inirun: dry mass of atmosphere will be held at ', mdryatm, ' kg/m^2'
         end if
!-------------------------------------------------------------------
!       ... chem routines initializers
!-------------------------------------------------------------------
         call moz_hook_inti( plonl, platl, pplon )
         call lght_inti( plonl, platl, pplon, lght_no_prd_factor )
         call lifetime_inti( plonl, platl, pplon )
         call imp_slv_lt_inti
      end if Comp_nodes_only_1

#ifdef USE_MPI
      if( ded_io_node ) then
         call mpi_bcast( mdryatm, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, iret )
         if( iret /= MPI_SUCCESS ) then
         write(*,*) 'inirun: Bcast for mdryatm failed; code = ',iret
            call endrun
         end if
      end if
#endif

      call iniele

      end subroutine inirun

      subroutine hst_files( hstinst, hsttimav )
!-------------------------------------------------------------------
!       ... Check and setup "match" variable history files
!-------------------------------------------------------------------

      use mo_histout, only : hst_file_max, match_file_cnt
      use m_types,    only : hst_list

      implicit none

!-------------------------------------------------------------------
!       ... Dummy arguments
!-------------------------------------------------------------------
      type(hst_list), intent(inout) :: &
        hstinst(hst_file_max), &        ! names of instantaneous namelist output fields
        hsttimav(hst_file_max)          ! names of time averaged namelist output fields

!-------------------------------------------------------------------
!       ... Local variables
!-------------------------------------------------------------------
      integer :: file

!-------------------------------------------------------------------
!       ... set count of namelist history files
!-------------------------------------------------------------------
      do file = hst_file_max,1,-1
         if( hstinst(file)%list(1) /= ' ' .or. hsttimav(file)%list(1) /= ' ' ) then
            exit
         end if
      end do
      match_file_cnt = file

      end subroutine hst_files

      subroutine time_conversion( time_string, frequency, time_token, dtime, valid_timec )
!-------------------------------------------------------------------
!       ... convert character timing to integer
!-------------------------------------------------------------------

      use mo_control, only : secpday
      use mo_charutl, only : upcase

      implicit none

!-------------------------------------------------------------------
!       ... dummy arguments
!-------------------------------------------------------------------
      integer, optional, intent(in)  :: dtime
      integer, intent(out) :: frequency
      character(len=*), intent(in)  :: time_string
      character(len=*), optional, intent(out) :: time_token
      character(len=*), optional, intent(in)  :: valid_timec

!-------------------------------------------------------------------
!       ... local variables
!-------------------------------------------------------------------
      integer :: l, m
      integer :: n
      integer :: iret
      character(len=16) :: up_case      ! upper case wrk variable
      character(len=6)  :: valid_char
      logical :: case_found

      valid_char = 'SMHDWY'
      m = len_trim( time_string )
      call upcase( time_string, up_case )
!-------------------------------------------------------------------
!       ... check for frequency setting
!-------------------------------------------------------------------
      if( present( dtime ) ) then
         case_found = .true.
         select case( up_case )
            case( 'YEARLY' )
               frequency  = 365*secpday/dtime
               time_token = 'y'
            case( 'MONTHLY' )
               frequency  = 1
               time_token = 'm'
            case( 'WEEKLY' )
               frequency  = 7*secpday/dtime
               time_token = 'w'
            case( 'DAILY' )
               frequency  = secpday/dtime
               time_token = 'd'
            case( 'HOURLY' )
               frequency = 3600/dtime
               time_token = 'h'
            case default
               case_found = .false.
         end select
      else
         case_found = .false.
      end if
!-------------------------------------------------------------------
!       ... if not frequency then check for allowed strings
!-------------------------------------------------------------------
      if( .not. case_found ) then
         if( present( valid_timec ) ) then
            n = scan( valid_timec, up_case(m:m) )
         else
            n = scan( valid_char, up_case(m:m) )
         end if
         if( n > 0 ) then
            n = m - 1
            l = verify( up_case(1:n), '0123456789' )
            if( l /= 0 ) then
               write(*,*) ' '
               write(*,*) 'inirun: in the input string, ',trim(time_string), &
                          ', ',trim(time_string(1:n)),' does not contain a valid integer'
               write(*,*) ' '
               call endrun
            end if
         else
            n = verify( up_case(1:m), '0123456789' )
            if( n /= 0 ) then
               write(*,*) ' '
               write(*,*) 'inirun: input string : ',trim(time_string),' does not contain a valid integer'
               if( present( valid_timec ) ) then
                  write(*,*) '        or does not end in a time character in the set {',valid_timec,'}'
               else
                  write(*,*) '        or does not end in a time character in the set {',valid_char,'}'
               end if
               write(*,*) ' '
               call endrun
            end if
            n = m
         end if
         read(time_string(1:n),*,iostat=iret) frequency
         if( iret /= 0 ) then
            write(*,*) 'inirun: time_conversion read failed; error = ',iret
            call endrun
         end if
         if( present( dtime ) ) then
            select case( up_case(m:m) )
               case( 'S' )
                  frequency  = max( 1,abs( frequency )/dtime )
               case( 'M' )
                  frequency  = max( 1,abs( frequency )*60/dtime )
               case( 'H' )
                  frequency  = max( 1,abs( frequency )*3600/dtime )
               case( 'D' )
                  frequency  = max( 1,abs( frequency )*secpday/dtime )
               case( 'W' )
                  frequency  = max( 1,abs( frequency )*7*secpday/dtime )
               case( 'Y' )
                  frequency  = max( 1,abs( frequency )*365*secpday/dtime )
            end select
         else
            select case( up_case(m:m) )
               case( 'M' )
                  frequency  = max( 1,abs( frequency )*60 )
               case( 'H' )
                  frequency  = max( 1,abs( frequency )*3600 )
               case( 'D' )
                  frequency  = max( 1,abs( frequency )*secpday )
               case( 'W' )
                  frequency  = max( 1,abs( frequency )*7*secpday )
               case( 'Y' )
                  frequency  = max( 1,abs( frequency )*365*secpday )
            end select
         end if
         if( present( time_token ) ) then
            select case( up_case(m:m) )
               case( 'S' )
                  time_token = 's'
               case( 'M' )
                  time_token = 'm'
               case( 'H' )
                  time_token = 'h'
               case( 'D' )
                  time_token = 'd'
               case( 'W' )
                  time_token = 'w'
               case( 'Y' )
                  time_token = 'y'
               case default
                  time_token = ' '
            end select
         end if
      end if

      end subroutine time_conversion

      integer function get_sim_duration( sdate, ssec, sim_duration, tstep )
!---------------------------------------------------------------------
!	... get simulation duration in time steps
!---------------------------------------------------------------------
      
      use mo_calendar, only : diffdat
      use mo_charutl,  only : upcase
      use mo_control,  only : secpday

      implicit none

!---------------------------------------------------------------------
!	... dummy args
!---------------------------------------------------------------------
      integer, intent(in)          :: sdate                ! start date in yyyymmdd format
      integer, intent(in)          :: ssec                 ! start seconds in start date
      integer, intent(in)          :: tstep                ! model time step (s)
      character(len=*), intent(in) :: sim_duration

!---------------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------------
      integer :: m, n, wrk_date
      integer :: steps
      integer :: iret
      integer :: yrs, mnths, hrs
      real    :: days
      character(len=16) :: up_case

      yrs  = 0
      mnths = 0
      days = 0.
      hrs  = 0
      m = len_trim( sim_duration )
      call upcase( sim_duration, up_case )
      if( up_case(m:m) == 'H' ) then                                  ! hours
        read(sim_duration(1:m-1),*,iostat=iret) hrs
        if( iret /= 0 ) then
          write(*,*) 'get_sim_duration: read failed; error = ',iret
          call endrun
        end if
        steps = hrs *3600
      else if( up_case(m:m) == 'D' ) then                             ! days
        read(sim_duration(1:m-1),*,iostat=iret) steps
        if( iret /= 0 ) then
          write(*,*) 'get_sim_duration: read failed; error = ',iret
          call endrun
        end if
        steps = steps *secpday
      else if( up_case(m:m) == 'M' ) then                             ! months
        read(sim_duration(1:m-1),*,iostat=iret) mnths
        wrk_date = mod( sdate,10000 )/100 + mnths
        n        = wrk_date/12 + sdate/10000
        m        = mod( wrk_date,12 )
        wrk_date = 10000*n + 100*m + mod( sdate,100 )
        if( mod( mnths,12 ) == 0 .and. mod( sdate,10000 ) == 229 ) then
          wrk_date = wrk_date - 1
        end if
        days     = diffdat( sdate, ssec, wrk_date, ssec )
        steps    = int( days )*secpday
      else if( up_case(m:m) == 'Y' ) then                             ! years
        read(sim_duration(1:m-1),*,iostat=iret) yrs
        wrk_date = sdate + 10000*yrs
        if( mod( sdate,10000 ) == 229 ) then
          wrk_date = wrk_date - 1
        end if
        days  = diffdat( sdate, ssec, wrk_date, ssec )
        steps = int( days )*secpday
      else
        read(sim_duration(1:m),*,iostat=iret) steps
        if( iret /= 0 ) then
          write(*,*) 'get_sim_duration: read failed; error = ',iret
          call endrun
        end if
        steps = -steps
      end if

      if( steps > 0 ) then
        get_sim_duration = steps/tstep
      else
        get_sim_duration = abs(steps)
      end if

      end function get_sim_duration

      subroutine check_filespecs( local_data_path )
!---------------------------------------------------------------------
!	... check input data file specs
!---------------------------------------------------------------------

      use m_types,    only : filespec
      use mo_control, only : ic_flsp, dyn_flsp, frc_flsp
      use mo_control, only : dvel_flsp, ubc_flsp, lbc_flsp
      use mo_control, only : sulf_flsp, surf_flsp, sad_flsp
      use mo_control, only : photo_flsp, emis_flsp, dust_flsp
      use mo_control, only : photo_xs_long_flsp, photo_rsf_flsp
      use mo_control, only : photo_xs_short_flsp, photo_xs_o2_flsp
      use mo_control, only : solar_parms_flsp, woods_flsp, neckel_flsp
      use mo_control, only : rstflg

      implicit none

!---------------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------------
      character(len=*), intent(in) :: local_data_path

!---------------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------------
      integer, parameter :: nfiles = 10
      character(len=14), parameter :: sub_dir(nfiles) = &
             (/ 'dvel          ', 'ub            ', 'lb            ', &
                'sulf          ', 'surf          ', 'emis          ', &
                'phot          ', 'dust          ', 'sad           ', &
                'extfrc        ' /)
      integer :: slen
      integer :: flen
      integer :: m
      logical :: exists
      character(len=168) :: wrk_filepath
      character(len=128) :: filepath

!---------------------------------------------------------------------
!	... check local data path
!---------------------------------------------------------------------
      filepath = trim( local_data_path )
      flen     = len_trim( filepath )
      if( flen > 0 ) then
         if( filepath(flen:flen) /= '/' ) then
           flen = flen + 1
           filepath(flen:) = '/'
         end if
      end if

is_initial_run : &
      if( rstflg == 0 ) then
!---------------------------------------------------------------------
!	... initial condition filespec
!---------------------------------------------------------------------
         if( ic_flsp%local_path == 'NONE' ) then
            if( flen > 0 ) then
               ic_flsp%local_path = trim( filepath ) // 'ic/'
            else
               write(*,*) 'check_filespecs: ic_flsp local path not specified'
               call endrun
            end if
         end if
#if( !defined(IFORT) )
         slen = len_trim( ic_flsp%local_path )
         inquire( file=ic_flsp%local_path(:slen-1), exist=exists )
         if( .not. exists ) then
            write(*,*) 'check_filespecs: ic_flsp local path ',ic_flsp%local_path(:slen-1),' does not exist'
            call endrun
         end if
#endif
!---------------------------------------------------------------------
!	... dynamics filespec
!---------------------------------------------------------------------
         if( dyn_flsp%local_path(1:4) == 'NONE' ) then
            if( flen > 0 ) then
               dyn_flsp%local_path = trim( filepath ) // 'dyn/'
            else
               write(*,*) 'check_filespecs: dyn_flsp local path not specified'
               call endrun
            end if
         end if
#if( !defined(IFORT) )
         slen = len_trim( dyn_flsp%local_path )
         inquire( file=dyn_flsp%local_path(:slen-1), exist=exists )
         if( .not. exists ) then
            write(*,*) 'check_filespecs: dyn_flsp local path ',dyn_flsp%local_path(:slen-1),' does not exist'
            call endrun
         end if
#endif
      end if is_initial_run

files_loop : &
      do m = 1,nfiles
         select case( m )
            case(1)
               wrk_filepath = dvel_flsp%local_path
            case(2)
               wrk_filepath = ubc_flsp%local_path
            case(3)
               wrk_filepath = lbc_flsp%local_path
            case(4)
               wrk_filepath = sulf_flsp%local_path
            case(5)
               wrk_filepath = surf_flsp%local_path
            case(6)
               wrk_filepath = emis_flsp%local_path
            case(7)
               wrk_filepath = photo_flsp%local_path
            case(8)
               wrk_filepath = dust_flsp%local_path
            case(9)
               wrk_filepath = sad_flsp%local_path
            case(10)
               wrk_filepath = frc_flsp%local_path
         end select
!---------------------------------------------------------------------
!	... filespec
!---------------------------------------------------------------------
         if( wrk_filepath == 'NONE' ) then
            if( flen > 0 ) then
               wrk_filepath = trim( filepath ) // trim(sub_dir(m)) // '/'
            else
               write(*,*) 'check_filespecs: ',trim(sub_dir(m)),'_flsp local path not specified'
               call endrun
            end if
         end if
#if( !defined(IFORT) )
         slen = len_trim( wrk_filepath )
         inquire( file=wrk_filepath(:slen-1), exist=exists )
         if( .not. exists ) then
            write(*,*) 'check_filespecs: ',trim(sub_dir(m)),'_flsp local path ',wrk_filepath(:slen-1),' does not exist'
            call endrun
         end if
#endif
         select case( m )
            case(1)
               dvel_flsp%local_path = wrk_filepath
            case(2)
               ubc_flsp%local_path = wrk_filepath
            case(3)
               lbc_flsp%local_path = wrk_filepath
            case(4)
               sulf_flsp%local_path = wrk_filepath
            case(5)
               surf_flsp%local_path = wrk_filepath
            case(6)
               emis_flsp%local_path = wrk_filepath
            case(7)
               photo_flsp%local_path = wrk_filepath
            case(8)
               dust_flsp%local_path = wrk_filepath
            case(9)
               sad_flsp%local_path = wrk_filepath
            case(10)
               frc_flsp%local_path = wrk_filepath
         end select
      end do files_loop

      if( photo_xs_long_flsp%local_path == 'NONE' ) then
         photo_xs_long_flsp%local_path = photo_flsp%local_path
      end if
      if( photo_xs_long_flsp%remote_path == 'NONE' ) then
         photo_xs_long_flsp%remote_path = photo_flsp%remote_path
      end if
      if( photo_rsf_flsp%local_path == 'NONE' ) then
         photo_rsf_flsp%local_path = photo_flsp%local_path
      end if
      if( photo_rsf_flsp%remote_path == 'NONE' ) then
         photo_rsf_flsp%remote_path = photo_flsp%remote_path
      end if

      if( photo_xs_short_flsp%local_path == 'NONE' ) then
         photo_xs_short_flsp%local_path = photo_flsp%local_path
      end if
      if( photo_xs_short_flsp%remote_path == 'NONE' ) then
         photo_xs_short_flsp%remote_path = photo_flsp%remote_path
      end if
      if( photo_xs_o2_flsp%local_path == 'NONE' ) then
         photo_xs_o2_flsp%local_path = photo_flsp%local_path
      end if
      if( photo_xs_o2_flsp%remote_path == 'NONE' ) then
         photo_xs_o2_flsp%remote_path = photo_flsp%remote_path
      end if

      if( solar_parms_flsp%local_path == 'NONE' ) then
         solar_parms_flsp%local_path = photo_flsp%local_path
      end if
      if( solar_parms_flsp%remote_path == 'NONE' ) then
         solar_parms_flsp%remote_path = photo_flsp%remote_path
      end if
      if( woods_flsp%local_path == 'NONE' ) then
         woods_flsp%local_path = photo_flsp%local_path
      end if
      if( woods_flsp%remote_path == 'NONE' ) then
         woods_flsp%remote_path = photo_flsp%remote_path
      end if
      if( neckel_flsp%local_path == 'NONE' ) then
         neckel_flsp%local_path = photo_flsp%local_path
      end if
      if( neckel_flsp%remote_path == 'NONE' ) then
         neckel_flsp%remote_path = photo_flsp%remote_path
      end if

      end subroutine check_filespecs

      end module mo_inirun
