
      module mo_mass

      use mo_grid, only : plat, plev

      implicit none

      save

      real :: &
        cwava, &     ! 1./(gravit*plon)
        gw(plat), &  ! gauss weights (these sum to 2)
        etamid(plev) ! hybrid coordinate a layer midpoints

      contains

      subroutine inimass( gravit, xgw, xetamid )
!-----------------------------------------------------------------------
! 	... initialize module
!-----------------------------------------------------------------------

      use mo_grid, only : plat, plev, plon

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      real, intent(in) :: &
        gravit, &      ! gravitational acceleration (m/s^2)
        xgw(plat), &   ! gauss weights
        xetamid(plev)  ! hybrid coord at layer midpoints

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i, k

      cwava = 1./(real(plon)*gravit)

      do i = 1,plat
         gw(i) = xgw(i)
      end do

      do k = 1,plev
         etamid(k) = xetamid(k)
      end do

      end subroutine inimass

      subroutine qmassa( lat, ncnst, q3, pdel, hw1lat, plonl )
!-----------------------------------------------------------------------
! 	... calculate area average mass of tracer fields.
!           input concentration units are wet mixing ratio (kg tracer / kg moist air).
!           output masses are in kg/m^2.
!-----------------------------------------------------------------------

      use mo_grid, only : plev
      use mo_mpi,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments:
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &                  ! latitude index
        ncnst, &                ! number of fields
        plonl                   ! lon tile dim

      real, intent(in) :: &
        q3(plonl,plev,ncnst), & ! advected species (wet mixing ratio)
        pdel(plonl,plev)        ! pressure difference across model layer

      real, intent(out) :: &
        hw1lat(plonl,ncnst)     ! latitude contribution to mass of each specie

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i, k, m
      real    :: const0

      const0 = .5 * cwava * gw(lat+base_lat)
      do m = 1,ncnst
         hw1lat(:,m) = 0.
         do k = 1,plev
            hw1lat(:,m) = hw1lat(:,m) + q3(:,k,m)*pdel(:,k)
         end do
         hw1lat(:,m) = const0 * hw1lat(:,m)
      end do

      end subroutine qmassa

      subroutine qmassd( lat, ncnst, q1, q2, pdel, hwn, plonl )
!-----------------------------------------------------------------------
! 	... compute latitude contribution to integral of q2*etamid*|q2 - q1|
!           input concentration units are wet mixing ratio (kg tracer / kg moist air).
!           output units are in kg/m^2.
!-----------------------------------------------------------------------

      use mo_grid, only : plev
      use mo_mpi,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments:
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &                  ! latitude index
        ncnst, &                ! number of fields
        plonl                   ! lon tile dim

      real, dimension(plonl,plev,ncnst), intent(in) :: &
        q1, &                ! tracer before slt
        q2                   ! tracer after slt
      real, intent(in) :: &
        pdel(plonl,plev)     ! pressure difference across model layers

      real, intent(out) :: &
        hwn(plonl,ncnst)     ! latitude contribution to integral

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i, k, m, jglob
      real    :: hwava

      jglob = lat + base_lat
      do m = 1,ncnst
         do i = 1,plonl
            hwava = 0.
            do k = 1,plev
               hwava = hwava + q2(i,k,m)*etamid(k) &
                       * abs( q1(i,k,m) - q2(i,k,m) ) * pdel(i,k)
            end do
            hwn(i,m) = cwava * gw(jglob) * hwava * .5
         end do
      end do

      end subroutine qmassd

      subroutine gatmass( lat, ps, tmassj, plonl )
!-----------------------------------------------------------------------
! 	... compute the contribution to the global average of the total
!           atmospheric mass from a latitude slice.
!-----------------------------------------------------------------------

      use mo_mpi,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &          ! latitude index
        plonl           ! lon tile dim

      real, intent(in) :: &
        ps(plonl)       ! pressure in pascals

      real, intent(out) :: &
        tmassj(plonl)   ! total mass in kg/m^2

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: jglob

      jglob          = lat + base_lat
      tmassj(:plonl) = ps(:plonl) * cwava * gw(jglob) * .5

      end subroutine gatmass

      subroutine gamdry( lat, ncnst, as, sh, pdel, massj, plonl )
!-----------------------------------------------------------------------
! 	... calculate global average mass of tracer fields.
!           input concentration units are dry mixing ratio (kg tracer / kg dry air).
!           output masses are in kg/m^2.
!-----------------------------------------------------------------------

      use mo_grid, only : plev
      use mo_mpi,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &                  ! latitude index
        ncnst, &                ! number of fields
        plonl                   ! lon tile dim

      real, intent(in) :: &
        as(plonl,plev,ncnst), & ! advected species (dry mixing ratio)
        sh(plonl,plev), &       ! specific humidity
        pdel(plonl,plev)        ! pressure difference across model layer

      real, intent(out) :: &
        massj(plonl,ncnst)      ! latitude contribution to mass of each specie

!-----------------------------------------------------------------------
! 	... local variables:
!-----------------------------------------------------------------------
      integer :: i, k, m
      real    :: const0

      const0 = .5 * cwava * gw(lat+base_lat)
      do m = 1,ncnst
         massj(:,m) = 0.
         do k = 1,plev
            massj(:,m) = massj(:,m) + as(:,k,m)*(1. - sh(:,k))*pdel(:,k)
         end do
         massj(:,m) = const0 * massj(:,m)
      end do

      end subroutine gamdry

      subroutine gamdry2( lat, ncnst, as, pdel, massj, plonl )
!-----------------------------------------------------------------------
! 	... calculate global average mass of tracer fields.
!           input concentration units are dry mixing ratio (kg tracer / kg dry air).
!           output masses are in kg/m^2.
!           this routine is the same as gamdry except that the pdel
!           argument assumes a dry atmosphere.
!-----------------------------------------------------------------------

      use mo_grid, only : plev
      use mo_mpi,  only : base_lat

      implicit none

!-----------------------------------------------------------------------
! 	... input arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &                  ! latitude index
        ncnst, &                ! number of fields
        plonl                   ! lon tile dim

      real, intent(in) :: &
        as(plonl,plev,ncnst), & ! advected species (dry mixing ratio)
        pdel(plonl,plev)        ! pressure difference across model layer in a dry atmosphere.

      real, intent(out) :: &
        massj(plonl,ncnst)      ! latitude contribution to mass of each specie

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i, k, m
      real    :: const0

      const0 = .5 * cwava * gw(lat+base_lat)
      do m = 1,ncnst
         massj(:,m) = 0.
         do k = 1,plev
            massj(:,m) = massj(:,m) + as(:,k,m)*pdel(:,k)
         end do
         massj(:,m) = const0 * massj(:,m)
      end do

      end subroutine gamdry2

      subroutine spc_gamdry( lat, ncnst, as, q, pdel, &
			     massj, indx, plonl )
!-----------------------------------------------------------------------
! 	... calculate global average mass of tracer fields.
!           input concentration units are dry mixing ratio (kg tracer / kg dry air).
!           output masses are in kg/m^2 using a wet mixing ratio
!-----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------------
! 	... input arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &                  ! latitude index
        plonl, &                ! lon tile dim
        ncnst, &                ! number of fields
        indx                    ! constituent index

      real, intent(in) :: &
        as(plonl,plev,ncnst), & ! advected species (dry mixing ratio)
        pdel(plonl,plev), &     ! pressure difference across model layer
        q(plonl,plev)           ! specific humidity

      real, intent(out) :: &
        massj(plonl,plev)       ! latitude contribution to mass of each layer

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: i, k

      do k = 1,plev
         massj(:,k) = cwava * as(:,k,indx)*(1. - q(:,k)) *  pdel(:,k)
      end do

      end subroutine spc_gamdry

      subroutine spc_gamwet( lat, ncnst, as, pdel, massj, &
			     indx, plonl )
!-----------------------------------------------------------------------
! 	... calculate global average mass of tracer fields.
!           input concentration units are wet mixing ratio (kg tracer / kg wet air).
!           output masses are in kg/m^2
!-----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &                  ! latitude index
        plonl, &                ! lon tile dim
        ncnst, &                ! number of fields
        indx                    ! constituent index

      real, intent(in) :: &
        as(plonl,plev,ncnst), & ! advected species (dry mixing ratio)
        pdel(plonl,plev)        ! pressure difference across model layer

      real, intent(out) :: &
        massj(plonl,plev)       ! latitude contribution to mass of each layer

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: k

      do k = 1,plev
         massj(:,k) = cwava * as(:,k,indx) * pdel(:,k)
      end do

      end subroutine spc_gamwet

      end module mo_mass
