#include <params.h>

module mo_offline_sources
!---------------------------------------------------------------------
! 	... offline sources module
!---------------------------------------------------------------------

  use mo_grid,    only : plon, plev, plevp, plat
  use mo_offline_data

  implicit none

  private
  public :: init_offline_sources, advance_offline_sources
  public :: mxsrcsflds, srcs_offline, srcs_names, srcs_data

  save

  type(offline_data) :: srcs_data

  integer :: mxsrcsflds

  real, pointer :: srcs_offline(:,:,:,:,:)           

  character(len=8), pointer :: srcs_names(:)

contains

  subroutine init_offline_sources( icdate, icsec, offset, lun, mxnbrsrcs, &
                                   offl_srcs, remove, sourcenm, sourcesc, &
                                   plonl, platl, pplon )
!---------------------------------------------------------------------------
! 	... initialize offline sources arrays, and a few other variables.
!           the sources fields will be linearly interpolated to the initial time (icdate,icsec).
!---------------------------------------------------------------------------

    use mo_control,     only : srcs_flsp
    use mo_chem_utls,   only : get_extfrc_ndx

    implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in) :: mxnbrsrcs
    integer, intent(in) :: plonl
    integer, intent(in) :: platl
    integer, intent(in) :: pplon
    integer, intent(in) :: icdate              ! date of initial conditions in yymmdd format
    integer, intent(in) :: icsec               ! seconds relative to date for initial conditions
    integer, intent(in) :: offset              ! add offset (seconds) to dates read from headers
    integer, intent(in) :: lun                 ! unit number for input
    real, intent(in)    :: sourcesc(*)         ! scale factor to convert units in input files to the internal units

    character(len=32), intent(in)   :: sourcenm(*)     ! names of source fields in offline file
    character(len=8), intent(inout) :: offl_srcs(*)    ! names of offline source invariants

    logical, intent(in) :: remove              ! true => remove old files from local disk

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
    integer              :: astat
    integer              :: m
    integer              :: ndx
    character(len=8)     :: temp_string
    logical, allocatable :: is_extfrc(:)

!---------------------------------------------------------------------------
! 	... check that offline sources are simulation external forcings
!---------------------------------------------------------------------------
    allocate( is_extfrc(mxnbrsrcs),stat=astat )
    if( astat /= 0 ) then
       write(*,*) 'init_offline_sources: failed to allocate is_extfrc; error = ',astat
       call endrun
    end if
    is_extfrc(:) = .false.
    do m = 1,mxnbrsrcs
       if( len_trim( offl_srcs(m) ) > 0 ) then
          ndx = get_extfrc_ndx( trim(offl_srcs(m)) )
          is_extfrc(m) = ndx > 0
       else
          exit
       end if
    end do

    ndx = 0
    do m = 1,mxnbrsrcs
       if( is_extfrc(m) ) then
          ndx = ndx + 1
          if( ndx /= m ) then
             offl_srcs(ndx) = offl_srcs(m)
          end if
       end if
    end do

    if( ndx < mxnbrsrcs ) then
       offl_srcs(ndx+1:mxnbrsrcs) = ' '
    end if

    deallocate( is_extfrc )

    call init_offline_data( icdate, icsec, offset, lun, mxnbrsrcs, &
                            offl_srcs, remove, sourcenm, sourcesc, srcs_data, &
                            mxsrcsflds, srcs_offline, srcs_names, srcs_flsp, plonl, &
                            platl, pplon )

  end subroutine init_offline_sources

  subroutine advance_offline_sources( ncdate, ncsec, plonl, platl, pplon )
!---------------------------------------------------------------------------
! advance the offline sources data -- interpolates, reads more data,
!                                     gets another file if needed 
!                                     time interpolates to ncdate,ncsec
!---------------------------------------------------------------------------

    use mo_control,    only : srcs_flsp

    implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
    integer, intent(in) :: ncdate, ncsec, plonl, platl, pplon

!---------------------------------------------------------------------------
! if there are no offline offline data fields then proceed  no farther
!---------------------------------------------------------------------------
    if( srcs_data%mxflds == 0 ) then 
       return
    endif

!---------------------------------------------------------------------------
! Check that data needed for the specified time is loaded.
! Read the appropriate data, get another file if needed
!---------------------------------------------------------------------------
    call chk_offline_data( ncdate, ncsec, plonl, platl, pplon, srcs_data, srcs_flsp )

!---------------------------------------------------------------------------
! time interpolate to ncdate,ncsec
!---------------------------------------------------------------------------
    call interpflds( ncdate, ncsec, plonl, platl, pplon, srcs_data, srcs_offline )

  end subroutine advance_offline_sources

end module mo_offline_sources
