! lke (April 2007) - took out h2so4_aer restart info

#include <params.h>

      module mo_restart
!-----------------------------------------------------------------------
! 	... Restart info file data
!-----------------------------------------------------------------------

      use mo_grid, only : plon, plev, plevp, plat, pcnst

      implicit none

      private
      public :: inirest, rdrst, wrrst, qrsti, qrstc, prtrst

      save

      integer :: &
        lunrst0, &        ! unit number for restart info file
        lunrst1, &        ! unit number for restart data file
        rstrrt, &         ! retention period (days) for restart file
        rstfrq, &         ! output frequency (steps) for restart files
        ncid, &           ! netcdf file id
        lon_id, &         ! dimension id
        lev_id, &         ! dimension id
        ilev_id, &        ! dimension id
        lat_id            ! dimension id

      logical :: &
        async, &          ! t => dispose to remote device asynchronously
        rmrst, &          ! t => remove local copy after disposing
        stratchem, &      ! t => stratospheric chemistry
        tropchem, &       ! t => tropospheric chemistry
        radon, &          ! t => calc radon surface fluxes and apply decay 
        diconvccm, &      ! t => diagnose CCM convection
        dicldphys         ! t => diagnose cloud physics

      character(len=1)   :: &
        rst_time_token = ' '
      character(len=32)  :: &
        case_name
      character(len=128) :: &
        lpthrst0, &       ! local pathname for restart info file
        rpthrst0, &       ! remote pathname for restart info file
        lpthrst1, &       ! local pathname for restart data
        rpthrst1          ! remote pathname for restart data
      character(len=128) :: &
        blpthrst0, &       ! local pathname for branch run restart info file
        brpthrst0, &       ! remote pathname for branch run restart info file
        blpthrst1, &       ! local pathname for branch run restart data
        brpthrst1          ! remote pathname for branch run restart data
      
      contains

      subroutine inirest( rstflg, iunrst0, iunrst1, nrt, xrstfrq, &
                          rrootd, case, xasync, xrmrst, &
                          xstratchem, xtropchem, xdiconvccm, xdicldphys, date, &
                          sec, itimrst, ntshst, partial_ta, hst_flsp, &
                          rst_date, rst_datesec, rst_days, rst_secs, rst_lnhstf, &
                          brest0_flsp, xrst_time_token, rhst_lpath, rhst_rpath )
!-----------------------------------------------------------------------
! 	... Initialize module,  acquire and read restart file.  Reset namelist parameters.
!-----------------------------------------------------------------------

      use mo_mpi
      use mo_file_utils, only : attach
      use mo_control,    only : rest0_flsp, rest1_flsp, dyn_flsp
      use mo_histout,    only : hfile, sim_file_cnt, hst_file_max
      use mo_charutl,    only : lastsl, glc, upcase
      use m_types,       only : filespec
      use mo_control,    only : constit_flsp, srcs_flsp

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        rstflg, &            ! restart flag( 0 == initial run, 1 == restart run, 2 == branch run )
        iunrst0, &           ! unit number for restart info file
        iunrst1, &           ! unit number for restart data file
        nrt, &               ! retention period for restart file
        xrstfrq              ! output frequency for restart files
      integer, dimension(hst_file_max), intent(inout) :: &
        rst_date, &          ! date retrieved from rstrt file (yyyymmdd) 
        rst_datesec, &       ! seconds in date retrieved from rstrt file (s) 
        rst_days, &          ! simulation elapsed days retrieved from rstrt file
        rst_secs             ! seconds in simulation elapsed days retrieved from rstrt file

      character(len=*), intent(inout) :: rhst_lpath(hst_file_max)
      character(len=*), intent(inout) :: rhst_rpath(hst_file_max)
      character(len=80), dimension(hst_file_max), intent(inout) :: &
        rst_lnhstf           ! path of first hst file for this case
      character(len=16), intent(in) :: &
        rrootd               ! Default root directory for files written to remote storage.
      character(len=32), intent(in) :: &
        case                 ! case name from the environment variable CASE
      character(len=1), intent(in) :: &
        xrst_time_token
      type(filespec), dimension(hst_file_max), intent(inout) :: &
        hst_flsp
      type(filespec), intent(in) :: &
        brest0_flsp          ! restart info file for branch run

      logical, intent(in) :: &
        xasync, &            ! t => dispose to remote device asynchronously
        xrmrst, &            ! t => remove restart data files after disposing
!-----------------------------------------------------------------------
! 	... the following flags are passed to the restart module so that it will
!           know to call the corresponding read and write restart file functions
!           for the parameterization module.
!-----------------------------------------------------------------------
        xstratchem, &        ! t => stratospheric chemistry
        xtropchem, &         ! t => tropospheric chemistry
        xdiconvccm, &        ! t => diagnose CCM convection
        xdicldphys           ! t => cloud fraction calcs for tropospheric chem

      integer, intent(inout) :: &
        date, &      ! date of restart data in yymmdd format
        sec          ! seconds relative to date
      integer, dimension(hst_file_max) ,intent(inout) :: &
        ntshst       ! number of time samples on history file
      integer, intent(out) :: &
        itimrst      ! time index of restart.  (not used on an initial run)
      logical, dimension(hst_file_max), intent(inout) :: &
        partial_ta   ! partial time averaged history file flag

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: &
        istat, &        ! return status
        indx1, &        ! character index in string
        indx2, &        ! character index in string
        ios, &          ! file operation status
        lun_loc, &      ! for COS blocked files, attach can modify lun
        hst_file_cnt, & ! local simulation history file count
        file            ! file index
      integer :: &
        idummy(hst_file_max)      ! wrk variable

      logical :: &
        ldum            ! dummy
      logical :: &
        ldummy(hst_file_max)      ! wrk variable

      character(len=168) ::  lpthdyn                       ! local path for dynamics file containing (date,sec).
      character(len=168) ::  rpthdyn                       ! remote path for dynamics file containing (date,sec).
      character(len=168) ::  lpthhst(hst_file_max)         ! local path for history file
      character(len=168) ::  rpthhst(hst_file_max)         ! remote path for history file
      character(len=168) ::  cdummy(hst_file_max)          ! dummy char variable
      character(len=168) ::  wrkflsp                       ! temp path for restart file
      character(len=128) :: &
        lrst0, &        ! local pathname for restart info file
        rrst0, &        ! remote pathname for restart info file
        blrst0, &       ! local pathname for branch run restart info file
        brrst0, &       ! remote pathname for branch run restart info file
        lrst1, &        ! local pathname for restart data
        rrst1, &        ! remote pathname for restart data
        blrst1, &       ! temp remote pathname for branch run restart data
        brrst1, &       ! temp remote pathname for branch run restart data
        wrkrst          ! temp remote filespec
      character(len=32) :: &
        homedir, &
        logname, &
        ctemp
      character(len=168) :: &
        lpthcnst, &      ! local path for offline constits file
        rpthcnst         ! remote path for offline constits file 
      character(len=168) :: &
        lpthsrcs, &      ! local path for offline sources file
        rpthsrcs         ! remote path for offline sources file 

!-----------------------------------------------------------------------
!	... function declarations
!-----------------------------------------------------------------------
      integer :: fgetenv

!-----------------------------------------------------------------------
!     	... set default filenames if values not supplied via namelist
!-----------------------------------------------------------------------
      case_name = case
      rst_time_token = xrst_time_token
      lrst0 = trim( rest0_flsp%local_path ) // trim( rest0_flsp%nl_filename )
      if( rest0_flsp%remote_path(1:1) == ' ' ) then
         rest0_flsp%remote_path = '/' // rrootd(:GLC(rrootd)) // '/mozart/' // case(:GLC(case)) // '/rest/'
      end if
      rrst0 = trim( rest0_flsp%remote_path ) // trim( rest0_flsp%nl_filename )
      lrst1 = trim( rest1_flsp%local_path ) // trim( rest1_flsp%nl_filename )
      if( rest1_flsp%remote_path(1:1) == ' ' ) then
         rest1_flsp%remote_path = '/' // rrootd(:GLC(rrootd)) // '/mozart/' // case(:GLC(case)) // '/rest/'
      end if
      rrst1 = trim( rest1_flsp%remote_path ) // trim( rest1_flsp%nl_filename )
      if( rstflg == 2 ) then
         blrst0 = trim( brest0_flsp%local_path ) // trim( brest0_flsp%nl_filename )
         brrst0 = trim( brest0_flsp%remote_path ) // trim( brest0_flsp%nl_filename )
      end if
      if( masternode ) then
         write(*,*) 'inirest: lrst0 = ',trim(lrst0)
         write(*,*) 'inirest: rrst0 = ',trim(rrst0)
         if( rstflg == 2 ) then
            write(*,*) 'inirest: blrst0 = ',blrst0(:glc(blrst0))
            write(*,*) 'inirest: brrst0 = ',brrst0(:glc(brrst0))
         end if
         write(*,*) 'inirest: lrst1 = ',trim(lrst1)
         write(*,*) 'inirest: rrst1 = ',trim(rrst1)
      end if
Restarting : &
      if( rstflg >= 1 ) then
!-----------------------------------------------------------------------
!	... get restart info file
!-----------------------------------------------------------------------
         if( io_node ) then
            istat = fgetenv( 'HOME', homedir )
            if( istat /= 0 ) then
               write(*,*) 'inirest: failed to get home directory; error = ',istat
               call endrun
            end if
            istat = fgetenv( 'LOGNAME', logname )
            if( istat /= 0 ) then
               write(*,*) 'inirest: failed to get user logname; error = ',istat
               call endrun
            end if
            lun_loc = iunrst0
            wrkflsp = trim( homedir) // '/' // trim( case_name) // '-rpntr'
            write(*,*) ' '
            write(*,*) 'inirest: will open and read file = ', trim( wrkflsp )
            write(*,*) ' '
            open( unit = lun_loc, file = trim(wrkflsp), iostat=istat )
            if( istat /= 0 ) then
               write(*,*) 'inirest: failed to open ' // trim(wrkflsp) // '; error = ',istat
               call endrun
            end if
            read(lun_loc,'(a)') wrkflsp
            close( lun_loc )
            if( rstflg == 1 ) then
	       call upcase( logname, ctemp )
               logname = ctemp
               wrkrst = '/' // trim( logname ) // '/mozart/' // trim( case_name ) // '/rest/'
               wrkrst = trim( wrkrst ) // wrkflsp(lastsl(wrkflsp)+1:)
               istat   = attach( trim(wrkflsp), trim(wrkrst), lun_loc, .false., ldum )
            else if( rstflg == 2 ) then
               istat   = attach( trim(blrst0), trim(brrst0), lun_loc, .false., ldum )
            end if
            if( istat /= 0 ) then
               write(*,*) 'inirest: attach returned ', istat
               call endrun
            end if
         end if
#ifdef USE_MPI
         call mpi_barrier( MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'inirest: Barrier failed; code = ',istat
            call endrun
         end if
         call mpi_bcast( wrkflsp, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'inirest: Bcast failed for wrkflsp; error = ',istat
            call endrun
         end if
#endif
         if( .not. io_node ) then
            if( rstflg == 1 ) then
               open( unit = iunrst0, &
                     file = trim( wrkflsp ), &
                     status = 'OLD', &
                     iostat = istat )
               if( istat /= 0 ) then
                  write(*,*) 'inirest: Failed to open file ',trim( wrkflsp )
                  call endrun
               end if
            else if( rstflg == 2 ) then
               open( unit = iunrst0, &
                     file = trim( blrst0 ), &
                     status = 'OLD', &
                     iostat = istat )
               if( istat /= 0 ) then
                  write(*,*) 'inirest: Failed to open file ',trim( blrst0 )
                  call endrun
               end if
            end if
         end if
!-----------------------------------------------------------------------
!     	... read restart info file
!-----------------------------------------------------------------------
         read(iunrst0,'(4i10)',iostat=istat) date, sec, itimrst, hst_file_cnt
         if( istat /= 0 ) then
            write(*,*) 'inirest: Failed to read date,sec,itimrst,hst_file_cnt; error = ',istat
	    call endrun
	 end if
	 if( rstflg == 1 ) then
            read(iunrst0,'(10i10)',iostat=istat) ntshst(:hst_file_cnt)
	 else if( rstflg == 2 ) then
            read(iunrst0,'(10i10)',iostat=istat) idummy(:hst_file_cnt)
	 end if
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to ntshst; error = ',istat
	    call endrun
	 end if
	 if( rstflg == 1 ) then
            read(iunrst0,*,iostat=ios) partial_ta(:hst_file_cnt)
	 else if( rstflg == 2 ) then
            read(iunrst0,*,iostat=ios) ldummy(:hst_file_cnt)
	 end if
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read partial_ta; error = ',istat
	    call endrun
	 end if
	 do file = 1,hst_file_cnt
	    if( rstflg == 1 ) then
               read(iunrst0,'(4i10)',iostat=istat) rst_date(file), rst_datesec(file), &
                                                   rst_days(file), rst_secs(file)
	    else if( rstflg == 2 ) then
               read(iunrst0,'(4i10)',iostat=istat) idummy(1:4)
	    end if
	    if( istat /= 0 ) then
	       write(*,*) 'inirest: Failed to read restart date,datesec,days, secs for file ',file,'; error = ',istat
              call endrun
            end if
            if( rstflg == 1 ) then
               read(iunrst0,'(a80)',iostat=istat) rst_lnhstf(file)
            else if( rstflg == 2 ) then
               read(iunrst0,'(a80)',iostat=istat) cdummy(1)
            end if
            if( istat /= 0 ) then
               write(*,*) 'inirest: Failed to read lnhstf; error = ',istat
               call endrun
            end if
         end do
         read(iunrst0,'(a168)',iostat=istat) lpthdyn
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read lpthdyn; error = ',istat
	    call endrun
	 end if
         read(iunrst0,'(a168)',iostat=istat) rpthdyn
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read rpthdyn; error = ',istat
	    call endrun
	 end if
         read(iunrst0,'(a168)',iostat=istat) lpthcnst
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read lpthcnst; error = ',istat
	    call endrun
	 end if
         read(iunrst0,'(a168)',iostat=istat) rpthcnst
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read rpthcnst; error = ',istat
	    call endrun
	 end if

         read(iunrst0,'(a168)',iostat=istat) lpthsrcs
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read lpthsrcs; error = ',istat
	    call endrun
	 end if
         read(iunrst0,'(a168)',iostat=istat) rpthsrcs
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read rpthsrcs; error = ',istat
	    call endrun
	 end if
	 if( rstflg == 1 ) then
            read(iunrst0,'(a168)',iostat=istat) lpthhst(:hst_file_cnt)
	 else if( rstflg == 2 ) then
            read(iunrst0,'(a168)',iostat=istat) cdummy(:hst_file_cnt)
	 end if
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read lpthhst; error = ',istat
	    call endrun
	 end if
	 if( rstflg == 1 ) then
            read(iunrst0,'(a168)',iostat=istat) rpthhst(:hst_file_cnt)
	 else if( rstflg == 2 ) then
            read(iunrst0,'(a168)',iostat=istat) cdummy(:hst_file_cnt)
	 end if
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read rpthhst; error = ',istat
	    call endrun
	 end if
	 if( rstflg == 1 ) then
            read(iunrst0,'(a128)',iostat=istat) lrst1
	 else if( rstflg == 2 ) then
            read(iunrst0,'(a128)',iostat=istat) blrst1
	 end if
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read local restart data filespec; error = ',istat
	    call endrun
	 end if
	 if( rstflg == 1 ) then
            read(iunrst0,'(a128)',iostat=istat) rrst1
	 else if( rstflg == 2 ) then
            read(iunrst0,'(a128)',iostat=istat) brrst1
	 end if
	 if( istat /= 0 ) then
	    write(*,*) 'inirest: Failed to read remote restart data filespec; error = ',istat
	    call endrun
	 end if
	 do file = 1,hst_file_cnt
	    if( partial_ta(file) ) then
	       if( rstflg == 1 ) then
                  read(iunrst0,'(a168)',iostat=istat) rhst_lpath(file)
               else if( rstflg == 2 ) then
                  read(iunrst0,'(a168)',iostat=istat) cdummy(file)
               end if
	       if( istat /= 0 ) then
	          write(*,*) 'inirest: Failed to read local history restart filespec; error = ',istat
	          call endrun
	       end if
	       if( rstflg == 1 ) then
                  read(iunrst0,'(a168)',iostat=istat) rhst_rpath(file)
               else if( rstflg == 2 ) then
                  read(iunrst0,'(a168)',iostat=istat) cdummy(file)
               end if
	       if( istat /= 0 ) then
	          write(*,*) 'inirest: Failed to read remote history restart filespec; error = ',istat
	          call endrun
	       end if
            end if
	 end do
!-----------------------------------------------------------------------
!     	... save file/pathnames to global file/path arrays
!-----------------------------------------------------------------------
         dyn_flsp%nl_filename = lpthdyn(lastsl(lpthdyn)+1:len_trim(lpthdyn))
         if( dyn_flsp%nl_filename /= rpthdyn(lastsl(rpthdyn)+1:len_trim(rpthdyn)) ) then
            write(*,*) 'inirest: ERROR, Local and remote dynamics filenames differ in file ',trim(lrst0)
            write(*,*) 'local = ',trim( dyn_flsp%nl_filename ), &
                       ', remote = ',rpthdyn(lastsl(rpthdyn)+1:len_trim(rpthdyn))
            call endrun
         end if
         dyn_flsp%local_path  = lpthdyn(1:lastsl(lpthdyn))
         dyn_flsp%remote_path = rpthdyn(1:lastsl(rpthdyn))

         constit_flsp%nl_filename = lpthcnst(lastsl(lpthcnst)+1:len_trim(lpthcnst))
         if( constit_flsp%nl_filename /= rpthcnst(lastsl(rpthcnst)+1:len_trim(rpthcnst)) ) then
            write(*,*) 'inirest: ERROR, Local and remote offline constituents filenames differ in file ',trim(lrst0)
            write(*,*) 'local = ',trim( constit_flsp%nl_filename ), &
                       ', remote = ',rpthcnst(lastsl(rpthcnst)+1:len_trim(rpthcnst))
            call endrun
         end if
         constit_flsp%local_path  = lpthcnst(1:lastsl(lpthcnst))
         constit_flsp%remote_path = rpthcnst(1:lastsl(rpthcnst))

         srcs_flsp%nl_filename = lpthsrcs(lastsl(lpthsrcs)+1:len_trim(lpthsrcs))
         if( srcs_flsp%nl_filename /= rpthsrcs(lastsl(rpthsrcs)+1:len_trim(rpthsrcs)) ) then
            write(*,*) 'inirest: ERROR, Local and remote offline sources filenames differ in file ',trim(lrst0)
            write(*,*) 'local = ',trim( srcs_flsp%nl_filename ), &
                       ', remote = ',rpthsrcs(lastsl(rpthsrcs)+1:len_trim(rpthsrcs))
            call endrun
         end if
         srcs_flsp%local_path  = lpthcnst(1:lastsl(lpthsrcs))
         srcs_flsp%remote_path = rpthcnst(1:lastsl(rpthsrcs))

         if( rstflg == 1 ) then
            do file = 1,hst_file_cnt
               hst_flsp(file)%nl_filename = lpthhst(file)(lastsl(lpthhst(file))+1:len_trim(lpthhst(file)) )
               if( hst_flsp(file)%nl_filename /= rpthhst(file)(lastsl(rpthhst(file))+1:len_trim(rpthhst(file))) ) then
                  write(*,*) 'inirest: ERROR, Local and remote history filenames differ in file ',trim(lrst0)
                  write(*,*) 'local = ',trim( hst_flsp(file)%nl_filename ),', remote = ', &
                              rpthhst(file)(lastsl(rpthhst(file))+1:len_trim(rpthhst(file)) )
                  call endrun
               end if
               hst_flsp(file)%local_path  = lpthhst(file)(1:lastsl(lpthhst(file)))
               hst_flsp(file)%remote_path = rpthhst(file)(1:lastsl(rpthhst(file)))
            end do
            sim_file_cnt = hst_file_cnt                                   ! for now just override
         end if

!-----------------------------------------------------------------------
!     	... if branch run check that branch remote directory
!            /= main truck remote directory
!-----------------------------------------------------------------------
         if( rstflg == 2 ) then
            indx1 = lastsl( rrst1 ) - 1
            indx2 = lastsl( brrst1 ) - 1
            if( rrst1(:indx1) == brrst1(:indx2) ) then
               write(*,*) 'inirest : ERROR, Trunk and remote restart filepaths are identical : ',rrst1(:indx1)
               write(*,*) '          This would result in potentially overwritting existing files'
               call endrun
            end if
         end if
         rest1_flsp%nl_filename = lrst1(lastsl(lrst1)+1:len_trim(lrst1))
         if( rest1_flsp%nl_filename /= rrst1(lastsl(rrst1)+1:len_trim(rrst1)) ) then
            write(*,*) 'inirest: ERROR, Local and remote restart filenames differ in file ',trim(lrst0)
            write(*,*) 'local = ',trim( rest1_flsp%nl_filename ), &
                                ', remote = ',rrst1(lastsl(rrst1)+1:len_trim(rrst1))
            call endrun
         end if
         rest1_flsp%local_path  = lrst1(1:lastsl(lrst1))
         rest1_flsp%remote_path = rrst1(1:lastsl(rrst1))
         close( iunrst0 )
      else
         itimrst = 0
      end if Restarting

!-----------------------------------------------------------------------
!     	... set module variables
!-----------------------------------------------------------------------
      lunrst0  = iunrst0
      lunrst1  = iunrst1
      rstrrt   = nrt
      rstfrq   = xrstfrq
      async    = xasync
      rmrst    = xrmrst
      lpthrst0 = lrst0
      rpthrst0 = rrst0
      lpthrst1 = lrst1
      rpthrst1 = rrst1
      if( rstflg == 2 ) then
         blpthrst1 = blrst1
         brpthrst1 = brrst1
      end if
      if( masternode ) then
         write(*,*) 'inirest: lpthrst0 = ',trim(lpthrst0)
         write(*,*) 'inirest: rpthrst0 = ',trim(rpthrst0)
         write(*,*) 'inirest: lpthrst1 = ',trim(lpthrst1)
         write(*,*) 'inirest: rpthrst1 = ',trim(rpthrst1)
         if( rstflg == 2 ) then
            write(*,*) 'inirest: blpthrst1 = ',trim(blpthrst1)
            write(*,*) 'inirest: brpthrst1 = ',trim(brpthrst1)
         end if
      end if
!-----------------------------------------------------------------
!	... this is a temporary override to turn off restart
!           handling specific to "chemistry"  S.W. 6/4/97
!-----------------------------------------------------------------
      stratchem = .false.
      tropchem  = .false.
      diconvccm = xdiconvccm
      dicldphys = xdicldphys

      if( io_node ) then
         if( rstflg <= 1 ) then
!-----------------------------------------------------------------------
!     	... open restart info file
!-----------------------------------------------------------------------
            open( unit = lunrst0, file = trim( lpthrst0 ), status = 'unknown' )
         else if( rstflg == 2 ) then
            if( lrst0 /= blrst0 ) then
               close( iunrst0 )
               open( unit = lunrst0, file = trim( lrst0 ), status = 'unknown' )
               write(*,*) 'inirest : opened restart info file : ',trim(lrst0),' on unit ',lunrst0
            end if
         end if
      end if

      end subroutine inirest

      subroutine rest_file_inti( nstep, ncdate, ncsec )
!------------------------------------------------------------------------------------------
!	... initialize the restart netcdf file
!           Note : the netcdf file is expected to be open for variable definition at entry
!------------------------------------------------------------------------------------------

      use netcdf
      use m_tracname,   only : tracnam
      use mo_constants, only : r2d, phi, latwts
      use mo_calendar,  only : diffdat
      use plevs,        only : hyam, hybm, hyai, hybi, p0 => ps0
      use mo_control,   only : xactive_drydep, xactive_emissions
      use mo_xemis,     only : has_xemis_no
      use mo_aerosols,  only : has_aerosols
      use mo_bvoc,      only : megan_species, megan_cnt

      implicit none

!------------------------------------------------------------------------------------------
!	... dummy arguments
!------------------------------------------------------------------------------------------
      integer, intent(in) :: nstep, ncdate, ncsec

!------------------------------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------------------------------
      integer :: dim_cnt, i, ind, var_id
      integer :: dims(3)
      real    :: ndys0
      real    :: latdeg(plat)
      real    :: lam(plon)
      character(len=32) :: varname
      character(len=80) :: attribute

!-----------------------------------------------------------------------
!     	... define dimensions
!-----------------------------------------------------------------------
      call handle_ncerr( nf_def_dim( ncid, 'lon', plon, lon_id ), &
                         'rest_file_inti: Failed to create longitude dimension' )
      call handle_ncerr( nf_def_dim( ncid, 'lev', plev, lev_id ), &
                         'rest_file_inti: Failed to create level dimension' )
      call handle_ncerr( nf_def_dim( ncid, 'ilev', plev+1, ilev_id ), &
                         'rest_file_inti: Failed to create level dimension' )
      call handle_ncerr( nf_def_dim( ncid, 'lat', plat, lat_id ), &
                         'rest_file_inti: Failed to create latitude dimension' )
!-----------------------------------------------------------------------
!     	... set global attributes
!-----------------------------------------------------------------------
      attribute = 'NCAR-CSM'
      call handle_ncerr( nf_put_att_text( ncid, nf_global, 'Conventions', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create global conventions attribute' )
      call handle_ncerr( nf_put_att_text( ncid, nf_global, 'case', 32, case_name ), &
                         'rest_file_inti: Failed to create global case attribute' )
!-----------------------------------------------------------------------
!     	... define variables and attributes; first the scalars
!-----------------------------------------------------------------------
      call handle_ncerr( nf_def_var( ncid, 'P0', nf_double, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create P0 variable' )
      attribute = 'reference pressure'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create P0 attribute' )
      attribute = 'Pa'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create P0 attribute' )
      call handle_ncerr( nf_def_var( ncid, 'drymass', nf_double, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create drymass variable' )
      attribute = 'atmosphere dry mass'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create drymass attribute' )
      attribute = 'Pa'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create drymass attribute' )
      call handle_ncerr( nf_def_var( ncid, 'days', nf_int, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create days variable' )
      attribute = 'elapsed simulation days for this case'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create days attribute' )
      call handle_ncerr( nf_def_var( ncid, 'secs', nf_int, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create secs variable' )
      attribute = 'seconds to complete elapsed days'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create secs attribute' )
      attribute = 's'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create secs attribute' )
      call handle_ncerr( nf_def_var( ncid, 'date', nf_int, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create date variable' )
      attribute = 'current date as 6 digit integer (YYMMDD)'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create date attribute' )
      call handle_ncerr( nf_def_var( ncid, 'datesec', nf_int, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create datesec variable' )
      attribute = 'seconds to complete current date'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create datesec attribute' )
      attribute = 's'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create datesec attribute' )
      call handle_ncerr( nf_def_var( ncid, 'time', nf_double, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create time variable' )
      attribute = 'simulation time'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create time attribute' )
      attribute = 'days since 0000-01-01 00:00:00'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create time attribute' )
      attribute = '365_days'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'calendar', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create time attribute' )
      call handle_ncerr( nf_def_var( ncid, 'timestep_index', nf_int, 0, dims, var_id ), &
                         'rest_file_inti: Failed to create timestep_index variable' )
      attribute = 'iteration counter for current case'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create timestep_index attribute' )

!-----------------------------------------------------------------------
!     	... the arrays
!-----------------------------------------------------------------------
      dims(1) = ilev_id
      call handle_ncerr( nf_def_var( ncid, 'hyai', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create hyai variable' )
      attribute = 'hybrid A coefficient at layer interfaces'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create hyai attribute' )
      call handle_ncerr( nf_def_var( ncid, 'hybi', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create hybi variable' )
      attribute = 'hybrid B coefficient at layer interfaces'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create hybi attribute' )
      call handle_ncerr( nf_def_var( ncid, 'ilev', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create ilev variable' )
      attribute = 'hybrid level at layer interface (1000*(A+B))'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create ilev attribute' )
      attribute = 'hybrid_sigma_pressure'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'down'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'positive', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create ilev attribute' )
      attribute = 'hyam'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'A_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create ilev attribute' )
      attribute = 'hybm'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'B_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create ilev attribute' )
      attribute = 'P0'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'P0_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create ilev attribute' )
      attribute = 'PS'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'PS_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create ilev attribute' )
      attribute = 'ilev'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'bounds', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      dims(1) = lev_id
      call handle_ncerr( nf_def_var( ncid, 'hyam', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create hyam variable' )
      attribute = 'hybrid A coefficient at layer midpoints'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create hyam attribute' )
      call handle_ncerr( nf_def_var( ncid, 'hybm', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create hybm variable' )
      attribute = 'hybrid B coefficient at layer midpoints'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create hybm attribute' )
      call handle_ncerr( nf_def_var( ncid, 'lev', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create lev variable' )
      attribute = 'hybrid level at layer midpoints (1000*(A+B))'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'hybrid_sigma_pressure'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'down'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'positive', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'hyam'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'A_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'hybm'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'B_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'P0'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'P0_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'PS'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'PS_var', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      attribute = 'ilev'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'bounds', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lev attribute' )
      dims(1) = lat_id
      call handle_ncerr( nf_def_var( ncid, 'lat', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create lat variable' )
      attribute = 'latitude'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lat attribute' )
      attribute = 'degrees_north'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lat attribute' )
      call handle_ncerr( nf_def_var( ncid, 'gw', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create gw variable' )
      attribute = 'gauss weights'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create gw attribute' )
      dims(1) = lon_id
      call handle_ncerr( nf_def_var( ncid, 'lon', nf_double, 1, dims, var_id ), &
                         'rest_file_inti: Failed to create lon variable' )
      attribute = 'longitude'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lon attribute' )
      attribute = 'degrees_east'
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create lon attribute' )

      dim_cnt = 2
      dims(:dim_cnt) = (/ lon_id, lat_id /)
      call handle_ncerr( nf_def_var( ncid, 'PS' , nf_double, dim_cnt, dims, var_id ), &
                         'rest_file_inti: Failed to create PS variable' )
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', 16, 'Surface Pressure' ), &
                         'rest_file_inti: Failed to create PS longname attribute' )
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', 2, 'Pa' ), &
                         'rest_file_inti: Failed to create PS units attribute' )
      if( xactive_drydep .or. xactive_emissions ) then
         call handle_ncerr( nf_def_var( ncid, 'precip' , nf_double, dim_cnt, dims, var_id ), &
                            'rest_file_inti: Failed to create precip variable' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', 16, 'Surface Precip' ), &
                            'rest_file_inti: Failed to create precip longname attribute' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', 2, 'm' ), &
                            'rest_file_inti: Failed to create precip units attribute' )
      end if
      if( has_xemis_no ) then
         call handle_ncerr( nf_def_var( ncid, 'is_pulsing' , nf_int, dim_cnt, dims, var_id ), &
                            'rest_file_inti: Failed to create is_pulsing variable' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', len_trim('Precip pulsing cnt'), 'Precip pulsing cnt' ), &
                            'rest_file_inti: Failed to create is_pulsing longname attribute' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', 1, ' ' ), &
                            'rest_file_inti: Failed to create is_pulsing units attribute' )
         call handle_ncerr( nf_def_var( ncid, 'soil_wetness' , nf_double, dim_cnt, dims, var_id ), &
                            'rest_file_inti: Failed to create soil_wetness variable' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', len_trim('Soil wetness') , 'Soil wetness' ), &
                            'rest_file_inti: Failed to create soil_wetness longname attribute' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', 1, ' ' ), &
                            'rest_file_inti: Failed to create soil_wetness units attribute' )
         call handle_ncerr( nf_def_var( ncid, 'time_pulsing' , nf_double, dim_cnt, dims, var_id ), &
                            'rest_file_inti: Failed to create time_pulsing variable' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', len_trim('time_pulsing'), 'time_pulsing' ), &
                            'rest_file_inti: Failed to create time_pulsing longname attribute' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', 1, ' ' ), &
                            'rest_file_inti: Failed to create time_pulsing units attribute' )
      end if
      if( megan_cnt > 0 ) then
         do i = 1,megan_cnt
            varname = 'megan_map_' // trim( megan_species(i) )
            call handle_ncerr( nf_def_var( ncid, trim( varname ), nf_double, dim_cnt, dims, var_id ), &
                               'rest_file_inti: Failed to create ' // trim( varname ) // ' variable' )
            call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                                len_trim(tracnam(i)), trim(tracnam(i)) ), &
                               'rest_file_inti: Failed to create ' // trim(tracnam(i)) // ' attribute' )
            call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                             len_trim(attribute), trim(attribute) ), &
                               'rest_file_inti: Failed to create ' // trim(tracnam(i)) // ' attribute' )
         end do
      end if
# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
# if defined TROP_CHEM || defined STRAT_CHEM
      call handle_ncerr( nf_def_var( ncid, 'cldtop' , nf_double, dim_cnt, dims, var_id ), &
                         'rest_file_inti: Failed to create cldtop variable' )
      call handle_ncerr( nf_def_var( ncid, 'cldbot' , nf_double, dim_cnt, dims, var_id ), &
                         'rest_file_inti: Failed to create cldbot variable' )
      dim_cnt = 3
      dims(:dim_cnt) = (/ lon_id, lat_id, lev_id /)
      call handle_ncerr( nf_def_var( ncid, 'geohgt-mid' , nf_double, dim_cnt, dims, var_id ), &
                         'rest_file_inti: Failed to create geo midpoint height variable' )
      dims(:dim_cnt) = (/ lon_id, lat_id, ilev_id /)
      call handle_ncerr( nf_def_var( ncid, 'geohgt-int' , nf_double, dim_cnt, dims, var_id ), &
                         'rest_file_inti: Failed to create geo interface height' )
      dims(:dim_cnt) = (/ lon_id, lat_id, lev_id /)
# else
      dim_cnt = 3
      dims(:dim_cnt) = (/ lon_id, lat_id, lev_id /)
# endif
# else
      dim_cnt = 3
      dims(:dim_cnt) = (/ lon_id, lat_id, lev_id /)
# endif
      attribute = 'kg/kg'
      call handle_ncerr( nf_def_var( ncid, 'sh', nf_double, dim_cnt, dims, var_id ), &
                         'rest_file_inti: Failed to create specific humidity variable' )
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                          len_trim( 'specific humidity' ), 'specific humidity' ), &
                         'rest_file_inti: Failed to create specific humidity attribute' )
      call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                       len_trim(attribute), trim(attribute) ), &
                         'rest_file_inti: Failed to create specific humidity attribute' )
      attribute = 'kg/kg'
      do i = 1,pcnst
         call handle_ncerr( nf_def_var( ncid, trim( tracnam(i) ), nf_double, dim_cnt, dims, var_id ), &
                            'rest_file_inti: Failed to create ' // trim( tracnam(i) ) // ' variable' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'long_name', &
                                             len_trim(tracnam(i)), trim(tracnam(i)) ), &
                            'rest_file_inti: Failed to create ' // trim(tracnam(i)) // ' attribute' )
         call handle_ncerr( nf_put_att_text( ncid, var_id, 'units', &
                                          len_trim(attribute), trim(attribute) ), &
                            'rest_file_inti: Failed to create ' // trim(tracnam(i)) // ' attribute' )
      end do

!-----------------------------------------------------------------------
!     	... leave define and set spatial variables
!-----------------------------------------------------------------------
      call handle_ncerr( nf_enddef( ncid ), 'rest_file_inti: Failed to leave define mode' )
      call handle_ncerr( nf_inq_varid( ncid, 'date', var_id ), &
                         'rest_file_inti: Failed to get date variable id' )
      call handle_ncerr( nf_put_var_int( ncid, var_id, ncdate ), &
                         'rest_file_inti: Failed to write date variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'datesec', var_id ), &
                         'rest_file_inti: Failed to get datesec variable id' )
      call handle_ncerr( nf_put_var_int( ncid, var_id, ncsec ), &
                         'rest_file_inti: Failed to write datesec variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'timestep_index', var_id ), &
                         'rest_file_inti: Failed to get timestep_index variable id' )
      call handle_ncerr( nf_put_var_int( ncid, var_id, nstep ), &
                         'rest_file_inti: Failed to write timestep_index variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'time', var_id ), &
                         'rest_file_inti: Failed to get time variable id' )
      ndys0 = diffdat( 101, 0, ncdate, ncsec )
      call handle_ncerr( nf_put_var_double( ncid, var_id, ndys0 ), &
                         'rest_file_inti: Failed to write time variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'lon', var_id ), &
                         'rest_file_inti: Failed to get longitudes variable id' )
      lam(:) = (/ (360.*real(i)/real(plon),i=0,plon-1) /)
      call handle_ncerr( nf_put_var_double( ncid, var_id, lam ), &
                         'rest_file_inti: Failed to write longitudes variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'lat', var_id ), &
                         'rest_file_inti: Failed to get latitude variable id' )
      latdeg(:) = r2d * phi(:)
      call handle_ncerr( nf_put_var_double( ncid, var_id, latdeg ), &
                         'rest_file_inti: Failed to write latitudes variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'gw', var_id ), &
                         'rest_file_inti: Failed to get gaussian weights variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, latwts ), &
                         'rest_file_inti: Failed to write gaussian weights variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'hyam', var_id ), &
                         'rest_file_inti: Failed to get hyam variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, hyam ), &
                         'rest_file_inti: Failed to write hyam variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'hybm', var_id ), &
                         'rest_file_inti: Failed to get hybm variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, hybm ), &
                         'rest_file_inti: Failed to write hybm variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'lev', var_id ), &
                         'rest_file_inti: Failed to get lev variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, 1000.*(hybm+hyam) ), &
                         'rest_file_inti: Failed to write ilev variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'hyai', var_id ), &
                         'rest_file_inti: Failed to get hyai variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, hyai ), &
                         'rest_file_inti: Failed to write hyai variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'hybi', var_id ), &
                         'rest_file_inti: Failed to get hybi variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, hybi ), &
                         'rest_file_inti: Failed to write hybi variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'ilev', var_id ), &
                         'rest_file_inti: Failed to get ilev variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, 1000.*(hybi+hyai) ), &
                         'rest_file_inti: Failed to write ilev variable' )
      call handle_ncerr( nf_inq_varid( ncid, 'P0', var_id ), &
                         'rest_file_inti: Failed to get P0 variable id' )
      call handle_ncerr( nf_put_var_double( ncid, var_id, p0 ), &
                         'rest_file_inti: Failed to write P0 variable' )

!-----------------------------------------------------------------------
!     	... close file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), 'rest_file_inti: Failed to close history netcdf file ' )

      end subroutine rest_file_inti

# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
      subroutine rdrst( as, cldtop, cldbot, zm, zint, &
                        rstflg, plonl, platl, pplon, ps, &
                        sh, precip )
# else
      subroutine rdrst( as, rstflg, plonl, platl, pplon, ps, sh )
# endif
!-----------------------------------------------------------------------
! 	... read restart data file
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_drymass,    only : mdryatm
      use mo_file_utils, only : open_netcdf_file
      use mo_charutl,    only : lastsl

      implicit none

!-----------------------------------------------------------------------
! 	... dummy argument
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        rstflg, &                  ! restart flag( 1 == restart run, 2 == branch run )
        plonl, &
        platl, &
        pplon
      real, intent(inout) :: &
        ps(plonl,-3:platl+4,pplon), &       ! surface pressure (Pa)
        sh(plonl,plev,platl,pplon)          ! specific humidity (kg/kg)
      real, intent(out) :: &
# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
        as(plonl,plev,pcnst,platl,pplon), & ! advected species (kg/kg)
        cldtop(plonl,platl,pplon), &        ! cloud top index
        cldbot(plonl,platl,pplon), &        ! cloud bottom index
        zm(plonl,plev,platl,pplon), &       ! geo height above surface at midpoints (m)
        zint(plonl,plevp,platl,pplon)       ! geo height above surface at interfaces (m)
      real, optional, intent(out) :: &
        precip(plonl,platl,pplon)           ! precip at the ground (m)
#else
        as(plonl,plev,pcnst,platl,pplon)    ! advected species (kg/kg)
# endif

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: &
        istat       ! return status
      character(len=256) :: &
        cmd         !  Command string
      character(len=168) :: &
        filename, lpath, mspath


!-----------------------------------------------------------------------
!	... function declarations
!-----------------------------------------------------------------------
      integer :: fsystem

!-----------------------------------------------------------------------
!     	... get and open restart data file
!-----------------------------------------------------------------------
      if( rstflg == 1 ) then
         filename = lpthrst1(lastsl( lpthrst1 )+1:len_trim( lpthrst1 ))
         lpath    = lpthrst1(:lastsl( lpthrst1 ))
         mspath   = rpthrst1(:lastsl( rpthrst1 ))
      else if( rstflg == 2 ) then
         filename = blpthrst1(lastsl( blpthrst1 )+1:len_trim( blpthrst1 ))
         lpath    = blpthrst1(:lastsl( blpthrst1 ))
         mspath   = brpthrst1(:lastsl( brpthrst1 ))
      end if
      write(*,*) 'rdrst: filename = ',trim(filename)
      write(*,*) 'rdrst: lpath    = ',trim(lpath)
      write(*,*) 'rdrst: mspath   = ',trim(mspath)
      ncid = open_netcdf_file( filename, lpath, mspath )
!-----------------------------------------------------------------------
!     	... read netcdf restart file
!-----------------------------------------------------------------------
# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
# if defined TROP_CHEM || defined STRAT_CHEM
      if( present( precip ) ) then
         call rest_file_input( as, plonl, platl, pplon, ps, &
                               sh, zm, zint, cldtop, cldbot, precip )
      else
         call rest_file_input( as, plonl, platl, pplon, ps, &
                               sh, zm, zint, cldtop, cldbot )
      end if
# else
      call rest_file_input( as, plonl, platl, pplon, ps, sh )
# endif
# else
      call rest_file_input( as, plonl, platl, pplon, ps, sh )
# endif
!-----------------------------------------------------------------------
!     	... close netcdf restart file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), &
                         'rdrst: Failed to close file ' // trim( lpthrst1 ) )
      if( masternode .and. rmrst ) then
!-----------------------------------------------------------------------
!    	... remove restart data file
!-----------------------------------------------------------------------
         cmd = 'rm -f ' // trim(lpthrst1)
         write(*,*) 'rdrst: fsystem issuing command - '
         write(*,*) trim(cmd)
         istat = fsystem( cmd )
      end if

      end subroutine rdrst

      subroutine rest_file_input( as, plonl, platl, pplon, ps, &
                                  sh, zm, zint, cldtop, cldbot, precip )
!-----------------------------------------------------------------------
! 	... get time dependent variables from the netcdf file
!-----------------------------------------------------------------------

      use netcdf
      use mo_mpi,     only : base_lat, masternode, lastnode
      use mo_grid,    only : plon, plev, plat
      use mo_drymass, only : mdryatm
      use mo_xemis,   only : has_xemis_no
      use mo_soil_no, only : soil_no_rdrst
      use mo_aerosols,only : has_aerosols
      use mo_bvoc,    only : bvoc_rdrst
      use mo_bvoc,    only : megan_cnt
      use m_tracname, only : tracnam

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
         plonl, &
         platl, &
         pplon
      real, intent(out) :: &
        as(plonl,plev,pcnst,platl,pplon) ! advected species (kg/kg)
      real, optional, intent(out) :: &
        ps(plonl,-3:platl+4,pplon), &    ! surface pressure (Pa)
        precip(plonl,platl,pplon), &     ! precip at the ground (m)
        sh(plonl,plev,platl,pplon), &    ! specific humidity (kg/kg)
        cldtop(plonl,platl,pplon), &     ! cloud top index
        cldbot(plonl,platl,pplon), &     ! cloud bottom index
        zm(plonl,plev,platl,pplon), &    ! geo height above surface at midpoints (m)
        zint(plonl,plevp,platl,pplon)    ! geo height above surface at interfaces (m)

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: j, jl, ju, jll, jlu, lev, m
      integer :: var_id, dim_id, dim_size
      integer :: astat
      integer, dimension(3) :: start, cnt
      real, allocatable :: wrk2d(:,:)
      real :: wrk3d(plon,platl,plev)
      real :: wrk3dp(plon,platl,plevp)
      real :: tmp2d(plonl,pplon)

!-----------------------------------------------------------------------
!     	... check restart file spatial dimensions against model
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dim_id ), &
                         'rest_file_input: failed to get lon coordinate id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dim_id, dim_size ), &
                         'rest_file_input: failed to get length of dimension lon' )
      if( dim_size /= plon ) then
         write(*,*) 'rest_file_input: restart longitude dimension ',dim_size
         write(*,*) '                 does not match model size   ',plon
         call endrun
      end if
      call handle_ncerr( nf_inq_dimid( ncid, 'lev', dim_id ), &
                         'rest_file_input: failed to get lev coordinate id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dim_id, dim_size ), &
                         'rest_file_input: failed to get length of dimension lev' )
      if( dim_size /= plev ) then
         write(*,*) 'rest_file_input: restart levels dimension  ',dim_size
         write(*,*) '                 does not match model size ',plev
         call endrun
      end if
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dim_id ), &
                         'rest_file_input: failed to get lat coordinate id' )
      call handle_ncerr( nf_inq_dimlen( ncid, dim_id, dim_size ), &
                         'rest_file_input: failed to get length of dimension lat' )
      if( dim_size /= plat ) then
         write(*,*) 'rest_file_input: restart latitude dimension ',dim_size
         write(*,*) '                 does not match model size  ',plat
         call endrun
      end if
!-----------------------------------------------------------------------
!     	... read drymass
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'drymass', var_id ), &
                         'rest_file_input: Failed to get drymass variable id' )
      call handle_ncerr( nf_get_var1_double( ncid, var_id, 1, mdryatm ), &
                         'rest_file_input: Failed to read drymass variable' )
      write(*,*) 'rest_file_input: read drymass'
!-----------------------------------------------------------------------
!     	... read advected species
!-----------------------------------------------------------------------
      start(:) = (/ 1, base_lat+1, 1 /)
      cnt(:) = (/ plon, platl, plev /)
#ifdef DEBUG
      write(*,*) 'rest_file_input: Before mixing ratios read'
      write(*,*) 'rest_file_input: start = ',start
      write(*,*) 'rest_file_input: count = ',cnt
#endif
      do m = 1,pcnst
         call handle_ncerr( nf_inq_varid( ncid, trim( tracnam(m) ), var_id ), &
                            'rest_file_input: Failed to get ' //  trim( tracnam(m) ) // ' variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk3d ), &
                            'rest_file_input: Failed to read ' //  trim( tracnam(m) ) // ' variable' )
#ifdef DEBUG
         if( m == 1 ) then
            write(*,*) 'rest_file_input: OX zero values = ',COUNT( wrk3d == 0. )
         end if
#endif
         do lev = 1,plev
            do j = 1,platl
               tmp2d(:,:) = reshape( wrk3d(:,j,lev), (/ plonl, pplon /) )
               as(:,lev,m,j,:) = tmp2d(:,:)
            end do
         end do
#ifdef DEBUG
         if( m == 1 ) then
            write(*,*) 'rest_file_input: OX zero values = ',COUNT( as(:,:,1,:,:) == 0. )
         end if
#endif
      end do
      write(*,*) 'rest_file_input: Mixing ratios read'
      if( present( sh ) ) then
!-----------------------------------------------------------------------
!     	... read specific humidity
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'sh', var_id ), &
                            'rest_file_input: Failed to get specific humidity variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk3d ), &
                            'rest_file_input: Failed to read specific humidity variable' )
	 do lev = 1,plev
	    do j = 1,platl
	       tmp2d(:,:)    = reshape( wrk3d(:,j,lev), (/ plonl, pplon /) )
	       sh(:,lev,j,:) = tmp2d(:,:)
	    end do
	 end do
         write(*,*) 'rest_file_input: read sh'
      end if
      if( present( zm ) ) then
!-----------------------------------------------------------------------
!     	... read geo midpoint height
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'geohgt-mid', var_id ), &
                            'rest_file_input: Failed to get geo midpoint height variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk3d ), &
                            'rest_file_input: Failed to read geo midpoint height variable' )
	 do lev = 1,plev
	    do j = 1,platl
	       tmp2d(:,:)    = reshape( wrk3d(:,j,lev), (/ plonl, pplon /) )
	       zm(:,lev,j,:) = tmp2d(:,:)
	    end do
	 end do
         write(*,*) 'rest_file_input: read zmid'
      end if
      if( present( zint ) ) then
!-----------------------------------------------------------------------
!     	... read geo interface height
!-----------------------------------------------------------------------
	 cnt(3) = plevp
         call handle_ncerr( nf_inq_varid( ncid, 'geohgt-int', var_id ), &
                            'rest_file_input: Failed to get geo interface height variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk3dp ), &
                            'rest_file_input: Failed to read geo interface height variable' )
	 do lev = 1,plevp
	    do j = 1,platl
	       tmp2d(:,:)      = reshape( wrk3dp(:,j,lev), (/ plonl, pplon /) )
	       zint(:,lev,j,:) = tmp2d(:,:)
	    end do
	 end do
         write(*,*) 'rest_file_input: read zint'
	 cnt(3) = plev
      end if
!-----------------------------------------------------------------------
!     	... read precip
!-----------------------------------------------------------------------
      if( present( precip ) ) then
	 if( allocated( wrk2d ) ) then
	    deallocate( wrk2d )
	 end if
	 allocate( wrk2d(plon,platl),stat=astat )
	 if( astat /= 0 ) then
	    write(*,*) 'rest_file_input: failed to allocate wrk2d; error = ',astat
	    call endrun
	 end if
         call handle_ncerr( nf_inq_varid( ncid, 'precip', var_id ), &
                            'rest_file_input: Failed to get precip variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start(:2), cnt(:2), wrk2d ), &
                            'rest_file_input: Failed to read precip variable' )
	 do j = 1,platl
	    tmp2d(:,:)    = reshape( wrk2d(:,j), (/ plonl, pplon/) )
	    precip(:,j,:) = tmp2d(:,:)
	 end do
         write(*,*) 'rest_file_input: read precip'
      end if
      if( present( cldtop ) ) then
	 if( allocated( wrk2d ) ) then
	    deallocate( wrk2d )
	 end if
	 allocate( wrk2d(plon,platl),stat=astat )
	 if( astat /= 0 ) then
	    write(*,*) 'rest_file_input: failed to allocate wrk2d; error = ',astat
	    call endrun
	 end if
!-----------------------------------------------------------------------
!     	... read cldtop index
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'cldtop', var_id ), &
                            'rest_file_input: Failed to get cldtop variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start(:2), cnt(:2), wrk2d ), &
                            'rest_file_input: Failed to read cldtop variable' )
	 do j = 1,platl
	    tmp2d(:,:)    = reshape( wrk2d(:,j), (/ plonl, pplon/) )
	    cldtop(:,j,:) = tmp2d(:,:)
	 end do
         write(*,*) 'rest_file_input: read cldtop'
      end if
      if( present( cldbot ) ) then
	 if( .not. allocated( wrk2d ) ) then
	    allocate( wrk2d(plon,platl),stat=astat )
	    if( astat /= 0 ) then
	       write(*,*) 'rest_file_input: failed to allocate wrk2d; error = ',astat
	       call endrun
	    end if
	 end if
!-----------------------------------------------------------------------
!     	... read cldbot index
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'cldbot', var_id ), &
                            'rest_file_input: Failed to get cldbot variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start(:2), cnt(:2), wrk2d ), &
                            'rest_file_input: Failed to read cldbot variable' )
	 do j = 1,platl
	    tmp2d(:,:) = reshape( wrk2d(:,j), (/ plonl, pplon/) )
	    cldbot(:,j,:) = tmp2d(:,:)
	 end do
         write(*,*) 'rest_file_input: read cldbot'
      end if
      if( present( ps ) ) then
!-----------------------------------------------------------------------
!     	... read surface pressure
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'PS', var_id ), &
                            'rest_file_input: Failed to get surface pressure variable id' )
         jl = base_lat-3
         ju = base_lat+platl+4
         if( masternode ) then
            jl = base_lat+1
         end if
         if( lastnode ) then
            ju = base_lat+platl
         end if
         jll = jl - base_lat
         jlu = ju - base_lat
         if( allocated( wrk2d ) ) then
            deallocate( wrk2d )
         end if
         allocate( wrk2d(plon,jll:jlu),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'rest_file_input: failed to allocate wrk2d; error = ',astat
         call endrun
         end if
         start(:2) = (/ 1, jl /)
         cnt(:2)   = (/ plon, ju - jl + 1 /)
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start(:2), cnt(:2), wrk2d ), &
                            'rest_file_input: Failed to read PS variable' )
         do j = jll,jlu
            tmp2d(:,:) = reshape( wrk2d(:,j), (/ plonl, pplon/) )
            ps(:,j,:)  = tmp2d(:,:)
         end do
         write(*,*) 'rest_file_input: read PS'
      end if
!-----------------------------------------------------------------------
!     	... read soil srf no emission variables
!-----------------------------------------------------------------------
      if( has_xemis_no ) then
         call soil_no_rdrst( plonl, platl, pplon, ncid )
      end if
!-----------------------------------------------------------------------
!     	... read megan maps
!-----------------------------------------------------------------------
      if( megan_cnt > 0 ) then
         call bvoc_rdrst( plonl, platl, pplon, ncid )
      end if

      if( allocated( wrk2d ) ) then
         deallocate( wrk2d )
      end if

      end subroutine rest_file_input

# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
      subroutine wrrst( nstep, ncdate, ncsec, ps, as, &
                        partial_ta, plonl, platl, pplon, sh, &
                        cldtop, cldbot, zm, zint, precip )
# else
      subroutine wrrst( nstep, ncdate, ncsec, ps, as, &
                        partial_ta, plonl, platl, pplon )
# endif
!-----------------------------------------------------------------------
! 	... write and dispose restart info and data files if it is time
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_drymass,      only : mdryatm
      use mo_dyninp,       only : qdyn, date, datesec
      use mo_dyninp,       only : last_filename
      use mo_histout,      only : qhstc, qhsti, sim_file_cnt, hst_file_max, hfile
      use mo_control,      only : rest0_flsp, rest1_flsp, dyn_flsp, endofrun
      use mo_control,      only : srcs_flsp, constit_flsp
      use mo_file_utils,   only : dispose, make_filename
      use mo_charutl,      only : lastsl, incstr
      use mo_calendar,     only : diffdat
      use mo_mscomment,    only : datecom
      use mo_offline_data, only : qdata
      use mo_offline_sources, only: srcs_data
      use mo_offline_constits, only:cnst_data

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        nstep, &                            ! Model timestep number.
        ncdate, &                           ! current date in yymmdd format
        ncsec, &                            ! seconds relative to ncdate
        plonl, &
        platl, &
        pplon

      real, intent(in) :: &
        as(plonl,plev,pcnst,platl,pplon), & ! advected species (kg/kg)
        sh(plonl,plev,platl,pplon), &       ! specific humidity (kg/kg)
        ps(plonl,-3:platl+4,pplon)          ! Surface pressure (Pa)
# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
      real, intent(in) :: &
        cldtop(plonl,platl,pplon), &        ! cloud top index
        cldbot(plonl,platl,pplon), &        ! cloud bottom index
        zm(plonl,plev,platl,pplon), &       ! geo height above surface at midpoints (m)
        zint(plonl,plevp,platl,pplon)       ! geo height above surface at interfaces (m)
      real, optional, intent(in) :: &
        precip(plonl,platl,pplon)           ! precip at ground
# endif
      logical, dimension(hst_file_max), intent(in) :: &
        partial_ta                          ! partial history file time average flag

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: &
        istat, &   ! return status
        ntsdyn, &  ! number of most recently read dynamics time sample
        mfilt, &   ! maximum number of time samples on a history file
        spos, &    ! string position marker
        var_id, &  ! variable id
        ldate, &   ! local dynamics file date
        ldatesec, &  ! local dynamics file seconds in date
        file       ! file index
      integer :: &
        yr, &
        mnth, &
        day
      integer, dimension(5) :: &
        dims, &    ! variable dimensions
        start, &   ! variable start position
        count      ! variable count
      integer, dimension(hst_file_max) :: &
        ntshst     ! number of time samples written on current history file

      character(len=256) :: &
        cmd         !  command string
      character(len=168) :: &
        lpthdyn, &  ! local path for dynamics file
        rpthdyn     ! remote path for dynamics file
      character(len=168), dimension(hst_file_max) :: &
        lpthhst, &  ! local path for history file
        rpthhst     ! remote path for history file
      character(len=168) :: &
        annotated_filename, &
        lwrkflsp, &  ! local temp path for restart file
        rwrkflsp    ! remote temp path for restart file
      character(len=80) :: &
        com         ! comment field
      character(len=32) :: &
        homedir
      character(len=20) :: &
        fl_suffix   ! restart file suffix
      character(len=6) :: &
        time_string
      character(len=1) :: &
        msvol, wpasswd
      character(len=168) :: &
        lpthcnst, &  ! local path for offline constits file
        rpthcnst     ! remote path for offline constits file
      character(len=168) :: &
        lpthsrcs, &  ! local path for offline sources file
        rpthsrcs     ! remote path for offline sources file

      logical :: &
        remove      ! .true. => remove local file after disposing to remote device

!-----------------------------------------------------------------------
!	... function declarations
!-----------------------------------------------------------------------
      integer :: fsystem
      integer :: fgetenv

!-----------------------------------------------------------------------
!     	... update restart info file
!-----------------------------------------------------------------------
      if( masternode ) then
         call qdyn( ncdate, ncsec, lpthdyn, rpthdyn, ntsdyn )
         ldate    = date(1)
         ldatesec = datesec(1)
         write(*,*) 'wrrst dynamics: QDYN->'//trim(lpthdyn)
         write(*,*) 'wrrst dynamics: file->'//trim( dyn_flsp%nl_filename )
         write(*,*) 'NTSDYN = ',ntsdyn
         call qdata( cnst_data, ncdate, ncsec, lpthcnst, rpthcnst )
         write(*,*) 'wrrst offline constits: QDATA->'//trim(lpthcnst)
         write(*,*) 'wrrst offline constits: file->'//trim( constit_flsp%nl_filename )
         call qdata( srcs_data, ncdate, ncsec, lpthsrcs, rpthsrcs )
         write(*,*) 'wrrst offline sources: QDATA->'//trim(lpthsrcs)
         write(*,*) 'wrrst offline sources: file->'//trim( srcs_flsp%nl_filename )
      end if
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... all nodes wait for masternode to get dynamics restart information
!-----------------------------------------------------------------------
      call mpi_barrier( MPI_COMM_WORLD, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'inirest: Barrier failed; code = ',istat
         call endrun
      end if
!-----------------------------------------------------------------------
!     	... if dedicated io node then bcast dynamics restrt info
!-----------------------------------------------------------------------
      if( ded_io_node ) then
         call mpi_bcast( lpthdyn, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for lpthdyn; error = ',istat
            call endrun
         end if
         call mpi_bcast( rpthdyn, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for rpthdyn; error = ',istat
            call endrun
         end if
         call mpi_bcast( ntsdyn, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for ntsdyn; error = ',istat
            call endrun
         end if
         call mpi_bcast( ldate, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for dynamics date; error = ',istat
            call endrun
         end if
         call mpi_bcast( ldatesec, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for dynamics datesec; error = ',istat
            call endrun
         end if
         call mpi_bcast( lpthcnst, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for lpthcnst; error = ',istat
            call endrun
         end if
         call mpi_bcast( rpthcnst, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for rpthcnst; error = ',istat
            call endrun
         end if

         call mpi_bcast( lpthsrcs, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for lpthsrcs; error = ',istat
            call endrun
         end if
         call mpi_bcast( rpthsrcs, 168, MPI_CHARACTER, 0, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'wrrst: Bcast failed for rpthsrcs; error = ',istat
            call endrun
         end if
      end if
#endif
is_io_node : &
      if( io_node ) then
         rewind( lunrst0 )
!-----------------------------------------------------------------------
!     	... get name of dynamics file for current time.  If the most
!           recently read time sample was the first on the file then
!	    must decrement file number to get correct starting file.
!-----------------------------------------------------------------------
         if( ntsdyn == 1 ) then
            if( diffdat( ncdate, ncsec, ldate, ldatesec ) > 0. ) then
               spos = len_trim( lpthdyn )
               spos = index( lpthdyn(:spos), '/', back=.true. )
               if( spos > 0 ) then
                  lpthdyn(spos+1:) = trim(last_filename)
               else
                  lpthdyn = trim(last_filename)
               end if
               spos = len_trim( rpthdyn )
               spos = index( rpthdyn(:spos), '/', back=.true. )
               if( spos > 0 ) then
                  rpthdyn(spos+1:) = trim(last_filename)
               else
                  rpthdyn = trim(last_filename)
               end if
            end if
         end if
!-----------------------------------------------------------------------
!     	... get history file paths and number of time samples currently written
!-----------------------------------------------------------------------
         do file = 1,sim_file_cnt
            istat = qhstc( 'LPATH', lpthhst(file), file )
            if( istat /= 0 ) then
               write(*,*) 'wrrst: qhstc returned error for LPTHHST'
               call endrun
            end if
            istat = qhstc( 'RPATH', rpthhst(file), file )
            if( istat /= 0 ) then
               write(*,*) 'wrrst: qhstc returned error for RPTHHST'
               call endrun
            end if
            istat = qhsti( 'NTSHST', ntshst(file), file )
            if( istat /= 0 ) then
               write(*,*) 'wrrst: qhsti returned error for NTSHST'
               call endrun
            end if
         end do

!-----------------------------------------------------------------------
!     	... increment the restart data filenames
!-----------------------------------------------------------------------
         spos = len_trim( lpthrst1 )
         if ( index( lpthrst1(:spos), 'nc', back=.true. ) == spos-1 ) then
            spos = spos - 3
         end if
         istat = incstr( lpthrst1(:spos), 1 )
         if( istat /= 0 ) then
            write(*,*) 'wrrst: incstr returned ', istat
            write(*,*) '       while trying to increment ', trim(lpthrst1)
            call endrun
         end if
         spos = len_trim( rpthrst1 )
         if( index( rpthrst1(:spos), 'nc', back=.true. ) == spos-1 ) then
            spos = spos - 3
         end if
         istat = incstr( rpthrst1(:spos), 1 )
         if( istat /= 0 ) then
            write(*,*) 'wrrst: incstr returned ', istat
            write(*,*) '       while trying to increment ', trim(rpthrst1)
            call endrun
         end if
         rest1_flsp%nl_filename = lpthrst1(lastsl(lpthrst1 )+1:len_trim(lpthrst1 ))
         call make_filename( ncdate, ncsec, 0, .false., case_name, annotated_filename )
         write(*,*) 'wrrst: new restart file:'
         write(*,*) trim(annotated_filename)

!-----------------------------------------------------------------------
!     	... write restart info file
!-----------------------------------------------------------------------
         write(*,*) 'wrrst: writing restart file at date = ',ncdate,'  seconds = ', ncsec
         write(lunrst0,'(4i10)') ncdate, ncsec, nstep, sim_file_cnt
         write(lunrst0,'(10i10)') ntshst(:sim_file_cnt)
         write(lunrst0,*) partial_ta(:sim_file_cnt)
         do file = 1,sim_file_cnt
            write(lunrst0,'(4i10)') hfile(file)%hdi(26), hfile(file)%hdi(27), &
                                    hfile(file)%hdi(22), hfile(file)%hdi(23)
            write(lunrst0,'(a80)') hfile(file)%lnhstf
         end do
         write(lunrst0,'(a168)') lpthdyn
         write(lunrst0,'(a168)') rpthdyn
         write(lunrst0,'(a168)') lpthcnst
         write(lunrst0,'(a168)') rpthcnst
         write(lunrst0,'(a168)') lpthsrcs
         write(lunrst0,'(a168)') rpthsrcs
         write(lunrst0,'(a168)') lpthhst(:sim_file_cnt)
         write(lunrst0,'(a168)') rpthhst(:sim_file_cnt)
         lwrkflsp = lpthrst1(:lastsl(lpthrst1)) // trim(annotated_filename)
         write(lunrst0,'(a128)') lwrkflsp
         rwrkflsp = rpthrst1(:lastsl(rpthrst1 )) // trim(annotated_filename)
         write(lunrst0,'(a128)') rwrkflsp
         do file = 1,sim_file_cnt
            if( partial_ta(file) ) then
               write(lunrst0,'(a168)') hfile(file)%lpath_rhst
               write(lunrst0,'(a168)') hfile(file)%rpath_rhst
            end if
         end do
         close( lunrst0 )

!-----------------------------------------------------------------------
!     	... dispose restart info file
!-----------------------------------------------------------------------
         msvol   = ' '
         wpasswd = ' '
         com     = ' '
         yr = ncdate/10000 + 10000
         write(time_string(1:5),'(i5)') yr
         fl_suffix = '-' // time_string(2:5) // '-'
         mnth = mod( ncdate,10000 )/100 + 100
         write(time_string(1:3),'(i3)') mnth
         fl_suffix = trim( fl_suffix ) // time_string(2:3) // '-'
         day = mod( mod( ncdate,1000 ),100 ) + 100
         write(time_string(1:3),'(i3)') day
         fl_suffix = trim( fl_suffix ) // time_string(2:3) // '-'
         write(time_string,'(i6)') 100000 + ncsec
         fl_suffix = trim( fl_suffix ) // time_string(2:6)
         lwrkflsp = trim( lpthrst0 ) // trim( fl_suffix )
         rwrkflsp = trim( rpthrst0 ) // trim( fl_suffix )
         cmd   = 'mv ' // trim( lpthrst0 ) // ' ' // trim( lwrkflsp )
         istat = fsystem( cmd )
         write(*,*) 'wrrst: system command:'
         write(*,*) trim(cmd)
         if( istat /= 0 ) then
            write(*,*) 'wrrst: failed; error = ',istat
            call endrun
         end if

         if( .not. endofrun ) then
            remove = rmrst
         else
            remove = .false.
         end if
         call dispose( lwrkflsp, rwrkflsp, rstrrt, async, com, &
                       msvol, wpasswd, remove )

!-----------------------------------------------------------------------
!     	... open, write, close restart pointer file
!-----------------------------------------------------------------------
         istat = fgetenv( 'HOME', homedir )
         if( istat /= 0 ) then
            write(*,*) 'wrrst: failed to get home directory; error = ',istat
            call endrun
         end if
         rwrkflsp = trim( homedir) // '/' // trim( case_name) // '-rpntr'
         write(*,*) ' '
         write(*,*) 'wrrst: will open and write file = ', trim( rwrkflsp )
         write(*,*) ' '
         open( unit = lunrst0, file = trim(rwrkflsp), iostat=istat )
         if( istat /= 0 ) then
            write(*,*) 'wrrst: failed to open ' // trim(rwrkflsp) // '; error = ',istat
            call endrun
         end if
         write(lunrst0,'(a)') trim( lwrkflsp )
         close( lunrst0 )

!-----------------------------------------------------------------------
!     	... open temp restart info file
!-----------------------------------------------------------------------
         if( .not. endofrun ) then
            open( unit = lunrst0, file = trim( lpthrst0 ), iostat=istat )
            if( istat /= 0 ) then
               write(*,*) 'wrrst: failed to open ',trim(lpthrst0),'; error = ',istat
               call endrun
            end if
         end if

!-----------------------------------------------------------------------
!     	... Create new restart data file
!-----------------------------------------------------------------------
         call handle_ncerr( nf_create( trim( lpthrst1 ), or(nf_clobber,nf_64bit_offset), ncid ), &
                            'wrrst: Failed to create ' // trim( lpthrst1 ) )
         write(*,*) 'wrrst: Created restart file ',trim( lpthrst1 )
!-----------------------------------------------------------------------
!     	... Initialize restart netcdf file
!-----------------------------------------------------------------------
         call rest_file_inti( nstep, ncdate, ncsec )
         write(*,*) 'wrrst: file = ',trim( lpthrst1 )
         write(*,*) 'wrrst: ncid = ',ncid
!-----------------------------------------------------------------------
!     	... Open netcdf restart file
!-----------------------------------------------------------------------
         call handle_ncerr( nf_open( trim( lpthrst1 ), nF_WRITE, ncid ), &
                            'wrrst: Failed to open ' // trim( lpthrst1 ) )
         write(*,*) 'wrrst: Opened file ',trim( lpthrst1 )
      end if is_io_node
!-----------------------------------------------------------------------
!     	... Write netcdf restart file
!-----------------------------------------------------------------------
# if defined AR_CLOUD_PHYS || defined DI_CLOUD_PHYS
# if defined TROP_CHEM || defined STRAT_CHEM
      if( present( precip ) ) then
         call rest_file_output( ps, as, sh, plonl, platl, &
                                pplon, zm, zint, cldtop, cldbot, precip )
      else
         call rest_file_output( ps, as, sh, plonl, platl, &
                                pplon, zm, zint, cldtop, cldbot )
      end if
# else
      call rest_file_output( ps, as, sh, plonl, platl, pplon )
# endif
# else
      call rest_file_output( ps, as, sh, plonl, platl, pplon )
# endif
      if( io_node ) then
!-----------------------------------------------------------------------
!     	... Close netcdf restart file
!-----------------------------------------------------------------------
         call handle_ncerr( nf_close( ncid ), 'wrrst: Failed to close file ' // trim( lpthrst1 ) )
         write(*,*) 'wrrst: Closed ' // trim( lpthrst1 )
!-----------------------------------------------------------------------
!     	... Dispose restart file
!-----------------------------------------------------------------------
         if( .not. endofrun ) then
            remove = rmrst
         else
            remove = .false.
         end if
         lwrkflsp = lpthrst1(:lastsl(lpthrst1)) // trim(annotated_filename)
         cmd   = 'mv ' // trim( lpthrst1 ) // ' ' // trim( lwrkflsp )
         istat = fsystem( cmd )
         write(*,*) 'wrrst: system command:'
         write(*,*) trim(cmd)
         if( istat /= 0 ) then
            write(*,*) 'wrrst: failed; error = ',istat
            call endrun
         end if
         rwrkflsp = rpthrst1(:lastsl(rpthrst1)) // trim(annotated_filename)
         call datecom( ncdate, ncsec, com )
         call dispose( lwrkflsp, rwrkflsp, rstrrt, async, com, &
                       msvol, wpasswd, remove )
      end if

      end subroutine wrrst

      subroutine rest_file_output( ps, as, sh, plonl, platl, &
				   pplon, zm, zint, cldtop, cldbot, precip )
!-----------------------------------------------------------------------
! 	... Put time dependent variables into the netcdf file
!-----------------------------------------------------------------------

      use netcdf
      use mo_mpi
      use mo_drymass, only : mdryatm
      use mo_xemis,   only : has_xemis_no
      use mo_soil_no, only : soil_no_wrrst
      use mo_aerosols,only : has_aerosols
      use mo_bvoc,    only : megan_cnt
      use mo_bvoc,    only : bvoc_wrrst
      use m_tracname, only : tracnam

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      real, intent(in) :: &
        as(plonl,plev,pcnst,platl,pplon), & ! advected species (kg/kg)
        sh(plonl,plev,platl,pplon), &       ! specific humidity (kg/kg)
        ps(plonl,-3:platl+4,pplon)          ! surface pressure (Pa)
      real, optional, intent(in) :: &
        cldtop(plonl,platl,pplon), &        ! cloud top index
        cldbot(plonl,platl,pplon), &        ! cloud bottom index
        zm(plonl,plev,platl,pplon), &       ! geo height above surface at midpoints (m)
        precip(plonl,platl,pplon), &        ! precip at ground
        zint(plonl,plevp,platl,pplon)       ! geo height above surface at interfaces (m)

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: j, lev, m, offset, node, nodes, var_id
      integer :: istat, astat
      integer, dimension(3) :: start, count
      real, allocatable :: gather_2d(:,:,:,:)
      real, allocatable :: gather_3d(:,:,:,:,:)
      real, allocatable :: gather_4d(:,:,:,:,:,:)
      real, allocatable :: wrk2d(:,:)
      real, allocatable :: wrk3d(:,:,:)
      character(len=32) :: varname

!-----------------------------------------------------------------------
!     	... Allocate work buffers and write drymass
!-----------------------------------------------------------------------
      if( io_node ) then
	 allocate( wrk2d(plon,plat),stat=astat )
	 if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocate wrk2d; error = ',astat
	    call endrun
	 end if
	 allocate( wrk3d(plon,plat,plev),stat=astat )
	 if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocate wrk3d; error = ',astat
	    call endrun
	 end if
         call handle_ncerr( nf_inq_varid( ncid, 'drymass', var_id ), &
                            'rest_file_output: Failed to get drymass variable id' )
         call handle_ncerr( nf_put_var1_double( ncid, var_id, 1, mdryatm ), &
                            'rest_file_output: Failed to write drymass variable' )
         write(*,*) 'rest_file_output: drymass written'
      end if
#ifdef USE_MPI
      if( ded_io_node ) then
	 nodes = maxnodes+1
      else
	 nodes = maxnodes
      end if
!-----------------------------------------------------------------------
!     	... Gather surface pressure
!-----------------------------------------------------------------------
      count(1) = plon*(platl+8)
      if( io_node ) then
         allocate( gather_2d(plonl,-3:platl+4,pplon,nodes),stat=astat )
      else
         allocate( gather_2d(1,1,1,1),stat=astat )
      end if
      if( astat /= 0 ) then
         write(*,*) 'rest_file_output: Failed to allocated gather_2d; error = ',astat
         call endrun
      end if
      call mpi_gather( ps, count(1), MPI_DOUBLE_PRECISION, &
                       gather_2d, count(1), MPI_DOUBLE_PRECISION, &
                       gather_node, MPI_COMM_WORLD, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'rest_file_output: mpi_gather failed for ps; error code = ',istat
         call endrun
      end if
#endif
!-----------------------------------------------------------------------
!     	... Write surface pressure
!-----------------------------------------------------------------------
      if( io_node ) then
         call handle_ncerr( nf_inq_varid( ncid, 'PS', var_id ), &
                            'rest_file_output: Failed to get PS variable id' )
#ifdef USE_MPI
         do node = 1,maxnodes
            offset = (node-1)*platl
            do j = 1,platl
               wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
            end do
         end do
#else
         do j = 1,plat
            wrk2d(:,j) = reshape( ps(:,j,:), (/ plon /) )
         end do
#endif
         call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d ), &
                            'rest_file_output: Failed to write PS variable' )
         write(*,*) 'rest_file_output: ps written'
      end if
      if( allocated( gather_2d ) ) then
         deallocate( gather_2d )
      end if
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather constituent mass mixing ratios
!-----------------------------------------------------------------------
      count(1) = plon*platl*pcnst*plev
      if( io_node ) then
         allocate( gather_4d(plonl,plev,pcnst,platl,pplon,nodes),stat=astat )
      else
         allocate( gather_4d(1,1,1,1,1,1),stat=astat )
      end if
      if( astat /= 0 ) then
         write(*,*) 'rest_file_output: Failed to allocated gather_4d; error = ',astat
         call endrun
      end if
      call mpi_gather( as, count(1), MPI_DOUBLE_PRECISION, &
                       gather_4d, count(1), MPI_DOUBLE_PRECISION, &
                       gather_node, MPI_COMM_WORLD, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'rest_file_output: mpi_gather failed for as; error code = ',istat
         call endrun
      end if
#endif
      if( io_node ) then
         do m = 1,pcnst
            call handle_ncerr( nf_inq_varid( ncid, trim( tracnam(m) ), var_id ), &
                               'rest_file_output: Failed to get ' //  trim( tracnam(m) ) // ' variable id' )
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do lev = 1,plev
	          do j = 1,platl
	             wrk3d(:,j+offset,lev) = reshape( gather_4d(:,lev,m,j,:,node), (/ plon /) )
	          end do
	       end do
	    end do
#else
	    do lev = 1,plev
	       do j = 1,plat
	          wrk3d(:,j,lev) = reshape( as(:,lev,m,j,:), (/ plon /) )
	       end do
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk3d ), &
                               'rest_file_output: Failed to write ' //  trim( tracnam(m) ) // ' variable' )
         end do
         write(*,*) 'rest_file_output: Mixing ratios written'
      end if
      if( allocated( gather_4d ) ) then
         deallocate( gather_4d )
      end if
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather specific humidity
!-----------------------------------------------------------------------
      count(1) = plon*platl*plev
      if( io_node ) then
         allocate( gather_3d(plonl,plev,platl,pplon,nodes),stat=astat )
      else
         allocate( gather_3d(1,1,1,1,1),stat=astat )
      end if
      if( astat /= 0 ) then
	 write(*,*) 'rest_file_output: Failed to allocated gather_3d; error = ',astat
         call endrun
      end if
      call mpi_gather( sh, count(1), MPI_DOUBLE_PRECISION, &
                       gather_3d, count(1), MPI_DOUBLE_PRECISION, &
                       gather_node, MPI_COMM_WORLD, istat )
      if( istat /= MPI_SUCCESS ) then
         write(*,*) 'rest_file_output: mpi_gather failed for sh; error code = ',istat
	 call endrun
      end if
#endif
!-----------------------------------------------------------------------
!     	... Write specific humidity
!-----------------------------------------------------------------------
      if( io_node ) then
         call handle_ncerr( nf_inq_varid( ncid, 'sh', var_id ), &
                            'rest_file_output: Failed to get specific humidity variable id' )
#ifdef USE_MPI
         do node = 1,maxnodes
            offset = (node-1)*platl
            do lev = 1,plev
               do j = 1,platl
                  wrk3d(:,j+offset,lev) = reshape( gather_3d(:,lev,j,:,node), (/ plon /) )
               end do
            end do
         end do
#else
         do lev = 1,plev
            do j = 1,platl
               wrk3d(:,j,lev) = reshape( sh(:,lev,j,:), (/ plon /) )
            end do
         end do
#endif
         call handle_ncerr( nf_put_var_double( ncid, var_id, wrk3d ), &
                            'rest_file_output: Failed to write specific humidity variable' )
         write(*,*) 'rest_file_output: sh written'
      end if
      if( allocated( gather_3d ) ) then
         deallocate( gather_3d )
      end if

      if( present( zm ) ) then
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather geo midpoint height
!-----------------------------------------------------------------------
         count(1) = plon*platl*plev
	 if( io_node ) then
            allocate( gather_3d(plonl,plev,platl,pplon,nodes),stat=astat )
         else
            allocate( gather_3d(1,1,1,1,1),stat=astat )
         end if
         if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocated gather_3d; error = ',astat
            call endrun
         end if
         call mpi_gather( zm, count(1), MPI_DOUBLE_PRECISION, &
                          gather_3d, count(1), MPI_DOUBLE_PRECISION, &
                          gather_node, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'rest_file_output: mpi_gather failed for zm; error code = ',istat
	    call endrun
         end if
#endif
!-----------------------------------------------------------------------
!     	... Write geo midpoint height
!-----------------------------------------------------------------------
	 if( io_node ) then
            call handle_ncerr( nf_inq_varid( ncid, 'geohgt-mid', var_id ), &
                               'rest_file_output: Failed to get geo midpoint height variable id' )
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do lev = 1,plev
	          do j = 1,platl
	             wrk3d(:,j+offset,lev) = reshape( gather_3d(:,lev,j,:,node), (/ plon /) )
	          end do
	       end do
	    end do
#else
	    do lev = 1,plev
	       do j = 1,platl
	          wrk3d(:,j,lev) = reshape( zm(:,lev,j,:), (/ plon /) )
	       end do
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk3d ), &
                               'rest_file_output: Failed to write geo midpoint height variable' )
            write(*,*) 'rest_file_output: zmid written'
         end if
	 if( allocated( gather_3d ) ) then
            deallocate( gather_3d )
	 end if
      end if
      if( present( zint ) ) then
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather geo interface height
!-----------------------------------------------------------------------
         count(1) = plon*platl*plevp
	 if( io_node ) then
            allocate( gather_3d(plonl,plevp,platl,pplon,nodes),stat=astat )
         else
            allocate( gather_3d(1,1,1,1,1),stat=astat )
         end if
         if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocated gather_3d; error = ',astat
	    call endrun
         end if
         call mpi_gather( zint, count(1), MPI_DOUBLE_PRECISION, &
			  gather_3d, count(1), MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'rest_file_output: mpi_gather failed for zint; error code = ',istat
	    call endrun
         end if
#endif
!-----------------------------------------------------------------------
!     	... Write geo interface height
!-----------------------------------------------------------------------
	 if( io_node ) then
            call handle_ncerr( nf_inq_varid( ncid, 'geohgt-int', var_id ), &
                               'rest_file_output: Failed to get geo interface height variable id' )
	    if( allocated( wrk3d ) ) then
	       deallocate( wrk3d )
	       allocate( wrk3d(plon,plat,plevp),stat=astat )
	       if( astat /= 0 ) then
	          write(*,*) 'rest_file_output: Failed to allocate wrk3d; error = ',astat
	          call endrun
	       end if
	    end if
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do lev = 1,plevp
	          do j = 1,platl
	             wrk3d(:,j+offset,lev) = reshape( gather_3d(:,lev,j,:,node), (/ plon /) )
	          end do
	       end do
	    end do
#else
	    do lev = 1,plevp
	       do j = 1,platl
	          wrk3d(:,j,lev) = reshape( zint(:,lev,j,:), (/ plon /) )
	       end do
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk3d ), &
                               'rest_file_output: Failed to write geo interface height variable' )
            write(*,*) 'rest_file_output: zint written'
         end if
	 if( allocated( gather_3d ) ) then
            deallocate( gather_3d )
	 end if
      end if
      if( present( cldtop ) ) then
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather cldtop
!-----------------------------------------------------------------------
         count(1) = plon*platl
	 if( io_node ) then
            allocate( gather_2d(plonl,platl,pplon,nodes),stat=astat )
	 else
            allocate( gather_2d(1,1,1,1),stat=astat )
         end if
         if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocated gather_2d; error = ',astat
	    call endrun
         end if
         call mpi_gather( cldtop, count(1), MPI_DOUBLE_PRECISION, &
			  gather_2d, count(1), MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'rest_file_output: mpi_gather failed for cldtop; error code = ',istat
	    call endrun
         end if
#endif
	 if( io_node ) then
            call handle_ncerr( nf_inq_varid( ncid, 'cldtop', var_id ), &
                               'rest_file_output: Failed to get cldtop variable id' )
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do j = 1,platl
	          wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
	       end do
	    end do
#else
	    do j = 1,platl
	       wrk2d(:,j) = reshape( cldtop(:,j,:), (/ plon /) )
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d), &
                              'rest_file_output: Failed to write cldtop variable' )
            write(*,*) 'rest_file_output: cldtop written'
         end if
	 if( allocated( gather_2d ) ) then
            deallocate( gather_2d )
	 end if
      end if
      if( present( cldbot ) ) then
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather cldbot
!-----------------------------------------------------------------------
         count(1) = plon*platl
	 if( io_node ) then
            allocate( gather_2d(plonl,platl,pplon,nodes),stat=astat )
	 else
            allocate( gather_2d(1,1,1,1),stat=astat )
         end if
         if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocated gather_2d; error = ',astat
	    call endrun
         end if
         call mpi_gather( cldbot, count(1), MPI_DOUBLE_PRECISION, &
			  gather_2d, count(1), MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'rest_file_output: mpi_gather failed for cldbot; error code = ',istat
	    call endrun
         end if
#endif
	 if( io_node ) then
            call handle_ncerr( nf_inq_varid( ncid, 'cldbot', var_id ), &
                               'rest_file_output: Failed to get cldbot variable id' )
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do j = 1,platl
	          wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
	       end do
	    end do
#else
	    do j = 1,platl
	       wrk2d(:,j) = reshape( cldbot(:,j,:), (/ plon /) )
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d ), &
                               'rest_file_output: Failed to write cldbot variable' )
            write(*,*) 'rest_file_output: cldbot written'
         end if
	 if( allocated( gather_2d ) ) then
            deallocate( gather_2d )
	 end if
      end if
      if( present( precip ) ) then
#ifdef USE_MPI
!-----------------------------------------------------------------------
!     	... Gather precip
!-----------------------------------------------------------------------
         count(1) = plon*platl
	 if( io_node ) then
            allocate( gather_2d(plonl,platl,pplon,nodes),stat=astat )
	 else
            allocate( gather_2d(1,1,1,1),stat=astat )
         end if
         if( astat /= 0 ) then
	    write(*,*) 'rest_file_output: Failed to allocated gather_2d; error = ',astat
	    call endrun
         end if
         call mpi_gather( precip, count(1), MPI_DOUBLE_PRECISION, &
			  gather_2d, count(1), MPI_DOUBLE_PRECISION, &
			  gather_node, MPI_COMM_WORLD, istat )
         if( istat /= MPI_SUCCESS ) then
	    write(*,*) 'rest_file_output: mpi_gather failed for precip; error code = ',istat
	    call endrun
         end if
#endif
	 if( io_node ) then
            call handle_ncerr( nf_inq_varid( ncid, 'precip', var_id ), &
                               'rest_file_output: Failed to get precip variable id' )
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do j = 1,platl
	          wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
	       end do
	    end do
#else
	    do j = 1,platl
	       wrk2d(:,j) = reshape( precip(:,j,:), (/ plon /) )
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d ), &
                               'rest_file_output: Failed to write precip variable' )
            write(*,*) 'rest_file_output: precip written'
	    if( allocated( gather_2d ) ) then
               deallocate( gather_2d )
	    end if
         end if
      end if
!-----------------------------------------------------------------------
!     	... write soil srf no emission variables
!-----------------------------------------------------------------------
      if( has_xemis_no ) then
         call soil_no_wrrst( plonl, platl, pplon, ncid )
      end if
!-----------------------------------------------------------------------
!     	... write megan maps
!-----------------------------------------------------------------------
      if( megan_cnt > 0 ) then
         call bvoc_wrrst( plonl, platl, pplon, ncid )
      end if

      if( allocated( wrk2d ) ) then
         deallocate( wrk2d )
      end if
      if( allocated( wrk3d ) ) then
         deallocate( wrk3d )
      end if

      end subroutine rest_file_output

      integer function qrsti( name, ival )
!-----------------------------------------------------------------------
! 	... Query restart module for integer data
!           Return value:
!            0 => success
!           -1 => variable name not found
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(out)         :: ival      ! requested value
      character(len=*), intent(in) :: name      ! variable name of requested integer data

      qrsti = 0

      if( name == 'RSTFRQ' ) then
         ival = rstfrq
      else
         qrsti = -1
      end if

      end function qrsti

      integer function qrstc( name, cval )
!-----------------------------------------------------------------------
! 	... Query history file module for character data
!           Return value:
!           0 => success
!          -1 => variable name not found
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in)  :: name      ! variable name of requested character string
      character(len=*), intent(out) :: cval      ! requested character string

      qrstc = 0
      select case( name )
         case( 'MONTHLY' )
            cval = rst_time_token
         case default
            qrstc = -1
      end select

      end function qrstc

      subroutine prtrst
!-----------------------------------------------------------------------
! 	... Print module contents
!-----------------------------------------------------------------------

      implicit none

      write(*,*) 'module MO_RESTART variables'
      write(*,*) 'lunrst0 =',lunrst0
      write(*,*) 'lunrst1 =',lunrst1
      write(*,*) 'rstrrt =',rstrrt
      write(*,*) 'rstfrq =',rstfrq
      write(*,*) 'async =',async
      write(*,*) 'rmrst =',rmrst
      write(*,*) 'lpthrst0 =',lpthrst0
      write(*,*) 'rpthrst0 =',rpthrst0
      write(*,*) 'lpthrst1 =',lpthrst1
      write(*,*) 'rpthrst1 =',rpthrst1

      end subroutine prtrst

      end module mo_restart
