

      module mo_tracinp
!-----------------------------------------------------------------------
! 	... Tracer initial conditions
!-----------------------------------------------------------------------

      implicit none

      private
      public  :: initrac

      save

      integer :: nlon                 ! ic longitudes
      integer :: nlat                 ! ic latitudes
      integer :: nlev                 ! ic levels
      integer :: ncid                 ! netcdf file id
      integer :: lat_limits(2)

      real, allocatable :: ic_lons(:)
      real, allocatable :: ic_lats(:)

      contains

      subroutine initrac( icdate, icsec, lun, remove, as, ps, plonl, pplon, platl )
!-----------------------------------------------------------------------
! 	... Initialize /tracinp/, and tracer arrays.   The tracer fields
!           must come directly from a time sample that exists on the input
!           file (though not necessarily the first time sample).
!-----------------------------------------------------------------------

      use netcdf
      use mo_grid,       only : plon, plat, plev, pcnst
      use mo_constants,  only : pi, d2r, phi, lam
      use plevs,         only : ps0, hyam, hybm
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits
      use mo_mpi,        only : base_lat
      use mo_calendar,   only : diffdat
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : ic_flsp

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        plonl, pplon, platl
      integer, intent(in) :: &
        icdate, &     ! date of initial conditions in yymmdd format
        icsec , &     ! seconds relative to date for initial conditions
        lun           ! unit number for input

      real, intent(in) :: &
        ps(plonl,-3:platl+4,pplon)       ! surface press( pascals )
      real, intent(out) :: &
              as(plonl,plev,pcnst,platl,pplon) ! advected species
      logical, intent(in) :: remove

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        istat, &             ! return status
        astat, &             ! allocation status
        gndx, &              ! regridding index
        input_date, input_secs, &
        dimid, varid, &
        i, ii, ip, j, jl, ju, k, kl, ku, m

      real, allocatable, dimension(:,:,:,:) :: &
                ic_vmr_int   ! temp storage for ic interpolated vmr
      real, allocatable :: ic_vmr(:,:,:)         ! temp storage for ic vmr
      real, allocatable :: ic_pmid(:,:,:,:)      ! ic pressure field
      real, allocatable, dimension(:,:) :: &
                ic_ps        ! temp storage for ic surface pressure
      real, allocatable, dimension(:,:,:) :: &
                ic_ps_int    ! temp storage for ic interpolated surface pressure

      real :: ic_p0
      real, allocatable, dimension(:) :: &
                ic_hyam, &   ! intial condition hybrid coefficients
                ic_hybm      ! intial condition hybrid coefficients
      real :: pinterp, delp
!-----------------------------------------------------------------------
!	... These variables are all dimensioned for the model simulation grid
!-----------------------------------------------------------------------
      real :: pmid(plonl,plev,platl,pplon)
      real :: wrk(plon,platl)
      real :: dummyP0(1)

      logical :: cosbtra
      character(len=128) :: &
        ctmp        ! temp string for system command
      character(len=168) :: &
        ncfile, &  ! NetCDF filename for tracer initial conditions
        lpath, &   ! Local path for tracer initial conditions
        mspath     ! Remote path for tracer initial conditions

!-----------------------------------------------------------------------
!     	... Open netcdf ic file
!-----------------------------------------------------------------------
      ncfile  = ic_flsp%nl_filename
      lpath   = ic_flsp%local_path
      mspath  = ic_flsp%remote_path
      ncid = open_netcdf_file( ncfile, lpath, mspath )

!-----------------------------------------------------------------------
!     	... Get intial condition dimension info
!-----------------------------------------------------------------------
      call handle_ncerr( NF_INQ_DIMID( ncid, 'lon', dimid ), &
                         'initrac: Failed to get longitude dimension id' )
      call handle_ncerr( NF_INQ_DIMLEN( ncid, dimid, nlon  ), &
                         'initrac: Failed to get longitude dimension size' )
      call handle_ncerr( NF_INQ_DIMID( ncid, 'lev', dimid ), &
                         'initrac: Failed to get level dimension id' )
      call handle_ncerr( NF_INQ_DIMLEN( ncid, dimid, nlev  ), &
                         'initrac: Failed to get level dimension size' )
      call handle_ncerr( NF_INQ_DIMID( ncid, 'lat', dimid ), &
                         'initrac: Failed to get latitude dimension id' )
      call handle_ncerr( NF_INQ_DIMLEN( ncid, dimid, nlat  ), &
                         'initrac: Failed to get latitude dimension size' )
!-----------------------------------------------------------------------
!     	... Get date information
!-----------------------------------------------------------------------
      call handle_ncerr( NF_INQ_VARID( ncid, 'date', varid ), &
                         'initrac: Failed to get date id' )
      call handle_ncerr( NF_GET_VAR_INT( ncid, varid, input_date ), &
                         'initrac: Failed to get date' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'datesec', varid ), &
                         'initrac: Failed to get datesec id' )
      call handle_ncerr( NF_GET_VAR_INT( ncid, varid, input_secs ), &
                         'initrac: Failed to get date' )

!-----------------------------------------------------------------------
!  	... Make sure time sample matches initial condition time
!-----------------------------------------------------------------------
      if( diffdat(  mod( icdate, 10000 ), icsec, &
                    mod( input_date, 10000 ), input_secs ) /= 0. ) then
         write(*,*) 'initrac: first time sample= ', input_date, input_secs
         call endrun
      end if

!-----------------------------------------------------------------------
!   	... Allocate memory for ic vertical grid
!-----------------------------------------------------------------------
      allocate( ic_hyam(nlev),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_hyam allocation error = ',astat
	 call endrun
      end if
      allocate( ic_hybm(nlev),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_hyam allocation error = ',astat
	 call endrun
      end if
!-----------------------------------------------------------------------
!     	... Get ic vertical coefficients
!-----------------------------------------------------------------------
      call handle_ncerr( NF_INQ_VARID( ncid, 'P0', varid ), 'initrac: Failed to get P0 id' )
      call handle_ncerr( NF_GET_VAR_DOUBLE( ncid, varid, dummyP0 ), 'initrac: Failed to get P0' )
      ic_p0 = dummyP0(1)
      call handle_ncerr( NF_INQ_VARID( ncid, 'hyam', varid ), 'initrac: Failed to get hyam id' )
      call handle_ncerr( NF_GET_VAR_DOUBLE( ncid, varid, ic_hyam ), 'initrac: Failed to get hyam' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'hybm', varid ), 'initrac: Failed to get hybm id' )
      call handle_ncerr( NF_GET_VAR_DOUBLE( ncid, varid, ic_hybm ), 'initrac: Failed to get hybm' )

!-----------------------------------------------------------------------
!   	... Allocate memory for ic lon and lat grids
!-----------------------------------------------------------------------
      allocate( ic_lons(nlon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_lons allocation error = ',astat
	 call endrun
      end if
      allocate( ic_lats(nlat),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_lats allocation error = ',astat
	 call endrun
      end if
!-----------------------------------------------------------------------
!     	... Get ic lon and lat grids
!-----------------------------------------------------------------------
      call handle_ncerr( NF_INQ_VARID( ncid, 'lon', varid ), 'initrac: Failed to get lon id' )
      call handle_ncerr( NF_GET_VAR_DOUBLE( ncid, varid, ic_lons ), 'initrac: Failed to get ic_lons' )
      call handle_ncerr( NF_INQ_VARID( ncid, 'lat', varid ), 'initrac: Failed to get lat id' )
      call handle_ncerr( NF_GET_VAR_DOUBLE( ncid, varid, ic_lats ), 'initrac: Failed to get ic_lats' )
!-----------------------------------------------------------------------
!     	... Initialize the regridding
!-----------------------------------------------------------------------
      write(*,'(1x,''initrac: nlat, nlon = '',2i4)') nlat, nlon
      write(*,'(1x,''initrac: plat, plon = '',2i4)') plat, plon
      ic_lats(:nlat) = d2r * ic_lats(:nlat)
      ic_lons(:nlon) = d2r * ic_lons(:nlon)
      gndx =  regrid_inti( nlat, plat, nlon, plon, ic_lons, lam, ic_lats, phi, 0, platl )
      if( gndx < 0 ) then
         write(*,*) 'initrac: REGRID_INTI error = ',gndx
	 call endrun
      else
         write(*,*) 'initrac: Using regrid index = ',gndx
      end if

      jl = base_lat + 1
      ju = base_lat + platl
      if( gndx > 0 ) then
         lat_limits = regrid_lat_limits( gndx )
      else
         lat_limits = (/ jl, ju /)
      end if
      write(*,'(1x,''initrac: lat_limits = '',2i4)') lat_limits
!-----------------------------------------------------------------------
!   	... Allocate memory for ic and ic surface pressure
!-----------------------------------------------------------------------
      allocate( ic_vmr(nlon,lat_limits(1):lat_limits(2),nlev),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_vmr allocation error = ',astat
	 call endrun
      end if
      allocate( ic_vmr_int(plonl,nlev,platl,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_vmr_int allocation error = ',astat
	 call endrun
      end if
      allocate( ic_ps(nlon,lat_limits(1):lat_limits(2)),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_ps allocation error = ',astat
	 call endrun
      end if
      allocate( ic_ps_int(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_ps_int allocation error = ',astat
	 call endrun
      end if
      allocate( ic_pmid(plonl,nlev,platl,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'initrac: ic_pmid allocation error = ',astat
	 call endrun
      end if
      call gettrac( ic_ps=ic_ps )
!--------------------------------------------------------------------------
!	... Regrid the initial condition surface pressure to model grid
!--------------------------------------------------------------------------
      call regrid_2d( ic_ps, wrk, gndx, jl, ju, do_poles=.true. )
      ic_ps_int = reshape( wrk, (/ plonl,platl,pplon /), order=(/1,3,2/) )
      deallocate( ic_ps )
!--------------------------------------------------------------------------
!	... Form initial condition midpoint pressure field
!--------------------------------------------------------------------------
      if( pplon > 1 ) then
!$OMP PARALLEL DO private( ip, k, i, j )
         do ip = 1,pplon
            do k = 1,nlev
	       do j = 1,platl
                  ic_pmid(:plonl,k,j,ip) = ic_hyam(k) * ic_p0 + ic_hybm(k) * ic_ps_int(:plonl,j,ip)
               end do
            end do
!--------------------------------------------------------------------------
!	... Form simulation grid midpoint pressure field
!--------------------------------------------------------------------------
            do k = 1,plev
	       do j = 1,platl
                  pmid(:,k,j,ip) = hyam(k) * ps0 + hybm(k) * ps(:,j,ip)
               end do
            end do
         end do
      else
         do k = 1,nlev
	    do j = 1,platl
               ic_pmid(:plonl,k,j,1) = ic_hyam(k) * ic_p0 + ic_hybm(k) * ic_ps_int(:plonl,j,1)
            end do
         end do
!--------------------------------------------------------------------------
!	... Form simulation grid midpoint pressure field
!--------------------------------------------------------------------------
         do k = 1,plev
	    do j = 1,platl
               pmid(:,k,j,1) = hyam(k) * ps0 + hybm(k) * ps(:,j,1)
            end do
         end do
      end if

species_loop : &
      do m = 1,pcnst-1
         call gettrac( ic_vmr=ic_vmr, ndx=m )
!--------------------------------------------------------------------------
!	... Regrid the initial condition vmr to simulation grid
!--------------------------------------------------------------------------
	 do k = 1,nlev
            call regrid_2d( ic_vmr(:,:,k), wrk, gndx, jl, ju, do_poles=.true. )
	    do ip = 1,pplon
	       ic_vmr_int(:plonl,k,:platl,ip) = wrk((ip-1)*plonl+1:ip*plonl,:platl)
            end do
         end do
         if( pplon > 1 ) then
!$OMP PARALLEL DO private( ip, k, i, j, pinterp, delp, kl, ku )
            do ip = 1,pplon
!--------------------------------------------------------------------------
!	... Pressure interpolate intial conditions to simulation vertical grid
!--------------------------------------------------------------------------
               do j = 1,platl
                  do k = 1,plev
                     do i = 1,plonl
                        pinterp = pmid(i,k,j,ip)
                        if( pinterp <= ic_pmid(i,1,j,ip) ) then
                           as(i,k,m,j,ip) = ic_vmr_int(i,1,j,ip)
                        else if( pinterp > ic_pmid(i,nlev,j,ip) ) then
                           as(i,k,m,j,ip) = ic_vmr_int(i,nlev,j,ip)
                        else
                           do ku = 2,nlev
                              if( pinterp <= ic_pmid(i,ku,j,ip) ) then
                                 kl = ku - 1
                                 delp = log( pinterp/ic_pmid(i,kl,j,ip) ) &
                                        / log( ic_pmid(i,ku,j,ip)/ic_pmid(i,kl,j,ip) )
                                 as(i,k,m,j,ip) = ic_vmr_int(i,kl,j,ip) &
                                        + delp * (ic_vmr_int(i,ku,j,ip) &
                                                  - ic_vmr_int(i,kl,j,ip))
                                 exit
                              end if
                           end do
                        end if
                     end do
                  end do
               end do
            end do
!$OMP END PARALLEL DO
         else
!--------------------------------------------------------------------------
!	... Pressure interpolate intial conditions to simulation vertical grid
!--------------------------------------------------------------------------
            do j = 1,platl
               do k = 1,plev
                  do i = 1,plonl
                     pinterp = pmid(i,k,j,1)
                     if( pinterp <= ic_pmid(i,1,j,1) ) then
                        as(i,k,m,j,1) = ic_vmr_int(i,1,j,1)
                     else if( pinterp > ic_pmid(i,nlev,j,1) ) then
                        as(i,k,m,j,1) = ic_vmr_int(i,nlev,j,1)
                     else
                        do ku = 2,nlev
                           if( pinterp <= ic_pmid(i,ku,j,1) ) then
                              kl = ku - 1
                              delp = log( pinterp/ic_pmid(i,kl,j,1) ) &
                                     / log( ic_pmid(i,ku,j,1)/ic_pmid(i,kl,j,1) )
                              as(i,k,m,j,1) = ic_vmr_int(i,kl,j,1) &
                                     + delp * (ic_vmr_int(i,ku,j,1) &
                                               - ic_vmr_int(i,kl,j,1))
                              exit
                           end if
                        end do
                     end if
                  end do
               end do
            end do
         end if
      end do species_loop

      deallocate( ic_vmr )
!--------------------------------------------------------------------------
!	... Set initial liquid cloud water to zero
!--------------------------------------------------------------------------
      as(:,:plev,pcnst,:,:) = 0.

!--------------------------------------------------------------------------
!	... Cleanup memory
!--------------------------------------------------------------------------
      deallocate( ic_ps_int )
      deallocate( ic_vmr_int )
      deallocate( ic_pmid )
      deallocate( ic_hyam )
      deallocate( ic_hybm )
      deallocate( ic_lats )
      deallocate( ic_lons )
!--------------------------------------------------------------------------
!  	... Close file
!--------------------------------------------------------------------------
      call handle_ncerr( NF_CLOSE( ncid ), &
                         'initrac: Failed to close file ' // lpath(:len_trim(lpath))  )

      end subroutine initrac

      subroutine gettrac( ic_ps, ic_vmr, ndx )
!-----------------------------------------------------------------------
! 	... Read constituent data from netcdf intial condition file
!-----------------------------------------------------------------------

      use netcdf
      use m_tracname, only : tracnam
      use mo_grid,    only : pcnst
      use mo_mpi,     only : masternode

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, optional, intent(in) :: ndx                                                       ! species index
      real, optional, intent(out)   :: ic_ps(nlon,lat_limits(1):lat_limits(2))                   ! ic surf press
      real, optional, intent(out)   :: ic_vmr(nlon,lat_limits(1):lat_limits(2),nlev)             ! ic initial concentrations

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: &
        j, k, m, &  ! indicies
        varid, &    ! variable id
        ncret       ! netcdf return code
      integer :: &
        start2(2), &
        count2(2), &
        start3(3), &
        count3(3)
      character(len=8) :: fldname

      start2(:) = (/ 1,lat_limits(1) /)
      count2(:) = (/ nlon,lat_limits(2)-lat_limits(1)+1 /)
      start3(:) = (/ 1,lat_limits(1),1 /)
      count3(:) = (/ nlon,lat_limits(2)-lat_limits(1)+1,nlev /)

      if( present( ic_ps ) ) then
!-----------------------------------------------------------------------
! 	... Get surface pressure
!-----------------------------------------------------------------------
         call handle_ncerr( NF_INQ_VARID( ncid, 'PS', varid ), &
                            'gettrac: Failed to get variable id for PS' )
         call handle_ncerr( NF_GET_VARA_DOUBLE( ncid, varid, start2, count2, ic_ps ), &
                            'gettrac: Failed to get variable PS' )
      end if
      if( present( ic_vmr ) .and. present( ndx ) ) then
!-----------------------------------------------------------------------
! 	... Get transported variables
!-----------------------------------------------------------------------
	 fldname = tracnam(ndx)
         ncret = nf_inq_varid( ncid, trim( fldname ), varid )
         if( ncret /= NF_NOERR ) then
            if( masternode ) then
               write(*,*) 'gettrac: Failed to get variable id for ' // trim( fldname ) &
                          // ', will initialize with 10^-38'
            end if
            ic_vmr(:,:,:) = 1.e-38
         else
            call handle_ncerr( NF_GET_VARA_DOUBLE( ncid, varid, start3, count3, ic_vmr(1,lat_limits(1),1) ), &
                            'gettrac: Failed to get variable ' // trim( fldname ) )
            if( masternode ) then
               write(*,*) 'gettrac: Loaded variable ' // trim( fldname )
            end if
         end if
      end if

      end subroutine gettrac

      end module mo_tracinp
