!-----------------------------------------------------------------------
!++bee, 7 february 1996
! vertical diffusion/pbl codes from omega0_9 modified for match as follows:
! * use pmgrid.com from match rather than ccm3s pmgrid.h
! * put qmincg into /comvd/
! * remove assumption that water vapor is the first constituent in the
!   constituent and surface flux arrays.  pass in arrays for water vapor
!   in separate arguments.  change loops over the non-water species to
!   "do m=1,pcnst" instead of "do m=2,pcnst".
! * change 1st dimension of cflx from plonl to plonl.  will change match
!   to use the plonl convention.
! * use matchs version of virtem.
! * could speed up and reduce memory requirements by removing calcs for
!   diffusion of momentum coefficients which arent being used (in vdiff).
!-----------------------------------------------------------------------

      module mo_vdiff

      use mo_grid, only : plevp, pcnst

      implicit none

      private
      public :: vdinti, vdiffdr

      save
!-----------------------------------------------------------------------
! 	... pbl constants
!-----------------------------------------------------------------------

      real :: &
           betam = 15., &   ! constant in wind gradient expression
           betas = 5., &    ! constant in surface layer gradient expression
           betah = 15., &   ! constant in temperature gradient expression 
           fak = 8.5, &     ! constant in surface temperature excess         
           g, &             ! gravitational acceleration
           onet, &          ! 1/3 power in wind gradient expression
           fakn = 7.2, &    ! constant in turbulent prandtl number
           ricr = .3, &     ! critical richardson number
           sffrac = .1, &   ! surface layer fraction of boundary layer
           vk = .4, &       ! von karmans constant
           ccon, &          ! fak * sffrac * vk
           binm, &          ! betam * sffrac
           binh             ! betah * sffrac

!-----------------------------------------------------------------------
! 	... constants used in vertical diffusion and pbl
!-----------------------------------------------------------------------
      real :: &
           cpair, &         ! specific heat of dry air
           rcpair, &        ! 1./cpair
           cpvir, &         ! derived constant for cp moist air
           gravit, &        ! acceleration due to gravity
           rair, &          ! gas const for dry air
           gor, &           ! g/rair
           gorsq, &         ! gor**2
           zkmin            ! minimum kneutral*f(ri)
      real :: &
           ml2(plevp), &    ! mixing lengths squared
           qmincg(pcnst)    ! min. constituent concentration counter-gradient term
      integer :: &
           ntopfl, &        ! top level to which vertical diffusion is applied.
           npbl             ! maximum number of levels in pbl from surface

      contains

      subroutine pbinti( gravx )
!-----------------------------------------------------------------------
! 	... initialize time independent variables of pbl package
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      real, intent(in) :: gravx     !  acceleration of gravity

!-----------------------------------------------------------------------
! 	... basic constants
!-----------------------------------------------------------------------
      g    = gravx
      onet = 1./3.

!-----------------------------------------------------------------------
! 	... derived constants
!-----------------------------------------------------------------------
      ccon = fak*sffrac*vk
      binm = betam*sffrac
      binh = betah*sffrac

      end subroutine pbinti

      subroutine vdiff( lat, ip, um1, vm1, tm1, &
                        pmidm1, pintm1, rpdel, rpdeli, ztodt, &
                        zm, taux, tauy, shflx, cflx, &
			thp, qp1, pblh, ustar, kvh, &
			kvm, tpert, qpert, cgs, shp1, &
			wvflx, plonl )
!-----------------------------------------------------------------------
! 	... driver routine to compute vertical diffusion of momentum,
!           moisture, trace constituents and potential temperature.
!
!           free atmosphere diffusivities are computed first; then modified
!           by the boundary layer scheme; then passed to individual
!           parameterizations mvdiff, qvdiff
!
!           the free atmosphere diffusivities are based on standard mixing length 
!           forms for the neutral diffusivity multiplied by functns of richardson 
!           number. k = l^2 * |dv/dz| * f(ri). the same functions are used for 
!           momentum, potential temperature, and constitutents.
!           the stable richardson num function (ri>0) is taken from holtslag and 
!           beljaars (1989), ecmwf proceedings. f = 1 / (1 + 10*ri*(1 + 8*ri))
!           the unstable richardson number function (ri<0) is taken from  ccm1.
!           f = sqrt(1 - 18*ri)
!-----------------------------------------------------------------------

      use mo_virtem, only : virtem
      use mo_qneg,   only : qneg3, shneg
      use mo_grid,   only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lat, ip             ! latitude index, long tile index
      integer, intent(in) :: plonl               ! number of local longitudes
      real, intent(in) :: &
           ztodt                     ! 2 delta-t
      real, intent(in) :: &
           um1(plonl,plev), &        ! u wind input
           vm1(plonl,plev), &        ! v wind input
           tm1(plonl,plev), &        ! temperature input
           pmidm1(plonl,plev), &     ! midpoint pressures
           pintm1(plonl,plevp), &    ! interface pressures
           rpdel(plonl,plev), &      ! 1./pdel  (thickness bet interfaces)
           rpdeli(plonl,plev), &     ! 1./pdeli (thickness bet midpoints)
           zm(plonl,plev), &         ! midpoint geoptl height above sfc
           taux(plonl), &            ! x surface stress (n)
           tauy(plonl), &            ! y surface stress (n)
           shflx(plonl), &           ! surface sensible heat flux (w/m2)
           cflx(plonl,pcnst), &      ! surface constituent flux (kg/m2/s)
           wvflx(plonl)              ! water vapor flux (kg/m2/s)

      real, intent(inout) :: &
           qp1(plonl,plev,pcnst), &  ! moist, tracers after vert. diff
           shp1(plonl,plev), &       ! specific humidity (kg/kg)
           thp(plonl,plev)           ! pot temp after vert. diffusion

      real, intent(out) :: &
           pblh(plonl), &            ! planetary boundary layer height
           ustar(plonl), &           ! surface friction velocity
           kvh(plonl,plevp), &       ! coefficient for heat and tracers
           kvm(plonl,plevp), &       ! coefficient for momentum
           tpert(plonl), &           ! convective temperature excess
           qpert(plonl), &           ! convective humidity excess
           cgs(plonl,plevp)          ! counter-grad star (cg/flux)

!-----------------------------------------------------------------------
!	... local varaibles
!-----------------------------------------------------------------------
      integer :: &
              i, &                   ! longitude index
              k, &                   ! vertical index
              kp1, &                 ! vertical index
              m                      ! constituent index
      integer :: &
              nval, &                ! num of values which meet criteria
              ii                     ! longitude index of found points
      integer :: &
              indx(plonl)            ! array of indices of potential q<0
      real :: &
           dvdz2 , &                 ! (du/dz)**2 + (dv/dz)**2
           dz , &                    ! delta z between midpoints
           fstab , &                 ! stable f(ri)
           funst , &                 ! unstable f(ri)
           rinub, &                  ! richardson no=(g/theta)(dtheta/dz)/(du/dz**2+dv/dz**2)
           sstab, &                  ! static stability = g/th  * dth/dz
           kvn, &                    ! neutral kv
           rcpair, &                 ! 1./cpair
           ztodtgor                  ! ztodt*gravit/rair
      real :: &
           tmp2(plonl)               ! temporary storage
      real :: &
           sup(plonl,plev), &        ! -upper diag for heat and constituts
           sub(plonl,plev), &        ! -lower diag for heat and constits
           cgh(plonl,plevp), &       ! countergradient term for heat
           cgq(plonl,plevp,pcnst), & ! countergrad term for constituent
           cgsh(plonl,plevp), &      ! countergrad term for sh
           kvf(plonl,plevp)          ! free atmosphere kv at interfaces
      real :: &
           potbar(plonl,plevp), &    ! pintm1(k)/(.5*(tm1(k)+tm1(k-1))
           tmp1(plonl), &            ! temporary storage
           dtbot(plonl), &           ! lowest layer t change from heat flx
           dqbot(plonl,pcnst), &     ! lowest layer q change from const flx
           dshbot(plonl), &          ! lowest layer sh change from wvflx
           thx(plonl,plev), &        ! temperature input + counter gradient
           thv(plonl,plev), &        ! virtual potential temperature
           qmx(plonl,plev,pcnst), &  ! constituents input + counter grad
           shmx(plonl,plev), &       ! sh input + counter grad
           zeh(plonl,plev), &        ! term in tri-diag. matrix system (t & q)
           termh(plonl,plev)         ! 1./(1. + sup(k) + sub(k) - sub(k)*zeh(k-1))
      logical :: adjust(plonl)

!-----------------------------------------------------------------------
! 	... convert the surface fluxes to lowest level tendencies
!-----------------------------------------------------------------------
      rcpair       = 1./cpair
      tmp1(:)      = ztodt*gravit*rpdel(:,plev)
      dshbot(:)    = wvflx(:)*tmp1(:)
      dtbot(:)     = shflx(:)*tmp1(:)*rcpair
      kvf(:,plevp) = 0.
      do m = 1,pcnst
         dqbot(:,m) = cflx(:,m)*tmp1(:)
      end do

!-----------------------------------------------------------------------
! 	... set the vertical diffusion coefficient above the top diffusion level
!-----------------------------------------------------------------------
      do k = 1,ntopfl
         kvf(:,k) = 0.
      end do

!-----------------------------------------------------------------------
! 	... compute virtual potential temperature for use in static stability 
!           calculation.  0.61 is 1. - r(water vapor)/r(dry air).  use 0.61 instead
!           of a computed variable in order to obtain an identical simulation to
!           case 414.
!-----------------------------------------------------------------------
      call virtem( thp, shp1, thv, plonl )

!-----------------------------------------------------------------------
! 	... compute the free atmosphere vertical diffusion coefficients
!           kvh = kvq = kvm. 
!-----------------------------------------------------------------------
      do k = ntopfl,plev-1
         kp1 = k + 1
         do i = 1,plonl
!-----------------------------------------------------------------------
! 	... vertical shear squared, min value of (delta v)**2 prevents zero shear.
!-----------------------------------------------------------------------
            dvdz2 = (um1(i,k) - um1(i,kp1))**2 + (vm1(i,k) - vm1(i,kp1))**2
            dvdz2 = max( dvdz2,1.e-36 )
            dz    = zm(i,k) - zm(i,kp1)
            dvdz2 = dvdz2/(dz**2)
!-----------------------------------------------------------------------
! 	... static stability (use virtual potential temperature)
!-----------------------------------------------------------------------
            sstab = gravit*2.*(thv(i,k) - thv(i,kp1))/((thv(i,k) + thv(i,kp1))*dz)
!-----------------------------------------------------------------------
! 	... richardson number, stable and unstable modifying functions
!-----------------------------------------------------------------------
            rinub = sstab/dvdz2
            fstab = 1.0/(1.0 + 10.0*rinub*(1.0 + 8.0*rinub))
            funst = max( 1. - 18.*rinub,0. )
!-----------------------------------------------------------------------
! 	... select the appropriate function of the richardson number
!-----------------------------------------------------------------------
            if( rinub < 0. ) then
	       fstab = sqrt( funst )
	    end if
!-----------------------------------------------------------------------
! 	... neutral diffusion coefficient
!           compute mixing length (z), where z is the interface height estimated
!           with an 8 km scale height.
!-----------------------------------------------------------------------
            kvn = ml2(k)*sqrt( dvdz2 )
!-----------------------------------------------------------------------
! 	... full diffusion coefficient (modified by f(ri)),
!-----------------------------------------------------------------------
            kvf(i,kp1) = max( zkmin,kvn*fstab )
         end do
      end do

!-----------------------------------------------------------------------
! 	... determine the boundary layer kvh (=kvq), kvm, 
!           counter gradient terms (cgh, cgq, cgs)
!           boundary layer height (pblh) and 
!           the perturbation temperature and moisture (tpert and qpert)
!           the free atmosphere kv is returned above the boundary layer top.
!-----------------------------------------------------------------------
      call pbldif( thp, shp1, zm, um1, vm1, &
                   tm1, pmidm1, kvf, cflx, shflx, &
                   taux, tauy, ustar, kvm, kvh, &
                   cgh, cgq, cgs, pblh, tpert, qpert, &
                   wvflx, cgsh, plonl )

!-----------------------------------------------------------------------
! 	... add the counter grad terms to potential temp, specific humidity
!           and other constituents in the bdry layer. note, npbl gives the max
!           num of levels which are permitted to be within the boundary layer.
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! 	... first set values above boundary layer
!-----------------------------------------------------------------------
      do k = 1,plev-npbl
         thx(:,k)  = thp(:,k)
         shmx(:,k) = shp1(:,k)
      end do
      do m = 1,pcnst
         do k = 1,plev-npbl
            qmx(:,k,m) = qp1(:,k,m)
         end do
      end do
      do k = 2,plev
         potbar(:,k) = pintm1(:,k)/(.5*(tm1(:,k) + tm1(:,k-1)))
      end do
      potbar(:,plevp) = pintm1(:,plevp)/tm1(:,plev)

!-----------------------------------------------------------------------
! 	... now focus on the boundary layer
!-----------------------------------------------------------------------
      ztodtgor = ztodt*gor
      do k = plev-npbl+1,plev
         kp1 = k + 1
         tmp1(:) = ztodtgor*rpdel(:,k)
         thx(:,k) = thp(:,k) + tmp1(:) &
                               *(potbar(:,kp1)*kvh(:,kp1)*cgh(:,kp1) &
                                 - potbar(:,k)*kvh(:,k)*cgh(:,k))
         shmx(:,k) = shp1(:,k) + tmp1(:) &
                                 *(potbar(:,kp1)*kvh(:,kp1)*cgsh(:,kp1) &
                                   - potbar(:,k)*kvh(:,k)*cgsh(:,k))
         do m = 1,pcnst
            qmx(:,k,m) = qp1(:,k,m) + tmp1(:) &
                                      *(potbar(:,kp1)*kvh(:,kp1)*cgq(:,kp1,m) &
                                        - potbar(:,k)*kvh(:,k)*cgq(:,k,m))
         end do
      end do

!-----------------------------------------------------------------------
! 	... check for neg qs in each constituent and put the original vertical
!           profile back if a neg value is found. a neg value implies that the
!           quasi-equilibrium conditions assumed for the countergradient term are
!           strongly violated.
!-----------------------------------------------------------------------
      do m = 1,pcnst
         adjust(:) = .false.
         do k = plev-npbl+1,plev
            do i = 1,plonl
               if( qmx(i,k,m) < qmincg(m) ) then
	          adjust(i) = .true.
	       end if
            end do
         end do
!-----------------------------------------------------------------------
! 	... find long indices of those columns for which negatives were found
!-----------------------------------------------------------------------
	 nval = count( adjust(:) )
!-----------------------------------------------------------------------
! 	... replace those columns with original values
!-----------------------------------------------------------------------
         if( nval > 0 ) then
            do k = plev-npbl+1,plev
	       where( adjust(:) )
                  qmx(:,k,m) = qp1(:,k,m)
	       endwhere
            end do
         end if
      end do
!-----------------------------------------------------------------------
! 	... repeat above for sh
!-----------------------------------------------------------------------
      adjust(:) = .false.
      do k = plev-npbl+1,plev
         do i = 1,plonl
!-----------------------------------------------------------------------
! 	... 1.e-12 is the value of qmin (=qmincg) used in ccm2.
!-----------------------------------------------------------------------
            if( shmx(i,k) < 1.e-12 ) then
               adjust(i) = .true.
	    end if
         end do
      end do
!-----------------------------------------------------------------------
! 	... find long indices of those columns for which negatives were found
!-----------------------------------------------------------------------
      nval = count( adjust(:) )
!-----------------------------------------------------------------------
! 	... replace those columns with original values
!-----------------------------------------------------------------------
      if( nval > 0 ) then
         do k = plev-npbl+1,plev
	    where( adjust(:) )
               shmx(:,k) = shp1(:,k)
	    endwhere
         end do
      end if

!-----------------------------------------------------------------------
! 	... determine superdiagonal and subdiagonal coeffs
!           of the tridiagonal diffusion matrix. the diagonal elements are a
!           combination of sub and sup; they are not explicitly provided to the 
!           solver
!-----------------------------------------------------------------------
      do k = ntopfl,plev-1
         kp1 = k + 1
         tmp2(:) = ztodt*gorsq*rpdeli(:,k)*(potbar(:,kp1)**2)*kvh(:,kp1)
         sup(:,k  ) = tmp2(:)*rpdel(:,k)
         sub(:,kp1) = tmp2(:)*rpdel(:,kp1)
      end do
!-----------------------------------------------------------------------
! 	... the last element of the upper diagonal is zero.
!-----------------------------------------------------------------------
      sup(:,plev) = 0.
!-----------------------------------------------------------------------
! 	... calculate e(k) for heat & momentum vertical diffusion.  this term is 
!           required in solution of tridiagonal matrix defined by implicit diffusion eqn.
!-----------------------------------------------------------------------
      termh(:,ntopfl) = 1./(1. + sup(:,ntopfl))
      zeh(:,ntopfl)   = sup(:,ntopfl)*termh(:,ntopfl)
      do k = ntopfl+1,plev-1
         termh(:,k) = 1./(1. + sup(:,k) + sub(:,k)*(1. - zeh(:,k-1)))
         zeh(:,k)   = sup(:,k)*termh(:,k)
      end do

!-----------------------------------------------------------------------
! 	... diffuse constituents
!-----------------------------------------------------------------------
      call qvdiff( pcnst, qmx, dqbot, sub, zeh, &
		   termh, qp1, plonl )

!-----------------------------------------------------------------------
! 	... identify and correct constituents exceeding user defined bounds
!-----------------------------------------------------------------------
      call qneg3( 'vdiff   ', lat, qp1, plonl )

!-----------------------------------------------------------------------
! 	... diffuse sh
!-----------------------------------------------------------------------
      call qvdiff( 1, shmx, dshbot, sub, zeh, &
		   termh, shp1, plonl )

!-----------------------------------------------------------------------
! 	... correct sh
!-----------------------------------------------------------------------
      call shneg( 'vdiff:sh', lat, shp1, plonl )

!-----------------------------------------------------------------------
! 	... diffuse potential temperature
!-----------------------------------------------------------------------
      call qvdiff( 1, thx, dtbot, sub, zeh, &
		   termh, thp, plonl )

      end subroutine vdiff

      subroutine pbldif( th      ,q       ,z       ,u       ,v, &
                         t       ,pmid    ,kvf     ,cflx    ,shflx, &
                         taux    ,tauy    ,ustar   ,kvm     ,kvh, &
                         cgh     ,cgq     ,cgs     ,pblh    ,tpert, &
                         qpert   ,wvflx   ,cgsh    ,plonl )
!------------------------------------------------------------------------
! 	... atmospheric boundary layer computation.
!
!           nonlocal scheme that determines eddy diffusivities based on a
!           diagnosed boundary layer height and a turbulent velocity scale;
!           also, countergradient effects for heat and moisture, and constituents
!           are included, along with temperature and humidity perturbations which 
!           measure the strength of convective thermals in the lower part of the 
!           atmospheric boundary layer.
!
!           for more information, see holtslag, a.a.m., and b.a. boville, 1993:
!           local versus nonlocal boundary-layer diffusion in a global climate
!           model. j. clim., vol. 6., p. 1825--1842.
!------------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst

      implicit none

!------------------------------------------------------------------------
! 	... dummy arguments
!------------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, intent(in) :: &
           th(plonl,plev), &          ! potential temperature [k]
           q(plonl,plev), &           ! specific humidity [kg/kg]
           z(plonl,plev), &           ! height above surface [m]
           u(plonl,plev), &           ! windspeed x-direction [m/s]
           v(plonl,plev), &           ! windspeed y-direction [m/s]
           t(plonl,plev), &           ! temperature (used for density)
           pmid(plonl,plev), &        ! midpoint pressures
           kvf(plonl,plevp), &        ! free atmospheric eddy diffsvty [m2/s]
           cflx(plonl,pcnst), &       ! surface constituent flux (kg/m2/s)
           wvflx(plonl), &            ! water vapor flux (kg/m2/s)
           shflx(plonl), &            ! surface heat flux (w/m2)
           taux(plonl), &             ! surface u stress (n)
           tauy(plonl)                ! surface v stress (n)

      real, intent(out) :: &
           ustar(plonl), &            ! surface friction velocity [m/s]
           kvm(plonl,plevp), &        ! eddy diffusivity for momentum [m2/s]
           kvh(plonl,plevp), &        ! eddy diffusivity for heat [m2/s]
           cgh(plonl,plevp), &        ! counter-gradient term for heat [k/m]
           cgq(plonl,plevp,pcnst), &  ! counter-gradient term for constituents
           cgsh(plonl,plevp), &       ! counter-gradient term for sh
           cgs(plonl,plevp), &        ! counter-gradient star (cg/flux)
           pblh(plonl), &             ! boundary-layer height [m]
           tpert(plonl), &            ! convective temperature excess
           qpert(plonl)               ! convective humidity excess

!------------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------------
      real, parameter :: tiny = 1.e-36           ! lower bound for wind magnitude

      integer :: &
              i, &                 ! longitude index
              k, &                 ! level index
              m                    ! constituent index
      logical :: &
              unstbl(plonl), &     ! pts w/unstbl pbl (positive virtual ht flx)
              stblev(plonl), &     ! stable pbl with levels within pbl
              unslev(plonl), &     ! unstbl pbl with levels within pbl
              unssrf(plonl), &     ! unstb pbl w/lvls within srf pbl lyr
              unsout(plonl), &     ! unstb pbl w/lvls in outer pbl lyr
              check(plonl)         ! true=>chk if richardson no.>critcal
      real :: &
           heatv(plonl), &         ! surface virtual heat flux
           thvsrf(plonl), &        ! sfc (bottom) level virtual temperature
           thvref(plonl), &        ! reference level virtual temperature
           tkv, &                  ! model level potential temperature
           therm(plonl), &         ! thermal virtual temperature excess
           phiminv(plonl), &       ! inverse phi function for momentum
           phihinv(plonl), &       ! inverse phi function for heat 
           wm(plonl), &            ! turbulent velocity scale for momentum
           vvk, &                  ! velocity magnitude squared
           zm(plonl), &            ! current level height
           zp(plonl), &            ! current level height + one level up
           khfs(plonl), &          ! surface kinematic heat flux [mk/s]
           kqfs(plonl,pcnst), &    ! sfc kinematic constituent flux [m/s]
           kshfs(plonl), &         ! sfc kinematic moisture flux [m/s]
           zmzp                    ! level height halfway between zm and zp
      real :: &
           rino(plonl,plev), &     ! bulk richardson no. from level to ref lev
           tlv(plonl), &           ! ref. level pot tmp + tmp excess
           fak1(plonl), &          ! k*ustar*pblh
           fak2(plonl), &          ! k*wm*pblh
           fak3(plonl), &          ! fakn*wstr/wm 
           pblk(plonl), &          ! level eddy diffusivity for momentum
           pr(plonl), &            ! prandtl number for eddy diffusivities
           zl(plonl), &            ! zmzp / obukhov length
           zh(plonl), &            ! zmzp / pblh
           zzh(plonl), &           ! (1-(zmzp/pblh))**2
           wstr(plonl), &          ! w*, convective velocity scale
           rrho(plonl), &          ! 1./bottom level density (temporary)
           obklen(plonl), &        ! obukhov length
           ustr                    ! unbounded ustar
      real :: &
           term, &                 ! intermediate calculation
           fac, &                  ! interpolation factor
           pblmin                  ! min pbl height due to mechanical mixing

!------------------------------------------------------------------------
! 	... compute kinematic surface fluxes
!------------------------------------------------------------------------
      do i = 1,plonl
         rrho(i)  = rair*t(i,plev)/pmid(i,plev)
         ustr     = sqrt( sqrt( taux(i)**2 + tauy(i)**2 )*rrho(i) )
         ustar(i) = max( ustr,.01 )
         khfs(i)  = shflx(i)*rrho(i)/cpair
         kshfs(i) = wvflx(i)*rrho(i)
      end do
      do m = 1,pcnst
         kqfs(:plonl,m) = cflx(:plonl,m)*rrho(:plonl)
      end do

!------------------------------------------------------------------------
! 	... initialize output arrays with free atmosphere values
!------------------------------------------------------------------------
      do k = 1,plevp
         kvm(:,k)  = kvf(:,k)
         kvh(:,k)  = kvf(:,k)
         cgh(:,k)  = 0.
         cgsh(:,k) = 0.
         cgs(:,k)  = 0.
      end do
      do m = 1,pcnst
         do k = 1,plevp
            cgq(:,k,m) = 0.
         end do
      end do

!------------------------------------------------------------------------
! 	... compute various arrays for use later:
!------------------------------------------------------------------------
      do i = 1,plonl
         thvsrf(i) = th(i,plev)*(1.0 + 0.61*q(i,plev))
         heatv(i)  = khfs(i) + 0.61*th(i,plev)*kshfs(i)
         wm(i)     = 0.
         therm(i)  = 0.
         qpert(i)  = 0.
         tpert(i)  = 0.
         fak3(i)   = 0.  
         zh(i)     = 0.  
         obklen(i) = -thvsrf(i)*ustar(i)**3 &
                      /(g*vk*(heatv(i) + sign( 1.e-10,heatv(i) )))
      end do

!------------------------------------------------------------------------
! 	... define first a new factor fac=100 for use in richarson number
!           calculate virtual potential temperature first level
!           and initialize pbl height to z1
!------------------------------------------------------------------------
      fac = 100.
      do i = 1,plonl
         thvref(i) = th(i,plev)*(1.0 + 0.61*q(i,plev))
         pblh(i)   = z(i,plev)
         check(i)  = .true.
!------------------------------------------------------------------------
! 	... initialization of lowest level ri number 
!           (neglected in initial holtslag implementation)
!------------------------------------------------------------------------
         rino(i,plev) = 0.
      end do

!------------------------------------------------------------------------
! 	... pbl height calculation:
!           search for level of pbl. scan upward until the richardson number between
!           the first level and the current level exceeds the "critical" value.
!------------------------------------------------------------------------
      do k = plev-1,plev-npbl+1,-1
         do i = 1,plonl
            if( check(i) ) then
               vvk = (u(i,k) - u(i,plev))**2 + (v(i,k) - v(i,plev))**2 + fac*ustar(i)**2
               vvk = max( vvk,tiny )
               tkv = th(i,k)*(1. + .61*q(i,k))
               rino(i,k) = g*(tkv - thvref(i))*(z(i,k)-z(i,plev))/(thvref(i)*vvk)
               if( rino(i,k) >= ricr ) then
                  pblh(i) = z(i,k+1) &
                            + (ricr - rino(i,k+1)) &
                              /(rino(i,k) - rino(i,k+1))*(z(i,k) - z(i,k+1))
                  check(i) = .false.
               end if
            end if
         end do
      end do

!------------------------------------------------------------------------
! 	... set pbl height to maximum value where computation exceeds number of
!           layers allowed
!------------------------------------------------------------------------
      do i = 1,plonl
         if( check(i) ) then
	    pblh(i) = z(i,plevp-npbl)
	 end if
      end do

!------------------------------------------------------------------------
! 	... improve estimate of pbl height for the unstable points.
!           find unstable points (virtual heat flux is positive):
!------------------------------------------------------------------------
      do i = 1,plonl
         if( heatv(i) > 0. ) then
            unstbl(i) = .true.
            check(i)  = .true.
         else
            unstbl(i) = .false.
            check(i)  = .false.
         end if   
      end do

!------------------------------------------------------------------------
! 	... for the unstable case, compute velocity scale and the
!           convective temperature excess:
!------------------------------------------------------------------------
      do i = 1,plonl
         if( check(i) ) then
            phiminv(i)   = (1. - binm*pblh(i)/obklen(i))**onet
            wm(i)        = ustar(i)*phiminv(i)
            therm(i)     = heatv(i)*fak/wm(i)       
            rino(i,plev) = 0.
            tlv(i)       = thvref(i) + therm(i)
         end if
      end do

!------------------------------------------------------------------------
! 	... improve pblh estimate for unstable conditions using the
!           convective temperature excess:
!------------------------------------------------------------------------
      do k = plev-1,plev-npbl+1,-1
         do i = 1,plonl
            if( check(i) ) then
               vvk = (u(i,k) - u(i,plev))**2 + (v(i,k) - v(i,plev))**2 &
                    + fac*ustar(i)**2
               vvk = max( vvk,tiny )
               tkv = th(i,k)*(1. + 0.61*q(i,k))
               rino(i,k) = g*(tkv - tlv(i))*(z(i,k)-z(i,plev)) &
                           /(thvref(i)*vvk)
               if( rino(i,k) >= ricr ) then
                  pblh(i) = z(i,k+1) + (ricr - rino(i,k+1)) &
                            /(rino(i,k) - rino(i,k+1))*(z(i,k) - z(i,k+1))
                  check(i) = .false.
               end if
            end if
         end do
      end do

!------------------------------------------------------------------------
! 	... points for which pblh exceeds number of pbl layers allowed;
!           set to maximum
!------------------------------------------------------------------------
      do i = 1,plonl
         if( check(i) ) then
	    pblh(i) = z(i,plevp-npbl)
	 end if
      end do

!------------------------------------------------------------------------
! pbl height must be greater than some minimum mechanical mixing depth
! several investigators have proposed minimum mechanical mixing depth
! relationships as a function of the local friction velocity, u*.  we 
! make use of a linear relationship of the form h = c u* where c=700.
! the scaling arguments that give rise to this relationship most often 
! represent the coefficient c as some constant over the local coriolis
! parameter.  here we make use of the experimental results of koracin 
! and berkowicz (1988) [blm, vol 43] for wich they recommend 0.07/f
! where f was evaluated at 39.5 n and 52 n.  thus we use a typical mid
! latitude value for f so that c = 0.07/f = 700.
!------------------------------------------------------------------------
      do i = 1,plonl
         pblmin  = 700.*ustar(i)
         pblh(i) = max( pblh(i),pblmin )
      end do

!------------------------------------------------------------------------
! 	... pblh is now available; do preparation for diffusivity calculation:
!------------------------------------------------------------------------
      do i = 1,plonl
         pblk(i) = 0.
         fak1(i) = ustar(i)*pblh(i)*vk
!------------------------------------------------------------------------
! 	... do additional preparation for unstable cases only, set temperature
!           and moisture perturbations depending on stability.
!------------------------------------------------------------------------
         if( unstbl(i) ) then
            phiminv(i) = (1. - binm*pblh(i)/obklen(i))**onet
            phihinv(i) = sqrt(1. - binh*pblh(i)/obklen(i))
            wm(i)      = ustar(i)*phiminv(i)
            fak2(i)    = wm(i)*pblh(i)*vk
            wstr(i)    = (heatv(i)*g*pblh(i)/thvref(i))**onet 
            fak3(i)    = fakn*wstr(i)/wm(i)
            tpert(i)   = max( khfs(i)*fak/wm(i),0. )   
            qpert(i)   = max( kshfs(i)*fak/wm(i),0. )    
         else
            tpert(i)   = max( khfs(i)*fak/ustar(i),0. ) 
            qpert(i)   = max( kshfs(i)*fak/ustar(i),0. ) 
         end if
      end do

!------------------------------------------------------------------------
! 	... main level loop to compute the diffusivities and counter-gradient terms
!------------------------------------------------------------------------
      do k = plev,plev-npbl+2,-1
!------------------------------------------------------------------------
! 	... find levels within boundary layer
!------------------------------------------------------------------------
         do i = 1,plonl
            unslev(i) = .false.
            stblev(i) = .false.
            zm(i) = z(i,k)
	    zp(i) = z(i,k-1)
            if( zkmin == 0. .and. zp(i) > pblh(i) ) then
	       zp(i) = pblh(i)
	    end if
            if( zm(i) < pblh(i) ) then
               zmzp = 0.5*(zm(i) + zp(i))
               zh(i) = zmzp/pblh(i)
               zl(i) = zmzp/obklen(i)
               if( zh(i) <= 1. ) then
	          zzh(i) = (1. - zh(i))**2
	       else
                  zzh(i) = 0.
	       end if
!------------------------------------------------------------------------
! 	... stblev for points zm < plbh and stable and neutral
!           unslev for points zm < plbh and unstable
!------------------------------------------------------------------------
               if( unstbl(i) ) then
                  unslev(i) = .true.
               else
                  stblev(i) = .true.
               end if
            end if
         end do
!------------------------------------------------------------------------
! 	... stable and neutral points; set diffusivities; counter-gradient
!           terms zero for stable case:
!------------------------------------------------------------------------
         do i = 1,plonl
            if( stblev(i) ) then
               if( zl(i) <= 1. ) then
                  pblk(i) = fak1(i)*zh(i)*zzh(i)/(1. + betas*zl(i))
               else
                  pblk(i) = fak1(i)*zh(i)*zzh(i)/(betas + zl(i))
               end if
               kvm(i,k) = max( pblk(i),kvf(i,k) )
               kvh(i,k) = kvm(i,k)
            end if
         end do
!------------------------------------------------------------------------
! 	... unssrf, unstable within surface layer of pbl
!           unsout, unstable within outer   layer of pbl
!------------------------------------------------------------------------
         do i = 1,plonl
            unssrf(i) = .false.
            unsout(i) = .false.
            if( unslev(i) ) then
               if( zh(i) < sffrac ) then
                  unssrf(i) = .true.
               else
                  unsout(i) = .true.
               end if
            end if
         end do
!------------------------------------------------------------------------
! 	... unstable for surface layer; counter-gradient terms zero
!------------------------------------------------------------------------
         do i = 1,plonl
            if( unssrf(i) ) then
               term    = (1. - betam*zl(i))**onet
               pblk(i) = fak1(i)*zh(i)*zzh(i)*term
               pr(i)   = term/sqrt(1. - betah*zl(i))
            end if
         end do
!------------------------------------------------------------------------
! 	... unstable for outer layer; counter-gradient terms non-zero:
!------------------------------------------------------------------------
         do i = 1,plonl
            if( unsout(i) ) then
               pblk(i)   = fak2(i)*zh(i)*zzh(i)
               cgs(i,k)  = fak3(i)/(pblh(i)*wm(i))
               cgh(i,k)  = khfs(i)*cgs(i,k)
               pr(i)     = phiminv(i)/phihinv(i) + ccon*fak3(i)/fak
               cgsh(i,k) = kshfs(i)*cgs(i,k)
            end if
         end do
         do m = 1,pcnst
            do i = 1,plonl
               if( unsout(i) ) then
	          cgq(i,k,m) = kqfs(i,m)*cgs(i,k)
	       end if
            end do
         end do
!------------------------------------------------------------------------
! 	... for all unstable layers, set diffusivities
!------------------------------------------------------------------------
         do i = 1,plonl
            if( unslev(i) ) then
               kvm(i,k) = max( pblk(i),kvf(i,k) )
               kvh(i,k) = max( pblk(i)/pr(i),kvf(i,k) )
            end if
         end do
      end do

      end subroutine pbldif

      subroutine qvdiff( ncnst, qin, qflx, sub, ze, &
			 term, qout, plonl )
!-----------------------------------------------------------------------
! 	... solve vertical diffusion eqtn for constituent with explicit srfc flux.
!           procedure for solution of the implicit equation follows richtmyer and 
!           morton (1967,pp 198-199).
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
           plonl
      integer, intent(in) :: &
           ncnst                    ! num of constituents being diffused
      real, intent(in) :: &
           qin(plonl,plev,ncnst), & ! initial constituent
           qflx(plonl,ncnst), &     ! sfc q flux into lowest model level
           sub(plonl,plev), &       ! -lower diag coeff.of tri-diag matrix
           term(plonl,plev)         ! 1./(1. + sup(k) + sub(k) - sub(k)*ze(k-1))

      real, intent(inout) :: &
           ze(plonl,plev)           ! term in tri-diag. matrix system

      real, intent(out) :: &
           qout(plonl,plev,ncnst)   ! final constituent

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      real :: &
           zfq(plonl,plev,pcnst), & ! terms appear in soln of tri-diag sys
           tmp1d(plonl)             ! temporary workspace (1d array)
      integer :: &
              k, &                  ! vertical index
              m                     ! constituent index

!-----------------------------------------------------------------------
! 	... calculate fq(k).  terms fq(k) and e(k) are required in solution of 
!           tridiagonal matrix defined by implicit diffusion eqn.
!           note that only levels ntopfl through plev need be solved for.
!           no vertical diffusion is applied above this level
!-----------------------------------------------------------------------
      do m = 1,ncnst
         zfq(:,ntopfl,m) = qin(:,ntopfl,m)*term(:,ntopfl)
         do k = ntopfl+1,plev-1
            zfq(:,k,m) = (qin(:,k,m) + sub(:,k)*zfq(:,k-1,m))*term(:,k)
         end do
      end do
!-----------------------------------------------------------------------
! 	... bottom level: (includes  surface fluxes)
!-----------------------------------------------------------------------
      tmp1d(:) = 1./(1. + sub(:,plev)*(1. - ze(:,plev-1)))
      ze(:,plev) = 0.
      do m = 1,ncnst
         zfq(:,plev,m) = (qin(:,plev,m) + qflx(:,m) &
                          + sub(:,plev)*zfq(:,plev-1,m))*tmp1d(:)
      end do
!-----------------------------------------------------------------------
! 	... perform back substitution
!-----------------------------------------------------------------------
      do m = 1,ncnst
         qout(:,plev,m) = zfq(:,plev,m)
         do k = plev-1,ntopfl,-1
            qout(:,k,m) = zfq(:,k,m) + ze(:,k)*qout(:,k+1,m)
         end do
      end do

      end subroutine qvdiff

      subroutine vdiffar( lat     ,tm1, &
                          pmidm1  ,pintm1  ,rpdel   ,rpdeli  ,ztodt, &
                          cflx    ,qp1     ,kvh     ,cgs     ,plonl )
!-----------------------------------------------------------------------
! 	... driver routine to compute vertical diffusion of trace constituents
!           using archived coefficients for cgs and kvh.  this is a gutted
!           version of vdiff.
!-----------------------------------------------------------------------

      use mo_qneg, only : qneg3
      use mo_grid, only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lat     ! latitude index
      integer, intent(in) :: plonl   ! lon tile dim
      real, intent(in) :: &
           tm1(plonl,plev), &        ! temperature input
           pmidm1(plonl,plev), &     ! midpoint pressures
           pintm1(plonl,plevp), &    ! interface pressures
           rpdel(plonl,plev), &      ! 1./pdel  (thickness bet interfaces)
           rpdeli(plonl,plev), &     ! 1./pdeli (thickness bet midpoints)
           ztodt, &                  ! 2 delta-t
           cflx(plonl,pcnst), &      ! surface constituent flux (kg/m2/s)
           kvh(plonl,plevp), &       ! coefficient for heat and tracers
           cgs(plonl,plevp)          ! counter-grad star (cg/flux)

      real, intent(inout) :: &
           qp1(plonl,plev,pcnst)     ! moist, tracers after vert. diff

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      real :: &
           cah(plonl,plev), &       ! -upper diag for heat and constituts
           cch(plonl,plev), &       ! -lower diag for heat and constits
           cgq(plonl,plevp,pcnst), &! countergrad term for constituent
           potbar(plonl,plevp), &   ! pintm1(k)/(.5*(tm1(k)+tm1(k-1))
           tmp1(plonl), &           ! temporary storage
           tmp2, &                  ! temporary storage
           ztodtgor, &              ! ztodt*gravit/rair
           gorsq, &                 ! (gravit/rair)**2
           dqbot(plonl,pcnst), &    ! lowest layer q change from const flx
           qmx(plonl,plev,pcnst), & ! constituents input + counter grad
           zeh(plonl,plev), &       ! term in tri-diag. matrix system (t & q)
           termh(plonl,plev)        ! 1./(1. + cah(k) + cch(k) - cch(k)*zeh(k-1))
      integer :: &
              indx(plonl), &        ! array of indices of potential q<0
              ilogic(plonl), &      ! 1 => adjust vertical profile
              nval, &               ! num of values which meet criteria
              ii                    ! longitude index of found points
      integer :: &
              i, &                  ! longitude index
              k, &                  ! vertical index
              m                     ! constituent index

!-----------------------------------------------------------------------
! 	... convert the surface fluxes to lowest level tendencies
!-----------------------------------------------------------------------
      do i = 1,plonl
         tmp1(i) = ztodt*gravit*rpdel(i,plev)
      end do
      do m = 1,pcnst
         do i = 1,plonl
            dqbot(i,m) = cflx(i,m)*tmp1(i)
         end do
      end do

!-----------------------------------------------------------------------
! 	... counter gradient terms:
!-----------------------------------------------------------------------
      call pbldifar( tm1, pmidm1, cflx, cgs, cgq, plonl )

!-----------------------------------------------------------------------
! 	... add the counter grad terms to potential temp, specific humidity
!           and other constituents in the bdry layer. note, npbl gives the max
!           num of levels which are permitted to be within the boundary layer.
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
! 	... first set values above boundary layer
!-----------------------------------------------------------------------
      do k = 1,plev-npbl
         do m = 1,pcnst
            qmx(:,k,m) = qp1(:,k,m)
         end do
      end do
      do k = 2,plev
         potbar(:,k) = pintm1(:,k)/(.5*(tm1(:,k) + tm1(:,k-1)))
      end do
      potbar(:,plevp) = pintm1(:,plevp)/tm1(:,plev)

!-----------------------------------------------------------------------
! 	... now focus on the boundary layer
!-----------------------------------------------------------------------
      ztodtgor = ztodt*gravit/rair
      do k = plev-npbl+1,plev
         do i = 1,plonl
            tmp1(i) = ztodtgor*rpdel(i,k)
         end do
         do m = 1,pcnst
            do i = 1,plonl
               qmx(i,k,m) = qp1(i,k,m) + tmp1(i)*(potbar(i,k+1)*kvh(i,k+1)*cgq(i,k+1,m) &
	                                          - potbar(i,k)*kvh(i,k)*cgq(i,k,m))
            end do
         end do
      end do

!-----------------------------------------------------------------------
! 	... check for neg qs in each constituent and put the original vertical
!           profile back if a neg value is found. a neg value implies that the
!           quasi-equilibrium conditions assumed for the countergradient term are
!           strongly violated.
!           original code rewritten by rosinski 7/8/91 to vectorize in longitude.
!-----------------------------------------------------------------------
      do m = 1,pcnst
         ilogic(:plonl) = 0
         do k = plev-npbl+1,plev
            do i = 1,plonl
               if( qmx(i,k,m) < qmincg(m) ) then
	          ilogic(i) = 1
	       end if
            end do
         end do
!-----------------------------------------------------------------------
! 	... find long indices of those columns for which negatives were found
!-----------------------------------------------------------------------
	 nval = count( ilogic(:plonl) == 1 )

!-----------------------------------------------------------------------
! 	... replace those columns with original values
!-----------------------------------------------------------------------
         if( nval > 0 ) then
            do k = plev-npbl+1,plev
	       where( ilogic(:plonl) == 1 )
                  qmx(:plonl,k,m) = qp1(:plonl,k,m)
	       endwhere
            end do
         end if
      end do

!-----------------------------------------------------------------------
! 	... determine superdiagonal (ca(k)) and subdiagonal (cc(k)) coeffs
!           of the tridiagonal diffusion matrix. the diagonal elements are a
!           combination of ca and cc; they are not explicitly provided to the 
!           solver
!-----------------------------------------------------------------------
      gorsq = (gravit/rair)**2
      do k = ntopfl,plev-1
         do i = 1,plonl
            tmp2 = ztodt*gorsq*rpdeli(i,k)*(potbar(i,k+1)**2)
            cah(i,k  ) = kvh(i,k+1)*tmp2*rpdel(i,k  )
            cch(i,k+1) = kvh(i,k+1)*tmp2*rpdel(i,k+1)
         end do
      end do
!-----------------------------------------------------------------------
! 	... the last element of the upper diagonal is zero.
!-----------------------------------------------------------------------
      do i = 1,plonl
         cah(i,plev) = 0.
      end do
!-----------------------------------------------------------------------
! 	... calculate e(k) for heat vertical diffusion.  this term is 
!           required in solution of tridiagonal matrix defined by implicit diffusion eqn.
!-----------------------------------------------------------------------
      do i = 1,plonl
         termh(i,ntopfl) = 1./(1. + cah(i,ntopfl))
         zeh(i,ntopfl) = cah(i,ntopfl)*termh(i,ntopfl)
      end do
      do k = ntopfl+1,plev-1
         do i = 1,plonl
            termh(i,k) = 1./(1. + cah(i,k) + cch(i,k) - cch(i,k)*zeh(i,k-1))
            zeh(i,k) = cah(i,k)*termh(i,k)
         end do
      end do
!-----------------------------------------------------------------------
! 	... diffuse constituents
!-----------------------------------------------------------------------
      call qvdiff( pcnst, qmx, dqbot, cch, zeh, &
		   termh, qp1, plonl )
!-----------------------------------------------------------------------
! 	... identify and correct constituents exceeding user defined bounds
!-----------------------------------------------------------------------
      call qneg3( 'vdiff   ', lat, qp1(1,1,1), plonl )

      end subroutine vdiffar

      subroutine pbldifar( t, pmid, cflx, cgs, cgq, plonl )
!-----------------------------------------------------------------------
! 	... modified version of pbldif which only calculates cgq given cgs.
!-----------------------------------------------------------------------
      use mo_grid, only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, intent(in) :: &
           t(plonl,plev), &        ! temperature (used for density)
           pmid(plonl,plev), &     ! midpoint pressures
           cflx(plonl,pcnst), &     ! surface constituent flux (kg/m2/s)
           cgs(plonl,plevp)        ! counter-gradient star (cg/flux)

      real, intent(out) :: &
           cgq(plonl,plevp,pcnst)  ! counter-gradient term for constituents

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: &
              i, &                 ! longitude index
              k, &                 ! level index
              m                    ! constituent index
      real :: &
           rrho(plonl), &          ! 1./bottom level density
           kqfs(plonl,pcnst)       ! sfc kinematic constituent flux [m/s]

!------------------------------------------------------------------------
! 	... compute kinematic surface fluxes
!------------------------------------------------------------------------
      rrho(:) = rair*t(:,plev)/pmid(:,plev)
      do m = 1,pcnst
         kqfs(:,m) = cflx(:,m)*rrho(:)
      end do
!------------------------------------------------------------------------
! 	... initialize output arrays with free atmosphere values
!------------------------------------------------------------------------
      do m = 1,pcnst
         do k = 1,plevp
            cgq(:,k,m) = 0.
         end do
      end do
!------------------------------------------------------------------------
! 	... compute the counter-gradient terms:
!------------------------------------------------------------------------
      do k = plev,plev-npbl+2,-1
         do m = 1,pcnst
            cgq(:,k,m) = kqfs(:,m)*cgs(:,k)
         end do
      end do

      end subroutine pbldifar

      subroutine vdinti( cpwvx, cpairx, gravx, rairx, qmin, ref_pmid )
!-----------------------------------------------------------------------
! 	... initialize time independent fields for vertical diffusion.
!           call initialization routine for boundary layer scheme.
!-----------------------------------------------------------------------

      use mo_grid, only : pcnst, plev, plevp

      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      real, intent(in) :: &
           cpwvx, &        ! spec. heat of water vapor at const. pressure
           cpairx, &       ! specific heat of dry air
           gravx, &        ! acceleration due to gravity
           rairx, &        ! gas constant for dry air
           qmin(pcnst), &  ! global minimum constituent concentration
           ref_pmid(plev)  ! reference midpoint pressure (pa)

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      real, parameter :: pbl_press = 400.e2         ! pressure cap for pbl (pa)
      integer :: k, &                               ! vertical loop index
                 m

!-----------------------------------------------------------------------
! 	... hard-wired numbers.
!           zkmin = minimum k = kneutral*f(ri)
!-----------------------------------------------------------------------
      zkmin = .01

!-----------------------------------------------------------------------
! 	... set physical constants for vertical diffusion and pbl
!-----------------------------------------------------------------------
      cpair  = cpairx
      rcpair = 1./cpair
      gravit = gravx
      rair   = rairx
      gor    = gravit/rair
      gorsq  = gor*gor

!-----------------------------------------------------------------------
! 	... derived constants
!           ntopfl = top level to which v-diff is applied
!           npbl   = max number of levels (from bottom) in pbl
!-----------------------------------------------------------------------
      do k = plev,1,-1
	 if( ref_pmid(k) < pbl_press ) then
	    exit
	 end if
      end do
      npbl = max( 1,plev - k )
      write(*,*) 'vdinti: pbl height will be limited to bottom ',npbl, &
                 ' model levels. top is ',1.e-2*ref_pmid(plevp-npbl),' hpa'
      if( plev == 1 ) then
         ntopfl = 0
      else
         ntopfl = 1
      end if
      cpvir = cpwvx/cpairx - 1.

!-----------------------------------------------------------------------
! 	... set the square of the mixing lengths
!-----------------------------------------------------------------------
      ml2(1) = 0.
      do k = 2,plev
         ml2(k) = 30.**2
      end do
      ml2(plevp) = 0.
!-----------------------------------------------------------------------
! 	... set the minimum mixing ratio for the counter-gradient term.
!           normally this should be the same as qmin.
!-----------------------------------------------------------------------
      do m = 1,pcnst
         qmincg(m) = qmin(m)
      end do

!-----------------------------------------------------------------------
! 	... initialize pbl variables
!-----------------------------------------------------------------------
      call pbinti( gravx )

      end subroutine vdinti

      subroutine vdiffdr( divdiff, arvdiff, lat, ip, um1, &
                          vm1, tadv, t1, pmid, pint, &
			  rpdel, rpdeli, dtime, zm, taux, &
			  tauy, hflx, sflx, shflx, cgs, &
			  kvh, thp, as2, shp, pblh, &
			  tpert, qpert, plonl )
!-----------------------------------------------------------------------
! 	... vertical diffusion calculations on a latitude slice of data
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp, pcnst

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
        lat, &            ! latitude index
        ip, &             ! longitude index
        plonl             ! number of local longitudes

      real, intent(in) :: &
        dtime                  ! timestep size (s)
      real, intent(in) :: &
        um1(plonl,plev), &     ! u wind
        vm1(plonl,plev), &     ! v wind
        tadv(plonl,plev), &    ! temperature corresponding to thm (k)
        t1(plonl,plev), &      ! temperature from input data interpolated to end of current timestep, (k)
        pmid(plonl,plev), &    ! pressure at layer midpoints
        pint(plonl,plevp), &   ! pressure at layer interfaces
        rpdel(plonl,plev), &   ! 1/pdel
        rpdeli(plonl,plev), &  ! 1./(pmid(k+1)-pmid(k))
        zm(plonl,plev), &      ! potential height above surface at midpoints
        taux(plonl), &         ! x surface stress (n)
        tauy(plonl), &         ! y surface stress (n)
        hflx(plonl), &         ! surface sensible heat flux (w/m2)
        sflx(plonl,pcnst), &   ! surface fluxes for advected species
        shflx(plonl)           ! surface water vapor flux (kg/m2/s)

      real, intent(inout) :: &
!-----------------------------------------------------------------------
! 	... cgs and kvh are input for archived vdiff and output for diagnosed vdiff
!-----------------------------------------------------------------------
        cgs(plonl,plevp), &       ! counter-gradient coefficient
        kvh(plonl,plevp), &       ! vertical diffusion coefficient
        shp(plonl,plev), &        ! specific humidity (kg/kg)
        as2(plonl,plev,pcnst), &  ! advected species
        thp(plonl,plev)           ! potential temperature

      real, intent(out) :: &
        pblh(plonl), &           ! planetary boundary layer height
        tpert(plonl), &          ! convective temperature excess
        qpert(plonl)             ! convective humidity excess

      logical, intent(in) :: &
        divdiff, &        ! t => diagnosed vertical diffusion
        arvdiff           ! t => archived vertical diffusion (from ccm2 run)

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
!-----------------------------------------------------------------------
!     	... diagnosed vdiff
!-----------------------------------------------------------------------
      real :: &
        kvm(plonl,plevp), &     ! momentum k
        ustar(plonl)            ! surface friction velocity

      if( divdiff ) then
         call vdiff( lat, ip, um1, vm1, tadv, &
                     pmid, pint, rpdel, rpdeli, dtime, &
                     zm, taux, tauy, hflx, sflx, &
		     thp, as2, pblh, ustar, kvh, &
		     kvm, tpert, qpert, cgs, shp, &
		     shflx, plonl )

      else if( arvdiff ) then
!-----------------------------------------------------------------------
!  	... vertical diffusion using archived values of cgs and kvh.
!-----------------------------------------------------------------------
         call vdiffar( lat, t1, &
                       pmid, pint, rpdel, rpdeli, dtime, &
                       sflx, as2, kvh, cgs, plonl )
      end if

      end subroutine vdiffdr

      end module mo_vdiff
