
      module mo_z0
!-----------------------------------------------------------------------
! 	... module for geopotential height calculations
!-----------------------------------------------------------------------

      implicit none

      private
      public :: iniz, zint, zmid, zr2a, z2phi, zr2phi

      save

      real, private :: &
        rair, &    ! Gas constant for dry air
        gravit, &  ! acceleration due to gravity (m/s^2)
        rog, &     ! rair/gravit
        rgrav      ! 1/gravit

      contains

      subroutine iniz(  xrair, xgravit )
!-----------------------------------------------------------------------
! 	... Initialize module
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      real, intent(in) :: &
        xrair, &    ! Gas constant for dry air
        xgravit     ! acceleration due to gravity (m/s^2)

      rair   = xrair
      gravit = xgravit
      rog    = rair / gravit
      rgrav  = 1. / gravit

      end subroutine iniz

      subroutine zint( piln, pmln, tv, zm, zi, plonl )
!-----------------------------------------------------------------------
! 	... Compute the geopotential height at the interface points
!           from *HEIGHT ABOVE THE SURFACE* at midlayer midpoints,
!           using the supplied virtual temperatures and pressures
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, intent(in) :: &
           piln(plonl,plevp), & ! Log interface pressures
           pmln(plonl,plev), &  ! Log midpoint pressures
           tv(plonl,plev), &    ! Virtual temperature
           zm(plonl,plev)       ! Height above surface at midpoints

      real, intent(out) :: &
           zi(plonl,plevp)      ! Height above surface at interfaces

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i, &           ! Longitude index
                 k              ! Level index

!-----------------------------------------------------------------------
! 	... Add increment to midlayer height
!-----------------------------------------------------------------------
      do i = 1,plonl
         zi(i,1) = zm(i,1) + rog*(pmln(i,1) - piln(i,1))*tv(i,1)
      end do

      do k = 2,plev
         do i = 1,plonl
            zi(i,k) = zm(i,k) + rog*(pmln(i,k) - piln(i,k))*0.5* &
                      (2.0*tv(i,k) - (tv(i,k) - tv(i,k-1))/ &
                       (pmln(i,k) - pmln(i,k-1))* &
                       (pmln(i,k) - piln(i,k)))
         end do
      end do

!-----------------------------------------------------------------------
! 	... The surface height is zero by definition.
!-----------------------------------------------------------------------
      do i = 1,plonl
         zi(i,plevp) = 0.
      end do
      
      end subroutine zint

      subroutine zmid( piln, pmln, tv, z, plonl )
!-----------------------------------------------------------------------
! 	... Compute the geopotential height *ABOVE THE SURFACE*
!           at layer midpoints from the virtual temperatures and pressures
!-----------------------------------------------------------------------

      use mo_grid, only : plev, plevp

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, intent(in) :: &
           piln(plonl,plevp), &  ! Log interface pressures
           pmln(plonl,plev), &   ! Log midpoint pressures
           tv(plonl,plev)        ! Virtual temperature

      real, intent(out) :: &
           z(plonl,plev)         ! Height above surface at midpoints

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: i, &            ! Longitude index
                 k, &            ! Level index
                 l               ! Level index

!-----------------------------------------------------------------------
! 	... Diagonal term of hydrostatic equation
!-----------------------------------------------------------------------
      do k = 1,plev-1
         do i = 1,plonl
            z(i,k) = rog*tv(i,k)*.5*(pmln(i,k+1) - pmln(i,k))
         end do
      end do
      do i = 1,plonl
         z(i,plev) = rog*tv(i,plev)*(piln(i,plevp) - pmln(i,plev))
      end do

!-----------------------------------------------------------------------
! 	... Bottom level term of hydrostatic equation
!-----------------------------------------------------------------------
      do  k = 1,plev-1
         do i = 1,plonl
            z(i,k) = z(i,k) + rog*tv(i,plev)*(piln(i,plevp)  &
			     - .5*(pmln(i,plev-1) + pmln(i,plev)))
         end do
      end do

!-----------------------------------------------------------------------
! 	... Interior terms of hydrostatic equation
!-----------------------------------------------------------------------
      do k = 1,plev-2
         do l = k+1,plev-1
            do i = 1,plonl
               z(i,k) = z(i,k) + rog*(tv(i,l)) &
                                 *.5*(pmln(i,l+1) - pmln(i,l-1))
            end do
         end do
      end do

      end subroutine zmid

      subroutine zr2a( zr, nlev, phis, za, plonl )
!-----------------------------------------------------------------------
! 	... Convert from relative to absolute geopotential height.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      integer, intent(in) :: &
        nlev               ! number of levels

      real, intent(in) :: &
        zr(plonl,nlev), &  ! geopotential height relative to the surface
        phis(plonl)        ! surface geopotential

      real, intent(out) :: &
        za(plonl,nlev)     ! absolute geopotential height

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: k
	
      do k = 1,nlev
         za(:,k) = zr(:,k) + phis(:)*rgrav
      end do

      end subroutine zr2a

      subroutine z2phi( z, nlev, phi, plonl )
!-----------------------------------------------------------------------
! 	... Convert from geopotential height to geopotential.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      integer, intent(in) :: &
        nlev            ! number of levels

      real, intent(in) :: &
        z(plonl,nlev)   ! geopotential height

      real, intent(out) :: &
        phi(plonl,nlev)      ! geopotential

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: k
	
      do k = 1,nlev
         phi(:,k) = z(:,k)*gravit
      end do

      end subroutine z2phi

      subroutine zr2phi( z, nlev, phis, phi, plonl )
!-----------------------------------------------------------------------
! 	... Convert from geopotential height ABOVE SURFACE to geopotential.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... Dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      integer, intent(in) :: &
        nlev              ! number of levels

      real, intent(in) :: &
        z(plonl,nlev), &  ! geopotential height
        phis(plonl)       ! surface geopotential

      real, intent(out) :: &
        phi(plonl,nlev)   ! geopotential

!-----------------------------------------------------------------------
! 	... Local variables
!-----------------------------------------------------------------------
      integer :: k
	
      do k = 1,nlev
         phi(:,k) = z(:,k)*gravit + phis(:)
      end do

      end subroutine zr2phi

      end module mo_z0
